/*!
 *******************************************************************************
 * \file              spi_tclOnCarInputHandler.cpp
 * \brief             SPI input handler for OnCar devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Input handler class to send input events from Headunit to
                OnCar Device
 AUTHOR:         Rishav Sardar
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 16.05.2018  | Rishav Sardar         | Initial Version
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclOnCarInputHandler.h"
#include "spi_tclOnCarCmdInput.h"
#include "spi_tclOnCarManager.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclOnCarInputHandler.cpp.trc.h"
#endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported  


/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
** FUNCTION:  spi_tclOnCarInputHandler::spi_tclOnCarInputHandler()
***************************************************************************/
spi_tclOnCarInputHandler::spi_tclOnCarInputHandler()
:m_poOnCarCmdInput(NULL)
{
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarInputHandler::spi_tclOnCarInputHandler() entered "));

}//!end of spi_tclOnCarInputHandler()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarInputHandler::~spi_tclOnCarInputHandler()
***************************************************************************/
spi_tclOnCarInputHandler::~spi_tclOnCarInputHandler()
{
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarInputHandler::~spi_tclOnCarInputHandler() entered "));
   m_poOnCarCmdInput = NULL;
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarInputHandler::~spi_tclOnCarInputHandler() left "));
}//!end of ~spi_tclOnCarInputHandler()

/***************************************************************************
** FUNCTION:  t_Void  spi_tclOnCarInputHandler::vSelectDevice
***************************************************************************/
t_Void  spi_tclOnCarInputHandler::vSelectDevice(const t_U32 cou32DevId,
   const tenDeviceConnectionReq coenConnReq)
{
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vSelectDevice() entered : Device-0x%x \n",cou32DevId));
  SPI_INTENTIONALLY_UNUSED(coenConnReq);
  tenErrorCode enErrCode = e8UNKNOWN_ERROR;
  if(m_poOnCarCmdInput)
  {
      if(e8DEVCONNREQ_SELECT == coenConnReq)
      {
          enErrCode = (bInitializeInputSession()) ? e8NO_ERRORS : e8SELECTION_FAILED;
      }//if(e8DEVCONNREQ_SELECT == coenConnReq)
      else if(e8DEVCONNREQ_DESELECT == coenConnReq)
      {
          enErrCode = e8NO_ERRORS;
          m_poOnCarCmdInput->vUnInitializeInputEndpoint();
      }//else if(e8DEVCONNREQ_DESELECT == coenConnReq)
  }//if(m_poOnCarCmdInput)
  ETG_TRACE_USR2(("[DESC]::spi_tclOnCarInputHandler::vSelectDevice() : Error Code - %d",enErrCode));
  if(m_rInputCallbacks.fvSelectDeviceResp)
  {
     (m_rInputCallbacks.fvSelectDeviceResp)(cou32DevId,e8NO_ERRORS);
  }//if(m_rVideoCallbacks.fpvSelectDeviceCb
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vSelectDevice() left "));
}//!end of vSelectDevice()

/***************************************************************************
** FUNCTION:  t_Void  spi_tclOnCarInputHandler::vRegisterVideoCallbacks
***************************************************************************/
t_Void  spi_tclOnCarInputHandler::vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks)
{
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:vRegisterInputCallbacks() entered "));
   //Copy
   m_rInputCallbacks = corfrInputCallbacks;
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:vRegisterInputCallbacks() left "));
}//!end of vRegisterInputCallbacks()

/***************************************************************************
** FUNCTION:  spi_tclOnCarInputHandler::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclOnCarInputHandler::vOnSelectDeviceResult(const t_U32 cou32DevId,
                                                        const tenDeviceConnectionReq coenConnReq,
                                                        const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vOnSelectDeviceResult() eneterd : Dev-0x%x ConnReq-%d, RespCode-%d",
                   cou32DevId,ETG_ENUM(CONNECTION_REQ,coenConnReq),coenRespCode));
   if((e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode)
           &&(m_poOnCarCmdInput))
   {
       m_poOnCarCmdInput->vUnInitializeInputEndpoint();
       m_poOnCarCmdInput->vDestroyInputEnpointInstance();
   }//if((e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode)&&(m_poOnCarCmdInput))
   if((e8DEVCONNREQ_DESELECT == coenConnReq) && (m_poOnCarCmdInput))
   {
       m_poOnCarCmdInput->vDestroyInputEnpointInstance();
   }//if((e8DEVCONNREQ_DESELECT == coenConnReq) && (m_poOnCarCmdInput))
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vOnSelectDeviceResult() left"));
}//!end of vOnSelectDeviceResult()


/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarInputHandler::bInitialise()
 ***************************************************************************/
t_Bool spi_tclOnCarInputHandler::bInitialise()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::bInitialise() entered "));

   t_Bool bRet = false;
   spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
   if(poOnCarManager)
   {
        m_poOnCarCmdInput = poOnCarManager->poGetInputInstance();
        //! Register with OnCar manager for Input callbacks
        bRet = poOnCarManager->bRegisterObject((spi_tclOnCarRespInput*)this);
   }//if(poOnCarManager)
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarInputHandler::vProcessKeyEvents
***************************************************************************/
t_Void spi_tclOnCarInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle,
                                                tenKeyMode enKeyMode, tenKeyCode enKeyCode)const
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vProcessKeyEvents() entered : Dev-0x%x Key Mode-%d, Key Code-%d",
                                       u32DeviceHandle,enKeyMode,enKeyCode));
   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
   if(m_poOnCarCmdInput)
   {
       m_poOnCarCmdInput->vReportkey(u32DeviceHandle,enKeyMode,enKeyCode);
   } //if(m_poOnCarCmdInput)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler::vProcessKeyEvents() left "));
}//!end of vProcessKeyEvents()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarInputHandler::vPostNotifyErrorRequest
***************************************************************************/
t_Void spi_tclOnCarInputHandler::vPostNotifyErrorRequest(tenOnCarSinkError enOnCarSinkError)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:vPostNotifyErrorRequest() entered : OnCarSinkError = %d",
                    enOnCarSinkError));
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:vPostNotifyErrorRequest() left "));
}//!end of vPostNotifyErrorRequest()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarInputHandler::bInitializeInputSession
***************************************************************************/
t_Bool spi_tclOnCarInputHandler::bInitializeInputSession()
{
    //! Get Video settings instance for screen properties
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:bInitializeInputSession() entered"));
    t_Bool bInitStatus = false;
    if((m_poOnCarCmdInput) && ( m_poInputHandlerSettings))
    {
        trOnCarInputConfig rOnCarInputConfig;
        trVideoConfigData rVideoConfigData;
        tvecVideoConfigList vecVideoConfigList;
        m_poInputHandlerSettings->vGetVideoConfigData(e8DEV_TYPE_ONCAR,vecVideoConfigList);
        m_poInputHandlerSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);
        rOnCarInputConfig.u16TouchLayerID = static_cast <t_U16> (rVideoConfigData.u32TouchLayerId);
        rOnCarInputConfig.u16TouchSurfaceID = static_cast <t_U16> (rVideoConfigData.u32TouchSurfaceId);
        rOnCarInputConfig.u16TouchHeight = static_cast <t_U16> (rVideoConfigData.u32ProjScreen_Height);
        rOnCarInputConfig.u16TouchWidth = static_cast <t_U16> (rVideoConfigData.u32ProjScreen_Width);
        rOnCarInputConfig.u32DisplayHeight = rVideoConfigData.u32ProjScreen_Height;
        rOnCarInputConfig.u32DisplayWidth = rVideoConfigData.u32ProjScreen_Width;
        rOnCarInputConfig.enOnCarTouchEventType = (e8MULTI_TOUCH == rVideoConfigData.enTouchEventType) ? e8_ONCAR_TOUCH_EVENT_TYPE_MULTI_TOUCH : e8_ONCAR_TOUCH_EVENT_TYPE_SINGLE_TOUCH;
        rOnCarInputConfig.enOnCarTouchScreenType = (e8RESISTIVE_SCREEN == rVideoConfigData.enDisplayType) ? e8_ONCAR_TOUCH_SCREEN_TYPE_RESISTIVE : e8_ONCAR_TOUCH_SCREEN_TYPE_CAPACITIVE;
        bInitStatus = m_poOnCarCmdInput->bInitializeInputEndpoint(rOnCarInputConfig);
    } //if((m_poOnCarCmdInput) && (m_poInputHandlerSettings))
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarInputHandler:bInitializeInputSession() left with result:%d",bInitStatus));
    return bInitStatus;
}//!end of bInitializeInputSession()

// <EOF>
