/*!
 *******************************************************************************
 * \file             spi_tclOnCarDataService.cpp
 * \brief            OnCar Data Service class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Data Service class implements Data Service Info Management for
 OnCar capable devices. This class must be derived from base Data Service class.
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 26.04.2018  | Rishav Sardar         | Initial Version

 \endverbatim
 ******************************************************************************/

#include "spi_tclOnCarDataService.h"
#include "spi_tclMediator.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclOnCarCmdSensor.h"
#include "Timer.h"
//#include "NmeaEncoder.h"
//#include "spi_tclExtCompManager.h"
//#include "spi_tclExtCmdNavData.h"
//#include <algorithm>
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
      #include "trcGenProj/Header/spi_tclOnCarDataService.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const trOnCarSelectedDeviceInfo corEmptyDeviceInfo;

//! Vehicle Speed timer data
static timer_t srVehSpeedTimerID;
static t_Bool sbVehSpeedTimerRunning = false;
static const t_U32 scou32VehSpeedUpdateTimerVal = 100;

/****************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::spi_tclOnCarDataService
 ***************************************************************************/
spi_tclOnCarDataService::spi_tclOnCarDataService(const trDataServiceCb& rfcorDataServiceCb)
             :m_poOnCarCmdSensor(NULL),
              m_rDataServiceCb(rfcorDataServiceCb),
              m_enDayNightMode(e8_ONCAR_DAYNIGHT_MODE_DAY),
              m_enDrivingState(e8_ONCAR_DRIVING_STATE_PARKED),
              m_enVehMovState(e8VEHICLE_MOVEMENT_STATE_INVALID),
              m_enGearPosition(e8GEAR_POS_UNKNOWN),
              m_s16VehicleSpeed(0),
              m_bParkBrakeActive(false),
              m_s32OnCarDriveModeRestrictionInfo(0)
              
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::spi_tclOnCarDataService
 ***************************************************************************/
spi_tclOnCarDataService::~spi_tclOnCarDataService()
{
   ETG_TRACE_USR1(("~spi_tclOnCarDataService() entered "));
   vStopVehicleSpeedUpdate();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarDataService::bInitialise()
 ***************************************************************************/
t_Bool spi_tclOnCarDataService::bInitialise()
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::bInitialise() entered "));
   t_Bool bInitSuccess = false;

   spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
   if(NULL != poOnCarManager)
   {
       m_poOnCarCmdSensor = poOnCarManager->poGetSensorInstance();
   }// if (NULL != poOnCarManager)
   if (NULL != m_poOnCarCmdSensor)
   {
      bInitSuccess = true;
      vRegisterCallbacks();
   }// if (NULL != m_poOnCarCmdSensor)

   ETG_TRACE_USR2(("[DESC]:spi_tclOnCarDataService::bInitialise() left with result = %u ", bInitSuccess));
   return bInitSuccess;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclOnCarDataService::bUninitialise()
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::bUninitialise() entered "));
   m_poOnCarCmdSensor = NULL;
   return true;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarDataService::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclOnCarDataService::vRegisterCallbacks()
{
   ETG_TRACE_USR1((" spi_tclOnCarDataService::vRegisterCallbacks() Entered "));

   trOnCarSensorCallback rDayNightCbs;
   rDayNightCbs.fvSetDayNightMode = std::bind(
      &spi_tclOnCarDataService::vOnSetDayNightMode, this,
      std::placeholders::_1);
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vRegisterCallbacks(rDayNightCbs);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarDataService::vOnSetDayNightMode()
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vOnSetDayNightMode(tenVehicleConfiguration enVehicleConfig)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarDataService::vOnSetDayNightMode() entered "
         "with DayNight mode =%d ", enVehicleConfig));

   m_enDayNightMode = (e8_DAY_MODE == enVehicleConfig)?(e8_ONCAR_DAYNIGHT_MODE_DAY):(e8_ONCAR_DAYNIGHT_MODE_NIGHT);

   vReportDayNightMode();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarDataService::bUninitialise()
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vSelectDevice(const t_U32 cou32DevId,
         tenDeviceConnectionReq enConnReq)
{
    ETG_TRACE_USR1(("spi_tclOnCarDataService::vSelectDevice() entered "));
    if(e8DEVCONNREQ_DESELECT == enConnReq)
    {
        if(NULL != m_poOnCarCmdSensor && cou32DevId == m_rSelDevInfo.u32DeviceHandle)
        {
            //! Clear Selected device's info
            m_rSelDevInfo = corEmptyDeviceInfo;
            //! Stop timers for fixed frequency updates
            vStopVehicleSpeedUpdate();
            vSetDataSubscription(false);
            //Destroy Endpoints
            m_poOnCarCmdSensor->bUnInitializeSensorEndpoint();
        }//if(NULL != m_poOnCarCmdSensor && cou32DevId == m_rSelDevInfo.u32DeviceHandle)
    }//if(e8DEVCONNREQ_DESELECT == enConnReq)
    else if(e8DEVCONNREQ_SELECT == enConnReq)
    {
        //! Clear Selected device's info & update the newly selected device's handle.
        m_rSelDevInfo = corEmptyDeviceInfo;
        m_rSelDevInfo.u32DeviceHandle = cou32DevId;
        // Initialize Sensor Endpoint
        if ((NULL != m_poOnCarCmdSensor) &&
           (true == m_poOnCarCmdSensor->bInitializeSensorEndpoint(m_rOnCarDataServiceConfigData)))
        {
           //! Subscribe for interested data
           //vSetDataSubscription(true);

           //! Update current data values
           vReportDayNightMode();
           vReportDrivingStatus();

           //! Start timers for fixed frequency updates
           //vStartVehicleSpeedUpdate();

        }//if ((NULL != m_poCmdSensor) && ...)
     }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnSelectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
        const tenDeviceConnectionReq coenConnReq,
        const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR2(("[DESC]::vOnSelectDeviceResult() entered: u32DeviceHandle = %u ", u32DeviceHandle));
   t_Bool bIsSelectFailure = ((e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq));
   if ((bIsSelectFailure) && (NULL != m_poOnCarCmdSensor) && (u32DeviceHandle == m_rSelDevInfo.u32DeviceHandle))
   {
         //! Clear Selected device's info
         m_rSelDevInfo = corEmptyDeviceInfo;

         //! Stop timers for fixed frequency updates
         vStopVehicleSpeedUpdate();

         //! Unsubscribe for interested data
         vSetDataSubscription(false);

         m_poOnCarCmdSensor->bUnInitializeSensorEndpoint();
   }//if ((bIsSelectFailure) && ...)   
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnDeselectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR2(("[DESC]::spi_tclOnCarDataService() entered: u32DeviceHandle = %u ", u32DeviceHandle));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarDataService::vSetDataSubscription(...)
***************************************************************************/
t_Void spi_tclOnCarDataService::vSetDataSubscription(t_Bool bSubscriptionOn)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarDataService::vSetDataSubscription() entered: SubscriptionOn = %u ",
      ETG_ENUM(BOOL, bSubscriptionOn)));

   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8VEHICLE_DATA);
      if (m_rOnCarDataServiceConfigData.bParkBrakeData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8PARKBRAKE_DATA);
      }
   }//if (NULL != m_rDataServiceCb.fvSubscribeForData)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(t_U32...)
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vOnData(const trVehicleData& rfcoVehicleData, t_Bool bSolicited)
{
    ETG_TRACE_USR2(("spi_tclOnCarDataService::vOnData entered"));
    //! Store Vehicle speed info
    m_s16VehicleSpeed = rfcoVehicleData.s16Speed;
    //! Send data to selected phone on vehicle data change
    if (false == bSolicited)
    {
       vSetVehicleData(rfcoVehicleData.bParkBrakeActive,
               rfcoVehicleData.enVehMovState, rfcoVehicleData.enGearPosition);
    }//if (false == bSolicited)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(const trSensorData&...)
***************************************************************************/
t_Void spi_tclOnCarDataService::vOnData(const trSensorData& rfcorSensorData)
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vOnData entered"));
   if((true == m_rOnCarDataServiceConfigData.bLocDataAvailable) && 
      (false == m_rOnCarDataServiceConfigData.bDeadReckonedData) &&
      (0 != m_rSelDevInfo.u32DeviceHandle))
   {
       if(NULL != m_poOnCarCmdSensor)  
       {
           m_poOnCarCmdSensor -> vReportLocationData(rfcorSensorData);
       }
   }        
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarDataService::vSetSensorDataAvailablility(...)
***************************************************************************/
t_Void spi_tclOnCarDataService::vSetSensorDataAvailablility(
      const trDataServiceConfigData& rfrDataServiceConfigInfo)
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vSetSensorDataAvailablility() entered "));

   //! Store config data
   m_rOnCarDataServiceConfigData = rfrDataServiceConfigInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarDataService::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
      const t_U8 cou8DriveModeRestrictionInfo)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarDataService::vSetFeatureRestrictions() entered with"
         "Park mode Restriction = %d, Drive mode Restriction = %d ",
         cou8ParkModeRestrictionInfo, cou8DriveModeRestrictionInfo));

   SPI_INTENTIONALLY_UNUSED(cou8DriveModeRestrictionInfo);
   vReportDrivingStatus();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(...)
***************************************************************************/
t_Void spi_tclOnCarDataService::vOnData(const trGPSData& rfcorGpsData)
{
    SPI_INTENTIONALLY_UNUSED(rfcorGpsData);
    
    ETG_TRACE_USR1(("spi_tclOnCarDataService::vOnData entered"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::vReportDayNightMode()
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vReportDayNightMode()
{
   //! Send Day/Night mode data
   if ((0 != m_rSelDevInfo.u32DeviceHandle) && (NULL != m_poOnCarCmdSensor))
   {
       m_poOnCarCmdSensor->vReportDayNightMode(m_enDayNightMode);
   }//if ((0 != m_rSelDevInfo.u32DeviceHandle))
}

/***************************************************************************
** FUNCTION: t_Void spi_tclOnCarDataService::vSetVehicleData()
***************************************************************************/
t_Void spi_tclOnCarDataService::vSetVehicleData(t_Bool bParkBrakeActive,
      tenVehicleMovementState enVehMovState, tenGearPosition enGearPosition)
{
   //! Send changed ParkBrake and/or restriction data
   if ((bParkBrakeActive != m_bParkBrakeActive) || (enVehMovState != m_enVehMovState)) /*if states have changed*/
   {
      if ((bParkBrakeActive != m_bParkBrakeActive) &&
               (true == m_rOnCarDataServiceConfigData.bParkBrakeData))
      {
         //! Store latest state
         m_bParkBrakeActive = bParkBrakeActive;
      }
      //! Store latest state
      m_enVehMovState = enVehMovState;
      vReportDrivingStatus();
      if(enGearPosition != m_enGearPosition)
      {
          m_enGearPosition = enGearPosition;
      }
   }//if ((bParkBrakeActive != m_bParkBrakeActive) || (enVehMovState != m_enVehMovState))
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::vReportVehicleSpeedInfo()
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vReportVehicleSpeedInfo()
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vReportVehicleSpeedInfo() entered ")); 
   //! Send Vehicle speed data
   if ((NULL != m_poOnCarCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      // convert speed cm/s to m/s and multiply by E3 as per AAUTO spec
      // (m_s16VehicleSpeed) * 10) == (m_s16VehicleSpeed / 100) * 1000)
      t_S32 s32SpeedE3 = ((m_s16VehicleSpeed) * 10);
      s32SpeedE3 = (e8GEAR_POS_REVERSE_GEAR == m_enGearPosition) ? ((-1)*(abs(s32SpeedE3))):(s32SpeedE3);
      m_poOnCarCmdSensor->vReportSpeedData(s32SpeedE3);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::vStartVehicleSpeedUpdate
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vStartVehicleSpeedUpdate()
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vStartVehicleSpeedUpdate() entered "));

   //! Start timer to send VehicleSpeed data at 10Hz rate
   Timer* poTimer = Timer::getInstance();
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vStartVehicleSpeedUpdate()  timer value is %d"
   ,m_rOnCarDataServiceConfigData.u32SpeedTimerIntervalInMs));
   if ((false == sbVehSpeedTimerRunning) && (NULL != poTimer))
   {
      poTimer->StartTimer(srVehSpeedTimerID,m_rOnCarDataServiceConfigData.u32SpeedTimerIntervalInMs,
              m_rOnCarDataServiceConfigData.u32SpeedTimerIntervalInMs, this,
            &spi_tclOnCarDataService::bVehicleSpeedTimerCb, NULL);
      sbVehSpeedTimerRunning = true;

      ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer started"));

   }//if ((false == sbVehSpeedTimerRunning) &&...)
   else if (true == sbVehSpeedTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer already running"));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::vStopVehicleSpeedUpdate
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vStopVehicleSpeedUpdate()
{
   ETG_TRACE_USR1(("spi_tclOnCarDataService::vStopVehicleSpeedUpdate() entered "));

   //! Stop timer
   Timer* poTimer = Timer::getInstance();
   if ((true == sbVehSpeedTimerRunning) && (NULL != poTimer))
   {
      poTimer->CancelTimer(srVehSpeedTimerID);
      sbVehSpeedTimerRunning = false;

      ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer stopped "));

   }//if ((true == sbVehSpeedTimerRunning) && ...)
   else if (false == sbVehSpeedTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer not running"));
   }
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::bVehicleSpeedTimerCb
 ***************************************************************************/
t_Bool spi_tclOnCarDataService::bVehicleSpeedTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   //ETG_TRACE_USR1(("spi_tclOnCarDataService::bVehicleSpeedTimerCb entered "));

   spi_tclOnCarDataService* poOnCarDataService = static_cast<spi_tclOnCarDataService*> (pvObject);
   if (NULL != poOnCarDataService)
   {
      poOnCarDataService->vReportVehicleSpeedInfo();
   }

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarDataService::vReportDrivingStatus()
 ***************************************************************************/
t_Void spi_tclOnCarDataService::vReportDrivingStatus()
{
   //! Send park/Drive mode data
   if ((0 != m_rSelDevInfo.u32DeviceHandle) && (NULL != m_poOnCarCmdSensor))
   {
       tenOnCarDrivingState enOnCarDrivingState =
            ((m_bParkBrakeActive) || (e8VEHICLE_MOVEMENT_STATE_PARKED == m_enVehMovState)) ?
            (e8_ONCAR_DRIVING_STATE_PARKED) : (e8_ONCAR_DRIVING_STATE_MOVING);
       m_poOnCarCmdSensor->vReportVehicleDrivingState(enOnCarDrivingState);
       if(enOnCarDrivingState == e8_ONCAR_DRIVING_STATE_MOVING)
       {
           t_S32 s32OnCarRestrictionInfo = m_s32OnCarDriveModeRestrictionInfo;
           if(NULL != m_poDataServiceSettings)
           {
               s32OnCarRestrictionInfo = static_cast<t_S32>(m_poDataServiceSettings->u8GetRestrictionsInfo(e8DEV_TYPE_ONCAR,true));
               m_poOnCarCmdSensor->vSetDriveModeRestriction(s32OnCarRestrictionInfo);
           }
      }
   }//if ((0 != m_rSelDevInfo.u32DeviceHandle) && (NULL != m_poOnCarCmdSensor))
}

//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
