/*!
 *******************************************************************************
 * \file             spi_tclOnCarDataService.h
 * \brief            OnCar Data Service class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Data Service class implements Data Service Info Management for
 OnCar capable devices. This class must be derived from base Data Service class.
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 26.04.2018  | Rishav Sardar         | Initial version


 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLONCARDATASERVICE_H_
#define SPI_TCLONCARDATASERVICE_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "SPI_OnCarTypes.h"
#include "spi_tclDataServiceDevBase.h"
#include "spi_tclDataServiceTypes.h"
//#include "spi_tclConfigReaderIntf.h"
/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
struct trOnCarSelectedDeviceInfo
{
   t_U32 u32DeviceHandle;

   trOnCarSelectedDeviceInfo():
      u32DeviceHandle(0)
   {
   }
};

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/* Forward declarations */
class spi_tclOnCarCmdSensor;

/*!
 * \class spi_tclOnCarDataService
 * \brief OnCar Connection class implements Data Service Info Management for
 *        OnCar capable devices. This class must be derived from base
 *        Data Service class.
 */

class spi_tclOnCarDataService : public spi_tclDataServiceDevBase  //! Base Connection Class
{
public:
   /***************************************************************************
    *********************************PUBLIC*************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::spi_tclOnCarDataService(const trDataServiceCb...))
    ***************************************************************************/
   /*!
    * \fn     spi_tclOnCarDataService(const trDataServiceCb& rfcorDataServiceCb)
    * \brief  Parameterised Constructor
    * \param  rfcorDataServiceCb: [IN] Structure containing callbacks to
    *            DataService Manager.
    * \sa     ~spi_tclOnCarDataService()
    **************************************************************************/
    spi_tclOnCarDataService(const trDataServiceCb& rfcorDataServiceCb);

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::~spi_tclOnCarDataService
    ***************************************************************************/
   /*!
    * \fn     ~spi_tclOnCarDataService()
    * \brief  Destructor
    * \sa     spi_tclOnCarDataService()
    **************************************************************************/
   virtual ~spi_tclOnCarDataService();


   /***** Start of Methods overridden from spi_tclDataServiceDevBase *********/

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarDataService::bInitialise();
   ***************************************************************************/
   /*!
   * \fn      bInitialise()
   * \brief   Method to initialises the service handler. (Performs initialisations which
   *          are not device specific.)
   *          Mandatory interface to be implemented.
   * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
   * \sa      bUninitialise()
   ***************************************************************************/
   virtual t_Bool bInitialise();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarDataService::bUninitialise();
   ***************************************************************************/
   /*!
   * \fn      bUninitialise()
   * \brief   Method to uninitialise the service handler.
   *          Mandatory interface to be implemented.
   * \retval  t_Bool: TRUE - If ServiceHandler is uninitialised successfully, else FALSE.
   * \sa      bInitialise()
   ***************************************************************************/
   virtual t_Bool bUninitialise();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarDataService::vSelectDevice(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vSelectDevice
    * \brief   Called when a device is selection request is received
    * \param   [IN] cou32DevId: Unique handle of selected device
    * \param   [IN] enConnReq: Indicated the category of the device
    * \retval  None
    **************************************************************************/
   t_Void vSelectDevice(const t_U32 cou32DevId,
            tenDeviceConnectionReq enConnReq);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnSelectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnSelectDeviceResult(t_U32 u32DeviceHandle, tenDeviceConnectionReq coenConnReq,
    *           tenResponseCode coenRespCode)
    * \brief   Called when a device is selected.
    *          Mandatory interface to be implemented.
    * \param   u32DeviceHandle: [IN] Unique handle of selected device
    * \param   coenConnReq    : [IN] Identifies the Connection Request.
    * \param   coenRespCode   : [IN] Response code. Success/Failure
    * \retval  None
    * \sa      vOnDeselectDevice()
    **************************************************************************/
   virtual t_Void vOnSelectDeviceResult(t_U32 u32DeviceHandle, tenDeviceConnectionReq coenConnReq,
           tenResponseCode coenRespCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarDataService::vSetSensorDataAvailablility(t_Bool...)
   ***************************************************************************/
   /*!
   * \fn      vSetSensorDataAvailablility(t_Bool bLocDataAvailable)
   * \brief   Interface to set the availability of LocationData
   * \param   rfrDataServiceConfigInfo: the structure consists of values TRUE
   *           for location data, dead reckoning data, environment data,
   *           gear status, accelerometer data and gyroscope data if available,
   *           FALSE if not.
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetSensorDataAvailablility(
         const trDataServiceConfigData& rfrDataServiceConfigInfo);
           
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnDeselectDeviceResult(t_U32...)
    ***************************************************************************/
   /*!
    * \fn      vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
    * \brief   Called when currently selected device is de-selected.
    *          Mandatory interface to be implemented.
    * \param   [IN] u32DeviceHandle: Unique handle of selected device
    * \retval  None
    * \sa      vOnSelectDevice()
    **************************************************************************/
   virtual t_Void vOnDeselectDeviceResult(t_U32 u32DeviceHandle);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(
                 const trVehicleData& rfcoVehicleData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trVehicleData& rfcoVehicleData)
   * \brief   Method to receive Vehicle data.
   *          Optional interface to be implemented.
   * \param   rfcoVehicleData: [IN] Vehicle data
   * \retval  None
   ***************************************************************************/
   t_Void vOnData(const trVehicleData& rfcoVehicleData, t_Bool bSolicited);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(const trSensorData&...)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trSensorData& rfcorSensorData)
   * \brief   Method to receive Sensor data.
   *          Optional interface to be implemented.
   * \param   rfcorSensorData: [IN] Sensor data
   * \retval  None
   ***************************************************************************/
   t_Void vOnData(const trSensorData& rfcorSensorData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarDataService::vOnData(const trGPSData& rfcorGpsData)
   ***************************************************************************/
   /*!
   * \fn      vOnData(const trGPSData& rfcorGpsData)
   * \brief   Method to receive GPS data.
   *          Optional interface to be implemented.
   * \param   rfcorGpsData: [IN] GPS data
   * \retval  None
   ***************************************************************************/
   t_Void vOnData(const trGPSData& rfcorGpsData);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarDataService::vSetFeatureRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
    *          const t_U8 cou8ParkModeRestrictionInfo,
    *          const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   cou8ParkModeRestrictionInfo : Park mode restriction
    *          cou8DriveModeRestrictionInfo : Drive mode restriction
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
         const t_U8 cou8DriveModeRestrictionInfo);


private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:t_Void spi_tclOnCarDataService::vOnSetDayNightMode(...)
    ***************************************************************************/
   /*!
    * \fn      vOnSetDayNightMode(tenVehicleConfiguration enVehicleConfig)
    * \brief   To day/night data
    * \param   [IN] enVehicleConfig: contains day/night mode information
    * \retval  None
    **************************************************************************/
   t_Void vOnSetDayNightMode(tenVehicleConfiguration enVehicleConfig);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclOnCarDataService::vSetVehicleData()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetVehicleData()
   * \brief  Interface to set the Vehicle data
   * \param  [IN] bParkBrakeActive : Identifies Park brake status
   * \param  [IN] enVehMovState    : Identifies Vehicle Movement status
   * \param  [IN] enGearPosition   : Identifies Vehicle gear status
   **************************************************************************/
   t_Void vSetVehicleData(t_Bool bParkBrakeActive, tenVehicleMovementState enVehMovState,
            tenGearPosition enGearPosition);

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::vReportVehicleSpeedInfo
    ***************************************************************************/
   /*!
   * \fn      vReportVehicleSpeedInfo()
   * \brief   Method to send Vehicle Speed Info.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportVehicleSpeedInfo();

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::vReportDayNightMode()
    ***************************************************************************/
   /*!
   * \fn      vReportDayNightMode()
   * \brief   Method to send Day/Night mode data.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportDayNightMode();

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::vReportDrivingStatus()
    ***************************************************************************/
   /*!
   * \fn      vReportDrivingStatus()
   * \brief   Method to send Park/Drive mode data.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vReportDrivingStatus();

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::vStartVehicleSpeedUpdate
    ***************************************************************************/
   /*!
   * \fn      vStartVehicleSpeedUpdate()
   * \brief   Method to start timer to send Vehicle Speed info
   * \retval  t_Void
   ***************************************************************************/
   t_Void vStartVehicleSpeedUpdate();

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarDataService::vStopVehicleSpeedUpdate
    ***************************************************************************/
   /*!
   * \fn      vStopVehicleSpeedUpdate()
   * \brief   Method to stop timer to send Vehicle Speed info
   * \retval  t_Void
   ***************************************************************************/
   t_Void vStopVehicleSpeedUpdate();

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarDataService::bVehicleSpeedTimerCb
   ***************************************************************************/
   /*!
   * \fn     bVehicleSpeedTimerCb
   * \brief  called on expiry of VehicleSpeed timer
   * \param  rTimerID: ID of the timer which has expired
   * \param  pvObject: pointer to object passed while starting the timer
   * \param  pvUserData: data passed during start of the timer
   **************************************************************************/
   static t_Bool bVehicleSpeedTimerCb(timer_t rTimerID, t_Void *pvObject,
           const t_Void *pvUserData);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclOnCarDataService::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterCallbacks()
   * \brief   Method to Register callback
   * \param   None
   * \retval  None
   ***************************************************************************/
   t_Void vRegisterCallbacks();

   /***************************************************************************
    ** FUNCTION:t_Void spi_tclOnCarDataService::vSetDataSubscription(...)
    ***************************************************************************/
   /*!
    * \fn      vSetDataSubscription(t_Bool bSubscriptionOn)
    * \brief   Used to subscribe for sensors data
    * \param   [IN] bSubscriptionOn: Indicates if data should be subscribed/unsubscribed.
    * \retval  None
    **************************************************************************/
   t_Void vSetDataSubscription(t_Bool bSubscriptionOn);

   /***************************************************************************
   ** Data Members
   ***************************************************************************/

   /***************************************************************************
   ** Pointer to OnCar Cmd Sensor
   ***************************************************************************/
   spi_tclOnCarCmdSensor*        m_poOnCarCmdSensor;

   /***************************************************************************
    ** Selected device's information
    ***************************************************************************/
   trOnCarSelectedDeviceInfo     m_rSelDevInfo;

   /***************************************************************************
    ** DataService callbacks structure
    ***************************************************************************/
   trDataServiceCb            m_rDataServiceCb;

   /***************************************************************************
    ** Day Night mode info
    ***************************************************************************/
   tenOnCarDayNightMode      m_enDayNightMode;

   /***************************************************************************
    ** Driving Status info
    ***************************************************************************/
   tenOnCarDrivingState      m_enDrivingState;

   /***************************************************************************
    ** Vehicle Movement status
    ***************************************************************************/
   tenVehicleMovementState    m_enVehMovState;

   /***************************************************************************
    ** Vehicle gear status
    ***************************************************************************/
   tenGearPosition m_enGearPosition;

   /***************************************************************************
    ** Vehicle park brake status
    ***************************************************************************/
   t_Bool                     m_bParkBrakeActive;

   /***************************************************************************
   ** Vehicle Speed info
   ***************************************************************************/
   t_S16    m_s16VehicleSpeed;

   /***************************************************************************
   ** Drive Mode Info
   ***************************************************************************/
   t_S32     m_s32OnCarDriveModeRestrictionInfo;

   /****************************************************************************
   ** Config info
   *****************************************************************************/
   trDataServiceConfigData m_rOnCarDataServiceConfigData;


};
#endif // SPI_TCLONCARDATASERVICE_H_
