/*!
 *******************************************************************************
 * \file         spi_tclOnCarConnMngr.cpp
 * \brief        OnCar Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    
 AUTHOR:         jun5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 01.02.2018  | Unmukt Jain           | Initial Version
 \endverbatim
 07.12.2018  | Ashwini Savadi        | Session error Handling adaptation
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclOnCarConnMngr.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "spi_tclMediator.h"
#include <algorithm>
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclOnCarConnMngr.cpp.trc.h"
#endif
#endif

static const t_U32 scou32OnCarSessionStatusTimeout_ms = 12000;

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::spi_tclOnCarConnMngr
 ***************************************************************************/
spi_tclOnCarConnMngr::spi_tclOnCarConnMngr():
m_u32SelDevID(0),
m_rSessionStatusTimerID(0)
{
   ETG_TRACE_USR1((" spi_tclOnCarConnMngr::spi_tclOnCarConnMngr()"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::~spi_tclOnCarConnMngr
 ***************************************************************************/
spi_tclOnCarConnMngr::~spi_tclOnCarConnMngr()
{
   ETG_TRACE_USR1((" spi_tclOnCarConnMngr::~spi_tclOnCarConnMngr()"));
   m_u32SelDevID = 0;
   m_rSessionStatusTimerID = 0;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclOnCarConnMngr::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclOnCarConnMngr::vRegisterCallbacks(trConnCallbacks &rfrConnCallbacks)
{
   m_rConnCbs = rfrConnCallbacks;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclOnCarConnMngr::bInitializeConnection
 **************************************************************************/
t_Bool spi_tclOnCarConnMngr::bInitializeConnection()
{
   ETG_TRACE_USR1(("spi_tclOnCarConnMngr::bInitializeConnection"));

   t_Bool bRet = false;

   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   if(NULL!= poOnCarManager)
   {
       bRet = poOnCarManager->bRegisterObject((spi_tclOnCarRespSession*)this);
   }
   return bRet;
}

/**************************************************************************
 ** FUNCTION: t_Void spi_tclOnCarConnMngr::vOnLoadSettings
 *************************************************************************/
t_Void spi_tclOnCarConnMngr::vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType)
{
    SPI_INTENTIONALLY_UNUSED(enCertificateType);
    
    ETG_TRACE_USR1(("spi_tclOnCarConnMngr::vOnLoadSettings entered "));
    m_rHeadUnitInfo = rfrheadUnitInfo;
}

/**************************************************************************
 ** FUNCTION: t_Void spi_tclConnection::vUnInitializeConnection
 **************************************************************************
t_Void spi_tclMySPINConnMngr::vUnInitializeConnection()
{
   ETG_TRACE_USR1((" spi_tclMySPINConnMngr::vUnInitializeConnection"));

   for (t_U8 u8Ind = 0; u8Ind < NUM_MYSPIN_CONN_HNDLRS; u8Ind++)
   {
      if (NULL != m_apoMySPINConnHndlrs[u8Ind])
      {
         m_apoMySPINConnHndlrs[u8Ind]->vUnInitializeConnection();
      }
      RELEASE_MEM(m_apoMySPINConnHndlrs[u8Ind]);
   }

}

**************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINConnMngr::vOnSaveSettings
 **************************************************************************
t_Void spi_tclMySPINConnMngr::vOnSaveSettings()
{

}

**************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINConnMngr::bSetDevProjUsage
 **************************************************************************
t_Bool spi_tclMySPINConnMngr::bSetDevProjUsage(tenEnabledInfo enServiceStatus)
{
   SPI_INTENTIONALLY_UNUSED(enServiceStatus)
   ETG_TRACE_USR1((" spi_tclMySPINConnMngr::bSetDevProjUsage"));

   //@todo - it' un used for now
   return true;
}*/

/**************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarConnMngr::vOnSelectDevice()
 **************************************************************************/
t_Void spi_tclOnCarConnMngr::vOnSelectDevice(const t_U32 cou32DevID, tenDeviceConnectionType enDevConnType,
         tenDeviceConnectionReq enDevSelectType, tenEnabledInfo enDAPUsage, tenEnabledInfo enCDBUsage,
         tenSelectReason enSelectionReason, const trUserContext corfUsrCntxt, tenDeviceType enDeviceType)
{
   SPI_INTENTIONALLY_UNUSED(enDevConnType);
   SPI_INTENTIONALLY_UNUSED(enDAPUsage);
   SPI_INTENTIONALLY_UNUSED(enCDBUsage);
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);
   SPI_INTENTIONALLY_UNUSED(corfUsrCntxt);

   ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::vOnSelectDevice entered for device(0x%x), Select Type-%d, DeviceType-%d",
                    cou32DevID, ETG_ENUM(CONNECTION_REQ,enDevSelectType), ETG_ENUM(SELECT_REASON, enDeviceType)));

   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdSession *poOnCarCmdSession = NULL;
   if(NULL != poOnCarManager)
   {
       poOnCarCmdSession = poOnCarManager->poGetSessionInstance();
   }

   //Fetch the serial number of the device
   spi_tclDeviceIDDataIntf oDeviceIDDataIntf;
   t_String szDeviceSerial;
   oDeviceIDDataIntf.vGetSerialNumber(szDeviceSerial,cou32DevID);

   if(e8DEVCONNREQ_SELECT == enDevSelectType && e8_ANDROID_DEVICE == enDeviceType)
   {
       m_u32SelDevID = cou32DevID;
       if((NULL != poOnCarCmdSession) && (NULL != m_poConnSettings))
       {
           OnCarConfig config;
           tvecVideoConfigList vecVideoConfigList;
           trVideoConfigData rVideoConfigData;
           t_U16 u16VehicleType = 0;
           t_U8 u8RegionType = 0;
           t_U8 u8VendorCode = 0;
           m_poConnSettings->bGetSysVehicleInfo(u8RegionType, u8VendorCode, u16VehicleType);
           m_poConnSettings->vGetVideoConfigData(e8DEV_TYPE_ONCAR,vecVideoConfigList);
           m_poConnSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);
           config.height = rVideoConfigData.u32Screen_Height;
           config.width = rVideoConfigData.u32Screen_Width;
           config.HUmanufacturer = m_rHeadUnitInfo.szHUManufacturer;
         
           config.HUvendor = "RENAULT";//m_rHeadUnitInfo.szVehicleManufacturer;
           config.VehicleCode = u16VehicleType;
           config.RegionCode = u8RegionType;
           //std::transform(m_rHeadUnitInfo.szVehicleManufacturer.begin(), m_rHeadUnitInfo.szVehicleManufacturer.end(), config.HUvendor.begin(), ::toupper);
           config.HUvendor_code = u8VendorCode;
           ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::Values :height(%u), width(%u), HUvendor_code(%u)",
                   config.height, config.width,config.HUvendor_code));
           ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::Values :HUmanufacturer(%s),",config.HUmanufacturer.c_str()));
           ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::Values :HUvendor(%s),",config.HUvendor.c_str()));
           t_Bool bRet = poOnCarCmdSession->bInitializeSession(config);

           bRet = bRet && poOnCarCmdSession->bStartTransport(szDeviceSerial);

           if(true == bRet)
           {
               //start the session status timer
               vStartOnCarSessionStatusTimer();
           }
           else if((false == bRet) && (NULL != m_rConnCbs.fvSelectDeviceResult))
           {
               (m_rConnCbs.fvSelectDeviceResult)(e8INTERNAL_ERROR);
           }
       }
   }
   else if(e8DEVCONNREQ_DESELECT == enDevSelectType && e8_ANDROID_DEVICE == enDeviceType)
   {
       if(NULL != poOnCarCmdSession)
       {
           poOnCarCmdSession->bStopTransport(szDeviceSerial);
           if(NULL != m_rConnCbs.fvSelectDeviceResult)
           {
               (m_rConnCbs.fvSelectDeviceResult)(e8NO_ERRORS);
           }
       }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::vStartOnCarSessionStatusTimer
 ***************************************************************************/
t_Void spi_tclOnCarConnMngr::vStartOnCarSessionStatusTimer()
{
    ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::vStartOnCarSessionStatusTimer entered"));
    Timer* poTimer = Timer::getInstance();
    if(NULL != poTimer)
    {
        timer_t rTimerID;
        //! Start timer and wait for the OnCar Session message
        poTimer->StartTimer(rTimerID, scou32OnCarSessionStatusTimeout_ms, 0, this,
               &spi_tclOnCarConnMngr::bOnCarSessionStatusTimerCb, NULL);
        m_rSessionStatusTimerID = rTimerID;
    }
}
/***************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::vStopOnCarSessionStatusTimer
 ***************************************************************************/
t_Void spi_tclOnCarConnMngr::vStopOnCarSessionStatusTimer()
{
    ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::vStopOnCarSessionStatusTimer entered"));
    Timer* poTimer = Timer::getInstance();
    if ((0 != m_rSessionStatusTimerID) && (NULL != poTimer))
    {
        poTimer->CancelTimer(m_rSessionStatusTimerID);
        m_rSessionStatusTimerID = 0;
    }
}
/***************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::bOnCarSessionStatusTimerCb
 ***************************************************************************/
t_Bool spi_tclOnCarConnMngr::bOnCarSessionStatusTimerCb(timer_t rTimerID, t_Void *pvObject,
                                                        const t_Void *pvUserData)
{
    SPI_INTENTIONALLY_UNUSED(rTimerID);
    SPI_INTENTIONALLY_UNUSED(pvUserData);
    
    ETG_TRACE_USR2(("[FUNC]spi_tclOnCarConnMngr::bOnCarSessionStatusTimerCb entered")); 
    if(pvObject)
    {
        spi_tclOnCarConnMngr* poOnCarConnMngr = static_cast<spi_tclOnCarConnMngr*>(pvObject);
        if (NULL != poOnCarConnMngr)
        {
            poOnCarConnMngr->vStopOnCarSessionStatusTimer();
            if(NULL != poOnCarConnMngr->m_rConnCbs.fvSelectDeviceResult)
            {
                poOnCarConnMngr->m_rConnCbs.fvSelectDeviceResult(e8INTERNAL_ERROR);
            }
        }
    }
    else{
        ETG_TRACE_ERR(("pvObject is NULL"));
    }
    return true;
}
/**************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::vOnSelectDeviceResult()
 **************************************************************************/
t_Void spi_tclOnCarConnMngr::vOnSelectDeviceResult(const t_U32 cou32DevID,
         const tenDeviceConnectionReq enDevSelectType, const tenResponseCode coenRespCode, tenDeviceCategory enDevCat,
         tenSelectReason enSelectionReason, tenDeviceType enDeviceType)
{
   ETG_TRACE_USR2(("spi_tclOnCarConnMngr::vOnSelectDeviceResult entered for 0x%x. Select Type-%d, Response Code -%d", cou32DevID, ETG_ENUM(CONNECTION_REQ,
            enDevSelectType), ETG_ENUM(RESPONSE_CODE, coenRespCode)));

   SPI_INTENTIONALLY_UNUSED(cou32DevID);
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);

   //Fetch the serial number of the device
   spi_tclDeviceIDDataIntf oDeviceIDDataIntf;
   t_String szDeviceSerial;
   oDeviceIDDataIntf.vGetSerialNumber(szDeviceSerial,cou32DevID);
   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdSession *poOnCarCmdSession = NULL;
   if(NULL != poOnCarManager)
   {
       poOnCarCmdSession = poOnCarManager->poGetSessionInstance();
   }

   if(poOnCarCmdSession)
   {
       if(((e8DEVCONNREQ_SELECT == enDevSelectType) && (e8FAILURE == coenRespCode)) && e8_ANDROID_DEVICE == enDeviceType )
       {
           ETG_TRACE_USR2(("Selection failed, trigger destroy connection!!"));
           poOnCarCmdSession->bStopTransport(szDeviceSerial);
       }
       if( (
         ( (e8DEVCONNREQ_DESELECT == enDevSelectType) || (e8DEVCONNREQ_SELECT == enDevSelectType && e8FAILURE == coenRespCode))
         && e8_ANDROID_DEVICE == enDeviceType ))
       {
            m_u32SelDevID = 0;
            poOnCarCmdSession->vUnInitializeSession();
       }
   }
   else
   {
       ETG_TRACE_ERR(("poOnCarCmdSession is NULL"));
    }
}

/**************************************************************************
 ** FUNCTION:  spi_tclOnCarConnMngr::vPostOnCarSessionErrorMsg()
 **************************************************************************/
t_Void spi_tclOnCarConnMngr::vPostOnCarSessionErrorMsg(const tenOnCarSinkError enOnCarSinkError)
{
    ETG_TRACE_USR1(("[FUNC]::spi_tclOnCarConnMngr::vPostOnCarSessionErrorMsg entered"));

    if(0 != m_rSessionStatusTimerID) //process session message here only if selection has started
    {
        vStopOnCarSessionStatusTimer();
        switch(enOnCarSinkError)
        {
            case e32_ONCAR_SINK_APP_NOT_INSTALLED_ERROR :
            {
                ETG_TRACE_USR2(("e32_ONCAR_SINK_APP_NOT_INSTALLED_ERROR occured"));
                (m_rConnCbs.fvSelectDeviceResult)(e8APP_NOT_INSTALLED);
            }
            break;
            case e32_ONCAR_SINK_PHONE_UNSUPPORTED_ERROR :
            {
                ETG_TRACE_USR2(("e32_ONCAR_SINK_PHONE_UNSUPPORTED_ERROR occured"));
                (m_rConnCbs.fvSelectDeviceResult)(e8PHONE_UNSUPPORTED);
            }
            break;
            case e32_ONCAR_SINK_ERROR_TIMEOUT:
            {
                ETG_TRACE_USR2(("e32_ONCAR_SINK_ERROR_TIMEOUT occured"));
                (m_rConnCbs.fvSelectDeviceResult)(e8CONNECT_TIME_OUT);
            }
            break;
            default:
            {
                ETG_TRACE_USR2(("sink error is : %d", enOnCarSinkError));
                (m_rConnCbs.fvSelectDeviceResult)(e8INTERNAL_ERROR);
            }
            break;
        }
    }
    else
    {
        ETG_TRACE_USR2(("spi_tclOnCarConnMngr::vPostOnCarSessionErrorMsg triggering deselection because of error!!"));
        spi_tclMediator *poMediator = spi_tclMediator::getInstance();
        if(NULL != poMediator)
        {
            poMediator->vPostAutoDeviceSelection(m_u32SelDevID, e8DEVCONNREQ_DESELECT);
        }
   }
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclOnCarConnMngr::vPostOnCarSessionStateMsg(...
 ***************************************************************************/
t_Void spi_tclOnCarConnMngr::vPostOnCarSessionStateMsg(const tenOnCarSessionState enOnCarSessionState)
        {
   ETG_TRACE_USR1(("spi_tclOnCarConnMngr::vPostOnCarSessionStateMsg session state=%d",enOnCarSessionState));
   if(0 != m_rSessionStatusTimerID) //process session message here only if selection has started
   {
       vStopOnCarSessionStatusTimer();
      if(e8_ONCAR_SESSION_STATE_START == enOnCarSessionState)
      {
          ETG_TRACE_USR2(("e8_ONCAR_SESSION_STATE_START occured"));
          (m_rConnCbs.fvSelectDeviceResult)(e8NO_ERRORS);
      }
      else if(e8_ONCAR_SESSION_STATE_STOP == enOnCarSessionState )
      {
          ETG_TRACE_USR2(("e8_ONCAR_SESSION_STATE_STOP occured"));
         (m_rConnCbs.fvSelectDeviceResult)(e8INTERNAL_ERROR);
      }
      else if(e8_ONCAR_SESSION_STATE_UNKNOWN == enOnCarSessionState)
      {
          ETG_TRACE_USR2(("e8_ONCAR_SESSION_STATE_UNKNOWN occured"));
          (m_rConnCbs.fvSelectDeviceResult)(e8UNKNOWN_ERROR);
        }
    }
}

/**************************************************************************
 ** FUNCTION:  t_Bool  spi_tclMySPINConnMngr::bValidateClient()
 **************************************************************************
t_Bool spi_tclMySPINConnMngr::bValidateConnHndlr(t_U8 u8Index)
{
   //Assert if the Index is greater than the array size
   SPI_NORMAL_ASSERT( u8Index == MYSPIN_CONN_HNDLRS);

   return ((u8Index < MYSPIN_CONN_HNDLRS) && (NULL != m_apoMySPINConnHndlrs[u8Index]));
}

**************************************************************************
 ** FUNCTION:  spi_tclMySPINConnMngr::vSetDeviceSwitchInfo
 **************************************************************************
t_Void spi_tclMySPINConnMngr::vSetDeviceSwitchInfo(trDeviceSwitchInfo &rfrDeviceSwitchInfo)
{
   ETG_TRACE_USR1(("MySPIN Device Switch Info set to VendorID = 0x%x ProductID = 0x%x, SerialNumber = %s", rfrDeviceSwitchInfo.u32VendorID, rfrDeviceSwitchInfo.u32ProductID, rfrDeviceSwitchInfo.szSerialNumber.c_str()));

   m_rDeviceSwitchInfo = rfrDeviceSwitchInfo;

   //Set the switch progress info for IAP devices
   if (true == bValidateConnHndlr(e8DEVTYPE_IAP))
   {
      m_apoMySPINConnHndlrs[e8DEVTYPE_IAP]->vSetDeviceSelectionState(m_rDeviceSwitchInfo.bSwitchInProgress);
   }
}

**************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINConnMngr::vSetSelectErrorCb()
 **************************************************************************
t_Void spi_tclMySPINConnMngr::vSetSelectErrorCb(t_U32 u32DevID, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR2(("[DESC]::Error in Device 0x%x selection. ErrorCode-%d", u32DevID, ETG_ENUM(ERROR_CODE, enErrorCode)));
   // Unused for now.
}

**************************************************************************
 ** FUNCTION:
 **************************************************************************
t_Void spi_tclMySPINConnMngr::vSetRoleSwitchRequestedInfo(const t_U32 cou32DeviceHandle)
{
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle)
}
*/
