/*!
 *******************************************************************************
 * \file         spi_tclOnCarBluetooth.h
 * \brief        OnCar Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Bluetooth handler class for SPI
 AUTHOR:         jun5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 02.02.2018  | jun5kor               | Initial Version
 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLONCARBLUETOOTH_H
#define _SPI_TCLONCARBLUETOOTH_H

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "SPI_OnCarTypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclBluetoothDevBase.h"
#include "spi_tclOnCarRespBluetooth.h"
#include "spi_tclExtRespBluetooth.h"
#include "Lock.h"

class spi_tclExtCmdBluetoothIntf;

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
 
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/* Forward Declarations. */
class spi_tclBluetoothIntf;
class spi_tclBluetoothPolicyBase;
class spi_tclOnCarCmdBluetooth;

/*!
 * \class spi_tclOnCarBluetooth
 * \brief This is the OnCar Bluetooth class that handles the Bluetooth
 *        connection logic during a OnCar device session
 */
class spi_tclOnCarBluetooth: public spi_tclBluetoothDevBase, public spi_tclOnCarRespBluetooth, public spi_tclExtRespBluetooth
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclOnCarBluetooth::spi_tclOnCarBluetooth(...)
       ***************************************************************************/
      /*!
       * \fn      spi_tclOnCarBluetooth(spi_tclBluetoothIntf* poBTInterface)
       * \brief   Parameterized Constructor
       * \param   [IN] poBTInterface: Pointer to Bluetooth manager interface
       ***************************************************************************/
       spi_tclOnCarBluetooth(spi_tclBluetoothIntf* poBTInterface);

      /***************************************************************************
       ** FUNCTION:  spi_tclOnCarBluetooth::~spi_tclOnCarBluetooth();
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclOnCarBluetooth()
       * \brief   Virtual Destructor
       ***************************************************************************/
       virtual ~spi_tclOnCarBluetooth();

      /*********Start of functions overridden from spi_tclBluetoothDevBase*******/
   
      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclOnCarBluetooth::bInitialise();
      ***************************************************************************/
      /*!
      * \fn      bInitialise()
      * \brief   Method to initialise the service handler. (Performs initialisations which
      *          are not device specific.)
      *          Optional interface to be implemented.
      * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
      * \sa      bUninitialise()
      ***************************************************************************/
      t_Bool bInitialise();
   
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceRequest(
      ***************************************************************************/
      /*!
      * \fn      vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
 
      * \brief   Called when SelectDevice request is received by SPI.
      *          Mandatory interface to be implemented.
      * \param   [IN] corfrSelectReq: Device Selection Info
      * \retval  None
      **************************************************************************/
      t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceResponse(
      ***************************************************************************/
      /*!
      * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
      *             tenResponseCode enRespCode)
      * \brief   Called when SelectDevice operation is complete, with the result
      *          of the operation.
      *          Mandatory interface to be implemented.
      * \param   [IN] corfrSelectReq: Device Selection Info
      * \param   [IN] enRespCode: Response code enumeration
      * \retval  None
      **************************************************************************/
      t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq, tenResponseCode enRespCode);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceRequest(
       ***************************************************************************/
      /*!
      * \fn      vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
      * \brief   Called when DeselectDevice request is received by SPI.
      *          Mandatory interface to be implemented.
      * \param   [IN] corfrSelectReq: Device Selection Info
      * \retval  None
      **************************************************************************/
      t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceResponse(
      ***************************************************************************/
      /*!
       * \fn      vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
       *             tenResponseCode enRespCode)
       * \brief   Called when DeselectDevice operation is complete, with the result
       *          of the operation.
       *          Mandatory interface to be implemented.
       * \param   [IN] corfrSelectReq: Device Selection Info
       * \param   [IN] enRespCode: Response code enumeration
       * \retval  None
       **************************************************************************/
       t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /*********End of functions overridden from spi_tclBluetoothDevBase*********/
   
   /********Start of functions overridden from spi_tclOnCarRespBluetooth********/

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarRespBluetooth::vPostBTPairingRequest()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingRequest()
    * \brief   Called when Pairing request is sent by OnCar device.
    *          Mandatory interface to be implemented.
    * \param   [IN] poszOnCarBTAddress: BT address of OnCar device
    * \param   [IN] enPairingMethod: Pairing method selected by OnCar device
    * \sa      None
    **************************************************************************/
   t_Void vPostBTPairingRequest(t_String poszOnCarBTAddress,
           tenOnCarBTPairingMethod enPairingMethod);

   /**********End of functions overridden from spi_tclOnCarRespBluetooth********/
   
   /**************************************************************************
    ****************************END OF PUBLIC**********************************
    **************************************************************************/

   private:

   /***************************************************************************
    *********************************PRIVATE************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclOnCarBluetooth(const spi_tclOnCarBluetooth &rfcoBluetooth)
    ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarBluetooth(const spi_tclOnCarBluetooth &rfcoBluetooth)
   * \brief   Copy constructor not implemented hence made private
   **************************************************************************/
   spi_tclOnCarBluetooth(const spi_tclOnCarBluetooth &rfcoBluetooth);

   /***************************************************************************
   ** FUNCTION: const spi_tclOnCarBluetooth & operator=(
   **                                 const spi_tclOnCarBluetooth &rfcoBluetooth);
   ***************************************************************************/
   /*!
   * \fn      const spi_tclOnCarBluetooth & operator=(const spi_tclOnCarBluetooth &rfcoBluetooth);
   * \brief   assignment operator not implemented hence made private
   **************************************************************************/
   const spi_tclOnCarBluetooth & operator=(const spi_tclOnCarBluetooth &rfcoBluetooth);

   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarBluetooth::enGetPairingStatus()
   ***************************************************************************/
   /*!
   * \fn      enGetPairingStatus()
   * \brief   Interface to read current pairing status of OnCar device
   * \retval  tenDevicePairingStatus
   **************************************************************************/
   tenDevicePairingStatus enGetPairingStatus();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vSetPairingStatus()
   ***************************************************************************/
   /*!
   * \fn      vSetPairingStatus()
   * \brief   Interface to set pairing status of OnCar device
   * \param   enPairingStatus : [IN] Pairing state enum
   **************************************************************************/
   t_Void vSetPairingStatus(tenDevicePairingStatus enPairingStatus);

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclOnCarBluetooth::vDeselectOnCarDevice()
   ***************************************************************************/
   /*!
   * \fn      vDeselectOnCarDevice()
   * \brief   Triggers deselection request for OnCar device
   * \retval  None
   **************************************************************************/
   t_Void vDeselectOnCarDevice();

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclOnCarBluetooth::vSendDevicePairingResponse()
   ***************************************************************************/
   /*!
   * \fn      vSendDevicePairingResponse(t_Bool bReadyToPair)
   * \brief   Sends a BTPairingResponse to OnCar device
   * \param   bReadyToPair [IN]: True - if HU is ready to pair, else False
   * \retval  None
   **************************************************************************/
   t_Void vSendDevicePairingResponse(t_Bool bReadyToPair, 
             tenOnCarBTPairingState enOnCarBTPairingState);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTPairingInfoMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingInfoMsg()
    * \brief   Called when BT Pairing message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTLimitationModeMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTLimitationModeMsg()
    * \brief   Called when BT limitation mode message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTDeviceNameUpdateMsg()
    * \brief   Called when BT Device Name update message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTDeviceNameUpdateMsg(t_String szBTDeviceAddress,t_String szBTDeviceName);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTServiceAvailabilityMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTConnectionChangedMsg()
    * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
    virtual t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** BT Manager interface pointer
   ***************************************************************************/
   spi_tclBluetoothIntf* const m_cpoBTInterface;

   /***************************************************************************
   ** Pairing status of OnCar device
   ***************************************************************************/
   tenDevicePairingStatus  m_enPairingStatus;

   /***************************************************************************
   ** Lock object for device pairing status
   ***************************************************************************/
   Lock  m_oPairingStateLock;

   /***************************************************************************
   ** Action of pending SetBTLimitationMode request
   ***************************************************************************/
   tenBTLimitationAction m_enPendingBTLimAction;

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

};

#endif // _SPI_TCLONCARBLUETOOTH_H
