/*!
 *******************************************************************************
 * \file             spi_tclOnCarAudio.cpp
 * \brief            Implements the Audio functionality for OnCar
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for OnCar
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                      | Modification
 20.04.2018  | Rishav Sardar               | Initial Version
 \endverbatim
 20..8.2018  | Ashwini Savadi              | Audio Resource Manager adaptation for
                                             Media.
 10.10.2018  | Ashwini Savadi              | Audio Resource Manager implementation for
                                             Guidance
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_tclOnCarAudio.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclOnCarCmdAudio.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclOnCarAudio.cpp.trc.h"
#endif
#endif
static const t_U8 scou8AudioUnduck      = 1;
static const t_U32 cou32MaxAudioStreams = 5;
static t_U32 su32CurSelectedDevID       = 0;
static const t_U8 scou8AudioDuckDefdB   = 41;
static const t_U16 scou16AudioRampDefDuration  = 1000;  //In msec
//static timer_t srStopAudioTimerID;
//static t_Bool bStopAudTimerRunning = false;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::spi_tclOnCarAudio()
 ***************************************************************************/
spi_tclOnCarAudio::spi_tclOnCarAudio() : m_poOnCarCmdAudio(NULL),
m_enOnCarMediaPlaybackState(e8_ONCAR_MEDIA_PLAYBACK_STATE_NOT_KNOWN)
{
   ETG_TRACE_USR1(("[Constructor]:spi_tclOnCarAudio::spi_tclOnCarAudio() Entered"));

   SPI_NORMAL_ASSERT(!m_poOnCarCmdAudio);

   //! Initialize audio stream states.
   for(t_U8 u8Cnt = 0; u8Cnt < cou32OnCarMaxAudioStreams; u8Cnt++)
   {
       m_aenStreamState[u8Cnt] = e8_ONCAR_AUD_STREAM_CLOSED;
   }//for(t_U8 u8Cnt = 0; u8Cnt < cou32MaxAudioStreams; u8Cnt++)

   //! Initialize audio channel states.
   for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)
   {
       m_aenChannelStatus[u8Cnt] = e8_AUD_NOT_ACTIVE;
   }//for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)
   ETG_TRACE_USR1(("[Constructor]:spi_tclOnCarAudio::spi_tclOnCarAudio() Entered"));

}//spi_tclOnCarAudio::spi_tclOnCarAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::~spi_tclOnCarAudio()
 ***************************************************************************/
spi_tclOnCarAudio::~spi_tclOnCarAudio()
{
   ETG_TRACE_USR1(("[Destructor]:spi_tclOnCarAudio::~spi_tclOnCarAudio() Entered"));
   su32CurSelectedDevID = 0;
   m_poOnCarCmdAudio = NULL;
   ETG_TRACE_USR1(("[Destructor]:spi_tclOnCarAudio::~spi_tclOnCarAudio() Left"));
   m_enOnCarMediaPlaybackState = e8_ONCAR_MEDIA_PLAYBACK_STATE_NOT_KNOWN;
}//spi_tclOnCarAudio::~spi_tclOnCarAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAudio::vRegisterCallbacks Entered "));
   m_rAudCallbacks = rfrAudCallbacks;
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAudio::vRegisterCallbacks Left "));

}//t_Void spi_tclOnCarAudio::vRegisterCallbacks()

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::vSetAudioPipeConfig
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vSetAudioPipeConfig(const tmapAudioPipeConfig& crfmapAudioPipeConfig)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAudio::vSetAudioPipeConfig Entered "));
   m_mapAudioPipeConfig = crfmapAudioPipeConfig;
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::bSelectAudioDevice()
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bSelectAudioDevice(t_U32 u32DeviceId)
{
   ETG_TRACE_USR1(("spi_tclOnCarAudio::bSelectAudioDevice Entered with Device ID:0x%x", u32DeviceId));
   t_Bool bRetVal = false;
   if(m_poOnCarCmdAudio)
   {
      tvectrOnCarAudioConfig vOnCarAudioConfig;
      trOnCarAudioConfig rConfig_media,rConfig_guidance;

      rConfig_media.u16threadPriority = 42;
      rConfig_media.enOnCarAudStreamtype = e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA;
      rConfig_media.szAudioPipeline = m_mapAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE];

      rConfig_guidance.u16threadPriority= 42;
      rConfig_guidance.enOnCarAudStreamtype = e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE;
      rConfig_guidance.szAudioPipeline = m_mapAudioPipeConfig[e8AUDIOCONFIG_ALTERNATEAUDIO];

      vOnCarAudioConfig.push_back(rConfig_media);
      vOnCarAudioConfig.push_back(rConfig_guidance);
      bRetVal = m_poOnCarCmdAudio->bInitialize(vOnCarAudioConfig);
   }//if (m_poOnCarCmdAudio)

   //! Acknowldege Device Selector with response on creation of OnCar Audio Endpoints
   if (NULL != m_rAudCallbacks.fvSelectDeviceResp)
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_SELECT, bRetVal);

   }//if (NULL != m_rAudCallbacks.fvSelectDeviceResp)
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarAudio::bSelectAudioDevice Left with result :%d",bRetVal));
   return bRetVal;
}//t_Bool spi_tclOnCarAudio::bSelectAudioDevice(t_U32)

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::vDeselectAudioDevice()
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vDeselectAudioDevice(t_U32 u32DeviceId)
{
   ETG_TRACE_USR2(("[FUNC]spi_tclOnCarAudio::vDeselectAudioDeviceEntered with Device ID:0x%x", u32DeviceId));

   if (NULL != m_poOnCarCmdAudio)
   {
      m_poOnCarCmdAudio->vUninitialize();
      m_poOnCarCmdAudio->vDestroyAudioEndpointInstance();
   }
   
   if ((e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_MAIN_OUT]) || 
         (m_enOnCarMediaPlaybackState == e8_ONCAR_MEDIA_PLAYBACK_STATE_PAUSED))
   {
      ETG_TRACE_USR1(("Device Deselected. Requesting Main Audio Channel Deactivation/Mute"));
      vDeactivateChannel(e8AUD_MAIN_OUT);
   }

   if(e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_DUCK])
   {
       ETG_TRACE_USR1(("Device Deselected. Requesting Mix Audio Channel Deactivation"));
       vDeactivateChannel(e8AUD_DUCK);
   }//if(e8_AUD_ACT_GRANTED == m_aenChannelStatus[e8AUD_DUCK])
       
   su32CurSelectedDevID = 0;
   m_enOnCarMediaPlaybackState = e8_ONCAR_MEDIA_PLAYBACK_STATE_STOPPED;
   
   //! Acknowldege Device Selector
   if (NULL != m_rAudCallbacks.fvSelectDeviceResp)
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_DESELECT, true);
   }
   
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclOnCarAudio::vUpdateDeviceSelection()
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vUpdateDeviceSelection(t_U32 u32DevID,
                                      tenDeviceCategory enDevCat,
                                      tenDeviceConnectionReq enDeviceConnReq,
                                      tenResponseCode enRespCode,
                                      tenErrorCode enErrorCode)
{
   m_oAudioStreamLock.s16Lock();
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);
   ETG_TRACE_USR4(("spi_tclOnCarAudio::vUpdateDeviceSelection entered"));

   if(e8DEVCONNREQ_SELECT == enDeviceConnReq)
   {
      su32CurSelectedDevID = (e8SUCCESS == enRespCode)?(u32DevID):(0);
      if((e8FAILURE == enRespCode) && (NULL != m_poOnCarCmdAudio))
      {
         m_poOnCarCmdAudio->vUninitialize();
         m_poOnCarCmdAudio->vDestroyAudioEndpointInstance();

      }//if((e8FAILURE == enRespCode) && (NULL != m_poOnCarCmdAudio))

      //! Clear all the stream states.
      for(t_U8 u8Cnt = 0; u8Cnt < cou32MaxAudioStreams; u8Cnt++)
      {
          ETG_TRACE_USR4(("Stream State for %d stream is %d", u8Cnt, m_aenStreamState[u8Cnt]));
          m_aenStreamState[u8Cnt] = e8_ONCAR_AUD_STREAM_CLOSED;
      }//for(t_U8 u8Cnt = 0; u8Cnt < cou32MaxAudioStreams; u8Cnt++)

      //! Clear all the channel states.
      for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)
      {
          ETG_TRACE_USR4(("Channel State for %d stream is %d",ETG_ENUM(SPI_AUDIO_DIRECTION, u8Cnt),
                                         ETG_ENUM(AUD_CHANNEL_STATE, m_aenChannelStatus[u8Cnt])));
          m_aenChannelStatus[u8Cnt] = e8_AUD_NOT_ACTIVE;
      }//for(t_U8 u8Cnt = 0; u8Cnt < e8AUD_INVALID; u8Cnt++)


      ETG_TRACE_USR4(("spi_tclOnCarAudio::vUpdateDeviceSelection. Selected Device ID %d", su32CurSelectedDevID));

   }//if(e8DEVCONNREQ_SELECT == enDeviceConnReq)

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR4(("spi_tclOnCarAudio:vUpdateDeviceSelection() left"));

}//t_Void spi_tclOnCarAudio::vUpdateDeviceSelection(t_U32 u32DevID,...)

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarAudio::bInitialise();
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bInitialise()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAudio::bInitialise() entered "));

   t_Bool bRet = false;
   spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
   if (poOnCarManager)
   {
      m_poOnCarCmdAudio = poOnCarManager->poGetAudioInstance();
      bRet = poOnCarManager->bRegisterObject((spi_tclOnCarRespAudio*) this);
      ETG_TRACE_USR2(("[DESC]:spi_tclOnCarAudio():Registered to Audio Callbacks from OnCar Session [%d]", ETG_ENUM(BOOL, bRet)));
   }//if (poOnCarManager)
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::bInitializeAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate)
{
   SPI_INTENTIONALLY_UNUSED(enSamplingRate);
   SPI_INTENTIONALLY_UNUSED(enNativeSamplingRate);
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   ETG_TRACE_USR2(("spi_tclOnCarAudio::bInitializeAudioPlayback Entered:Dev-0x%x for Direction [%d]",
                 u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarAudio::bGetOnCarAudioStreamStype(...)
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bGetOnCarAudioStreamStype(tenAudioDir enAudDir,
      tenOnCarAudioStreamType& rfenOnCarAudioStreamType)
{
   t_Bool bValidAudDir = true;
   switch (enAudDir)
   {
      case e8AUD_MAIN_OUT:
         rfenOnCarAudioStreamType = e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA;
      break;
      case e8AUD_MIX_OUT:
      case e8AUD_DUCK:
         rfenOnCarAudioStreamType = e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE;
      break;
      case e8AUD_PHONE_IN:
      case e8AUD_VR_IN:

      case e8AUD_ALERT_OUT:
      case e8AUD_INVALID:
      default:
      {
         ETG_TRACE_USR1((" bGetOnCarAudioStreamStype: Unsupported Audio direction! "));
         bValidAudDir = false;
      }
      break;
   }

   ETG_TRACE_USR1((" spi_tclOnCarAudio::bGetOnCarAudioStreamStype: left with bValidAudDir = %d, StreamType %d ",
         ETG_ENUM(BOOL, bValidAudDir),rfenOnCarAudioStreamType));
   return bValidAudDir;
}//t_Bool  spi_tclOnCarAudio::bGetOnCarAudioStreamStype(tenAudioDir enAudDir,...)

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bStartAudio(t_U32 u32DeviceId, t_String szAudioDev, tenAudioDir enAudioDir)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclOnCarAudio::bStartAudio() entered "));
   m_oAudioStreamLock.s16Lock();

   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   ETG_TRACE_USR1(("spi_tclOnCarAudio::bStartAudio entered for Audio Source %d with Audio Device - %s ",
         enAudioDir, szAudioDev.c_str()));

   tenOnCarAudioStreamType enAudStreamType = e8_ONCAR_AUDIOSTREAM_TYPE_UNKNOWN;
   t_Bool bRetVal = bGetOnCarAudioStreamStype(enAudioDir, enAudStreamType);

   if ((NULL != m_poOnCarCmdAudio) && (false == szAudioDev.empty()) && (true == bRetVal))
   {
      ETG_TRACE_USR4(("Current Stream State for Activated Audio Channel is %d",
               m_aenStreamState[enAudStreamType]));
      m_aenChannelStatus[enAudioDir] = e8_AUD_ACT_GRANTED;

      //! Start playback only if the Audio Stream from MD is blocked for Activation of Audio channel
      //! Do nothing for cases like restoration of Audio channel by Audio manager where the MD may not have activated stream
      //! When playback start is received, request for Audio channel and SPI Audio Manager will mock the necessary response to complete sequence
      if (e8_ONCAR_AUD_STREAM_OPEN == m_aenStreamState[enAudStreamType])
      {
         ETG_TRACE_USR4(("Starting Audio playback for OnCar Stream %d",  enAudStreamType));
         m_aenStreamState[enAudStreamType] = e8_ONCAR_AUD_STREAMING;
         if(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA == enAudStreamType)
         {
            m_enOnCarMediaPlaybackState = e8_ONCAR_MEDIA_PLAYBACK_STATE_PLAYING;
         }
         
         m_poOnCarCmdAudio->vAcknowledgeStartAudioRequest(enAudStreamType);
      }//if (e8_AUD_STREAM_OPEN == m_aenStreamState[enAudStreamType])

      if((e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE == enAudStreamType) && (NULL != m_rAudCallbacks.fbSetAudioDucking) && (e8AUD_DUCK == enAudioDir))
      {
         (m_rAudCallbacks.fbSetAudioDucking)(scou16AudioRampDefDuration, scou8AudioDuckDefdB,
                  e8_DUCKINGTYPE_DUCK,e8DEV_TYPE_ONCAR);
      }//if((NULL != m_rAudCallbacks.fbSetAudioDucking) && (e8AUD_DUCK == enAudDir))

   }//if ((NULL != m_poOnCarCmdAudio) && (false == szAudioDev.empty()))

   //! In any case, Acknowledge Start of playback to SPI Audio Manager.
   if (NULL != m_rAudCallbacks.fvStartAudioResp)
   {
      (m_rAudCallbacks.fvStartAudioResp)(enAudioDir, bRetVal);
   }//if (NULL != m_rAudCallbacks.fvStartAudioResp)

   //! If the device is already deselected, request for de-activation of Audio channel
   //! This has to be initiated only after sending an acknowledgement for Source Activity (On) to Audio Mananger

   if (0 == su32CurSelectedDevID)
   {
       ETG_TRACE_USR4(("Audio Channel Activated but Device is already deselected. Deactivate Audio Channel"));
       //vDeactivateChannel(enAudDir);
   }//if(0 == su32CurSelectedDevID)

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1((" spi_tclOnCarAudio::bStartAudio() left "));

   return bRetVal;

}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
        t_String szInputAudioDev, tenAudioDir enAudDir)
{
    ETG_TRACE_USR2(("spi_tclOnCarAudio::bStartAudio u32DeviceId = %d",u32DeviceId));
    ETG_TRACE_USR2(("spi_tclOnCarAudio::bStartAudio szOutputAudioDev = %s",szOutputAudioDev.c_str()));
    ETG_TRACE_USR2(("spi_tclOnCarAudio::bStartAudio szInputAudioDev = %s",szInputAudioDev.c_str()));
    ETG_TRACE_USR2(("spi_tclOnCarAudio::bStartAudio enAudDir = %d",ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir)));
    t_Bool bIsAudioStarted = bStartAudio(u32DeviceId,szInputAudioDev,enAudDir);

    return bIsAudioStarted;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudio::vActivateChannel(tenAudioDir enAudDir)
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vActivateChannel(tenAudioDir enAudDir)
{
   /*lint -esym(40,fvLaunchAudioReq)fvLaunchAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclOnCarAudio::vActivateChannel() entered for Audio direction %d, Current DevID 0x%x ", ETG_ENUM(
               SPI_AUDIO_DIRECTION, enAudDir), su32CurSelectedDevID));

   tenOnCarAudioStreamType enAudStream = e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA;
   t_Bool bRet = bGetOnCarAudioStreamStype(enAudDir, enAudStream);

   if ((NULL != m_poOnCarCmdAudio) && (NULL != m_rAudCallbacks.fvLaunchAudioReq)
       && (bRet) && (0 != su32CurSelectedDevID))
   {
      ETG_TRACE_USR4(("[PARAM]::spi_tclOnCarAudio::vActivateChannel: Current channel status = %d, Stream state = %d", m_aenChannelStatus[enAudDir], m_aenStreamState[enAudStream]));
      trAudSampleRate rSampleRate;
      switch (enAudDir)
      {
            case e8AUD_MAIN_OUT:
            {
                rSampleRate.enSampleRate = e8AUD_SAMPLERATE_DEFAULT;
                m_aenChannelStatus[enAudDir] = e8_AUD_ACT_REQUESTED;
               (m_rAudCallbacks.fvLaunchAudioReq)(su32CurSelectedDevID,
                  e8DEV_TYPE_ONCAR, e8AUD_MAIN_OUT, rSampleRate);

            }//case e8AUD_MAIN_OUT:
               break;
            case e8AUD_DUCK:
            {
                rSampleRate.enSampleRate = e8AUD_SAMPLERATE_DEFAULT;
                ETG_TRACE_USR4(("Request for Activation of Guidance Audio Channel"));
                m_aenChannelStatus[enAudDir] = e8_AUD_ACT_REQUESTED;
               (m_rAudCallbacks.fvLaunchAudioReq)(su32CurSelectedDevID,
                  e8DEV_TYPE_ONCAR, e8AUD_DUCK, rSampleRate);

            }//case e8AUD_DUCK:
               break;
            default:
            {
               ETG_TRACE_ERR(("No action taken for invalid audio direction %d ", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
            }//default:
               break;
         }//switch (enAudDir)
   
   }//if ((NULL != m_poOnCarCmdAudio) && (NULL != m_rAudCallbacks.fvLaunchAudioReq))
}//t_Void spi_tclOnCarAudio::vActivateChannel()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudio::vDeactivateChannel(tenAudioDir enAudioDirection)
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vDeactivateChannel(tenAudioDir enAudioDirection)
{
   /*lint -esym(40,fvTerminateAudioReq)fvTerminateAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclOnCarAudio::vDeactivateChannel() entered for Audio direction %d ",
         ETG_ENUM(SPI_AUDIO_DIRECTION, enAudioDirection)));

   if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
   {
      m_aenChannelStatus[enAudioDirection] = e8_AUD_DEACT_REQUESTED;
      (m_rAudCallbacks.fvTerminateAudioReq)(su32CurSelectedDevID, enAudioDirection, e8REASON_NOMEDIA);
   }//if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
{
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   /*lint -esym(40,tfbSetAudioDucking)fbSetAudioDucking Undeclared identifier */
   /*lint -esym(746,fbSetAudioDucking)call to function fbSetAudioDucking() not made in the presence of a prototype */

   m_oAudioStreamLock.s16Lock();

   ETG_TRACE_USR1((" spi_tclOnCarAudio::bFinalizeAudioPlayback entered: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   //! Audio channel is deactivated. Update the audio channel state.
   m_aenChannelStatus[enAudDir] = e8_AUD_NOT_ACTIVE;

   //! Media Source is deactivated.
   if(e8AUD_MAIN_OUT == enAudDir)
   {
       ETG_TRACE_USR4(("Media audio channel is deactivated."));

       //! MD had setup a new Media Audio stream during the deactivation of Media Audio channel
       if (e8_ONCAR_AUD_STREAM_OPEN == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA])
       {
           ETG_TRACE_USR4(("Media Audio Stream was opened by MD. Requesting Media audio channel Activation"));
           vActivateChannel(enAudDir);
       }//if(e8_ONCAR_AUD_STREAM_OPEN == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA])

   }//if(e8AUD_MAIN_OUT == enAudDir)

   //! Mix Audio Source (Guidance) is deactivated.
   else if(e8AUD_DUCK == enAudDir)
   {
      ETG_TRACE_USR4(("Mix audio channel is deactivated."));

      if (e8_ONCAR_AUD_STREAM_OPEN == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE])
      {
         ETG_TRACE_USR4(("TTS Stream is open. Hence requesting for re-activation of Mix audio channel "));
         vActivateChannel(enAudDir);
      }
      else if (NULL != m_rAudCallbacks.fbSetAudioDucking)
      {
         m_rAudCallbacks.fbSetAudioDucking(scou16AudioRampDefDuration, scou8AudioUnduck,
               e8_DUCKINGTYPE_UNDUCK, e8DEV_TYPE_ONCAR);
      }
   }//else if(e8AUD_DUCK == enAudDir)
   m_oAudioStreamLock.vUnlock();

   ETG_TRACE_USR1((" spi_tclOnCarAudio::bFinalizeAudioPlayback left: AudDir %d", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudio::vStopAudio()
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused)
{
   ETG_TRACE_USR2(("[FUNC]spi_tclOnCarAudio::vStopAudio: Stop Audio Stream for Device [%d] for Audio Dir [%d]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));

   m_oAudioStreamLock.s16Lock();

   ETG_TRACE_USR4(("Media Stream State - %d, Media Channel Status - %d",
         ETG_ENUM(ONCAR_AUDSTREAM_STATE,m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA]),
         ETG_ENUM(ONCAR_AUD_CHANNEL_STATE, m_aenChannelStatus[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA])));

    tenOnCarAudioStreamType enAudStreamType = e8_ONCAR_AUDIOSTREAM_TYPE_UNKNOWN;
    t_Bool bRetVal = bGetOnCarAudioStreamStype(enAudDir, enAudStreamType);

   if(NULL != m_poOnCarCmdAudio)
   {
       if ((e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA == enAudStreamType) && (e8AUD_MAIN_OUT == enAudDir) &&
             (e8_ONCAR_AUD_STREAMING == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA]))
       {
          ETG_TRACE_USR2(("Media is streaming. Wait for Media PlaybackStop to send StopAudio response "));
          m_poOnCarCmdAudio->vStopALSAStreaming(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA);
          
          vSendStopAudioResponse(e8AUD_MAIN_OUT);
       }//if ((e8AUD_MAIN_OUT == enAudDir) && ...)
       //! Send StopAudio response
       else if((e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE == enAudStreamType) &&
            (e8AUD_DUCK == enAudDir))
       {
          if(e8_ONCAR_AUD_STREAMING == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE])
          {
              m_poOnCarCmdAudio->vStopALSAStreaming(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE);
              m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE] = e8_ONCAR_AUD_STREAM_CLOSED;
          }
          else
          {
              m_poOnCarCmdAudio->vAcknowledgeStopAudioRequest(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE);
          }
          vSendStopAudioResponse(e8AUD_DUCK);
          ETG_TRACE_USR2(("spi_tclOnCarAudio::vStopAudio: Guidance Stream playback stopped. Deactivate Guidance/Mix Audio channel"));
       }//end of if-else
   }
   if ((false == bIsPaused) && (e8AUD_MAIN_OUT == enAudDir) &&
      (e8_ONCAR_AUD_STREAM_CLOSED == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA]))
   {
      vSendStopAudioResponse(e8AUD_MAIN_OUT);
   }
   if(true == bIsPaused)
   {
      m_enOnCarMediaPlaybackState = e8_ONCAR_MEDIA_PLAYBACK_STATE_PAUSED;
   }
   ETG_TRACE_USR1(("spi_tclOnCarAudio::vStopAudio u32DeviceId = %d enAudDir = %d  ",
            u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   m_oAudioStreamLock.vUnlock();
   ETG_TRACE_USR1((" spi_tclOnCarAudio::vStopAudio() left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudio::vSendStopAudioResponse(...)
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vSendStopAudioResponse(tenAudioDir enAudDir)
{
   /*lint -esym(40,fvStopAudioResp) fvStopAudioResp is not referenced */
   /*lint -esym(40,fvStopAudioResp) fvStopAudioResp Undeclared identifier */

   ETG_TRACE_USR4(("spi_tclOnCarAudio:vSendStopAudioResponse entered"));
   if (NULL != m_rAudCallbacks.fvStopAudioResp)
   {
      (m_rAudCallbacks.fvStopAudioResp)(enAudDir, true);
   }//if (NULL != m_rAudCallbacks.fvStopAudioResp)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclOnCarAudio::vPoststartAudioPlaybackRequest(..
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vPoststartAudioPlaybackRequest(const tenOnCarAudioStreamType enStreamType)
{
    ETG_TRACE_USR1((" spi_tclOnCarAudio::vPoststartAudioPlaybackRequest() Entered"));

    m_oAudioStreamLock.s16Lock();
    tenAudioDir enAudioDir = e8AUD_INVALID;
//  tenOnCarAudioStreamType enAudStream = e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA ;
//  t_Bool bRet = bGetOnCarAudioStreamStype(e8AUD_MAIN_OUT, enAudStream);
    t_Bool bRet = bGetAudioDir(enStreamType, enAudioDir);
    if(true == bRet)
    {
            m_aenStreamState[enStreamType] = e8_ONCAR_AUD_STREAM_OPEN;

            if((e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA == enStreamType) && (e8_AUD_DEACT_REQUESTED != m_aenChannelStatus[enAudioDir]))
            {
                vProcessOnCarMediaStart(enAudioDir);
            } //if((e8_AUDIOTYPE_MEDIA == enAudStreamType) && ((e8_AUD_DEACT_REQUESTED !=..))
            //! Guidance Audio channel status is not checked here since it needs to be evaluated in context of Mic Stream, Speech Audio channel and
            //! Voice Session status. This is handled in vProcessGuidanceStart().
            else if(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE == enStreamType)
            {
                vProcessOnCarGuidanceStart(enAudioDir);
            }//else if(e8_AUDIOTYPE_GUIDANCE == enAudStreamType)
    }   
    m_oAudioStreamLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclOnCarAudio::vPoststopAudioPlaybackRequest(..
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vPoststopAudioPlaybackRequest(const tenOnCarAudioStreamType enStreamType)
{
    ETG_TRACE_USR1((" spi_tclOnCarAudio::vPoststopAudioPlaybackRequest() Entered"));
    m_oAudioStreamLock.s16Lock();

    tenAudioDir enAudioDir = e8AUD_INVALID;
    t_Bool bRetVal = bGetAudioDir(enStreamType, enAudioDir);
    if(true == bRetVal)
    {
        ETG_TRACE_USR4(("Updated %d Stream State to %d",ETG_ENUM(AAP_AUDSTREAM_TYPE, enStreamType),
              ETG_ENUM(AAP_AUDSTREAM_STATE,m_aenStreamState[enStreamType])));

        if(NULL != m_poOnCarCmdAudio)
        {
            //! Media Stream playback stopped. Not required to deactivate audio channel.
            if(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA == enStreamType)
            {
                ETG_TRACE_USR4(("Media Stream playback stopped. Audio channel is not required to be deactivated"));
            
                //! Send StopAudio response if timer was started to wait for Media PlaybackStop
                m_poOnCarCmdAudio->vAcknowledgeStopAudioRequest(enStreamType);
            }//if(e8_AUDIOTYPE_MEDIA == enStreamType)
            else if(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE == enStreamType)
            {
                if(e8_ONCAR_AUD_STREAMING == m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE])
                {
                    ETG_TRACE_USR2(("Guidance Stream playback stopped. Deactivate Guidance/Mix Audio channel"));
                    vDeactivateChannel(enAudioDir);
                    m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE] = e8_ONCAR_AUD_STREAM_CLOSED;
                }
                else
                {
                    ETG_TRACE_USR2(("Guidance Stream playback stopped and deactivate Guidance/Mix Audio channel is done. Ack stop audio req"));
                    m_poOnCarCmdAudio->vAcknowledgeStopAudioRequest(enStreamType);
                }
            }
        }
    }
    m_oAudioStreamLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudio::vProcessOnCarMediaStart(...)
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vProcessOnCarMediaStart(tenAudioDir enAudioDirection)
{
     ETG_TRACE_USR1(("spi_tclOnCarAudio::vProcessOnCarMediaStart() entered"));

     m_oAudioStreamLock.s16Lock();
    if (NULL != m_poOnCarCmdAudio)
    {
         if((e8AUD_MAIN_OUT == enAudioDirection) && (e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudioDirection]))
         {
             ETG_TRACE_USR1(("Media Audio Channel not active. Requesting Activation of Media Audio Channel"));
             vActivateChannel(enAudioDirection);
         }
         else if(e8_AUD_NOT_ACTIVE != m_aenChannelStatus[enAudioDirection])
         {
             ETG_TRACE_USR1(("Media Audio Channel is active. acknowledging device"));
             m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA] = e8_ONCAR_AUD_STREAMING;
             m_enOnCarMediaPlaybackState = e8_ONCAR_MEDIA_PLAYBACK_STATE_PLAYING;
             
             m_poOnCarCmdAudio->vAcknowledgeStartAudioRequest(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA);
         }
     }

     m_oAudioStreamLock.vUnlock();
     ETG_TRACE_USR1(("spi_tclOnCarAudio::vProcessOnCarMediaStart() left"));
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudio::vProcessOnCarMediaStart(...)
 ***************************************************************************/
t_Void spi_tclOnCarAudio::vProcessOnCarGuidanceStart(tenAudioDir enAudioDirection)
      {
    ETG_TRACE_USR1(("spi_tclOnCarAudio::vProcessOnCarGuidanceStart() entered"));

    m_oAudioStreamLock.s16Lock();
    if(NULL != m_poOnCarCmdAudio)
    {
        if((e8AUD_DUCK == enAudioDirection) && ((e8_AUD_NOT_ACTIVE == m_aenChannelStatus[enAudioDirection]) ||
                 (e8_AUD_DEACT_REQUESTED == m_aenChannelStatus[enAudioDirection] )))
        {
            ETG_TRACE_USR1(("Guidance Audio Channel not active. Requesting Activation of Guidance Audio Channel"));
            vActivateChannel(enAudioDirection);
        }
        else if((e8AUD_DUCK == enAudioDirection) && (e8_AUD_ACT_GRANTED == m_aenChannelStatus[enAudioDirection]))
        {
            ETG_TRACE_USR1(("Guidance Audio Channel is already active. Requesting Activation of Guidance Audio Channel"));
            m_aenStreamState[e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE] = e8_ONCAR_AUD_STREAMING;
            m_poOnCarCmdAudio->vAcknowledgeStartAudioRequest(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE);
      }
    }
    m_oAudioStreamLock.vUnlock();
    ETG_TRACE_USR1(("spi_tclOnCarAudio::vProcessOnCarGuidanceStart() left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclOnCarAudio::bGetAudioDir()
 ***************************************************************************/
t_Bool spi_tclOnCarAudio::bGetAudioDir(tenOnCarAudioStreamType enAudStreamType,
      tenAudioDir& rfenAudioDir)
{
    ETG_TRACE_USR1(("spi_tclOnCarAudio::bGetAudioDir() entered"));
   t_Bool bValidAudDir = true;

   switch (enAudStreamType)
   {
      case e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA:
      {
         rfenAudioDir = e8AUD_MAIN_OUT;
      }
         break;
      case e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE:
     // case e8_AUDIOTYPE_SYSTEM_AUDIO:
      {
         rfenAudioDir = e8AUD_DUCK;
         //rfenAudioDir = e8AUD_MIX_OUT;
      }
         break;
      case e8_ONCAR_AUDIOSTREAM_TYPE_UNKNOWN:
      default:
      {
         ETG_TRACE_USR1((" bGetAudioDir: Unsupported Audio direction! "));
         bValidAudDir = false;
}
      break;
   } //switch(enAudStreamType)
    ETG_TRACE_USR1((" spi_tclOncarAudio::bGetAudioDir: left with bValidAudDir = %d, Aud Dir %d ",
         ETG_ENUM(BOOL, bValidAudDir), ETG_ENUM(SPI_AUDIO_DIRECTION, rfenAudioDir)));
   return bValidAudDir;
}//t_Bool spi_tclAAPAudio::bGetAudioDir(tenAAPAudStreamType enAudStreamType,..)
//lint -restore

///////////////////////////////////////////////////EOF///////////////////////////////////////////////////////////////
