/*!
 *******************************************************************************
 * \file             spi_tclOnCarAudio.h
 * \brief
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for MySpin
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                      | Modification
 19.04.2018  | Rishav Sardar               | Initial Version
 20..8.2018  | Ashwini Savadi              | Audio Resource Manager adaptation for
                                             Media.
 \endverbatim

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLONCARAUDIO_H
#define SPI_TCLONCARAUDIO_H

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclAudioDevBase.h"
#include "spi_tclOnCarRespAudio.h"
#include "SPI_OnCarTypes.h"
#include "SPITypes.h"
#include "Lock.h"
#include "Timer.h"

static const t_U32 cou32OnCarMaxAudioStreams = 5;
class spi_tclOnCarCmdAudio;
/**
 * This class implements the Audio for OnCar
 */

class spi_tclOnCarAudio: public spi_tclAudioDevBase, public spi_tclOnCarRespAudio
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclOnCarAudio::spi_tclOnCarAudio();
       ***************************************************************************/
      /*!
       * \fn      spi_tclOnCarAudio()
       * \brief   Default Constructor
       **************************************************************************/
       spi_tclOnCarAudio();

      /***************************************************************************
       ** FUNCTION:  spi_tclOnCarAudio::~spi_tclOnCarAudio();
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclOnCarAudio()
       * \brief   Virtual Destructor
       **************************************************************************/
      virtual ~spi_tclOnCarAudio();


      /*********Start of functions overridden from spi_tclAudioDevBase*******/


      /***************************************************************************
       ** FUNCTION:  spi_tclOnCarAudio::vRegisterCallbacks();
       ***************************************************************************/
      /*!
       * \fn      vRegisterCallbacks()
       * \brief   Register callbacks for audio
       * \param   [rfrAudCallbacks]: Used to hold the callbacks to be registered by the
       *                          creator of the class object
       * \retval t_Void
       **************************************************************************/
      t_Void vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bInitialise();
       ***************************************************************************/
      /*!
       * \fn      bInitialise()
       * \brief   Method to initialises the service handler. (Performs initialisations which
       *          are not device specific.)
       *          Optional interface to be implemented.
       * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
       * \sa      bUninitialise()
       ***************************************************************************/
      t_Bool bInitialise();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vSetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetAudioPipeConfig()
       * \brief   Set the Audio pipeline configuration for alsa devices
       * \param   crfmapAudioPipeConfig: Contains audio pipeline configuration
       **************************************************************************/
      t_Void vSetAudioPipeConfig(const tmapAudioPipeConfig& crfmapAudioPipeConfig);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bInitializeAudioPlayback(t_U32)
       ***************************************************************************/
      /*!
       * \fn      bInitializeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions to prepare for an Audio Playback.
       *          Function will be called prior to a Play Command from Audio Manager.
       *          Mandatory Interface to be implemented by Device Class.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device
        * \param   [enAudDir]: Audio route being allocated
       * \param   [enAudSampleRate]: Specifies the Audio Sample Rate.
       * \retval  Bool value
       **************************************************************************/
      virtual t_Bool bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,
            tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate);

      /***************************************************************************
      ** FUNCTION:  t_Void  spi_tclOnCarAudio::vUpdateDeviceSelection()
      ***************************************************************************/
      /*!
      * \fn      t_Void vUpdateDeviceSelection()
      * \brief   To update the device selection.
      * \param   u32DevID : [IN] Resource Manager callbacks structure.
      * \param   enDevCat : [IN] Category of the device
      * \param   enDeviceConnReq : [IN] Select/ deselect.
      * \param   enRespCode : [IN] Response code (success/failure)
      * \param   enErrorCode : [IN] Error
      * \retval  t_Void
      **************************************************************************/
      t_Void vUpdateDeviceSelection(t_U32 u32DevID, tenDeviceCategory enDevCat,
                              tenDeviceConnectionReq enDeviceConnReq,
                              tenResponseCode enRespCode, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vStopAudio()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Void vStopAudio
       * \brief   To stop the audio that is currently playing
       * \param    u32DeviceId : [IN] Unique Device Id
       * \param   enAudDir : [IN] Direction of audio
       * \retval  t_Void
       **************************************************************************/
      t_Void vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bSelectAudioDevice()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bSelectAudioDevice
       * \brief   To select the audio device
       * \param   u32DeviceId : [IN] Unique Device Id
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSelectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vDeselectAudioDevice()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool vDeselectAudioDevice
       * \brief   To deselect the audio device
       * \param   u32DeviceId : [IN] Unique Device Id
       * \retval  t_Void
       **************************************************************************/
      t_Void vDeselectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bIsAudioLinkSupported()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bIsAudioLinkSupported
       * \brief   Perfom necessary actions specific to a device on de-selection.
       * \param   u32DeviceId : [IN] Unique Device Id
       * \param   enLink : [IN] used to identify the audio streaming link
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bIsAudioLinkSupported(t_U32 u32DeviceId, tenAudioLink enLink)
      {
          SPI_INTENTIONALLY_UNUSED(u32DeviceId);
          SPI_INTENTIONALLY_UNUSED(enLink);
          return true;
      }

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vOnAudioError()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Void vOnAudioError
       * \brief   To deselect the audio device
       * \param   enAudDir : [IN] Used to identify the audio stream direction
       * \param   enAudioError : [IN] used to define the audio errors
       * \retval  t_Void
       **************************************************************************/
      t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
      {
      }

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bStartAudio()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bStartAudio
       * \brief   To Start audio streaming
       * \param    cou32DevId : [IN] Unique Device Id
       * \param   coenConnReq : [IN] connected/disconnected
       * \retval  t_Void
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szAudioDev, tenAudioDir enAudioDir);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bStartAudio(t_U32,t_String, t_String,
       **                   tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
       *          t_String szInputAudioDev, tenAudioDir enAudDir)
       * \brief   Overloaded method to handle audio stream for Phone and VR.
       *          Start Streaming of Audio from the CE Device to the Audio Output
       *          Device assigned by the Audio Manager for the Source.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       *          [szOutputAudioDev]: Output ALSA Audio Device
       *          [szInputAudioDev] : Input ALSA Audio Device
       *          [enAudDir]        : Specify the Audio Direction(Phone or VR Audio).
       * \retval  Bool value
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
              t_String szInputAudioDev, tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bFinalizeAudioPlayback(t_U32,tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions on completion of Audio Playback.
       *          Function will be called after to a Stop Command from Audio Manager.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       * \param   [enAudDir]: Audio route being deallocated
       * \retval  Bool value
       **************************************************************************/
      t_Bool bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir);

      /************End of functions overridden from spi_tclAudioDevBase***********/

      /***********Start of functions overridden from spi_tclOnCarRespAudio**********/

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vPoststartAudioPlaybackRequest(...)
       ***************************************************************************/
      /*!
       * \fn      t_Void vPoststartAudioPlaybackRequest(const AUDIOSTREAM_TYPE type)
       * \brief   Callback to start audio streaming from IOnCarAudioSinkCallbacks.
       * \sa      vstopAudioPlaybackRequest()
       **************************************************************************/
      virtual t_Void vPoststartAudioPlaybackRequest(const tenOnCarAudioStreamType enStreamType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vPoststopAudioPlaybackRequest(...)
       ***************************************************************************/
      /*!
       * \fn      t_Void vPoststopAudioPlaybackRequest(const AUDIOSTREAM_TYPE type)
       * \brief   Callback to start audio streaming from IOnCarAudioSinkCallbacks.
       * \sa      None
       **************************************************************************/
      virtual t_Void vPoststopAudioPlaybackRequest(const tenOnCarAudioStreamType enStreamType);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bGetAudioDir()
       ***************************************************************************/
      /*!
       * \fn      bGetAudioDir
       * \brief  Function to convert OnCar Audio stream type to Audio direction.
       * \param  enAudStreamType : OnCar Audio stream type.
       * \param  rfenAudioDir : reference to Audio direction type.
       * \retval  Bool value, TRUE if retrieved, FALSE otherwise
       **************************************************************************/
       t_Bool bGetAudioDir(tenOnCarAudioStreamType enAudStreamType,
                 tenAudioDir& rfenAudioDir);
   protected:

   private:

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOncarAudio::vProcessMediaStart(tenAudioDir enAudioDirection)
       ***************************************************************************/
      /*!
       * \fn      vProcessOnCarMediaStart
       * \brief  Function to process Media playback start request from OnCar device.
       * \param  enAudioDirection : Audio direction type.
       * \retval  None
       **************************************************************************/
       t_Void vProcessOnCarMediaStart(tenAudioDir enAudioDirection);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclOncarAudio::vProcessMediaStart(tenAudioDir enAudioDirection)
        ***************************************************************************/
       /*!
        * \fn      vProcessOnCarGuidanceStart
        * \brief  Function to process Guidance playback start request from OnCar device.
        * \param  enAudioDirection : Audio direction type.
        * \retval  None
        **************************************************************************/
       t_Void vProcessOnCarGuidanceStart(tenAudioDir enAudioDirection);
      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclOnCarAudio::bGetOnCarAudioStreamStype()
       ***************************************************************************/
      /*!
       * \fn      bGetOnCarAudioStreamStype
       * \brief  Function to convert Audio direction to OnCar audio stream type.
       * \param  enAudDir : Audio Direction of type tenAudioDir.
       * \param  rfenOnCarAudioStreamType : reference to OnCar Audio stream type.
       * \retval  Bool value, TRUE if retrieved, FALSE otherwise
       **************************************************************************/
      t_Bool bGetOnCarAudioStreamStype(tenAudioDir enAudDir,
            tenOnCarAudioStreamType& rfenOnCarAudioStreamType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vActivateChannel(tenAudioDir enAudDir)
       ***************************************************************************/
      /*!
       * \fn     vActivateChannel(tenAudioDir enAudDir)
       * \brief  Function to validate and request for activation of Guidance channel
       * \param  enAudDir : Audio direction type
       * \retval None
       **************************************************************************/
      t_Void vActivateChannel(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vDeactivateChannel(tenAudioDir enAudioDirection)
       ***************************************************************************/
      /*!
       * \fn     vDeactivateChannel(tenAudioDir enAudioDirection)
       * \brief  Function to request for deactivation of Guidance channel
       * \param  enAudioDirection : Audio direction type
       * \retval None
       **************************************************************************/
      t_Void vDeactivateChannel(tenAudioDir enAudioDirection);


      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclOnCarAudio::vSendStopAudioResponse(...)
       ***************************************************************************/
      /*!
       * \fn     vSendStopAudioResponse
       * \brief  Sends response for a StopAudio request.
       * \param  enAudDir: Audio direction for which StopAudio response is to be sent
       **************************************************************************/
      t_Void vSendStopAudioResponse(tenAudioDir enAudDir);

      //! Structure object for Function pointers .
      //! This will be used by Audio Manager to register for response callbacks from  Audio
      trAudioCallbacks m_rAudCallbacks;

      //! Pointer to spi_tclOnCarCmdAudio class
      spi_tclOnCarCmdAudio* m_poOnCarCmdAudio;

      //! To remove hard coding of number of streams.
      //! For every audio stream from MD, maintain the state of playback
      tenOnCarStreamState m_aenStreamState[cou32OnCarMaxAudioStreams];

      //! Audio Channel Status used for every stream
      tenAudioChannelStatus m_aenChannelStatus[e8AUD_INVALID];

      //! Lock for Audio Streams and Activations
      Lock    m_oAudioStreamLock;

      //! Map to store audio pipeline configuration for different audio streams
      tmapAudioPipeConfig m_mapAudioPipeConfig;
      
      tenOnCarMediaPlaybackState m_enOnCarMediaPlaybackState;
};
#endif // SPI_TCLONCARAUDIO_H
