/***********************************************************************/
/*!
 * \file  spi_tclOnCarAppMngr.cpp
 * \brief OnCar Application Manager
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Application Manager
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 09.02.2018  | jun5kor               | Initial Version
 01.06.2018  | Rishav Sardar               | Metadata Implementation
 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "spi_tclOnCarAppMngr.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclOnCarCmdMetadata.h"
#include <vector>

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_APPMNGR
#include "trcGenProj/Header/spi_tclOnCarAppMngr.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
static const t_U32 scou32UnknownDevId = 0;

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
//static t_U32 su32CurSelectedDevId = 0;
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAppMngr::spi_tclOnCarAppMngr()
 ***************************************************************************/
spi_tclOnCarAppMngr::spi_tclOnCarAppMngr():m_u32SelectedDevId(scou32UnknownDevId)
{
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarAppMngr::spi_tclOnCarAppMngr() entered "));
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarAppMngr::spi_tclOnCarAppMngr() left "));
}//!end of spi_tclOnCarAppMngr()

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAppMngr::~spi_tclOnCarAppMngr()
 ***************************************************************************/
spi_tclOnCarAppMngr::~spi_tclOnCarAppMngr()
{
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarAppMngr::~spi_tclOnCarAppMngr() entered "));
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarAppMngr::~spi_tclOnCarAppMngr() left "));
}//!end of ~spi_tclOnCarAppMngr()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarAppMngr::bInitialize()
 ***************************************************************************/
t_Bool spi_tclOnCarAppMngr::bInitialize()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::bInitialize() entered "));
   t_Bool bInitSuccess = false;
   spi_tclOnCarManager *poOnCarManager =  spi_tclOnCarManager::getInstance();
   if(poOnCarManager && (poOnCarManager->bRegisterObject((spi_tclOnCarRespMetadata*) this)))
   {
       bInitSuccess = true;
   }//if(poOnCarManager)
   return bInitSuccess;
   ETG_TRACE_USR2(("[FUNC]:spi_tclOnCarAppMngr::bInitialise() left with result = %d", bInitSuccess));
}//!end of bInitialize()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vUninitialize()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vUnInitialize()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vUnInitialize() entered "));
   t_Bool bUnInitSuccess = false;
   spi_tclOnCarManager *poOnCarManager =  spi_tclOnCarManager::getInstance();
   if(poOnCarManager && (poOnCarManager->bUnRegisterObject((spi_tclOnCarRespMetadata*) this)))
   {
       bUnInitSuccess = true;
   }//if(poOnCarManager)
   ETG_TRACE_USR2(("[FUNC]:spi_tclOnCarAppMngr::vUnInitialize() left with result = %d", bUnInitSuccess));
}//!end of vUnInitialize()

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclOnCarAppMngr::vRegisterAppMngrCallbacks()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vRegisterAppMngrCallbacks(const trAppMngrCallbacks& corfrAppMngrCbs)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vRegisterAppMngrCallbacks() entered "));
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vRegisterAppMngrCallbacks() left "));
   m_rAppMngrCallbacks = corfrAppMngrCbs;
}//!end of vRegisterAppMngrCallbacks()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR2(("[FUNC]:spi_tclOnCarAppMngr::vSelectDevice() entered : Dev-0x%x, ConnReq - %d",
                    corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ, corfrSelectReq.m_enDevConnReq)));

   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdMetadata *poCmdMetadata = (NULL != poOnCarManager)?
           (poOnCarManager->poGetMetadataInstance()):(NULL);

   if(e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
   {
       m_u32SelectedDevId = corfrSelectReq.m_u32DeviceHandle;
       if(poCmdMetadata)
       {
           t_Bool bInitSuccess = false;
           bInitSuccess = poCmdMetadata->bInitialiseMetadataEndpoint();
           ETG_TRACE_USR2(("[DESC]:spi_tclOnCarAppMngr::vSelectDevice() : Metadata Initialisation result:%d",bInitSuccess));
       }//if(poCmdMetadata)
   }
   else
   {
       m_u32SelectedDevId = 0;
       if(poCmdMetadata)
       {
           poCmdMetadata->vUninitialiseMetadataEndpoint();
       }//if(poCmdMetadata)
   }//if(e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)

   //Always return true, there is nothing major to send device selection failure
   if (m_rAppMngrCallbacks.fpvSelectDeviceResult)
   {
      (m_rAppMngrCallbacks.fpvSelectDeviceResult)(true);
   }//if (m_rAppMngrCallbacks.fpvSelectDeviceResult)
   ETG_TRACE_USR2(("[FUNC]:spi_tclOnCarAppMngr::vSelectDevice() left "));
}//!end of vSelectDevice()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarAppMngr::bLaunchApp()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vLaunchApp(const t_U32 cou32DevId, t_U32 u32AppId,
                                       const trUserContext& rfrcUsrCntxt, tenDiPOAppType enDiPOAppType,
                                       t_String szTelephoneNumber, tenEcnrSetting enEcnrSetting)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclOnCarAppMngr::vLaunchApp-Dev-0x%x App-0x%x ", cou32DevId, u32AppId));
   SPI_INTENTIONALLY_UNUSED(enEcnrSetting);
   SPI_INTENTIONALLY_UNUSED(szTelephoneNumber);
   
   spi_tclOnCarManager* poOncarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdInput* poOncarCmdInputInstance = (NULL != poOncarManager) ?
            (poOncarManager->poGetInputInstance()) : (NULL);

   if (NULL != poOncarCmdInputInstance)
   {
      switch (enDiPOAppType)
      {
         case e8DIPO_NOT_USED:
         case e8DIPO_NO_URL:
         {
            // Just ask for video focus.
            ETG_TRACE_USR1((" spi_tclOnCarAppMngr::vLaunchApp:NO URL"));
         }
            break;
         case e8DIPO_MAPS:
         {
            //Send Key code for Navigation
            poOncarCmdInputInstance->vReportkey(cou32DevId, e8KEY_PRESS, e32APP_KEYCODE_NAVIGATION);
            poOncarCmdInputInstance->vReportkey(cou32DevId, e8KEY_RELEASE, e32APP_KEYCODE_NAVIGATION);
         }
            break;
         case e8DIPO_MUSIC:
         {
            //Send Key event for Music
            poOncarCmdInputInstance->vReportkey(cou32DevId, e8KEY_PRESS, e32APP_KEYCODE_MEDIA);
            poOncarCmdInputInstance->vReportkey(cou32DevId, e8KEY_RELEASE, e32APP_KEYCODE_MEDIA);
         }
            break;
         default:
         {
            // Unsupported Application Type
            ETG_TRACE_USR1((" spi_tclAAPAppMngr::vLaunchApp:Unsupported"));
         }
            break;
      } //switch (enDiPOAppType)
   } //if (NULL != poOncarCmdInputInstance)
   
   if (NULL != m_rAppMngrCallbacks.fpvLaunchAppResult)
   {
      (m_rAppMngrCallbacks.fpvLaunchAppResult)(cou32DevId, u32AppId,
               enDiPOAppType, e8NO_ERRORS, rfrcUsrCntxt);
   }//if( NULL != m_rAppMngrCallbacks.fpvLaunchAppResult)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vTerminateApp()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vTerminateApp(const t_U32 cou32devId, const t_U32 cou32AppId,
         const trUserContext& rfrcUsrCntxt)
{
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vTerminateApp() entered : Dev-0x%x App-0x%x \n", cou32devId, cou32AppId));
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vTerminateApp() left \n"));
   //! Add code
}//!end of vTerminateApp()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vGetAppDetailsList()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vGetAppDetailsList(const t_U32 cou32DevId, t_U32& u32NumAppInfoList, std::vector<
         trAppDetails>& corfvecrAppDetailsList)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vGetAppDetailsList() entered : Dev-0x%x NumAppInfoList-0x%x",
                   cou32DevId,u32NumAppInfoList));
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(u32NumAppInfoList);
   SPI_INTENTIONALLY_UNUSED(corfvecrAppDetailsList);
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vGetAppDetailsList() left \n"));
/*
   trAppDetails rAppDetails;
   rAppDetails.u32AppHandle = 1;
   rAppDetails.szAppName = "Launcher";

   corfvecrAppDetailsList.push_back(rAppDetails);
*/

}//!end of vGetAppDetailsList()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vGetAppIconData()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vGetAppIconData(t_String szAppIconUrl, const trUserContext& rfrcUsrCntxt)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vGetAppIconData() entered : AppIconUrl = %s",szAppIconUrl.c_str()));
    SPI_INTENTIONALLY_UNUSED(szAppIconUrl);
   //Post response to HMI
   if (m_rAppMngrCallbacks.fpvCbAppIconDataResult)
   {
      (m_rAppMngrCallbacks.fpvCbAppIconDataResult)(e8ICON_INVALID, NULL, 0, rfrcUsrCntxt);
   }//if (m_rAppMngrCallbacks.fpvCbAppIconDataResult)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vGetAppIconData() left \n"));
}//!end of vGetAppIconData()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::bCheckAppValidity()
 ***************************************************************************/
t_Bool spi_tclOnCarAppMngr::bCheckAppValidity(const t_U32 cou32DevId, const t_U32 cou32AppId)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::bCheckAppValidity() entered : Dev-0x%x AppId-0x%x",
                    cou32DevId,cou32AppId));
    SPI_INTENTIONALLY_UNUSED(cou32DevId);
    SPI_INTENTIONALLY_UNUSED(cou32AppId);
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::bCheckAppValidity() left "));
    return true;
}//!end of bCheckAppValidity()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vSetVehicleConfig()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vOnSelectDeviceResult(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vOnSelectDeviceResult() entered : Dev-0x%x",cou32DevId));
   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdMetadata *poCmdMetadata = (NULL !=  poOnCarManager) ?
                     (poOnCarManager->poGetMetadataInstance()): (NULL);
   if(e8FAILURE == coenRespCode && e8DEVCONNREQ_SELECT == coenConnReq && (poCmdMetadata))
   {
       poCmdMetadata->vUninitialiseMetadataEndpoint();
       poCmdMetadata->DestroyMetadataEndpointInstance();
   }//if(e8FAILURE == coenRespCode && e8DEVCONNREQ_SELECT == coenConnReq && NULL != poCmdMetadata)
   if(e8DEVCONNREQ_DESELECT == coenConnReq && (poCmdMetadata))
   {
       poCmdMetadata->DestroyMetadataEndpointInstance();
   }//if(e8DEVCONNREQ_DESELECT == coenConnReq && (poCmdMetadata))
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vOnSelectDeviceResult() left "));
}//!end of vOnSelectDeviceResult()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vSetVehicleConfig()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vSetVehicleConfig(tenVehicleConfiguration enVehicleConfig, t_Bool bSetConfig)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetVehicleConfig() entered : SetConfig = %d",bSetConfig));
   SPI_INTENTIONALLY_UNUSED(bSetConfig);
   SPI_INTENTIONALLY_UNUSED(enVehicleConfig);
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetVehicleConfig() left "));
}//!end of vSetVehicleConfig()
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::bGetAudioHandlingCaps()
 ***************************************************************************/
t_Bool spi_tclOnCarAppMngr::bGetAudioHandlingCaps()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::bGetAudioHandlingCaps() entered "));
   t_Bool bIsAdvAudioCapable = false;
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::bGetAudioHandlingCaps() left "));
   return bIsAdvAudioCapable;
}//!end of bGetAudioHandlingCaps()
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate(const trOnCarMediaMetadata* rOnCarMediaMetadata)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() entered"));
    trUserContext rfcorUsrCntxt;
    m_rAppMediaMetaData.bMediaMetadataValid = true;
    if(rOnCarMediaMetadata)
    {
        ETG_TRACE_USR4(("size of album art is %d", (rOnCarMediaMetadata->u8AlbumArt).size()));
        m_rAppMediaMetaData.szArtist.assign(rOnCarMediaMetadata->cArtist,STR_ONCAR_METADATA_LENGTH);
        m_rAppMediaMetaData.szAlbum.assign(rOnCarMediaMetadata->cAlbum,STR_ONCAR_METADATA_LENGTH);
        m_rAppMediaMetaData.szTitle.assign(rOnCarMediaMetadata->cSong,STR_ONCAR_METADATA_LENGTH);
        m_rAppMediaMetaData.szImageMIMEType = "";
        m_rAppMediaMetaData.rAppMediaAlbumArt.u8Items.clear();
        m_rAppMediaMetaData.szImageMIMEType = "PNG";
        m_rAppMediaMetaData.u32ImageSize = rOnCarMediaMetadata->u8AlbumArt.size();
        if(!rOnCarMediaMetadata->u8AlbumArt.empty())
        {
            copy(rOnCarMediaMetadata->u8AlbumArt.begin(), rOnCarMediaMetadata->u8AlbumArt.end(), back_inserter(m_rAppMediaMetaData.rAppMediaAlbumArt.u8Items));
            
        }//if(!rOnCarMediaMetadata->u8AlbumArt.empty())
        m_rAppMediaMetaData.u8MediaRating = static_cast<t_U8> (rOnCarMediaMetadata->u32Rating);
        m_rAppMediaPlaytime.u32TotalPlayTime = rOnCarMediaMetadata->u32DurationSeconds;

        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : MetadataValid  = %d, \n", ETG_ENUM(BOOL, m_rAppMediaMetaData.bMediaMetadataValid)));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : Song  = %s", m_rAppMediaMetaData.szTitle.c_str()));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : Album  = %s",m_rAppMediaMetaData.szAlbum.c_str()));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : Artist  = %s",m_rAppMediaMetaData.szArtist.c_str()));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : MediaRating  = %d ", m_rAppMediaMetaData.u8MediaRating));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : Image MIME Type  = %s ", m_rAppMediaMetaData.szImageMIMEType.c_str()));
        ETG_TRACE_USR4(("[DESC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() : TotalPlayTime  = %d ", m_rAppMediaPlaytime.u32TotalPlayTime));
    }//if(rOnCarMediaMetadata)
    //! Forward metadata to Connection Manager
    if (m_rAppMngrCallbacks.fvAppMediaMetaData)
    {
      (m_rAppMngrCallbacks.fvAppMediaMetaData)(m_rAppMediaMetaData, rfcorUsrCntxt);
    }//if (m_rDiPoConnCallbacks.fvAppMetadata)
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vPostOnCarMediaMetadataUpdate() entered"));
}//!end of vPostOnCarMediaMetadataUpdate()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAppMngr::vPostOnCarMediaPlaybackStatusUpdate()
 ***************************************************************************/
t_Void spi_tclOnCarAppMngr::vPostOnCarMediaPlaybackStatusUpdate(const trOnCarMediaPlaybackStatus* rOnCarMediaPlaybackStatus)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vPostOnCarMediaPlaybackStatusUpdate() entered "));
    trUserContext rfcorUsrCntxt;
    t_Bool bSolicited = false;
    m_rAppMediaMetaData.bMediaMetadataValid = true;
    if(rOnCarMediaPlaybackStatus)
    {
        m_rAppMediaPlaytime.u32ElapsedPlayTime = rOnCarMediaPlaybackStatus->u32PlaybackSeconds;
        tenMediaPlayBackState enMediaPlayBackState = static_cast<tenMediaPlayBackState> ((rOnCarMediaPlaybackStatus->enState));
        if(enMediaPlayBackState != m_rAppMediaMetaData.enMediaPlayBackState)
        {
            m_rAppMediaMetaData.enMediaPlayBackState = enMediaPlayBackState;
            bSolicited = true;
        }//if(enMediaPlayBackState != m_rAppMediaMetaData.enMediaPlayBackState)
        tenMediaPlayBackShuffleState enMediaPlayBackShuffleState = (rOnCarMediaPlaybackStatus->bShuffle) ?
                                    (e8SHUFFLE_ON) : (e8SHUFFLE_OFF);
        if(enMediaPlayBackShuffleState != m_rAppMediaMetaData.enMediaPlayBackShuffleState)
        {
             m_rAppMediaMetaData.enMediaPlayBackShuffleState = enMediaPlayBackShuffleState;
             bSolicited = true;
        }//if(enMediaPlayBackShuffleState != m_rAppMediaMetaData.enMediaPlayBackShuffleState)
          vSetPlaybackRepeatState(rOnCarMediaPlaybackStatus, bSolicited);

          //Set the playback media source
          vSetPlaybackMediaSource(rOnCarMediaPlaybackStatus, bSolicited);
     }//if(rOnCarMediaPlaybackStatus)
     //! Forward Play time data to Connection Manager
     if (m_rAppMngrCallbacks.fvAppMediaPlaytime)
     {
         (m_rAppMngrCallbacks.fvAppMediaPlaytime)(m_rAppMediaPlaytime, rfcorUsrCntxt);
     }//if (m_rDiPoConnCallbacks.fvAppMediaPlaytime)

     //! Forward meta data to Connection Manager
     if ((m_rAppMngrCallbacks.fvAppMediaMetaData) && (true == bSolicited))
     {
         (m_rAppMngrCallbacks.fvAppMediaMetaData)(m_rAppMediaMetaData, rfcorUsrCntxt);
     }//if ((m_rDiPoConnCallbacks.fvAppMediaMetaData)&& (true == bSolicited))
     ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vPostOnCarMediaPlaybackStatusUpdate() left "));
}//!end of vPostOnCarMediaPlaybackStatusUpdate()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAppMngr::vSetPlaybackRepeatState(const...
***************************************************************************/
t_Void spi_tclOnCarAppMngr::vSetPlaybackRepeatState(
      const trOnCarMediaPlaybackStatus* coprOnCarMediaPlaybackStatus,
      t_Bool& bSolicited)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetPlaybackRepeatState() entered : Solicited Flag = %d",bSolicited));
    // Get Info about repeat all and repeat one states
    tenMediaPlayBackRepeatState enMediaPlayBackRepeatAllState = (true == coprOnCarMediaPlaybackStatus->bRepeatAll)
        ? (e8REPEAT_ALL) : (m_rAppMediaMetaData.enMediaPlayBackRepeatState);

    tenMediaPlayBackRepeatState enMediaPlayBackRepeatOneState = (true == coprOnCarMediaPlaybackStatus->bRepeatOne)
        ? (e8REPEAT_ONE) : (m_rAppMediaMetaData.enMediaPlayBackRepeatState);

    // Set the repeat all state of AppMedia metadata
    if((enMediaPlayBackRepeatAllState != m_rAppMediaMetaData.enMediaPlayBackRepeatState))
    {
        m_rAppMediaMetaData.enMediaPlayBackRepeatState = enMediaPlayBackRepeatAllState;
        bSolicited = true;
        ETG_TRACE_USR2(("[DESC]:spi_tclOnCarAppMngr::vSetPlaybackRepeatState() : Repeat All state = %d",m_rAppMediaMetaData.enMediaPlayBackRepeatState));
    }// if((enMediaPlayBackRepeatAllState != m_rAppMediaMetaData.enMediaPlayBackRepeatState))

    // Set the repeat one state of AppMedia metadata
    else if (enMediaPlayBackRepeatOneState != m_rAppMediaMetaData.enMediaPlayBackRepeatState)
    {
        m_rAppMediaMetaData.enMediaPlayBackRepeatState = enMediaPlayBackRepeatOneState;
        bSolicited = true;
        ETG_TRACE_USR2(("[DESC]:spi_tclOnCarAppMngr::vSetPlaybackRepeatState() : Repeat One state = %d",m_rAppMediaMetaData.enMediaPlayBackRepeatState));
    }// else if (enMediaPlayBackRepeatOneState != m_rAppMediaMetaData.enMediaPlayBackRepeatState)
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetPlaybackRepeatState() left "));
}//!end of vSetPlaybackRepeatState()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAppMngr::vSetPlaybackMediaSource(const...
***************************************************************************/
t_Void spi_tclOnCarAppMngr::vSetPlaybackMediaSource(
      const trOnCarMediaPlaybackStatus* coprOnCarMediaPlaybackStatus,
      t_Bool& bSolicited)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetPlaybackMediaSource() entered : Solicited Flag = %d",bSolicited));
    //Set the media source of the app media metadata
    if (0 != strcmp(m_rAppMediaMetaData.szAppName.c_str(),coprOnCarMediaPlaybackStatus->cMediaSource))
    {
        m_rAppMediaMetaData.szAppName.assign(coprOnCarMediaPlaybackStatus->cMediaSource, STR_METADATA_LENGTH);
        bSolicited = true;
        ETG_TRACE_USR2(("[DESC]:spi_tclAAPAppMngr:: vSetPlaybackMediaSource : Media Source  = %s",m_rAppMediaMetaData.szAppName.c_str()));
    }//if (0 != strcmp(m_rAppMediaMetaData.szAppName.c_str(),coprOnCarMediaPlaybackStatus->cMediaSource))
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarAppMngr::vSetPlaybackMediaSource() left "));
}//!end of vSetPlaybackMediaSource()

// <EOF>

