/**
 *******************************************************************************
 * \file              OnCarVideoEndpoint.h
 * \brief             class handles video frames that must be projected on the head unit.
 *******************************************************************************/

#ifndef ONCAR_VIDEO_ENDPOINT_HPP
#define ONCAR_VIDEO_ENDPOINT_HPP 
 
#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"
  
 /**
 *This class represents an interface that every sink implementation 
 *must subclass if they wish to implement a OnCarVideoEndpoint.  
 */

class IOnCarVideoSinkCallbacks
{
	public:
	  virtual ~IOnCarVideoSinkCallbacks() { }
	  
	 /**
     * Called when MD wants to start video streaming. This callback should be called after
	 * HU has granted video focus to MD through setVideoFocus .It is expected that when this 
	 * callback is called MD will start sending video packets as soon as HU acknowledges 
	 * this request by invoking  acknowledgeStartVideoRequest().
     */
	  virtual void startVideoPlaybackRequestCallback() = 0;
	  
	/**
     * Called when error has occured during video playback.
	 * @param error enum containing error code.
     */
	  virtual void videoErrorCallback(ONCARSINKERROR error) = 0;
	  
	 /**
     * Called when MD wants to stop video streaming. This callback should be called after
	 * HU has granted video focus to MD through setVideoFocus .It is expected that when this 
	 * callback is called MD will stop sending video packets as soon as HU acknowledges 
	 * this request by invoking  acknowledgeStopVideoRequest().
     */
	  virtual void stopVideoPlaybackRequestCallback() = 0;
	  
	/**
     * Called when the OnCar source wishes to change the Video focus.
	 * @param focus Enumeration of VIDEOFOCUS.
	 * @param reason Enumeration of VIDEOFOCUS_REASON.
     */
	  virtual void requestVideoFocusCallback(const VIDEOFOCUS focus,const VIDEOFOCUS_REASON reason) = 0;
	  
	 /**
     * Called by MD to tell HU about its supported resolution.
	 * @param width  width of the projection in pixels.
	 * @param height height of the projection in pixels.
     */
	  virtual void supportedResolution (uint32_t width,uint32_t height) = 0;
};


 /**
 * OnCarVideoEndpoint: class handles video frames that must be projected on the head unit.
 * \image html Video.jpg
 * \image latex Video.eps "OnCar VideoEndPoint Sequence" width=\textwidth
 */

class OnCarVideoEndpoint : public OnCarEndPointBase
{
	public:
	
	/**
     * @brief Constructor
     * @param endpointID End point ID
     */
	   OnCarVideoEndpoint(ONCARENDPOINTID endpointID):OnCarEndPointBase(endpointID)
	   {}
	   
    /**
     * @brief destructor
     */
	   virtual ~OnCarVideoEndpoint(){}

	/**
     * Set the Video Configuration like width, height etc
     * @param config reference to object of VideoConfig type struct
     */ 
     virtual void setVideoConfig(const VideoConfig &config) = 0;
	   
	 /**
     * When this API is invoked by HU, OnCar sink app should start projecting video.
	 * This is the acknowledgement from HU side for requested startVideoPlaybackRequestCallback.
     */
     virtual void acknowledgeStartVideoRequest() = 0;
	   
	/**
     * When this API is invoked by HU, OnCar sink app should stop sending video packets to HU.
	 * This is the acknowledgement from HU side for requested stopVideoPlaybackRequestCallback.
     */
     virtual void acknowledgeStopVideoRequest() = 0;
	   
     /**
     * Called by HU to set MD's video focus.
	 * This API basically informs MD that whether MD is allowed to project video or not.
	 * If MD wants to project video, MD should send vRequestVideoFocusCallback() and should wait
	 * for HU to invoke this API. HU can also invoke this API if HU wants to inform MD about
	 * change in ownership of video resource and then MD should take proper action.
	 * @param focus video focus type.
     */
     virtual void setVideoFocus(VIDEOFOCUS focus) = 0;
	   
	/**
     * Register callback interface to receive Video-related notifications.
     * @param callbacks object implementing the IOnCarVideoCallbacks interface
	 */
     virtual void registerCallbacks(const shared_ptr <IOnCarVideoSinkCallbacks> & callbacks) = 0;
};

#endif //ONCAR_VIDEO_ENDPOINT_HPP