/**
 *******************************************************************************
 * \file              OnCarSensorEndpoint.h
 * \brief             class handles location and speed data that must be sent to MD.
 *******************************************************************************/

#ifndef ONCAR_SENSOR_ENDPOINT_HPP
#define ONCAR_SENSOR_ENDPOINT_HPP
 
#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"

 /**
 *OnCarSensorEndpoint: class handles location and speed data that must be sent to MD.
 */
class OnCarSensorEndpoint : public OnCarEndPointBase
{
	public:
	
	/**
     * @brief Constructor
     * @param endpointID End point ID
     */
	OnCarSensorEndpoint(ONCARENDPOINTID endpointID):OnCarEndPointBase(endpointID)
	{}
	
    /**
     * @brief destructor
     */
	virtual ~OnCarSensorEndpoint(){}

   /**
    * Called by HU to report DayNightMode.
    * @param mode Enumeration of DAYNIGHT_MODE 
    */
    virtual void reportDayNightMode(const DAYNIGHT_MODE mode) = 0;
	
   /**
    * Called by HU to report vehicle movement state.
    * @param state Enumeration of DRIVING_STATE 
    */
	virtual void reportVehicleDrivingState(const DRIVING_STATE state) = 0;
	
	/**
     * Use this function to report the current speed of the vehicle. The value reported here might
     * be used in dead reckoning the position of the vehicle in the event of a GPS signal loss.
     * @param speedE3 The speed in m/s multiplied by 1e3..
     */
	virtual void reportSpeedData(int32_t speedE3) = 0;
	   
	/**
     * Use this method to report a GPS location. The eX multiplication is to allow for
     * fixed point representation of decimal values using an int32. For example, the value
     * 3.1415 becomes 31415000 in E7 notation and can be represented as an integer.
     * @param timestamp The timestamp of this GPS location in nanoseconds.
     * @param latitudeE7 Latitude value [-90.0, +90.0] multiplied by 1e7.
     * @param longitudeE7 Longitude value [-180.0, +180.0) multiplied by 1e7.
     * @param hasAccuracy True if accuracy of this location is known and accuracyE3 param is valid.
     * @param accuracyE3 Horizontal 68% radius meters value multiplied by 1e3.
     * @param hasAltitude True if the altitudeE2 param should be considered, false otherwise.
     * @param altitudeE2 The altitude in meters above sea level multipled by 1e2.
     * @param hasSpeed True if the speedE3 param should be considered, false otherwise.
     * @param speedE3 The speed in m/s absolute velocity multiplied by 1e3.
     * @param hasBearing True if the bearingE6 param should be considered, false otherwise.
     * @param bearingE6 The compass bearing [0, 360) multiplied by 1e6.
	 *
	 * \note This method is kept for future use. Will be used if OnCar requires location data
     */
    virtual void reportLocationData(uint64_t timestamp, int32_t latitudeE7, int32_t longitudeE7,
            bool hasAccuracy, int32_t accuracyE3, bool hasAltitude, int32_t altitudeE2,
            bool hasSpeed, int32_t speedE3, bool hasBearing, int32_t bearingE6) = 0;

	/**
     * Use this function to set drive mode restriction.
     * @param driveModeRestrictions Indicates the level of restrictions available and is defined as a bitmask.
	 *
	    Bit Position | Description
	   --------------|----------------
        0th bit      | VideoPlayback (Value 0 - VideoPlayback not restricted, Value 1 - VideoPlayback restricted) 
        1st bit      | TextInput (Value 0 - TextInput not restricted, Value 1 - TextInput restricted) 
        2nd bit      | VoiceInput(Value 0 - VoiceInput not restricted, Value 1 - VoiceInput restricted) 
        3rd bit      | SetUpAndConfiguration (Value 0 - SetUpAndConfiguration not restricted, Value 1 - SetUpAndConfiguration restricted) 
        4th bit      | MessageLength (Value 0 - Do not limit display message length, Value 1 - Limit display message length)  
     */
    virtual void setDriveModeRestriction (uint16_t driveModeRestrictions) = 0;
	 
  /**
   * Called by HU to report which gear the vehicle is in. 
   * @param gear Enumeration of ONCAR_GEARINFO
   */
     virtual void reportGearData(ONCAR_GEARINFO gear) = 0;
};

#endif //ONCAR_SENSOR_ENDPOINT_HPP
