/*!
 *******************************************************************************
 * \file             spi_tclBluetoothPolicyBase.h
 * \brief            Base Class for Bluetooth Policy
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Base Class for Bluetooth Policy
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 21.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 24.11.2014 |  Ramya Murthy (RBEI/ECP2)         | Added BT block/unblock

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef SPI_TCLBLUETOOTHPOLICYBASE_H
#define SPI_TCLBLUETOOTHPOLICYBASE_H

#include "BaseTypes.h"
#include "spi_BluetoothTypedefs.h"

/**
 *  class definitions.
 */


/**
 * This class is the base class for the Bluetooth policy class which implements the
 * Bluetooth interface for different projects.
 */
class spi_tclBluetoothPolicyBase
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothPolicyBase::spi_tclBluetoothPolicyBase();
   ***************************************************************************/
   /*!
   * \fn      spi_tclBluetoothPolicyBase()
   * \brief   Parameterised Constructor
   * \param   NONE
   **************************************************************************/
   spi_tclBluetoothPolicyBase(){}

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothPolicyBase::~spi_tclBluetoothPolicyBase();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclBluetoothPolicyBase()
   * \brief   Virtual Destructor
   **************************************************************************/
   virtual ~spi_tclBluetoothPolicyBase(){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vRegisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForProperties()
   * \brief   Registers for interested properties to Bluetooth Service.
   **************************************************************************/
   virtual t_Void vRegisterForProperties() {};

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vUnregisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vUnregisterForProperties()
   * \brief   Registers for interested properties to Bluetooth Service.
   **************************************************************************/
   virtual t_Void vUnregisterForProperties() {};

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterCallbacks(trBluetoothCallbacks rBTRespCallbacks)
   * \brief   Interface to register for BT connection callbacks.
   *          Optional interface to be implemented.
   * \param   [IN] rBTRespCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterCallbacks(trBluetoothCallbacks rBTRespCallbacks) {};

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vRegisterPairingInfoCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterPairingInfoCallbacks(trBluetoothPairingCallbacks rBTPairInfoCallbacks)
   * \brief   Interface to register for BT pairing info callbacks.
   *          Optional interface to be implemented.
   * \param   [IN] rBTPairInfoCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterPairingInfoCallbacks(trBluetoothPairingCallbacks rBTPairInfoCallbacks) {};

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vRegisterDeviceNameCallback()
   ***************************************************************************/
   /*!
   * \fn      vRegisterDeviceNameCallback(trBTDeviceNameCbInfo rBTDeviceNameCbInfo)
   * \brief   Interface to register for device name callback of a specific device
   *          Optional interface to be implemented.
   * \param   [IN] rBTDeviceNameCbInfo: Callback info structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterDeviceNameCallback(trBTDeviceNameCbInfo rBTDeviceNameCbInfo) {};

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetoothPolicyBase::bGetPairingStatus(const...)
   ***************************************************************************/
   /*!
   * \fn      bGetPairingStatus(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request from the Bluetooth Manager to the Bluetooth service
   *          for the pairing status of a BT device.
   *          Optional interface to be implemented.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  Bool value: TRUE - if Pairing is required, else FALSE
   **************************************************************************/
   virtual t_Bool bGetPairingStatus(const t_String& rfcoszDeviceBTAddress) { return true; };
   
   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclBluetoothPolicyBase::u32GetBTDeviceHandle(const...)
   ***************************************************************************/
   /*!
   * \fn      u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request from Bluetooth Manager to the Bluetooth Client
   *          for the BT DeviceHandle of a device.
   *          Optional interface to be implemented.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  t_U32 : BT device handle of device with BT address = rfcoszDeviceBTAddress.
   *          If the device is not found in BT DeviceList, zero is returned.
   **************************************************************************/
   virtual t_U32 u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)  { return 0; };
   
   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclBluetoothPolicyBase::vGetBTProfileStatus(...)
   ***************************************************************************/
   /*!
   * \fn      vGetBTProfileStatus(t_String szBTAddress)
   * \brief   Request from Bluetooth Manager to the Bluetooth Client
   *          for the BT DeviceHandle of a device.
   *          Optional interface to be implemented.
   * \param   [IN] u8DeviceHandle: BT device handle.
   * \param   [OUT] rfbIsHFPConn: HFP connection status.
   * \param   [OUT] rfbIsA2DPConn: A2DP connection status.
   * \retval  NONE
   **************************************************************************/   
   virtual t_Void vGetBTProfileStatus(t_U8 u8DeviceHandle, 
         t_Bool& rfbIsHFPConn, t_Bool& rfbIsA2DPConn) = 0;
		 

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothPolicyBase::szGetBTDeviceName()
   ***************************************************************************/
   /*!
   * \fn      szGetBTDeviceName()
   * \brief   Interface to fetch name of a BT device.
   * \param   [IN] rfcszDeviceBTAddress: BT address of device.
   * \retval  t_String : BT device name of requested device.
   *            If device name is not available, returns empty string.
   **************************************************************************/
   virtual t_String szGetBTDeviceName(const t_String& rfcszDeviceBTAddress)  { return ""; };
   
      /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothPolicyBase::szGetVehicleBTFriendlyName()
   ***************************************************************************/
   /*!
   * \fn      szGetVehicleBTFriendlyName()
   * \brief   Interface to fetch BT friendly name of vehicle
   * \retval  t_String : HU's BT friendly name.
   **************************************************************************/
   t_String szGetVehicleBTFriendlyName(){return ""; };

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetoothPolicyBase::bGetBTServiceAvailability(...)
   ***************************************************************************/
   /*!
   * \fn      bGetBTServiceAvailability()
   * \brief   Interface to check Bluetooth Serviec Availability in HU.
   *          Optional interface to be implemented.
   * \param   None
   * \retval  t_Bool : True - if BT Service is available, else false.
   **************************************************************************/
   virtual t_Bool bGetBTServiceAvailability() { return true; };
   
   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclBluetoothPolicyBase::bSetBTLimitationMode(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a Bluetooth limitation in HU.
   *          Optional interface to be implemented.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool : True - if request to set limitation mode is sent successfully,
   *              else false.
   **************************************************************************/
   virtual t_Bool bSetBTLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName,
         tenBTTechnology enTechnology,
         tenBTCommunicationChannel enCommChannel,
         tenBTLimitationAction enAction)
   {
      SPI_INTENTIONALLY_UNUSED(rfcszBTAddress);
      SPI_INTENTIONALLY_UNUSED(rfcszBTDeviceName);
      SPI_INTENTIONALLY_UNUSED(enTechnology);
      SPI_INTENTIONALLY_UNUSED(enCommChannel);
      return true;
   }

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothPolicyBase::szGetBtMacAddress()
   ***************************************************************************/
   /*!
   * \fn      szGetBtMacAddress()
   * \brief   Interface to fetch name of a BT device.
   * \param   [IN] rfcszDeviceBTAddress: BT address of device.
   * \retval  t_String : BT device name of requested device.
   *            If device name is not available, returns empty string.
   **************************************************************************/
   virtual t_Void vGetBtMacAddress(const t_String& rfcszDeviceFrndName, t_String& rfszMACAddress)  { };
};

#endif //SPI_TCLBLUETOOTHPOLICYBASE_H
