/*!
 *******************************************************************************
 * \file             spi_TBTTypes.h
 * \brief            Type defines used TBT
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Type defines used for TBT
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
03.05.2017  |  Unmukt Jain (RBEI/ECP2)          | Initial Version
24.03.2018  |  Jiji Anna Jacob                  | Added ML TBT types

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef _SPI_TBTTYPES_H_
#define _SPI_TBTTYPES_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "SPITypes.h"
#include <vector>
#include <algorithm>

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

enum tenTBTDistanceUnits
{
    e8TBT_DISTANCE_UNITS_UNKNOWN       = 0,
    e8TBT_DISTANCE_UNITS_METERS        = 1,
    e8TBT_DISTANCE_UNITS_KILOMETERS    = 2,
    e8TBT_DISTANCE_UNITS_KILOMETERS_P1 = 3,
    e8TBT_DISTANCE_UNITS_MILES         = 4,
    e8TBT_DISTANCE_UNITS_MILES_P1      = 5,
    e8TBT_DISTANCE_UNITS_FEET          = 6,
    e8TBT_DISTANCE_UNITS_YARDS         = 7
};

enum tenDiPoManeuverType
{
    e8DIPO_MANEUVER_TYPE_NO_TURN                           = 0,
    e8DIPO_MANEUVER_TYPE_LEFT_TURN                         = 1,
    e8DIPO_MANEUVER_TYPE_RIGHT_TURN                        = 2,
    e8DIPO_MANEUVER_TYPE_STRAIGHT_AHEAD                    = 3,
    e8DIPO_MANEUVER_TYPE_MAKE_U_TURN                       = 4,
    e8DIPO_MANEUVER_TYPE_CONTINUE                          = 5,
    e8DIPO_MANEUVER_TYPE_ENTER_ROUNDABOUT                  = 6,
    e8DIPO_MANEUVER_TYPE_EXIT_ROUNDABOUT                   = 7,
    e8DIPO_MANEUVER_TYPE_OFF_RAMP                          = 8,
    e8DIPO_MANEUVER_TYPE_ON_RAMP                           = 9,
    e8DIPO_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION          = 10,
    e8DIPO_MANEUVER_TYPE_PROCEED_TO_THE_BEGINNING_OF_ROUTE = 11,
    e8DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION             = 12,
    e8DIPO_MANEUVER_TYPE_KEEP_LEFT                         = 13,
    e8DIPO_MANEUVER_TYPE_KEEP_RIGHT                        = 14,
    e8DIPO_MANEUVER_TYPE_ENTER_FERRY                       = 15,
    e8DIPO_MANEUVER_TYPE_EXIT_FERRY                        = 16,
    e8DIPO_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY         = 17,
    e8DIPO_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED_TO_ROUTE  = 18,
    e8DIPO_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_U_TURN     = 19,
    e8DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_LEFT             = 20,
    e8DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_RIGHT            = 21,
    e8DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT             = 22,
    e8DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT            = 23,
    e8DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT        = 24,
    e8DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT       = 25,
    e8DIPO_MANEUVER_TYPE_MAKE_U_TURN_WHEN_POSSIBLE         = 26,
    e8DIPO_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTIONS          = 27,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_1                 = 28,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_2                 = 29,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_3                 = 30,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_4                 = 31,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_5                 = 32,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_6                 = 33,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_7                 = 34,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_8                 = 35,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_9                 = 36,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_10                = 37,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_11                = 38,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_12                = 30,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_13                = 40,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_14                = 41,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_15                = 42,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_16                = 43,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_17                = 44,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_18                = 45,
    e8DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_19                = 46,
    e8DIPO_MANEUVER_TYPE_SHARP_LEFT_TURN                   = 47,
    e8DIPO_MANEUVER_TYPE_SHARP_RIGHT_TURN                  = 48,
    e8DIPO_MANEUVER_TYPE_SLIGHT_LEFT_TURN                  = 49,
    e8DIPO_MANEUVER_TYPE_SLIGHT_RIGHT_TURN                 = 50,
    e8DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY                    = 51,
    e8DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT               = 52,
    e8DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT              = 53
};

enum tenDiPoDrivingSide
{
    e8DIPO_DRIVING_SIDE_RIGHT        = 0,
    e8DIPO_DRIVING_SIDE_LEFT         = 1,
    e8DIPO_DRIVING_SIDE_UNSPECIFIED  = 2
};

enum tenDiPoJunctionType
{
    e8DIPO_JUNCTION_TYPE_SINGLE_INTERSECTION = 0,
    e8DIPO_JUNCTION_TYPE_ROUNDABOUT          = 1,
    e8DIPO_JUNCTION_TYPE_UNSPECIFIED         = 2
};

enum tenDiPoRouteGuidanceState
{
    e8DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET   = 0,
    e8DIPO_ROUTE_GUIDANCE_STATE_ROUTE_SET      = 1,
    e8DIPO_ROUTE_GUIDANCE_STATE_ARRIVED        = 2,
    e8DIPO_ROUTE_GUIDANCE_STATE_LOADING        = 3,
    e8DIPO_ROUTE_GUIDANCE_STATE_LOCATING       = 4,
    e8DIPO_ROUTE_GUIDANCE_STATE_REROUTING      = 5,
    e8DIPO_ROUTE_GUIDANCE_STATE_PROCEEDTOROUTE = 6
};

enum tenDiPoRouteManeuverState
{
    e8DIPO_MANEUVER_STATE_CONTINUE = 0,
    e8DIPO_MANEUVER_STATE_INITIAL  = 1,
    e8DIPO_MANEUVER_STATE_PREPARE  = 2,
    e8DIPO_MANEUVER_STATE_EXECUTE  = 3
};

enum tenAAutoNavigationAppState
{
    AAUTO_STATE_NAV_UNKNOWN   = 0,
    AAUTO_STATE_NAV_ACTIVE    = 1,
    AAUTO_STATE_NAV_NOTACTIVE = 2
};

enum tenAAutoNextTurnEvent
{
    e8AA_NEXT_TURN_UNKNOWN                   = 0,
    e8AA_NEXT_TURN_DEPART                    = 1,
    e8AA_NEXT_TURN_NAME_CHANGE               = 2,
    e8AA_NEXT_TURN_SLIGHT_TURN               = 3,
    e8AA_NEXT_TURN_TURN                      = 4,
    e8AA_NEXT_TURN_SHARP_TURN                = 5,
    e8AA_NEXT_TURN_U_TURN                    = 6,
    e8AA_NEXT_TURN_ON_RAMP                   = 7,
    e8AA_NEXT_TURN_OFF_RAMP                  = 8,
    e8AA_NEXT_TURN_FORK                      = 9,
    e8AA_NEXT_TURN_MERGE                     = 10,
    e8AA_NEXT_TURN_ROUNDABOUT_ENTER          = 11,
    e8AA_NEXT_TURN_ROUNDABOUT_EXIT           = 12,
    e8AA_NEXT_TURN_ROUNDABOUT_ENTER_AND_EXIT = 13,
    e8AA_NEXT_TURN_STRAIGHT                  = 14,
    e8AA_NEXT_TURN_FERRY_BOAT                = 16,
    e8AA_NEXT_TURN_FERRY_TRAIN               = 17,
    e8AA_NEXT_TURN_DESTINATION               = 19
};

enum tenAAutoTurnSide
{
    e8TURN_SIDE_LEFT        = 1,
    e8TURN_SIDE_RIGHT       = 2,
    e8TURN_SIDE_UNSPECIFIED = 3
};

enum tenBDCLNextTurnType
{
    e8CL_NEXT_TURN_UNKNOWN                        = 0,
    e8CL_TURN_KIND_FRONT                          = 1,
    e8CL_TURN_KIND_RIGHT_FRONT                    = 2,
    e8CL_TURN_KIND_RIGHT                          = 3,
    e8CL_TURN_KIND_RIGHT_BACK                     = 4,
    e8CL_TURN_KIND_BACK                           = 5,
    e8CL_TURN_KIND_LEFT_BACK                      = 6,
    e8CL_TURN_KIND_LEFT                           = 7,
    e8CL_TURN_KIND_LEFT_FRONT                     = 8,
    e8CL_TURN_KIND_LEFT_SIDE                      = 9,
    e8CL_TURN_KIND_RIGHT_SIDE                     = 10,
    e8CL_TURN_KIND_BRANCH_LEFT                    = 11,
    e8CL_TURN_KIND_BRANCH_CENTER                  = 12,
    e8CL_TURN_KIND_BRANCH_RIGHT                   = 13,
    e8CL_TURN_KIND_RING                           = 14,
    e8CL_TURN_KIND_INFERRY                        = 15,
    e8CL_TURN_KIND_OUTFERRY                       = 16,
    e8CL_TURN_KIND_TOLLGATE                       = 17,
    e8CL_TURN_KIND_LEFT_SIDE_MAIN                 = 18,
    e8CL_TURN_KIND_RIGHT_SIDE_MAIN                = 19,
    e8CL_TURN_KIND_LEFT_SIDE_IC                   = 20,
    e8CL_TURN_KIND_RIGHT_SIDE_IC                  = 21,
    e8CL_TURN_KIND_LEFT_SIDE_STRAIGHT_IC          = 22,
    e8CL_TURN_KIND_RIGHT_SIDE_STRAIGHT_IC         = 23,
    e8CL_TURN_KIND_LEFT_SIDE_STRAIGHT             = 24,
    e8CL_TURN_KIND_RIGHT_SIDE_STRAIGHT            = 25,
    e8CL_TURN_KIND_BRANCH_LEFT_STRAIGHT           = 26,
    e8CL_TURN_KIND_BRANCH_CENTER_STRAIGHT         = 27,
    e8CL_TURN_KIND_BRANCH_RIGHT_STRAIGHT          = 28,
    e8CL_TURN_KIND_BRANCH_LEFT_IC                 = 29,
    e8CL_TURN_KIND_BRANCH_CENTER_IC               = 30,
    e8CL_TURN_KIND_BRANCH_RIGHT_IC                = 31,
    e8CL_TURN_KIND_BRANCH_LEFT_IC_STRAIGHT        = 32,
    e8CL_TURN_KIND_BRANCH_CENTER_IC_STRAIGHT      = 33,
    e8CL_TURN_KIND_BRANCH_RIGHT_IC_STRAIGHT       = 34,
    e8CL_TURN_KIND_STRAIGHT_2BRANCH_LEFT_BASE     = 35,
    e8CL_TURN_KIND_STRAIGHT_2BRANCH_RIGHT_BASE    = 36,
    e8CL_TURN_KIND_STRAIGHT_3BRANCH_LEFT_BASE     = 37,
    e8CL_TURN_KIND_STRAIGHT_3BRANCH_MIDDLE_BASE   = 38,
    e8CL_TURN_KIND_STRAIGHT_3BRANCH_RIGHT_BASE    = 39,
    e8CL_TURN_KIND_LEFT_2BRANCH_LEFT_BASE         = 40,
    e8CL_TURN_KIND_LEFT_2BRANCH_RIGHT_BASE        = 41,
    e8CL_TURN_KIND_LEFT_3BRANCH_LEFT_BASE         = 42,
    e8CL_TURN_KIND_LEFT_3BRANCH_MIDDLE_BASE       = 43,
    e8CL_TURN_KIND_LEFT_3BRANCH_RIGHT_BASE        = 44,
    e8CL_TURN_KIND_RIGHT_2BRANCH_LEFT_BASE        = 45,
    e8CL_TURN_KIND_RIGHT_2BRANCH_RIGHT_BASE       = 46,
    e8CL_TURN_KIND_RIGHT_3BRANCH_LEFT_BASE        = 47,
    e8CL_TURN_KIND_RIGHT_3BRANCH_MIDDLE_BASE      = 48,
    e8CL_TURN_KIND_RIGHT_3BRANCH_RIGHT_BASE       = 49,
    e8CL_TURN_KIND_LEFT_FRONT_2BRANCH_LEFT_BASE   = 50,
    e8CL_TURN_KIND_LEFT_FRONT_2BRANCH_RIGHT_BASE  = 51,
    e8CL_TURN_KIND_RIGHT_FRONT_2BRANCH_LEFT_BASE  = 52,
    e8CL_TURN_KIND_RIGHT_FRONT_2BRANCH_RIGHT_BASE = 53,
    e8CL_TURN_KIND_BACK_2BRANCH_LEFT_BASE         = 54,
    e8CL_TURN_KIND_BACK_2BRANCH_RIGHT_BASE        = 55,
    e8CL_TURN_KIND_BACK_3BRANCH_LEFT_BASE         = 56,
    e8CL_TURN_KIND_BACK_3BRANCH_MIDDLE_BASE       = 57,
    e8CL_TURN_KIND_BACK_3BRANCH_RIGHT_BASE        = 58
};

enum tenBDCLNavigationAppState
{
    e8BDCL_NAV_STATE_ACTION_INVALID = 0,
    e8BDCL_NAV_STATE_ACTION_SHOW    = 1,
    e8BDCL_NAV_STATE_ACTION_UPDATE  = 2,
    e8BDCL_NAV_STATE_ACTION_HIDE    = 3
};

enum tenBDCLAssistantType
{
    e8BDCL_ASSISTANT_TYPE_INVALID            = 0,
    e8BDCL_ASSISTANT_TYPE_TRAFFICSIGN        = 1,
    e8BDCL_ASSISTANT_TYPE_SPEEDCAMERA        = 2,
    e8BDCL_ASSISTANT_TYPE_TRAFFICLIGHTCAMERA = 3,
    e8BDCL_ASSISTANT_TYPE_PECCANRYCAMERA     = 4,
    e8BDCL_ASSISTANT_TYPE_INTERVALCAMERA     = 5
};

enum tenBDCLTrafficSignType
{
    e8BDCL_TRAFFICSIGNTYPE_INVALID = 0
};


enum tenMLNaviUpdateType
{
   e8ML_NAVIGATION_INFO = 0,
   e8ML_NAVIGATION_NEXT_MANEUVER = 1,
   e8ML_NAVIGATION_SPD_LIMIT = 2,
   e8ML_NAVIGATION_NEXT_DISTANCE = 3,
   e8ML_NAVIGATION_LANE_GUIDANCE = 4,
   e8ML_NAVIGATION_TRIP_INFO = 5,
   e8ML_NAVIGATION_UNKNOWN = 255
};

enum tenMLNaviGuidanceState
{
   E8ML_NAVI_GUIDANCE_UNKNOWN = 0,
   E8ML_NAVI_GUIDANCE_NO_DESTINATION_SET = 1,
   E8ML_NAVI_GUIDANCE_CALCULATING_ROUTE = 2,
   E8ML_NAVI_GUIDANCE_NEW_ROUTE = 3,
   E8ML_NAVI_GUIDANCE_NO_ROUTE = 4,
   E8ML_NAVI_GUIDANCE_NORMAL_OPERATION = 5,
   E8ML_NAVI_GUIDANCE_NORMAL_OFF_ROAD = 6,
   E8ML_NAVI_GUIDANCE_NORMAL_OFF_MAP = 7,
   E8ML_NAVI_GUIDANCE_DESTINATION_AREA = 8,
   E8ML_NAVI_GUIDANCE_DESTINATION_REACHED = 9
};

enum tenMLNaviGuidanceActive
{
   E8ML_NAVI_GUIDANCE_NONE = 0,
   E8ML_NAVI_GUIDANCE_SERVER = 1,
   E8ML_NAVI_GUIDANCE_CLIENT = 2
};

enum tenMLNaviDistanceUnit
{
   E8ML_NAVI_DISTANCE_METER = 0,
   E8ML_NAVI_DISTANCE_KM = 1,
   E8ML_NAVI_DISTANCE_FEET = 2,
   E8ML_NAVI_DISTANCE_YARDS = 3,
   E8ML_NAVI_DISTANCE_MILES = 4,
   E8ML_NAVI_DISTANCE_UNKNOWN = 255
};

enum tenMLNaviManeuverDirection
{
   E8ML_NAVI_MANEUVER_NO_SYMBOL = 0,
   E8ML_NAVI_MANEUVER_NO_INFO = 1,
   E8ML_NAVI_MANEUVER_FOLLOW_STREET = 2,
   E8ML_NAVI_MANEUVER_TURN_STRAIGHT = 3,
   E8ML_NAVI_MANEUVER_TURN_SLIGHT_RIGHT = 4,
   E8ML_NAVI_MANEUVER_TURN_SLIGHT_LEFT = 5,
   E8ML_NAVI_MANEUVER_TURN_RIGHT = 6,
   E8ML_NAVI_MANEUVER_TURN_LEFT = 7,
   E8ML_NAVI_MANEUVER_TURN_SHARP_RIGHT = 8,
   E8ML_NAVI_MANEUVER_TURN_SHARP_LEFT = 9,
   E8ML_NAVI_MANEUVER_UTURN_RIGHT = 10,
   E8ML_NAVI_MANEUVER_UTURN_LEFT = 11,
   E8ML_NAVI_MANEUVER_KEEP_RIGHT = 12,
   E8ML_NAVI_MANEUVER_KEEP_LEFT = 13,
   E8ML_NAVI_MANEUVER_EXIT_RIGHT = 14,
   E8ML_NAVI_MANEUVER_EXIT_LEFT = 15,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_SLIGHT_RIGHT_AND_SLIGHT_RIGHT_AGAIN = 16,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_SLIGHT_LEFT_AND_SLIGHT_LEFT_AGAIN = 17,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_SLIGHT_RIGHT_AND_STRAIGHT = 18,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_SLIGHT_LEFT_AND_STRAIGHT = 19,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_RIGHT_AND_RIGHT = 20,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_LEFT_AND_LEFT = 21,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_RIGHT_AND_LEFT = 22,
   E8ML_NAVI_MANEUVER_DOUBLE_TURN_LEFT_AND_RIGHT = 23,
   E8ML_NAVI_MANEUVER_MERGE = 24,
   E8ML_NAVI_MANEUVER_HIGHWAY_FOLLOW = 25,
   E8ML_NAVI_MANEUVER_HIGHWAY_SLIGHT_RIGHT = 26,
   E8ML_NAVI_MANEUVER_HIGHWAY_SLIGHT_LEFT = 27,
   E8ML_NAVI_MANEUVER_HIGHWAY_DOUBLE_TURN_SLIGHT_RIGHT_AND_SLIGHT_RIGHT = 28,
   E8ML_NAVI_MANEUVER_HIGHWAY_DOUBLE_TURN_SLIGHT_LEFT_AND_SLIGHT_LEFT = 29,
   E8ML_NAVI_MANEUVER_HIGHWAY_DOUBLE_TURN_SLIGHT_RIGHT_AND_STRAIGHT = 30,
   E8ML_NAVI_MANEUVER_HIGHWAY_DOUBLE_TURN_SLIGHT_LEFT_AND_STRAIGHT = 31,
   E8ML_NAVI_MANEUVER_MICHIGAN_TURN_VARIANT_1_RIGHT = 32,
   E8ML_NAVI_MANEUVER_MICHIGAN_TURN_VARIANT_1_LEFT = 33,
   E8ML_NAVI_MANEUVER_MICHIGAN_TURN_VARIANT_2_RIGHT = 34,
   E8ML_NAVI_MANEUVER_MICHIGAN_TURN_VARIANT_2_LEFT = 35,
   E8ML_NAVI_MANEUVER_TUNNEL_ENTER = 36,
   E8ML_NAVI_MANEUVER_TUNNEL = 37,
   E8ML_NAVI_MANEUVER_TUNNEL_EXIT = 38,
   E8ML_NAVI_MANEUVER_FERRY_ENTER = 39,
   E8ML_NAVI_MANEUVER_FERRY = 40,
   E8ML_NAVI_MANEUVER_FERRY_EXIT = 41,
   E8ML_NAVI_MANEUVER_PUBLIC_TRANSPORTATION = 42,
   E8ML_NAVI_MANEUVER_WALK = 43,
   E8ML_NAVI_MANEUVER_COMPASS = 44,
   E8ML_NAVI_MANEUVER_DIRECTION_TO_DESTINATION = 45,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_RIGHT_EXIT_NOW = 46,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_LEFT_EXIT_NOW = 47,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_RIGHT_UNKNOWN_EXIT_NUMBER = 48,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_RIGHT_KNOWN_EXIT_NUMBER = 49,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_LEFT_UNKNOWN_EXIT_NUMBER = 64,
   E8ML_NAVI_MANEUVER_ROUNDABOUT_LEFT_KNOWN_EXIT_NUMBER = 65,
   E8ML_NAVI_MANEUVER_FINAL_DESTINATION = 80,
   E8ML_NAVI_MANEUVER_FINAL_DESTINATION_ON_THE_RIGHT = 81,
   E8ML_NAVI_MANEUVER_FINAL_DESTINATION_ON_THE_LEFT = 82,
   E8ML_NAVI_MANEUVER_INTERMEDIATE_DESTINATION = 83,
   E8ML_NAVI_MANEUVER_INTERMEDIATE_DESTINATION_ON_THE_RIGHT = 84,
   E8ML_NAVI_MANEUVER_INTERMEDIATE_DESTINATION_ON_THE_LEFT = 85

};

enum tenDisplayComponentId
{
   E8DISPLAY_ID_HEADUNIT_DISPLAY_1 = 1,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_2 = 2,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_3 = 3,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_4 = 4,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_5 = 5,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_6 = 6,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_7 = 7,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_8 = 8,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_9 = 9,
   E8DISPLAY_ID_HEADUNIT_DISPLAY_10 = 10,
   E8DISPLAY_ID_CLUSTER_1 = 11,
   E8DISPLAY_ID_CLUSTER_2 = 12,
   E8DISPLAY_ID_CLUSTER_3 = 13,
   E8DISPLAY_ID_CLUSTER_4 = 14,
   E8DISPLAY_ID_CLUSTER_5 = 15,
   E8DISPLAY_ID_CLUSTER_6 = 16,
   E8DISPLAY_ID_CLUSTER_7 = 17,
   E8DISPLAY_ID_CLUSTER_8 = 18,
   E8DISPLAY_ID_CLUSTER_9 = 19,
   E8DISPLAY_ID_CLUSTER_10 = 20,
   DISPLAY_ID_HEAD_UP_DISPLAY_1 = 21,
   DISPLAY_ID_HEAD_UP_DISPLAY_2 = 22,
   DISPLAY_ID_HEAD_UP_DISPLAY_3 = 23,
   DISPLAY_ID_HEAD_UP_DISPLAY_4 = 24,
   DISPLAY_ID_HEAD_UP_DISPLAY_5 = 25,
   DISPLAY_ID_HEAD_UP_DISPLAY_6 = 26,
   DISPLAY_ID_HEAD_UP_DISPLAY_7 = 27,
   DISPLAY_ID_HEAD_UP_DISPLAY_8 = 28,
   DISPLAY_ID_HEAD_UP_DISPLAY_9 = 29,
   DISPLAY_ID_HEAD_UP_DISPLAY_10 = 30,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_1 = 31,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_2 = 32,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_3 = 33,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_4 = 34,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_5 = 35,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_6 = 36,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_7 = 37,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_8 = 38,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_9 = 39,
   DISPLAY_ID_MISCELLANEOUS_DISPLAY_10 = 40,
   DISPLAY_ID_UNKNOWN = 255
};

//! RegisterRouteGuidanceDisplayConfiguration FI data
struct trDiPORouteGuidanceDisplayComponent
{
    tenDisplayComponentId enDisplayComponent;
    t_String szDisplayName;
    t_U16 u16MaxCurrentRoadNameLength;
    t_U16 u16MaxDestinationNameLength;
    t_U16 u16MaxAfterManeuverRoadNameLength;
    t_U16 u16MaxManeuverDescriptionLength;
    t_U16 u16MaxGuidanceManeuverCapacity;

    trDiPORouteGuidanceDisplayComponent():enDisplayComponent(DISPLAY_ID_UNKNOWN),
                                u16MaxCurrentRoadNameLength(0),
                                u16MaxDestinationNameLength(0),
                                u16MaxAfterManeuverRoadNameLength(0),
                                u16MaxManeuverDescriptionLength(0),
                                u16MaxGuidanceManeuverCapacity(0)
    {}
};

//! StartRouteGuidanceUpdates FI data
struct trStartRouteGuidanceUpdates
{
    std::vector<tenDisplayComponentId> vecDisplayComponent;
    t_Bool bSourceName;
    t_Bool bSourceSupportsRouteGuidance;
    t_Bool bNeedAutoStartStopTBTUpdates;
    t_String szDeviceSerialNumber;

    trStartRouteGuidanceUpdates():bSourceName(false),
                                  bSourceSupportsRouteGuidance(false),
                                  bNeedAutoStartStopTBTUpdates(false)
    {}
};

//! StopRouteGuidanceUpdates FI data
struct trStopRouteGuidanceUpdates
{
    std::vector<tenDisplayComponentId> vecDisplayComponent;
    t_String szDeviceSerialNumber;

    trStopRouteGuidanceUpdates(){}
};

struct trDiPORouteGuidanceUpdate
{
    t_U32 u32DeviceHandle;
    t_U64 u64EstimatedTimeOfArrival;
    std::vector<tenDisplayComponentId> vecDisplayComponentList;
    std::vector<t_U16> vecRouteGuidanceManeuverCurrentList;
    t_U64 u64TimeRemainingToDestination;
    t_String szCurrentRoadName;
    t_String szDestinationName;
    t_String szDistanceRemainingDisplayStr;
    t_String szDistanceToNextManeuverDisplayStr;
    t_String szSourceName;
    t_String szDeviceSerialNumber;
    t_U32 u32DistanceToNextManeuver;
    t_U32 u32DistanceRemaining;
    t_U16 u16RouteGuidanceManeuverCount;
    tenDiPoRouteGuidanceState enRouteGuidanceState;
    tenDiPoRouteManeuverState enManeuverState;
    tenTBTDistanceUnits enDistanceRemainingDisplayUnits;
    tenTBTDistanceUnits enDistanceToNextManeuverDisplayUnits;
    t_Bool bIsRouteGuidanceVisibleInApp;
    t_Bool bIsSourceSupportsRouteGuidance;
    t_Bool bIsDisplayComponentListAvailable;
    t_Bool bIsRouteGuidanceStateAvailable;
    t_Bool bIsManeuverStateAvailable;
    t_Bool bIsCurrentRoadNameAvailable;
    t_Bool bIsDestinationNameAvailable;
    t_Bool bIsEstimatedTimeOfArrivalAvailable;
    t_Bool bIsTimeRemainingToDestAvailable;
    t_Bool bIsDistanceRemainingAvailable;
    t_Bool bIsDistanceRemainingDisplayStrAvailable;
    t_Bool bIsDistanceRemainingDisplayUnitsAvailable;
    t_Bool bIsDistanceToNextManeuverAvailable;
    t_Bool bIsDistanceToNextManeuverDispStrAvailable;
    t_Bool bIsDistanceToNextManeuverDIspUnitsAvailable;
    t_Bool bIsRouteGuidanceManeuverCurrentListAvailable;
    t_Bool bIsRouteGuidanceManeuverCountAvailable;
    t_Bool bIsRouteGuidanceVisibleInAppAvailable;
    t_Bool bIsSourceNameAvailable;
    t_Bool bIsSourceSupportsRouteGuidanceAvailable;

    trDiPORouteGuidanceUpdate():u32DeviceHandle(0),
                                u64EstimatedTimeOfArrival(0),
                                u64TimeRemainingToDestination(0),
                                u32DistanceToNextManeuver(0),
                                u32DistanceRemaining(0),
                                u16RouteGuidanceManeuverCount(0),
                                enRouteGuidanceState(e8DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET),
                                enManeuverState(e8DIPO_MANEUVER_STATE_CONTINUE),
                                enDistanceRemainingDisplayUnits(e8TBT_DISTANCE_UNITS_UNKNOWN),
                                enDistanceToNextManeuverDisplayUnits(e8TBT_DISTANCE_UNITS_UNKNOWN),
                                bIsRouteGuidanceVisibleInApp(false),
                                bIsSourceSupportsRouteGuidance(false),
                                bIsDisplayComponentListAvailable(false),
                                bIsRouteGuidanceStateAvailable(false),
                                bIsManeuverStateAvailable(false),
                                bIsCurrentRoadNameAvailable(false),
                                bIsDestinationNameAvailable(false),
                                bIsEstimatedTimeOfArrivalAvailable(false),
                                bIsTimeRemainingToDestAvailable(false),
                                bIsDistanceRemainingAvailable(false),
                                bIsDistanceRemainingDisplayStrAvailable(false),
                                bIsDistanceRemainingDisplayUnitsAvailable(false),
                                bIsDistanceToNextManeuverAvailable(false),
                                bIsDistanceToNextManeuverDispStrAvailable(false),
                                bIsDistanceToNextManeuverDIspUnitsAvailable(false),
                                bIsRouteGuidanceManeuverCurrentListAvailable(false),
                                bIsRouteGuidanceManeuverCountAvailable(false),
                                bIsRouteGuidanceVisibleInAppAvailable(false),
                                bIsSourceNameAvailable(false),
                                bIsSourceSupportsRouteGuidanceAvailable(false)
    {}
};

struct trDiPORouteGuidanceManeuverUpdate
{
    std::vector<tenDisplayComponentId> vecDisplayComponentList;
    t_String szManeuverDescription;
    t_String szAfterManeuverRoadName;
    t_String szDistanceBetweenManeuverDisplayStr;
    t_String szDeviceSerialNumber;
    t_U32 u32DeviceHandle;
    t_U32 u32DistanceBetweenManeuver;
    t_U16 u16Index;
    std::vector<t_S16> vecJunctionElementAngleList;
    t_S16 s16JunctionElementExitAngle;
    tenDiPoManeuverType enManeuverType;
    tenTBTDistanceUnits enDistanceBetweenManeuverDisplayUnits;
    tenDiPoDrivingSide enDrivingSide;
    tenDiPoJunctionType enJunctionType;
    t_Bool bIsDisplayComponentListAvailable;
    t_Bool bIsIndexAvailable;
    t_Bool bIsManeuverDescriptionAvailable;
    t_Bool bIsManeuverTypeAvailable;
    t_Bool bIsAfterManeuverRoadNameAvailable;
    t_Bool bIsDistanceBetweenManeuverAvailable;
    t_Bool bIsDistanceBetweenManeuverDispStrAvailable;
    t_Bool bIsDistanceBetweenManeuverDispUnitsAvailable;
    t_Bool bIsDrivingSideAvailable;
    t_Bool bIsJunctionTypeAvailable;
    t_Bool bIsJunctionElementAngleAvailable;
    t_Bool bIsJunctionElementExitAngleAvailable;

    trDiPORouteGuidanceManeuverUpdate():u32DeviceHandle(0),
                                        u32DistanceBetweenManeuver(0),
                                        u16Index(0),
                                        s16JunctionElementExitAngle(0),
                                        enManeuverType(e8DIPO_MANEUVER_TYPE_NO_TURN),
                                        enDistanceBetweenManeuverDisplayUnits(e8TBT_DISTANCE_UNITS_UNKNOWN),
                                        enDrivingSide(e8DIPO_DRIVING_SIDE_UNSPECIFIED),
                                        enJunctionType(e8DIPO_JUNCTION_TYPE_UNSPECIFIED),
                                        bIsDisplayComponentListAvailable(false),
                                        bIsIndexAvailable(false),
                                        bIsManeuverDescriptionAvailable(false),
                                        bIsManeuverTypeAvailable(false),
                                        bIsAfterManeuverRoadNameAvailable(false),
                                        bIsDistanceBetweenManeuverAvailable(false),
                                        bIsDistanceBetweenManeuverDispStrAvailable(false),
                                        bIsDistanceBetweenManeuverDispUnitsAvailable(false),
                                        bIsDrivingSideAvailable(false),
                                        bIsJunctionTypeAvailable(false),
                                        bIsJunctionElementAngleAvailable(false),
                                        bIsJunctionElementExitAngleAvailable(false)
    {}
};

#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
struct trCarPlayTBTUpdate
{
    trDiPORouteGuidanceUpdate rDiPORouteGuidanceUpdate;
    trDiPORouteGuidanceManeuverUpdate rDiPORouteGuidanceManeuverUpdate;
    t_Bool isDiPoRouteGuidanceUpdateValid;
    t_Bool isDiPoRouteGuidanceManeuverUpdateValid;

    trCarPlayTBTUpdate():isDiPoRouteGuidanceUpdateValid(false),
                         isDiPoRouteGuidanceManeuverUpdateValid(false)
    {}
};
#endif

struct trAAutoNavigationNextTurnData
{
    std::vector<t_U8> u8Image;
    t_String szRoadName;
    t_S32 s32TurnAngle;
    t_S32 s32TurnNumber;
    tenAAutoTurnSide enTurnSide;
    tenAAutoNextTurnEvent enNextTurnEvent;

    trAAutoNavigationNextTurnData():s32TurnAngle(0),
                                    s32TurnNumber(0),
                                    enTurnSide(e8TURN_SIDE_UNSPECIFIED),
                                    enNextTurnEvent(e8AA_NEXT_TURN_UNKNOWN)
    {}
};

struct trAAutoNavigationNextTurnDistanceData
{
    t_S32 s32DistanceInMeters;
    t_S32 s32TimeInSec;
    t_S32 s32DistanceToManeuver;
    tenTBTDistanceUnits enManeuverDistanceUnits ;

    trAAutoNavigationNextTurnDistanceData():s32DistanceInMeters(0),
                                            s32TimeInSec(0),
                                            s32DistanceToManeuver(0),
                                            enManeuverDistanceUnits(e8TBT_DISTANCE_UNITS_UNKNOWN)
    {}
};

struct trAAutoTBTUpdate
{
    trAAutoNavigationNextTurnData AAutoNavigationNextTurnData;
    trAAutoNavigationNextTurnDistanceData AAutoNavigationNextTurnDistanceData;
    tenAAutoNavigationAppState enAAutoNavigationStatus;
    t_Bool isAAutoNavigationNextTurnDataValid;
    t_Bool isAAutoNavigationNextTurnDistanceDataValid;
    t_Bool isAAutoNavigationStatusValid;

    trAAutoTBTUpdate():enAAutoNavigationStatus(AAUTO_STATE_NAV_UNKNOWN),
                       isAAutoNavigationNextTurnDataValid(false),
                       isAAutoNavigationNextTurnDistanceDataValid(false),
                       isAAutoNavigationStatusValid(false)

    {}
};

struct trCarlifeNaviNextTurnInfo
{
    std::vector<t_U8> u8TurnIconData;
    t_String szRoadName;
    t_U32 u32TotalDistance;
    t_U32 u32RemainDistance;
    tenBDCLNavigationAppState enBDCLNavigationAppState;
    tenBDCLNextTurnType enBDCLNaviTurnType;

    trCarlifeNaviNextTurnInfo():u32TotalDistance(0),
                                u32RemainDistance(0),
                                enBDCLNavigationAppState(e8BDCL_NAV_STATE_ACTION_INVALID),
                                enBDCLNaviTurnType(e8CL_NEXT_TURN_UNKNOWN)
    {}
};

struct trCarlifeNaviAssistantGuideInfo
{
    t_U32 u32TotalDistance;
    t_U32 u32RemainDistance;
    t_U32 u32CameraSpeed;
    tenBDCLNavigationAppState enBDCLAssistantGuideAppState;
    tenBDCLAssistantType enAssistantType;
    tenBDCLTrafficSignType enTrafficSignType;

    trCarlifeNaviAssistantGuideInfo():u32TotalDistance(0),
                                      u32RemainDistance(0),
                                      u32CameraSpeed(0),
                                      enBDCLAssistantGuideAppState(e8BDCL_NAV_STATE_ACTION_INVALID),
                                      enAssistantType(e8BDCL_ASSISTANT_TYPE_INVALID),
                                      enTrafficSignType(e8BDCL_TRAFFICSIGNTYPE_INVALID)
    {}
};

struct trBDCLTBTUpdate
{
    trCarlifeNaviNextTurnInfo CarlifeNaviNextTurnInfo;
    trCarlifeNaviAssistantGuideInfo CarlifeNaviAssistantGuideInfo;
    t_Bool isCarlifeNaviNextTurnInfoValid;
    t_Bool isCarlifeNaviAssistantGuideInfoValid;

    trBDCLTBTUpdate():isCarlifeNaviNextTurnInfoValid(false),
                      isCarlifeNaviAssistantGuideInfoValid(false)
    {}
};

struct trMLNavigationInfo
{
   t_U32 u32AppID;
   t_String szAppName;
   tenMLNaviGuidanceState enGuidanceState;
   tenMLNaviGuidanceActive enGuidanceActive;
   t_Bool bMetric;
   t_Bool bRightDrive;

   trMLNavigationInfo():u32AppID(0),
                        enGuidanceState(E8ML_NAVI_GUIDANCE_UNKNOWN),
                        enGuidanceActive(E8ML_NAVI_GUIDANCE_NONE),
                        bMetric(false),
                        bRightDrive(false)
   {

   }
   trMLNavigationInfo(const trMLNavigationInfo& oData)
   {
      u32AppID = oData.u32AppID;
      szAppName = oData.szAppName.c_str();
      enGuidanceState = oData.enGuidanceState;
      enGuidanceActive = oData.enGuidanceActive;
      bMetric = oData.bMetric;
      bRightDrive = oData.bRightDrive;
   }
   bool operator ==(const trMLNavigationInfo& oData)
   {
      return ((u32AppID == oData.u32AppID) && (szAppName == oData.szAppName) &&(enGuidanceState == oData.enGuidanceState)
                  && (enGuidanceActive == oData.enGuidanceActive) && (bMetric == oData.bMetric)
                  && (bRightDrive == oData.bRightDrive));
   }
   trMLNavigationInfo& operator=(const trMLNavigationInfo& oData)
   {
      if (this != &oData)
      {
         u32AppID = oData.u32AppID;
         szAppName = oData.szAppName.c_str();
         enGuidanceState = oData.enGuidanceState;
         enGuidanceActive = oData.enGuidanceActive;
         bMetric = oData.bMetric;
         bRightDrive = oData.bRightDrive;
      }
      return *this;
   }
};

struct trMLNavigationConfig
{
	t_U32 u32MaxSideStAngles_1;
	t_U32 u32MaxSideStAngles_2;
	t_U32 u32MaxSideStAngles_3;
	t_U32 u32MaxLaneGuidance;

	trMLNavigationConfig() :
		u32MaxSideStAngles_1(0),
		u32MaxSideStAngles_2(0),
		u32MaxSideStAngles_3(0),
		u32MaxLaneGuidance(0)
	{

	}
	trMLNavigationConfig(const trMLNavigationConfig& oData)
	{
	   u32MaxSideStAngles_1 = oData.u32MaxSideStAngles_1;
	   u32MaxSideStAngles_2 = oData.u32MaxSideStAngles_2;
	   u32MaxSideStAngles_3 = oData.u32MaxSideStAngles_3;
	   u32MaxLaneGuidance = oData.u32MaxLaneGuidance;
	}
	bool operator ==(const trMLNavigationConfig& oData)
	{
		return ((u32MaxSideStAngles_1 == oData.u32MaxSideStAngles_1) && (u32MaxSideStAngles_2 == oData.u32MaxSideStAngles_2)
			&& (u32MaxSideStAngles_3 == oData.u32MaxSideStAngles_3) && (u32MaxLaneGuidance == oData.u32MaxLaneGuidance));
	}
	trMLNavigationConfig& operator=(const trMLNavigationConfig& oData)
	{
	   if (this != &oData)
	   {
	      u32MaxSideStAngles_1 = oData.u32MaxSideStAngles_1;
	      u32MaxSideStAngles_2 = oData.u32MaxSideStAngles_2;
	      u32MaxSideStAngles_3 = oData.u32MaxSideStAngles_3;
	      u32MaxLaneGuidance = oData.u32MaxLaneGuidance;
	   }
	   return *this;
	}
};

struct trMLNavigationNextManeuver
{
	tenMLNaviManeuverDirection enManeuverDirection;
	t_U32 u32Index;
	t_U32 u32NextAngle;
	std::vector<t_U32> nextSideStAngles_1;
	std::vector<t_U32> nextSideStAngles_2;
	std::vector<t_U32> nextSideStAngles_3;
	t_String szNextStName;
	t_String szCurrStName;

	trMLNavigationNextManeuver() :
		enManeuverDirection(E8ML_NAVI_MANEUVER_NO_SYMBOL),
		u32Index(0),
		u32NextAngle(0)
	{

	}
	trMLNavigationNextManeuver(const trMLNavigationNextManeuver& oData)
	{
	   enManeuverDirection = oData.enManeuverDirection;
	   u32Index = oData.u32Index;
	   u32NextAngle = oData.u32NextAngle;
	   nextSideStAngles_1 = oData.nextSideStAngles_1;
	   nextSideStAngles_2 = oData.nextSideStAngles_2;
	   nextSideStAngles_3 = oData.nextSideStAngles_3;
	   szNextStName = oData.szNextStName.c_str();
	   szCurrStName = oData.szCurrStName.c_str();

	}
	bool operator ==(const trMLNavigationNextManeuver& oData)
	{
		return ((enManeuverDirection == oData.enManeuverDirection) && (u32Index == oData.u32Index)
			&& (u32NextAngle == oData.u32NextAngle)
			&& (equal(nextSideStAngles_1.begin(), nextSideStAngles_1.end(), oData.nextSideStAngles_1.begin()))
			&& (equal(nextSideStAngles_2.begin(), nextSideStAngles_2.end(), oData.nextSideStAngles_2.begin()))
			&& (equal(nextSideStAngles_3.begin(), nextSideStAngles_3.end(), oData.nextSideStAngles_3.begin()))
			&& (szNextStName == oData.szNextStName) && (szCurrStName == szCurrStName));
	}
	trMLNavigationNextManeuver& operator=(const trMLNavigationNextManeuver& oData)
	{
	   if (this != &oData)
	   {
	      enManeuverDirection = oData.enManeuverDirection;
	      u32Index = oData.u32Index;
	      u32NextAngle = oData.u32NextAngle;
	      nextSideStAngles_1 = oData.nextSideStAngles_1;
	      nextSideStAngles_2 = oData.nextSideStAngles_2;
	      nextSideStAngles_3 = oData.nextSideStAngles_3;
	      szNextStName = oData.szNextStName.c_str();
	      szCurrStName = oData.szCurrStName.c_str();
	   }
	   return *this;
	}
};

struct trMLNavigationSpeedLimit
{
	t_U32 u32CurrentLimit;
	t_U32 u32NextLimit;
	t_U32 u32Distance;
	tenMLNaviDistanceUnit enDistanceUnit;

	trMLNavigationSpeedLimit() :
		u32CurrentLimit(0),
		u32NextLimit(0),
		u32Distance(0),
		enDistanceUnit(E8ML_NAVI_DISTANCE_UNKNOWN)
	{

	}
	trMLNavigationSpeedLimit(const trMLNavigationSpeedLimit& oData)
	{
	   u32CurrentLimit = oData.u32CurrentLimit;
	   u32NextLimit = oData.u32NextLimit;
	   u32Distance = oData.u32Distance;
	   enDistanceUnit = oData.enDistanceUnit;
	}
	bool operator ==(const trMLNavigationSpeedLimit& oData)
	{
		return ((u32CurrentLimit == oData.u32CurrentLimit) && (u32NextLimit == oData.u32NextLimit)
			&& (u32Distance == oData.u32Distance) && (enDistanceUnit == oData.enDistanceUnit));
	}
	trMLNavigationSpeedLimit& operator=(const trMLNavigationSpeedLimit& oData)
	{
	   if (this != &oData)
	   {
	      u32CurrentLimit = oData.u32CurrentLimit;
	      u32NextLimit = oData.u32NextLimit;
	      u32Distance = oData.u32Distance;
	      enDistanceUnit = oData.enDistanceUnit;
	   }
	   return *this;
	}
};

struct trMLNavigationNextDistance
{
	t_U32 u32Distance;
	tenMLNaviDistanceUnit enDistanceUnit;
	t_U64 u64TimeToManeuver;
	t_U32 u32Percent;

	trMLNavigationNextDistance() :
		u32Distance(0),
		enDistanceUnit(E8ML_NAVI_DISTANCE_UNKNOWN),
		u64TimeToManeuver(0),
		u32Percent(0)
	{

	}
	trMLNavigationNextDistance(const trMLNavigationNextDistance& oData)
	{
	   u32Distance = oData.u32Distance;
	   enDistanceUnit = oData.enDistanceUnit;
	   u64TimeToManeuver = oData.u64TimeToManeuver;
	   u32Percent = oData.u32Percent;
	}
	bool operator ==(const trMLNavigationNextDistance& oData)
	{
		return ((u32Distance == oData.u32Distance) && (enDistanceUnit == oData.enDistanceUnit)
			&& (u64TimeToManeuver == oData.u64TimeToManeuver) && (u32Percent == oData.u32Percent));
	}
	trMLNavigationNextDistance& operator=(const trMLNavigationNextDistance& oData)
	{
	   if (this != &oData)
	   {
	      u32Distance = oData.u32Distance;
	      enDistanceUnit = oData.enDistanceUnit;
	      u64TimeToManeuver = oData.u64TimeToManeuver;
	      u32Percent = oData.u32Percent;
	   }

	   return *this;
	}
};

struct trMLNavigationLaneGuidance
{
	std::vector<t_U32> nextLaneGuidances;
	bool operator ==(const trMLNavigationLaneGuidance& oData)
	{
		return (std::equal(nextLaneGuidances.begin(), nextLaneGuidances.end(), oData.nextLaneGuidances.begin()));
	}
	trMLNavigationLaneGuidance& operator=(const trMLNavigationLaneGuidance& oData)
	{
		if (this != &oData)
		{
		   nextLaneGuidances = oData.nextLaneGuidances;
		}
		return *this;
	}
};

struct trMLNavigationTripInfo
{
	t_String szDestination;
	t_U32 u32Distance;
	tenMLNaviDistanceUnit enDistanceUnit;
	t_U64 u64TravelTime; //TO DO: Check what type "TIME" maps onto
	t_U64 u64TrafficTime; //TO DO: Check what type "TIME" maps onto

	trMLNavigationTripInfo():
		u32Distance(0),
		enDistanceUnit(E8ML_NAVI_DISTANCE_UNKNOWN),
		u64TravelTime(0),
		u64TrafficTime(0)
	{

	}

	trMLNavigationTripInfo(const trMLNavigationTripInfo& oData)
	{
	   szDestination = oData.szDestination.c_str();
	   u32Distance = oData.u32Distance;
	   enDistanceUnit = oData.enDistanceUnit;
	   u64TravelTime = oData.u64TravelTime;
	   u64TrafficTime = oData.u64TrafficTime;

	}

	bool operator ==(const trMLNavigationTripInfo& oData)
	{
		return ((szDestination == oData.szDestination) && (u32Distance == oData.u32Distance)
			&& (enDistanceUnit == oData.enDistanceUnit) && (u64TravelTime == oData.u64TravelTime)
			&& (u64TrafficTime == oData.u64TrafficTime));
	}
	trMLNavigationTripInfo& operator=(const trMLNavigationTripInfo& oData)
	{
	   if (this != &oData)
	   {
	      szDestination = oData.szDestination.c_str();
	      u32Distance = oData.u32Distance;
	      enDistanceUnit = oData.enDistanceUnit;
	      u64TravelTime = oData.u64TravelTime;
	      u64TrafficTime = oData.u64TrafficTime;
	   }
	   return *this;
	}
};

struct trMLTBTUpdate
{
	trMLNavigationInfo MLNaviInfo;
	trMLNavigationNextManeuver MLNaviNextManeuver;
	trMLNavigationSpeedLimit MLNaviSpdLimit;
	trMLNavigationNextDistance MLNaviNextDistance;
	trMLNavigationLaneGuidance MLNaviLaneGuidance;
	trMLNavigationTripInfo MLNaviTripInfo;
	tenMLNaviUpdateType enMLNaviUpdateType;

	trMLTBTUpdate() :
		enMLNaviUpdateType(e8ML_NAVIGATION_UNKNOWN)
	{}
};


struct trTurnByTurnInfo
{
    t_U32 u32DeviceHandle;
    tenDeviceCategory enDeviceCategory;
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
    trCarPlayTBTUpdate rCarPlayTBTUpdate;
#endif
    trAAutoTBTUpdate rAAutoTBTUpdate;
    trBDCLTBTUpdate rBDCLTBTUpdate;
    trMLTBTUpdate rMLTBTUpdate;

    trTurnByTurnInfo(): u32DeviceHandle(0),
                        enDeviceCategory(e8DEV_TYPE_UNKNOWN)
    {}
};

typedef std::function<t_Void(const trDiPORouteGuidanceUpdate&)> tfvDiPORouteGuidanceUpdateCallback;
typedef std::function<t_Void(const trDiPORouteGuidanceManeuverUpdate&)> tfvDiPORouteGuidanceManeuverUpdateCallback;
typedef std::function<t_Void(const trAAutoNavigationNextTurnData&)> tfvAAutoNavigationNextTurnDataCallback;
typedef std::function<t_Void(const trAAutoNavigationNextTurnDistanceData&)> tfvAAutoNavigationNextTurnDistanceDataCallback;
typedef std::function<t_Void(const tenAAutoNavigationAppState&)> tfvAAutoNavigationAppStateCallback;
typedef std::function<t_Void(const trCarlifeNaviNextTurnInfo&)> tfvCarlifeNaviNextTurnInfoCallback;
typedef std::function<t_Void(const trCarlifeNaviAssistantGuideInfo&)> tfvCarlifeNaviAssistantGuideInfoCallback;

typedef std::function<t_Void(const trMLNavigationInfo&, const trUserContext&)> tfvUpdateMLNaviInfoCallback;
typedef std::function<t_Void(const trMLNavigationNextManeuver&, const trUserContext&)> tfvUpdateMLNaviTBTCallback;
typedef std::function<t_Void(const trMLNavigationSpeedLimit&, const trUserContext&)> tfvUpdateMLNaviSpdLimitCallback;
typedef std::function<t_Void(const trMLNavigationNextDistance&, const trUserContext&)> tfvUpdateMLNaviNextDistanceCallback;
typedef std::function<t_Void(const trMLNavigationLaneGuidance&, const trUserContext&)> tfvUpdateMLNaviLaneGuidanceCallback;
typedef std::function<t_Void(const trMLNavigationTripInfo&, const trUserContext&)> tfvUpdateMLNaviTripInfoCallback;

struct trTBTCallbacks
{
      tfvDiPORouteGuidanceUpdateCallback fvOnDiPORouteGuidanceUpdateCallback;
      tfvDiPORouteGuidanceManeuverUpdateCallback fvOnDiPORouteGuidanceManeuverUpdateCallback;
      tfvAAutoNavigationNextTurnDataCallback fvOnAAutoNavigationNextTurnDataCallback;
      tfvAAutoNavigationNextTurnDistanceDataCallback fvOnAAutoNavigationNextTurnDistanceDataCallback;
      tfvAAutoNavigationAppStateCallback fvOnAAutoNavigationAppStateCallback;
      tfvCarlifeNaviNextTurnInfoCallback fvOnCarlifeNaviNextTurnInfoCallback;
      tfvCarlifeNaviAssistantGuideInfoCallback fvOnCarlifeNaviAssistantGuideInfoCallback;
      //!ML CDB Navi Metadata Callbacks
      tfvUpdateMLNaviInfoCallback fvMLNaviInfoCallback;
      tfvUpdateMLNaviTBTCallback fvMLNaviTBTCallback;
      tfvUpdateMLNaviSpdLimitCallback fvMLNaviSpdLimitCallback;
      tfvUpdateMLNaviNextDistanceCallback fvMLNaviNextDistanceCallback;
      tfvUpdateMLNaviLaneGuidanceCallback fvMLNaviLaneGuidanceCallback;
      tfvUpdateMLNaviTripInfoCallback fvMLNaviTripInfoCallback;

      trTBTCallbacks() :
                        fvOnDiPORouteGuidanceUpdateCallback(NULL),
                        fvOnDiPORouteGuidanceManeuverUpdateCallback(NULL),
                        fvOnAAutoNavigationNextTurnDataCallback(NULL),
                        fvOnAAutoNavigationNextTurnDistanceDataCallback(NULL),
                        fvOnAAutoNavigationAppStateCallback(NULL),
                        fvOnCarlifeNaviNextTurnInfoCallback(NULL),
                        fvOnCarlifeNaviAssistantGuideInfoCallback(NULL),
                        fvMLNaviInfoCallback(NULL),
                        fvMLNaviTBTCallback(NULL),
                        fvMLNaviSpdLimitCallback(NULL),
                        fvMLNaviNextDistanceCallback(NULL),
                        fvMLNaviLaneGuidanceCallback(NULL),
                        fvMLNaviTripInfoCallback(NULL)
      {

      }
};

#endif //_SPI_TBTTYPES_H_
