/*!
 *******************************************************************************
 * \file              SPITypes.h
 * \brief             common types used for mirrorlink and IPODout
 *******************************************************************************
 \verbatim
 PROJECT:       GM Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   provides definition of common types used
 for mirrorlink and IPODout
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 13.09.2013 |  Pruthvi Thej Nagaraju       | Initial Version
 07.01.2014 |  Priju K Padiyath            | Updated with types and data structure for SPI message mediator
 21.04.2014 |  Shiva Kumar G               | Updated with the elements for Content Attestation
 28.04.2014 |  Shihabudheen P M            | Updated with types for audio context handling
 28.10.2014 |  Hari Priya E R (RBEI/ECP2)  | Updated with types for Vehicle Data
 25.06.2015 |  Shiva kaumr G               | Dynamic display configuration
 25.06.2015 |  Sameer Chandra              | Added ML XDeviceKey Support for PSA
 03.07.2015 |  Shiva Kumar Gurija          | improvements in ML Version checking
 17.07.2015 |  Sameer Chandra              | Added new command ID for knob Encoder and respectiv
 keycodes
 10.08.2015 |  Vinoop U                    | Updated VehicleConfiguration with Right
 and left and drive enums
 18.08.2015 |  Shiva Kumar Gurija          | Added elements for ML Dynamic Audio
 05.02.2016 |  Rachana L Achar             | Removed tenAAPKeycode and tenAAPkeyCodes
 10.03.2016 |  Rachana L Achar             | Added structure for Notification Acknowledgment data
 08.11.2016 |  Noopur R K                  | Added attributes for Session Configuration of AAP
 29.11.2016 |  Noopur R K                  | Added attributes for HMI Synchronization
 01.01.2017 |  Noopur R K                  | Added changes for Delete device interface implementation
 18.07.2017 |  Noopur R K                  | Added new COMMAND ID for General Restriction parameter
 10.05.2019 |  Dhanya S B                  | SPITypes.h Clean up, all Mirrorlink related structures and functions ared moved to MLTypes.h

 \endverbatim
 ******************************************************************************/

#ifndef SPITYPES_H_
#define SPITYPES_H_

#include <functional>
#include <vector>
#include <map>
#include <list>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "BaseTypes.h"

static const t_U32 BUFFERSIZE = 256; //TODO Check buffer size
static const t_U32 MAXKEYSIZE = 256;
static const t_U32 cou32MAX_DEVICEHANDLE = 0xFFFFFFFF;
#define ARR_SIZE 3

#define SINGLE_TOUCH 1
#define DOUBLE_TOUCH 2

#define STR_LENGTH 250
#define MAX_LENGTH 1024
#define PATH_LENGTH 20

#define NUM_AUD_CLIENTS (e8DEV_TYPE_MAX)
#define NUM_APPMNGR_CLIENTS (e8DEV_TYPE_MAX)
#define NUM_VIDEO_CLIENTS (e8DEV_TYPE_MAX)
#define NUM_RSRCMNGR_CLIENTS (e8DEV_TYPE_MAX)
#define NUM_INPUT_CLIENTS (e8DEV_TYPE_MAX)
#define MAX_DEV_CATEGORY (e8DEV_TYPE_MAX)
#define PRIMARY_INDEX 0
#define NUM_WIFI_CLIENTS (e8DEV_TYPE_MAX)
//TODO consider using templates to avoid misuse of macros
#define RELEASE_MEM(VAR)      \
   if(NULL!=VAR)              \
   {                          \
      delete VAR;             \
      VAR= NULL;              \
   }

#define RELEASE_ARRAY_MEM(VAR)  \
   if(NULL!=VAR)                \
   {                            \
      delete[] VAR;             \
      VAR= NULL;                \
   }

#define CPY_TO_USRCNTXT(RCVDMSG, USRCNTXT)\
         USRCNTXT.u32FuncID    = RCVDMSG->u16GetFunctionID( );\
         USRCNTXT.u32SrcAppID  = RCVDMSG->u16GetSourceAppID( );\
         USRCNTXT.u32DestAppID = RCVDMSG->u16GetTargetAppID( );\
         USRCNTXT.u32RegID     = RCVDMSG->u16GetRegisterID( );\
         USRCNTXT.u32CmdCtr    = RCVDMSG->u16GetCmdCounter( );\
         USRCNTXT.u32ServcID   = RCVDMSG->u16GetServiceID( );\

//@note: most_fi_tcl_String and midw_fi_tclString szGet() function performs deep copy.
//Hence this macro extracts the FI string data to a std::string object & de-allocates the deep-copied data.
#define GET_STRINGDATA_FROM_FI_STRINGOBJ(FI_STRING_OBJ, CHAR_SET, STD_STRING_OBJ)  \
{                                                                                  \
   t_Char* poszFiString = static_cast<t_Char*>(FI_STRING_OBJ.szGet(CHAR_SET));     \
   STD_STRING_OBJ = "";                               \
   if (NULL != poszFiString)                          \
   {                                                  \
      STD_STRING_OBJ.assign(poszFiString);            \
      delete[] poszFiString;                          \
      poszFiString = NULL;                            \
   }                                                  \
}

//! Macro to place 1 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_1 std::placeholders::_1
//! Macro to place 2 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_2 std::placeholders::_1,std::placeholders::_2
//! Macro to place 3 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_3 std::placeholders::_1,std::placeholders::_2,std::placeholders::_3 
//! Macro to place 4 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_4 std::placeholders::_1,std::placeholders::_2,std::placeholders::_3,\
   std::placeholders::_4 
//! Macro to place 5 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_5 std::placeholders::_1,std::placeholders::_2,std::placeholders::_3,\
   std::placeholders::_4,std::placeholders::_5
//! Macro to place 6 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_6 std::placeholders::_1,std::placeholders::_2,std::placeholders::_3,\
   std::placeholders::_4,std::placeholders::_5,std::placeholders::_6
//! Macro to place 7 place holders for call back registration
#define SPI_FUNC_PLACEHOLDERS_7 std::placeholders::_1,std::placeholders::_2,std::placeholders::_3,\
   std::placeholders::_4,std::placeholders::_5,std::placeholders::_6,std::placeholders::_7

//! Macro to check whether the map is empty
#define SPI_MAP_NOT_EMPTY(var)   var.size()>0

#define LOW_FIDELITY_TOUCH 0x02
#define HIGH_FIDELITY_TOUCH 0x04
#define LOW_FIDELITY_TOUCH_WITH_KNOB 0x08
#define HIGH_FIDELITY_TOUCH_WITH_KNOB 0x10

#define FOUR_EIGHTY 480

//! Identifies result of an operation
typedef enum
{
   //! Identifies an unsuccessful Operation.
   e8FAILURE = 0,
   //! Identifies a successful Operation.
   e8SUCCESS = 1
} tenResponseCode;

//! Identifies different Error Codes
typedef enum
{
   //! No Error
   e8NO_ERRORS = 0x00,
   //! Unknown Error
   e8UNKNOWN_ERROR = 0x01,
   //! Invalid Argument
   e8INVALID_ARGUMENT = 0x02,
   //!Invalid Device Handle
   e8INVALID_DEV_HANDLE = 0x03,
   //!Application Handle Invalid
   e8INVALID_APP_HANDLE = 0x04,
   //!Resource Busy
   e8RESOURCE_BUSY = 0x05,
   //!Operation Rejected
   e8OPERATION_REJECTED = 0x06,
   //!Application Launch failed
   e8LAUNCH_FAILED = 0x07,
   //!Application Termination failed
   e8TERMINATE_FAILED = 0x08,
   //!Unsupported Key
   e8UNSUPPORTED_KEY = 0x09,
   //!Unsupported Operation
   e8UNSUPPORTED_OPERATION = 0x0A,
   //!Invalid Service Handle
   e8INVALID_SERVICE_HANDLE = 0x0B,
   //!Device Selection Failed
   e8SELECTION_FAILED = 0x0C,
   //! Device not connected
   e8DEVICE_NOT_CONNECTED = 0x0D,
   //! Could not launch remote launcher
   e8REMOTE_APP_LAUNCH_ERROR = 0x0E,
   //! Connection Time out
   e8CONNECT_TIME_OUT = 0x0F,
   //! Device disconnected during an operation
   e8DEVICE_DISCONNECTED_DURING_SELECTION = 0x10,
   //! Projection is not enabled for the device
   e8PROJECTION_NOT_ENABLED = 0x11,
   //! Projection is not enabled for the device
   e8OPERATION_CANCELLED_BY_USER = 0x12,
   //! Projection is not enabled for the device
   e8DEVICE_SWITCH_FAILED = 0x13,
   //!Unrecoverable errors
   e8FATAL_ERROR = 0x14,
   //!Unsupported phone connected.
   e8PHONE_UNSUPPORTED = 0x15,
   //app not installed.
   e8APP_NOT_INSTALLED = 0x16,
   //! Operation canceled by Factory reset
   e8OPERATION_CANCELLED_BY_FACTORY_RESET = 0x17,
   //! AP Could not be setup
   e8AP_SETUP_ERROR = 0x18,
   //! Station Disconnection from the Wireless Access Point
   e8STA_DISCONNECTED = 0x19,
   //! Internal SPI error
   e8INTERNAL_ERROR = 0xFF
//! \note : Error Codes will be extended in future as required.

} tenErrorCode;

//! Identifies the different types of devices
typedef enum
{
   e8LOW_FIDELITY = 0, e8HIGH_FIDELITY = 1
} tenFidelityType;

//! Identifies the different types of devices
typedef enum
{
   //! Default value for Initialization
   e8DEV_TYPE_UNKNOWN = 0,
   //! Digital iPod Out Device
   e8DEV_TYPE_DIPO = 1,
   //! Mirror Link Device
   e8DEV_TYPE_MIRRORLINK = 2,
   //! Android Audio Device
   e8DEV_TYPE_ANDROIDAUTO = 3,
   //! mySPIN Device
   e8DEV_TYPE_MYSPIN = 4,
   //! Baidu Carlife Device
   e8DEV_TYPE_CARLIFE = 5,
   //! OnCar Device
   e8DEV_TYPE_ONCAR = 6,

   e8DEV_TYPE_MAX,
   //! Default technology preference to use device in default mode
   e8DEV_TYPE_DEFAULT = 0x0F

//Device category has the dependency in Video.
//If there is any change in enum values. tcl_Video must be revisisted.
//or else it can cause crashes because of array out of bounds error

} tenDeviceCategory;

//! Identifies the different types of character encoding
typedef enum
{
   e8UTF8 = 0,

   e8ASCII = 1
} tenEncodingType;

//! Device Type (Android, Apple etc)
enum tenDeviceType
{
   e8_UNKNOWN_DEVICE = 0x00, e8_ANDROID_DEVICE = 0x01, e8_APPLE_DEVICE = 0x02
};

//! Indicates type of SPI feature supported
struct trSpiFeatureSupport
{
   public:
      trSpiFeatureSupport() :
                        m_bMirrorlinkSupport(false),
                        m_bDipoSupport(false),
                        m_bAndroidAutoSupport(false),
                        m_bMySPINSupport(false),
                        m_bCarlifeAndroidSupport(false),
                        m_bOnCarSupport(false),
                        m_bCPWSupport(false),
                        m_bCarlifeIOSSupport(false)
      {
      }
      t_Bool bMirrorLinkSupported() const
      {
         return m_bMirrorlinkSupport;
      }
      t_Void vSetMirrorLinkSupport(t_Bool bMirrorLinkSupported)
      {
         m_bMirrorlinkSupport = bMirrorLinkSupported;
      }

      t_Bool bDipoSupported() const
      {
         return m_bDipoSupport;
      }
      t_Void vSetDipoSupport(t_Bool bDipoSupported)
      {
         m_bDipoSupport = bDipoSupported;
      }

      t_Bool bAndroidAutoSupported() const
      {
         return m_bAndroidAutoSupport;
      }
      t_Void vSetAndroidAutoSupport(t_Bool bAndroidAutoSupported)
      {
         m_bAndroidAutoSupport = bAndroidAutoSupported;
      }
      t_Bool bMySPINSupported() const
      {
         return m_bMySPINSupport;
      }
      t_Void vSetmySPINSupport(t_Bool bMySPINSupported)
      {
         m_bMySPINSupport = bMySPINSupported;
      }
      t_Bool bCarlifeSupportedForIOS() const
      {
         return m_bCarlifeIOSSupport;
      }
      t_Bool bCarlifeSupportedForAndroid() const
      {
         return m_bCarlifeAndroidSupport;
      }
      t_Void vSetCarlifeSupport(t_Bool bCarlifeSupported, tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE)
      {
         switch (enDeviceType)
         {
            case e8_APPLE_DEVICE:
            {
               m_bCarlifeIOSSupport = bCarlifeSupported;
            }
               break;
            case e8_ANDROID_DEVICE:
            {
               m_bCarlifeAndroidSupport = bCarlifeSupported;
            }
               break;
            default:
               m_bCarlifeIOSSupport = m_bCarlifeAndroidSupport = bCarlifeSupported;
         }
      }
      t_Bool bOnCarSupported() const
      {
         return m_bOnCarSupport;
      }
      t_Void vSetOnCarSupport(t_Bool bOnCarSupported)
      {
         m_bOnCarSupport = bOnCarSupported;
      }

      t_Bool bCPWSupported() const
      {
         return m_bCPWSupport;
      }
      t_Void vSetCPWSupport(t_Bool bCPWSupported)
      {
         m_bCPWSupport = bCPWSupported;
      }

      t_Bool bGetFeatureSupport(tenDeviceCategory enDeviceCategory, tenDeviceType enDeviceType) const
      {
         t_Bool bRetval = false;
         switch (enDeviceCategory)
         {
            case e8DEV_TYPE_MIRRORLINK:
            {
               bRetval = bMirrorLinkSupported();
               break;
            }
            case e8DEV_TYPE_DIPO:
            {
               bRetval = bDipoSupported();
               break;
            }
            case e8DEV_TYPE_ANDROIDAUTO:
            {
               bRetval = bAndroidAutoSupported();
               break;
            }
            case e8DEV_TYPE_MYSPIN:
            {
               bRetval = bMySPINSupported();
               break;
            }
            case e8DEV_TYPE_CARLIFE:
            {
               switch (enDeviceType)
               {
                  case e8_ANDROID_DEVICE:
                  {
                     bRetval = bCarlifeSupportedForAndroid();
                     break;
                  }
                  case e8_APPLE_DEVICE:
                  {
                     bRetval = bCarlifeSupportedForIOS();
                     break;
                  }
                  default:
                  {
                     bRetval = (bCarlifeSupportedForAndroid() || bCarlifeSupportedForIOS());
                     break;
                  }
               }
               break;
            }
            case e8DEV_TYPE_ONCAR:
            {
               bRetval = bOnCarSupported();
               break;
            }
            default:
            {
               bRetval = false;
               break;
            }
         }
         return bRetval;
      }

      t_Bool bGetWirelessFeatureSupport(tenDeviceCategory enDeviceCategory) const
      {
         t_Bool bRetval = false;
         switch (enDeviceCategory)
         {
            case e8DEV_TYPE_DIPO:
            {
               bRetval = bCPWSupported();
               break;
            }
            default:
            {
               bRetval = false;
               break;
            }
         }
         return bRetval;
      }
   private:

      t_Bool m_bMirrorlinkSupport;
      t_Bool m_bDipoSupport;
      t_Bool m_bAndroidAutoSupport;
      t_Bool m_bMySPINSupport;
      t_Bool m_bCarlifeAndroidSupport;
      t_Bool m_bOnCarSupport;
      t_Bool m_bCPWSupport;
      t_Bool m_bCarlifeIOSSupport;
};

enum tenATBoxSupportedState
{
   ATBOX_DISABLED = 0x00,
   ATBOX_TYPE1_NISSAN = 0x01,
   ATBOX_TYPE2_RENAULT = 0x02,
   ATBOX_RESERVED = 0x03,
   ATBOX_UNKNOWN = 0x04
};

enum tenUSBPortType
{
   e8_PORT_TYPE_NOT_KNOWN = 0, e8_PORT_TYPE_OTG = 1, e8_PORT_TYPE_NON_OTG = 2
};

//! Indicated support to SPI technologies
enum tenSPISupport
{
   e8SPI_SUPPORT_UNKNOWN = 0, e8SPI_SUPPORTED = 1, e8SPI_NOTSUPPORTED = 2
};

//! Identifies Device connection Statuses
typedef enum
{
   /*!
    * Device is connected but not available to start a Mirror Link
    * session (USB Device/ Wi-Fi Au-thenticated devices).
    * Also the Default Status value at Startup.
    */
   e8DEV_NOT_CONNECTED = 0,

   /*!
    * Device is connected and available to
    * start a Mirror Link session or an Application has al-ready
    * been launched on the device (USB De-vice/Wi-Fi Authenticated devices)
    */
   e8DEV_CONNECTED = 1,
   //! Device Busy when session is starting or ending
   e8DEV_BUSY = 2

} tenDeviceConnectionStatus;

//! Identifies the device connection types
typedef enum
{
   //! Unknown Connection Type.
   e8UNKNOWN_CONNECTION = 0,
   //! Device connected via USB.
   e8USB_CONNECTED = 1,
   //! Device connected via Wi-Fi.
   e8WIRELESS_CONNECTED = 2
} tenDeviceConnectionType;

//! Identifies Connection Information of the devices
typedef enum
{
   /*!
    * Default value(Startup case).
    */
   e8DEVICE_STATUS_NOT_KNOWN = 0,

   /*!
    * Device List has been modified with addition of a device.
    */
   e8DEVICE_ADDED = 1,

   /*!
    * Device List has been modified with removal of an existing device.
    */
   e8DEVICE_REMOVED = 2,

   /*!
    * Properties of an existing device updated in the Device List.
    */
   e8DEVICE_CHANGED = 3
} tenDeviceStatusInfo;

//! Identifies Status Information of device applications.
typedef enum
{
   /*!
    * Default value(Startup case).
    */
   e8APP_STATUS_NOT_KNOWN = 0,

   /*!
    * Application List has been updated with addition/removal of devices and on change in drive modes.
    */
   e8APP_LIST_CHANGED = 1,
} tenAppStatusInfo;

//! Session status
enum tenSessionStatus
{
   //! Indicates the Default status
   e8_SESSION_UNKNOWN = 0,
   //! Indicates the Device is selected but a session has not been initiated
   e8_SESSION_INACTIVE = 1,
   //! Indicates the Session has started and the remote application active in foreground
   e8_SESSION_ACTIVE = 2,
   //! Indicates the Session has started and the remote application is active in background
   e8_SESSION_SUSPENDED = 3,
   //! Indicates any internal error during an active session
   e8_SESSION_ERROR = 4,
   //! Indicates that the Non Mirror link application is launched by the user
   e8_SESSION_SUSPENDED_NON_ML_APP = 5,
   //! Indicates that the Non drive certified application is launched by the user in drive mode
   e8_SESSION_SUSPENDED_NON_DRIVE_APP = 6,
   //! Indicates that for mySPIN 3rd party App is in transition and hence session is suspended
   e8_SESSION_SESPENDED_APP_TRANSITION = 7,
   //! Indicates that session failed due to authentication failure
   e8_SESSION_AUTHENTICATION_FAILURE = 8,
   //! Indicates Ping response failed
   e8_SESSION_PING_FAILURE = 9
};

//! Handset Interaction Status
enum tenHandsetInteractionStatus
{
   //! Indicates the Default status
   e8_HANDSET_INTERACTION_UNKNOWN = 0,
   //! Indicates that handset interaction is required to proceed with projection
   e8_HANDSET_INTERACTION_REQUIRED = 1,
   //! Indicates that handset interaction is not required to proceed with projection
   e8_HANDSET_INTERACTION_NOT_REQUIRED = 2
};

//! Identifies the Icon Mime Types
typedef enum
{
   //! Invalid Icon Format.
   e8ICON_INVALID = 0x00,
   //! Icon image in PNG format.
   e8ICON_PNG = 0x01,
   //! Icon image in JPEG format.
   e8ICON_JPEG = 0x02,
} tenIconMimeType;

//! Identifies the Orientation Modes
typedef enum
{
   //! Invalid Mode for display
   e8INVALID_MODE = 0x00,
   //! Portrait Mode for display
   e8PORTRAIT_MODE = 0x01,
   //! Landscape Mode for display. Default Mode
   e8LANDSCAPE_MODE = 0x02,
} tenOrientationMode;

//! Identifies the Blocking Modes
typedef enum
{
   //! Disable Audio/Video content blocking.
   e8DISABLE_BLOCKING = 0x00,
   //! Enable Audio/Video blocking.
   e8ENABLE_BLOCKING = 0x01,
} tenBlockingMode;

//! Identifies the Video Blocking Reason
typedef enum
{
   //! Blocking Reason User Action.
   e8USER_ACTION = 0x00,
   //! Blocking Reason NON ML App.
   e8NON_ML_APP = 0x01,
   //! Blocking Reason User Action.
   e8NON_DRIVE_APP = 0x02,
   //! Blocking Reason NON ML App.
   e8PHONE_REQUEST = 0x03
} tenVideoBlockingReason;

//! Identifies the Vehicle configurations
typedef enum
{
   //! Vehicle is in Non-Driving Mode
   e8PARK_MODE = 0x00,
   //! Vehicle is in Driving Mode
   e8DRIVE_MODE = 0x01,
   //! Day Mode
   e8_DAY_MODE = 0x02,
   //! Night mode
   e8_NIGHT_MODE = 0x03,
   //! Right Hand Drive
   e8_RIGHT_HAND_DRIVE = 0x04,
   //! Left Hand Drive
   e8_LEFT_HAND_DRIVE = 0x05,
   //! To enable or disable Key Lock ( pointer events & key events from the Phone)
   e8_KEY_LOCK = 0x06,
   //! To enable or disable device lick during ML session.
   e8_DEVICE_LOCK = 0x07,
   //! To Enable/Disable the Screen Saver
   e8_SCREEN_SAVER = 0x08,
   //! To enable/disable Voice recognition
   e8_VOICE_RECOG = 0x09,
   //! Top Enable/disable MicroPhone In
   e8_MICROPHONE_IN = 0x0A,
   //! Configuraion is unknown
   e8_UNKNOWN_CONFIG = 0xFF

} tenVehicleConfiguration;

//! Identifies the Touch Modes
typedef enum
{
   //! Touch pressure is Zero
   e8TOUCH_RELEASE = 0x00,
   //! Touch Pressure is Non-Zero
   e8TOUCH_PRESS = 0x01,
   //! Touch pressure is Non-Zero but is not
   //! the first Press (introduced for AndroidAuto)
   e8TOUCH_MOVED = 0x02,
   //! Non primary touch-point press [AAP specific]
   e8TOUCH_MULTI_PRESS = 0x03,
   //! Non primary touch-point release [AAP specific]
   e8TOUCH_MULTI_RELEASE = 0x04,
   //! Default value
   e8TOUCH_UNKNOWN = 0x05

} tenTouchMode;

//! Identifies the Key Modes
typedef enum
{
   //! Key is Released
   e8KEY_RELEASE = 0x00,
   //! Key is Pressed
   e8KEY_PRESS = 0x01,
   //! Long press of key
   e8KEY_LONGPRESS = 0x02,
   //! Soft Key is Pressed
   e8SOFTKEY_PRESS = 0x03,
   //! Soft Key is Released
   e8SOFTKEY_RELEASE = 0x04,

} tenKeyMode;

//! Identifies the Usage Preference
enum tenEnabledInfo
{
   //! Disables the chosen setting
   e8USAGE_DISABLED = 0x00,
   //! Enables the chosen setting
   e8USAGE_ENABLED = 0x01,
   //! Wait for user confirmation
   e8USAGE_CONF_REQD = 0x02,
   //! Value on virgin flash
   e8USAGE_UNKNOWN = 0xFF
};

//! Identifies the various screen aspect ratios
typedef enum
{
   //! Unknown Aspect Ratio
   e8ASPECT_UNKNOWN = 0x00,
   //! 4:3 Aspect Ratio
   e8ASPECT_4_3 = 0x01,
   //! 16:9 Aspect Ratio
   e8ASPECT_16_9 = 0x02,
} tenScreenAspectRatio;

//! Identifies the Connection Request Type
typedef enum
{
   //! Device Connect
   e8DEVCONNREQ_SELECT = 0x00,
   //! Device Disconnect
   e8DEVCONNREQ_DESELECT = 0x01,
   //! Device unknown
   e8DEVCONNREQ_UNKNOWN = 0x02,
} tenDeviceConnectionReq;


//! Identifies the Key Codes
typedef enum
{
   //! 0, ' '
   e32ITU_KEY_0 = 0x30000100,
   //! 1, '.', ','
   e32ITU_KEY_1 = 0x30000101,
   //! 2, a, b, c
   e32ITU_KEY_2 = 0x30000102,
   //! 3, d, e, f
   e32ITU_KEY_3 = 0x30000103,
   //! 4, g, h, i
   e32ITU_KEY_4 = 0x30000104,
   //! 5, j, k, l
   e32ITU_KEY_5 = 0x30000105,
   //! 6, m, n, 0
   e32ITU_KEY_6 = 0x30000106,
   //! 7, p,q, r, s
   e32ITU_KEY_7 = 0x30000107,
   //! 8, t, u, v
   e32ITU_KEY_8 = 0x30000108,
   //! 9, w, x, y, z
   e32ITU_KEY_9 = 0x30000109,
   //! *, +
   e32ITU_KEY_ASTERIX = 0x3000010A,
   //!#, shift
   e32ITU_KEY_POUND = 0x3000010B,
   //! Take a phone call
   e32DEV_PHONE_CALL = 0x30000200,
   //! End phone call
   e32DEV_PHONE_END = 0x30000201,
   //! Left soft key
   e32DEV_SOFT_LEFT = 0x30000202,
   //! Middle soft key
   e32DEV_SOFT_MIDDLE = 0x30000203,
   //! Right soft key
   e32DEV_SOFT_RIGHT = 0x30000204,
   //! Shortcut to the Application listing
   e32DEV_APPLICATION = 0x30000205,
   //! Ok
   e32DEV_OK = 0x30000206,
   //! Delete (Backspace)
   e32DEV_DELETE = 0x30000207,
   //! Zoom in
   e32DEV_ZOOM_IN = 0x30000208,
   //! Zoom out
   e32DEV_ZOOM_OUT = 0x30000209,
   //! Clear
   e32DEV_CLEAR = 0x3000020A,
   //! Go one step forward
   e32DEV_FORWARD = 0x3000020B,
   //! Go one step backward
   e32DEV_BACKWARD = 0x3000020C,
   //! Shortcut to the Home Screen
   e32DEV_HOME = 0x3000020D,
   //! Shortcut to the search function
   e32DEV_SEARCH = 0x3000020E,
   //! Shortcut to the (application) menu
   e32DEV_MENU = 0x3000020F,
   //! Shortcut to the (application) menu
   e32DEV_PTT = 0x30000210,
   //! Start media playing
   e32MULTIMEDIA_PLAY = 0x30000400,
   //! Pause media playing
   e32MULTIMEDIA_PAUSE = 0x30000401,
   //! Stop media playing
   e32MULTIMEDIA_STOP = 0x30000402,
   //! Forward
   e32MULTIMEDIA_FORWARD = 0x30000403,
   //! Rewind
   e32MULTIMEDIA_REWIND = 0x30000404,
   //! Go to next track in playlist
   e32MULTIMEDIA_NEXT = 0x30000405,
   //! Go to previous track in playlist
   e32MULTIMEDIA_PREVIOUS = 0x30000406,
   //! Mute the audio stream at source
   e32MULTIMEDIA_MUTE = 0x30000407,
   //! Un-mute the audio stream
   e32MULTIMEDIA_UNMUTE = 0x30000408,
   //! Take a Photo
   e32MULTIMEDIA_PHOTO = 0x30000409,
   //! Flash key
   e32DEV_PHONE_FLASH = 0x3000040A,
   // Below are internal keycodes used for launching the application.
   //! App Key Code for Media
   e32APP_KEYCODE_MEDIA = 0x3000040B,
   //! App Key Code for Telephony
   e32APP_KEYCODE_TELEPHONY = 0x3000040C,
   //! App Key Code for Navigation
   e32APP_KEYCODE_NAVIGATION = 0x3000040D,
   //Numeric Keys 0-9
   //! Numeric Key 0
   e32DEV_KEY_0 = 0x3000040E,
   //! Numeric Key 1
   e32DEV_KEY_1 = 0x3000040F,
   //! Numeric Key 2
   e32DEV_KEY_2 = 0x30000410,
   //! Numeric Key 3
   e32DEV_KEY_3 = 0x30000411,
   //! Numeric Key 4
   e32DEV_KEY_4 = 0x30000412,
   //! Numeric Key 5
   e32DEV_KEY_5 = 0x30000413,
   //! Numeric Key 6
   e32DEV_KEY_6 = 0x30000414,
   //! Numeric Key 7
   e32DEV_KEY_7 = 0x30000415,
   //! Numeric Key 8
   e32DEV_KEY_8 = 0x30000416,
   //! Numeric Key 9
   e32DEV_KEY_9 = 0x30000417,
   //! Dev Key *
   e32DEV_KEY_STAR = 0x30000418,
   //! Dev key #
   e32DEV_KEY_HASH = 0x30000419,

   //! Knob Right Shift
   e32TWODKNOB0_SHIFT_RIGHT = 0x3000041A,
   //! Knob Left Shift
   e32TWODKNOB0_SHIFT_LEFT = 0x3000041B,
   //! Knob up Shift
   e32TWODKNOB0_SHIFT_UP = 0x3000041C,
   //! Knob Down Shift
   e32TWODKNOB0_SHIFT_DOWN = 0x3000041D,
   //! Knob Shift Up Right
   e32TWODKNOB0_SHIFT_UP_RIGHT = 0x3000041E,
   //! Knob Shift Up Left
   e32TWODKNOB0_SHIFT_UP_LEFT = 0x3000041F,
   //! Knob Shift Down Right
   e32TWODKNOB0_SHIFT_DOWN_RIGHT = 0x30000420,
   //! Knob Shift Down Left
   e32TWODKNOB0_SHIFT_DOWN_LEFT = 0x30000421,
   //! Media
   e32DEV_MEDIA = 0x30000422,
   //! Naviagtion
   e32DEV_NAV = 0x30000423,
   //! Dev Key Plus
   e32DEV_KEY_PLUS = 0x30000424,
   //! Phone Mute
   e32DEV_PHONE_MUTE = 0x30000425,
   //! Play Pause Key Code
   e32MULTIMEDIA_PLAY_PAUSE = 0x30000426,
   //! Rotary Knob
   e32ROTARY_KNOB = 0x30000427,
   //! Merge Calls
   e32MERGE_CALLS = 0x30000428,
   //! Swap Calls
   e32SWAP_CALLS  = 0x30000429,
   //!Invalid Key Code
   e32INVALID_KEY = 0xFFFFFFFF
} tenKeyCode;

//! @note: Key Codes will be extended in future as required.

//! Identifies the application categories
typedef enum
{
   //! Unknown Application
   e32APPUNKNOWN = 0x00000000,
   //! General UI Framework
   e32APPUI = 0x00010000,
   //! Home screen/Start-up screen
   e32APP_UI_HOMESCREEN = 0x00010001,
   //! Menu
   e32APP_UI_MENU = 0x00010002,
   //! Notification
   e32APP_UI_NOTIFICATION = 0x00010003,
   //! Application Listing
   e32APP_UI_APPLICATIONLISTING = 0X00010004,
   //! Settings
   e32APP_UI_SETTINGS = 0X00010005,
   //! General Phone Call Application
   e32APP_PHONE = 0X00020000,
   //! Contact List
   e32APP_PHONE_CONTACTLIST = 0X00020001,
   //! Call Log
   e32APP_PHONE_CALLLOG = 0X00020002,
   //! General Media Applications
   e32APP_MEDIA = 0X00030000,
   //! Music
   e32APP_MEDIA_MUSIC = 0X00030001,
   //! Video
   e32APP_MEDIA_VIDEO = 0X00030002,
   //! Gaming
   e32APP_MEDIA_GAMING = 0X00030003,
   //! Image
   e32APP_MEDIA_IMAGE = 0X00030004,
   //! General Messaging Applications
   e32APP_MESSAGING = 0X00040000,
   //! SMS
   e32APP_MESSAGING_SMS = 0X00040001,
   //! MMS
   e32APP_MESSAGING_MMS = 0X00040002,
   //! Email
   e32APP_MESSAGING_EMAIL = 0X00040003,
   //! General Navigation Application
   e32APP_NAVIGATION = 0X00050000,
   //! General Browser Application
   e32APP_BROWSER = 0X00060000,
   //! Application Store
   e32APP_BROWSER_APPLICATIONSTORE = 0X00060001,
   //! General Productivity Application
   e32APP_PRODUCTIVITY = 0X00070000,
   //! Document Viewer
   e32APP_PRODUCTIVITY_DOCVIEWER = 0X00070001,
   //! Document Editor
   e32APP_PRODUCTIVITY_DOCEDITOR = 0X00070002,
   //! General Information
   e32APP_INFORMATION = 0X00080000,
   //! News
   e32APP_INFORMATION_NEWS = 0X00080001,
   //! Weather
   e32APP_INFORMATION_WEATHER = 0X00080002,
   //! Stocks
   e32APP_INFORMATION_STOCKS = 0X00080003,
   //! Travel
   e32APP_INFORMATION_TRAVEL = 0X00080004,
   //! Sports
   e32APP_INFORMATION_SPORTS = 0X00080005,
   //! Clock
   e32APP_INFORMATION_CLOCK = 0X00080006,
   //! General Social Networking Application
   e32APP_SOCIALNETWORKING = 0X00090000,
   //! General Personal Information Management(PIM)
   e32APP_PIM = 0X000A0000,
   //! Calendar
   e32APP_PIM_CALENDAR = 0X000A0001,
   //! Notes
   e32APP_PIM_NOTES = 0X000A0002,
   //! General UI less Application
   e32APP_NOUI = 0XF0000000,
   //! No UI Audio Server Application
   e32APP_NOUI_AUDIOSERVER = 0XF0000001,
   //! No UI Audio Client Application
   e32APP_NOUI_AUDIOCLIENT = 0XF0000002,
   //! Voice Command Engine
   e32APP_NOUI_VOICECMDENGINE = 0XF0000010,
   //! Conversational Audio
   e32APP_NOUI_CONVERSATIONAL_AUDIO = 0XF0000020,
   //! Switch to Mirror Link Client Native UI
   e32APP_SWITCHTO_CLIENT_NATIVE_UI = 0XF000FFFF,
   //! Mask for the Test&Certification Apps
   e32APP_CERT_TEST = 0XFFFE0000,
   //! General System
   e32APP_SYSTEM = 0XFFFF0000,
   //! PIN Input for Device Unlock
   e32APP_SYSTEM_INPUT_UNLOCK_PIN = 0XFFFF0001,
   //! Bluetooth PIN code Input
   e32APP_SYSTEM_INPUT_BTPIN = 0XFFFF0002,
   //! Other Password Input
   e32APP_SYSTEM_INPUT_OTHERPWD = 0XFFFF000F,
   //! Voice Command Confirmation
   e32APP_SYSTEM_VOICECMD_CONFIRMATION = 0XFFFF0010,
} tenAppCategory;

//! Identifies the content categories of an application
typedef enum
{
   //! Text Content
   e32CONTENT_TEXT = 0X00000001,
   //! Video Content
   e32CONTENT_VIDEO = 0X00000002,
   //! Image Content
   e32CONTENT_IMAGE = 0X00000004,
   //! Graphics Vector Content
   e32CONTENT_GRAPHICSVECTOR = 0X00000080,
   //! 3D Graphics Content
   e32CONTENT_GRAPHICS3D = 0X00000010,
   //! User Interface(e.g. ApplicationMenu)
   e32CONTENT_UI = 0X00000020,
   //! Car Mode (Application UI is complying with all rules for a restricted driving mode)
   e32CONTENT_CARMODE = 0X00010000,
   //! Miscellaneous Content
   e32CONTENT_MISC = 0X80000000,
} tenAppDisplayCategory;

//! Identifies the audio categories of an application
typedef enum
{
   //! Phone Audio
   e32PHONE_AUDIO = 0X00000001,
   //! Media Audio Out
   e32MEDIA_AUDIOOUT = 0X00000002,
   //! Media Audio In
   e32MEDIA_AUDIOIN = 0X00000004,
   //! Voice Command Out
   e32VOICE_COMMANDOUT = 0X00000080,
   //! Voice Command In
   e32VOICE_COMMANDIN = 0X00000010,
   //! Miscellaneous Content
   e32MISC_CONTENT = 0X80000000,
} tenAppAudioCategory;

//! Identifies the trust levels
typedef enum
{
   //! No Trust
   e8UNKNOWN = 0X00,
   //! Trust the User
   e8USER_CONFIGURATION = 0X40,
   //! Trust the application
   e8SELF_REGISTERED_APPLICATION = 0X60,
   //! Trust the VNC and UPnP MirrorLink server
   e8REGISTERED_APPLICATION = 0X80,
   //! Trust a 3rd party certification entity
   e8APPLICATION_CERTIFICATE = 0XA0,
} tenTrustLevel;

typedef enum
{
   //Certification status for ML1.0 devices & Not certified & Only CCC certified but cant be used in
   //Park mode & drive mode
   e8NOT_CERTIFIED = 0x00,
   // Don't use BASE CERTIFIED. App should be either NON CERTIFIED or PARK/DRIVE CERTIFIED,When we inform HMI.
   // If this is used, Updating Application List to HMi and App Filtering needs to be revisited.  
   // We still have this element in the SPI Interface. Commented just to prevent the usage.
   //CCC/CCC-Member certified
   //e8BASE_CERTIFIED = 0x01,
   //Only certified in Park mode
   e8NONDRIVE_CERTIFIED_ONLY = 0x02,
   //Only used in drive mode
   e8DRIVE_CERTIFIED_ONLY = 0x03,
   // Can be used in both the park mode & drive mode
   e8DRIVE_CERTIFIED = 0x04,
} tenAppCertificationInfo;

//! identifies App certfying entity
typedef enum
{
   e8UNKNOWN_ENTITY = 0, e8CCC_CERTIFIED = 1, e8CCC_MEMBER_CERTIFIED = 2
} tenAppCertificationEntity;

//! Identifies the Application Status Information
typedef enum
{
   //! Application running in Foreground.
   e8FOREGROUND = 0x00,
   //! Application running in Background
   e8BACKGROUND = 0x01,
   //! Application Not Running
   e8NOTRUNNING = 0x02,
   //! default value. This value is not available in SPI FI
   e8UNKNOWN_STATUS = 0x03
} tenAppStatus;

//! Identifies the display context of HMI or the projected device
typedef enum
{
   //! Context information not available.
   e8DISPLAY_CONTEXT_UNKNOWN = 0x00,
   //! Display context is speech recognition.
   e8DISPLAY_CONTEXT_SPEECH_REC = 0x01,
   //! Display context is phone display.
   e8DISPLAY_CONTEXT_PHONE = 0x02,
   //! Display context is navigation display.
   e8DISPLAY_CONTEXT_NAVIGATION = 0x03,
   //! Display context is media display.
   e8DISPLAY_CONTEXT_MEDIA = 0x04,
   //! Display context is informational.
   e8DISPLAY_CONTEXT_INFORMATIONAL = 0x05,
   //! Display context is emergency display.
   e8DISPLAY_CONTEXT_EMERGENCY = 0x06,
   //! Display context is a speaking display.
   e8DISPLAY_CONTEXT_SPEAKING = 0x07,
   //! Display Context is Turn-by-Type navigation display.
   e8DISPLAY_CONTEXT_TBT_NAVIGATION = 0x08,
   //! Display Context is sleep mode. System will wake up on incoming calls.
   e8DISPLAY_CONTEXT_SLEEP_MODE = 0x09,
   //! Display Context is sleep mode. System will not wake up on any event.
   e8DISPLAY_CONTEXT_STANDBY_MODE = 0x0A,
   //! Display Context is screen saver.
   e8DISPLAY_CONTEXT_SCREEN_SAVER = 0x0B,
   //! Display Context is native main screen.
   e8DISPLAY_CONTEXT_NATIVE = 0x0C,
   //! Display Context is projection screen.
   e8DISPLAY_CONTEXT_PROJECTION = 0x0D,
   //! Display context is user input. Settings screen which requires user input.
   e8DISPLAY_CONTEXT_USER_INPUT = 0x0E
} tenDisplayContext;

//! Identifies the location data usage request.
typedef enum
{
   //! Unknown Data Service. Default Value at StartUp.
   e8UNKNOWN_DATA_SERVICE = 0x00,
   //! GPS Data Service.
   e8GPS_DATA_SERVICE = 0x01,
   //! Location Data Service.
   e8LOCATION_DATA_SERVICE = 0x02,

} tenDataServiceType;
//! Identifies the RelativelyActiveMediaApp categoty.
typedef enum
{
   //! Online radio app is relatively active media app
   e8MEDIA_APP_TYPE_RADIO = 0x00,
   //! Music app is relatively active media app
   e8MEDIA_APP_TYPE_MUSIC = 0x01,
   //! Video app is relatively active media app.This is for future purpose
   e8APP_TYPE_VIDEO = 0x02,
   //! Game app is relatively active media app.This is for future purpose
   e8MEDIA_APP_TYPE_GAMING = 0x03,
   //! Image app is relatively active media app.This is for future purpose
   e8MEDIA_APP_TYPE_IMAGE = 0x04,
   //! Unknown app category. Default Value at Startup.
   e8MEDIA_APP_TYPE_UNKNOWN = 0xFF
} tenRelativelyActiveMediaAppCat;
//! Identifies the DAP Authentication progress information.
typedef enum
{
   //! Unknown DAP Status. Default Value at Startup.
   e8DAP_UNKNOWN = 0x00,
   //! DAP Authentication Not supported.
   e8DAP_NOT_SUPPORTED = 0x01,
   //! DAP Authentication In Progress.
   e8DAP_IN_PROGRESS = 0x02,
   //! DAP Authentication Successful.
   e8DAP_SUCCESS = 0x03,
   //! DAP Authentication failed.
   e8DAP_FAILED = 0x04,

} tenDAPStatus;

//! Identifies the the application to be launched on a DiPO device.
typedef enum
{
   //! Default Value at Startup.
   e8DIPO_NOT_USED = 0x00,
   //! DiPO HomeScreen.
   e8DIPO_NO_URL = 0x01,
   //! DiPO Maps Application.
   e8DIPO_MAPS = 0x02,
   //! DiPO MobilePhone Application.
   e8DIPO_MOBILEPHONE = 0x03,
   //! DiPO Telephone Number Application.
   e8DIPO_TEL_NUMBER = 0x04,
   //! DiPO Siri Application for Pre Warning.
   e8DIPO_SIRI_PREWARN = 0x05,
   //! DiPO Siri Application for Button Down Event.
   e8DIPO_SIRI_BUTTONDOWN = 0x06,
   //! DiPO Siri Application Button Up Event.
   e8DIPO_SIRI_BUTTONUP = 0x07,
   //! DiPO music application
   e8DIPO_MUSIC = 0x08,
   //! DiPO now playing application.
   e8DIPO_NOWPLAYING = 0x09
} tenDiPOAppType;

//! Identifies the ECNR configuration for launching DiPO application.
typedef enum
{
   //! Settings unchanged for this dial event.
   e8ECNR_NOCHANGE = 0x00,
   //! For Telephone call.
   e8ECNR_VOICE = 0x01,
   //! For Voice Recognition.
   e8ECNR_SERVER = 0x02,

} tenEcnrSetting;

//! Identifies the change in Bluetooth device
enum tenBTChangeInfo
{
   //! Default value
   e8NO_CHANGE = 0,
   //! Switching from BT to ML device
   e8SWITCH_BT_TO_ML = 1,
   //! Switching from BT to DiPO device
   e8SWITCH_BT_TO_DIPO = 2,
   //! Switching from ML to BT device
   e8SWITCH_ML_TO_BT = 3,
   //! Switching from DiPO to BT device
   e8SWITCH_DIPO_TO_BT = 4,
   //! Switching from BT to AAP device
   e8SWITCH_BT_TO_AAP = 5,
   //! Switching from AAP to BT device
   e8SWITCH_AAP_TO_BT = 6
};

//! This enumeration identifies State of the current playing song
enum tenMediaPlayBackState
{
   //! Current track is stopped playing
   e8PLAYBACK_STOPPED = 0x00,
   //! Current track is in playing state
   e8PLAYBACK_PLAYING = 0x01,
   //! Current track is in paused state
   e8PLAYBACK_PAUSED = 0x02,
   //! Current track is seeking forward
   e8PLAYBACK_SEEKFORWARD = 0x03,
   //! Current track is seeking backward
   e8PLAYBACK_SEEKBACKWARD = 0x04,
   //! Identifies Default value(Startup case) or status is not known
   e8PLAYBACK_NOT_KNOWN = 0xFF
};

//! This enumeration identifies shuffle state of the song
enum tenMediaPlayBackShuffleState
{
   //! Shuffle is off
   e8SHUFFLE_OFF = 0x00,
   //! Shuffle is on for Songs
   e8SHUFFLE_SONGS = 0x01,
   //! Shuffle is on for albums
   e8SHUFFLE_ALBUMS = 0x02,
   //! Shuffle is on in the device, however whether it is for songs or albums etc is upto the device specific
   e8SHUFFLE_ON = 0x03,
   //! Identifies Default value(Startup case) or status is not known
   e8SHUFFLE_NOT_KNOWN = 0xFF
};

//! This enumeration identifies Repeat state of the song
enum tenMediaPlayBackRepeatState
{
   //! Repeat is off
   e8REPEAT_OFF = 0x00,
   //! Repeat is on for one song
   e8REPEAT_ONE = 0x01,
   //! Repeat is on for all the songs
   e8REPEAT_ALL = 0x02,
   //! Identifies Default value(Startup case) or status is not known
   e8REPEAT_NOT_KNOWN = 0xFF
};

//! This enumeration identifies current playing media type
enum tenMediaPlayBackMediaType
{
   //! Its a music app
   e8MEDIATYPE_MUSIC = 0x00,
   //! It is pod cast<
   e8MEDIATYPE_PODCAST = 0x01,
   //! Its a audio book.
   e8MEDIATYPE_AUDIOBOOK = 0x02,
   //! Its a iTunes.
   e8MEDIATYPE_ITUESU = 0x03,
   //! Identifies Default value(Startup case) or status is not known
   e8MEDIATYPE_NOT_KNOWN = 0xFF
};

//! This enumeration identifies current playing media type
enum tenSignalStrength
{
   //! Zero bars to be shown on display
   e8ZERO_BARS = 0x00,
   //! One bar to be shown on display
   e8ONE_BAR = 0x01,
   //! Two bars to be shown on display
   e8TWO_BARS = 0x02,
   //! Three bars to be shown on display
   e8THREE_BARS = 0x03,
   //! Four bars to be shown on display
   e8FOUR_BARS = 0x04,
   //! Five bars to be shown on display
   e8FIVE_BARS = 0x05,
   //! Identifies Default value(Startup case) or status is not known
   e8SIGNAL_STRENGTH_NOT_KNOWN = 0xFF
};

//! This enumeration identifies registration status of the SIM on phone side
enum tenRegistrationStatus
{
   //! Not registered
   e8NOT_REGISTERED = 0x01,
   e8SEARCHING = 0x02,
   e8DENIED = 0x03,
   e8REGISTERED_HOME = 0x04,
   e8REGISTERED_ROAMING = 0x05,
   e8EMERGENCY_CALLS_ONLY = 0x06,
   //! Identifies Default value(Startup case) or status is not known
   e8PHONE_REGISTRATION_NOT_KNOWN = 0xFF
};
//! This enumeration identifies AirPlane status on the phone
enum tenAirPlaneModeStatus
{
   //! AirPlane is off
   e8AIRPLANE_OFF = 0x00,
   //! AirPlne is on
   e8AIRPLANE_ON = 0x01,
   //! Identifies Default value(Startup case) or status is not known
   e8PHONE_AIRPLANE_MODE_NOT_KNOWN = 0xFF
};

//! This enumeration identifies AirPlane status on the phone
enum tenPhoneMuteStatus
{
   //! mute is off
   e8PHONE_MUTE_OFF = 0x00,
   //! mute is on
   e8PHONE_MUTE_ON = 0x01,
   //! Identifies Default value(Startup case) or status is not known
   e8PHONE_MUTESTATE_NOT_KNOWN = 0xFF
};

//! This enumeration identifies call direction from the phone
enum tenPhoneCallDirection
{
   //! call is Incoming
   e8CALL_INCOMING = 0x01,
   //! call is Outgoing
   e8CALL_OUTGOING = 0x02,
   //! Identifies Default value(Startup case) or status is not known
   e8PHONE_CALL_DIRECTION_NOT_KNOWN = 0xFF
};

//! This enumeration identifies call state on the phone
//! This enum is used in populating phone call metadata
enum tenPhoneCallState
{
   //! Phone call is disconnected or no active call
   e8CALL_DISCONNECTED = 0x00,
   e8CALL_SENDING = 0x01,
   //! Phone call is ringing.
   e8CALL_RINGING = 0x02,
   //! Phone call is connecting
   e8CALL_CONNECTING = 0x03,
   //! Phone call is active
   e8CALL_ACTIVE = 0x04,
   //! Phone call is on hold.
   e8CALL_HELD = 0x05,
   //! Phone call is disconnecting.
   e8CALL_DISCONNECTING = 0x06,
   //! Identifies Default value(Startup case) or status is not known
   e8PHONE_CALL_STATE_NOT_KNOWN = 0xFF
};

enum tenCallStateUpdateService
{
   //! call Service : Telephony
   e8CALLSTATE_SERVICE_TELEPHONY = 0x00,
   //! call Service : Face Time Audio
   e8CALLSTATE_SERVICE_FACETIMEAUDIO = 0x01,
   //! call Service : Face Time Video
   e8CALLSTATE_SERVICE_FACETIMEVIDEO = 0x02,
   //! call Service : Unknown
   e8CALLSTATE_SERVICE_UNKNOWN = 0xFF
};

enum tenCallStateUpdateDisconnectReason
{
   //! call disconnect reason update : Call Ended
   e8CALLSTATE_DISCONNECT_REASON_ENDED = 0x00,
   //! call disconnect reason update : Call Declined
   e8CALLSTATE_DISCONNECT_REASON_DECLINED = 0x01,
   //! call disconnect reason update : Call Failed
   e8CALLSTATE_DISCONNECT_REASON_FAILED = 0x02,
   //! call disconnect reason update : Unknown
   e8CALLSTATE_DISCONNECT_REASON_UNKNOWN = 0xFF
};

/************structures***************************/
//! \brief This section lists the different structures used by the interfaces noted below
//! This provides Mirror Link version information and contains the following elements
struct trVersionInfo
{
      //! Major Version
      t_U32 u32MajorVersion; //TODO update in API doc
      //! Minor Version
      t_U32 u32MinorVersion;
      //! Patch Version (Set to NULL if not available)
      t_U32 u32PatchVersion;

//! Initialise structure members to 0
      trVersionInfo() :
               u32MajorVersion(0), u32MinorVersion(0), u32PatchVersion(0)
      {

      }
//! Assignment operator
      trVersionInfo& operator=(const trVersionInfo& rfcorVersionInfo)
      {
         if (&rfcorVersionInfo != this)
         {
            u32MajorVersion = rfcorVersionInfo.u32MajorVersion;
            u32MinorVersion = rfcorVersionInfo.u32MinorVersion;
            u32PatchVersion = rfcorVersionInfo.u32PatchVersion;
         }      //if (&rfcorVersionInfo != this)
         return *this;
      }
};

//!Contains Key Capabilities
struct trKeyCapabilities
{
      /*BitField*/
      //! Contains support information for knob keys.
      t_U32 u32KnobKeySupport;
      /*BitField*/
      //! Contains support information for various device keys.
      t_U32 u32DeviceKeySupport;
      /*BitField*/
      //! Contains support information for various multimedia keys.
      t_U32 u32MultimediaKeySupport;
      /*BitField*/
      //! Contains support information for various miscellaneous keys.
      t_U32 u32MiscKeySupport;
      /*BitField*/
      //! Contains support pointer or touch support
      t_U32 u32PointerTouchSupport;

//! Initialise structure members to 0
      trKeyCapabilities() :
                        u32KnobKeySupport(0),
                        u32DeviceKeySupport(0),
                        u32MultimediaKeySupport(0),
                        u32MiscKeySupport(0),
                        u32PointerTouchSupport(0)
      {

      }
//! Assignment operator
      trKeyCapabilities& operator=(const trKeyCapabilities& rfcoKeyCapabilities)
      {
         if (&rfcoKeyCapabilities != this)
         {
            u32KnobKeySupport = rfcoKeyCapabilities.u32KnobKeySupport;
            u32DeviceKeySupport = rfcoKeyCapabilities.u32DeviceKeySupport;
            u32MultimediaKeySupport = rfcoKeyCapabilities.u32MultimediaKeySupport;
            u32MiscKeySupport = rfcoKeyCapabilities.u32MiscKeySupport;
            u32PointerTouchSupport = rfcoKeyCapabilities.u32PointerTouchSupport;
         }      //if (&rfcoKeyCapabilities != this)
         return *this;
      }
};

//! This provides information about projection details

struct trProjectionCapability
{
      //! Port type. Indicates OTG or non OTG port
      tenUSBPortType enUSBPortType;
      //! Device type. indicates the OS running on the device
      tenDeviceType enDeviceType;
      //! Carplay capability
      tenSPISupport enCarplaySupport;
      //! Carplay Wireless capability
      tenSPISupport enCarplayWirelessSupport;
      //! Android Auto capability
      tenSPISupport enAndroidAutoSupport;
      //! Mirrorlink capability
      tenSPISupport enMirrorlinkSupport;
      tenSPISupport enMySPINSupport;
      tenSPISupport enCarlifeSupport;
      //! OnCar capability
      tenSPISupport enOnCarSupport;

      trProjectionCapability() :
                        enUSBPortType(e8_PORT_TYPE_NOT_KNOWN),
                        enDeviceType(e8_UNKNOWN_DEVICE),
                        enCarplaySupport(e8SPI_SUPPORT_UNKNOWN),
                        enCarplayWirelessSupport(e8SPI_SUPPORT_UNKNOWN),
                        enAndroidAutoSupport(e8SPI_SUPPORT_UNKNOWN),
                        enMirrorlinkSupport(e8SPI_SUPPORT_UNKNOWN),
                        enMySPINSupport(e8SPI_SUPPORT_UNKNOWN),
                        enCarlifeSupport(e8SPI_SUPPORT_UNKNOWN),
                        enOnCarSupport(e8SPI_SUPPORT_UNKNOWN)
      {

      }
};

//! Enum for authorization status
enum tenUserAuthorizationStatus
{
   e8_USER_AUTH_UNKNOWN = 0x00,
   e8_USER_AUTH_UNAUTHORIZED = 0x01,
   e8_USER_AUTH_AUTHORIZED = 0x02,
   e8_USER_AUTH_ABORTED = 0x03,
   e8_USER_AUTH_CANCELLED = 0x04
};

//! indictaes the device selection status
enum tenDeviceSelectionState
{
   e8_SELECTION_STATE_DEVICE_NOT_SELECTED = 0,
   e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS = 1,
   e8_SELECTION_STATE_DEVICE_SELECTED = 2,
   e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS = 3,
   e8_SELECTION_STATE_DEVICE_USER_DESELECTED = 4,
   e8_SELECTION_STATE_DEVICE_SELECTION_ERROR = 5,
   e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR = 6,
   e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS = 7,
   e8_SELECTION_STATE_DEVICE_RETRY_INPROGRESS = 8
};

//! indictaes the Device Selection ProgressState
enum tenDeviceSelectionProgressState
{
   e8_SELECTION_PROGRESS_STATE_NOT_APPLICABLE = 0,
   e8_SELECTION_PROGRESS_STATE_SWITCHING_PROFILE = 1,
   e8_SELECTION_PROGRESS_STATE_PREPARING_ENDPOINTS = 2,
   e8_SELECTION_PROGRESS_STATE_SERVICE_DISCOVERY = 3,
   e8_SELECTION_PROGRESS_STATE_AUTHENTICATING = 4,
   e8_SELECTION_PROGRESS_STATE_RESOURCE_INITIALIZATION = 5
};

typedef std::function<t_Void(const t_U32, tenDeviceSelectionProgressState)> tfvSelectionProgressState;

//! indicates the error occured during device selection
enum tenDeviceSelectionErrorType
{
   e8_DEVICE_SELECTION_ERROR_NONE = 0,
   e8_DEVICE_SELECTION_ERROR_NOT_CONNECTED = 1,
   e8_DEVICE_SELECTION_ERROR_RESOURCE_BUSY = 2,
   e8_DEVICE_SELECTION_DEVICE_ERROR = 3,
   e8_DEVICE_SELECTION_ERROR_USBRESET_FAILED = 4,
   e8_DEVICE_SELECTION_CANCELLED_BY_USER = 5,
   e8_DEVICE_SELECTION_DEVICE_ERROR_FATAL = 6,
   e8_DEVICE_SELECTION_ERROR_PHONE_UNSUPPORTED = 7,
   e8_DEVICE_SELECTION_ERROR_APP_NOT_INSTALLED = 8,
   e8_DEVICE_SELECTION_WIFI_AP_SETUP_ERROR = 9,
   e8_DEVICE_SELECTION_STA_DISCONNECTED = 10,
   e8_DEVICE_SELECTION_ERROR_SWITCH_FAILED = 11
};

//! Structure to hold device selection information such as status and error
struct trDeviceSelectionInfo
{
      tenDeviceSelectionState enDeviceSelectionState;
      tenDeviceSelectionErrorType enDeviceSelectionErrorType;
      tenDeviceSelectionProgressState enDeviceSelectionProgressState;

      trDeviceSelectionInfo() :
                        enDeviceSelectionState(e8_SELECTION_STATE_DEVICE_NOT_SELECTED),
                        enDeviceSelectionErrorType(e8_DEVICE_SELECTION_ERROR_NONE),
                        enDeviceSelectionProgressState(e8_SELECTION_PROGRESS_STATE_NOT_APPLICABLE)
      {

      }
};

enum tenDeviceProfile
{
   e8_PROFILE_UNKNOWN = 0,
   e8_PROFILE_DEFAULT = 1,
   e8_PROFILE_AOAP = 2,
   e8_PROFILE_CDCNCM_MIRRORLINK = 3,
   e8_PROFILE_CDCNCM_CARPLAY = 4,
   e8_PROFILE_NATIVETRANSPORT_MYSPIN = 5,
   e8_PROFILE_HOSTMODE_CARPLAY = 6,
   e8_PROFILE_IAP2BT_CPW_FEASIBLE = 7,
   e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE = 8,
   e8_PROFILE_NATIVETRANSPORT_CARLIFE = 9,
   e8_PROFILE_CDCNCM_UNKNOWNMODE = 10
};

enum tenSessionTransport
{
   e8_SESSION_TRANSPORT_UNKNOWN = 0,
   e8_SESSION_TRANSPORT_USB = 1,
   e8_SESSION_TRANSPORT_WIFI = 2,
   e8_SESSION_TRANSPORT_INVALID = 3
};

struct trConnectionStatus
{
      tenDeviceConnectionStatus enUSBConnectionStatus;
      tenDeviceConnectionStatus enWiFiConnectionStatus;
      tenDeviceConnectionStatus enBTConnectionStatus;
      trConnectionStatus() :
                        enUSBConnectionStatus(e8DEV_NOT_CONNECTED),
                        enWiFiConnectionStatus(e8DEV_NOT_CONNECTED),
                        enBTConnectionStatus(e8DEV_NOT_CONNECTED)
      {

      }
};

//! This provides attributes of the connected Device
struct trDeviceInfo
{
      //! Unique device identifier
      t_U32 u32DeviceHandle;
      //! Name of the Device
      std::string szDeviceName;
      //! Type of the Device
      tenDeviceCategory enDeviceCategory;
      //! Model Name of the Device.
      std::string szDeviceModelName;
      //! Manufacturer Name of the Device.
      std::string szDeviceManufacturerName;
      //! Connection Status of the device
      tenDeviceConnectionStatus enDeviceConnectionStatus;
      //! Connection Type of the device (Deprecated instead use trConnectionStatus)
      tenDeviceConnectionType enDeviceConnectionType;
      //! indicates the type of device connection (BT/WiFi/USB etc)
      trConnectionStatus rConnectionStatus;
      //! Contains version based on the Device Category(MirrorLink/iPodOut)
      trVersionInfo rVersionInfo;
      //!Additional field is added in place of t_Bool bDeviceUsageEnabled; this will be changed during the release.
      tenEnabledInfo enDeviceUsageEnabled;
      //! This value is set to TRUE if a device is automatically selected to initiate a ses-sion, FALSE otherwise. 
      //! Specifically used in case of startup.
      t_Bool bSelectedDevice;
      //! Set to TRUE if device sup-ports DAP, FALSE other-wise.
      t_Bool bDAPSupport;
      //! Bluetooth address of the device, Set to NULL if not available.
      std::string szBTAddress;
      //! provides information about projection details
      trProjectionCapability rProjectionCapability;
      //! provides information about user authorization status
      tenUserAuthorizationStatus enUserAuthorizationStatus;
      //! Unique ID to identify device across connection modes
      t_String szUUID;
      //! Serial number
      t_String szSerialNumber;
      //! system path information (CRQ 430789: [A-IVI][G3G] - Interface to update USB port using system path info)
      t_String szSystemPath;
      //! Device profile
      tenDeviceProfile enDeviceProfile;
      trDeviceSelectionInfo rDeviceSelectionInfo;
      tenSessionTransport enSessionTransport;
      //! Vendor ID
      t_U32 u32VendorID;
      //! Product ID
       t_U32 u32ProductID;
      //! BT Limited Functionality
       t_Bool bIsBTFunctionalityLimited;

      //! Note: Please add corresponding elements to assignment operator
      trDeviceInfo() :
                        u32DeviceHandle(0),
                        enDeviceCategory(e8DEV_TYPE_UNKNOWN),
                        enDeviceConnectionStatus(e8DEV_NOT_CONNECTED),
                        enDeviceConnectionType(e8UNKNOWN_CONNECTION),
                        enDeviceUsageEnabled(e8USAGE_UNKNOWN),
                        bSelectedDevice(false),
                        bDAPSupport(false),
                        enUserAuthorizationStatus(e8_USER_AUTH_UNKNOWN),
                        enDeviceProfile(e8_PROFILE_UNKNOWN),
                        enSessionTransport(e8_SESSION_TRANSPORT_UNKNOWN),
                        u32VendorID(0), u32ProductID(0), bIsBTFunctionalityLimited(false)
      {
      }
};

//! This provides information about the Remoting Info used to interact with the application after it is launched and contains the following elements
struct trAppRemotingInfo
{
      //! Protocol Identifier of the Remoting protocol that will be used to access the application.
      std::string szRemotingProtocolID;

      //! Format of the data being transferred using the Remoting Protocol.
      std::string szRemotingFormat;
      //! Direction of the content stream.
      std::string szRemotingDirection;
      //! Audio Initial Playback Latency.
      t_U32 u32RemotingAudioIPL;
      //! Audio Maximum Playback Latency.
      t_U32 u32RemotingAudioMPL;

      //! Initialise structure members to 0
      trAppRemotingInfo() :
               u32RemotingAudioIPL(0xFFFFFFFF), u32RemotingAudioMPL(0xFFFFFFFF)
      {

      }
      //Assignment operator
      trAppRemotingInfo& operator=(const trAppRemotingInfo& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            szRemotingProtocolID = corfrSrc.szRemotingProtocolID.c_str();
            szRemotingFormat = corfrSrc.szRemotingFormat.c_str();
            szRemotingDirection = corfrSrc.szRemotingDirection.c_str();
            u32RemotingAudioIPL = corfrSrc.u32RemotingAudioIPL;
            u32RemotingAudioMPL = corfrSrc.u32RemotingAudioMPL;
         }
         return *this;
      }

};

//! This provides Application Display Information and contains the following elements
struct trAppAudioInfo
{
      //! Application Audio Content Type.
      std::string szAppAudioType;

      //! Application Audio Content Category.
      tenAppAudioCategory enAppAudioCategory;
      //! Application Audio Content Trust Level.
      tenTrustLevel enTrustLevel;
      //! Application Audio Content Rules.
      t_U32 u32AppAudioRules;

//! Initialise structure members to 0
      trAppAudioInfo() :
               enAppAudioCategory(e32MISC_CONTENT), enTrustLevel(e8UNKNOWN), u32AppAudioRules(0xFFFFFFFF)
      {

      }
      //Assignment operator
      trAppAudioInfo& operator=(const trAppAudioInfo& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            szAppAudioType = corfrSrc.szAppAudioType.c_str();
            enAppAudioCategory = corfrSrc.enAppAudioCategory;
            enTrustLevel = corfrSrc.enTrustLevel;
            u32AppAudioRules = corfrSrc.u32AppAudioRules;
         }
         return *this;
      }

};

//! This contains Icon attributes.
struct trIconAttributes
{
      //! Width of the icon
      t_U32 u32IconWidth;
      //! Height of the icon
      t_U32 u32IconHeight;
      //! Depth of the icon
      t_U32 u32IconDepth;
      //! Mime Type of the icon
      tenIconMimeType enIconMimeType;
      //! URL of the icon.Data for the icon can be retrieved via GetApplicationIcon Interface
      std::string szIconURL;
      //! Initialise structure members to 0
      trIconAttributes() :
                        u32IconWidth(0xFFFFFFFF),
                        u32IconHeight(0xFFFFFFFF),
                        u32IconDepth(0xFFFFFFFF),
                        enIconMimeType(e8ICON_INVALID)
      {

      }
      //Assignment operator
      trIconAttributes& operator=(const trIconAttributes& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            u32IconWidth = corfrSrc.u32IconWidth;
            u32IconHeight = corfrSrc.u32IconHeight;
            u32IconDepth = corfrSrc.u32IconDepth;
            enIconMimeType = corfrSrc.enIconMimeType;
            szIconURL = corfrSrc.szIconURL.c_str();
         }
         return *this;
      }

};

//! This provides Application Display Information and contains the following elements
struct trAppDisplayInfo
{
      //! Application Display Content Category.
      tenAppDisplayCategory enAppDisplayCategory;
      //! Application Display Content Trust Level.
      tenTrustLevel enTrustLevel;
      //! Application Display Content Rules.
      t_U32 u32AppDisplayRules;
      //! Application Display Content Orientation.
      std::string szAppDisplayOrientation;

//! Initialise structure members to 0
      trAppDisplayInfo() :
               enAppDisplayCategory(e32CONTENT_MISC), enTrustLevel(e8UNKNOWN), u32AppDisplayRules(0xFFFFFFFF)
      {

      }
      trAppDisplayInfo& operator=(const trAppDisplayInfo& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            enAppDisplayCategory = corfrSrc.enAppDisplayCategory;
            enTrustLevel = corfrSrc.enTrustLevel;
            u32AppDisplayRules = corfrSrc.u32AppDisplayRules;
            szAppDisplayOrientation = corfrSrc.szAppDisplayOrientation.c_str();
         }
         return *this;
      }
};

//! This provides detailed application information and contains the following elements
struct trAppDetails
{
      //! Unique application identifier on the device.
      t_U32 u32AppHandle;

      //! String  Equivalent App ID
      t_U32 u32ConvAppHandle;

      //! Name of the application.
      std::string szAppName;
      //! Status of the application.
      tenAppStatus enAppStatus;
      //! Application Variant.
      std::string szAppVariant;
      //! Application Provider Name.
      std::string szAppProviderName;
      //! Application Provider URL.
      std::string szAppProviderURL;
      //! Application Description.
      std::string szAppDescription;
      //! Allowed Profiles for the application. //TODO
      std::vector<std::string> AppAllowedProfiles;
      //! URL of application certificate.
      std::string szAppCertificateURL;
      //! Application Category.
      tenAppCategory enAppCategory;
      //! Application Trust Level.
      tenTrustLevel enTrustLevel;
      //! Application Display Information
      trAppDisplayInfo rAppDisplayInfo;
      //! Application Audio Information
      trAppAudioInfo rAppAudioInfo;
      //! Application Remoting Information
      trAppRemotingInfo rAppRemotingInfo;
      //! Number of Application Icons
      t_U32 u32NumAppIcons;
      //! Application Icon Details. //TODO
      std::vector<trIconAttributes> tvecAppIconList;
      //! Notification support info
      t_Bool bNotificationSupport;

      tenAppCertificationInfo enAppCertificationInfo;
      tenAppCertificationEntity enAppCertificationEntity;

      //! Initialise structure members to 0
      trAppDetails() :
                        u32AppHandle(0),
                        u32ConvAppHandle(0),
                        enAppStatus(e8NOTRUNNING),
                        enAppCategory(e32APPUNKNOWN),
                        enTrustLevel(e8UNKNOWN),
                        u32NumAppIcons(0),
                        bNotificationSupport(false),
                        enAppCertificationInfo(e8NOT_CERTIFIED),
                        enAppCertificationEntity(e8UNKNOWN_ENTITY)
      {

      }
      //Assignment operator
      trAppDetails& operator=(const trAppDetails& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            //copy info from corfrSrc
            u32AppHandle = corfrSrc.u32AppHandle;
            u32ConvAppHandle = corfrSrc.u32ConvAppHandle;
            szAppName = corfrSrc.szAppName.c_str();
            enAppStatus = corfrSrc.enAppStatus;
            szAppVariant = corfrSrc.szAppVariant.c_str();
            szAppProviderName = corfrSrc.szAppProviderName.c_str();
            szAppProviderURL = corfrSrc.szAppProviderURL.c_str();
            szAppDescription = corfrSrc.szAppDescription.c_str();
            AppAllowedProfiles = corfrSrc.AppAllowedProfiles;
            szAppCertificateURL = corfrSrc.szAppCertificateURL.c_str();
            enAppCategory = corfrSrc.enAppCategory;
            enTrustLevel = corfrSrc.enTrustLevel;
            rAppDisplayInfo = corfrSrc.rAppDisplayInfo;
            rAppAudioInfo = corfrSrc.rAppAudioInfo;
            rAppRemotingInfo = corfrSrc.rAppRemotingInfo;
            u32NumAppIcons = corfrSrc.u32NumAppIcons;
            tvecAppIconList = corfrSrc.tvecAppIconList;
            enAppCertificationInfo = corfrSrc.enAppCertificationInfo;
            bNotificationSupport = corfrSrc.bNotificationSupport;
         }
         return *this;
      }

};

//! This provides information of the application
struct trAppInfo
{
      //! Unique device identifier
      t_U32 u32DeviceHandle;
      //! Number of applications on the device.
      t_U32 u32NumOfApps;
      //! std::vector<>Application Details //TODO
      std::vector<trAppDetails> tvecAppDetailsList;

//! Initialise structure members to 0
      trAppInfo() :
               u32DeviceHandle(0), u32NumOfApps(0)
      {

      }

};

//! This provides session status information for a device and contains the following elements
struct trSessionStatusInfo
{
      //! Unique device Identifier.
      t_U32 u32DeviceHandle;
      //! Current Session Status
      tenSessionStatus enSessionStatus;

//! Initialise structure members to 0
      trSessionStatusInfo() :
               u32DeviceHandle(0)
      {

      }
};

//! This identifies the X and Y Co-ordinates and contains the following elements
struct trTouchCoordinates
{
      //! Touch Mode information
      tenTouchMode enTouchMode;
      //! X Co-ordinate value
      t_S32 s32XCoordinate;
      //! Y Co-ordinate value
      t_S32 s32YCoordinate;
      //!Touch Identifier
      t_U8 u8Identifier;      //TODO : added update API document

//! Initialise structure members to 0
      trTouchCoordinates() :
               enTouchMode(e8TOUCH_UNKNOWN), s32XCoordinate(0), s32YCoordinate(0), u8Identifier(0)
      {

      }
};

//! This identifies Touch information and contains the following elements
struct trTouchInfo
{
      //! Information for various Touch points.
      std::vector<trTouchCoordinates> tvecTouchCoordinatesList;
      //! TimeStamp
      t_U32 u32TimeStamp; //TODO : added update API document
      t_U32 u32Pressure; //TODO : added update API document

//! Initialise structure members to 0
      trTouchInfo() :
               u32TimeStamp(0), u32Pressure(0)
      {

      }
};

//! This identifies Single/Multi-Touch information and contains the following elements
struct trTouchData
{
      //! Number of Touch Descriptors. In case of Single touch//! This value is 1
      t_U32 u32TouchDescriptors;
      //! Touch Information. In case of Single touch list has only one entry
      std::vector<trTouchInfo> tvecTouchInfoList;

//! Initialise structure members to 0
      trTouchData() :
               u32TouchDescriptors(0)
      {

      }
};

//! Provides Attributes to scale
struct trScalingAttributes
{
      //! New Height of the screen
      t_U32 u32ScreenHeight;
      //! New width of the screen
      t_U32 u32ScreenWidth;
      //! New X axis start coordinate
      t_S32 s32XStartCoordinate;
      //! New Y axis start coordinate
      t_S32 s32YStartCoordinate;
      //! Scaling value for phone width
      t_Float fWidthScaleValue;
      //! Scaling value of phone height
      t_Float fHeightScalingValue;

      trScalingAttributes() :
                        u32ScreenHeight(0),
                        u32ScreenWidth(0),
                        s32XStartCoordinate(0),
                        s32YStartCoordinate(0),
                        fWidthScaleValue(1.0),
                        fHeightScalingValue(1.0)
      {

      }

};

//!Provides screen attributes
struct trScreenAttributes
{
      //! Height of the screen
      t_U32 u32ScreenHeight;
      //! Width of the screen
      t_U32 u32ScreenWidth;
      //! Aspect Ratio.
      tenScreenAspectRatio enScreenAspectRatio;

//! Initialise structure members to 0
      trScreenAttributes() :
               u32ScreenHeight(0), u32ScreenWidth(0), enScreenAspectRatio(e8ASPECT_UNKNOWN)
      {

      }
};

//! Identifies the attributes associated with Video.
struct trVideoAttributes
{
      //! Orientation Mode.
      tenOrientationMode enOrientationMode;
      //! Screen size of the CE device.
      trScreenAttributes rScreenAttributes;

//! Default Constructor
      trVideoAttributes() :
               enOrientationMode(e8INVALID_MODE)
      {

      }
};

//!It identifies the application IDs for which notification has to be enabled / disabled.
struct trNotiEnable
{
      //! Unique Application Identifier
      t_U32 u32AppHandle;
      //! Enable/Disable Info
      tenEnabledInfo enEnabledInfo;

//! Initialise structure members to 0
      trNotiEnable() :
               u32AppHandle(0), enEnabledInfo(e8USAGE_UNKNOWN)
      {

      }
};

//! Provides Notification Action details
struct trNotiAction
{
      //! Notification action Identifier
      t_U32 u32NotiActionID;
      //! Notification action Name
      std::string szNotiActionName;
      //! Number of Notification Action Icons
      t_U32 u32NotiActionIconCount;
      //! Launch the application on this particular user action             
      t_Bool bLaunchAppReq;
      //! Details of Notification action icons
      std::vector<trIconAttributes> tvecNotiActionIconList;

//! Initialize structure members to 0
      trNotiAction() :
               u32NotiActionID(0), u32NotiActionIconCount(0), bLaunchAppReq(false)
      {

      }
};

//!Provides detailed information for a Notification Event
struct trNotiData
{
      //! Notification Event ID
      t_U32 u32NotiID;
      //! Title of the notification
      std::string szNotiTitle;
      //! Body of the Notification
      std::string szNotiBody;
      //! Number of Notification Icons
      t_U32 u32NotiIconCount;
      //! Details of Notification icons
      std::vector<trIconAttributes> tvecNotiIconList;
      //! Application ID of the notification
      t_U32 u32NotiAppID;
      //! Details of Notification Actions
      std::vector<trNotiAction> tvecNotiActionList;

//! Initialise structure members to 0
      trNotiData() :
               u32NotiID(0), u32NotiIconCount(0), u32NotiAppID(0)
      {

      }
};

//!Structure containing Details of supported XdeviceKeys
struct trXDeviceKeyDetails
{
      //! X-Device Key Id (Internal to SPI)
      t_U8 u8KeyId;

      //!X-Device key Name
      std::string szXDeviceName;

      //!X-Device Manadatory Status
      t_Bool bIsMandatory;

      //!X-Device Symbol Value
      tenKeyCode enSymbolValue;

      //!List of Icons for each X-Device key
      std::vector<trIconAttributes> tvecKeyIconList;
      //! Initialise structure members to 0
      trXDeviceKeyDetails() :
               u8KeyId(0), bIsMandatory(false), enSymbolValue(e32INVALID_KEY)
      {

      }
      //Assignment operator
      trXDeviceKeyDetails& operator=(const trXDeviceKeyDetails& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            //copy info from corfrSrc
            bIsMandatory = corfrSrc.bIsMandatory;
            szXDeviceName = corfrSrc.szXDeviceName;
            enSymbolValue = corfrSrc.enSymbolValue;
            tvecKeyIconList = corfrSrc.tvecKeyIconList;
         }
         return *this;
      }

};

//!Contains client capabilities
struct trClientCapabilities
{
      //! Provides Key Capabilities
      trKeyCapabilities rKeyCapabilities;

//!Default Constructor
      trClientCapabilities()
      {
      }
};

//!Contains Server Key capabilities
struct trMLSrvKeyCapabilities
{
      //! Provides Key Capabilities
      trKeyCapabilities rKeyCapabilities;

      //! Provides XDevice Keys Capabilities
      std::vector<trXDeviceKeyDetails> vecrXDeviceKeyInfo;

      //!Default Constructor
      trMLSrvKeyCapabilities()
      {
      }
};

//!Provides Media album art image data
struct trAppMediaAlbumArt
{
      //! URL for image data
      std::string szImageUrl;
      //! Image data
      std::vector<t_U8> u8Items;
      //! Initialize structure members to default values
      trAppMediaAlbumArt() :
               u8Items(0)
      {

      }
};

/*
 //!Provides application specific metadata information associated with currently active application (audio/ video).
 struct trAppMetaData
 {
 //! Validity flag of Metadata
 t_Bool bMetadataValid;
 //! Name of the artist.
 std::string szArtist;
 //! Title of currently active application.
 std::string szTitle;
 //! Name of the album.
 std::string szAlbum;
 //! Name of the Genre.
 std::string szGenre;
 //! Total Play time of the media object in seconds.
 t_U32 u32TotalPlayTime;
 //! Elapsed play time of the media object in seconds.
 t_U32 u32ElapsedPlayTime;
 //! Track number
 t_U32 u32TrackNumber;
 //! Name of the composer
 std::string szComposerName;
 //! URL for image data.
 std::string szImageUrl;
 //! Image size in bytes.
 t_U32 u32ImageSize;
 //! MIME type of the image pointed by ImageUrl.
 std::string szImageMIMEType;
 //! Name of the application.
 std::string szAppName;
 //! Uutgoing caller name, in-coming caller ID or phone number.
 std::string szPhoneCaller;
 //! Formatted string of phone call information.
 std::string szPhoneCallInfo;

 //! Initialise structure members to 0
 trAppMetaData() :
 bMetadataValid(false), u32TotalPlayTime(0), u32ElapsedPlayTime(0),
 u32ImageSize(0), u32TrackNumber(0)
 {

 }

 };
 */

//!Provides Media application specific metadata information associated with currently active application (audio/ video).
struct trAppMediaMetaData
{
      //! Validity flag of Metadata
      t_Bool bMediaMetadataValid;
      //! Name of the application.
      std::string szAppName;
      //! Title of currently active application.
      std::string szTitle;
      //! Name of the artist.
      std::string szArtist;
      //! Name of the album.
      std::string szAlbum;
      //! Name of the albumartist.
      std::string szAlbumArtist;
      //! Name of the Genre.
      std::string szGenre;
      //! Name of the composer
      std::string szComposerName;
      //! Album Track number
      t_U32 u32TrackNumber;
      //! Album Track Count
      t_U32 u32AlbumTrackCount;
      //! Album Disc number
      t_U32 u32AlbumDiscNumber;
      //! Album Disc Count
      t_U32 u32AlbumDiscCount;
      //! Chapter Count
      t_U32 u32ChapterCount;
      //! Track number
      t_U32 u32QueueIndex;
      //! Total Track number
      t_U32 u32QueueCount;
      //! MIME type of the image pointed by ImageUrl.
      std::string szImageMIMEType;
      //! Image size in bytes.
      t_U32 u32ImageSize;
      //! URL for image data
      std::string szImageUrl;
      //! Album Art
      trAppMediaAlbumArt rAppMediaAlbumArt;
      //! Current Playback state of the song
      tenMediaPlayBackState enMediaPlayBackState;
      //! Shuffle state
      tenMediaPlayBackShuffleState enMediaPlayBackShuffleState;
      //! Repeat state of the song
      tenMediaPlayBackRepeatState enMediaPlayBackRepeatState;
      //! Current playing media type.
      tenMediaPlayBackMediaType enMediaPlayBackMediaType;
      //! Now playing an iTunes ad.
      t_Bool bITunesRadioAd;
      //! Name of the now playing iTunes Radio station name.
      std::string szITunesRadioStationName;
      //! Indicates what is media rating. It is can 0...5
      t_U8 u8MediaRating;
      //! Initialise structure members to default values
      trAppMediaMetaData() :
                        bMediaMetadataValid(false),
                        u32TrackNumber(0xFFFFFFFF),
                        u32AlbumTrackCount(0xFFFFFFFF),
                        u32AlbumDiscNumber(0xFFFFFFFF),
                        u32AlbumDiscCount(0xFFFFFFFF),
                        u32ChapterCount(0xFFFFFFFF),
                        u32QueueIndex(0xFFFFFFFF),
                        u32QueueCount(0xFFFFFFFF),
                        u32ImageSize(0),
                        enMediaPlayBackState(e8PLAYBACK_NOT_KNOWN),
                        enMediaPlayBackShuffleState(e8SHUFFLE_NOT_KNOWN),
                        enMediaPlayBackRepeatState(e8REPEAT_NOT_KNOWN),
                        enMediaPlayBackMediaType(e8MEDIATYPE_NOT_KNOWN),
                        bITunesRadioAd(false),
                        u8MediaRating(0xFF)
      {

      }
};

//This structure contains time in hour, minute and seconds
struct trTime
{
      //! Hour component
      t_U8 u8Hour;
      //! Minute component
      t_U8 u8Minute;
      //! Second component
      t_U8 u8Second;
      trTime():
         u8Hour(0),
         u8Minute(0),
         u8Second(0)
      {

      }
};

//! This structure contains Phone call specific extended metadata information
struct trPhoneCallExtendedMetaData
{
      //! Phone number
      std::string szPhoneNumber;
      //! Name of the caller in the Phone contacts.
      std::string szDisplayName;
      //! Call UUID
      std::string szCallUUID;
      //! If caller label is in contacts, than mobile, work or home would be populated.
      std::string szCallerLabel;
      //! Call service of the current device
      tenCallStateUpdateService enCallStateUpdateService;
      //! Call disconnect reason of the current device
      tenCallStateUpdateDisconnectReason enCallStateUpdateDisconnectReason;
      //! Identifies the state of the call
      tenPhoneCallState enPhoneCallState;
      //! Incoming or outgoing call
      tenPhoneCallDirection enPhoneCallDirection;
      //! Conference group number
      t_U8 u8ConferenceGroup;
      //! Notification support info
      t_Bool bConferencedCall;
      //! Start Timestamp of call in seconds since reference date (Jan 1, 1970, GMT)
      t_U64 u64StartTimeStamp;
      //! Provides call progress information. Value gets incremented for every sec, until call is ended.
      trTime rCallProgress;
      //! CallProgress + some duration of call that was active before the device was connected to HU
      trTime rCallProgressRefFromDevice;
      //! Initialise structure members to default vaules
      trPhoneCallExtendedMetaData() :
                        enCallStateUpdateService(e8CALLSTATE_SERVICE_UNKNOWN),
                        enCallStateUpdateDisconnectReason(e8CALLSTATE_DISCONNECT_REASON_UNKNOWN),
                        enPhoneCallState(e8PHONE_CALL_STATE_NOT_KNOWN),
                        enPhoneCallDirection(e8PHONE_CALL_DIRECTION_NOT_KNOWN),
                        u8ConferenceGroup(0),
                        bConferencedCall(false),
                        u64StartTimeStamp(0)

      {
      }
};

//!Data to hold Phone application information like phone number, call progress in secs, call direction,etc
struct trAppPhoneCallData
{
      //! std::vector<>Phone call details Details
      std::vector<trPhoneCallExtendedMetaData> tvecPhoneCallExtendedMetaDataList;

      //! deafult constructor
      trAppPhoneCallData ()
      {

      }
};

//! This structure contains Phone call specific metadata information
struct trPhoneCallMetaData
{
      //! Phone number
      std::string szPhoneNumber;
      //! Name of the caller in the Phone contacts.
      std::string szDisplayName;
      //! Identifies the state of the call
      tenPhoneCallState enPhoneCallState;
      //! Incoming or outgoing call
      tenPhoneCallDirection enPhoneCallDirection;
      //! Call duration in secs
      t_U32 u32CallDuration;
      //! Call UUID
      std::string szCallUUID;
      //! If caller label is in contacts, than mobile, work or home would be populated.
      std::string szCallerLabel;
      //! Notification support info
      t_Bool bConferencedCall;
      //! Conference group number
      t_U8 u8ConferenceGroup;
      //! Call service of the current device
      tenCallStateUpdateService enCallStateUpdateService;
      //! Call disconnect reason of the current device
      tenCallStateUpdateDisconnectReason enCallStateUpdateDisconnectReason;
      //! Initialise structure members to default vaules
      trPhoneCallMetaData() :
                        enPhoneCallState(e8PHONE_CALL_STATE_NOT_KNOWN),
                        enPhoneCallDirection(e8PHONE_CALL_DIRECTION_NOT_KNOWN),
                        u32CallDuration(0xFFFFFFFF),
                        bConferencedCall(false),
                        u8ConferenceGroup(0),
                        enCallStateUpdateService(e8CALLSTATE_SERVICE_UNKNOWN),
                        enCallStateUpdateDisconnectReason(e8CALLSTATE_DISCONNECT_REASON_UNKNOWN)
      {
      }
};

enum tenBatteryChargingState
{
   e8_BATTERY_CHARGING_STATE_DISABLED = 0,
   e8_BATTERY_CHARGING_STATE_CHARGING = 1,
   e8_BATTERY_CHARGING_STATE_CHARGED = 2,
   e8_BATTERY_CHARGING_STATE_UNKNOWN = 3
};

//!Data to hold Phone application information like, call status, signal strength to the client.
struct trAppPhoneData
{
      //! Validity flag of Phone Metadata
      t_Bool bPhoneMetadataValid;
      //! std::vector<>Phone call details Details
      std::vector<trPhoneCallMetaData> tvecPhoneCallMetaDataList;
      //! Signal Strength on the Phone
      tenSignalStrength enSignalStrength;
      //! Registration status of the phone connection
      tenRegistrationStatus enRegistrationStatus;
      //! AirPlane Mode status on the phone
      tenAirPlaneModeStatus enAirPlaneModeStatus;
      //! Mute status of the call..
      tenPhoneMuteStatus enMute;
      //! Carrier Name on the phone.
      std::string szCarrierName;
      //! Current call count
      t_U8 u8CurrentCallCount;
      //! Current battery charge level
      t_U8 u8BatteryChargeLevel;
      //! Current battery charging state
      tenBatteryChargingState enBatteryChargingState;
      //! Initialise structure members to default vaules
      trAppPhoneData() :
                        bPhoneMetadataValid(false),
                        enSignalStrength(e8SIGNAL_STRENGTH_NOT_KNOWN),
                        enRegistrationStatus(e8PHONE_REGISTRATION_NOT_KNOWN),
                        enAirPlaneModeStatus(e8PHONE_AIRPLANE_MODE_NOT_KNOWN),
                        enMute(e8PHONE_MUTESTATE_NOT_KNOWN),
                        u8CurrentCallCount(0),
                        u8BatteryChargeLevel(0xFF),
                        enBatteryChargingState(e8_BATTERY_CHARGING_STATE_UNKNOWN)
      {

      }
};

//!Structure to hold playtime of the current playing media song.
struct trAppMediaPlaytime
{
      //! Total play time of the media object in seconds.
      t_U32 u32TotalPlayTime;
      //! Elapsed play time of the media object in seconds.
      t_U32 u32ElapsedPlayTime;
      //! Initialise structure members to default vaules
      trAppMediaPlaytime() :
               u32TotalPlayTime(0), u32ElapsedPlayTime(0)
      {

      }
};

//! This provides ScreenSize information
struct trScreenSize
{
      t_U32 u32Screen_Height;
      t_U32 u32Screen_Width;
//! Initialise structure members to 0
      trScreenSize() :
               u32Screen_Height(0), u32Screen_Width(0)
      {
      }
      trScreenSize& operator=(const trScreenSize& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            u32Screen_Height = corfrSrc.u32Screen_Height;
            u32Screen_Width = corfrSrc.u32Screen_Width;
         } //if( &corfrSrc != this)
         return *this;
      }
};

//! This provides ScreenOffset information
struct trScreenOffset
{
      t_U32 u32Screen_X_Offset;
      t_U32 u32Screen_Y_Offset;
//! Initialise structure members to 0
      trScreenOffset() :
               u32Screen_X_Offset(0), u32Screen_Y_Offset(0)
      {
      }
      trScreenOffset& operator=(const trScreenOffset& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            u32Screen_X_Offset = corfrSrc.u32Screen_X_Offset;
            u32Screen_Y_Offset = corfrSrc.u32Screen_Y_Offset;
         } //if( &corfrSrc != this)
         return *this;
      }
};

static const t_String cszClientProfileId = "0";
static const t_U32 cou32MLClientMajorversion = 1;
static const t_U32 cou32MLClientMinorVersion = 1;

//! PosixTime will be in seconds for NDS & TM, where as for Sensor_PV it will be in milliseconds.
//! Posix time (Time since the Epoch (00:00:00 UTC, January 1, 1970),
typedef t_U64 t_PosixTime;

//! Indicates the direction of the Heading caluculation
enum tenHeadingDir
{
   //! Clockwise direction (0 degrees is North, 90 degrees is East, etc.)
   e8CLOCKWISE_0_DEG_NORTH = 0,
   //!Anti-clockwise Direction (0 degrees is North, 90 degrees is West, etc.)
   e8ANTICLOCKWISE_0_DEG_NORTH = 1,
   //!Anti-clockwise Direction (0 degrees is East, 90 degrees is North, etc.)
   e8ANTICLOCKWISE_0_DEG_EAST = 2
};

//! Indicates Sensor type available in the vehicle
enum tenSensorType
{
   e8SENSOR_TYPE_UNKNOWN = 0, e8SENSOR_TYPE_COMBINED_LEFT_RIGHT_WHEEL = 1
};

//! Indicates Geo Coordinate System
enum tenGeoCoordinateSystemType
{
   e8GEO_COORDINATE_SYSTEM_UNKNOWN = 0,
   // Represents WGS-84 Coordinate system
   e8GEO_COORDINATE_SYSTEM_WGS = 1,
   // Represents GCJ-02 Coordinate system
   e8GEO_COORDINATE_SYSTEM_GCJ = 2

};

// Indicates source of deadreckoning information
enum tenDRSourceType
{
   // do not use the deadreckoning information since the source is invalid
   e8DR_SRC_INVALID = 0,
   // information is taken from a GNSS receiver
   e8DR_SRC_GNSS = 1,
   // information is calculated by deadreckoning component
   e8DR_SRC_DR = 2
};

/* Structure holding Location data */
struct trPositionData
{
      //! Time in Posix format
      t_PosixTime PosixTime;
      //! Precision of Posix time (in milliseconds) of received data
      t_U16 u16ExactTime;
      //! Latitude (in decimal) of vehicle position
      t_Double d64Latitude;
      //! Longitude (in decimal) of vehicle position
      t_Double d64Longitude;
      //! Resolution of Latitude/Longitude data
      //! @Note: Indicates scaling factor of Latitude/Longitude values.
      //! Actual Lat/Long value = <LatLong_in_decimal> / <Resolution>
      t_Double dLatLongResolution;
      //! Altitude (in meters)
      t_Float f32Altitude;
      //! Heading (in decimal) of vehicle
      t_Double dHeading;
      //! Resolution of Heading data
      //! @Note: Indicates scaling factor of Heading value.
      //! Actual Heading value = <Heading_in_decimal> / <Resolution>
      t_Double dHeadingResolution;
      //! Speed (in centimeter per second) of vehicle
      t_Float f32Speed;
      //! Value that indicates the Heading Direction-Clockwise or anti-clockwise
      tenHeadingDir enHeadingDir;
      //! Sensor type used
      tenSensorType enSensorType;
      //! Turn rate (in centidegree per second)
      t_Float f32TurnRate;
      //! Acceleration (cm/s)
      t_S16 s16Acceleration;

      //! Data availability flags
      t_Bool bLatitudeAvailable;
      t_Bool bLongitudeAvailable;
      t_Bool bAltitudeAvailable;
      t_Bool bHeadingAvailable;
      t_Bool bSpeedAvailable;

      //! Indicates Geo Coordinate System
      tenGeoCoordinateSystemType enGeoCoordinateSystemType;

      //! Indicates source of deadreckoning information
      tenDRSourceType enDRSourceType;

      //! Initialise structure members to default values
      trPositionData() :
                        PosixTime(0),
                        u16ExactTime(0),
                        d64Latitude(0),
                        d64Longitude(0),
                        dLatLongResolution(1),
                        f32Altitude(0),
                        dHeading(0),
                        dHeadingResolution(1),
                        f32Speed(0),
                        enHeadingDir(e8CLOCKWISE_0_DEG_NORTH),
                        enSensorType(e8SENSOR_TYPE_UNKNOWN),
                        f32TurnRate(0),
                        s16Acceleration(0),
                        bLatitudeAvailable(false),
                        bLongitudeAvailable(false),
                        bAltitudeAvailable(false),
                        bHeadingAvailable(false),
                        bSpeedAvailable(false),
                        enGeoCoordinateSystemType(e8GEO_COORDINATE_SYSTEM_UNKNOWN),
                        enDRSourceType(e8DR_SRC_INVALID)
      {
      }
};

//! Structure for holding GPS data
typedef trPositionData trGPSData;
//! Structure for holding DeadReckoning data
typedef trPositionData trDeadReckoningData;

/* Identifies the source of NMEA data */
enum tenNmeaDataSource
{
   //! GPS as source of NMEA data
   e8NMEA_SOURCE_GPS = 0,
   //! Sensor as source of NMEA data
   e8NMEA_SOURCE_SENSOR = 1,
   //! Unknown source of NMEA data
   e8NMEA_SOURCE_UNKNOWN
};

/* Identifies the type of Location data */
enum tenSensorDataType
{
   //! Indicates GPS data
   e8GPS_DATA = 0,
   //! Indicates GPS Dead Reckoning data
   e8DEADRECKONING_DATA = 1,
   //! Indicates GPS Dead Reckoning data
   e8GNSS_DATA = 2,
   //! Indicates Accelerometer data
   e8ACCELEROMETER_DATA = 3,
   //! Indicates Gyroscope data
   e8GYROSCOPE_DATA = 4,
   //! Indicates Environment data
   e8ENVIRONMENT_DATA = 5,
   //! Indicates Vehicle specific data
   e8VEHICLE_DATA = 6,
   //! Indicates Vehicle Park Brake data
   e8PARKBRAKE_DATA = 7,
   //! Indicates VD Clock data
   e8VDCLOCK_DATA = 8
};

/* Identifies NMEA sentence types */
enum tenNmeaSentenceType
{
   e8NMEA_GPGGA = 0,
   e8NMEA_GPRMC,
   e8NMEA_GPGSV,
   e8NMEA_GPHDT,
   e8NMEA_PASCD,
   e8NMEA_PAGCD,
   e8NMEA_PAACD,
   e8NMEA_UNKNOWN = 255
};

typedef enum
{
   e8GNSSQUALITY_NOFIX = 0x00,
   e8GNSSQUALITY_AUTONOMOUS = 0x01,
   e8GNSSQUALITY_DIFFERENTIAL = 0x02,
   e8GNSSQUALITY_UNKNOWN = 0x0ff
} tenGnssQuality;

typedef enum
{
   e8GNSSMODE_NOFIX = 0x01, e8GNSSMODE_2DFIX = 0x02, e8GNSSMODE_3DFIX = 0x03, e8GNSSMODE_UNKNOWN = 0x0255
} tenGnssMode;

//! Structure for holding sensor data
struct trSensorData
{
      //! Timestamp
      t_U32 u32TimeStamp;
      //! UTC time
      t_PosixTime PosixTime;
      //! Gnss Quality indicator
      tenGnssQuality enGnssQuality;
      //! Gnss Mode
      tenGnssMode enGnssMode;
      //! Number of Satellites used
      t_U16 u16SatellitesUsed;
      //! Horizontal dilution of precision
      t_Double dHDOP;
      //! Geoidal Separation (in meters)
      t_Double dGeoidalSeparation;
      //! Accuracy (in meters)
      t_Float fAccuracy;
      //! Speed (in meters per sec)
      t_Float fSpeed;
      //! Heading (or Bearing) (in radian)
      t_Float fHeading;
      //! Latitude (in radian)
      t_Double dLatitude;
      //! Longitude (in radian)
      t_Double dLongitude;
      //! Altitude (in meters)
      t_Float fAltitude;
      //! GyroZ (in deg/s)
      t_Float fGyroZ;
      //! GyroY (in deg/s)
      t_Float fGyroY;
      //! GyroX (in deg/s)
      t_Float fGyroX;
      //! Acceleration-Z (in m/s2)
      t_Float fAccZ;
      //! Acceleration-Y (in m/s2)
      t_Float fAccY;
      //! Acceleration-X (in m/s2)
      t_Float fAccX;

      //! Data availability flags
      t_Bool bNumSatUsedAvailable;
      t_Bool bHDOPAvailable;
      t_Bool bGeoidalSepAvailable;

      trSensorData() :
                        u32TimeStamp(0),
                        PosixTime(0),
                        enGnssQuality(e8GNSSQUALITY_UNKNOWN),
                        enGnssMode(e8GNSSMODE_UNKNOWN),
                        u16SatellitesUsed(0),
                        dHDOP(0.0),
                        dGeoidalSeparation(0.0),
                        fAccuracy(0),
                        fSpeed(0),
                        fHeading(0),
                        dLatitude(0),
                        dLongitude(0),
                        fAltitude(0),
                        fGyroZ(0),
                        fGyroY(0),
                        fGyroX(0),
                        fAccZ(0),
                        fAccY(0),
                        fAccX(0),
                        bNumSatUsedAvailable(false),
                        bHDOPAvailable(false),
                        bGeoidalSepAvailable(false)

      {

      }
};

//! Structure for holding Acceleration sensor data
struct trAccSensorData
{
      //! Acceleration-Z (in m/s2)
      t_Float fAccZ;
      //! Acceleration-Y (in m/s2)
      t_Float fAccY;
      //! Acceleration-X (in m/s2)
      t_Float fAccX;
      //Timestamp
      t_U64 u64AccTimeStamp;

      trAccSensorData() :
               fAccZ(0), fAccY(0), fAccX(0), u64AccTimeStamp(0)
      {
         //Nothing to do
      }
};

//! Structure for holding Gyro sensor data
struct trGyroSensorData
{
      //! GyroZ (in deg/s)
      t_Float fGyroZ;
      //! GyroY (in deg/s)
      t_Float fGyroY;
      //! GyroX (in deg/s)
      t_Float fGyroX;
      //Timestamp
      t_U64 u64GyroTimeStamp;
      trGyroSensorData() :
               fGyroZ(0), fGyroY(0), fGyroX(0), u64GyroTimeStamp(0)
      {
         //Nothing to do
      }
};

//Indicates the Gear Position
enum tenGearPosition
{
   e8GEAR_POS_NEUTRAL_GEAR = 0,
   e8GEAR_POS_FIRST_GEAR = 1,
   e8GEAR_POS_SECOND_GEAR = 2,
   e8GEAR_POS_THIRD_GEAR = 3,
   e8GEAR_POS_FOURTH_GEAR = 4,
   e8GEAR_POS_FIFTH_GEAR = 5,
   e8GEAR_POS_SIXTH_GEAR = 6,
   e8GEAR_POS_FORWARD_GEAR = 100,
   e8GEAR_POS_PARK_GEAR = 101,
   e8GEAR_POS_REVERSE_GEAR = 102,
   e8GEAR_POS_UNKNOWN = 255
//@Note: Please do not change above values!
//Values are assigned as per SPI FI.
};

enum tenVehicleMovementState
{
   e8VEHICLE_MOVEMENT_STATE_PARKED = 0,
   e8VEHICLE_MOVEMENT_STATE_NEUTRAL = 1,
   e8VEHICLE_MOVEMENT_STATE_FORWARD = 2,
   e8VEHICLE_MOVEMENT_STATE_REVERSE = 3,
   e8VEHICLE_MOVEMENT_STATE_INVALID = 4
};

//Indicates the Vehicle Data for PASCD sentence
struct trVehicleData
{
      //! Posix time
      t_PosixTime PosixTime;

      //! Speed (in centimeter per second) of vehicle
      t_S16 s16Speed;

      //! Transmitted Gear Position
      tenGearPosition enGearPosition;

      //! Parking Brake
      t_Bool bParkBrakeActive;

      //! Data availability flags
      t_Bool bSpeedAvailable;

      tenVehicleMovementState enVehMovState;

      //! Initialise all members to defaults
      trVehicleData() :
                        PosixTime(0),
                        s16Speed(0),
                        enGearPosition(e8GEAR_POS_UNKNOWN),
                        bParkBrakeActive(false),
                        bSpeedAvailable(false),
                        enVehMovState(e8VEHICLE_MOVEMENT_STATE_INVALID)
      {
      }
};

enum tenVDClockTimeFormat
{
   e8VD_CLOCK_TIMEFORMAT_12HOUR = 1, e8VD_CLOCK_TIMEFORMAT_24HOUR = 2, e8VD_CLOCK_TIMEFORMAT_UNKNOWN = 3
};

enum tenATBoxReversegearEngagementState
{
   e8ATBOX_REARGEAR_INVALID = -1, e8ATBOX_REARGEAR_DISENGAGED = 0x00, e8ATBOX_REARGEAR_ENGAGED = 0x01
};

enum tenPrivateCDBObjectType
{
   e8PRIVATECDB_OBJECT_INVALID = 0x00,
   e8PRIVATECDB_OBJECT_DEVICETIMEFORMAT = 0x01,
   e8PRIVATECDB_OBJECT_LANGUAGESETTING = 0x02,
   e8PRIVATECDB_OBJECT_VEHICLESPEED = 0x03,
   e8PRIVATECDB_OBJECT_REVERSEGEARENGAGEMENT = 0x04,
   e8PRIVATECDB_OBJECT_VEHICLEPOWERMODE = 0x05,
   e8PRIVATECDB_OBJECT_DISTANCETOEMPTYTANK = 0x06,
   e8PRIVATECDB_OBJECT_AMBIENTCOLOUR = 0x07
};

//! Callback signatures definition for GPS Data
typedef std::function<t_Void(trGPSData)> vOnGpsData;
typedef std::function<t_Void(trDeadReckoningData)> vOnDeadReckoningData;

//! \brief   Structure holding the callbacks to subscriber.
//!          Used by Client-handler to notify subscriber on interested events.
struct trLocationDataCallbacks
{
      //! Called when GPS data is received by client handler.
      vOnGpsData fvOnGpsData;

      //! Called when DeadReckoning data is received by client handler.
      vOnDeadReckoningData fvOnDeadReckoningData;

      trLocationDataCallbacks() :
               fvOnGpsData(NULL), fvOnDeadReckoningData(NULL)
      {
      }
};

//! Callback signatures definition for Sensor Data
typedef std::function<t_Void(trSensorData)> vOnSensorData;

//! Callback signatures definition for Sensor Data
typedef std::function<t_Void(const std::vector<trGyroSensorData>&)> vOnGyroSensorData;

//! Callback signatures definition for Sensor Data
typedef std::function<t_Void(const std::vector<trAccSensorData>&)> vOnAccSensorData;

//! \brief   Structure holding the callbacks to subscriber.
//!          Used by Client-handler to notify subscriber on interested events.
struct trSensorDataCallbacks
{
      //! Called when Sensor data is received by client handler.
      vOnSensorData fvOnSensorData;

      //! Called when Gyro Sensor data is received by client handler.
      vOnGyroSensorData fvOnGyroSensorData;

      //! Called when Acceleration Sensor data is received by client handler.
      vOnAccSensorData fvOnAccSensorData;

      trSensorDataCallbacks() :
               fvOnSensorData(NULL), fvOnGyroSensorData(NULL), fvOnAccSensorData(NULL)
      {
      }
};

//! Callback signatures definition for Vehicle Data
typedef std::function<t_Void(trVehicleData, t_Bool)> vOnVehicleData;

//! Callback signatures definition for DIPO Location DATA
typedef std::function<t_Void(t_Bool, const std::vector<tenNmeaSentenceType>&, t_U8)> vOnDipoLocationInfo;

//! Callback signatures definition for DIPO GPRMC Data Status Values DATA
typedef std::function<t_Void(t_U32, t_U8)> vOnDipoGPRMCDataStatusValues;

//! \brief   Structure holding the callbacks to subscriber.
//!          Used by Client-handler to notify subscriber on interested events.
struct trDipoLocationInfoCallbacks
{
      //! Called when dipo location info is received by client handler
      vOnDipoLocationInfo fvOnDipoLocationInfo;
      vOnDipoGPRMCDataStatusValues fvOnDipoGPRMCDataStatusValues;

      trDipoLocationInfoCallbacks() :
               fvOnDipoLocationInfo(NULL), fvOnDipoGPRMCDataStatusValues(NULL)
      {
      }
};

typedef std::function<t_Void(t_U8)> vOnClockData;

struct trClockCbs
{
      //! Called when Clock data is received by client handler.
      vOnClockData fvOnClockData;

      trClockCbs() :
               fvOnClockData(NULL)
      {

      }
};
//Callback for OutsideTemperature update
typedef std::function<t_Void(t_Bool, t_Double)> tvOutsideTempUpdate;

/*****************************************************
 **************** COMMON ******************************
 *****************************************************/
#define MAX_KEYSIZE 256
/**
 * typedef enum tenServiceId
 * \Enumeration to identify different services or functionalities in the SPI core lib
 *  Use it along with mediator.
 *  TODO : Append new service Ids as various services are identified for SPI core
 */
/*
 typedef enum
 {
 e8SPI_SERVICE_VIDEO,
 e8SPI_SERVICE_AUDIO,
 e8SPI_SERVICE_SPIMANAGER,
 e8SPI_SERVICE_INVALID
 }tenServiceId;*/

/**
 * struct rUserContext
 * \brief  Structure to store the User Context info
 */
struct trUserContext
{
      /*
       * Source Application Id
       */
      t_U32 u32SrcAppID;
      /*
       * Registration Id
       */
      t_U32 u32RegID;
      /*
       * Command Counter
       */
      t_U32 u32CmdCtr;
      /*
       * Function ID
       */
      t_U32 u32FuncID;

      /*
       * Destination Application Id
       */
      t_U32 u32DestAppID;

      /*
       * Service Id
       */
      t_U32 u32ServcID;

      /*************************************************************************
       ** FUNCTION:  rUserContext::rUserContext()
       *************************************************************************/
      /*!
       * \fn    rUserContext()
       * \brief Constructor
       *************************************************************************/
      trUserContext() :
               u32SrcAppID(0), u32RegID(0), u32CmdCtr(0), u32FuncID(0), u32DestAppID(0), u32ServcID(0)
      {

      }

      t_Bool operator==(const trUserContext& rfcorUserContext) const
      {
         //compare all members
         return ((rfcorUserContext.u32SrcAppID == u32SrcAppID) && (rfcorUserContext.u32RegID == u32RegID)
                  && (rfcorUserContext.u32CmdCtr == u32CmdCtr) && (rfcorUserContext.u32FuncID == u32FuncID)
                  && (rfcorUserContext.u32DestAppID == u32DestAppID) && (rfcorUserContext.u32ServcID == u32ServcID));
      }

      t_Bool operator!=(const trUserContext& rfcorUserContext) const
      {
         //use equality operator
         return !((*this).operator==(rfcorUserContext));
      }

      trUserContext& operator=(const trUserContext& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            u32SrcAppID = corfrSrc.u32SrcAppID;
            u32RegID = corfrSrc.u32RegID;
            u32CmdCtr = corfrSrc.u32CmdCtr;
            u32FuncID = corfrSrc.u32FuncID;
            u32DestAppID = corfrSrc.u32DestAppID;
            u32ServcID = corfrSrc.u32ServcID;
         }   //if( &corfrSrc != this)
         return *this;
      }

};

//! Empty context
static const trUserContext corEmptyUsrContext;

/**
 * struct rAppContext
 * \brief  Structure to store the application Context info
 */

struct trAppContext
{
      //! Response ID
      t_U32 u32ResID;

      //! Service ID
      t_U32 u32ServiceID;

      //! Initialise structure members to 0
      trAppContext() :
               u32ResID(0), u32ServiceID(0)
      {
      }

      //Assignment operator
      trAppContext& operator=(const trAppContext& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            u32ResID = corfrSrc.u32ResID;
            u32ServiceID = corfrSrc.u32ServiceID;
         }   //if( &corfrSrc != this)
         return *this;
      }
};

/*
 struct trAppContext
 {
 //! Response ID
 t_U32 u32ResID;

 //! Service ID list
 std::list<t_U32> lstReceiverServiceId;

 rAppContext() : u32ResID(0)
 {
 }
 } ;
 */
/**
 * struct rMsgContext
 * \brief  Structure to store the complete message Context info
 */
struct trMsgContext
{
      trUserContext rUserContext;
      trAppContext rAppContext;
//! Default Constructor
      trMsgContext()
      {
      }

      //Assignment operator
      trMsgContext& operator=(const trMsgContext& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            rUserContext = corfrSrc.rUserContext;
            rAppContext = corfrSrc.rAppContext;
         } //if( &corfrSrc != this)
         return *this;
      }

};

//! Identifies the Audio Streaming Link 
typedef enum
{
   e8LINK_RTPOUT = 0, e8LINK_RTPIN = 1, e8LINK_BTA2DP = 2, e8LINK_BTHFP = 3
} tenAudioLink;

/***************************************************************************************************************************
 AudioDir                   Purpose                                   GM-AudioSource        G3G-AudioSource
 ----------------------------------------------------------------------------------------------------------------------------
 e8AUD_MAIN_OUT          To enable MAIN Audio channel                 LC_MAIN              ARL_SRC_SPI_MAIN
 e8AUD_MIX_OUT           To enable Mono MIX channel with out Duck     LC_ALERTTONE         ARL_SRC_SPI_INFO
 e8AUD_ALERT_OUT         To play Ringtones                            LC_INCOMING_RINGTONE ARL_SRC_SPI_ALERT
 e8AUD_PHONE_IN          To enable Telephony                          LC_PHONE             ARL_SRC_SPI_PHONE
 e8AUD_VR_IN             To enable Speech Rec                         LC_SPEECH            ARL_SRC_SPI_VR
 e8AUD_DUCK              To enable Mono SPI_MIX with Duck             LC_MIXALERT          ARL_SRC_SPI_MIX & bSetAudioDucking
 e8AUD_TRANSIENT         To enable SPI_TRANSIENT                        -                  ARL_SRC_SPI_TRANSIENT
 e8AUD_STEREO_MIX_OUT    To enable Stereo SPI_MIX                       -                  ARL_SRC_SPI_ML_INFO
 e8AUD_VR_STEREO         To enable Voice Recognition for sterep source  -                  ARL_SRC_SPI_VR_STEREO
 e8AUD_INVALID           Native Audio Source is Active
 ******************************************************************************************************************************/

/****************************************************************************************************
 e8AUD_MIX_OUT and e8AUD_DUCK are used to differentiate Audio channels which needs to mix SPI audio
 with Duck and with out Duck.
 ------------------------------------------------------------------------------------------------------
 In case of G3G:
 To Mix SPI audio with MAIN audio with Ducking => SPI has to use SPI_MIX audio channel and use bSetAudioDucking API
 To Mix SPI audio with MAIN audio with out Ducking => SPI just has to use SPI_MIX audio channel

 In case of GM:
 To Mix SPI audio with MAIN audio with Ducking => SPI has to use LC_MIXALERT audio channel
 To Mix SPI audio with MAIN audio with out Ducking => SPI has to use LC_ALERTTONE audio channel
 ****************************************************************************************************/

//! Identifies the Audio Stream Direction 
typedef enum
{
   //! Audio Output: CE Device -> Head Unit
   e8AUD_MAIN_OUT = 0x00,
   //! Audio Output: CE Device -> Head Unit
   e8AUD_MIX_OUT = 0x01,
   //! Audio Output: CE Device -> Head Unit
   e8AUD_ALERT_OUT = 0x02,
   //! Audio Input: CE Device <-> Head Unit
   e8AUD_PHONE_IN = 0x03,
   //! Audio Input: CE Device <-> Head Unit
   e8AUD_VR_IN = 0x04,
   //! Audio Input: CE Device -> Head Unit
   e8AUD_DUCK = 0x05,
   //! Audio Output: CE->HU; For general info updates.
   e8AUD_TRANSIENT = 0x06,
   //! Audio Stereo Mix channel without Duck CE -> HU
   e8AUD_STEREO_MIX_OUT = 0x07,
   //! Audio Input: CE Device <-> Head Unit
   e8AUD_DEFAULT = 0x08,
   //! Audio Input:
   e8AUD_VR_STEREO = 0x09,
   //! Audio Input: Invalid for Default Value
   e8AUD_INVALID = 0x0A
} tenAudioDir;

//! Identifies the Audio  Sampling Rate
typedef enum
{
   //! Default Sample Rate
   e8AUD_SAMPLERATE_DEFAULT = 0x00,
   //! 8KHz Sample Rate
   e8AUD_SAMPLERATE_8KHZ = 0x01,
   //! 16KHz Sample Rate
   e8AUD_SAMPLERATE_16KHZ = 0x02,
   //! 24KHz Sample Rate
   e8AUD_SAMPLERATE_24KHZ = 0x03,
   //!Add entries above this for sampling rates
   e8AUD_SAMPLERATE_MAX = 0x04
} tenAudioSamplingRate;

//! \brief:  DataSet values for SPI
enum tenAudioInDataSet
{
   //! Enum values should match with the config IDS in ECNR configuration
   e32_AUDIOIN_DATASET_UNKNOWN = 0,
   e32_AUDIOIN_DATASET_PHONE_NB = 1,
   e32_AUDIOIN_DATASET_PHONE_WB = 2,
   e32_AUDIOIN_DATASET_VR_NB = 3,
   e32_AUDIOIN_DATASET_VR_WB = 4,
   e32_AUDIOIN_DATASET_VR_MIC_ONLY = 5
};

//! \brief:  DTC_STATUS values for SPI
typedef enum
{
   //Device connection status is in progress/not sure
   e8DTC_NOT_SURE = 0,
   //Device is not connected
   e8DTC_NOT_CONNECTED = 1,
   //Device is connected
   e8DTC_CONNECTED = 2,

} tenDTCStatus;

/**
 * class trMsgBase
 * \brief  Base class for all messages used for communication and data exchange between
 *  classes in Smart Phone integration
 */

class trMsgBase
{
   public:

      trMsgBase()
      {
      }

      virtual ~trMsgBase()
      {

      }

      t_U32 u32GetServiceID()
      {
         return rMsgCtxt.rAppContext.u32ServiceID;
      }

      t_Void vSetServiceID(t_U32 u32ServiceID)
      {
         rMsgCtxt.rAppContext.u32ServiceID = u32ServiceID;
      }

      t_Void vGetUserContext(trUserContext &rfrUserContext)
      {
         rfrUserContext = rMsgCtxt.rUserContext;
      }

      t_Void vSetUserContext(const trUserContext &rfrUserContext)
      {
         rMsgCtxt.rUserContext = rfrUserContext;
      }

      //virtual t_Void vAllocateMsg() = 0;

      //virtual t_Void vDeAllocateMsg() = 0;
   private:
      trMsgContext rMsgCtxt;

};

//! Defines Command IDs for HMI interface
enum tenCommandID
{
   e32COMMANDID_INVALID = 0,
   e32COMMANDID_GETDEVICEINFOLIST = 1,
   e32COMMANDID_SELECTDEVICE = 2,
   e32COMMANDID_GETAPPLIST = 3,
   e32COMMANDID_LAUNCHAPP = 4,
   e32COMMANDID_TERMINATEAPP = 5,
   e32COMMANDID_GETAPPICONDATA = 6,
   e32COMMANDID_SETAPPICONATTR = 7,
   e32COMMANDID_SETDEVICEUSAGEPREF = 8,
   e32COMMANDID_SETORIENTATIONMODE = 9,
   e32COMMANDID_SETSCREENSIZE = 10,
   e32COMMANDID_SETVIDEOBLOCKINGMODE = 11,
   e32COMMANDID_SETAUDIOBLOCKINGMODE = 12,
   e32COMMANDID_SETVEHICLECONFIG = 13,
   e32COMMANDID_SENDTOUCHEVENT = 14,
   e32COMMANDID_SENDKEYEVENT = 15,
   e32COMMANDID_ALLOCATEAUDIOROUTE = 16,
   e32COMMANDID_DEALLOCATEAUDIOROUTE = 17,
   e32COMMANDID_AUDIOSRCACTIVITY = 18,
   e32COMMANDID_SETSCREENVARIANT = 19,
   e32COMMANDID_BTADDRESS = 21,
   e32COMMANDID_MLNOTIONOFF = 22,
   e32COMMANDID_SETACCESSORY_DISPLAYCONTEXT = 23,
   e32COMMANDID_SETACCESSORY_AUDIOCONTEXT = 24,
   e32COMMANDID_SETACCESSORY_APPSTATE = 25,
   e32COMMANDID_SETREGION = 26,
   e32COMMANDID_AUDIO_ROUTEALLOCATE_RESULT = 27,
   e32COMMANDID_AUDIO_ROUTEDEALLOCATE_RESULT = 28,
   e32COMMANDID_AUDIO_STARTSOURCEACTIVITY = 29,
   e32COMMANDID_AUDIO_STOPSOURCEACTIVITY = 30,
   e32COMMANDID_AUDIO_REQAV_DEACTIVATION_RESULT = 31,
   e32COMMANDID_AUDIO_ERROR = 32,
   e32COMMANDID_MLNOTIFICATION_ENABLED_INFO = 33,
   e32COMMANDID_INVOKE_NOTIACTION = 34,
   e32COMMANDID_SETCLIENTCAPABILITIES = 35,
   e32COMMANDID_SENSORDATA = 37,
   e32COMMANDID_GPSDATA = 38,
   e32COMMANDID_VEHICLEDATA = 39,
   e32COMMANDID_GETKEYICONDATA = 40,
   e32COMMANDID_SENDKNOBKEYEVENT = 41,
   e32COMMANDID_ACCSENSORDATA = 42,
   e32COMMANDID_GYROSENSORDATA = 43,
   e32COMMANDID_SETFEATURERESTRICTIONDATA = 44,
   e32COMMANDID_ACKNOTIFICATIONEVENT = 45,
   e32COMMANDID_AUDIO_PAUSESOURCEACTIVITY = 46,
   e32COMMANDID_BTVRSTATUS = 47,
   e32COMMANDID_SETDEVICEAUTHRIZATION = 48,
   e32COMMANDID_SETTECHNOLOGYSELECTION = 49,
   e32COMMANDID_SETNATIVEVIDEORENDERINGSTATUS = 50,
   e32COMMANDID_DELETEDEVICE = 51,
   e32COMMANDID_INITIATEWIRELESSDISCOVERY = 52,
   e32COMMANDID_SETGENERALRESTRICTIONDATA = 53,
   e32COMMANDID_SENDVENDOREXTENSION = 54,
   e32COMMANDID_ETCSTATUS = 55,
   e32COMMANDID_LANGUAGESETTING = 56,
   e32COMMANDID_VEHICLEPOWERMODESTATE = 57,
   e32COMMANDID_AMBIENTCOLORDATA = 58,
   e32COMMANDID_DISTTOEMPTYTANK = 59,
   e32COMMANDID_LOADSETTINGSCOMPLETED = 60,
   e32COMMANDID_GETDEVICELIST = 61,
   e32COMMANDID_REGISTERKEYCODES = 62
};

//! IDS for SPI components
typedef enum
{
   e32COMPID_UNKNOWN = 0,
   e32COMPID_CONNECTIONMANAGER = 1,
   e32COMPID_AUDIO = 2,
   e32COMPID_VIDEO = 3,
   e32COMPID_APPMANAGER = 4,
   e32COMPID_BLUETOOTH = 5,
   e32COMPID_DATASERVICE = 6,
   e32COMPID_INPUTHANDLER = 7,
   e32COMPID_RESOURCEMNGR = 8,
   e32COMPID_LEGALDISCLAIMER = 9,
   e32COMPID_TECHNOLOGYSELECTOR = 10,
   e32COMPID_DISCOVERER = 11,
   e32COMPID_WIFI = 12,
   //! Add Components above this. Please follow sequential numbering
   //! so that enum size is maintained
   e32COMPID_SIZE
} tenCompID;

typedef enum
{
   e8SPI_STATE_OFF = 0, e8SPI_STATE_ON = 1, e8SPI_STATE_UNKNOWN = 2
} tenSPIState;

struct trAudDevices
{
      t_String szOutputDev;
      t_String szInputDev;
};

struct trAudSrcInfo
{
      trAudDevices rMainAudDevNames;
      trAudDevices rEcnrAudDevNames;
};

typedef enum
{
   e8BASESYSTEM = 0,           //0 - Base System
   e8CONNSYSTEM = 1,           //1 - Connectivity System -> 4.2" system
   e8COLORSYSTEM8INCH = 2,      //2 - Color HMI System    -> 8" system
   e8NAVISYSTEM8INCH = 3,       //3 - Premium Navi System -> 8" system
   e8COLORSYSTEM10DOT2INCH = 4, //4 - Color HMI System    -> 10.2" system
   e8NAVISYSTEM10DOT2INCH = 5   //5 - Premium Navi System -> 10.2" system
} tenGMSystemVariant;

//! Context info of an application
struct trVehicleBrandInfo
{
      t_U8 u8GMVehicleBrandStartRange;
      t_U8 u8GMVehicleBrandEndRange;
      t_Char szOemName[STR_LENGTH];
      t_Char szOemIconPath[STR_LENGTH];
      t_Char szOemIconsPath[STR_LENGTH];
      t_Char szManufacturer[STR_LENGTH];
      t_Char szModel[STR_LENGTH];
      t_Bool bIsRotarySupported;
      t_Bool bIsOEMIconEnable;

};

struct trVehicleInfo
{
      t_U8 u8GMVehicleBrandStartRange;
      t_U8 u8GMVehicleBrandEndRange;
      t_String szManufacturer;
      t_String szModel;
      t_String szVehicleVariant;

      trVehicleInfo() :
               u8GMVehicleBrandStartRange(0), u8GMVehicleBrandEndRange(0)
      {

      }

};

struct trDiagEOLEntry
{
      t_U8 u8Table;
      t_U16 u16Offset;
      t_U16 u16EntryLength;
      t_U8* pu8EntryData;
};

//! Identifies the blocking type
enum tenBlockingType
{
   e8BLOCKINGTYPE_UNKNOWN = 0, e8BLOCKINGTYPE_AUDIO = 1, e8BLOCKINGTYPE_VIDEO = 2
};

//! MirrorLink Client Profile
struct trClientProfile
{
      //! Profile Id of the Client Profile
      t_String szClientProfileId;
      //! Client Manufacturer or OEM - This is required for fetching member certified applications
      t_String szProfileManufacturer;

      //! Icon Attributes which needs to be set
      trIconAttributes rIconPreferences;

      //! MirrorLink Client BT Address
      t_String szProfileBdAddr;

      //! various RTP payload types supported by the MLC
      t_String szRtpPayLoadTypes;

      //! ML Major version supported by the ML Client
      t_U32 u32ProfileMajorVersion;
      //! ML Minor version supported by the ML Client
      t_U32 u32ProfileMinorVersion;

      //! Support for native notification UI
      t_Bool bNotiUISupport;
      //! Maximum number of actions in notification
      t_U8 u8NotiMaxActions;

      //! Friendly name of the MLC
      t_String szFriendlyName;

      //! Model Number of the MLC
      t_String szModelNumber;
      trClientProfile() :
               u32ProfileMajorVersion(0), u32ProfileMinorVersion(0), bNotiUISupport(false), u8NotiMaxActions(0)
      {
         szClientProfileId = "0";
         szRtpPayLoadTypes = "99";
      }
};

/*! 
 * \enum tenAudioContext
 * \brief SPI internal audio mapping
 */
typedef enum
{
   //! Idenify the main audio
   e8SPI_AUDIO_MAIN = 0,

   //! Idenify audio from iApps
   e8SPI_AUDIO_INTERNET_APP = 1,

   //! idenify the speech_rec audio
   e8SPI_AUDIO_SPEECH_REC = 2,

   //! Idenify advisor phone audio
   e8SPI_AUDIO_ADVISOR_PHONE = 3,

   //! Identify emergency phone audio
   e8SPI_AUDIO_EMER_PHONE = 4,

   //! Idenify phone audio
   e8SPI_AUDIO_PHONE = 5,

   //! Idenitify phone incoming ringtone audio
   e8SPI_AUDIO_INCOM_TONE = 6,

   //! Identify traffic alert audio
   s8SPI_AUDIO_TRAFFIC = 7,

   //! Idenify emergency message audio
   e8SPI_AUDIO_EMER_MSG = 8,

   //! Identify synchronous message audio
   e8SPI_AUDIO_SYNC_MSG = 9,

   //! Identify mix alert message audio
   e8SPI_AUDIO_MIX_ALERT_MSG = 10,

   //! Identify short mix alert message
   e8SPI_AUDIO_SHORT_MIX_ALERT_MSG = 11,

   //! Identify audio sleep mode, and it ready to wake up on incoming call.
   e8SPI_AUDIO_SLEEP_MODE = 12,

   //! Identify audio sleep mode, and it will not wake up on any event on the projection device.
   e8SPI_AUDIO_STANDBY_MODE = 13,

   //! Identify audio mute
   e8SPI_AUDIO_MUTE = 14,

   //! Identifies Main before session start
   e8SPI_AUDIO_MAIN_DEFAULT = 15,

   //! Identifies projection audio
   e8SPI_AUDIO_PROJECTION = 16,

   //! Identify alert tone audio
   e8SPI_AUDIO_ALERT_TONE = 250,

   //! Identify asynchronous message audio
   e8SPI_AUDIO_ASYNC_MSG = 251,

   //! Identify lvm message audio
   e8SPI_AUDIO_LVM = 252,

   //! Identify audio cues message audio
   e8SPI_AUDIO_CUE = 253,

   //! Identify no audio
   e8SPI_AUDIO_NONE = 254,

   //! Identify unknown audio
   e8SPI_AUDIO_UNKNOWN = 255
} tenAudioContext;

//! Audio ducking type
enum tenDuckingType
{
   e8_DUCKINGTYPE_UNDUCK = 0,
   e8_DUCKINGTYPE_DUCK = 1,
   e8_DUCKINGTYPE_DEVICEDUCK = 2,
   e8_DUCKINGTYPE_NATIVEDUCK = 3,
   e8_DUCKINGTYPE_UNKNOWN = 4
};

//Structure for audio context info//
struct trAudioContext
{
      t_Bool AudioFlag;
      tenAudioContext AudioContext;
      trAudioContext() :
               AudioFlag(false), AudioContext(e8SPI_AUDIO_UNKNOWN)
      {
      }
};

/*!
 * \typedef tenRegion
 */
enum tenRegion
{
   //! Indicates All countries
   e8_WORLD = 0,
   //! Indicates European Union member states
   e8_EU = 1,
   //! Indicates Europe without countries listed separately or EU member states
   e8_EPE = 2,
   //! Indicates Canada
   e8_CAN = 3,
   //! Indicates USA
   e8_USA = 4,
   //! Indicates Americas without countries listed separately
   e8_AMERICA = 5,
   //! Indicates Australia
   e8_AUS = 6,
   //! Indicates Korea
   e8_KOR = 7,
   //! Indicates Japan
   e8_JPN = 8,
   //! Indicates China
   e8_CHN = 9,
   //! Indicates Hongkong
   e8_HKG = 10,
   //! Indicates Taiwan
   e8_TPE = 11,
   //! Indicates India
   e8_IND = 12,
   //! Indicates APAC states without countries listed separately
   e8_APAC = 13,
   //! Indicates African countries without countries listed separately
   e8_AFRICA = 14,
   //! Invalid index
   e8_INVALID = 15
};

/*! 
 * \enum tenSpeechAppState
 * \brief SPI speech states
 */
typedef enum
{
   //! Unknown speech state/ not applicable
   e8SPI_SPEECH_UNKNOWN = 0,

   //! Indentify the end of speaking/recognizing state
   e8SPI_SPEECH_END = 1,

   //! Identify the recognizing state
   e8SPI_SPEECH_RECOGNIZING = 2,

   //! Identify the speaking state
   e8SPI_SPEECH_SPEAKING = 3
} tenSpeechAppState;

/*! 
 * \enum tenDisplayType
 * \brief SPI display screen type
 * */
typedef enum
{
   e8RESISTIVE_SCREEN = 0, e8CAPACITIVE_SCREEN = 1
} tenDisplayType;

/*!
 * \enum tenPhoneAppState
 * \brief SPI phone states
 */
typedef enum
{
   //! Phone state is unknown/ not applicable
   e8SPI_PHONE_UNKNOWN = 0,

   //! Phone call is active
   e8SPI_PHONE_ACTIVE = 1,

   //! Phone call is not active.
   e8SPI_PHONE_NOT_ACTIVE = 2
} tenPhoneAppState;

/*! 
 * \enum tenPhoneAppState
 * \brief SPI Nav states
 */
typedef enum
{
   //! Nav state is unknown
   e8SPI_NAV_UNKNOWN = 0,

   //! Nav state is active
   e8SPI_NAV_ACTIVE = 1,

   //! Nav state is not active
   e8SPI_NAV_NOT_ACTIVE = 2
} tenNavAppState;

typedef enum
{
   e8SINGLE_TOUCH = 0, e8MULTI_TOUCH = 1
} tenTouchEventType;

typedef enum
{
   e8Unknown = 0, e8AAP_SINGLE_TOUCH = 3, e8AAP_MULTI_TOUCH = 4
} tenAAPTouchEventType;

//!Vehicle Display Data
struct trVideoConfigData
{
      tenGMSystemVariant enSysVariant;
      t_U32 u32Screen_Width;
      t_U32 u32Screen_Height;
      t_U32 u32LayerId;
      t_U32 u32SurfaceId;
      t_U32 u32TouchLayerId;
      t_U32 u32TouchSurfaceId;
      t_U32 u32Screen_Width_Millimeter;
      t_U32 u32Screen_Height_Millimeter;
      t_U32 u32ProjScreen_Width;
      t_U32 u32ProjScreen_Height;
      t_U32 u32ProjScreen_Width_Mm;
      t_U32 u32ProjScreen_Height_Mm;
      t_U32 u32ResolutionSupported;
      t_Char szPixelAspectRatio[MAXKEYSIZE];
      tenTouchEventType enTouchEventType;
      tenDisplayType enDisplayType;
      t_U16 u16dpi;
      t_U16 u16PriorityIndex;
      t_Char szDisplayDev[PATH_LENGTH];

      //Please make sure that config reader files for all projects are updated,
      //when this structure is extended/updated.
};

//Vector to store the video configurations for AAP
typedef std::vector<trVideoConfigData> tvecVideoConfigList;

//Map to store the video configuration for AAP on the baisis of primary index
typedef std::map<t_U16, trVideoConfigData> tmapAAPVideoConfigList;
//!Android Auto Certificates Details
struct trAAPCertConfigData
{
      t_String szKeyId;
      t_String szCertificateFolderPath;
};

//! Pixel formats for video rendering
enum tenPixelFormat
{
   e8_PIXELFORMAT_UNKNOWN = 0,
   e8_PIXELFORMAT_ARGB888 = 1,
   e8_PIXELFORMAT_RGB565 = 2,
   e8_PIXELFORMAT_RGBA8888 = 3,
   e8_PIXELFORMAT_RGB888 = 4
};

//! SR Button event result
enum tenSpeechRecBtnEventResult
{
   e8NEW_SESSION_NOT_STARTED = 0, e8NEW_SESSION_STARTED = 1
};

//! Enum for Navigation Guidance
enum tenGuidanceStatus
{
   e8GUIDANCE_INACTIVE = 0x00, e8GUIDANCE_ACTIVE = 0x01, e8GUIDANCE_SUSPENDED = 0x02
//@Note: Please do not change above values!
//Values are assigned as per Navigation FI values.
};

//! defines audio errors
enum tenAudioError
{
   e8_AUDIOERROR_NONE = 0,
   e8_AUDIOERROR_UNKNOWN = 1,
   e8_AUDIOERROR_AVACTIVATION = 2,
   e8_AUDIOERROR_AVDEACTIVATION = 3,
   e8_AUDIOERROR_ALLOCATE = 4,
   e8_AUDIOERROR_DEALLOCATE = 5,
   e8_AUDIOERROR_STARTSOURCEACT = 6,
   e8_AUDIOERROR_STOPSOURCEACT = 7,
   e8_AUDIOERROR_GENERIC = 8
};
//! Pixel resolutions
enum tenPixelResolution
{
   e8_PIXEL_RES_800x480 = 0, e8_PIXEL_RES_640x320 = 1, e8_PIXEL_RES_320x240 = 2, e8_PIXEL_RES_1200x720 = 3
};

//! Indicates Drive side of vehicle
enum tenDriveSideInfo
{
   e8UNKNOWN_DRIVE_SIDE = 0, e8LEFT_HAND_DRIVE = 1, e8RIGHT_HAND_DRIVE = 2
};

enum tenAudioStatus
{
   e8AUDIO_STATUS_UNKNOWN = 0,
   e8AUDIO_STATUS_MEDIA_SETUP = 1,
   e8AUDIO_STATUS_MEDIA_TEARDOWN = 2,
   e8AUDIO_STATUS_MEDIA_RELEASE = 3,
   e8AUDIO_STATUS_RELEASE_AUDIO_DEVICE = 4,
};

enum tenTouchReleaseValidity
{
   e8P1Release = 0, e8P2Release = 1, e8P1P2Release = 2
};

typedef enum
{

   e8_ACTION_DOWN = 0,
   e8_ACTION_UP = 1,
   e8_ACTION_MOVED = 2,
   e8_ACTION_POINTER_DOWN = 5,
   e8_ACTION_POINTER_UP = 6,
   e8_ACTION_INVALID
} tenAAPPointerAction;

typedef enum
{
   e8_DISPLAY_CONTEXT = 0, e8_DISPLAY_CONSTRAINT = 1
}__attribute__ ((packed, aligned (1))) tenDisplayInfo;

struct trDisplayContext
{
      tenDisplayContext enDisplayContext;
      t_Bool bDisplayFlag;
};

typedef enum
{
   e8DIPO_TRANSFERTYPE_NA = 0,
   e8DIPO_TRANSFERTYPE_TAKE = 1,
   e8DIPO_TRANSFERTYPE_UNTAKE = 2,
   e8DIPO_TRANSFERTYPE_BORROW = 3,
   e8DIPO_TRANSFERTYPE_UNBORROW = 4
}__attribute__ ((packed, aligned (1))) tenDiPOTransferType;

typedef enum
{
   e8DIPO_TRANSFERPRIO_NA = 0, e8DIPO_TRANSFERPRIO_NICETOHAVE = 1, e8DIPO_TRANSFERPRIO_USERINITIATED = 2
}__attribute__ ((packed, aligned (1))) tenDiPOTransferPriority;

typedef enum
{
   e8DIPO_CONSTRAINT_NA = 0,
   e8DIPO_CONSTRAINT_ANYTIME = 1,
   e8DIPO_CONSTRAINT_USERINITIATED = 2,
   e8DIPO_CONSTRAINT_NEVER = 3
}__attribute__ ((packed, aligned (1))) tenDiPOConstraint;

/*! 
 * \struct rDisplayConstraint
 * \brief structure to hold video constraint data
 */
struct trDisplayConstraint
{
      //! Transfer type
      tenDiPOTransferType enTransferType;

      //! Transfer priority
      tenDiPOTransferPriority enTransferPriority;

      //! Transfer constraint
      tenDiPOConstraint enTakeConstraint;

      //! Transfer borrow constraint
      tenDiPOConstraint enBorrowConstraint;

      //! Borrow Id to identify the borrow request
      t_Char szBorrowId[STR_LENGTH];

      trDisplayConstraint()
      {
         enTransferType = e8DIPO_TRANSFERTYPE_NA;
         enTransferPriority = e8DIPO_TRANSFERPRIO_NA;
         enTakeConstraint = e8DIPO_CONSTRAINT_NA;
         enBorrowConstraint = e8DIPO_CONSTRAINT_NA;
         memset(szBorrowId, 0, STR_LENGTH);
      }

      trDisplayConstraint& operator=(trDisplayConstraint &DiPODispConst)
      {
         enTransferType = DiPODispConst.enTransferType;
         enTransferPriority = DiPODispConst.enTransferPriority;
         enTakeConstraint = DiPODispConst.enTakeConstraint;
         enBorrowConstraint = DiPODispConst.enBorrowConstraint;
         memset(szBorrowId, 0, STR_LENGTH);
         strcpy(szBorrowId, DiPODispConst.szBorrowId);
         return *this;
      }
};

struct trDisplayLayerAttributes
{
      tenDeviceCategory enDevCat;
      t_U16 u16TouchLayerID;
      t_U16 u16TouchSurfaceID;
      t_U16 u16VideoLayerID;
      t_U16 u16VideoSurfaceID;
      t_U16 u16LayerWidth;
      t_U16 u16LayerHeight;
      t_U16 u16DisplayHeightMm;
      t_U16 u16DisplayWidthMm;
      tenTouchEventType enTouchEventType;
      t_String szPixelAspectRatio;
      t_U16 u16dpi;
      t_U16 u16PriorityIndex;

      trDisplayLayerAttributes() :
                        enDevCat(e8DEV_TYPE_UNKNOWN),
                        u16TouchLayerID(0),
                        u16TouchSurfaceID(0),
                        u16VideoLayerID(0),
                        u16VideoSurfaceID(0),
                        u16LayerWidth(0),
                        u16LayerHeight(0),
                        u16DisplayHeightMm(0),
                        u16DisplayWidthMm(0),
                        enTouchEventType(e8MULTI_TOUCH),
                        szPixelAspectRatio("1"),
                        u16dpi(0),
                        u16PriorityIndex(0)
      {
      }

      trDisplayLayerAttributes(tenDeviceCategory enDevCategory, t_U16 u16TouchID, t_U16 u16TouchSurfID,
               t_U16 u16VideoLayID, t_U16 u16VideoSurfID, t_U16 u16LayWidth, t_U16 u16LayHeight, t_U16 u16DispHeightMm,
               t_U16 u16DispWidthMm, tenTouchEventType enTouchEvntTyp, t_String szPixelAspRatio, t_U16 u16DPI,
               t_U16 u16PrioIndex) :
                        enDevCat(enDevCategory),
                        u16TouchLayerID(u16TouchID),
                        u16TouchSurfaceID(u16TouchSurfID),
                        u16VideoLayerID(u16VideoLayID),
                        u16VideoSurfaceID(u16VideoSurfID),
                        u16LayerWidth(u16LayWidth),
                        u16LayerHeight(u16LayHeight),
                        u16DisplayHeightMm(u16DispHeightMm),
                        u16DisplayWidthMm(u16DispWidthMm),
                        enTouchEventType(enTouchEvntTyp),
                        szPixelAspectRatio(szPixelAspRatio),
                        u16dpi(u16DPI),
                        u16PriorityIndex(u16PrioIndex)
      {

      }

      trDisplayLayerAttributes& operator=(const trDisplayLayerAttributes& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            //copy info from corfrSrc
            enDevCat = corfrSrc.enDevCat;
            u16TouchLayerID = corfrSrc.u16TouchLayerID;
            u16TouchSurfaceID = corfrSrc.u16TouchSurfaceID;
            u16VideoLayerID = corfrSrc.u16VideoLayerID;
            u16VideoSurfaceID = corfrSrc.u16VideoSurfaceID;
            u16LayerWidth = corfrSrc.u16LayerWidth;
            u16LayerHeight = corfrSrc.u16LayerHeight;
            u16DisplayHeightMm = corfrSrc.u16DisplayHeightMm;
            u16DisplayWidthMm = corfrSrc.u16DisplayWidthMm;
            enTouchEventType = corfrSrc.enTouchEventType;
            szPixelAspectRatio = corfrSrc.szPixelAspectRatio;
            u16dpi = corfrSrc.u16dpi;
            u16PriorityIndex = corfrSrc.u16PriorityIndex;
         }   //if( &corfrSrc != this)
         return *this;
      }   //trDisplayLayerAttributes&

      t_Bool operator!=(const trDisplayLayerAttributes& corfrSrc)
      {
         t_Bool bRet = false;

         bRet = ((enDevCat != corfrSrc.enDevCat) || (u16TouchLayerID != corfrSrc.u16TouchLayerID)
                  || (u16TouchSurfaceID != corfrSrc.u16TouchSurfaceID) || (u16VideoLayerID != corfrSrc.u16VideoLayerID)
                  || (u16VideoSurfaceID != corfrSrc.u16VideoSurfaceID) || (u16LayerWidth != corfrSrc.u16LayerWidth)
                  || (u16LayerHeight != corfrSrc.u16LayerHeight) || (u16DisplayHeightMm != corfrSrc.u16DisplayHeightMm)
                  || (u16DisplayWidthMm != corfrSrc.u16DisplayWidthMm)
                  || (enTouchEventType != corfrSrc.enTouchEventType)
                  || (szPixelAspectRatio != corfrSrc.szPixelAspectRatio) || (u16dpi != corfrSrc.u16dpi)
                  || (u16PriorityIndex != corfrSrc.u16PriorityIndex));

         return bRet;
      }   //t_Bool operator!=(const trDis

};

struct trDisplayAttributes
{
      t_U16 u16ScreenHeight;
      t_U16 u16ScreenWidth;
      t_U16 u16ScreenHeightMm;
      t_U16 u16ScreenWidthMm;
      tenDisplayType enDisplayType;
      std::vector<trDisplayLayerAttributes> vecDisplayLayerAttr;

      trDisplayAttributes() :
                        u16ScreenHeight(0),
                        u16ScreenWidth(0),
                        u16ScreenHeightMm(0),
                        u16ScreenWidthMm(0),
                        enDisplayType(e8RESISTIVE_SCREEN)
      {
      }

      trDisplayAttributes& operator=(const trDisplayAttributes& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            //copy info from corfrSrc
            u16ScreenHeight = corfrSrc.u16ScreenHeight;
            u16ScreenWidth = corfrSrc.u16ScreenWidth;
            u16ScreenHeightMm = corfrSrc.u16ScreenHeightMm;
            u16ScreenWidthMm = corfrSrc.u16ScreenWidthMm;
            enDisplayType = corfrSrc.enDisplayType;
            vecDisplayLayerAttr = corfrSrc.vecDisplayLayerAttr;
         }   //if( &corfrSrc != this)
         return *this;
      }   //trDisplayAttributes& operator=(c

      t_Bool operator!=(const trDisplayAttributes& corfrSrc)
      {
         t_Bool bRet = false;

         bRet = ((u16ScreenHeight != corfrSrc.u16ScreenHeight) || (u16ScreenWidth != corfrSrc.u16ScreenWidth)
                  || (u16ScreenHeightMm != corfrSrc.u16ScreenHeightMm)
                  || (u16ScreenWidthMm != corfrSrc.u16ScreenWidthMm) || (enDisplayType != corfrSrc.enDisplayType)
                  || (vecDisplayLayerAttr.size() != corfrSrc.vecDisplayLayerAttr.size()));

         if (false == bRet)
         {
            for (t_U8 u8Index = 0; u8Index < vecDisplayLayerAttr.size(); u8Index++)
            {
               if (true == (vecDisplayLayerAttr[u8Index] != corfrSrc.vecDisplayLayerAttr[u8Index]))
               {
                  bRet = true;
                  break;
               }   //if(true == (vecDisplayLaye
            }   //for(t_U8 u8Index=0;u8Ind
         }   // if(false == bRet)
         return bRet;
      }
};

struct trOEMIconDetails
{
      t_String szOEMName;
      t_String szOEMPath;
      trOEMIconDetails()
      {
      }
      trOEMIconDetails& operator=(const trOEMIconDetails& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            szOEMName = corfrSrc.szOEMName;
            szOEMPath = corfrSrc.szOEMPath;
         }
         return *this;
      }
      t_Bool operator!=(const trOEMIconDetails& corfrSrc)
      {
         t_Bool bRet = false;
         bRet = ((szOEMName != corfrSrc.szOEMName) || (szOEMPath != corfrSrc.szOEMPath));
         return bRet;
      }
};
struct trVehicleManufacturerInformation
{
      t_String szVehicleMake;
      t_String szVehicleModel;
      t_String szVehicleModelYear;
      t_String szVehicleHUSwVersion;
      t_String szVehicleHUHwVersion;
      t_String szVehicleHUMake;
      t_String szVehicleHUModel;
      t_String szVehicleHUSwBuild;
      tenDriveSideInfo enDriverSideInfo;
      t_U16 u16SupportedSPITechnologiesInfo;
      t_Bool bIsNativeNavSupported;
      trOEMIconDetails rOEMIconDetails;
      t_Bool bIsRotaryCtrlSupported;
      tenFidelityType enFidelity;

      trVehicleManufacturerInformation() :
                        enDriverSideInfo(e8LEFT_HAND_DRIVE),
                        u16SupportedSPITechnologiesInfo(0),
                        bIsNativeNavSupported(false),
                        bIsRotaryCtrlSupported(false),
                        enFidelity(e8HIGH_FIDELITY)
      {
      }
      trVehicleManufacturerInformation& operator=(const trVehicleManufacturerInformation& corfrSrc)
      {
         if (&corfrSrc != this)
         {
            szVehicleMake = corfrSrc.szVehicleMake;
            szVehicleModel = corfrSrc.szVehicleModel;
            szVehicleModelYear = corfrSrc.szVehicleModelYear;
            szVehicleHUSwVersion = corfrSrc.szVehicleHUSwVersion;
            szVehicleHUHwVersion = corfrSrc.szVehicleHUHwVersion;
            szVehicleHUMake = corfrSrc.szVehicleHUMake;
            szVehicleHUModel = corfrSrc.szVehicleHUModel;
            szVehicleHUSwBuild = corfrSrc.szVehicleHUSwBuild;
            enDriverSideInfo = corfrSrc.enDriverSideInfo;
            u16SupportedSPITechnologiesInfo = corfrSrc.u16SupportedSPITechnologiesInfo;
            bIsNativeNavSupported = corfrSrc.bIsNativeNavSupported;
            rOEMIconDetails = corfrSrc.rOEMIconDetails;
            bIsRotaryCtrlSupported = corfrSrc.bIsRotaryCtrlSupported;
            enFidelity = corfrSrc.enFidelity;
         }   //if( &corfrSrc != this)
         return *this;
      }   //trVehicleManufacturerInformation& operator=(const trVehicleManufacturerInformation...
      t_Bool operator!=(const trVehicleManufacturerInformation& corfrSrc)
      {
         t_Bool bRet = false;
         bRet =
                  ((szVehicleMake != corfrSrc.szVehicleMake) || (szVehicleModel != corfrSrc.szVehicleModel)
                           || (szVehicleModelYear != corfrSrc.szVehicleModelYear)
                           || (szVehicleHUSwVersion != corfrSrc.szVehicleHUSwVersion)
                           || (szVehicleHUHwVersion != corfrSrc.szVehicleHUHwVersion)
                           || (szVehicleHUMake != corfrSrc.szVehicleHUMake)
                           || (szVehicleHUModel != corfrSrc.szVehicleHUModel)
                           || (szVehicleHUSwBuild != corfrSrc.szVehicleHUSwBuild)
                           || (enDriverSideInfo != corfrSrc.enDriverSideInfo)
                           || (u16SupportedSPITechnologiesInfo != corfrSrc.u16SupportedSPITechnologiesInfo)
                           || (bIsNativeNavSupported != corfrSrc.bIsNativeNavSupported)
                           || (rOEMIconDetails != corfrSrc.rOEMIconDetails)
                           || (bIsRotaryCtrlSupported != corfrSrc.bIsRotaryCtrlSupported)
                           || (enFidelity != corfrSrc.enFidelity));
         return bRet;
      }   //t_Bool operator!=(const trVehicleManufacturerInformation...
};
//! enum to define different audio configuration values
enum tenAudioStreamConfig
{
   e8AUDIOCONFIG_MAINAUDIO_24kHz = 0,
   e8AUDIOCONFIG_MAINAUDIO_16kHz = 1,
   e8AUDIOCONFIG_MAINAUDIO_8kHz = 2,
   e8AUDIOCONFIG_MAINAUDIO_MEDIA_BROWSING = 3,
   e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE = 4,
   e8AUDIOCONFIG_MAINAUDIO_ALERT = 5,
   e8AUDIOCONFIG_ALTERNATEAUDIO = 6,
   e8AUDIOCONFIG_MAINAUDIO_SPEECH = 7,
   e8AUDIOCONFIG_MAINAUDIO_DUMMY = 9,
   e8AUDIOCONFIG_ALERT = 10,
   e8AUDIOCONFIG_AUDIOIN_24kHz = 11,
   e8AUDIOCONFIG_AUDIOIN_16kHz = 12,
   e8AUDIOCONFIG_AUDIOIN_8kHz = 13,
   e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE_BDCL = 14,

   //! Please add new elements before ths line to ensure size is amintained
   e8AUDIOCONFIG_SIZE
};
//Session Configuration struature consisting of values for displaying clock,phone signal and battery level.
struct trSessionConfigurationInfo
{
      t_Bool bHideClock;
      t_Bool bHidePhoneSignal;
      t_Bool bHideBatteryLevel;

      trSessionConfigurationInfo() :
               bHideClock(false), bHidePhoneSignal(false), bHideBatteryLevel(false)
      {

      }
};

//! Structure holding the details of the headunit
struct trHeadUnitInfo
{
      t_String szHUManufacturer;
      t_String szModelYear;
      t_String szHUModelName;
      t_String szVehicleManufacturer;
      t_String szVehicleModelName;
      t_String szSoftwareBuild;
      tenDriveSideInfo enDrivePosition;
      t_String szSoftwareVersion;
      t_String szHardwareVersion;
      t_String szVehicleID;
      trSessionConfigurationInfo rSessionConfigurationInfo;
      trHeadUnitInfo() :
               enDrivePosition(e8UNKNOWN_DRIVE_SIDE)
      {

      }

      struct trAAPCertConfigData oAAAutoCertConfigData;
};

typedef std::map<tenAudioStreamConfig, t_String> tmapAudioPipeConfig;

enum tenAudioChannelStatus
{
   e8_AUD_NOT_ACTIVE = 0,      // Audio channel is not active
   e8_AUD_ACT_REQUESTED = 1,   // Audio channel activation is requested
   e8_AUD_ACT_GRANTED = 2,     // Audio channel is active
   e8_AUD_DEACT_REQUESTED = 3, // Audio channel deactivation is requested
};
//! indicates the certificate type to be used (used only for android auto as of now)
enum tenCertificateType
{
   e8_CERTIFICATETYPE_DEVELOPER = 0, e8_CERTIFICATETYPE_SDC = 1, e8_CERTIFICATETYPE_FFS = 2
};

struct trDisplayDpi
{
      t_U16 u16Dpi480p;
      t_U16 u16Dpi720p;
      t_U16 u16Dpi1080p;

      trDisplayDpi() :
               u16Dpi480p(160), u16Dpi720p(240), u16Dpi1080p(240)
      {

      }
};
//! Device Selection Modes
enum tenDeviceSelectionMode
{
   e16DEVICESEL_UNKNOWN = 0, e16DEVICESEL_MANUAL = 1, e16DEVICESEL_AUTOMATIC = 2, e16DEVICESEL_SEMI_AUTOMATIC = 3
};

typedef enum
{
   e8_TOUCHSCREEN_CAPACITIVE = 1, e8_TOUCHSCREEN_RESISTIVE = 2
} tenAAPTouchScreenType;

struct trDataServiceConfigData
{
      t_Bool bLocDataAvailable;
      t_Bool bDeadReckonedData;
      t_Bool bEnvData;
      t_Bool bGearStatus;
      t_Bool bAcclData;
      t_Bool bGyroData;
      t_Bool bParkBrakeData;
      t_Bool bSpeedData;
      t_U32 u32SpeedTimerIntervalInMs;
      t_Bool bTollCardStatus;
      //required for Baidu Carlife
      t_U32 u32GPSTimerIntervalInMs;
      t_U32 u32GyroscopeTimerIntervalInMs;
      t_U32 u32AccelerationTimerIntervalInMs;
      t_U32 u32GearTimerIntervalInMs;
      t_U32 u32OilTimerIntervalInMs;
      tenGeoCoordinateSystemType enGeoCoordinateSystemType;

      trDataServiceConfigData() :
                        bLocDataAvailable(true),
                        bDeadReckonedData(false),
                        bEnvData(false),
                        bGearStatus(false),
                        bAcclData(false),
                        bGyroData(false),
                        bParkBrakeData(false),
                        bSpeedData(true),
                        u32SpeedTimerIntervalInMs(100),
                        bTollCardStatus(false),
                        u32GPSTimerIntervalInMs(0),
                        u32GyroscopeTimerIntervalInMs(0),
                        u32AccelerationTimerIntervalInMs(0),
                        u32GearTimerIntervalInMs(0),
                        u32OilTimerIntervalInMs(0),
                        enGeoCoordinateSystemType(e8GEO_COORDINATE_SYSTEM_UNKNOWN)
      {

      }

};

//! \brief Structure holding notification's acknowledgement data
struct trNotificationAckData
{
      t_U32 u32DeviceHandle;
      tenDeviceCategory enDeviceCategory;
      t_String szNotifId;

      trNotificationAckData() :
               u32DeviceHandle(0), enDeviceCategory(e8DEV_TYPE_UNKNOWN), szNotifId("")
      {
      }
};

struct trConfigData
{
      t_Bool bAAPNavigationEnable;
      t_Bool bAAPNotificationEnable;
      t_Bool bAAPMediaPlaybackEnable;
      t_Bool bTBTUpdateEnabled;
      t_Bool bTBTAssisstantGuideEnabled;

      trConfigData() :
                        bAAPNavigationEnable(false),
                        bAAPNotificationEnable(false),
                        bAAPMediaPlaybackEnable(false),
                        bTBTUpdateEnabled(false),
                        bTBTAssisstantGuideEnabled(false)
      {

      }

};

struct trAppsAudioInfo
{
      t_U32 u32AppID;
      tenAppCategory enAppCat;
};
enum tenDeviceSubCategory
{
   e8DEVTYPE_UNKNWON = 0, e8DEVTYPE_AOAP = 1, e8DEVTYPE_IAP = 2
};

typedef std::function<void(tenResponseCode, tenErrorCode)> tfvOnRoleSwitchResult;
typedef std::function<void(t_U32)> tfvOnNativeTransportStart;
typedef std::function<void(t_U32)> tfvOnNativeTransportStop;
typedef std::function<void(t_U32, t_String&)> tfvGetSelectedDevSerialCallback;
typedef std::function<void(t_U32, t_Bool, t_Bool, t_String)> tfvBTProfileCallback;

struct trIAPDevInfoCallbacks
{
      //!Called when Native Transport Session has started
      tfvOnNativeTransportStart fvOnNativeTransportStart;

      //!Called when Native Transport Session has stopped
      tfvOnNativeTransportStop fvOnNativeTransportStop;

      trIAPDevInfoCallbacks() :
               fvOnNativeTransportStart(NULL), fvOnNativeTransportStop(NULL)
      {
      }
};
enum tenIapLaunchType
{
   e8LAUNCH_WITH_USER_ALERT = 0, e8LAUNCH_WITHOUT_USER_ALERT = 1

};
struct trMySPINIapAppInfo
{
      t_Bool bAudioSupport;
      t_String szDeviceId;
      t_String szProtocol;
      t_String szBundleSeedId;
      t_String szAppName;
      tenIapLaunchType enIapLaunchType;
};
struct trMySPINAoapAppInfo
{
      t_Bool bAudioSupport;
      t_String szVehicleModelName;
      t_String szVendorName;
      t_String szDescription;
      t_String szVersion;
      t_String szUrl;
      t_String szSerial;
};
struct trMySPINAppInfo
{
      trMySPINIapAppInfo rMySPINIapAppInfo;
      trMySPINAoapAppInfo rMySPINAoapAppInfo;
};

enum tenBTVoiceSessionStatus
{
   e8BTVOICESESSION_STATUS_UNAVAILABLE = 0,
   e8BTVOICESESSION_STATUS_IDLE = 1,
   e8BTVOICESESSION_STATUS_REQUEST_ACCEPTED = 2,
   e8BTVOICESESSION_STATUS_ACTIVE = 3,
   e8BTVOICESESSION_STATUS_REQUEST_DENIED_NOHFP = 4,
   e8BTVOICESESSION_STATUS_REQUEST_DENIED_VOICEALREADYACTIVE = 5,
   e8BTVOICESESSION_STATUS_REQUEST_DENIED = 6,
   e8BTVOICESESSION_STATUS_NONE = 7

};
enum tenProjVideoRenderingStatus
{
   e8_PROJECTION_VIDEO_RENDERING_STARTED = 1, e8_PROJECTION_VIDEO_RENDERING_STOPPED = 0
};

enum tenNativeVideoRenderingStatus
{
   e8_NATIVE_VIDEO_RENDERING_STOPPED = 0, e8_NATIVE_VIDEO_RENDERING_STARTED = 1, e8_NATIVE_VIDEO_RENDERING_UNKNOWN = 2

};

struct trProjectionVideoAttributes
{
      t_U16 u16ProjectionUIWidth;
      t_U16 u16ProjectionUIHeight;
      trProjectionVideoAttributes() :
               u16ProjectionUIWidth(0), u16ProjectionUIHeight(0)
      {

      }
};

enum tenSelectReason
{
   e8_REASON_UNKNOWN = 0,
   e8_REASON_HMI_TRIGGER = 1,
   e8_REASON_USER_TRIGGER = 2,
   e8_REASON_SPM_STATE_CHANGE = 3,
   e8_FACTORY_RESET = 4,
   e8_HU_WIFI_AP_OFF = 5,
   e8_RETRIAL_FAILED = 6,
   e8_STATION_DISCONNECTED = 7, //For WIFI
   e8_REASON_NO_DEVICE_ERROR = 8, //FOR BDCL - NO_DEVICE_ERROR
   e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER = 9,//! For CPW - to trigger PREPARE instead of DEACTIVATE
   e8_STATION_CONNECTION = 10
};

typedef std::function<t_Void(t_U32, trDeviceInfo)> tfvDeviceListUpdate;

typedef enum
{
   e8DIPO_ROLE_SWITCH_NOT_REQUIRED = 0,
   e8DIPO_ROLE_SWITCH_REQUIRED = 1,
   e8DIPO_ROLE_SWITCH_REQUIRED_FOR_NATIVE_TRANSPORT = 2,
   e8DIPO_ROLE_SWITCH_REQUIRED_FOR_CARPLAY_AND_NATIVE_TRANSPORT = 3,
   e8DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE = 4
} tenRoleSwitchResponse;

struct trEAPAppInfo
{
      t_String szAppName;
      t_String szProtocol;
      t_String szBundleID;
      t_Bool bLaunchType;

      trEAPAppInfo() :
               bLaunchType(false)
      {

      }

};

//! Holds the information for device selection
struct trSelectDeviceRequest
{
      t_U32 m_u32DeviceHandle;
      tenDeviceConnectionType m_enDevConnType;
      tenDeviceConnectionReq m_enDevConnReq;
      tenDeviceCategory m_enDevCategory;
      tenDeviceType m_enDeviceType;
      trUserContext m_corUsrCntxt;
      t_Bool m_bisSwitchRequired;
      tenSelectReason m_enSelectionReason;

      trSelectDeviceRequest(t_U32 cou32DeviceHandle, tenDeviceConnectionType enDevConnType,
               tenDeviceConnectionReq enDevConnReq, tenDeviceCategory enDevCategory, tenDeviceType enDeviceType,
               tenSelectReason enSelectionReason, trUserContext corUsrCntxt) :
                        m_u32DeviceHandle(cou32DeviceHandle),
                        m_enDevConnType(enDevConnType),
                        m_enDevConnReq(enDevConnReq),
                        m_enDevCategory(enDevCategory),
                        m_enDeviceType(enDeviceType),
                        m_corUsrCntxt(corUsrCntxt),
                        m_bisSwitchRequired(false),
                        m_enSelectionReason(enSelectionReason)
      {

      }
      ;
      trSelectDeviceRequest() :
                        m_u32DeviceHandle(0),
                        m_enDevConnType(e8UNKNOWN_CONNECTION),
                        m_enDevConnReq(e8DEVCONNREQ_SELECT),
                        m_enDevCategory(e8DEV_TYPE_UNKNOWN),
                        m_enDeviceType(e8_UNKNOWN_DEVICE),
                        m_bisSwitchRequired(false),
                        m_enSelectionReason(e8_REASON_UNKNOWN)
      {

      }
};

enum tenLastMode
{
   e8NOT_APPLICABLE = 0, e8PROJECTION = 1, e8NATIVE = 2
};

enum tenSrcAvailability
{
   e8SRC_NOT_AVAILABLE = 0, e8SRC_PLAYABLE_UNKNOWN = 1, e8SRC_PLAYABLE = 2, e8SRC_NOT_PLAYABLE = 3
};

/*!
 * \typedef enum arl_tenAvailabilityReason
 * \brief reasons for a change in the availability
 */
enum tenSrcAvailabilityReason
{
   e8REASON_NEWMEDIA = 0,           // Device is newly inserted
   e8REASON_SAMEMEDIA = 1,          // Device was already there before
   e8REASON_NOMEDIA = 2,            // Media removed
   e8REASON_TEMPERATURE = 3,        // Over-/Undertemperature
   e8REASON_VOLTAGE = 4,            // Over-/Undervoltage
   e8REASON_ERROR = 5,              // Device Access/ Read Error
   e8REASON_NO_CONTENT = 6,         // Device has no conttent
   e8REASON_OVERCURRENT = 7,        // Device has over current
   e8REASON_INTERNALDISCONNECT = 8	// Device has internal disconnect 
};

enum tenLastModeSupport
{
   e8LASTMODE_DISABLED = 0,   // Last mode is disabled for all technologies
   e8LASTMODE_CARPLAY = 1,    // Last mode is enabled only for CarPlay technology
   e8LASTMODE_ALLTECH = 2     // Last mode is enabled for all technologies
};

struct trAudioConfig
{
      t_Bool bMuteEnabled;
      t_Bool bVolLockEnabled;

      trAudioConfig() :
               bMuteEnabled(false), bVolLockEnabled(false)
      {
      }
};

struct trDeviceStatisticInfo
{
      t_U32 u32ConnectCount;
      t_U32 u32ConnectSuccessCount;
      t_U32 u32ConnectTime;

      trDeviceStatisticInfo() :
               u32ConnectCount(0), u32ConnectSuccessCount(0), u32ConnectTime(0)
      {

      }
};
//! Connection info structure
struct trDeviceConnectionCountInfo
{
      trDeviceStatisticInfo rDeviceStatisticInfo[MAX_DEV_CATEGORY];
};
enum class tenDisplayContextReason
{
   e8DISPLAY_CONTEXT_REASON_UNKNOWN = 0,
   e8DISPLAY_CONTEXT_REASON_HOME_SCREEN = 1,
   e8DISPLAY_CONTEXT_REASON_LAST_SCREEN = 2,
   e8DISPLAY_CONTEXT_REASON_PHONE_SCREEN_OFF = 3,
   e8DISPLAY_CONTEXT_REASON_SHOW_NATIVE_BT_SCREEN = 4
};

//! Dap attestation info
enum tenDAPAttestationInfo
{
   e8_DAP_ATTESTATION_NOT_REQ = 0, e8_DAP_ATTESTATION_REQ = 1
};

//! Selection State Details
struct trDeviceSelectionStateDetails
{
      t_U32 u32DeviceHandle;
      tenDeviceSelectionState enDeviceSelectionState;
      trDeviceSelectionStateDetails() :
               u32DeviceHandle(0), enDeviceSelectionState(e8_SELECTION_STATE_DEVICE_NOT_SELECTED)
      {

      }
};
//! Lanagugues supported by ATBox
enum tenLanguageSetting
{
   e32LANGUAGE_UNKNOWN = 0,
   e32LANGUAGE_ALBANIAN = 1,
   e32LANGUAGE_ARABIC = 2,
   e32LANGUAGE_BULGARIAN = 3,
   e32LANGUAGE_CHINESE_CANTONESE_SIMPLIFIED = 4,
   e32LANGUAGE_CHINESE_CANTONESE_TRADITIONAL = 5,
   e32LANGUAGE_CHINESE_MANDARIN_SIMPLIFIED = 6,
   e32LANGUAGE_CROATIAN = 7,
   e32LANGUAGE_CZECH = 8,
   e32LANGUAGE_DANISH = 9,
   e32LANGUAGE_DUTCH = 10,
   e32LANGUAGE_ENGLISH_AUSTRALIAN = 11,
   e32LANGUAGE_ENGLISH_UK = 12,
   e32LANGUAGE_ENGLISH_US = 13,
   e32LANGUAGE_ENGLISH_US_FOR_JPN = 14,
   e32LANGUAGE_ENGLISH_US_FOR_PRC = 15,
   e32LANGUAGE_ENGLISH_CANADIAN = 16,
   e32LANGUAGE_ENGLISH_INDIA = 17,
   e32LANGUAGE_ESTONIAN = 18,
   e32LANGUAGE_FINNISH = 19,
   e32LANGUAGE_FLEMISH = 20,
   e32LANGUAGE_FRENCH = 21,
   e32LANGUAGE_FRENCH_CANDIAN = 22,
   e32LANGUAGE_GERMAN = 23,
   e32LANGUAGE_GREEK = 24,
   e32LANGUAGE_HEBREW = 25,
   e32LANGUAGE_HINDI = 26,
   e32LANGUAGE_HUNGARIAN = 27,
   e32LANGUAGE_INDONESIAN = 28,
   e32LANGUAGE_ITALIAN = 29,
   e32LANGUAGE_JAPANESE = 30,
   e32LANGUAGE_KOREAN = 31,
   e32LANGUAGE_LATVIAN = 32,
   e32LANGUAGE_LITHUANIAN = 33,
   e32LANGUAGE_MALAY = 34,
   e32LANGUAGE_NORWEGIAN = 35,
   e32LANGUAGE_PERSIAN_FARSI_IRANIAN = 36,
   e32LANGUAGE_PHILIPPINES = 37,
   e32LANGUAGE_POLISH = 38,
   e32LANGUAGE_PORTUGUESE = 39,
   e32LANGUAGE_PORTUGUESE_BRAZILIAN = 40,
   e32LANGUAGE_ROMANIAN = 41,
   e32LANGUAGE_RUSSIAN = 42,
   e32LANGUAGE_SERBIAN = 43,
   e32LANGUAGE_SLOVAKIAN = 44,
   e32LANGUAGE_SLOVENIAN = 45,
   e32LANGUAGE_SPANISH = 46,
   e32LANGUAGE_SPANISH_LATIN_AMERICAN = 47,
   e32LANGUAGE_SPANISH_MEXICAN = 48,
   e32LANGUAGE_SWEDISH = 49,
   e32LANGUAGE_TAIWANESE = 50,
   e32LANGUAGE_THAI = 51,
   e32LANGUAGE_TURKISH = 52,
   e32LANGUAGE_UKRAINIAN = 53,
   e32LANGUAGE_VIETNAMESE = 54,
   e32LANGUAGE_UNSUPPORTED = -1
};

//! Identifies the different types of settings
typedef enum
{
   COMMON_SETTINGS = 1,
   RESOURCE_MNGR_SETTINGS = 2,
   INPUT_SETTINGS = 3,
   APP_MNGR_SETTINGS = 4,
   CONNECTION_SETTINGS = 5,
   AUDIO_SETTINGS = 6,
   DISCOVERER_SETTINGS = 7,
   BLUETOOTH_SETTINGS = 8,
   DATA_SERVICE_SETTINGS = 9,
} tenKDSCategory;

//! Identifies the vehicle Power mode state
typedef enum
{
   //! SLEEP state
   SLEEP_MODE = 0,
   //! Auto acceleration
   AUTO_ACC = 1,
   //! Ignition ON
   IGNITION_ON = 2,
   //! Cranking of the vehicle
   INITIAL_STARTING_IN_PROGRESS = 3,
   //! Engine is runnine
   ENGINE_RUNNING = 4,

   //! TODO - Take care that to add the values before this UNKNWON element if in future the FI is extended
   //! UNKNOWN Value
   VEHICLE_POWERMODE_UNSUPPORTED = 255
} tenVehiclePowerMode;

//! Structure contains Ambient color data details
struct trAmbientColorData
{
      t_Float fRed;
      t_Float fGreen;
      t_Float fBlue;
      t_Float fAlpha;
      trAmbientColorData() :
               fRed(0.0f), fGreen(0.0f), fBlue(0.0f), fAlpha(0.0f)
      {

      }
};

struct trAudSampleRate
{
   tenAudioSamplingRate enSampleRate;
   tenAudioSamplingRate enNativeSampleRate;
   trAudSampleRate(tenAudioSamplingRate sampleRate,tenAudioSamplingRate nativesampleRate)
   :enSampleRate(sampleRate),enNativeSampleRate(nativesampleRate){}
   trAudSampleRate():enSampleRate(e8AUD_SAMPLERATE_DEFAULT),enNativeSampleRate(e8AUD_SAMPLERATE_DEFAULT){}
};


enum tenClientState
{
   e8_CLIENT_STATUS_UNKNOWN =0,
   e8_CLIENT_STATUS_INITIALIZED =1
};

static const t_U8 scou8MaxKeys = 50;

typedef enum
{
  HARD_KEY = 0,
  SOFT_KEY = 1
}tenKeyCat;

//! Accepting phone call action options
enum tenPhoneCallAcceptAction
{
    e8_ACCEPT_OR_HOLD_AND_ACCEPT = 0,
    e8_END_AND_ACCEPT = 1
};

//! Ending phone call action options
enum tenPhoneCallEndAction
{
    e8_END_OR_DECLINE = 0,
    e8_END_ALL = 1
};

struct trDeviceTime
{
      //! time interval in seconds since epoch time
      t_U64 u64CurrDevTimeInSeconds;

      //! time zone
      t_S16 s16TimeZoneOffsetMinutes;

      //! daylight savings
      t_S8 s8DaylightSavingsOffsetMinutes;

      trDeviceTime()
      {
         u64CurrDevTimeInSeconds = 0;
         s16TimeZoneOffsetMinutes = 0;
         s8DaylightSavingsOffsetMinutes = 0;
      }
};

//! This enumeration identifies spm states set from SPI
typedef enum
{
   //! spm state change is temporary
   e8SPM_SUBSTATE_TEMPORARY = 0x00,
   //! spm state change is permanent
   e8SPM_SUBSTATE_PERMANENT = 0x01,
   //! Identifies Default value(Startup case) or status is not known
   e8SPM_SUBSTATE_NOT_KNOWN = 0xFF
}tenSpmSubState;

#endif //SPITYPES_H_
