/*!
*******************************************************************************
* \file              spi_tclExtCmdNavDataIntf.h
* \brief             Nav Data Interface class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Class which provides interfaces to register/un-register
                with Nav Data provider.
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
27.01.2017 | Dhiraj Asopa                 | Initial Version

\endverbatim
******************************************************************************/

#ifndef _SPI_TCLEXTCMDNAVDATAINTF_H_
#define _SPI_TCLEXTCMDNAVDATAINTF_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "SPITypes.h"
#include "spi_TBTTypes.h"
/**
* \brief Nav Data Intf class
*/
class spi_tclExtCmdNavDataIntf
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  virtual spi_tclExtCmdNavDataIntf::~spi_tclExtCmdNavDataIntf()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclExtCmdNavDataIntf()
   * \brief   Destructor
   * \sa
   **************************************************************************/
   virtual ~spi_tclExtCmdNavDataIntf(){}

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdNavDataIntf::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to Initialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bInitialize() = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdNavDataIntf::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
    * \brief   Method to UnInitialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bUnInitialize() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdNavDataIntf::bTransferGPSData(...)
   ***************************************************************************/
   /*!
    * \fn     bTransferGPSData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo, t_String szGPGGAData,
    *            t_String szGPRMCData, t_String szGPGSVData, t_String szGPHDTData)
    * \brief  Function to transfer GPS date to device.
    * \param  u32DeviceHandle : [IN] Unique device handle of active device
    * \param  szDeviceSerialNo : [IN] Unique serial number of active device
    * \param  szGPGGAData : [IN] GPGGA data
    * \param  szGPRMCData : [IN] GPRMS data
    * \param  szGPGSVData : [IN] GPGSV data
    * \param  szGPHDTData : [IN] GPHDT data
    * \retval bool : true if succees, false otherwise
   **************************************************************************/
   virtual t_Bool bTransferGPSData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
            t_String szGPGGAData, t_String szGPRMCData, t_String szGPGSVData, t_String szGPHDTData)= 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdNavDataIntf::bTransferDrData(...)
   ***************************************************************************/
   /*!
   * \fn     bTransferDrData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo, t_String szPASCDData,
   *            t_String szPAGCDData, t_String szPAACDData)
   * \brief  Function to transfer DR date to device.
   * \param  u32DeviceHandle : [IN] Unique device handle of active device
   * \param  szDeviceSerialNo : [IN] Unique serial number of active device
   * \param  szPASCDData : [IN] PASCD data
   * \param  szPAGCDData : [IN] PAGCD data
   * \param  szPAACDData : [IN] PAGCD data
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   virtual t_Bool bTransferDrData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
            t_String szPASCDData, t_String szPAGCDData, t_String szPAACDData)= 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdNavDataIntf::vRegisterForNavDataProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForNavDataProperties()
   * \brief   Registers for all interested properties to NavData
   **************************************************************************/
   virtual t_Void vRegisterForNavDataProperties() = 0;
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdNavDataIntf::vDipoStartRouteGuidanceUpdate()
   **************************************************************************/
   /*!
   * \fn      vDipoStartRouteGuidanceUpdate()
   * \brief   Function to start route guidance
   * \param   cou32DevHandle: Device handle of the device
   * \sa      vDipoStopRouteGuidance()
   **************************************************************************/
   virtual t_Void vDipoStartRouteGuidanceUpdate(const t_U32 cou32DevHandle) = 0;

    /***************************************************************************
    ** FUNCTION:  t_Void spi_tclExtCmdNavDataIntf::vDipoStopRouteGuidanceUpdate()
    **************************************************************************/
    /*!
    * \fn      vDipoStopRouteGuidanceUpdate()
    * \brief   Function to stop route guidance
    * \param   cou32DevHandle: Device handle of the device
    * \sa      vDipoStartRouteGuidance()
    **************************************************************************/
    virtual t_Void vDipoStopRouteGuidanceUpdate(const t_U32 cou32DevHandle) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdNavDataIntf::vUnregisterForNavDataProperties()
   **************************************************************************/
   /*!
   * \fn      vUnregisterForNavDataProperties()
   * \brief   Un-registers all subscribed properties to NavData
   **************************************************************************/
   virtual t_Void vUnregisterForNavDataProperties() = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclExtCmdNavDataIntf::bOnRegRouteGuidanceDisplayConf()
   ***************************************************************************/
   /*!
    * \fn     bOnRegRouteGuidanceDisplayConf(const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent)
    * \brief  Interface to send device list details
    * \param  [IN] corfvecRouteGuidanceDisplayComponent : registered display component list
    * \retval t_Bool  TRUE- Enabled FALSE-Disabled
    **************************************************************************/
   virtual t_Bool bOnRegRouteGuidanceDisplayConf(const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent) = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclExtCmdNavDataIntf::bOnStartRouteGuidanceUpdates
    ***************************************************************************/
   /*!
    * \fn     bOnStartRouteGuidanceUpdates(const trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates)
    * \brief  Interface to start route guidance updates
    * \param  [IN] cou32DevHandle : currently selected device handle
    * \param  [IN] trStartRouteGuidanceUpdates : start route guidance paramaeters
    * \retval t_Bool  TRUE- Enabled FALSE-Disabled
    **************************************************************************/
   virtual t_Bool bOnStartRouteGuidanceUpdates(t_U32 cou32DevHandle, trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates) = 0;

    /***************************************************************************
     ** FUNCTION: t_Void spi_tclExtCmdNavDataIntf::bOnStopRouteGuidanceUpdates
     ***************************************************************************/
    /*!
     * \fn     bOnStopRouteGuidanceUpdates(const trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates)
     * \brief  Interface to stop route guidance updates
    * \param  [IN] cou32DevHandle : currently selected device handle
     * \param  [IN] trStartRouteGuidanceUpdates : stop route guidance paramaeters
     * \retval t_Bool  TRUE- Enabled FALSE-Disabled
     **************************************************************************/
   virtual t_Bool bOnStopRouteGuidanceUpdates(t_U32 cou32DevHandle, trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates) = 0;


   /*************************************************************************
   ****************************END OF PUBLIC*********************************
   *************************************************************************/

private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   *! Data members
   ***************************************************************************/

   /*************************************************************************
   ****************************END OF PRIVATE********************************
   *************************************************************************/

};

#endif // #define _SPI_TCLEXTCMDNAVDATAINTF_H_

