/*!
*******************************************************************************
* \file              spi_tclExtCmdBluetoothIntf.h
* \brief             Bluetooth Adapter Interface class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Class which provides interfaces to register/un-register
                with Bluetooth data provider.
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
02.02.2017 |  Unmukt Jain                 | Initial Version

\endverbatim
******************************************************************************/

#ifndef _SPI_TCLEXTCMDBLUETOOTHINTF_H_
#define _SPI_TCLEXTCMDBLUETOOTHINTF_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "spi_BluetoothTypedefs.h"
/**
* \brief   Bluetooth Intf class
*/
class spi_tclExtCmdBluetoothIntf
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  virtual spi_tclExtCmdBluetoothIntf::~spi_tclExtCmdBluetoothIntf()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclExtCmdBluetoothIntf()
   * \brief   Destructor
   * \sa
   **************************************************************************/
   virtual ~spi_tclExtCmdBluetoothIntf(){}

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to Initialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bInitialize() = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
    * \brief   Method to UnInitialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bUnInitialize() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bGetPairingStatus(const...)
   ***************************************************************************/
   /*!
   * \fn      bGetPairingStatus(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request Bluetooth service for the pairing status of a BT device.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  Bool value: TRUE - if Pairing is required, else FALSE
   **************************************************************************/
   virtual t_Bool bGetPairingStatus(const t_String& rfcoszDeviceBTAddress) = 0;

   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclExtCmdBluetoothIntf::u32GetBTDeviceHandle(const...)
   ***************************************************************************/
   /*!
   * \fn      u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request Bluetooth Client for the BT DeviceHandle of a device.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  t_U32 : BT device handle of device with BT address = rfcoszDeviceBTAddress.
   *          If the device is not found in BT DeviceList, zero is returned.
   **************************************************************************/
   virtual t_U32 u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress) = 0;

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetoothIntf::szGetBTDeviceName()
   ***************************************************************************/
   /*!
   * \fn      szGetBTDeviceName()
   * \brief   Interface to fetch name of a BT device.
   * \param   [IN] rfcszDeviceBTAddress: BT address of device.
   * \retval  t_String : BT device name of requested device.
   *            If device name is not available, returns empty string.
   **************************************************************************/
   virtual t_String szGetBTDeviceName(const t_String& rfcszDeviceBTAddress) = 0;
   
   /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetoothIntf::szGetVehicleBTFriendlyName()
   ***************************************************************************/
   /*!
   * \fn      szGetVehicleBTFriendlyName()
   * \brief   Interface to fetch BT friendly name of vehicle
   * \retval  t_String : HU's BT friendly name.
   **************************************************************************/
   virtual t_String szGetVehicleBTFriendlyName() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bGetBTServiceAvailability(...)
   ***************************************************************************/
   /*!
   * \fn      bGetBTServiceAvailability()
   * \brief   Interface to check Bluetooth Service Availability in HU.
   * \param   None
   * \retval  t_Bool : True - if BT Service is available, else false.
   **************************************************************************/
   virtual t_Bool bGetBTServiceAvailability() = 0;

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bSetBTLimitationMode(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a Bluetooth limitation in HU.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] rfcszBTDeviceName: BT device name
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool : True - if request to set limitation mode is sent successfully,
   *              else false.
   **************************************************************************/
   virtual t_Bool bSetBTLimitationMode(const t_String& rfcszBTAddress,const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
                                       tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction) = 0;

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetoothIntf::bSetBTPairingResponse(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set BT pairing response .
   * \param   [IN] enBTPairingResponse: BT pairing response
   * \retval  t_Bool : True - if pairing response is sent successfully,
   *              else false.
   **************************************************************************/
   virtual t_Bool bSetBTPairingResponse(tenPairingResponseType enBTPairingResponse) = 0;

  /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetoothIntf::vGetBtMacAddress()
   ***************************************************************************/
   /*!
   * \fn      vGetBtMacAddress()
   * \brief   Interface to fetch BT MAC Address from the friendly Name.
   * \param   [IN] rfcszDevName: BT name of device.
   * \param   [IN/OUT] rfszMACAddress: BT MAC Address of device.
   * \retval  NONE.
   **************************************************************************/
   virtual t_Void vGetBtMacAddress(const t_String& rfcszDevName, t_String& rfszMACAddress) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetoothIntf::vRegisterDeviceNameCallback()
   ***************************************************************************/
   /*!
   * \fn      vRegisterDeviceNameCallback(const t_String& rfcszAAPBTAddress)
   * \brief   Interface to register for device name callback of a specific device
   * \param   [IN] rfcszAAPBTAddress: BT Address of the device
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterDeviceNameCallback(const t_String& rfcszAAPBTAddress) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetoothIntf::vRegisterBTMacAddressCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterBTMacAddressCallbacks()
   * \brief   Register for BT Mac address callbacks.
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterBTMacAddressCallbacks(trVehicleBTAddressCallback &rVehicleBTAddress) = 0;
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetoothIntf::vGetBTProfileStatus()
   ***************************************************************************/
   /*!
   * \fn      vGetBTProfileStatus(...)
   * \brief   Interface to register for device name callback of a specific device
   * \param   [IN] u8DeviceHandle: BT device handle
   * \param   [OUT] rfbIsHFPConn: HFP connection status
   * \param   [OUT] rfbIsA2DPConn: A2DP connection status
   * \retval  None
   **************************************************************************/   
   virtual t_Void vGetBTProfileStatus(t_U8 u8DeviceHandle,
         t_Bool& rfbIsHFPConn, t_Bool& rfbIsA2DPConn) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetoothIntf::vTriggerBTNameUpdate()
   ***************************************************************************/
   /*!
   * \fn      vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress)
   * \brief   Interface to get device name associated with a specific device
   * \param   [IN] rfcszAAPBTAddress: BT Address of the device
   * \retval  None
   **************************************************************************/
   virtual t_Void vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress) = 0;

   /*************************************************************************
   ****************************END OF PUBLIC*********************************
   *************************************************************************/

private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   *! Data members
   ***************************************************************************/

   /*************************************************************************
   ****************************END OF PRIVATE********************************
   *************************************************************************/

};

#endif // #define _SPI_TCLEXTCMDBLUETOOTHINTF_H_

