/*!
*******************************************************************************
* \file              spi_tclExtCmdBluetooth.h
* \brief             Bluetooth Adapter Implementation class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Bluetooth Adapter Implementation class
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
02.02.2017 |  Unmukt Jain                 | Initial Version
\endverbatim
******************************************************************************/

#ifndef SPI_TCLEXTCMDBLUETOOTH_H_
#define SPI_TCLEXTCMDBLUETOOTH_H_


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/


#include "spi_tclExtCmdBluetoothIntf.h"
#include "spi_BluetoothTypedefs.h"

class spi_tclBluetoothClient;

class spi_tclExtCmdBluetooth: public spi_tclExtCmdBluetoothIntf
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/
   /***************************************************************************
   ** FUNCTION:  spi_tclExtCmdBluetooth::spi_tclExtCmdBluetooth()
   ***************************************************************************/
  /*!
   * \fn      spi_tclExtCmdBluetooth()
   * \brief   Default constructor
   * \sa
   **************************************************************************/
	spi_tclExtCmdBluetooth();

   /***************************************************************************
   ** FUNCTION:  virtual spi_tclExtCmdBluetooth::~spi_tclExtCmdBluetooth()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclExtCmdBluetooth()
   * \brief   Destructor
   * \sa
   **************************************************************************/
   virtual ~spi_tclExtCmdBluetooth();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to Initialize
    * \sa      bInitialize()
    **************************************************************************/
   t_Bool bInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
    * \brief   Method to UnInitialize
    * \sa      bInitialize()
    **************************************************************************/
   t_Bool bUnInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bGetPairingStatus(const...)
   ***************************************************************************/
   /*!
   * \fn      bGetPairingStatus(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request Bluetooth service for the pairing status of a BT device.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  Bool value: TRUE - if Pairing is required, else FALSE
   **************************************************************************/
   t_Bool bGetPairingStatus(const t_String& rfcoszDeviceBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclExtCmdBluetooth::u32GetBTDeviceHandle(const...)
   ***************************************************************************/
   /*!
   * \fn      u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request Bluetooth Client for the BT DeviceHandle of a device.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  t_U32 : BT device handle of device with BT address = rfcoszDeviceBTAddress.
   *          If the device is not found in BT DeviceList, zero is returned.
   **************************************************************************/
   t_U32 u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetooth::szGetBTDeviceName()
   ***************************************************************************/
   /*!
   * \fn      szGetBTDeviceName()
   * \brief   Interface to fetch name of a BT device.
   * \param   [IN] rfcszDeviceBTAddress: BT address of device.
   * \retval  t_String : BT device name of requested device.
   *            If device name is not available, returns empty string.
   **************************************************************************/
   t_String szGetBTDeviceName(const t_String& rfcszDeviceBTAddress);
   
   /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetoothIntf::szGetVehicleBTFriendlyName()
   ***************************************************************************/
   /*!
   * \fn      szGetVehicleBTFriendlyName()
   * \brief   Interface to fetch BT friendly name of vehicle
   * \retval  t_String : HU's BT friendly name.
   **************************************************************************/
   t_String szGetVehicleBTFriendlyName();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bGetBTServiceAvailability(...)
   ***************************************************************************/
   /*!
   * \fn      bGetBTServiceAvailability()
   * \brief   Interface to check Bluetooth Service Availability in HU.
   * \param   None
   * \retval  t_Bool : True - if BT Service is available, else false.
   **************************************************************************/
   t_Bool bGetBTServiceAvailability();

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bSetBTLimitationMode(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a Bluetooth limitation in HU.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] rfcszBTDeviceName: BT device name
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool : True - if request to set limitation mode is sent successfully,
   *              else false.
   **************************************************************************/
   t_Bool bSetBTLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
                               tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction);

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclExtCmdBluetooth::bSetBTPairingResponse(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a BT Pairing Response
   * \param   [IN] enBTPairingResponse: BT Pairing response
   * \retval  t_Bool : True - if BT Pairing Response is sent successfully,
   *              else false.
   **************************************************************************/
   t_Bool bSetBTPairingResponse(tenPairingResponseType enBTPairingResponse);

  /***************************************************************************
   ** FUNCTION:  t_String spi_tclExtCmdBluetooth::vGetBtMacAddress()
   ***************************************************************************/
   /*!
   * \fn      vGetBtMacAddress()
   * \brief   Interface to fetch BT MAC Address from the friendly Name.
   * \param   [IN] rfcszDevName: BT name of device.
   * \param   [IN/OUT] rfszMACAddress: BT MAC Address of device.
   * \retval  NONE.
   **************************************************************************/
   t_Void vGetBtMacAddress(const t_String& rfcszDevName, t_String& rfszMACAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vRegisterDeviceNameCallback()
   ***************************************************************************/
   /*!
   * \fn      vRegisterDeviceNameCallback(const t_String& rfcszAAPBTAddress)
   * \brief   Interface to register for device name callback of a specific device
   * \param   [IN] rfcszAAPBTAddress: BT Address of the device
   * \retval  None
   **************************************************************************/
   t_Void vRegisterDeviceNameCallback(const t_String& rfcszAAPBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vRegisterBTMacAddressCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterBTMacAddressCallbacks()
   * \brief   Register for BT Mac address callbacks.
   * \retval  None
   **************************************************************************/
   t_Void vRegisterBTMacAddressCallbacks(trVehicleBTAddressCallback &rVehicleBTAddress);
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vRegisterDeviceNameCallback()
   ***************************************************************************/
   /*!
   * \fn      vGetBTProfileStatus(...)
   * \brief   Interface to register for device name callback of a specific device
   * \param   [IN] u8DeviceHandle: BT device handle
   * \param   [OUT] rfbIsHFPConn: HFP connection status
   * \param   [OUT] rfbIsA2DPConn: A2DP connection status
   * \retval  None
   **************************************************************************/   
   t_Void vGetBTProfileStatus(t_U8 u8DeviceHandle,
         t_Bool& rfbIsHFPConn, t_Bool& rfbIsA2DPConn);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vTriggerBTNameUpdate()
   ***************************************************************************/
   /*!
   * \fn      vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress)
   * \brief   Interface to get device name associated with a specific device
   * \param   [IN] rfcszAAPBTAddress: BT Address of the device
   * \retval  None
   **************************************************************************/
   t_Void vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress);

   /*************************************************************************
   ****************************END OF PUBLIC*********************************
   *************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTPairingStatusCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTPairingStatusCb(tenBTSetPairingStatus enBTSetPairingStatus)
   * \brief   Interface to receive BT Pairing Status.
   * \param   [IN] enBTSetPairingStatus: Bluetooth pairing status
   * \param   [IN] szBTDeviceAddress : Bluetooth MAC Address
   * \retval  None
   **************************************************************************/
   t_Void vOnBTPairingStatusCb(tenBTSetPairingStatus enBTSetPairingStatus, t_String szBTDeviceAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTConnectionChangedCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTConnectionChangedCb(t_String szBTDeviceAddress, tenBTConnectionResult enBTConnResult)
   * \brief   Interface to receive notification on a BT device connection/disconnection.
   * \param   [IN] szBTDeviceAddress: Bluetooth address of device
   * \param   [IN] enBTConnResult: Connection/Disconnection result.
   * \retval  None
   **************************************************************************/
   t_Void vOnBTConnectionChangedCb(t_String szBTDeviceAddress, tenBTConnectionResult enBTConnResult);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTDeviceNameUpdate(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTDeviceNameUpdate()
   * \brief   Interface to receive notification on device name of BT paired devices.
   * \param   [IN] szBTDeviceAddress: BT Address of paired device
   * \param   [IN] szBTDeviceName: BT Device name of paired device
   * \retval  None
   **************************************************************************/
   t_Void vOnBTDeviceNameUpdate(t_String szBTDeviceAddress,
         t_String szBTDeviceName);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTLimitationModeCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTLimitationModeCb()
   * \brief   Interface to receive notification when there is a change in BT modes.
   * \param   [IN] rBTLimitationMode: BT limitation mode info
   * \retval  None
   **************************************************************************/
   t_Void vOnBTLimitationModeCb(trBTLimitationModeInfo rBTLimitationModeInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnSetBTLimitationModeResultCb(...)
    ***************************************************************************/
   /*!
    * \fn      vOnSetBTLimitationModeResultCb()
    * \brief   Interface to receive notification BT Limitation Mode Result
    * \param   [IN] bSetBTLimitationModeResult: Set BT limitation mode Result info
    * \retval  None
    **************************************************************************/
   t_Void vOnSetBTLimitationModeResultCb(t_Bool bSetBTLimitationModeResult);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTServiceAvailabilityCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTServiceAvailabilityCb(t_Bool bServiceAvailable)
   * \brief   Interface to receive notification when BT Service availability changes.
   * \param   [IN] bServiceAvailable: True if BT Service is available, else False.
   * \retval  None
   **************************************************************************/
   t_Void vOnBTServiceAvailabilityCb(t_Bool bServiceAvailable);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTPairingInfoCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTPairingInfoCb()
   * \brief   Interface to receive notification when there is a pairing request to HU.
   * \param   [IN] rBTPairingReqInfo: Info about pairing request
   * \retval  None
   **************************************************************************/
   t_Void vOnBTPairingInfoCb(trBTPairingRequestInfo rBTPairingReqInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTDeviceNameUpdateCb(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTDeviceNameUpdateCb()
   * \brief   Interface to receive notification on device name of BT paired devices.
   * \param   [IN] szBTDeviceAddress: BT Address of paired device
   * \param   [IN] szBTDeviceName: BT Device name of paired device
   * \retval  None
   **************************************************************************/
   t_Void vOnBTDeviceNameUpdateCb(t_String szBTDeviceAddress, t_String szBTDeviceName);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vOnBTProfileStatus(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTProfileStatus()
   * \brief   Interface to receive notification on profile status of BT paired devices.
   * \retval  None
   **************************************************************************/   
   t_Void vOnBTProfileStatus();

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/

private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vRegisterBTCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterBTCallbacks()
   * \brief   register for BT connection callbacks.
   * \retval  None
   **************************************************************************/
   t_Void vRegisterBTCallbacks();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclExtCmdBluetooth::vRegisterPairingInfoCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterPairingInfoCallbacks()
   * \brief   Register for BT pairing info callbacks.
   * \retval  None
   **************************************************************************/
   t_Void vRegisterPairingInfoCallbacks();

   /***************************************************************************
   *! Data members
   ***************************************************************************/

   //! BT client handler pointer
   spi_tclBluetoothClient *m_poBTClientHandler;

   /*************************************************************************
   ****************************END OF PRIVATE********************************
   *************************************************************************/

};
#endif // #define SPI_TCLEXTCMDBLUETOOTH_H_
