/***********************************************************************/
/*!
* \file  spi_tclDiPoWiFi.cpp
* \brief Base class for DiPO WiFi specific impl
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Base class for DiPO WiFi specific impl
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                  | Modification
13.02.2018  | Shiva Kumar Gurija      | Initial Version
12.01.2019  | Ramesh Sidaray Mudalagi | Changes in handling the Device Connction and Disconnection Status

\endverbatim
*************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdWiFi.h"
#include "DiPOTypes.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPoWiFi.h"
#include "spi_tclDeviceIDDataIntf.h"

#include <string>
#include <algorithm>



#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_WIFI
      #include "trcGenProj/Header/spi_tclDiPoWiFi.cpp.trc.h"
   #endif
#endif

static const t_U8 scou8InvalidDeviceHandle = 0;

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static spi_tclDiPOManager* spoDiPOManager = NULL;

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
** FUNCTION:  spi_tclDiPoWiFi::spi_tclDiPoWiFi()
***************************************************************************/
spi_tclDiPoWiFi::spi_tclDiPoWiFi():m_enSessionTransport(e8_SESSION_TRANSPORT_INVALID),
         m_CurrentlySelectedDevice(0)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::spi_tclDiPoWiFi()"));
   
   spoDiPOManager = spi_tclDiPOManager::getInstance();
   if (NULL != spoDiPOManager)
   {
      //Register for the callbacks from the required callbacks.
       spoDiPOManager->bRegisterObject((spi_tclDiPORespSession*)this);
   }//if(NULL != spoVncManager )
}

/***************************************************************************
** FUNCTION:  spi_tclDiPoWiFi::~spi_tclDiPoWiFi()
***************************************************************************/
spi_tclDiPoWiFi::~spi_tclDiPoWiFi()
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::~spi_tclDiPoWiFi()"));
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoWiFi::bInitialize()
***************************************************************************/
t_Bool spi_tclDiPoWiFi::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::bInitialize()"));
   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vUninitialize()
****************************************************************************/
t_Void spi_tclDiPoWiFi::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vUninitialize()"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoWiFi::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclDiPoWiFi::vRegisterCallbacks(trWiFiCallbacks &rfrWifiCallbacks)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vRegisterCallbacks entered"));
   m_rDiPoWifiCallbacks = rfrWifiCallbacks;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vSelectDevice()
***************************************************************************/
t_Void spi_tclDiPoWiFi::vSelectDevice(const t_U32 cou32DevId,
   const tenDeviceConnectionReq coenConnReq)
{
  ETG_TRACE_USR1(("spi_tclDiPoWiFi::vSelectDevice()"));  
  //@TODO:currently everything is handled in common WiFi class. Some of the functionality to be moved in next releases.


  // Updating the map with Default details on the SelectDevice
  if( e8DEVCONNREQ_SELECT == coenConnReq)
  {
     //Device that will be selected
     m_CurrentlySelectedDevice = cou32DevId;

     m_STALock.s16Lock();
     auto itMapSTADetails = m_MapSTADetails.find(cou32DevId);
     if(itMapSTADetails == m_MapSTADetails.end())
     {
        //empty string,Currently Session active as false, Previously Session active as false, session transport as UNKNOWN- Initial values
        m_MapSTADetails[cou32DevId].szSessionIP = "";
        m_MapSTADetails[cou32DevId].bCurrentlySessionActive = false;
        m_MapSTADetails[cou32DevId].bPreviouslySessionActive = false;
        m_MapSTADetails[cou32DevId].enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
     }
     m_STALock.vUnlock();
  }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclDiPoWiFi::vOnSelectDeviceResult(const t_U32 cou32DevId,
   const tenDeviceConnectionReq coenConnReq,
   const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vOnSelectDeviceResult() Entered"));
   //@TODO:currently everything is handled in common WiFi class. Some of the functionality to be moved in next releases.

   //Map should be updated here also, what if the User has canceled the selection, Even in that case Device details should be saved
   //IP as empty, STA disconnedted as False, CanbeSelected as True, Transport as unknown
   if(e8DEVCONNREQ_SELECT == coenConnReq)
   {
      if( e8FAILURE == coenRespCode)
      {
         m_STALock.s16Lock();
         auto itMapSTADetails = m_MapSTADetails.find(cou32DevId);
         if(itMapSTADetails != m_MapSTADetails.end())
         {
            //Session active as false, Previously Session active as false, session transport as UNKNOWN- Initial values
            itMapSTADetails->second.bCurrentlySessionActive = false;
            itMapSTADetails->second.bPreviouslySessionActive = false;
            itMapSTADetails->second.enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
         }
         m_CurrentlySelectedDevice = scou8InvalidDeviceHandle;
         m_STALock.vUnlock();
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionMsg(...
***************************************************************************/
t_Void spi_tclDiPoWiFi::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                             tenDiPOSessionTransport enSessionTransport,
                             t_String szSessionIPAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vOnSessionMsg():Entered with SessionState:%d,SessionTransport:%d and Session IP as %s",
   enDiPOSessionState,enSessionTransport, szSessionIPAddress.c_str()));
  switch(enDiPOSessionState)
  {
     case e8DIPO_SESSION_START:
     {
        switch(enSessionTransport)
        {
            case e8_DIPO_SESSION_TRANSPORT_WIFI:
            {
               m_enSessionTransport = e8_SESSION_TRANSPORT_WIFI;
            }
            break;
            default:
            {
               m_enSessionTransport = e8_SESSION_TRANSPORT_USB;
            }
            break;
        }
        vOnSessionStartSetDeviceDetails(szSessionIPAddress);
     }
     break;
     case e8DIPO_SESSION_END:
     {
        //reset session transport
         vOnSessionEndSetDeviceDetails(szSessionIPAddress, enSessionTransport);
     }
     break;
     default:
     {
       m_CurrentlySelectedDevice = scou8InvalidDeviceHandle;
     }
     break;
  }

}

   
/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionMsg(...
***************************************************************************/
t_Void spi_tclDiPoWiFi::vEnableWiFiRestrictions(t_Bool bEnableRest)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vEnableWiFiRestrictions():Enable Restrictions:%d",
     ETG_ENUM(BOOL,bEnableRest)));

   spi_tclExtCmdWiFiIntf *poExtCmdWiFiIntf =  NULL;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   if(NULL != poExtCompMgr)
   {
      poExtCmdWiFiIntf = poExtCompMgr->poGetCmdWiFiIntfInst();
   }
   if(NULL != poExtCmdWiFiIntf)
   {
      poExtCmdWiFiIntf->vSetWiFiLimitationMode(e8DEV_TYPE_DIPO,bEnableRest);
      ETG_TRACE_USR1(("spi_tclDiPoWiFi::vEnableWiFiRestrictions():Enable Restrictions request sent"));
   }
}

/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclDiPoWiFi::u32GetDisconnectedStation (const std::vector<trStationInfo>& corfrStationsInfo)
 ***************************************************************************/
t_U32 spi_tclDiPoWiFi::u32GetDisconnectedStation (const std::vector<trStationInfo>& corfrStationsInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::u32GetDisconnectedStation():No.Of.Stations:%d",
            corfrStationsInfo.size()));
   ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation(): Currently Selected Device Handle %d", m_CurrentlySelectedDevice));

   t_Bool bIsDeviceDisconnected = false;
   t_U32 u32DeviceHandle = scou8InvalidDeviceHandle;
   t_String szSessionIP = szGetCurrentlyActiveSessionIP();

   m_STALock.s16Lock();
   for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation(): Device Handle %d with Session IP as %s",itMapSTADetails->first, (itMapSTADetails->second).szSessionIP.c_str()));
   }
   m_STALock.vUnlock();

   //If the session is active over WiFi, check whether the current IP address exist in the connected stations list. If CPW active and STA not available then only return true.
   if(e8_SESSION_TRANSPORT_WIFI == m_enSessionTransport)
   {
      //If the session is active, set the flag to TRUE initially. In case if the device is disconnected, it will not be available in the list of connected stations.
      if(0 != szSessionIP.length())
      {
         bIsDeviceDisconnected = true;
         for(t_U8 u8Index=0;u8Index<corfrStationsInfo.size();u8Index++)
         {
            ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation: Station:%d,HostName:%s",u8Index,corfrStationsInfo[u8Index].szDeviceName.c_str()));
            ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation: IPAddress:%s",corfrStationsInfo[u8Index].szIPAddress.c_str()));
            if((szSessionIP == corfrStationsInfo[u8Index].szIPAddress))
            {
               //Station with the active session is found.
               bIsDeviceDisconnected = false;
               ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation:Active CPW device is still available"));
               break;
            }
         }
      }
   }
   //1. iPhone might have reported the device disconnection first and then WBL, In that case Session Transport will be Invalid
   //2. No Active CPW session
   else if(e8_SESSION_TRANSPORT_INVALID == m_enSessionTransport)
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation():No active CPW session"));
   }

   //! WBL reports the device disconnection first and then the iPhone reports the SESSION_END. PreviouslySelected and CurrentlySelected should be set accordingly for the particular device
   if((e8_SESSION_TRANSPORT_WIFI == m_enSessionTransport) && (true == bIsDeviceDisconnected))
   {
      m_STALock.s16Lock();
      auto itMapSTADetails = m_MapSTADetails.find(m_CurrentlySelectedDevice);
      if(itMapSTADetails != m_MapSTADetails.end())
      {
         (itMapSTADetails->second).bCurrentlySessionActive = false;
         (itMapSTADetails->second).bPreviouslySessionActive = true;
         u32DeviceHandle = m_CurrentlySelectedDevice;
      }
      m_STALock.vUnlock();
   }

   /*
    ** Return Value false can be for 3 reasons:**
    * 1. Session is Still active with some of the device
    * 2. iPhone might have reported the device disconnection first and then WBL, In that case Session Transport will be Invalid
    * 3. No Active CPW session
   */
   if(false == bIsDeviceDisconnected)
   {
      m_STALock.s16Lock();
      for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
      {
         if(true == (itMapSTADetails->second).bCurrentlySessionActive)
         {
            ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation():Session Active"));
            break;
         }
         if((true == (itMapSTADetails->second).bPreviouslySessionActive)
                  &&(m_CurrentlySelectedDevice == itMapSTADetails->first) && (e8_SESSION_TRANSPORT_INVALID == m_enSessionTransport))
         {
            ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation():Previously Selected Device Handle %d Before Clearing the Details", m_CurrentlySelectedDevice));
            (itMapSTADetails->second).bCurrentlySessionActive = false;
            (itMapSTADetails->second).bPreviouslySessionActive = false;
            m_CurrentlySelectedDevice = scou8InvalidDeviceHandle;
            break;
         }
      }
      m_STALock.vUnlock();
   }
   ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetDisconnectedStation(): Left with Device Handle that will be De-selected is %d", u32DeviceHandle));
   return u32DeviceHandle;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoWiFi::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclDiPoWiFi::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vRestoreSettings Entered"));
   m_STALock.s16Lock();
   //clearing the private data on factory reset
   if(false == m_MapSTADetails.empty())
   {
      m_MapSTADetails.clear();
   }
   m_STALock.vUnlock();

   m_ConnectedSTALock.s16Lock();
   m_setConnectedStation.clear();
   m_ConnectedSTALock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails()
***************************************************************************/
t_Void spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(t_String szActiveSessionIPAddress)
{
   ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(): Session Active with the IP address %s", szActiveSessionIPAddress.c_str()));
   m_STALock.s16Lock();
   auto itMap = m_MapSTADetails.find(m_CurrentlySelectedDevice);
   if (itMap != m_MapSTADetails.end())
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(): Updating device details for Device Handle %d", m_CurrentlySelectedDevice));
      itMap->second.bCurrentlySessionActive = true;
      itMap->second.bPreviouslySessionActive = false;
      itMap->second.szSessionIP = szActiveSessionIPAddress;
      if (e8_SESSION_TRANSPORT_WIFI == m_enSessionTransport)
      {
         //Session is established successfully. Enable Feature restrictions.
         vEnableWiFiRestrictions(true);
         itMap->second.enSessionTransport = e8_SESSION_TRANSPORT_WIFI;
      }
      else if (e8_SESSION_TRANSPORT_USB == m_enSessionTransport)
      {
         itMap->second.enSessionTransport = e8_SESSION_TRANSPORT_USB;
      }
   }
   else
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(): Could not find the Map"));
   }

   //onSession Start Message Populating intial values for all the Devices except for the one with which Session has started
   for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
   {
      if(szActiveSessionIPAddress != (itMapSTADetails->second).szSessionIP)
      {
         (itMapSTADetails->second).bCurrentlySessionActive = false;
         (itMapSTADetails->second).bPreviouslySessionActive = false;
         (itMapSTADetails->second).enSessionTransport = e8_SESSION_TRANSPORT_UNKNOWN;
      }
   }
   m_STALock.vUnlock();

   m_STALock.s16Lock();
   for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(): Device Handle %d with Session IP as %s",itMapSTADetails->first, (itMapSTADetails->second).szSessionIP.c_str()));
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionStartSetDeviceDetails(): bCurrentlySessionActive %d and bPreviouslySessionActive %d",ETG_ENUM(BOOL,itMapSTADetails->second.bCurrentlySessionActive), ETG_ENUM(BOOL, itMapSTADetails->second.bPreviouslySessionActive)));
   }
   m_STALock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails()
***************************************************************************/
t_Void spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails(const t_String szSessionIPAddress, tenDiPOSessionTransport enSessionTransport)
{
   //reset session transport
   m_enSessionTransport = e8_SESSION_TRANSPORT_INVALID;

   //Session IP,Currently Session active as false, Previously Session active as true- on SessionEnd
   if (false == szSessionIPAddress.empty())
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails():Currently Selected Device Handle %d Before Clearing the Details of CurrentlySessionActive", m_CurrentlySelectedDevice));

      spi_tclDeviceIDDataIntf oDataIntf;
      t_U32 u32DeviceHandle = oDataIntf.u32GetDeviceIDFromIPAddress(szSessionIPAddress);

      if(u32DeviceHandle == m_CurrentlySelectedDevice)
      {
         m_STALock.s16Lock();

         auto itMap = m_MapSTADetails.find(m_CurrentlySelectedDevice);
         if(itMap != m_MapSTADetails.end())
         {
            itMap->second.szSessionIP = szSessionIPAddress;
            itMap->second.bCurrentlySessionActive = false;
            if (e8_DIPO_SESSION_TRANSPORT_WIFI == enSessionTransport)
            {
               //Session is terminated over WiFi. Disable Feature restrictions.
               vEnableWiFiRestrictions(false);
               itMap->second.enSessionTransport = e8_SESSION_TRANSPORT_WIFI;
               if (true == itMap->second.bPreviouslySessionActive)
               {
                  itMap->second.bPreviouslySessionActive = false;
                  m_CurrentlySelectedDevice = scou8InvalidDeviceHandle;
               }
               else
               {
                  itMap->second.bPreviouslySessionActive = true;
               }
            }
            else if (e8_DIPO_SESSION_TRANSPORT_USB == enSessionTransport)
            {
               itMap->second.enSessionTransport = e8_SESSION_TRANSPORT_USB;
               itMap->second.bPreviouslySessionActive = false;
               m_CurrentlySelectedDevice = scou8InvalidDeviceHandle;
            }
         }
         m_STALock.vUnlock();
      }
   }

   m_STALock.s16Lock();
   for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
   {
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails(): Currently Selected Device Handle %d", m_CurrentlySelectedDevice));
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails(): Device Handle %d with Session IP as %s",itMapSTADetails->first, (itMapSTADetails->second).szSessionIP.c_str()));
      ETG_TRACE_USR4(("spi_tclDiPoWiFi::vOnSessionEndSetDeviceDetails(): bCurrentlySessionActive %d and bPreviouslySessionActive %d",ETG_ENUM(BOOL,itMapSTADetails->second.bCurrentlySessionActive), ETG_ENUM(BOOL, itMapSTADetails->second.bPreviouslySessionActive)));
   }
   m_STALock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoWiFi::szGetCurrentlyActiveSessionIP()
***************************************************************************/
t_String spi_tclDiPoWiFi::szGetCurrentlyActiveSessionIP()
{
   t_String szSessionIP="";
   m_STALock.s16Lock();
   for(std::map<t_U32, trSTADetails>::iterator itMapSTADetails = m_MapSTADetails.begin(); itMapSTADetails != m_MapSTADetails.end(); itMapSTADetails++ )
   {
      if(true == (itMapSTADetails->second).bCurrentlySessionActive)
      {
         szSessionIP = (itMapSTADetails->second).szSessionIP;
         break;
      }
   }
   m_STALock.vUnlock();
   ETG_TRACE_USR4(("spi_tclDiPoWiFi::szGetCurrentlyActiveSessionIP():left with Session IP as %s ", szSessionIP.c_str()));
   return szSessionIP;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoWiFi::vPostWiFiConfig()
 ***************************************************************************/
t_Void spi_tclDiPoWiFi::vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPoWiFi::vPostWiFiConfig(): No.Of Stations:%d",corvecStationsInfo.size()));
   t_U32 u32ConnectedDeviceHandle = scou8InvalidDeviceHandle;
   vReportDisconnectedStation (rfcorWiFiConfig,corvecStationsInfo);

   if(("CarPlayWireless" == rfcorWiFiConfig.szType) && (true == rfcorWiFiConfig.bPowered))
   {
      u32ConnectedDeviceHandle = u32GetConnectedStationInfo(corvecStationsInfo);
      if((u32ConnectedDeviceHandle != scou8InvalidDeviceHandle)&& (NULL != m_rDiPoWifiCallbacks.fvDeviceSelectionOnConnection))
      {
         m_rDiPoWifiCallbacks.fvDeviceSelectionOnConnection(u32ConnectedDeviceHandle);
      }
   }
}

/********************************************************************************************************************************************************
 ** FUNCTION:  t_U32 spi_tclDiPoWiFi::vReportDisconnectedStation (const trWiFiConfig& rfcorWiFiConfig, const std::vector<trStationInfo>& corfrStationsInfo)
 ***************************************************************************************************************************************************************/

t_Void spi_tclDiPoWiFi::vReportDisconnectedStation(const trWiFiConfig& rfcorWiFiConfig, const std::vector<trStationInfo>& corvecStationsInfo)
{
   t_U32 u32DisconnectedDeviceHandle = scou8InvalidDeviceHandle;
   t_Bool isAPTypeNormal = false;
   vRemoveDisconnectedSTAfromSet(rfcorWiFiConfig,corvecStationsInfo);
   if(("Normal" == rfcorWiFiConfig.szType) || (false == rfcorWiFiConfig.bPowered))
   {
      isAPTypeNormal = true;
      m_ConnectedSTALock.s16Lock();
      m_setConnectedStation.clear();
      m_ConnectedSTALock.vUnlock();
   }
   if((("CarPlayWireless" == rfcorWiFiConfig.szType) && (true == rfcorWiFiConfig.bPowered)) || (true == isAPTypeNormal))
   {
      u32DisconnectedDeviceHandle = u32GetDisconnectedStation(corvecStationsInfo);
      if((u32DisconnectedDeviceHandle != scou8InvalidDeviceHandle) && (NULL != m_rDiPoWifiCallbacks.fvDeviceDeselectionOnDisconnection))
      {
         m_rDiPoWifiCallbacks.fvDeviceDeselectionOnDisconnection(u32DisconnectedDeviceHandle, e8_STATION_DISCONNECTED);
      }
   }

}
/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclDiPoWiFi::u32GetConnectedStationInfo (const std::vector<trStationInfo>& corfrStationsInfo)
 ***************************************************************************/
t_U32 spi_tclDiPoWiFi::u32GetConnectedStationInfo(const std::vector<trStationInfo>& corfrStationsInfo)
{
   t_U32 u32DeviceHandle = 0;
   t_String szIPAddress;

   if(0 != corfrStationsInfo.size())
   {
      m_ConnectedSTALock.s16Lock();
      for(trStationInfo itVec : corfrStationsInfo)
      {
         auto itSetofDevice = m_setConnectedStation.find(itVec.szIPAddress);
         if(itSetofDevice == m_setConnectedStation.end())
         {
            //Report the Device as Connected
            szIPAddress = itVec.szIPAddress;
            if(false == szIPAddress.empty())
            {
               auto itSet = m_setConnectedStation.insert(itVec.szIPAddress);
               break;
            }
         }
      }
      m_ConnectedSTALock.vUnlock();
   }

   if(false == szIPAddress.empty())
   {
      spi_tclDeviceIDDataIntf oDataIntf;
      u32DeviceHandle = oDataIntf.u32GetDeviceIDFromIPAddress(szIPAddress);
   }

   ETG_TRACE_USR4(("spi_tclDiPoWiFi::u32GetConnectedStationInfo():left with Device handle as %d ", u32DeviceHandle));
   return u32DeviceHandle;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoWiFi::vRemoveDisconnectedSTAfromSet (const std::vector<trStationInfo>& corfrStationsInfo)
 ***************************************************************************/
t_Void spi_tclDiPoWiFi::vRemoveDisconnectedSTAfromSet(const trWiFiConfig& rfcorWiFiConfig,const std::vector<trStationInfo>& corfrStationsInfo)
{
   t_Bool bDeviceReported = false;
   std::set<t_String> setOfDisconnectedStations;
   std::set<t_String> setOfWBLReportedStations;

   m_ConnectedSTALock.s16Lock();
   if(0 == corfrStationsInfo.size())
   {
      m_setConnectedStation.clear();
   }

   else if (("CarPlayWireless" == rfcorWiFiConfig.szType) && (true == rfcorWiFiConfig.bPowered))
   {
      for(trStationInfo rStaInfo : corfrStationsInfo)
      {
         setOfWBLReportedStations.insert(rStaInfo.szIPAddress);
      }

      std::set_difference(m_setConnectedStation.begin(), m_setConnectedStation.end(), setOfWBLReportedStations.begin(), setOfWBLReportedStations.end(),
               std::inserter(setOfDisconnectedStations, setOfDisconnectedStations.begin()));

      if(0 != setOfDisconnectedStations.size())
      {
         for(t_String szIPAddress : setOfDisconnectedStations)
         {
            ETG_TRACE_USR4(("spi_tclDiPoWiFi::vRemoveDisconnectedSTAfromSet(): IP Address as %s will be erased from Set", szIPAddress.c_str()));
         }

         for(t_String szDisconnectedIP : setOfDisconnectedStations)
         {
            auto itDisconnectedDevice  = m_setConnectedStation.find(szDisconnectedIP);
            if(itDisconnectedDevice != m_setConnectedStation.end())
            {
               m_setConnectedStation.erase(szDisconnectedIP);
            }
         }
      }
   }
   m_ConnectedSTALock.vUnlock();
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>

