/***********************************************************************/
/*!
* \file  spi_tclDiPoWiFi.h
* \brief Base class for WiFi specific impl
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Base class for DiPO WiFi specific impl
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
13.02.2018  | Shiva Kumar Gurija    | Initial Version

\endverbatim
*************************************************************************/

#ifndef _SPI_TCLDIPOWIFI_H_
#define _SPI_TCLDIPOWIFI_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "BaseTypes.h"
#include "spi_tclWiFiDevBase.h"
#include "spi_tclDiPORespSession.h"
#include "Lock.h"
#include <set>

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/


/****************************************************************************/
/*!
* \class spi_tclDiPoWiFi
* \brief Base class for DiPO WiFi specific impl
****************************************************************************/
class spi_tclDiPoWiFi : public spi_tclWiFiDevBase,public spi_tclDiPORespSession
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoWiFi::spi_tclDiPoWiFi()
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoWiFi()
   * \brief   Default Constructor
   * \sa      ~spi_tclDiPoWiFi()
   **************************************************************************/
   explicit spi_tclDiPoWiFi();

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoWiFi::~spi_tclDiPoWiFi()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclDiPoWiFi()
   * \brief   Destructor
   * \sa      spi_tclDiPoWiFi()
   **************************************************************************/
   ~spi_tclDiPoWiFi();
  
   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoWiFi::spi_tclDiPoWiFi(const spi_tclDiPoWiFi&)
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoWiFi(const spi_tclDiPoWiFi&) = delete;
   * \brief   deletes default copy Constructor
   * \sa      ~spi_tclDiPoWiFi()
   **************************************************************************/
   spi_tclDiPoWiFi(const spi_tclDiPoWiFi&) = delete;

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoWiFi& soperator=(const spi_tclDiPoWiFi&)
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoWiFi& operator=(const spi_tclDiPoWiFi&) = delete;
   * \brief   deletes default assignment operator for the class
   **************************************************************************/
   spi_tclDiPoWiFi& operator=(const spi_tclDiPoWiFi&) = delete;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoWiFi::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the WiFi related classes
   * \retval  t_Bool
   * \sa      vUninitialize()
   **************************************************************************/
   t_Bool bInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vUninitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialize()
   * \brief   To Uninitialize all the WiFi related classes
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Void vUninitialize();

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoWiFi::vRegisterCallbacks
    ***************************************************************************/
   /*!
    * \fn     vRegisterCallbacks()
    * \brief  interface for the creator class to register for the required
    *        callbacks.
    * \param rfrWifiCallbacks : reference to the callback structure
    *        populated by the caller
    **************************************************************************/
   t_Void vRegisterCallbacks(trWiFiCallbacks &rfrWifiCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vSelectDevice()
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vSelectDevice(const t_U32 cou32DevId,
   *          const tenDeviceConnectionReq coenConnReq)
   * \brief   To Initialize/UnInitialize Video setup for thr currently selected device
   * \pram    cou32DevId : [IN] Unique Device Id
   * \param   coenConnReq : [IN] connected/disconnected
   * \retval  t_Void
   **************************************************************************/
   t_Void vSelectDevice(const t_U32 cou32DevId,
      const tenDeviceConnectionReq coenConnReq);

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSelectDeviceResult(const t_U32 cou32DevId,
   *                 const tenDeviceConnectionReq coenConnReq,
   *                 const tenResponseCode coenRespCode)
   * \brief   To perform the actions that are required, after the select device is
   *           successful/failed
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    coenConnReq : [IN] Identifies the Connection Request.
   * \pram    coenRespCode: [IN] Response code. Success/Failure
   * \retval  t_Void
   **************************************************************************/
   t_Void vOnSelectDeviceResult(const t_U32 cou32DevId,
      const tenDeviceConnectionReq coenConnReq,
      const tenResponseCode coenRespCode);


   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoWiFi::vRestoreSettings
    ***************************************************************************/
   /*!
    * \fn     t_Void vRestoreSettings()
    * \brief  vRestoreSettings Method. Invoked on factory reset to clear the Private Data
    * \param  None
    **************************************************************************/
   t_Void vRestoreSettings();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnWiFiConfig(...)
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostWiFiConfig()
    * \brief   Called when ActiveWifiSetup structure is sent by WBL Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN] rfcorWiFiConfig: structure of type trWiFiConfig
    * \param   [IN] corvecStationsInfo: List of connected stations.
    * \sa      None
    **************************************************************************/
   t_Void vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo);



   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionMsg(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
   *                                       tenDiPOSessionTransport enSessionTransport,
   *                                       t_String szSessionIPAddress)
   * \brief   To Post the session message to SPI, when the session state changes.
   * \param   enDiPOSessionState : [IN] Session State : Session Started,Session_Ended,
   *                                                    CarPlayd_PlugIn_Loaded or UnLoaded.
   * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
   *                                    Applicable, only when SessionState is Session Started/ended.
   * \param   szSessionIPAddress : [IN] IP Address of the active carplay device.
   *                                    used, only if the Session Transport is WiFi
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                                tenDiPOSessionTransport enSessionTransport,
                                t_String szSessionIPAddress);

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vOnSessionMsg(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vEnableWiFiRestrictions(t_Bool bEnableRest)
   * \brief   Method to enable/disable restrictions with WBL
   * \param   bEnableRest : [IN] TRUE- Enable restrictions
   * \retval  t_Void
   ***************************************************************************/
   t_Void vEnableWiFiRestrictions(t_Bool bEnableRest);

   /***************************************************************************
   ** FUNCTION:  t_Void vOnSessionStartSetDeviceDetails()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionStartSetDeviceDetails()
   * \brief   Method to set the device details accordingly on Session start
   * \param   szActiveSessionIPAddress - Activce Session IP
   **************************************************************************/
   t_Void vOnSessionStartSetDeviceDetails(t_String szActiveSessionIPAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void vOnSessionEndSetDeviceDetails()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionEndSetDeviceDetails()
   * \brief   Method to set the device details accordingly on Session End
   * \param   szSessionIPAddress: Session IP Address with which the session has ended
   * \param   enSessionTransport: Session transport over which session has ended
   **************************************************************************/
   t_Void vOnSessionEndSetDeviceDetails(t_String szSessionIPAddress, tenDiPOSessionTransport enSessionTransport);

   /***************************************************************************
   ** FUNCTION:  t_String szGetCurrentlyActiveSessionIP()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetCurrentlyActiveSessionIP()
   * \brief   Method to get the session Ip with which the session is active
   * \retval  returns currently active Session IP
   **************************************************************************/
   t_String szGetCurrentlyActiveSessionIP();
   
   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclDiPoWiFi::u32GetDisconnectedStation()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetDisconnectedStation(
   *                 const std::vector<trStationInfo>& corfrStationsInfo)
   * \brief   Method to check whether station which with SPI session is active got disconnected.
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/
   t_U32 u32GetDisconnectedStation (const std::vector<trStationInfo>& corfrStationsInfo);

   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclDiPoWiFi::u32GetConnectedStationInfo()
   ***************************************************************************/
   /*!
   * \fn      t_Bool u32GetConnectedStationInfo(
   *                 const std::vector<trStationInfo>& corfrStationsInfo)
   * \brief   Method to Handle Station connecting to CPW AP
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/
   t_U32 u32GetConnectedStationInfo (const std::vector<trStationInfo>& corfrStationsInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vRemoveDisconnectedSTAfromSet()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRemoveDisconnectedSTAfromSet(
   *                 const trWiFiConfig& rfcorWiFiConfig,const std::vector<trStationInfo>& corfrStationsInfo)
   * \brief   Method to Handle Station connecting to CPW AP
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/
   t_Void vRemoveDisconnectedSTAfromSet (const trWiFiConfig& rfcorWiFiConfig,const std::vector<trStationInfo>& corfrStationsInfo);
   
      /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoWiFi::vReportDisconnectedStation()
   ***************************************************************************/
   /*!
   * \fn      t_Void vReportDisconnectedStation(
   *                const trWiFiConfig& rfcorWiFiConfig,, const std::vector<trStationInfo>& corfrStationsInfo)
   * \brief   Method to Handle Station connecting to CPW AP
   * \param   [IN] corvecStationsInfo: List of connected stations.
   **************************************************************************/
   
   t_Void vReportDisconnectedStation(const trWiFiConfig& rfcorWiFiConfig, const std::vector<trStationInfo>& corvecStationsInfo);

   struct trSTADetails
   {
         t_String szSessionIP;
         t_Bool bCurrentlySessionActive;
         t_Bool bPreviouslySessionActive;
         tenSessionTransport enSessionTransport;

         trSTADetails():
            bCurrentlySessionActive(false),
            bPreviouslySessionActive(false),
            enSessionTransport(e8_SESSION_TRANSPORT_UNKNOWN)
         {

         }
   };

   // Maintains the current session transport. 
   tenSessionTransport m_enSessionTransport;

  // Map to store the Details of the STA connected to Carplay Wireless AP
   std::map<t_U32, trSTADetails> m_MapSTADetails;

   std::set<t_String> m_setConnectedStation;

   Lock m_ConnectedSTALock;

   //Currently Selected Devie
   t_U32 m_CurrentlySelectedDevice;

   Lock m_STALock;

   trWiFiCallbacks m_rDiPoWifiCallbacks;
  
   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

}; //spi_tclDiPoWiFi

#endif //_SPI_TCLDIPOWIFI_H_


///////////////////////////////////////////////////////////////////////////////
// <EOF>

