/***********************************************************************/
/*!
 * \file  spi_tclDiPoVideoResourceMngr.cpp
 * \brief Manages DiPo Video resource
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Manages DiPo Video resource
 AUTHOR:         Anjan kumar sahu
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 17.08.2018  | Anjan kumar sahu      | Initial Version 
 27.08.2018  | Chaitra Srinivasa     | Improvements to add and remove methods and member variables

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclDiPoVideoResourceMngr.cpp.trc.h"
#endif

#include "SPITypes.h"
#include "spi_tclDiPoVideoResourceMngr.h"
#include "spi_tclDiPOManager.h"
#include "DiPOWrapperTypes.h"
#include "spi_tclDipoResourceMngrIntf.h"
#include "spi_tclDiPOCmdVideo.h"
#include "spi_tclDiPOCmdRsrcMngr.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION: spi_tclDiPoVideoResourceMngr::spi_tclDiPoVideoResourceMngr()
 ***************************************************************************/
spi_tclDiPoVideoResourceMngr::spi_tclDiPoVideoResourceMngr(spi_tclDipoResourceMngrIntf* poDipoResrcMngrIntf) :
                  m_bIsLastModeSupported(false),
                  m_bIsRequestUIRequired(false),
                  m_enCurDispCntxt(e8DISPLAY_CONTEXT_NATIVE),
                  m_enDisplayInfo(e8_DISPLAY_CONTEXT),
                  m_bIsVideoPlaybackStarted(false),
                  m_bCurDispFlag(false),
                  m_bDisplayOffActive(false),
                  m_enSpmSubState(e8SPM_SUBSTATE_NOT_KNOWN),
                  m_bSubStateData(false),
                  m_poDipoRsrcMngrIntf(poDipoResrcMngrIntf),
                  m_enLastDisplayMode(e8NOT_APPLICABLE),
                  m_bStartupDispFlag(false),
                  m_enStartupDispCntxt(e8DISPLAY_CONTEXT_NATIVE),
                  m_poResourceMngrSettings(NULL)
{
   m_rLastDispTakeContext.enBorrowConstraint = e8DIPO_CONSTRAINT_NA;
   m_rLastDispTakeContext.enDisplayContext = e8DISPLAY_CONTEXT_NATIVE;
   m_rLastDispTakeContext.enTakeConstraint = e8DIPO_CONSTRAINT_NA;
   m_rLastDispTakeContext.enTransferPriority = e8DIPO_TRANSFERPRIO_NA;
   m_rLastDispTakeContext.enTransferType = e8DIPO_TRANSFERTYPE_NA;
   memset(m_rLastDispTakeContext.szBorrowId, 0, MAX_STR_LEN);

   m_rCurDispConstraint.enTransferType = e8DIPO_TRANSFERTYPE_NA;
   m_rCurDispConstraint.enTransferPriority = e8DIPO_TRANSFERPRIO_NA;
   m_rCurDispConstraint.enTakeConstraint = e8DIPO_CONSTRAINT_NA;
   m_rCurDispConstraint.enBorrowConstraint = e8DIPO_CONSTRAINT_NA;
   memset(m_rCurDispConstraint.szBorrowId, 0, MAX_STR_LEN);

   spi_tclDiPOManager* poDiPOManager = spi_tclDiPOManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == poDiPOManager);
   SPI_NORMAL_ASSERT(NULL == m_poDipoRsrcMngrIntf);

   if (NULL != poDiPOManager)
   {
      m_poDiPOCmdRsrcMngr = poDiPOManager->poGetDiPORsrcMngrInstance();
      m_poDiPOCmdVideo = poDiPOManager->poGetDiPOVideoInstance();
      poDiPOManager->bRegisterObject((spi_tclDiPORespVideo*) this);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoVideoResourceMngr::~spi_tclDiPoVideoResourceMngr
 ***************************************************************************/
spi_tclDiPoVideoResourceMngr::~spi_tclDiPoVideoResourceMngr()
{
   m_poDipoRsrcMngrIntf = NULL;
   m_poResourceMngrSettings = NULL;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoVideoResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDeviceConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   SPI_INTENTIONALLY_UNUSED(u32DevID);
   SPI_INTENTIONALLY_UNUSED(enDeviceConnReq);
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);
   if (true == m_bIsLastModeSupported)
   {
      vUpdateLastDisplayMode(false);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayMode(t_U32...
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayMode(const trDisplayContext corDisplayContext,
         const trDisplayConstraint corDisplayConstraint, const tenDisplayInfo coenDisplayInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayMode: Resource Management handled by SPI:%d,DisplayInfo:%d", ETG_ENUM(BOOL,
            (e8_DISPLAY_CONTEXT == coenDisplayInfo)), coenDisplayInfo));

   m_enDisplayInfo = coenDisplayInfo;

   if (e8_DISPLAY_CONTEXT == coenDisplayInfo)
   {
      vSetAccessoryDisplayContext(corDisplayContext.bDisplayFlag, corDisplayContext.enDisplayContext);
   }
   else if (e8_DISPLAY_CONSTRAINT == coenDisplayInfo)
   {
      if (NULL != m_poDiPOCmdRsrcMngr)
      {
         trDiPOVideoContext rDiPOVideoContext;
         rDiPOVideoContext.enDisplayContext = e8DISPLAY_CONTEXT_NATIVE; // Dummy value keep it for backward compatibility.
         rDiPOVideoContext.enTransferType = corDisplayConstraint.enTransferType;
         rDiPOVideoContext.enTransferPriority = corDisplayConstraint.enTransferPriority;
         rDiPOVideoContext.enTakeConstraint = corDisplayConstraint.enTakeConstraint;
         rDiPOVideoContext.enBorrowConstraint = corDisplayConstraint.enBorrowConstraint;
         memset(rDiPOVideoContext.szBorrowId, 0, MAX_STR_LEN);
         strcpy(rDiPOVideoContext.szBorrowId, corDisplayConstraint.szBorrowId);

         if (true == m_bIsLastModeSupported)
         {
            // Store the last take context to determine the last mode
            if (e8DIPO_TRANSFERTYPE_TAKE == rDiPOVideoContext.enTransferType)
            {
               m_rLastDispTakeContext = rDiPOVideoContext;
            }
            // Check the last mode and send the display context according to it
            vHandleDisplayLastMode(rDiPOVideoContext);
         }

         m_poDiPOCmdRsrcMngr->bSetAccDisplayCntxt(rDiPOVideoContext);
      }

      m_rCurDispConstraint.enTransferType = corDisplayConstraint.enTransferType;
      m_rCurDispConstraint.enTransferPriority = corDisplayConstraint.enTransferPriority;
      m_rCurDispConstraint.enTakeConstraint = corDisplayConstraint.enTakeConstraint;
      m_rCurDispConstraint.enBorrowConstraint = corDisplayConstraint.enBorrowConstraint;
      memset(m_rCurDispConstraint.szBorrowId, 0, MAX_STR_LEN);
      strcpy(m_rCurDispConstraint.szBorrowId, corDisplayConstraint.szBorrowId);
      if ((true == m_bIsRequestUIRequired) && (NULL != m_poDiPOCmdVideo))
      {
         //Send request UI request along with unborrow on disengaging RVC/ on phone call end if last mode was CarPlay
         m_poDiPOCmdVideo->vRequestUI("");
         m_bIsRequestUIRequired = false;
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vOnPluginLoaded()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vOnPluginLoaded()
{
   ETG_TRACE_USR4(("[PARAM]:: spi_tclDiPOResourceMngr::vOnPluginLoaded m_enStartupDispCntxt = %d", ETG_ENUM(DISPLAY_CONTEXT,
            m_enStartupDispCntxt)));
   ETG_TRACE_USR4(("[PARAM]:: spi_tclDiPOResourceMngr::vOnPluginLoaded m_bStartupDispFlag = %d", ETG_ENUM(BOOL,
            m_bStartupDispFlag)));

   trDiPOVideoContext rDiPOVideoContext;
   rDiPOVideoContext.enDisplayContext = m_enStartupDispCntxt; // Dummy value keep it for backward compatibility.
   rDiPOVideoContext.enTransferType = m_rCurDispConstraint.enTransferType;
   rDiPOVideoContext.enTransferPriority = m_rCurDispConstraint.enTransferPriority;
   rDiPOVideoContext.enTakeConstraint = m_rCurDispConstraint.enTakeConstraint;
   rDiPOVideoContext.enBorrowConstraint = m_rCurDispConstraint.enBorrowConstraint;
   memset(rDiPOVideoContext.szBorrowId, 0, MAX_STR_LEN);
   strcpy(rDiPOVideoContext.szBorrowId, m_rCurDispConstraint.szBorrowId);

   m_bCurDispFlag = m_bStartupDispFlag; //the display flag is updated as true/false depending on the context update from the previous backed up state
   m_enCurDispCntxt = m_enStartupDispCntxt; //the display context is updated from the previous backed up state

   if (NULL != m_poDiPOCmdRsrcMngr)
   {
      m_poDiPOCmdRsrcMngr->bSetAccDisplayCntxt(rDiPOVideoContext);
   }

   // Initialize display last mode after system startup
   if (NULL != m_poResourceMngrSettings)
   {
      m_enLastDisplayMode = (true == m_poResourceMngrSettings->bIsDisplayLastModeProjection()) ? e8PROJECTION : e8NATIVE;
      ETG_TRACE_USR4(("[PARAM]:: spi_tclDiPOResourceMngr::vOnPluginLoaded m_enLastDisplayMode = %d", ETG_ENUM(BOOL, m_enLastDisplayMode)));

      //on system startup, set the initial values of the lastmode constraints fetched from datapool
      m_rLastDispTakeContext.enTransferType = e8DIPO_TRANSFERTYPE_TAKE;
      m_rLastDispTakeContext.enBorrowConstraint = e8DIPO_CONSTRAINT_ANYTIME;
      memset(m_rLastDispTakeContext.szBorrowId, 0, MAX_STR_LEN);

      if(e8NATIVE == m_enLastDisplayMode)
      {
         //If the last mode was NATIVE - set as (Take, userinitiated, userinitiated, anytime)
         ETG_TRACE_USR4((" spi_tclDiPOResourceMngr::vOnPluginLoaded - m_rLastDispTakeContext = e8DISPLAY_CONTEXT_NATIVE"));
         m_rLastDispTakeContext.enDisplayContext = e8DISPLAY_CONTEXT_NATIVE;
         m_rLastDispTakeContext.enTransferPriority = e8DIPO_TRANSFERPRIO_USERINITIATED;
         m_rLastDispTakeContext.enTakeConstraint = e8DIPO_CONSTRAINT_USERINITIATED;
      }
      else if(e8PROJECTION == m_enLastDisplayMode)
      {
         //If the last mode was PROJECTION - set as (Take, nicetohave, anytime, anytime)
         ETG_TRACE_USR4((" spi_tclDiPOResourceMngr::vOnPluginLoaded - m_rLastDispTakeContext = e8DISPLAY_CONTEXT_PROJECTION"));
         m_rLastDispTakeContext.enDisplayContext = e8DISPLAY_CONTEXT_PROJECTION;
         m_rLastDispTakeContext.enTransferPriority = e8DIPO_TRANSFERPRIO_NICETOHAVE;
         m_rLastDispTakeContext.enTakeConstraint = e8DIPO_CONSTRAINT_ANYTIME;
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayContext()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayContext(t_Bool bDisplayFlag,
         tenDisplayContext enDisplayContext)
{
   ETG_TRACE_USR1((" spi_tclDiPoVideoResourceMngr::vSetAccessoryDisplayContext entered"));
   ETG_TRACE_USR1(("[DESC]:vSetAccessoryDisplayContext - Display Context = %d", ETG_ENUM(DISPLAY_CONTEXT,
            enDisplayContext)));
   ETG_TRACE_USR1(("[DESC]:vSetAccessoryDisplayContext - Display Flag = %d", ETG_ENUM(BOOL, bDisplayFlag)));

   //!@ Glossary
   // Master Transfer : A transfer which cuases the display resource to transfer from the iPhone to Car or vice versa.
   //                   Possible values : Take, Borrow, NA
   // Master video context: Display context which causes a display resource transfer from iPhone to car or vice versa.
   // Please refer spi_tclDiPOContext.cfg for display context values.
   // Intermediate transfer: A transfer to update the resource access constraint and do no perform a resource transfer.

   //! Resource transfer requested only in any of these cases.
   // 1. The resource is with the mobile(controller).
   // 2. The last requested context is not same with the current one.
   trDiPOVideoContext rDiPOVideoContext;
   t_Bool bStatus = false;
   if (NULL != m_poDipoRsrcMngrIntf)
   {
      bStatus = m_poDipoRsrcMngrIntf->bGetVideoModeChangeMsg(enDisplayContext, bDisplayFlag, rDiPOVideoContext);
   }

   ETG_TRACE_USR4(("[PARAM]:: vSetAccessoryDisplayContext - Prev Display Context m_enCurDispCntxt = %d", ETG_ENUM(DISPLAY_CONTEXT,
            m_enCurDispCntxt)));
   ETG_TRACE_USR4(("[PARAM]:: vSetAccessoryDisplayContext - Prev Display Flag m_bCurDispFlag = %d", ETG_ENUM(BOOL,
            m_bCurDispFlag)));
   ETG_TRACE_USR4(("[PARAM]:: vSetAccessoryDisplayContext - Curr entity m_rCurrModeState.enScreen = %d", m_rCurrModeState.enScreen));

   if (true == bStatus)
   {
      //Store the last received valid display context - NCG3D-112606, NCG3D-112705
      m_bStartupDispFlag = bDisplayFlag;
      m_enStartupDispCntxt = enDisplayContext;
   }

   if ((true == bStatus)
            && ((enDisplayContext != m_enCurDispCntxt) || (bDisplayFlag != m_bCurDispFlag)
                     || (e8DIPO_ENTITY_MOBILE == m_rCurrModeState.enScreen)))
   {
      ETG_TRACE_USR4(("[PARAM]::vSetAccessoryDisplayContext: VideoContext = %d, BorrowConstraint = %d "
               "TakeConstraint = %d, TransferPriority = %d, TransferType = %d, BorrowId = %s", rDiPOVideoContext.enDisplayContext, rDiPOVideoContext.enBorrowConstraint, rDiPOVideoContext.enTakeConstraint, rDiPOVideoContext.enTransferPriority, rDiPOVideoContext.enTransferType, rDiPOVideoContext.szBorrowId));

      if (true == m_bIsLastModeSupported)
      {
         // Store  the last take context to determine the last mode
         if (e8DIPO_TRANSFERTYPE_TAKE == rDiPOVideoContext.enTransferType)
         {
            m_rLastDispTakeContext = rDiPOVideoContext;
         }
         // Check the last mode and send the display context according to it
         vHandleDisplayLastMode(rDiPOVideoContext);
      }

      bStatus = (NULL != m_poDiPOCmdRsrcMngr) ? (m_poDiPOCmdRsrcMngr->bSetAccDisplayCntxt(rDiPOVideoContext)) : (false);

      if (false == bStatus)
      {
         //! Reset the Current display context information,
         // if it failed to send the IPC message to CarPlay plugin.
         m_bCurDispFlag = false;
         m_enCurDispCntxt = e8DISPLAY_CONTEXT_NATIVE;
      }
      else //if(false == bStatus)
      {
         //! Update the latest sent display context to current context.
         m_enCurDispCntxt = enDisplayContext;
         m_bCurDispFlag = bDisplayFlag;
      } // if(false == bStatus)

      m_rCurDispConstraint.enTransferType = rDiPOVideoContext.enTransferType;
      m_rCurDispConstraint.enTransferPriority = rDiPOVideoContext.enTransferPriority;
      m_rCurDispConstraint.enTakeConstraint = rDiPOVideoContext.enTakeConstraint;
      m_rCurDispConstraint.enBorrowConstraint = rDiPOVideoContext.enBorrowConstraint;
      memset(m_rCurDispConstraint.szBorrowId, 0, MAX_STR_LEN);
      strcpy(m_rCurDispConstraint.szBorrowId, rDiPOVideoContext.szBorrowId);

      //activate display off mode as the screen resource is now borrowed from iPhone
      if(e8DISPLAY_CONTEXT_SLEEP_MODE == m_enCurDispCntxt)
      {
         vSetDisplayOffActive(m_bCurDispFlag);
      }
   }
   else //if((true == bStatus) && (NULL != pBuffer))
   {
      ETG_TRACE_ERR(("[ERR] Display Mode Change Request Failed"));
   }
   if ((true == m_bIsRequestUIRequired) && (NULL != m_poDiPOCmdVideo))
   {
      //Send request UI request along with unborrow on disengaging RVC/ on phone call end if last mode was CarPlay
      m_poDiPOCmdVideo->vRequestUI("");
      m_bIsRequestUIRequired = false;
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoVideoResourceMngr::vHandleDisplayLastMode()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vHandleDisplayLastMode(trDiPOVideoContext& rfrDiPOVideoContext)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vHandleDisplayLastMode entered"));

   if (e8DIPO_TRANSFERTYPE_UNBORROW == rfrDiPOVideoContext.enTransferType)
   {
      // If last mode was native and HU had borrowed the display while connecting the device,
      // borrow should be converted to take instead of sending unborrow. It should be unborrow + request UI, in case if the BT Telephony is active with the same device.
      if (e8NATIVE == m_enLastDisplayMode)
      {
         if (true == bIsCallActive())
         {
            ETG_TRACE_USR1(("[DESC]::vHandleDisplayLastMode: "
                     "Last mode is native and phone is connected while BT call was active. Hence send unborrow and then request UI"));
            m_bIsRequestUIRequired = true;
         }
         else
         {
            rfrDiPOVideoContext.enDisplayContext = e8DISPLAY_CONTEXT_NATIVE;
            rfrDiPOVideoContext.enBorrowConstraint = e8DIPO_CONSTRAINT_ANYTIME;
            rfrDiPOVideoContext.enTakeConstraint = e8DIPO_CONSTRAINT_USERINITIATED;
            rfrDiPOVideoContext.enTransferPriority = e8DIPO_TRANSFERPRIO_USERINITIATED;
            rfrDiPOVideoContext.enTransferType = e8DIPO_TRANSFERTYPE_TAKE;
            memset(rfrDiPOVideoContext.szBorrowId, 0, MAX_STR_LEN);
            m_rLastDispTakeContext = rfrDiPOVideoContext;
            ETG_TRACE_USR2(("[DESC]::vHandleDisplayLastMode: Last mode was native and "
                     "HU borrowed display before connecting device. Converting borrow to take."));
         }
      }
      // If last mode was CarPlay and HU had borrowed the display while connecting the device,
      // unborrow and request UI should be sent to phone so that phone takes the display
      else if (e8PROJECTION == m_enLastDisplayMode)
      {
         m_bIsRequestUIRequired = true;
         ETG_TRACE_USR2(("[DESC]::vHandleDisplayLastMode: Last mode was CarPlay and "
                  "HU borrowed display before connecting device. Sending RequestUI along with unborrow"));
      }
   }

   // Reset the last mode value as it is handled and no more required until the end of the session
   if (NULL != m_poDipoRsrcMngrIntf)
   {
      t_Bool bResetDisplayMode = (true == m_poDipoRsrcMngrIntf->bIsSessionActive());
      vUpdateLastDisplayMode(bResetDisplayMode);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoVideoResourceMngr::vVideoPlaybackStatus()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vVideoPlaybackStatus(t_Bool bIsPlaybackStarted)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vVideoPlaybackStatus entered with playback status %d", ETG_ENUM(BOOL,
            bIsPlaybackStarted)));

   m_bIsVideoPlaybackStarted = bIsPlaybackStarted;
   if ((e8DIPO_ENTITY_MOBILE == m_rCurrModeState.enScreen) && (NULL != m_poDipoRsrcMngrIntf))
   {
      m_poDipoRsrcMngrIntf->vPostDisplayContext(true, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN, m_rCurrModeState.enPermScreen);
      ETG_TRACE_USR4(("[DESC]::vVideoPlaybackStatus - Device Display Context Updated with Status =  %d and Context =  %d", ETG_ENUM(BOOL,
               bIsPlaybackStarted), ETG_ENUM(DISPLAY_CONTEXT, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN)));
   }

   // Store the last display context as projection when CarPlay UI is projected
   if (true == bIsPlaybackStarted)
   {
      if (true == m_bIsLastModeSupported)
      {
         m_rLastDispTakeContext.enTransferPriority = e8DIPO_TRANSFERPRIO_NICETOHAVE;
         m_rLastDispTakeContext.enTransferType = e8DIPO_TRANSFERTYPE_TAKE;
      }

      if (NULL != m_poResourceMngrSettings)
      {
         m_poResourceMngrSettings->bWriteDisplayLastMode(m_bIsLastModeSupported);
      }
   }        //if (true == bIsPlaybackStarted)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoVideoResourceMngr::vUpdateLastDisplayMode()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vUpdateLastDisplayMode(t_Bool bIsResetLastMode)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vUpdateLastDisplayMode entered with %d", ETG_ENUM(BOOL,
            bIsResetLastMode)));

   if (true == bIsResetLastMode)
   {
      m_enLastDisplayMode = e8NOT_APPLICABLE;
   }
   else if (e8DIPO_TRANSFERPRIO_USERINITIATED == m_rLastDispTakeContext.enTransferPriority)
   {
      m_enLastDisplayMode = e8NATIVE;
   }
   else if (e8DIPO_TRANSFERPRIO_NICETOHAVE == m_rLastDispTakeContext.enTransferPriority)
   {
      m_enLastDisplayMode = e8PROJECTION;
   }
   ETG_TRACE_USR2(("[PARAM]::vUpdateLastDisplayMode: Last display mode = %d", ETG_ENUM(LASTMODE, m_enLastDisplayMode)));

   if (NULL != m_poDipoRsrcMngrIntf)
   {
      m_poDipoRsrcMngrIntf->vSetDisplayLastMode(m_enLastDisplayMode); //To set video last mode
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vUpdateInitialVideoSettings()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vUpdateInitialVideoSettings(t_Bool bIsLastModeSupported)
{
   m_bIsLastModeSupported = bIsLastModeSupported;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::bIsCallActive(...)
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bIsCallActive()
{
   t_Bool bIsCallActiveBeforeDisableBT = false;
   if (NULL != m_poDipoRsrcMngrIntf)
   {
      bIsCallActiveBeforeDisableBT = m_poDipoRsrcMngrIntf->bGetCallStatus(); //To get the BT call status
   }

   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::bIsCallActive():DisplayContext:%d,WasCallActive:%d,ResourceMngt Handler:%d", ETG_ENUM(DISPLAY_CONTEXT,
            m_enCurDispCntxt), ETG_ENUM(BOOL, bIsCallActiveBeforeDisableBT), ETG_ENUM(DISPLAY_INFO, m_enDisplayInfo)));

   t_Bool bIsCallActive = false;
   if (e8_DISPLAY_CONTEXT == m_enDisplayInfo)
   {
      //Check whether the context is PHONE and whether the Phone call is active with the same device or different device.
      bIsCallActive = ((e8DISPLAY_CONTEXT_PHONE == m_enCurDispCntxt) && (true == bIsCallActiveBeforeDisableBT));
      ETG_TRACE_USR4(("[PARAM]::bIsCallActive():SPI handles resource Management: Call active before disable BT:%d and DisplayContext:%d", ETG_ENUM(BOOL,
               bIsCallActiveBeforeDisableBT), ETG_ENUM(DISPLAY_CONTEXT, m_enCurDispCntxt)));
   }
   else
   {
      //In case if resource mngt is handled by HMI, HMI is always expected to send Unborrow only after the Emergency context gets over for display even if the Telephony is ended.
      //Eg: Use case : RVC active (Borrow,Never)-> BT Phone call active (Borrow,Never) -> Connect BT device for CP -> DisableBT -> BT phone call ends (Unbrrow -> shouldn't be received)
      //-> RVC ends (Un borrow to be sent here).
      ETG_TRACE_USR4(("[PARAM]::bIsCallActive():HMI handles resource Mngt: Call active before disable BT:%d", ETG_ENUM(BOOL,
               bIsCallActiveBeforeDisableBT)));
      bIsCallActive = bIsCallActiveBeforeDisableBT;
   }

   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::bIsCallActive(): Left with call status:%d", ETG_ENUM(BOOL,
            bIsCallActive)));
   return bIsCallActive;
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPoVideoResourceMngr::vOnRequestUI()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vOnRequestUI(t_String szRequestUIURL)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vOnRequestUI() entered"));
   ETG_TRACE_USR2(("Video focus reason received : %s", szRequestUIURL.c_str()));

   // Notify HMI to switch to native screen by sending device display context with display flag as false (NCG3D-52691)
   if (NULL != m_poDipoRsrcMngrIntf)
   {
      if ("oem:back" == szRequestUIURL)
      {
         ETG_TRACE_USR1(("vOnRequestUI: e8DISPLAY_CONTEXT_REASON_LAST_SCREEN entered"));
         m_poDipoRsrcMngrIntf->vPostDisplayContext(false,
                  tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_LAST_SCREEN,
                  m_rCurrModeState.enPermScreen);
      }
      else
      {
         ETG_TRACE_USR1(("vOnRequestUI: e8DISPLAY_CONTEXT_REASON_HOME_SCREEN entered"));
         m_poDipoRsrcMngrIntf->vPostDisplayContext(false,
                  tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_HOME_SCREEN,
                  m_rCurrModeState.enPermScreen);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vUpdateDisplayContext()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vUpdateDisplayContext(const trDiPOModeState &corfrDiPOModeState)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vUpdateDisplayContext entered"));

   t_Bool bRenderStatus = true;        // True to render CarPlay, False otherwise
   m_rCurrModeState = corfrDiPOModeState;

   if ((e8DIPO_ENTITY_CAR == corfrDiPOModeState.enScreen) || (e8DIPO_ENTITY_NA == corfrDiPOModeState.enScreen))
   {
      bRenderStatus = false; // To show native HMI send false to HMI
      m_bIsVideoPlaybackStarted = false;
      if (NULL != m_poDipoRsrcMngrIntf)
      {
         m_poDipoRsrcMngrIntf->vPostDisplayContext(bRenderStatus,
                  tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN,
                  corfrDiPOModeState.enPermScreen);
      }
   }
   else if (e8DIPO_ENTITY_MOBILE == corfrDiPOModeState.enScreen)
   {
      bRenderStatus = true;
      if (true == m_bIsVideoPlaybackStarted)
      {
         //! Always the modeChanged message is expected to come first and then the video play back status.
         //! So this part of the code might not be executed. But extra logic added here to avoid any synchronization issues.
         if (NULL != m_poDipoRsrcMngrIntf)
         {
            m_poDipoRsrcMngrIntf->vPostDisplayContext(bRenderStatus,
                     tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN,
                     corfrDiPOModeState.enPermScreen);
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
         const t_U8 cou8DriveModeRestrictionInfo)
{
   if (NULL != m_poDiPOCmdVideo)
   {
      m_poDiPOCmdVideo->vSetFeatureResrictions(cou8ParkModeRestrictionInfo, cou8DriveModeRestrictionInfo);
   }
}

/********************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vOnSetNativeVideoRenderingStatus()
 ********************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vOnSetNativeVideoRenderingStatus(
         tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
{
   //The message from the Service file is received here. The native display is ON.
   //when the display is ON the message has to reach the video adapter to unlock
   //dispatch message to the IPC
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vOnSetNativeVideoRenderingStatus entered with %d ", ETG_ENUM(NATIVE_VIDEO_RENDERING_STATUS,
            enNativeVideoRenderingStatus)));

   if (NULL != m_poDiPOCmdVideo)
   {
      m_poDiPOCmdVideo->vSetNativeVideoRenderingStatus(enNativeVideoRenderingStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoVideoResourceMngr::vProjVideoRenderingStatus(...
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vProjVideoRenderingStatus(tenProjVideoRenderingStatus enProjRenderingStatus)
{
   if (NULL != m_poDipoRsrcMngrIntf)
   {
      m_poDipoRsrcMngrIntf->vProjVideoRenderingStatus(enProjRenderingStatus);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetResourceMngrSettingsInstance()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetResourceMngrSettingsInstance(
         spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vSetResourceMngrSettingsInstance entered"));
   SPI_NORMAL_ASSERT(NULL == poResourceMngrSettingsIntf);

   m_poResourceMngrSettings = poResourceMngrSettingsIntf;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vUpdateSpmSubState()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vUpdateSpmSubState(const trDiPOModeState & corfoPrevDiPOModeState, const trDiPOModeState & corfoCurrDiPOModeState)
{
   ETG_TRACE_USR2(("spi_tclDiPoVideoResourceMngr::vUpdateSpmSubState - m_enSpmSubState = %d, m_bSubStateData = %d", m_enSpmSubState, ETG_ENUM(BOOL,m_bSubStateData)));

      if ((NULL != m_poDipoRsrcMngrIntf) &&
          (true == bProcessModeStateEvent(corfoPrevDiPOModeState, corfoCurrDiPOModeState)))
      {
         m_poDipoRsrcMngrIntf->vPostDeviceSpmSubState(m_enSpmSubState, m_bSubStateData);
      }

      if(e8SPM_SUBSTATE_PERMANENT == m_enSpmSubState)
      {
         //once the state has been changed to permanent state,
         //system will wake up permanently until HMI state is changed explicitly to user off mode
         vSetDisplayOffActive(false);
      }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPoVideoResourceMngr::bSiriActivated()
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent(const trDiPOModeState & corfoPrevDiPOModeState, const trDiPOModeState & corfoCurrDiPOModeState)
{
   //check if any of the state in modesChanged message has a delta change
   t_Bool bStateChanged =  ((corfoPrevDiPOModeState.enPermScreen != corfoCurrDiPOModeState.enPermScreen) ||
         (corfoPrevDiPOModeState.enScreen != corfoCurrDiPOModeState.enScreen) ||
         (corfoPrevDiPOModeState.rSpeechState.enSpeechMode != corfoCurrDiPOModeState.rSpeechState.enSpeechMode) ||
         (corfoPrevDiPOModeState.enPhone != corfoCurrDiPOModeState.enPhone) ||
         (corfoPrevDiPOModeState.enNavigation != corfoCurrDiPOModeState.enNavigation));

   if(true == bStateChanged)
   {
      //This is to check the CP phone call state updates
      t_Bool bPrevAppStatePhone = (e8DIPO_ENTITY_MOBILE == corfoPrevDiPOModeState.enPhone) ? true : false;
      t_Bool bCurrAppStatePhone = (e8DIPO_ENTITY_MOBILE == corfoCurrDiPOModeState.enPhone) ? true : false;

      switch(corfoCurrDiPOModeState.enPermScreen)
      {
         case e8DIPO_ENTITY_MOBILE:
         {
            //For permanent entity change to device, SPM state will also change to e8SPM_SUBSTATE_PERMANENT
            //Examples are - starting CP music, CP navigation
            ETG_TRACE_USR2(("spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent - permanent state"));

            //state to be triggered only once entity is taken by device
            if(e8DIPO_ENTITY_MOBILE == corfoCurrDiPOModeState.enScreen)
            {
               vSetPermSpmSubState(true);
            }
         }
         break;

         case e8DIPO_ENTITY_CAR:
         {
            if(bPrevAppStatePhone != bCurrAppStatePhone)
            {
               //This is purely for CP phone call transition.
               //either CP phone call started or ended, the respective flag is set as TRUE/FALSE for SPM state e8SPM_SUBSTATE_TEMPORARY
               ETG_TRACE_USR2(("spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent - phone call app state updated while temp entity is active"));
               vSetTempSpmSubState(bCurrAppStatePhone);
            }
            else if(bSiriActivated(corfoPrevDiPOModeState, corfoCurrDiPOModeState) || bTempVideoResWithDevice(corfoPrevDiPOModeState, corfoCurrDiPOModeState))
            {
               //Either Siri is activated or temporary resource change (i.e; SMS),
               //the SPM substate to be set as e8SPM_SUBSTATE_TEMPORARY with flag = TRUE (to initiate spm state change)
               ETG_TRACE_USR2(("spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent - siri activated or temp state changed"));
               vSetTempSpmSubState(true);
            }
            else if(bStateRestored(corfoPrevDiPOModeState, corfoCurrDiPOModeState))
            {
               //Restored back to the initial state
               //Now, we end the transition state by resetting substate flag as FALSE
               ETG_TRACE_USR2(("spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent - state restored"));
               vSetTempSpmSubState(false);
            }
            else
            {
               //There are no final state and spm state not to be updated
               bStateChanged = false;
               ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::bProcessModeStateEvent - no action"));
            }
         }
         break;
         default:
         {
            //permanent entity is N/A and hence ignore sending SPM updates
            bStateChanged = false;
            break;
         }
      }
   }
   return bStateChanged;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPoVideoResourceMngr::bSiriActivated()
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bSiriActivated(const trDiPOModeState & corfoPrevDiPOModeState, const trDiPOModeState & corfoCurrDiPOModeState)
{
   t_Bool bStatus = ((e8DIPO_SPEECHMODE_NONE == corfoPrevDiPOModeState.rSpeechState.enSpeechMode) &&
         (e8DIPO_SPEECHMODE_NONE != corfoCurrDiPOModeState.rSpeechState.enSpeechMode) &&
         (corfoPrevDiPOModeState.enPermScreen == corfoCurrDiPOModeState.enPermScreen) &&
         (corfoPrevDiPOModeState.enPhone == corfoCurrDiPOModeState.enPhone));
   return bStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPoVideoResourceMngr::bTempVideoResWithDevice()
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bTempVideoResWithDevice(const trDiPOModeState & corfoPrevDiPOModeState, const trDiPOModeState & corfoCurrDiPOModeState)
{
   t_Bool bStatus = ((corfoPrevDiPOModeState.enScreen != corfoCurrDiPOModeState.enScreen) &&
         (corfoPrevDiPOModeState.enPermScreen == corfoCurrDiPOModeState.enPermScreen) &&
         (e8DIPO_ENTITY_NA == corfoCurrDiPOModeState.enPhone) &&
         (e8DIPO_SPEECHMODE_NONE == corfoCurrDiPOModeState.rSpeechState.enSpeechMode) &&
         (e8DIPO_ENTITY_MOBILE == corfoCurrDiPOModeState.enScreen));
   return bStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPoVideoResourceMngr::bStateRestored()
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bStateRestored(const trDiPOModeState & corfoPrevDiPOModeState, const trDiPOModeState & corfoCurrDiPOModeState)
{
   t_Bool bStatus = ((corfoPrevDiPOModeState.enPermScreen == corfoCurrDiPOModeState.enPermScreen) &&
         (e8DIPO_ENTITY_NA == corfoCurrDiPOModeState.enPhone) &&
         (e8DIPO_SPEECHMODE_NONE == corfoCurrDiPOModeState.rSpeechState.enSpeechMode) &&
         (e8DIPO_ENTITY_CAR == corfoCurrDiPOModeState.enScreen));
   return bStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::bIsDisplayOffActive()
 ***************************************************************************/
t_Bool spi_tclDiPoVideoResourceMngr::bIsDisplayOffActive()
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::bIsDisplayOffActive entered - m_bDisplayOffActive=%d", m_bDisplayOffActive));
   return m_bDisplayOffActive;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetDisplayOffActive()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetDisplayOffActive(t_Bool bIsDisplayOffActive)
{
   ETG_TRACE_USR1(("spi_tclDiPoVideoResourceMngr::vSetDisplayOffActive bIsDisplayOffActive=%d", bIsDisplayOffActive));

   //pre-condition - If user off mode was already active
   //whenever DisplayOff is deactivated, reset the display flag as false,
   //so that next time when user enters User Off mode, setAccessoryDispCtx will be send
   if((true == m_bDisplayOffActive) && (false == bIsDisplayOffActive))
   {
      m_bCurDispFlag = false;
   }

   m_bDisplayOffActive = bIsDisplayOffActive;

   //reset all LCM states whenever starting user off or ending user off mode
   m_enSpmSubState = e8SPM_SUBSTATE_NOT_KNOWN;
   m_bSubStateData = false;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetTempSpmSubState()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetTempSpmSubState(t_Bool bSpmSubStateDataEnable)
{
   m_enSpmSubState = e8SPM_SUBSTATE_TEMPORARY;
   m_bSubStateData = bSpmSubStateDataEnable;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoVideoResourceMngr::vSetPermSpmSubState()
 ***************************************************************************/
t_Void spi_tclDiPoVideoResourceMngr::vSetPermSpmSubState(t_Bool bSpmSubStateDataEnable)
{
   m_enSpmSubState = e8SPM_SUBSTATE_PERMANENT;
   m_bSubStateData = bSpmSubStateDataEnable;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
