/***********************************************************************/
/*!
 * \file  spi_tclDiPoAudioResourceMngr.cpp
 * \brief Manages DiPo Audio resource
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Manages DiPo Audio resource
 AUTHOR:         Pruthvi Thej Nagaraju
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 20.08.2014  | Pruthvi Thej Nagaraju | Initial Version 
 02.03.2015  | Ramya Murthy          | Added trigger for media release on Siri/Phone/Alert prepare 
                                       (Fix for GMMY16-25651)
06.05.2015   | Tejaswini HB          | Lint Fix             
 02.12.2016  | Ramya Murthy          | Changes to cleanup audio on SessionEnd or DeviceDeselection 
                                       (Fix for NCG3D-8442) 
 24.02.2016  | Ramya Murthy          | Implementation of default audio type
 22.08.2018  | Chaitra Srinivasa     | Added audio related functionality from DipoResourceMngr

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclDiPoAudioResourceMngr.cpp.trc.h"
#endif

#include "spi_tclDiPoAudioResourceMngr.h"
#include "spi_tclDiPOManager.h"
#include "DiPOWrapperTypes.h"
#include "spi_tclDiPOResourceMngr.h"
#include "spi_tclDiPOCmdRsrcMngr.h"
#include "spi_tclDipoResourceMngrIntf.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
static trDiPOAudioContext srLastModeCPAudCtxt;
static trDiPOAudioContext srLastModeNativeAudCtxt;
 
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudioResourceMngr::spi_tclDiPoAudioResourceMngr
 ***************************************************************************/
spi_tclDiPoAudioResourceMngr::spi_tclDiPoAudioResourceMngr(spi_tclDipoResourceMngrIntf* poDipoResrcMngrIntf):
         m_enCurAudioSource(e8SPI_AUDIO_MAIN),
         m_bCurAudioFlag(false),
         m_bIsLastModeSupported(false),
         m_poDipoRsrcMngrIntf(poDipoResrcMngrIntf),
         m_enLastAudioMode(e8NOT_APPLICABLE),
         m_poResourceMngrSettings(NULL)
{
   m_rLastAudioTakeContext.enAudioContext = e8SPI_AUDIO_UNKNOWN;
   m_rLastAudioTakeContext.enTransferType = e8DIPO_TRANSFERTYPE_NA;
   m_rLastAudioTakeContext.enTransferPriority = e8DIPO_TRANSFERPRIO_NA;
   m_rLastAudioTakeContext.enTakeConstraint = e8DIPO_CONSTRAINT_NA;
   m_rLastAudioTakeContext.enBorrowConstraint = e8DIPO_CONSTRAINT_NA;
   memset(m_rLastAudioTakeContext.szBorrowId, 0, MAX_STR_LEN);
   
   srLastModeCPAudCtxt.enAudioContext = e8SPI_AUDIO_MAIN;
   srLastModeCPAudCtxt.enBorrowConstraint = e8DIPO_CONSTRAINT_ANYTIME;
   srLastModeCPAudCtxt.enTakeConstraint = e8DIPO_CONSTRAINT_ANYTIME;
   srLastModeCPAudCtxt.enTransferPriority = e8DIPO_TRANSFERPRIO_NICETOHAVE;
   srLastModeCPAudCtxt.enTransferType = e8DIPO_TRANSFERTYPE_TAKE; 
   memset(srLastModeCPAudCtxt.szBorrowId, 0, MAX_STR_LEN);

   srLastModeNativeAudCtxt.enAudioContext = e8SPI_AUDIO_MAIN;
   srLastModeNativeAudCtxt.enBorrowConstraint = e8DIPO_CONSTRAINT_ANYTIME;
   srLastModeNativeAudCtxt.enTakeConstraint = e8DIPO_CONSTRAINT_USERINITIATED;
   srLastModeNativeAudCtxt.enTransferPriority = e8DIPO_TRANSFERPRIO_USERINITIATED;
   srLastModeNativeAudCtxt.enTransferType = e8DIPO_TRANSFERTYPE_TAKE; 
   memset(srLastModeNativeAudCtxt.szBorrowId, 0, MAX_STR_LEN);
   
   spi_tclDiPOManager* poDiPOManager = spi_tclDiPOManager::getInstance();
   SPI_NORMAL_ASSERT( NULL == poDiPOManager);
   SPI_NORMAL_ASSERT( NULL == m_poDipoRsrcMngrIntf);

   if (NULL != poDiPOManager)
   {
      m_poDiPOCmdRsrcMngr = poDiPOManager->poGetDiPORsrcMngrInstance();
      poDiPOManager->bRegisterObject((spi_tclDiPORespAudio*) this);
   }      

   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
     pRespRegister->bRegisterObject((spi_tclExtRespMediaDevInfo*)this);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudioResourceMngr::~spi_tclDiPoAudioResourceMngr
 ***************************************************************************/
spi_tclDiPoAudioResourceMngr::~spi_tclDiPoAudioResourceMngr()
{
   m_mapBTMediaDevInfo.clear();
   m_poResourceMngrSettings = NULL;
   m_poDipoRsrcMngrIntf = NULL;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vOnSPISelectDeviceResult(
      t_U32 u32DevID, tenDeviceConnectionReq enDeviceConnReq,
      tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   SPI_INTENTIONALLY_UNUSED(u32DevID);
   SPI_INTENTIONALLY_UNUSED(enDeviceConnReq);
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);
   if(true == m_bIsLastModeSupported)
   {
      vUpdateLastAudioMode(false);
   }
}//! end of spi_tclDiPoAudio::vOnSPISelectDeviceResult()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vUpdateInitialAudioSettings()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vUpdateInitialAudioSettings(t_Bool bIsLastModeSupported)
{
    m_bIsLastModeSupported = bIsLastModeSupported;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnAudioAllocateRequest(...
 ***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
         tenDiPOMainAudioType enAudioType,
         tenAudioReqType enAudioReqType,
         const trDiPOAudioFormat& corfrDiPOAudioFormat)
{
   ETG_TRACE_USR4(("[PARAM]::vOnAudioAllocateRequest:AudioChannelType:%d,AudioType:%d,AudioRequestType:%d",
   ETG_ENUM(DIPO_AUDIO_STREAMTYPE,enAudioChannelType),ETG_ENUM(DIPO_MAIN_AUD_TYPE,enAudioType),ETG_ENUM(AUDIO_REQUEST_TYPE,enAudioReqType)));
   SPI_INTENTIONALLY_UNUSED(corfrDiPOAudioFormat);
   
   if ((true == m_bIsLastModeSupported) &&
            (AudioChannelType_Main == (t_U8) enAudioChannelType) &&
            (e8AUDIO_MEDIA == enAudioType) &&
            (e8ALLOCATION_REQUEST == enAudioReqType))
   {
      vStoreLastAudioTakeContext(srLastModeCPAudCtxt);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vSetAccessoryAudioContext(const tenAudioContext coenAudioCntxt, 
                                                               t_Bool bReqFlag)
{
    ETG_TRACE_USR1((" spi_tclDiPoAudioResourceMngr::vSetAccessoryAudioContext entered"));
    ETG_TRACE_USR4(("[PARAM]::vSetAccessoryAudioContext - Audio Context = %d", ETG_ENUM(AUDIO_CONTEXT, coenAudioCntxt)));
    ETG_TRACE_USR4(("[PARAM]::vSetAccessoryAudioContext - Audio Flag = %d", ETG_ENUM(BOOL, bReqFlag)));
    ETG_TRACE_USR4(("[PARAM]::vSetAccessoryAudioContext - Current Audio Context = %d", ETG_ENUM(AUDIO_CONTEXT, m_enCurAudioSource)));
    ETG_TRACE_USR4(("[PARAM]::vSetAccessoryAudioContext - Current Audio Flag = %d", m_bCurAudioFlag));

    //! Resource transfer requested only in any of these cases.
    // 1. The resource is with the mobile(controller).
    // 2. The last requested context is not same with the current one.

    trDiPOAudioContext rDiPOAudioContext;
    t_Bool bStatus = false;
    t_Bool bIsAudCtxRequested = false;
    if(NULL != m_poDipoRsrcMngrIntf)
    {        
        bStatus = m_poDipoRsrcMngrIntf->bGetAudioModechangeMsg(coenAudioCntxt, bReqFlag, rDiPOAudioContext);
    }  
    ETG_TRACE_USR4(("[PARAM]::vSetAccessoryAudioContext - AudioContext message: AudioContext=%d, BorrowConstraint = %d"
    "TakeConstraint = %d, TransferPriority = %d, TransferType =%d, BorrowId = %s",
    ETG_ENUM(AUDIO_CONTEXT, rDiPOAudioContext.enAudioContext),
    ETG_ENUM(DIPO_CONSTRAINT, rDiPOAudioContext.enBorrowConstraint),
    ETG_ENUM(DIPO_CONSTRAINT, rDiPOAudioContext.enTakeConstraint),
    ETG_ENUM(DIPO_TRANSFERPRIO, rDiPOAudioContext.enTransferPriority),
    ETG_ENUM(DIPO_TRANSFERTYPE, rDiPOAudioContext.enTransferType),
    rDiPOAudioContext.szBorrowId));
    
    ETG_TRACE_USR4(("[PARAM]:vSetAccessoryAudioContext - Current Mode State = %d", m_rCurrModeState.enAudio));
    
    if ((true == bStatus) &&
        ((coenAudioCntxt != m_enCurAudioSource) || (bReqFlag != m_bCurAudioFlag) || (e8DIPO_ENTITY_CAR != m_rCurrModeState.enAudio)))
    {
        if (true == m_bIsLastModeSupported)
        {
            // Store the last take context to determine the last mode
            if (e8DIPO_TRANSFERTYPE_TAKE == rDiPOAudioContext.enTransferType)
            {
                vStoreLastAudioTakeContext(rDiPOAudioContext);
            }
            // Check the last mode and send the audio context according to it
            vHandleAudioLastMode(rDiPOAudioContext);
        }

        //if set accessory audio context is send by the HMI as take/borrow, the audio is explicitly requested by head unit and the flag is updated TRUE
        if((e8DIPO_TRANSFERTYPE_TAKE == rDiPOAudioContext.enTransferType) || (e8DIPO_TRANSFERTYPE_BORROW == rDiPOAudioContext.enTransferType))
        {
            //If the transition happened from CAR->MOBILE, it means headunit has requested and the flag is set to track the request
            bIsAudCtxRequested = (m_rCurrModeState.enAudio == e8DIPO_ENTITY_MOBILE) ? true : false;
        }
        if(NULL != m_poDiPOCmdRsrcMngr)
        {
            bStatus = m_poDiPOCmdRsrcMngr->bSetAccAudioCntxt(rDiPOAudioContext);
            m_poDiPOCmdRsrcMngr->vSetAudioCtxtRequestedbyCar(bIsAudCtxRequested);
        }
    
        if (false == bStatus)
        {
            //!The missed updates during startup time will be kept in a queue
            // and are sent to the carplay deamon later. This is important
            // to process the subsequent requests in proper way.
            trAudioContextInfo rAudioContextInfo;
            rAudioContextInfo.bAudioFlag = bReqFlag;
            rAudioContextInfo.enAudioContext = coenAudioCntxt;
            m_rAudioContextInfo.push(rAudioContextInfo);
            
            //! Reset the Current audio context information,
            // if it failed to send the IPC message to CarPlay plugin.
            m_bCurAudioFlag = false;
            m_enCurAudioSource = e8SPI_AUDIO_MAIN;
        }
        else //if(false == bStatus)
        {
            //! Update the latest sent display context to current context.
            m_enCurAudioSource = coenAudioCntxt;
            m_bCurAudioFlag = bReqFlag;
        }
    }
    else //if((true == bStatus) && (NULL != pBuffer))
    {
        ETG_TRACE_ERR(("Mode change request failed"));
    }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vUpdateAudioContext()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vUpdateAudioContext(const trDiPOModeState &corfoDiPOModeState)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::bUpdateAudioContext entered"));

   t_Bool bPlayStatus = true;
  
   m_rCurrModeState = corfoDiPOModeState; // For Audio response handling
   if ((e8DIPO_ENTITY_CAR == corfoDiPOModeState.enAudio) || (e8DIPO_ENTITY_NA == corfoDiPOModeState.enAudio))
   {
      bPlayStatus = false; //To play native audio send false to HMI
   }
   else if (e8DIPO_ENTITY_MOBILE == corfoDiPOModeState.enAudio)
   {
      bPlayStatus = true; // To play CarPlay audio send true to HMI
   }
   if(NULL != m_poDipoRsrcMngrIntf)
   {      
       m_poDipoRsrcMngrIntf->vPostAudioContext(bPlayStatus,e8SPI_AUDIO_MAIN,corfoDiPOModeState.enPermMainAudio);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vUpdateLastAudioMode()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vUpdateLastAudioMode(t_Bool bIsResetLastMode)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vUpdateLastAudioMode entered with %d",ETG_ENUM(BOOL,bIsResetLastMode)));
   ETG_TRACE_USR4(("[PARAM]::vUpdateLastAudioMode: LastAudioTakeContext - TransferPriority %d",
            ETG_ENUM(DIPO_TRANSFERPRIO,m_rLastAudioTakeContext.enTransferPriority)));
   
   if (true == bIsResetLastMode)
   {
      m_enLastAudioMode = e8NOT_APPLICABLE;
   }
   else if (e8DIPO_TRANSFERPRIO_USERINITIATED == m_rLastAudioTakeContext.enTransferPriority)
   {
      m_enLastAudioMode = e8NATIVE;
   }
   else if (e8DIPO_TRANSFERPRIO_NICETOHAVE == m_rLastAudioTakeContext.enTransferPriority)
   {
      m_enLastAudioMode = e8PROJECTION;
   }
   ETG_TRACE_USR4(("[PARAM]::vUpdateLastAudioMode: Last audio mode = %d", ETG_ENUM(LASTMODE, m_enLastAudioMode)));

   if(NULL != m_poDipoRsrcMngrIntf)
   {
       m_poDipoRsrcMngrIntf->vSetAudioLastMode(m_enLastAudioMode);   //To set the audio last mode
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vStoreLastAudioTakeContext()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vStoreLastAudioTakeContext(const trDiPOAudioContext& corfrDiPOAudioContext)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vStoreLastAudioTakeContext entered with: TransferPriority %d",
         ETG_ENUM(DIPO_TRANSFERPRIO, corfrDiPOAudioContext.enTransferPriority)));

   m_rLastAudioTakeContext = corfrDiPOAudioContext;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vHandleAudioLastMode()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vHandleAudioLastMode(trDiPOAudioContext& rfrDiPOAudioContext)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vHandleAudioLastMode entered"));
   SPI_INTENTIONALLY_UNUSED(rfrDiPOAudioContext);

   //RTC-788829, NCG3D-165849 : FACETS-41BDE when Native VR ends audCtx(Take, userinit, userinit, anytime)
   if (e8DIPO_TRANSFERTYPE_UNBORROW == rfrDiPOAudioContext.enTransferType)
   {
      t_Bool bIsCallActiveBeforeDisableBT = false;
      if (NULL != m_poDipoRsrcMngrIntf)
      {
         bIsCallActiveBeforeDisableBT = m_poDipoRsrcMngrIntf->bGetCallStatus(); //To get the BT call status
      }
      //If B/T phone call was already active while connecting iPhone, send unborrow
      if ((e8SPI_AUDIO_PHONE == m_enCurAudioSource) && (true == bIsCallActiveBeforeDisableBT))
      {
         ETG_TRACE_USR2(("spi_tclDiPoAudioResourceMngr::vHandleAudioLastMode - current aud ctx is phone and call was active, unborrow to be send"));
      }
      else
      {
         //depending on last mode, update change mode command by sending (Take, userinit/nicetohave)
         if(e8NATIVE == m_enLastAudioMode)
         {
            rfrDiPOAudioContext = srLastModeNativeAudCtxt;
            vStoreLastAudioTakeContext(srLastModeNativeAudCtxt);
         }
         else if(e8PROJECTION == m_enLastAudioMode)
         {
            rfrDiPOAudioContext = srLastModeCPAudCtxt;
            vStoreLastAudioTakeContext(srLastModeCPAudCtxt);
         }
      }
   }

   // Reset the last mode value as it is handled and no more required until the end of the session
   if(NULL != m_poDipoRsrcMngrIntf)
   {
       t_Bool bResetAudioMode = (true == m_poDipoRsrcMngrIntf->bIsSessionActive());
       vUpdateLastAudioMode(bResetAudioMode);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vOnPluginLoaded()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vOnPluginLoaded()
{
    while (!m_rAudioContextInfo.empty())
    {
       trAudioContextInfo rAudioContextInfo = m_rAudioContextInfo.front();
       m_rAudioContextInfo.pop();
       vSetAccessoryAudioContext(rAudioContextInfo.enAudioContext,
                rAudioContextInfo.bAudioFlag);
    }

   // Initialize display last mode after system startup
   if (NULL != m_poResourceMngrSettings)
   {
      m_enLastAudioMode = (true == m_poResourceMngrSettings->bIsAudioLastModeProjection()) ? e8PROJECTION : e8NATIVE;
      ETG_TRACE_USR4(("[PARAM]:: spi_tclDiPOResourceMngr::vOnPluginLoaded m_enLastAudioMode = %d", ETG_ENUM(BOOL, m_enLastAudioMode)));

      //on system startup, set the initial values of the lastmode constraints fetched from datapool
      if(e8NATIVE == m_enLastAudioMode)
      {
         //If the last mode was NATIVE - set as (Take, userinitiated, userinitiated, anytime)
         ETG_TRACE_USR4(("spi_tclDiPOResourceMngr::vOnPluginLoaded - set e8NATIVE"));
         vStoreLastAudioTakeContext(srLastModeNativeAudCtxt);
      }
      else if(e8PROJECTION == m_enLastAudioMode)
      {
         //If the last mode was PROJECTION - set as (Take, nicetohave, anytime, anytime)
         ETG_TRACE_USR4(("spi_tclDiPOResourceMngr::vOnPluginLoaded - set e8PROJECTION"));
         vStoreLastAudioTakeContext(srLastModeCPAudCtxt);
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vPostBTMediaDeviceInfo()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vPostBTMediaDeviceInfo(const trMediaDeviceInfo& corfrMediaDeviceInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vPostBTMediaDeviceInfo entered"));

   m_mapBTMediaDevInfoLock.s16Lock();
   m_mapBTMediaDevInfo[corfrMediaDeviceInfo.u32DevID] = corfrMediaDeviceInfo;
   m_mapBTMediaDevInfoLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: Static  t_Void spi_tclDiPoAudioResourceMngr::vOnEvaluateInitalModes()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vOnEvaluateInitalModes(const t_String& corfszBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vOnEvaluateInitalModes entered"));
   //! For CPW enabled HU: BT media last mode can be detected only if Resource manager gets Select request
   //! before BT is disconnected (i.e. before BT Limitation Preactivate)
   //! For CPW disabled HU: BT media last mode can be detected only after device appears on Bonjour (BTAddress needed)

   //! The implementation was done to handle seamless audio transition from BT media to CarPlay media when the same BT device is connected as CarPlay.
   //! We were checking if the BT media is in playing state using the Mediaplayer interface, which is not working in case of Source change from BT media to native media or when the BT media is in paused state.
   //! There is another interface named GetA2DPStreamingStatus available from BT component. However, we have observed issues with this interface when using some phones like LG.
   //! Also, Apple have agreed for sending Take, User-initiated, User-initiated, Anytime in Info Response message in this case, as they are already handling seamless transition from BT media to CarPlay media in the latest iOS versions (13.2)
   //! Hence, removing the implementation done to send Take, Nice-to-Have, Anytime, Anytime when same BT device is connected as CarPlay with active BT media.
#if 0
   if (true == m_bIsLastModeSupported)
   {
      t_Bool bIsBTMediaActiveWithCPDevice = false;
      m_mapBTMediaDevInfoLock.s16Lock();
      for (const auto& itrBTMediaDevInfo: m_mapBTMediaDevInfo)
      {
         if(corfszBTAddress == itrBTMediaDevInfo.second.szBTMACAddress)
         {
            bIsBTMediaActiveWithCPDevice = itrBTMediaDevInfo.second.bIsMediaPlaying;
            ETG_TRACE_USR2(("spi_tclDiPoAudioResourceMngr::vCheckBTMediaLastMode: found match with playing state = %d",
                     ETG_ENUM(BOOL, itrBTMediaDevInfo.second.bIsMediaPlaying)));

            if (true == bIsBTMediaActiveWithCPDevice)
               break;
         }
      }
      m_mapBTMediaDevInfoLock.vUnlock();
      ETG_TRACE_USR4(("spi_tclDiPoAudioResourceMngr::vOnEvaluateInitalModes: bIsBTMediaActiveWithCPDevice = %d",
               ETG_ENUM(BOOL, bIsBTMediaActiveWithCPDevice)));

      //! Value stored in configreader/datapool will be read by ConnMgr and sent to carplayd during selection sequence
      if ((true == bIsBTMediaActiveWithCPDevice) && (NULL != m_poResourceMngrSettings))
      {
         m_enLastAudioMode = e8PROJECTION;
         vStoreLastAudioTakeContext(srLastModeCPAudCtxt);
         m_poResourceMngrSettings->bWriteAudioLastMode(true);
      }
   }
#endif
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vSetResourceMngrSettingsInstance()
***************************************************************************/
t_Void spi_tclDiPoAudioResourceMngr::vSetResourceMngrSettingsInstance(spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudioResourceMngr::vSetResourceMngrSettingsInstance entered"));
   SPI_NORMAL_ASSERT(NULL == poResourceMngrSettingsIntf);

   m_poResourceMngrSettings = poResourceMngrSettingsIntf;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
