/*!
 *******************************************************************************
 * \file              spi_tclDiPOResourceMngr.cpp
 * \brief             DiPO Resource manager
 *******************************************************************************
 \verbatim
 PROJECT:        G3G
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO Resource manager implementation
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 24.03.2014 |  Shihabudheen P M            | Initial Version
 05.04.2014 |  Priju K Padiyath            | Modified to handle display & audio context
 14.04.2014 |  Shihabudheen P M            | Modified vSetAccessoryDisplayContext()
 19.04.2014 |  Shihabudheen P M            | Implemented CarPlay resource arbitration
 25.06.2014 |  Shihabudheen P M            | Adapted to the CarPlay design changes
 27.08.2014 |  Shihabudheen P M            | Changes to reset all resource arbitration parameters.
 16.12.2014 |  Shihabudheen P M            | Changed resource transfer requests.
 11.02.2015 |  Shihabudheen P M            | Added timer implementation
 06.05.2015 |  Tejaswini HB                  |Lint Fix
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)    | Added Lint comments to suppress C++11 Errors
 15.06.2015 |  Shihabudheen P M            | added vSetVehicleBTAddress.
 24.11.2016 |  Noopur R K                  | Added method for HMI synchronization
 01.03.2017 |  Shiva Kumar G               | Removed unused includes & created DiPoMsgReceiver
 07.06.2016 |  Ramya Murthy                | Handling transient audio cases for last mode
 17.08.2018 |  Anjan kumar sahu            | Removed Audio and Video related resource management
 22.08.2018 |  Chaitra Srinivasa           | Removed audio/video related functionality and
 added functions to set and get generic member variables

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "RespRegister.h"
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclResourceArbitrator.h"
#include "spi_tclMediator.h"
#include "spi_tclDiPoAudioResourceMngr.h"
#include "spi_tclDiPoVideoResourceMngr.h"

#include "DiPOWrapperTypes.h"
#include "spi_tclDiPOCmdBluetooth.h"
#include "spi_tclDiPOCmdRsrcMngr.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPOCmdSession.h"
#include "spi_tclExtCmdBluetooth.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclDeviceIDDataIntf.h"

#include "spi_tclDiPOResourceMngr.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclDiPOResourceMngr.cpp.trc.h"
#endif

static t_String szBtMacAddress = "";
static const t_U8 su8ETCEnabledMask = 0x01;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported  

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
static spi_tclDiPOManager* spoDiPOManager = NULL;
static spi_tclDiPOCmdBluetooth* spoDiPOCmdBluetooth = NULL;
static spi_tclDiPOCmdRsrcMngr* spoDiPOCmdRsrcMngr = NULL;

/***************************************************************************
 ** FUNCTION: spi_tclDiPOResourceMngr::spi_tclDiPOResourceMngr()
 ***************************************************************************/
spi_tclDiPOResourceMngr::spi_tclDiPOResourceMngr() :
                  m_poResArb(NULL),
                  m_u32DevId(0),
                  m_u8DriveModeRestrictionInfo(0),
                  m_enETCSupport(e8ETCInfo_NotSupported),
                  m_enLastDisplayMode(e8NOT_APPLICABLE),
                  m_enLastAudioMode(e8NOT_APPLICABLE),
                  m_bIsSessionActive(false),
                  m_bIsLastModeSupported(false),
                  m_poAudioResourceMngr(NULL),
                  m_poVideoResourceMngr(NULL),
                  m_bIsCallActiveBeforeDisableBT(false)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::spi_tclDiPOResourceMngr()"));

   m_poResArb = new spi_tclResourceArbitrator();
   SPI_NORMAL_ASSERT(NULL == m_poResArb);

   m_rDiPOAppState.enNavAppState = e8SPI_NAV_UNKNOWN;
   m_rDiPOAppState.enPhoneAppState = e8SPI_PHONE_UNKNOWN;
   m_rDiPOAppState.enSpeechAppState = e8SPI_SPEECH_UNKNOWN;

   m_poAudioResourceMngr = new spi_tclDiPoAudioResourceMngr(this);
   SPI_NORMAL_ASSERT(NULL == m_poAudioResourceMngr);
   m_poVideoResourceMngr = new spi_tclDiPoVideoResourceMngr(this);
   SPI_NORMAL_ASSERT(NULL == m_poVideoResourceMngr);

   spoDiPOManager = spi_tclDiPOManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == spoDiPOManager);

   if (NULL != spoDiPOManager)
   {
      spoDiPOCmdBluetooth = spoDiPOManager->poGetDiPOBluetoothInstance();
      spoDiPOCmdRsrcMngr = spoDiPOManager->poGetDiPORsrcMngrInstance();

      spoDiPOManager->bRegisterObject((spi_tclDiPORespSession*) this);
      spoDiPOManager->bRegisterObject((spi_tclDiPORespVideo*) this);
      spoDiPOManager->bRegisterObject((spi_tclDiPORespRsrcMngr*) this);
      spoDiPOManager->bRegisterObject((spi_tclDiPORespAudio*) this);

   } //if(NULL != spoDiPOManager )
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPOResourceMngr::~spi_tclDiPOResourceMngr()
 ***************************************************************************/
spi_tclDiPOResourceMngr::~spi_tclDiPOResourceMngr()
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::~spi_tclDiPOResourceMngr()"));

   RELEASE_MEM(m_poResArb);
   RELEASE_MEM(m_poAudioResourceMngr);
   RELEASE_MEM(m_poVideoResourceMngr);
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPOResourceMngr::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSelectDevice(const t_U32 cou32DevId, tenDeviceConnectionReq enConnReq,
         tenDeviceCategory enDevCat)
{
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSelectDevice entered"));

   if (e8DEVCONNREQ_SELECT == enConnReq)
   {
      m_u32DevId = cou32DevId;

      t_String szBTMACAddress;
      spi_tclDeviceIDDataIntf oDataIntf;
      oDataIntf.vGetBTAddress(szBTMACAddress, cou32DevId);
      vOnEvaluateInitalModes(szBTMACAddress);
   }
   else
   {
      m_u32DevId = 0;
      m_bIsSessionActive = false;
      vHandleSessionEnd();
   }

   //@TODO - check whether to wait till the session end message is received to respond to Deselect request. Or this shall be handled in DiPOConnection.
   if (NULL != (m_rRsrcMngrCallback.fvSelectDeviceResult))
   {
      //! Always send the true response
      (m_rRsrcMngrCallback.fvSelectDeviceResult)(true);
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPOResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDeviceConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   /*lint -esym(40,fvUpdateSessionStatus)fvUpdateSessionStatus Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnSPISelectDeviceResult entered"));

   if ((e8DEVCONNREQ_SELECT == enDeviceConnReq) && (e8SUCCESS == enRespCode))
   {
      m_u32DevId = u32DevID; //TODO - required? It is already set on select request
   }

   if ((NULL != m_rRsrcMngrCallback.fvUpdateSessionStatus) && (enDeviceConnReq == e8DEVCONNREQ_DESELECT))
   {
      ETG_TRACE_USR2(("[DESC]::vOnSPISelectDeviceResult: Posting the session status info with device id %d", u32DevID));
      (m_rRsrcMngrCallback.fvUpdateSessionStatus)(u32DevID, e8DEV_TYPE_DIPO, e8_SESSION_INACTIVE);
   }

   if (NULL != m_poAudioResourceMngr)
   {
      m_poAudioResourceMngr->vOnSPISelectDeviceResult(u32DevID, enDeviceConnReq, enRespCode, enErrorCode);
   }

   if (NULL != m_poVideoResourceMngr)
   {
      m_poVideoResourceMngr->vOnSPISelectDeviceResult(u32DevID, enDeviceConnReq, enRespCode, enErrorCode);
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPOResourceMngr::vOnModesChangedUpdate()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnModesChangedUpdate(const trDiPOModeState& corfrDiPOModeState)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnModesChangedUpdate"));

   if ((NULL != m_poVideoResourceMngr) && (m_rCurrModeState.enScreen != corfrDiPOModeState.enScreen))
   {
      m_poVideoResourceMngr->vUpdateDisplayContext(corfrDiPOModeState);
   }
   if ((NULL != m_poAudioResourceMngr) && (m_rCurrModeState.enAudio != corfrDiPOModeState.enAudio))
   {
      m_poAudioResourceMngr->vUpdateAudioContext(corfrDiPOModeState);
   }

   vUpdateAppState(corfrDiPOModeState);
   if((NULL != m_poVideoResourceMngr) && (m_poVideoResourceMngr->bIsDisplayOffActive()))
   {
      //The WakeUp of system is applicable only if system is in user-off mode
      m_poVideoResourceMngr->vUpdateSpmSubState(m_rCurrModeState, corfrDiPOModeState);
   }
   m_rCurrModeState = const_cast<trDiPOModeState&>(corfrDiPOModeState);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnSessionMsg(...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
         tenDiPOSessionTransport enSessionTransport, t_String szSessionIPAddress)
{
   SPI_INTENTIONALLY_UNUSED(enSessionTransport);
   SPI_INTENTIONALLY_UNUSED(szSessionIPAddress);
   /*lint -esym(40,fvUpdateSessionStatus)fvUpdateSessionStatus Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnSessionMsg entered with state: %d, Session Transport:%d, Session IP Address:%s", ETG_ENUM(DIPO_SESSION_STATE,
            enDiPOSessionState), ETG_ENUM(DIPO_SESSION_TRANSPORT, enSessionTransport), szSessionIPAddress.c_str()));

   tenSessionStatus enSessionStatus = e8_SESSION_UNKNOWN;
   if (e8DIPO_SESSION_END == enDiPOSessionState)
   {
      enSessionStatus = e8_SESSION_INACTIVE;
      m_bIsSessionActive = false;
      vHandleSessionEnd();
   } //if(e8DIPO_SESSION_END == m_enDiPOSessionState)
   else if (e8DIPO_SESSION_START == enDiPOSessionState)
   {
      enSessionStatus = e8_SESSION_ACTIVE;
      m_bIsSessionActive = true;
   } //else if (e8DIPO_SESSION_START == enDiPOSessionState)
   else if (e8DIPO_PLUGIN_LOADED == enDiPOSessionState)
   {

      if (NULL != m_poResourceMngrSettings)
      {
         trVehicleConfigData rVehicleConfigData;
         tvecVideoConfigList vecVideoConfigList;

         m_poResourceMngrSettings->vGetVideoConfigData(e8DEV_TYPE_DIPO, vecVideoConfigList);
         m_poResourceMngrSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,
                  rVehicleConfigData.rVideoConfigData);

         rVehicleConfigData.enDriveSideInfo = m_poResourceMngrSettings->enGetDriveSideInfo();
         m_poResourceMngrSettings->vGetOemIconData(rVehicleConfigData.rVehicleBrandInfo);

         m_poResourceMngrSettings->vGetDiPOExtendedFeaturesSupport(rVehicleConfigData.rExtendedFeaturesSupport);

         strncpy(rVehicleConfigData.szFirmwareRevision,
                  (m_poResourceMngrSettings->szGetSoftwareVersion()).c_str(),
                  MAX_STR_LEN);
         strncpy(rVehicleConfigData.szHardwareRevision,
                  (m_poResourceMngrSettings->szGetHardwareVersion()).c_str(),
                  MAX_STR_LEN);

         rVehicleConfigData.rVehicleInfo.electronicTollSupport = static_cast<ETCInfo>(m_enETCSupport);

         /*//! Check if rotary controllers are supported.
          rVehicleConfigData.rVehicleBrandInfo.bIsRotarySupported
          = m_poResourceMngrSettings->bGetRotaryCtrlSupport();*/

         //! Read the drive restriction value from configuration
         t_U8 u8DriveRestrictionInfo = m_poResourceMngrSettings->u8GetRestrictionsInfo(e8DEV_TYPE_DIPO, true);

         t_String szBtId = "";
         m_poResourceMngrSettings->vGetVehicleBTAddress(szBtId);
         vFormatBTMacAddress(szBtId);
         ETG_TRACE_USR2(("[PARAM] spi_tclDiPOResourceMngr::vOnSessionMsg BTMAC is %s ", szBtId.c_str()));

         if (NULL != spoDiPOCmdRsrcMngr)
         {
            tenVehicleConfiguration enDriveModeInfo;
            m_poResourceMngrSettings->vGetDriveModeInfo(enDriveModeInfo);

            tenVehicleConfiguration enNightModeInfo;
            m_poResourceMngrSettings->vGetNightModeInfo(enNightModeInfo);

            t_U8 u8DisplayInput;
            m_poResourceMngrSettings->vGetDisplayInputParam(u8DisplayInput);

            spoDiPOCmdRsrcMngr->vSendInfoMsg(szBtId.c_str(),
                     rVehicleConfigData,
                     u8DriveRestrictionInfo,
                     enDriveModeInfo,
                     enNightModeInfo,
                     u8DisplayInput);
         }

         if (NULL != m_poAudioResourceMngr)
         {
            m_poAudioResourceMngr->vOnPluginLoaded();
         }

         if (NULL != m_poVideoResourceMngr)
         {
            m_poVideoResourceMngr->vOnPluginLoaded();
         }

         trUserContext rDummyUsrCntxt;
         vSetAccessoryAppState(m_rDiPOAppState.enSpeechAppState,
                  m_rDiPOAppState.enPhoneAppState,
                  m_rDiPOAppState.enNavAppState,
                  rDummyUsrCntxt);

      } //if((NULL != poBluetooth) && (NULL != m_poResourceMngrSettings))
   }
   if ((e8_SESSION_UNKNOWN != enSessionStatus)
            && (NULL != (m_rRsrcMngrCallback.fvUpdateSessionStatus) && (0 != m_u32DevId)))
   {
      (m_rRsrcMngrCallback.fvUpdateSessionStatus)(m_u32DevId, e8DEV_TYPE_DIPO, enSessionStatus);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vRegRsrcMngrCallBack()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vRegRsrcMngrCallBack(trRsrcMngrCallback rRsrcMngrCallback)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vRegRsrcMngrCallBack entered "));
   m_rRsrcMngrCallback = rRsrcMngrCallback;

   //DiPOMsgRcvr to be started only after the ResourceMngr is created. Or else
   //SPI doesn't respond to SessionMsg(PlugInLoaded) and it leads to reset.
   if (NULL != spoDiPOManager)
   {
      spoDiPOManager->vInitiateMsgRcvr();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOResourceMngr::bGetVideoModeChangeMsg()
 ***************************************************************************/
t_Bool spi_tclDiPOResourceMngr::bGetVideoModeChangeMsg(const tenDisplayContext enDisplayContext, t_Bool bDisplayFlag,
         trDiPOVideoContext &rfrDiPOVideoContext)
{
   t_Bool bStatus = false;
   if (NULL != m_poResArb)
   {
      bStatus = m_poResArb->bGetVideoModeChangeMsg(enDisplayContext, bDisplayFlag, rfrDiPOVideoContext);
   }
   return bStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayContext()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayContext(const t_U32 cou32DevId, t_Bool bDisplayFlag,
         tenDisplayContext enDisplayContext, const trUserContext& rfrcUsrCntxt)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);
   if (NULL != m_poVideoResourceMngr)
   {
      m_poVideoResourceMngr->vSetAccessoryDisplayContext(bDisplayFlag, enDisplayContext);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOResourceMngr::bGetAudioModechangeMsg()
 ***************************************************************************/
t_Bool spi_tclDiPOResourceMngr::bGetAudioModechangeMsg(const tenAudioContext coenAudioCntxt, t_Bool bReqFlag,
         trDiPOAudioContext &rfrDiPOAudioContext)
{
   t_Bool bStatus = false;
   if (NULL != m_poResArb)
   {
      bStatus = m_poResArb->bGetAudioModechangeMsg(coenAudioCntxt, bReqFlag, rfrDiPOAudioContext);
   }
   return bStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryAudioContext()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetAccessoryAudioContext(const t_U32 cou32DevId, const tenAudioContext coenAudioCntxt,
         t_Bool bReqFlag, const trUserContext& rfrcUsrCntxt)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);
   if (NULL != m_poAudioResourceMngr)
   {
      m_poAudioResourceMngr->vSetAccessoryAudioContext(coenAudioCntxt, bReqFlag);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayMode(t_U32...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayMode(const t_U32 cu32DeviceHandle,
         const trDisplayContext corDisplayContext, const trDisplayConstraint corDisplayConstraint,
         const tenDisplayInfo coenDisplayInfo)
{
   SPI_INTENTIONALLY_UNUSED(cu32DeviceHandle);
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetAccessoryDisplayMode: Resource Management handled by SPI:%d,DisplayInfo:%d", ETG_ENUM(BOOL,
            (e8_DISPLAY_CONTEXT == coenDisplayInfo)), coenDisplayInfo));
   if (NULL != m_poVideoResourceMngr)
   {
      m_poVideoResourceMngr->vSetAccessoryDisplayMode(corDisplayContext, corDisplayConstraint, coenDisplayInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryAppState()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetAccessoryAppState(const tenSpeechAppState enSpeechAppState,
         const tenPhoneAppState enPhoneAppState, const tenNavAppState enNavAppState, const trUserContext& rfrcUsrCntxt)
{
   SPI_INTENTIONALLY_UNUSED(rfrcUsrCntxt);
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetAccessoryAppState entered: NAV = %d, PHONE = %d, SPEECH = %d", ETG_ENUM(NAV_APP_STATE,
            enNavAppState), ETG_ENUM(PHONE_APP_STATE, enPhoneAppState), ETG_ENUM(SPEECH_APP_STATE, enSpeechAppState)));

   if (NULL != spoDiPOCmdRsrcMngr)
   {
      trDiPOAppState rDiPOAppState;
      rDiPOAppState.enSpeechAppState = enSpeechAppState;
      rDiPOAppState.enPhoneAppState = enPhoneAppState;
      rDiPOAppState.enNavAppState = enNavAppState;
      m_rDiPOAppState = rDiPOAppState;

      spoDiPOCmdRsrcMngr->vSetAppState(rDiPOAppState);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vUpdateInitialSettings()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vUpdateInitialSettings(trRsrcSettings rRsrcSettings)
{
   m_enETCSupport = rRsrcSettings.m_enETCSupport;
   m_bIsLastModeSupported = rRsrcSettings.m_bIsLastModeSupported;

   if (NULL != m_poVideoResourceMngr)
   {
      m_poVideoResourceMngr->vSetResourceMngrSettingsInstance(m_poResourceMngrSettings);
      m_poVideoResourceMngr->vUpdateInitialVideoSettings(m_bIsLastModeSupported);
   }
   if (NULL != m_poAudioResourceMngr)
   {
      m_poAudioResourceMngr->vSetResourceMngrSettingsInstance(m_poResourceMngrSettings);
      m_poAudioResourceMngr->vUpdateInitialAudioSettings(m_bIsLastModeSupported);
   }
}

/* Private functions */

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vHandleSessionEnd()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vHandleSessionEnd()
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vHandleSessionEnd entered"));

   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if ((NULL != poMediator) && (0 != m_u32DevId))
   {
      //Trigger a device de-selection to switch back to the host mode.
      poMediator->vPostAutoDeviceSelection(m_u32DevId, e8DEVCONNREQ_DESELECT);
   }

   trDiPOModeState rDiPOModeState;
   rDiPOModeState.enScreen = e8DIPO_ENTITY_NA;
   rDiPOModeState.enAudio = e8DIPO_ENTITY_NA;
   rDiPOModeState.rSpeechState.enEntity = e8DIPO_ENTITY_NA;
   rDiPOModeState.rSpeechState.enSpeechMode = e8DIPO_SPEECHMODE_NA;
   rDiPOModeState.enPhone = e8DIPO_ENTITY_NA;
   rDiPOModeState.enNavigation = e8DIPO_ENTITY_NA;

   if ((NULL != m_poVideoResourceMngr) && (m_rCurrModeState.enScreen != rDiPOModeState.enScreen))
   {
      m_poVideoResourceMngr->vUpdateDisplayContext(rDiPOModeState);
   }
   if ((NULL != m_poAudioResourceMngr) && (m_rCurrModeState.enAudio != rDiPOModeState.enAudio))
   {
      m_poAudioResourceMngr->vUpdateAudioContext(rDiPOModeState);
   }

   //! Update the app states.(Clear all states)
   vUpdateAppState(rDiPOModeState);

   //Reset the flags to default
   m_rCurrModeState = rDiPOModeState;
   vSetCallStatus(false);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vUpdateAppState()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vUpdateAppState(const trDiPOModeState & corfoDiPOModeState)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vUpdateAppState entered"));

   /*lint -esym(40,fvPostDeviceAppState)fvPostDeviceAppState Undeclared identifier */
   /*lint -esym(40,fvSetDeviceAppState)fvSetDeviceAppState Undeclared identifier */

   t_Bool bAppState = false;
   tenSpeechAppState enSpeechAppState = e8SPI_SPEECH_UNKNOWN;
   tenPhoneAppState enPhoneAppState = e8SPI_PHONE_UNKNOWN;
   tenNavAppState enNavAppState = e8SPI_NAV_UNKNOWN;

   if (NULL != m_poResArb) // For App state handling
   {
      vAcquireDevAppStateLock();

      bAppState = m_poResArb->bGetDeviceSpeechState(corfoDiPOModeState, m_rCurrModeState, enSpeechAppState);
      bAppState = m_poResArb->bGetDevicePhoneState(corfoDiPOModeState, m_rCurrModeState, enPhoneAppState) || bAppState;
      bAppState = m_poResArb->bGetDeviceNavState(corfoDiPOModeState, m_rCurrModeState, enNavAppState) || bAppState;

      if ((true == bAppState) && (NULL != m_rRsrcMngrCallback.fvSetDeviceAppState))
      {
         (m_rRsrcMngrCallback.fvSetDeviceAppState)(enSpeechAppState, enPhoneAppState, enNavAppState);
      }

      vReleaseDevAppStateLock();

      if ((true == bAppState) && (NULL != (m_rRsrcMngrCallback.fvPostDeviceAppState)))
      {
         trUserContext rDummyUsrCntxt;
         (m_rRsrcMngrCallback.fvPostDeviceAppState)(enSpeechAppState, enPhoneAppState, enNavAppState, rDummyUsrCntxt);
      }
   } //if(NULL != m_poResArb)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetVehicleBTAddress()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetVehicleBTAddress(t_String szBtAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetVehicleBTAddress:BTAddress:%s", szBtAddress.c_str()));
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetVehicleBTAddress:Stored BTAddress:%s", szBtMacAddress.c_str()));

   if ((0 != strcmp(szBtAddress.c_str(), szBtMacAddress.c_str())) && (true == vFormatBTMacAddress(szBtAddress)))
   {
      if (NULL != spoDiPOCmdBluetooth)
      {
         spoDiPOCmdBluetooth->vUpdateBTAddr(szBtAddress.c_str());
      }
      szBtMacAddress = szBtAddress;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
         const t_U8 cou8DriveModeRestrictionInfo)
{
   //! Send the drive mode restriction data to CarPlay adapter if data has changed
   if (cou8DriveModeRestrictionInfo != m_u8DriveModeRestrictionInfo)
   {
      //! Store the latest drive mode restriction data
      m_u8DriveModeRestrictionInfo = cou8DriveModeRestrictionInfo;

      if (NULL != m_poVideoResourceMngr)
      {
         m_poVideoResourceMngr->vSetFeatureRestrictions(cou8ParkModeRestrictionInfo, cou8DriveModeRestrictionInfo);
      }
   } //if (cou8DriveModeRestrictionInfo != m_u8DriveModeRestrictionInfo)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vFormatBTMacAddress()
 ***************************************************************************/
t_Bool spi_tclDiPOResourceMngr::vFormatBTMacAddress(t_String &szMACAddress)
{
   // Insert the ":" charecter in the address string begin from second position and after the two charecters.
   const t_U8 u8InitPos = 2;
   const t_U8 u8Offset = 3;
   t_Bool bStatus = false;

   //! If string is empty return false.
   if (!szMACAddress.empty())
   {
      bStatus = true;
   }

   for (size_t siIndex = u8InitPos; siIndex < szMACAddress.length(); siIndex += u8Offset)
   {
      szMACAddress.insert(siIndex, ":");
   }

   ETG_TRACE_USR1(("Formatted MAC address : %s", szMACAddress.c_str()));
   return bStatus;
}

/********************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnSetNativeVideoRenderingStatus()
 ********************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnSetNativeVideoRenderingStatus(
         tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
{
   //The message from the Service file is received here. The native display is ON.
   //when the display is ON the message has to reach the video adapter to unlock
   //dispatch message to the IPC
   if (NULL != m_poVideoResourceMngr)
   {
      m_poVideoResourceMngr->vOnSetNativeVideoRenderingStatus(enNativeVideoRenderingStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vProjVideoRenderingStatus(...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vProjVideoRenderingStatus(tenProjVideoRenderingStatus enProjRenderingStatus)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vProjVideoRenderingStatus entered with %d", ETG_ENUM(PROJECTION_VIDEO_RENDERING_STATUS,
            enProjRenderingStatus)));

   if (NULL != m_rRsrcMngrCallback.fvProjectionVideoRenderingStatus)
   {
      (m_rRsrcMngrCallback.fvProjectionVideoRenderingStatus)(enProjRenderingStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetGeneralRestrictions(...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetGeneralRestrictions(const t_U16 u16GeneralRestrInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetGeneralRestrictions:u16GeneralRestrInfo:%d", u16GeneralRestrInfo));
   //TODO - Fetch The last Bit - SetEtcSatus and then send it the info response message

   //fetching the 0th bit from the GeneralRestriction - If 1 - Enabled 0 - Disabled
   m_enETCSupport = (u16GeneralRestrInfo & su8ETCEnabledMask) ? e8ETCInfo_InActive : e8ETCInfo_NotSupported;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetETCStatus(...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetETCStatus(t_Bool bETCStatus)
{
   //TODO - Based on the bETCStatus value - Carry on the implementation
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetETCStatus entered"));

   //Posting the message to Carplayd process through the DipoCmdRsrcMngr
   if (NULL != spoDiPOCmdRsrcMngr)
   {
      tenETCInfo enETCStatus;
      enETCStatus = (true == bETCStatus) ? e8ETCInfo_Active : e8ETCInfo_InActive;
      spoDiPOCmdRsrcMngr->vSetETCStatus(enETCStatus);
   }
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vSetETCStatus left"));
}

/***************************************************************************
 ** FUNCTION:  tVoid spi_tclDiPOResourceMngr::vOnTelephoneCallStatus()
 ***************************************************************************/
tVoid spi_tclDiPOResourceMngr::vOnTelephoneCallStatus(const std::vector<trTelCallStatusInfo>& rfcoTelCallStatusList)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnTelephoneCallStatus():Size:%d", rfcoTelCallStatusList.size()));

   m_CallStatusLock.s16Lock();
   mvecTelCallStatusInfo = rfcoTelCallStatusList;
   m_CallStatusLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger(...
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger(t_String szBTMACAddr)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger:Device BT Address:%s", szBTMACAddr.c_str()));

   t_U32 u32BTDevHandle = 0x00;

   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   if (NULL != poExtCompMgr)
   {
      spi_tclExtCmdBluetoothIntf* poExtCmdBluetoothIntf = poExtCompMgr->poGetCmdBluetoothIntfInst();
      if (NULL != poExtCmdBluetoothIntf)
      {
         u32BTDevHandle = poExtCmdBluetoothIntf->u32GetBTDeviceHandle(szBTMACAddr);
      }
   }

   ETG_TRACE_USR2(("Device Handle is:%d", u32BTDevHandle));

   if (0x00 != u32BTDevHandle)
   {
      m_CallStatusLock.s16Lock();
      //Check whether callStatus is active. If yes set a flag which can be used to send unborrow.
      auto it = mvecTelCallStatusInfo.begin();
      for (; it != mvecTelCallStatusInfo.end(); it++)
      {
         ETG_TRACE_USR4(("Device Handle is:%d and call Status is:%d", it->u32BTDeviceHandle, it->enCallStatus));
         if ((u32BTDevHandle == it->u32BTDeviceHandle) && (e8IDLE != it->enCallStatus))
         {
            ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger::BT call is active with the same device connected for CP."));
            vSetCallStatus(true);

            //send unborrow followed by app state incative for audio resource if B/T call was already active
            if (NULL != m_poAudioResourceMngr)
            {
               ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger - send audio unborrow"));
               m_poAudioResourceMngr->vSetAccessoryAudioContext(e8SPI_AUDIO_PHONE, false);
               
               trUserContext rfrcUsrCntxt;
               vSetAccessoryAppState(e8SPI_SPEECH_END, e8SPI_PHONE_NOT_ACTIVE, e8SPI_NAV_NOT_ACTIVE, rfrcUsrCntxt);
            }

            break;
         }
      }
      m_CallStatusLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vPostDisplayContext()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vPostDisplayContext(t_Bool bRenderStatus, tenDisplayContextReason enDispCntxtReason,
          tenDiPOEntity enPermScreen)
{
   if (NULL != (m_rRsrcMngrCallback.fvPostDeviceDisplayContext))
   {
      trUserContext rDummyUsrCntxt;
      (m_rRsrcMngrCallback.fvPostDeviceDisplayContext)(bRenderStatus, enDispCntxtReason, enPermScreen, rDummyUsrCntxt);
      ETG_TRACE_USR4(("[PARAM]::vPostDisplayContext - Device Display Context Update; Status: %d", ETG_ENUM(BOOL,
               bRenderStatus)));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vPostAudioContext()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vPostAudioContext(t_Bool bPlayStatus, t_U8 u8AudioCntxt, tenDiPOEntity enPermMainAudio)
{
   if (NULL != (m_rRsrcMngrCallback.fvPostDeviceAudioContext))
   {
      trUserContext rDummyUsrCntxt;
      (m_rRsrcMngrCallback.fvPostDeviceAudioContext)(bPlayStatus, u8AudioCntxt, enPermMainAudio, rDummyUsrCntxt);
      ETG_TRACE_USR4(("[PARAM]::vPostAudioContext - Device Audio Context Update; PlayStatus: %d, Context: %d", ETG_ENUM(BOOL,
               bPlayStatus), ETG_ENUM(AUDIO_CONTEXT, u8AudioCntxt)));
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOResourceMngr::bIsSessionActive()
 ***************************************************************************/
t_Bool spi_tclDiPOResourceMngr::bIsSessionActive()
{
   return m_bIsSessionActive;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetDisplayLastMode()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetDisplayLastMode(tenLastMode enLastDisplayMode)
{
   m_enLastDisplayMode = enLastDisplayMode;
   vUpdateCallStatus();
}

/***************************************************************************
 ** FUNCTION: tenLastMode spi_tclDiPOResourceMngr::enGetDisplayLastMode()
 ***************************************************************************/
tenLastMode spi_tclDiPOResourceMngr::enGetDisplayLastMode()
{
   return m_enLastDisplayMode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetAudioLastMode()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetAudioLastMode(tenLastMode enLastAudioMode)
{
   m_enLastAudioMode = enLastAudioMode;
   vUpdateCallStatus();
}

/***************************************************************************
 ** FUNCTION: tenLastMode spi_tclDiPOResourceMngr::enGetAudioLastMode()
 ***************************************************************************/
tenLastMode spi_tclDiPOResourceMngr::enGetAudioLastMode()
{
   return m_enLastAudioMode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetCallStatus()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vSetCallStatus(t_Bool bIsCallActiveBeforeDisableBT)
{
   ETG_TRACE_USR4(("[PARAM]spi_tclDiPOResourceMngr::vSetCallStatus = %d", ETG_ENUM(BOOL, bIsCallActiveBeforeDisableBT)));
   m_bIsCallActiveBeforeDisableBT = bIsCallActiveBeforeDisableBT;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOResourceMngr::bGetCallStatus()
 ***************************************************************************/
t_Bool spi_tclDiPOResourceMngr::bGetCallStatus()
{
   ETG_TRACE_USR4(("[PARAM]spi_tclDiPOResourceMngr::bGetCallStatus = %d", ETG_ENUM(BOOL, m_bIsCallActiveBeforeDisableBT)));
   return m_bIsCallActiveBeforeDisableBT;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vUpdateCallStatus()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vUpdateCallStatus()
{
   t_Bool bIsCallActiveBeforeDisableBT =
            ((e8NOT_APPLICABLE == m_enLastDisplayMode) && (e8NOT_APPLICABLE == m_enLastAudioMode)) ?
                     (false) : (m_bIsCallActiveBeforeDisableBT);
   vSetCallStatus(bIsCallActiveBeforeDisableBT);
}

/***************************************************************************
 ** FUNCTION: Static  t_Void spi_tclDiPOResourceMngr::vOnEvaluateInitalModes()
 ***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vOnEvaluateInitalModes(const t_String& corfszBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPOResourceMngr::vOnEvaluateInitalModes entered"));

   if (NULL != m_poAudioResourceMngr)
   {
      m_poAudioResourceMngr->vOnEvaluateInitalModes(corfszBTAddress);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vPostDeviceSpmSubState
***************************************************************************/
t_Void spi_tclDiPOResourceMngr::vPostDeviceSpmSubState(tenSpmSubState enSpmSubState, t_Bool m_bSubStateData)
{
   if (NULL != (m_rRsrcMngrCallback.fvPostDeviceSpmSubState))
   {
      (m_rRsrcMngrCallback.fvPostDeviceSpmSubState)(enSpmSubState, m_bSubStateData);
   }
}

//lint -restore
