/*!
*******************************************************************************
* \file              spi_tclDiPoKeys.cpp
* \brief             DiPO Supported Key Information
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO Supported Key Information implementation
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
29.04.2019 |  Ram Chaithanya S A          | Initial Version
\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <sstream>
#include <set>
#include "spi_tclDiPoKeys.h"
#include "HIDUtils.h"
#include "Trace.h"

#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclDiPoKeys.cpp.trc.h"
#endif
#endif

static const t_U8 scou8byteTobit              =  8;
static const t_U8 scou8HID_CONSUMER           = 0x0C;
static const t_U8 scou8HID_PHONE              = 0x0B;
static const t_U8 scou8HID_ADDUSAGE           = 0x01;
static const t_U8 scou8HID_COLLECTION         = 0x01;
static const t_U8 scou8HID_LOGICAL_MIN        = 0x00;
static const t_U8 scou8HID_LOGICAL_MAX        = 0x01;
static const t_U8 scou8HID_REPORT             = 0x01;
static const t_U8 scou8HID_ADDINPUT           = 0x01;
static const t_U8 scou8HID_INPUT              = 0x02;
static const t_U8 scou8HID_INPUT_ARRAY        = 0x03;
static const t_U8 scou8HID_KNOB_INPUT         = 0x06;
static const t_U8 scou8HID_KNOB_REPORTSIZE_8  = 0x08;
static const t_U8 scou8HID_KNOB_USAGEPAGE     = 0x09;
static const t_U8 scou8HID_KNOB_USAGEX        = 0x30;
static const t_U8 scou8HID_KNOB_USAGEY        = 0x31;
static const t_U8 scou8HID_KNOB_USAGEDIAL     = 0x37;
static const t_U8 scou8HID_KNOB_LOGICAL_MIN   = 0x81;
static const t_U8 scou8HID_KNOB_LOGICAL_MAX   = 0x7F;
static const t_U8 scou8CHAR_MAX               = 0xff;
static const t_U8 scou8BIT_NOT_SET            = 0;
static const t_U8 scou8BIT_SET                = 1;
static const t_U8 scou8HID_KNOB_REPORTCOUNT   = 0x02;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::spi_tclDiPoKeys()
***************************************************************************/
spi_tclDiPoKeys::spi_tclDiPoKeys()
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::spi_tclDiPoKeys()"));
   m_vecConsKeys = {
                    e32MULTIMEDIA_PLAY_PAUSE,e32MULTIMEDIA_NEXT,
                    e32MULTIMEDIA_PREVIOUS,e32MULTIMEDIA_PLAY,
                    e32MULTIMEDIA_PAUSE,e32DEV_BACKWARD,e32DEV_HOME
                   };
   m_vecPhnKeys  = {
                    e32DEV_PHONE_CALL,e32DEV_PHONE_FLASH,
                    e32DEV_PHONE_END,e32DEV_PHONE_MUTE
                   };
   m_vecKnobKeys = {
                    e32TWODKNOB0_SHIFT_RIGHT,e32TWODKNOB0_SHIFT_LEFT,
                    e32TWODKNOB0_SHIFT_UP,e32TWODKNOB0_SHIFT_DOWN
                   };

   m_DIPOConsumerkeyCodeMap =
   {
         { e32MULTIMEDIA_NEXT, e8_DIPO_KEYCODE_MEDIA_NEXT },
         { e32MULTIMEDIA_PREVIOUS, e8_DIPO_KEYCODE_MEDIA_PREVIOUS },
         { e32MULTIMEDIA_PLAY, e8_DIPO_KEYCODE_PLAY},
         { e32MULTIMEDIA_PAUSE, e8_DIPO_KEYCODE_PAUSE},
         { e32MULTIMEDIA_PLAY_PAUSE, e8_DIPO_KEYCODE_MEDIA_PLAY_PAUSE},
         { e32DEV_HOME, e16_DIPO_KEYCODE_HOME},
         { e32DEV_BACKWARD, e16_DIPO_BACK}
   };
   m_DIPOPhonekeyCodeMap =
   {
         { e32DEV_PHONE_CALL, e8_DIPO_KEYCODE_HOOK },
         { e32DEV_PHONE_FLASH, e8_DIPO_KEYCODE_FLASH },
         { e32DEV_PHONE_END, e8_DIPO_KEYCODE_DROP},
         { e32DEV_PHONE_MUTE, e8_DIPO_PHONE_MUTE}
   };
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::~spi_tclDiPoKeys()
***************************************************************************/
spi_tclDiPoKeys::~spi_tclDiPoKeys()
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::~spi_tclDiPoKeys()"));
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildConsumerDescriptor()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildConsumerDescriptor(const std::set<tenKeyCode> &rsetKeyCodes,std::vector<t_U8> &vecu8ConsumerKeys)
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildConsumerDescriptor"));
   HIDUtils oHIDUtils;

   std::vector<tenKeyCode> vecresKeys;
   std::vector<tenKeyCode> vecSortKeys = m_vecConsKeys;

   sort(vecSortKeys.begin(),vecSortKeys.end());

   std::set_intersection (rsetKeyCodes.begin(),rsetKeyCodes.end(),vecSortKeys.begin(),
                          vecSortKeys.end(),std::back_inserter(vecresKeys));

   if(!vecresKeys.empty())
   {
     t_U8 u8KeyCount = 0;
     std::set<tenKeyCode> setenTwoByteKeys = {e32DEV_BACKWARD,e32DEV_HOME};
     std::vector<t_U8> vecOneByteKeys,vecTwoByteKeys;
     oHIDUtils.vAddUsagePage(scou8HID_CONSUMER); //(CONSUMER HID)
     oHIDUtils.vAddUsage(scou8HID_ADDUSAGE);
     oHIDUtils.vStartCollection(scou8HID_COLLECTION);
     oHIDUtils.vAddLogicalMinMax(scou8HID_LOGICAL_MIN,scou8HID_LOGICAL_MAX);
     oHIDUtils.vAddReportSize(scou8HID_REPORT);

     for(auto enConsKeys : m_vecConsKeys)
     {
       if(rsetKeyCodes.find(enConsKeys) != rsetKeyCodes.end())
       {
        if(m_DIPOConsumerkeyCodeMap.find(enConsKeys) != m_DIPOConsumerkeyCodeMap.end())
        {
          if(setenTwoByteKeys.find(enConsKeys) == setenTwoByteKeys.end())
          {
             vecOneByteKeys.push_back(scou8HID_USAGE);
             vecOneByteKeys.push_back(m_DIPOConsumerkeyCodeMap[enConsKeys]);
             u8KeyCount++;
          }
          else
          {
             vecTwoByteKeys.push_back(scou8HID_USAGE_TWOBYTE);
             vecTwoByteKeys.push_back(static_cast<t_U8>((m_DIPOConsumerkeyCodeMap[enConsKeys] & scou8CHAR_MAX)));
             vecTwoByteKeys.push_back(static_cast<t_U8>((m_DIPOConsumerkeyCodeMap[enConsKeys] >> scou8byteTobit) & scou8CHAR_MAX));
             u8KeyCount++;
          }
        }
       }
     }
     oHIDUtils.vAddReportCount(u8KeyCount);
     if(0 < vecOneByteKeys.size() )
     {
        oHIDUtils.vAddUsage(vecOneByteKeys);
     }
     if(0 < vecTwoByteKeys.size())
     {
        oHIDUtils.vAddUsage(vecTwoByteKeys);
     }
     oHIDUtils.vAddInput(scou8HID_INPUT);
     oHIDUtils.vAddReportSize(static_cast<t_U8>(scou8byteTobit - u8KeyCount));
     oHIDUtils.vAddReportCount(scou8HID_REPORT);
     oHIDUtils.vAddInput(scou8HID_INPUT_ARRAY);
     oHIDUtils.vEndCollection();
     oHIDUtils.vGetHIDDescriptor(vecu8ConsumerKeys);
   }
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildConsumerBitmap()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildConsumerBitmap(const std::set<tenKeyCode> &rsetKeyCodes,std::vector<t_U8> &vecu8KeyMapBitValues)
{

  ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildConsumerBitmap"));
  ////////////// Consumer Key Bit Map Info///////////
  // Position 1 : Play/Pause Toggle Button         //
  // Position 2 : Next                             //
  // Position 3 : Previous                         //
  // Position 4 : Play                             //
  // Position 5 : Pause                            //
  // Position 6 : Back                             //

  t_U8 u8Pos = 0;
  for(auto enConsKeys : m_vecConsKeys)
  {
    if(rsetKeyCodes.find(enConsKeys) != rsetKeyCodes.end())
    {
      vecu8KeyMapBitValues.push_back(static_cast<t_U8>(pow(2,u8Pos)));
      u8Pos++;
    }
    else
    {
      vecu8KeyMapBitValues.push_back(scou8BIT_NOT_SET);
    }
  }
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildPhoneKeyDescriptor()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildPhoneKeyDescriptor(const std::set<tenKeyCode> &rsetKeyCodes,std::vector<t_U8> &vecu8PhoneKeys)
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildPhoneKeyDescriptor"));
   HIDUtils oHIDUtils;

   std::vector<tenKeyCode> vecresKeys;
   std::vector<tenKeyCode> vecSortKeys = m_vecPhnKeys;

   sort(vecSortKeys.begin(),vecSortKeys.end());

   std::set_intersection (rsetKeyCodes.begin(),rsetKeyCodes.end(),vecSortKeys.begin(),
                          vecSortKeys.end(),std::back_inserter(vecresKeys));

   if(!vecresKeys.empty())
   { 
     t_U8 u8KeyCount = 0;
     oHIDUtils.vAddUsagePage(scou8HID_PHONE); //(Telephone HID)
     oHIDUtils.vAddUsage(scou8HID_ADDUSAGE); //(Consumer Control)
     oHIDUtils.vStartCollection(scou8HID_COLLECTION); // (Application)
     oHIDUtils.vAddUsagePage(scou8HID_PHONE);
     for(auto enPhnKeys : m_vecPhnKeys)
     {
       if(rsetKeyCodes.find(enPhnKeys) != rsetKeyCodes.end())
       {
         if(m_DIPOPhonekeyCodeMap.find(enPhnKeys) != m_DIPOPhonekeyCodeMap.end())
         {
            oHIDUtils.vAddUsage(m_DIPOPhonekeyCodeMap[enPhnKeys]);
            u8KeyCount++;
         }
       }
     }
     oHIDUtils.vAddLogicalMinMax(scou8HID_LOGICAL_MIN,scou8HID_LOGICAL_MAX);
     oHIDUtils.vAddReportSize(scou8HID_REPORT);
     oHIDUtils.vAddReportCount(u8KeyCount);
     oHIDUtils.vAddInput(scou8HID_INPUT);
     oHIDUtils.vAddReportSize(static_cast<t_U8>(scou8byteTobit - u8KeyCount));
     oHIDUtils.vAddReportCount(scou8HID_REPORT);
     oHIDUtils.vAddInput(scou8HID_INPUT_ARRAY);
     oHIDUtils.vEndCollection();
     oHIDUtils.vGetHIDDescriptor(vecu8PhoneKeys);
   }
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildPhoneKeyBitmap()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildPhoneKeyBitmap(const std::set<tenKeyCode> &rsetKeyCodes,std::vector<t_U8> &vecu8PhoneKeyBitmap)
{
  ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildPhoneKeyBitmap"));

  t_U8 u8Pos = 0;
  for(auto enPhnKeys : m_vecPhnKeys)
  {
    if(rsetKeyCodes.find(enPhnKeys) != rsetKeyCodes.end())
    {
      vecu8PhoneKeyBitmap.push_back(static_cast<t_U8>(pow(2,u8Pos)));
      u8Pos++;
    }
    else
    {
      vecu8PhoneKeyBitmap.push_back(scou8BIT_NOT_SET);
    }
  }
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildKnobKeyDescriptor()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildKnobKeyDescriptor(const std::set<tenKeyCode>& corfrsetKeyCodes,
                                                std::vector<t_U8> &vecu8KnobKey,
                                                t_Bool bITCommanderSupport,t_Bool bRotaryKnobSupport)
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildKnobKeyDescriptor"));

   HIDUtils oHIDUtils;
   if(true == bITCommanderSupport || true == bRotaryKnobSupport)
   {
      oHIDUtils.vAddUsagePage(scou8HID_ADDUSAGE); //(Generic Desktop)
      oHIDUtils.vAddUsage(scou8HID_KNOB_REPORTSIZE_8); //(Consumer Control)
      oHIDUtils.vStartCollection(scou8HID_COLLECTION); // (Application)
      if(true == bRotaryKnobSupport)
      {
        if(corfrsetKeyCodes.find(e32DEV_MENU) != corfrsetKeyCodes.end())
        {
           oHIDUtils.vAddUsagePage(scou8HID_KNOB_USAGEPAGE);
           oHIDUtils.vAddUsage(scou8HID_ADDUSAGE);
           oHIDUtils.vAddLogicalMinMax(scou8HID_LOGICAL_MIN,scou8HID_LOGICAL_MAX);
           oHIDUtils.vAddReportSize(scou8HID_REPORT);
           oHIDUtils.vAddReportCount(scou8HID_REPORT);
           oHIDUtils.vAddInput(scou8HID_INPUT);
           oHIDUtils.vAddReportSize(static_cast<t_U8>(scou8byteTobit - scou8HID_REPORT));
           oHIDUtils.vAddReportCount(scou8HID_REPORT);
           oHIDUtils.vAddInput(scou8HID_ADDINPUT);
        }
        oHIDUtils.vAddUsagePage(scou8HID_ADDUSAGE);
        oHIDUtils.vAddUsage(scou8HID_KNOB_USAGEDIAL);
        oHIDUtils.vAddLogicalMinMax(scou8HID_KNOB_LOGICAL_MIN,scou8HID_KNOB_LOGICAL_MAX);
        oHIDUtils.vAddReportSize(scou8HID_KNOB_REPORTSIZE_8);
        oHIDUtils.vAddReportCount(scou8HID_REPORT);
        oHIDUtils.vAddInput(scou8HID_KNOB_INPUT);
      }
      if(true == bITCommanderSupport)
      {
        oHIDUtils.vAddUsagePage(scou8HID_ADDUSAGE);
        oHIDUtils.vAddUsage(scou8HID_KNOB_USAGEX);
        oHIDUtils.vAddUsage(scou8HID_KNOB_USAGEY);
        oHIDUtils.vAddLogicalMinMax(scou8HID_KNOB_LOGICAL_MIN,scou8HID_KNOB_LOGICAL_MAX);
        oHIDUtils.vAddReportSize(scou8HID_KNOB_REPORTSIZE_8);
        oHIDUtils.vAddReportCount(scou8HID_KNOB_REPORTCOUNT);
        oHIDUtils.vAddInput(scou8HID_INPUT);
       }
      oHIDUtils.vEndCollection();
   }
   oHIDUtils.vGetHIDDescriptor(vecu8KnobKey);
}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vBuildKnobKeyBitmap()
***************************************************************************/
t_Void spi_tclDiPoKeys::vBuildKnobKeyBitmap(std::vector<t_U8> &vecu8KnobKeyBitmap,
                                            t_Bool bITCommanderSupport,
                                            t_Bool bRotaryKnobSupport)
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::vBuildKnobKeyBitmap"));

   if(true == bITCommanderSupport || true == bRotaryKnobSupport)
   {
     vecu8KnobKeyBitmap.push_back(scou8BIT_SET);
   }
   else
   {
     vecu8KnobKeyBitmap.push_back(scou8BIT_NOT_SET);
   }

}

/***************************************************************************
** FUNCTION: spi_tclDiPoKeys::vGetKeySupported()
***************************************************************************/
t_Void spi_tclDiPoKeys::vGetKeySupported(std::vector<t_U8> &vecu8ConsumerKeys,std::vector<t_U8> &vecu8PhoneKeys,
                                         std::vector<t_U8> &vecu8KnobKeys,std::vector<t_U8> &vecu8ConsumerBitmap,
                                         std::vector<t_U8> &vecu8PhoneKeyBitmap,std::vector<t_U8> &vecu8KnobKeyBitmap,
                                         const std::set<tenKeyCode>& rsetKeyCodes,t_Bool& bITCommanderSupport,
                                         t_Bool& bRotaryKnobSupport)
{
   ETG_TRACE_USR1(("spi_tclDiPoKeys::vGetKeySupported"));
   // Need to check whether it can be done more cleanly
   for(auto keys : m_vecKnobKeys)
   {
     if(rsetKeyCodes.find(keys) != rsetKeyCodes.end())
     {
       bITCommanderSupport = true;
       //Even one of the 4 knob keys are sent by HMI,it means itCommander is supported
       break;
     }
   }
   if(rsetKeyCodes.find(e32ROTARY_KNOB) != rsetKeyCodes.end())
   {
     bRotaryKnobSupport = true;
   }
   vBuildConsumerDescriptor(rsetKeyCodes,vecu8ConsumerKeys);
   vBuildConsumerBitmap(rsetKeyCodes,vecu8ConsumerBitmap);
   vBuildPhoneKeyDescriptor(rsetKeyCodes,vecu8PhoneKeys);
   vBuildPhoneKeyBitmap(rsetKeyCodes,vecu8PhoneKeyBitmap);
   vBuildKnobKeyDescriptor(rsetKeyCodes,vecu8KnobKeys,bITCommanderSupport,bRotaryKnobSupport);
   vBuildKnobKeyBitmap(vecu8KnobKeyBitmap,bITCommanderSupport,bRotaryKnobSupport);
}

//lint -restore

