/*!
 *******************************************************************************
 * \file              spi_tclDiPoInputHandler.cpp
 * \brief             SPI input handler for DiPo devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   provides input handler to send input events from DiPo client to DiPo server
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 25.11.2013 |  Hari Priya E R              | Initial Version
 10.03.2014 |  Hari Priya E R              | Touch handling Implementation
 04.04.2014 |  Hari Priya E R              | Key handling Implementation
 03.07.2014 |  Hari Priya E R              | Added changes related to Input Response interface
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)    | Added Lint comments to suppress C++11 Errors
 15.07.2015 |  Sameer Chandra              | Added Knob key Support Implementation

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "Trace.h"
#include "BaseTypes.h"
#include "SPITypes.h"
#include "DiPOTypes.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPOCmdInput.h"
#include "spi_tclDiPoInputHandler.h"
#include "spi_tclDiPOCmdSession.h"
#include "spi_tclExtCmdInputIntf.h"
#include "spi_tclExtCompManager.h"

#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclDiPoInputHandler.cpp.trc.h"
#endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

static spi_tclDiPOCmdInput* spoDiPOCmdInput = NULL;

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoInputHandler::spi_tclDiPoInputHandler()
 ***************************************************************************/
spi_tclDiPoInputHandler::spi_tclDiPoInputHandler():m_HardKeysUpdate(false),
                                                   m_bIsSessionActive(false)
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::spi_tclDiPoInputHandler entered"));

   spi_tclDiPOManager* poDiPOMngr = spi_tclDiPOManager::getInstance();
   if (NULL != poDiPOMngr)
   {
      spoDiPOCmdInput = poDiPOMngr->poGetDiPOInputInstance();
   }

   m_poDiPoKeyCodeUtils = new spi_tclDiPoKeys();
   SPI_NORMAL_ASSERT( NULL == m_poDiPoKeyCodeUtils);

   //Dipo Supported Keys
   m_setenDiPoSuppKeys =
   {
     e32MULTIMEDIA_NEXT,
     e32MULTIMEDIA_PREVIOUS,
     e32MULTIMEDIA_PLAY,
     e32MULTIMEDIA_PAUSE,
     e32MULTIMEDIA_PLAY_PAUSE,
     e32DEV_HOME,
     e32DEV_BACKWARD,
     e32DEV_MENU,
     e32DEV_PHONE_CALL,
     e32DEV_PHONE_FLASH,
     e32DEV_PHONE_END,
     e32DEV_PHONE_MUTE,
     e32ROTARY_KNOB,
     e32TWODKNOB0_SHIFT_RIGHT,
     e32TWODKNOB0_SHIFT_LEFT,
     e32TWODKNOB0_SHIFT_UP,
     e32TWODKNOB0_SHIFT_DOWN,
     e32INVALID_KEY
   };

   m_setenDiPoPhoneKeys =
   {
     e32DEV_PHONE_CALL,
     e32DEV_PHONE_END,
     e32MERGE_CALLS,
     e32SWAP_CALLS,
     e32INVALID_KEY
   };

   m_setenDiPoSoftKeys = m_setenDiPoPhoneKeys;

}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoInputHandler::~spi_tclDiPoInputHandler()
 ***************************************************************************/
spi_tclDiPoInputHandler::~spi_tclDiPoInputHandler()
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::~spi_tclDiPoInputHandler entered"));
   RELEASE_MEM(m_poDiPoKeyCodeUtils);
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoInputHandler::bInitialize()
***************************************************************************/
t_Bool spi_tclDiPoInputHandler::bInitialise()
{
   ETG_TRACE_USR1((" spi_tclDiPoInputHandler::bInitialize()" ));
   spi_tclDiPOManager* poDiPOManager = spi_tclDiPOManager::getInstance();
   SPI_NORMAL_ASSERT( NULL == poDiPOManager);

   if (NULL != poDiPOManager)
   {
      poDiPOManager->bRegisterObject((spi_tclDiPORespSession*) this);
   }

   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vUnInitialize()
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vUnInitialize()"));
   //Add code
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessTouchEvent
 ***************************************************************************/
t_Void spi_tclDiPoInputHandler::vProcessTouchEvent(t_U32 u32DeviceHandle, trTouchData &rfrTouchData) const
{
   ETG_TRACE_USR4(("[PARAM]::vProcessTouchEvent()"));

   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
   std::vector<trTouchInfo>::iterator itTouchInfo;
   std::vector<trTouchCoordinates> vecTouchCoord;
   std::vector<trTouchCoordinates>::iterator itTouchCoord;

   //Iterate through the list and populate the touch co-ordinates
   for (itTouchInfo = rfrTouchData.tvecTouchInfoList.begin(); itTouchInfo != rfrTouchData.tvecTouchInfoList.end();
            ++itTouchInfo)
   {
      vecTouchCoord = itTouchInfo->tvecTouchCoordinatesList;

      for (itTouchCoord = vecTouchCoord.begin(); itTouchCoord != vecTouchCoord.end(); ++itTouchCoord)
      {
         if (NULL != spoDiPOCmdInput)
         {
            spoDiPOCmdInput->vSendTouchEvent(itTouchCoord->s32XCoordinate,
                     itTouchCoord->s32YCoordinate,
                     itTouchCoord->enTouchMode);
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessKeyEvents
 ***************************************************************************/
t_Void spi_tclDiPoInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle, tenKeyMode enKeyMode,
         tenKeyCode enKeyCode) const
{
   ETG_TRACE_USR4(("[PARAM]::vProcessKeyEvents - Key Mode: %d,Key Code : %d ", ETG_ENUM(KEY_MODE, enKeyMode), ETG_ENUM(SPI_KEY_CODE,
            enKeyCode)));

   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);

   if (NULL != spoDiPOCmdInput)
   {
     switch(enKeyMode)
     {
       case e8KEY_RELEASE:
       case e8KEY_PRESS:
       case e8KEY_LONGPRESS:
       {
         spoDiPOCmdInput->vSendKeyEvent(enKeyMode,enKeyCode,e8_BUTTON_TYPE,0);
         break;
       }
       case e8SOFTKEY_RELEASE:
       {
         ETG_TRACE_USR4(("[PARAM]::vProcessKeyEvents SoftKey Release is not supported Yet"));
         break;
       }
       case e8SOFTKEY_PRESS:
       {
         vProcessnValidate(u32DeviceHandle,enKeyMode,enKeyCode);
         break;
       }
       default:
        ETG_TRACE_USR4(("[PARAM]::vProcessKeyEvents:Wrong KeyMode Function"));
        break;
     }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclDiPoInputHandler::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclDiPoInputHandler::vSelectDevice(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq)
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vSelectDevice:Device Id-0x%x", cou32DevId));
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   if (NULL != m_rInputCallbacks.fvSelectDeviceResp)
   {
      (m_rInputCallbacks.fvSelectDeviceResp)(cou32DevId, e8NO_ERRORS);
   } //if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb

   if(e8DEVCONNREQ_DESELECT == coenConnReq && true == m_HardKeysUpdate)
   {
      vSetUpdatedKeyInfo();
      m_HardKeysUpdate = false;
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclDiPoInputHandler::vRegisterInputCallbacks()
 ***************************************************************************/
t_Void spi_tclDiPoInputHandler::vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks)
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vRegisterInputCallbacks entered"));
   //Copy
   m_rInputCallbacks = corfrInputCallbacks;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessKnobKeyEvents()
 ***************************************************************************/
t_Void spi_tclDiPoInputHandler::vProcessKnobKeyEvents(t_U32 u32DeviceHandle, t_S8 s8EncoderDeltaCnt) const
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vProcessKnobKeyEvents entered"));
   ETG_TRACE_USR2(("[DESC]:vProcessKnobKeyEvents: Device Id-0x%x Encoder Delta Count-%d", u32DeviceHandle, s8EncoderDeltaCnt));
   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);

   if (NULL != spoDiPOCmdInput)
   {
      tenKeyMode enKeyMode = e8KEY_RELEASE;
      spoDiPOCmdInput->vSendKeyEvent(enKeyMode, e32INVALID_KEY, e8_KNOB_TYPE, s8EncoderDeltaCnt);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::vOnSessionMsg()
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                        tenDiPOSessionTransport enSessionTransport,
                        t_String szSessionIPAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vOnSessionMsg entered"));
   SPI_INTENTIONALLY_UNUSED(enSessionTransport);
   SPI_INTENTIONALLY_UNUSED(szSessionIPAddress);

   if (e8DIPO_SESSION_END == enDiPOSessionState)
   {
      m_bIsSessionActive = false;
   }
   else if (e8DIPO_SESSION_START == enDiPOSessionState)
   {
      m_bIsSessionActive = true;
   } //if (e8DIPO_SESSION_START == enDiPOSessionState)
   else if (e8DIPO_PLUGIN_LOADED == enDiPOSessionState)
   {
      std::set<tenKeyCode> rsetHardKeys;
      std::set<tenKeyCode> rsetSoftKeys;
      if(NULL != m_poInputHandlerSettings)
      {
         //Set HardKeys
         m_poInputHandlerSettings->bGetKeyInfo(rsetHardKeys,e8DEV_TYPE_DIPO);

         //Get the configuration values for the input adapter and set it accordingly
         std::vector<t_U8> enConsumerKey;
         std::vector<t_U8> enPhoneKey;
         std::vector<t_U8> enKnobKey;
         std::vector<t_U8> u8KeyMapBitValues;
         std::vector<t_U8> u8TelKeyMapBitValues;
         std::vector<t_U8> u8KnobKeyMapBitValues;
         t_Bool bITCommanderSupport = false;
         t_Bool bRotaryKnobSupport = false;

         if (NULL != m_poDiPoKeyCodeUtils)
         {
             m_poDiPoKeyCodeUtils->vGetKeySupported(enConsumerKey, enPhoneKey, enKnobKey,
                                 u8KeyMapBitValues, u8TelKeyMapBitValues,u8KnobKeyMapBitValues,rsetHardKeys,
                                 bITCommanderSupport,bRotaryKnobSupport);
         }
         spi_tclDiPOManager* poDiPOManager = spi_tclDiPOManager::getInstance();
         if(NULL != poDiPOManager)
         {
            spi_tclDiPOCmdSession* poDiPOCmdSession = poDiPOManager->poGetDiPOSessionInstance();
            if(NULL != poDiPOCmdSession)
            {
               poDiPOCmdSession->vSetKeySupported(enConsumerKey, enPhoneKey,
                                                       enKnobKey,u8KeyMapBitValues,
                                                       u8TelKeyMapBitValues, u8KnobKeyMapBitValues,
                                                       bITCommanderSupport,bRotaryKnobSupport);
            }
         }

         // Set Soft Keys
         m_poInputHandlerSettings->bGetSoftKeyInfo(rsetSoftKeys,e8DEV_TYPE_DIPO);
         spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
         if (NULL != poExtCompMgr)
         {
            spi_tclExtCmdInputIntf* poExtCmdInputIntf = poExtCompMgr->poGetCmdInputInst();
            if (NULL != poExtCmdInputIntf)
            {
                poExtCmdInputIntf->bSetCallControlConfig(rsetSoftKeys);
            }
         }
       }
     }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::bCheckHardKeyValidity()
***************************************************************************/
t_Bool spi_tclDiPoInputHandler::bCheckHardKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeys,
                                                  std::set<tenKeyCode>& rfrsetUnSuppkeys) const
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::bCheckHardKeyValidity entered"));
   t_Bool bretVal = true;
   // Check keys received from client with DiPo supported keys if there is
   // any mismatch fill it in a set
   std::set_difference(rfrsetSuppkeys.begin(),rfrsetSuppkeys.end(),
                       m_setenDiPoSuppKeys.begin(),m_setenDiPoSuppKeys.end(),
                       std::inserter(rfrsetUnSuppkeys,rfrsetUnSuppkeys.end()));

   if(scou8MaxKeys < rfrsetSuppkeys.size() || 0 < rfrsetUnSuppkeys.size())
   {
      bretVal = false;
   }
   return bretVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::bCheckSoftKeyValidity()
***************************************************************************/
t_Bool spi_tclDiPoInputHandler::bCheckSoftKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeys,
                                                  std::set<tenKeyCode>& rfrsetUnSuppkeys) const
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::bCheckSoftKeyValidity entered"));
   t_Bool bretVal = true;
   // Check keys received from client with DiPo supported soft keys if there is
   // any mismatch fill it in a set
   std::set_difference(rfrsetSuppkeys.begin(),rfrsetSuppkeys.end(),
                       m_setenDiPoPhoneKeys.begin(),m_setenDiPoPhoneKeys.end(),
                       std::inserter(rfrsetUnSuppkeys,rfrsetUnSuppkeys.end()));

   if(scou8MaxKeys < rfrsetSuppkeys.size() || 0 < rfrsetUnSuppkeys.size())
   {
      bretVal = false;
   }
   return bretVal;

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::vUpdateKeyInfo
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vUpdateKeyInfo()
{

   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vUpdateKeyInfo entered"));
   if(true == m_bIsSessionActive)
   {
     //Update during deselection
     m_HardKeysUpdate = true;
   }
   else
   {
      vSetUpdatedKeyInfo();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::vUpdateKeyInfo
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vUpdateSoftKeyInfo()
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vUpdateSoftKeyInfo entered"));
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   if (NULL != poExtCompMgr)
   {
      spi_tclExtCmdInputIntf* poExtCmdInputIntf = poExtCompMgr->poGetCmdInputInst();
      if (NULL != poExtCmdInputIntf)
      {
         std::set<tenKeyCode> rsetKeyCodes;
         if(NULL != m_poInputHandlerSettings)
         {
           m_poInputHandlerSettings->bGetSoftKeyInfo(rsetKeyCodes,e8DEV_TYPE_DIPO);
           m_setenDiPoSoftKeys = rsetKeyCodes;
           poExtCmdInputIntf->bSetCallControlConfig(rsetKeyCodes);
        }
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessnValidate
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vProcessnValidate(t_U32 u32DeviceHandle,tenKeyMode enKeyMode, tenKeyCode enKeyCode) const
{
   SPI_INTENTIONALLY_UNUSED(enKeyMode);
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vProcessnValidate entered"));
   if(m_setenDiPoPhoneKeys.find(enKeyCode) != m_setenDiPoPhoneKeys.end())
   {
     spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
     if (NULL != poExtCompMgr)
     {
        spi_tclExtCmdInputIntf* poExtCmdInputIntf = poExtCompMgr->poGetCmdInputInst();
        if (NULL != poExtCmdInputIntf)
        {
           switch(enKeyCode)
           {
            case e32DEV_PHONE_CALL:
                 if (NULL != poExtCmdInputIntf)
                 {
                    if(m_setenDiPoSoftKeys.find(e32DEV_PHONE_CALL) != m_setenDiPoSoftKeys.end())
                    {
                       poExtCmdInputIntf->bDiPOAcceptCall(u32DeviceHandle, e8_ACCEPT_OR_HOLD_AND_ACCEPT);
                    }
                 }
                 break;
            case e32DEV_PHONE_END:
                 if (NULL != poExtCmdInputIntf)
                 {
                    if(m_setenDiPoSoftKeys.find(e32DEV_PHONE_END) != m_setenDiPoSoftKeys.end())
                    {
                       poExtCmdInputIntf->bDiPOEndCall(u32DeviceHandle, e8_END_OR_DECLINE);
                    }
                 }
                 break;
            default:
                 break;
          }
        }
     }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoInputHandler::vSetUpdatedKeyInfo
***************************************************************************/
t_Void spi_tclDiPoInputHandler::vSetUpdatedKeyInfo()
{
   ETG_TRACE_USR1(("spi_tclDiPoInputHandler::vSetUpdatedKeyInfo entered"));
   tenDiPOSessionState enDiPOSessionState = e8DIPO_PLUGIN_LOADED;
   tenDiPOSessionTransport enSessionTransport = e8_DIPO_SESSION_TRANSPORT_UNKNOWN;
   t_String szSessionIPAddress = "0.0.0.0";
   vOnSessionMsg(enDiPOSessionState,
                 enSessionTransport,
                 szSessionIPAddress);
}
