/*!
 *******************************************************************************
 * \file              spi_tclDiPoInputHandler.h
 * \brief             SPI input handler for DiPo devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Input handler class to send input events from DiPo client to DiPo server
 AUTHOR:        Hari Priya E R (RBEI/ECP2)
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 25.11.2013 |  Hari Priya E R              | Initial Version
 10.03.2014 |  Hari Priya E R              | Touch handling Implementation
 04.04.2014 |  Hari Priya E R              | Key handling Implementation
 03.07.2014 |  Hari Priya E R              | Added changes related to Input Response interface
 17.07.2015 |  Sameer Chandra              | Added knob key support.

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDIPOINPUTHANDLER_H_
#define SPI_TCLDIPOINPUTHANDLER_H_

/******************************************************************************
 | includes:
 | 1)RealVNC sdk - includes
 | 2)Typedefines
 |----------------------------------------------------------------------------*/
#include "Timer.h"
#include "DiPOTypes.h"
#include "spi_tclInputDevBase.h"
#include "spi_tclDiPORespSession.h"
#include "spi_tclDiPoKeys.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/


/*!
 * \class spi_tclDiPoInputHandler
 * \brief provides input handler to send input events from DiPo client to DiPo server
 *
 */
class spi_tclDiPoInputHandler: public spi_tclInputDevBase,public spi_tclDiPORespSession
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoInputHandler::spi_tclDiPoInputHandler
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoInputHandler()
   * \brief   Parameterised constructor
   * \sa      ~spi_tclDiPoInputHandler()
   **************************************************************************/
   spi_tclDiPoInputHandler();

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoInputHandler::~spi_tclDiPoInputHandler()
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoInputHandler()
   * \brief   destructor
   * \sa      spi_tclDiPoInputHandler()
   **************************************************************************/
   ~spi_tclDiPoInputHandler();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoInputHandler::bInitialize()
   ***************************************************************************/
   /*!^M
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the Dipo Input Handler required things
   * \retval  t_Bool
   * \sa      binitialize()
   **************************************************************************/
   t_Bool bInitialise();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vUnInitialize()
   ***************************************************************************/
   /*!^M
   * \fn      t_Void vUnInitialize()
   * \brief   To Uninitialize all the Dipo Input Handler required things
   * \retval  t_Void
   * \sa      vUnInitialize()
   **************************************************************************/
   t_Void vUnInitialize();


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessTouchEvent
   ***************************************************************************/
   /*!
   * \fn      vProcessTouchEvent()
   * \brief   Receives the Touch events and forwards it through IPC to DiPO Server 
   * \param   u32DeviceHandle  :[IN] unique identifier to ML Server
   * \param   rfrTouchData     :[IN] reference to touch data structure which contains
   *          touch details received /ref trTouchData
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessTouchEvent(t_U32 u32DeviceHandle,trTouchData &rfrTouchData)const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessKeyEvents
   ***************************************************************************/
   /*!
   * \fn      vProcessKeyEvents()
   * \brief   Receives hard key events and forwards it through IPC to DiPO Server
   * \param   u32DeviceHandle :[IN] unique identifier to ML Server
   * \param   enKeyMode       :[IN] indicates keypress or keyrelease
   * \param   enKeyCode       :[IN] unique key code identifier
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessKeyEvents(t_U32 u32DeviceHandle, tenKeyMode enKeyMode,
      tenKeyCode enKeyCode)const;

   /***************************************************************************
   ** FUNCTION: virtual t_Void spi_tclMLInputHandler::vSelectDevice()
   ***************************************************************************/
  /*!
   * \fn      t_Void vSelectDevice(const t_U32 cou32DevId,
   *                 const tenDeviceConnectionReq coenConnReq)
   * \brief   To setup Video related info when a device is selected or
   *          de selected.
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    coenConnReq : [IN] Identifies the Connection Type.
   * \retval  t_Void
   **************************************************************************/
   t_Void  vSelectDevice(const t_U32 cou32DevId,
      const tenDeviceConnectionReq coenConnReq);
  /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclMLInputHandler::vRegisterVideoCallbacks()
   ***************************************************************************/
  /*!
   * \fn      t_Void vRegisterInputCallbacks(const trVideoCallbacks& corfrVideoCallbacks)
   * \brief   To Register for the asynchronous responses that are required from
   *          ML/DiPo Video
   * \param   corfrVideoCallbacks : [IN] Video callabcks structure
   * \retval  t_Void
   **************************************************************************/
   t_Void  vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks);
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::vProcessKnobKeyEvents
   ***************************************************************************/
   /*!
   * \fn      vProcessKnobKeyEvents(t_U32 u32DeviceHandle,t_S8 s8EncoderDeltaCnt
   * \brief   Receives Knob key enocder change and forwards it to
   *          further handlers for processing
   * \param   u32DeviceHandle : [IN] unique identifier to CP Server
   * \param   s8EncoderDeltaCount : [IN] encoder delta count
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessKnobKeyEvents(t_U32 u32DeviceHandle,t_S8 s8EncoderDeltaCnt)
      const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::bCheckHardKeyValidity
   ***************************************************************************/
   /*!
   * \fn      bCheckHardKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeyCodes
   *                            std::set<tenKeyCode> rfrsetUnSuppkeyCodes
   * \brief   Validates DiPo Supported Keys
   * \param   rfrsetSuppkeys : [IN] Supported Key Codes sent by client
   * \param   rfrsetUnSuppkeys : [OUT] UnSupported Key codes by DiPo
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bCheckHardKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeys,
                            std::set<tenKeyCode>& rfrsetUnSuppkeys) const;
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoInputHandler::bCheckSoftKeyValidity
   ***************************************************************************/
   /*!
   * \fn      bCheckSoftKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeyCodes
   *                            std::set<tenKeyCode> rfrsetUnSuppkeyCodes
   * \brief   Validates DiPo Supported Keys
   * \param   rfrsetSuppkeys : [IN] Supported Key Codes sent by client
   * \param   rfrsetUnSuppkeys : [OUT] UnSupported Key codes by DiPo
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bCheckSoftKeyValidity(const std::set<tenKeyCode>& rfrsetSuppkeys,
                            std::set<tenKeyCode>& rfrsetUnSuppkeys) const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputDevBase::vUpdateKeyInfo
   ***************************************************************************/
   /*!
   * \fn      vUpdateKeyInfo()
   * \brief   Updates Key Info during runtime for carplay technology
   * \retval  t_Void
   **************************************************************************/
   t_Void vUpdateKeyInfo();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputDevBase::vUpdateSoftKeyInfo
   ***************************************************************************/
   /*!
   * \fn      vUpdateSoftKeyInfo()
   * \brief   Updates SoftKey Info during runtime for carplay technology
   * \retval  t_Void
   **************************************************************************/
   t_Void vUpdateSoftKeyInfo();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/


   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE***********************************
   ***************************************************************************/

   //! Dipo Supported Keys
   std::set <tenKeyCode> m_setenDiPoSuppKeys;

   //! Dipo Suported Phone Keys
   std::set <tenKeyCode> m_setenDiPoPhoneKeys;

   //! Dipo Soft Keys
   std::set <tenKeyCode> m_setenDiPoSoftKeys;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOInputHandler::vOnSessionMsg(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
   *                                       tenDiPOSessionTransport enSessionTransport,
   *                                       t_String szSessionIPAddress)
   * \brief   To Post the session message to SPI, when the session state changes.
   * \param   enDiPOSessionState : [IN] Session State : Session Started,Session_Ended,
   *                                                    CarPlayd_PlugIn_Loaded or UnLoaded.
   * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
   *                                    Applicable, only when SessionState is Session Started/ended.
   * \param   szSessionIPAddress : [IN] IP Address of the active carplay device.
   *                                    used, only if the Session Transport is WiFi
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                        tenDiPOSessionTransport enSessionTransport,
                        t_String szSessionIPAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputDevBase::vProcessnValidate
   ***************************************************************************/
   /*!
   * \fn      vProcessnValidate(tenKeyMode enKeyMode, tenKeyCode enKeyCode)
   * \brief   Validates and process the soft key events.
   * \param   u32DeviceHandle : unique identifier to CP Server
   * \param   enKeyMode       :[IN] indicates keypress or keyrelease
   * \param   enKeyCode       :[IN] unique key code identifier
   * \retval  t_Void
   **************************************************************************/
   t_Void vProcessnValidate(t_U32 u32DeviceHandle,tenKeyMode enKeyMode, tenKeyCode enKeyCode) const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOInputHandler::vSetUpdatedKeyInfo
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetUpdatedKeyInfo()
   * \brief   To Set the updated KeyInfo 
   *
   * \retval  t_Void
   ***************************************************************************/
   t_Void vSetUpdatedKeyInfo();

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoInputHandler::spi_tclDiPoInputHandler
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoInputHandler(const spi_tclMLInputHandler &corfrSrc)
   * \brief   Copy constructor, will not be implemented.
   * \note    This is a technique to disable the Copy constructor for this class.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclDiPoInputHandler(const spi_tclDiPoInputHandler& corfrSrc);

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoInputHandler& spi_tclDiPoInputHandler::operator= 
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoInputHandler& operator= (const spi_tclDiPoInputHandler &corfrSrc)
   * \brief   Assignment Operator, will not be implemented.
   * \note    This is a technique to disable the assignment operator for this class.
   *          So if an attempt for the assignment is made linker complains.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclDiPoInputHandler& operator=(const spi_tclDiPoInputHandler& corfrSrc);


   //! Input Callbacks structure
   trInputCallbacks m_rInputCallbacks;

   //! Pointer to Dipo Keys Utility;
   spi_tclDiPoKeys *m_poDiPoKeyCodeUtils;
   
   //! Member variable to Check whether HardKeyInfo needs to be updated or not
   t_Bool m_HardKeysUpdate;

   t_Bool m_bIsSessionActive;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

};

#endif /* SPI_TCLDIPOINPUTHANDLER_H_ */
