/***********************************************************************/
/*!
 * \file  spi_tclDiPOCmdSession.cpp
 * \brief Adapter class to send session data to CarPlayd process.
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Adapter class to send session data to CarPlayd process.
 AUTHOR:         Rachana L Achar
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 16.02.2018  | Rachana L Achar       | Initial Version

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "BaseTypes.h"
#include "IPCMessageQueue.h"
#include "SPITypes.h"
#include "DiPOTypes.h"
#include "DiPOWrapperTypes.h"
#include "spi_tclDiPOCmdSession.h"

#ifdef VARIANT_S_FTR_ENABLE_DIPOSPI
#include "spi_tclDiPOStartUp.h"
#include "CarPlayMain.h"
#endif

#include "spi_tclDiPOWrapperMsgQInterface.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DIPO
#include "trcGenProj/Header/spi_tclDiPOCmdSession.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | typedefs (scope: module-global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*******************************************************************************
 * \FUNCTION   : tVoid vOnLoadSettings()
 *******************************************************************************/
t_Void spi_tclDiPOCmdSession::vOnLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vOnLoadSettings()"));
}

t_Void spi_tclDiPOCmdSession::vOnLoadSettingsCompleted()
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vOnLoadSettingsCompleted()"));
#ifdef VARIANT_S_FTR_ENABLE_DIPOSPI
   spi_tclDiPOStartUp* p_DiPOStartUp = spi_tclDiPOStartUp::getInstance();
   if(p_DiPOStartUp)
   {
      p_DiPOStartUp->carplayLibraryEntryPoint();
   }
   if(false == m_bIsCarPlayMainThreadActive)
   {
      t_S32 s32Result = startCarPlay();
      if(0 != s32Result)
      {
         ETG_TRACE_ERR(("spi_tclDiPOCmdSession::vOnLoadSettingsCompleted() - startCarPlay error = %d", s32Result));
      }
      else
      {
         m_bIsCarPlayMainThreadActive = true;
         ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vOnLoadSettingsCompleted() - startCarPlay successful"));
      }
   }
#endif
}

/*******************************************************************************
 * \FUNCTION   : tVoid vOnSaveSettings()
 *******************************************************************************/
t_Void spi_tclDiPOCmdSession::vOnSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vOnSaveSettings()"));
#ifdef VARIANT_S_FTR_ENABLE_DIPOSPI
   spi_tclDiPOStartUp* p_DiPOStartUp = spi_tclDiPOStartUp::getInstance();
   if(p_DiPOStartUp)
   {
      p_DiPOStartUp->carplayLibraryExitPoint();
   }
#endif
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOCmdSession::spi_tclDiPOCmdSession()
 ***************************************************************************/
spi_tclDiPOCmdSession::spi_tclDiPOCmdSession()
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::spi_tclDiPOCmdSession()"));
   m_bIsCarPlayMainThreadActive = false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOCmdInput::~spi_tclDiPOCmdInput()
 ***************************************************************************/
spi_tclDiPOCmdSession::~spi_tclDiPOCmdSession()
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::~spi_tclDiPOCmdSession()"));
#ifdef VARIANT_S_FTR_ENABLE_DIPOSPI
   stopCarPlay();
#endif

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOCmdSession::vSendLastModeData()
 ***************************************************************************/
t_Void spi_tclDiPOCmdSession::vSendLastModeData(t_Bool bAudioLastMode, t_Bool bDisplayLastMode)
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vSendLastModeData() - Audio last mode: %d"
            "Display last mode: %d", ETG_ENUM(BOOL, bAudioLastMode), ETG_ENUM(BOOL, bDisplayLastMode)));

   trLastModeInfoMsg oLastModeInfoMsg;
   oLastModeInfoMsg.enMsgType = e8LASTMODEINFO_MESSAGE;
   oLastModeInfoMsg.bAudioLastMode = bAudioLastMode;
   oLastModeInfoMsg.bDisplayLastMode = bDisplayLastMode;
   spi_tclDiPOWrapperMsgQInterface* p_DiPOWrapperMsgQInterface = spi_tclDiPOWrapperMsgQInterface::getInstance();
   if (p_DiPOWrapperMsgQInterface)
   {
      p_DiPOWrapperMsgQInterface->bWriteMsgToQ(&oLastModeInfoMsg, sizeof(trLastModeInfoMsg));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOCmdSession::vSetKeySupported()
 ***************************************************************************/
t_Void spi_tclDiPOCmdSession::vSetKeySupported(std::vector<t_U8> enConsumerKey, std::vector<t_U8> enPhoneKey,
         std::vector<t_U8> enKnobKey, std::vector<t_U8> u8KeyMapBitValues, std::vector<t_U8> u8TelKeyMapBitValues,
         std::vector<t_U8> u8KnobKeyMapBitValues, t_Bool bIsKnobDirectionalPadEnabled,t_Bool bIsRotaryKnobSupported)
{
   ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vSetKeySupported() - Entered"));
   {
      trSupportedKeyMsg oSupportedKeyMsg;
      //The consumer key descripter along with the bitmap array are merged together in a single native char array as the data part in the IPC message.
      //The respective size of the descriptor and the bitmap are also set accordingly to the variables
      oSupportedKeyMsg.enMsgType = e8SUPPORTEDKEY_CONSKEY_MESSAGE;
      for (size_t k = 0; k < enConsumerKey.size(); k++)
      {
         //The descriptor data is copied in the first half of the char array
         oSupportedKeyMsg.u8Key[k] = enConsumerKey[k];
      }
      for (size_t k = 0; k < u8KeyMapBitValues.size(); k++)
      {
         //The corresponding bitmap array is concatenated with the above created data
         oSupportedKeyMsg.u8Key[enConsumerKey.size() + k] = u8KeyMapBitValues[k];
      }
      oSupportedKeyMsg.sKeySize = enConsumerKey.size(); //consumer descriptor size
      oSupportedKeyMsg.sKeyBitmapSize = u8KeyMapBitValues.size(); //consumer bitmap descriptor size
      ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vSetKeySupported() - sKeySize(%d), sKeyBitmapSize(%d)", oSupportedKeyMsg.sKeySize, oSupportedKeyMsg.sKeyBitmapSize));

      spi_tclDiPOWrapperMsgQInterface* p_DiPOWrapperMsgQInterface = spi_tclDiPOWrapperMsgQInterface::getInstance();
      if (p_DiPOWrapperMsgQInterface)
      {
         t_Bool bMessageSendStatus = p_DiPOWrapperMsgQInterface->bWriteMsgToQ(&oSupportedKeyMsg,
                  sizeof(trSupportedKeyMsg));
         ETG_TRACE_USR4(("spi_tclDiPOCmdSession::vSetKeySupported() - message send status(%d)", bMessageSendStatus));
      }
   }

   {
      trSupportedKeyMsg oSupportedKeyMsg;
      //The phone key descripter along with the bitmap array are merged together in a single native char array as the data part in the IPC message.
      //The respective size of the descriptor and the bitmap are also set accordingly to the variables
      oSupportedKeyMsg.enMsgType = e8SUPPORTEDKEY_TELKEY_MESSAGE;
      for (size_t k = 0; k < enPhoneKey.size(); k++)
      {
         //The descriptor data is copied in the first half of the char array
         oSupportedKeyMsg.u8Key[k] = enPhoneKey[k];
      }
      for (size_t k = 0; k < u8TelKeyMapBitValues.size(); k++)
      {
         //The corresponding bitmap array is concatenated with the above created data
         oSupportedKeyMsg.u8Key[enPhoneKey.size() + k] = u8TelKeyMapBitValues[k];
      }
      oSupportedKeyMsg.sKeySize = enPhoneKey.size(); //phone descriptor size
      oSupportedKeyMsg.sKeyBitmapSize = u8TelKeyMapBitValues.size(); //phone bitmap descriptor size
      ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vSetKeySupported() - sKeySize(%d), sKeyBitmapSize(%d)", oSupportedKeyMsg.sKeySize, oSupportedKeyMsg.sKeyBitmapSize));
      spi_tclDiPOWrapperMsgQInterface* p_DiPOWrapperMsgQInterface = spi_tclDiPOWrapperMsgQInterface::getInstance();
      if (p_DiPOWrapperMsgQInterface)
      {
         t_Bool bMessageSendStatus = p_DiPOWrapperMsgQInterface->bWriteMsgToQ(&oSupportedKeyMsg,
                  sizeof(trSupportedKeyMsg));
         ETG_TRACE_USR4(("spi_tclDiPOCmdSession::vSetKeySupported() - message send status(%d)", bMessageSendStatus));
      }
   }

   {
      trSupportedKeyMsg oSupportedKeyMsg;
      //The knob key descripter along with the bitmap array are merged together in a single native char array as the data part in the IPC message.
      //The respective size of the descriptor and the bitmap are also set accordingly to the variables
      oSupportedKeyMsg.enMsgType = e8SUPPORTEDKEY_KNOBKEY_MESSAGE;
      for (size_t k = 0; k < enKnobKey.size(); k++)
      {
         //The descriptor data is copied in the first half of the char array
         oSupportedKeyMsg.u8Key[k] = enKnobKey[k];
      }
      for (size_t k = 0; k < u8KnobKeyMapBitValues.size(); k++)
      {
         //The corresponding bitmap array is concatenated with the above created data
         oSupportedKeyMsg.u8Key[enKnobKey.size() + k] = u8KnobKeyMapBitValues[k];
      }
      oSupportedKeyMsg.sKeySize = enKnobKey.size(); //knob descriptor size
      oSupportedKeyMsg.sKeyBitmapSize = u8KnobKeyMapBitValues.size(); //knob bitmap descriptor size
      oSupportedKeyMsg.bIsKnobDirectionalPadEnabled = bIsKnobDirectionalPadEnabled;
      oSupportedKeyMsg.bIsRotaryKnobSupported = bIsRotaryKnobSupported;
      ETG_TRACE_USR1(("spi_tclDiPOCmdSession::vSetKeySupported() - sKeySize(%d), sKeyBitmapSize(%d), bIsKnobDirectionalPadEnabled(%d)", oSupportedKeyMsg.sKeySize, oSupportedKeyMsg.sKeyBitmapSize, bIsKnobDirectionalPadEnabled));
      spi_tclDiPOWrapperMsgQInterface* p_DiPOWrapperMsgQInterface = spi_tclDiPOWrapperMsgQInterface::getInstance();
      if (p_DiPOWrapperMsgQInterface)
      {
         t_Bool bMessageSendStatus = p_DiPOWrapperMsgQInterface->bWriteMsgToQ(&oSupportedKeyMsg,
                  sizeof(trSupportedKeyMsg));
         ETG_TRACE_USR4(("spi_tclDiPOCmdSession::vSetKeySupported() - message send status(%d)", bMessageSendStatus));
      }
   }
}

/////////////////////////////////////<EOF>//////////////////////////////////////////
