/*!
 *******************************************************************************
 * \file             spi_tclDipoDataService.cpp
 * \brief            DiPO Data Service class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO Data Service class implements Data Service Info Management for
 CarPlay capable devices. This class must be derived from base Data Service class.
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      		| Modifications
 27.03.2014 |  Ramya Murthy                		| Initial Version
 14.04.2014 |  Ramya Murthy                		| Implemented sending GPS data to MediaPlayer client.
 13.06.2014 |  Ramya Murthy                		| Implementation for:
                                             	 (1) MPlay FI extn to start/stop loc info
                                             	 (2) VDSensor data integration
                                             	 (3) NMEA-PASCD sentence for DiPO
 13.10.2014 |  Hari Priya E R (RBEI/ECP2)       | Added interface to get Vehicle Data for PASCD.
 23.04.2015 |  Ramya Murthy (RBEI/ECP2)         | Implementation to provide PASCD & PAGCD sentences (for China region)
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)         | Added Lint comments to suppress C++11 Errors
 16.03.2016 |  Ramya Murthy                     | Revised implementation of data subscription
 31.03.2016 |  Ramya Murthy                     | Implementation to support Sensors mode (PASCD) for all projects and 
                                                  determine GNSS/Sensors mode from StartLocationData request.

 \endverbatim
 ******************************************************************************/
#include <algorithm>

#include "NmeaEncoder.h"
#include "spi_tclDipoDataService.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdNavData.h"
#include "spi_tclDeviceIDDataIntf.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
      #include "trcGenProj/Header/spi_tclDipoDataService.cpp.trc.h"
   #endif
#endif


//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
static const t_U32 scou32SensorsModeUpdateTimerVal = 1000;
/*  GPRMCDataStatusValues have been declared as t_U8 to avoid
 *  Static code analyzer complaint for not matching data type */
static const t_U8 scou8GPRMCDataStatusValueA = 1;
static const t_U8 scou8GPRMCDataStatusValueV = 2;
static const t_U8 scou8GPRMCDataStatusValueX = 4;
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static timer_t srSensorsModeTimerID;
static t_Bool bSensorsModeTimerRunning = false;

/****************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDipoDataService::spi_tclDipoDataService
 ***************************************************************************/
spi_tclDipoDataService::spi_tclDipoDataService(const trDataServiceCb& rfcorDataServiceCb)
   : m_u32SelDevHandle(0),
     m_bIsLocDataTransferEnabled(false),
     m_rDataServiceCb(rfcorDataServiceCb),
     m_enGeoCoordinateSystemType(e8GEO_COORDINATE_SYSTEM_WGS)
{
   ETG_TRACE_USR1(("spi_tclDipoDataService() entered \n"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDipoDataService::spi_tclDipoDataService
 ***************************************************************************/
spi_tclDipoDataService::~spi_tclDipoDataService()
{
   ETG_TRACE_USR1(("~spi_tclDipoDataService() entered \n"));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDipoDataService::bInitialise()
 ***************************************************************************/
t_Bool spi_tclDipoDataService::bInitialise()
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::bInitialise() entered \n"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDipoDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclDipoDataService::bUninitialise()
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::bUninitialise() entered \n"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDipoDataService::vOnSelectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclDipoDataService::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
         const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::vOnSelectDeviceResult() entered: u32DeviceHandle = %u \n", u32DeviceHandle));
   SPI_INTENTIONALLY_UNUSED(coenConnReq);
   SPI_INTENTIONALLY_UNUSED(coenRespCode);
   //! Store selected device's handle
   m_u32SelDevHandle = u32DeviceHandle;

   m_szSelDevSerialNo.clear();
   spi_tclDeviceIDDataIntf oDataIntf;
   oDataIntf.vGetSerialNumber(m_szSelDevSerialNo, m_u32SelDevHandle);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDipoDataService::vOnDeselectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclDipoDataService::vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::vOnDeselectDeviceResult() entered: u32DeviceHandle = %u \n", u32DeviceHandle));

   //! Clear data
   if (u32DeviceHandle == m_u32SelDevHandle)
   {
      vStopLocationData(m_NmeaSentencesList);

      m_rDipoDataServiceConfigDatalock.s16Lock();
      // Resetting the coordinate system to default
      m_enGeoCoordinateSystemType = m_rDipoDataServiceConfigData.enGeoCoordinateSystemType;
      m_rDipoDataServiceConfigDatalock.vUnlock();

      //! Clear selected device handle
      m_u32SelDevHandle = 0;
      m_szSelDevSerialNo.clear();
   } //if (u32DeviceHandle == m_u32SelDevHandle)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDipoDataService::vStartLocationData(...)
 ***************************************************************************/
t_Void spi_tclDipoDataService::vStartLocationData(
      const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList, t_U8 u8RequestedBy)
{
   SPI_INTENTIONALLY_UNUSED(u8RequestedBy);
   ETG_TRACE_USR1(("spi_tclDipoDataService::vStartLocationData() entered: rfcoNmeaSentencesList size = %u \n",
         rfcoNmeaSentencesList.size()));

   m_bIsLocDataTransferEnabled = true;

   m_NmeaSentencesListlock.s16Lock();
   //Store complete Nmea sentences list. Only the requested sentences (if supported)
   //will be transferred. (In future, may need to support individual sentences transfer).
   m_NmeaSentencesList = rfcoNmeaSentencesList;
   m_NmeaSentencesListlock.vUnlock();

   //Register for Location data
   vSetDataSubscription(true);

   if (true == bIsSensorsModeSupported())
   {
      vStartSensorsModeUpdate();
   }//if (true == bIsSensorsModeSupported())
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDipoDataService::bStopLocationData(...)
 ***************************************************************************/
t_Void spi_tclDipoDataService::vStopLocationData(
      const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList,t_U8 u8RequestedBy)
{
   SPI_INTENTIONALLY_UNUSED(u8RequestedBy);

   m_bIsLocDataTransferEnabled = false;

   vStopSensorsModeUpdate();

   //Unregister for Location data
   vSetDataSubscription(false);

   m_NmeaSentencesListlock.s16Lock();
   ETG_TRACE_USR1(("spi_tclDipoDataService::vStopLocationData() entered: rfcoNmeaSentencesList size = %u \n",
         rfcoNmeaSentencesList.size()));
   //Clear Nmea sentences list since currently all sentences transfer is stopped.
   //(In future, may need to support stopping transfer of individual sentences).
   m_NmeaSentencesList.clear();
   m_NmeaSentencesListlock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vOnData(const trGPSData& rfcorGpsData)
***************************************************************************/
t_Void spi_tclDipoDataService::vOnData(const trGPSData& rfcorGpsData)
{
   //ETG_TRACE_USR1(("spi_tclDipoDataService::vOnData() entered \n"));

   //Store the GPS Data
   m_rGPSDatalock.s16Lock();
   m_rGPSData =  rfcorGpsData;
   m_rGPSData.enGeoCoordinateSystemType = m_enGeoCoordinateSystemType;
   m_rGPSDatalock.vUnlock();

   //! If a device is active & LocationData start request is received,
   //! send NMEA-formatted GPS data to Mediaplayer client handler.
   if (
      (0 != m_u32SelDevHandle)
      &&
      (true == m_bIsLocDataTransferEnabled)
      &&
      (false == m_NmeaSentencesList.empty())
      )
   {
      m_rGPSDatalock.s16Lock();
      trGPSData otrGPSData = m_rGPSData;
      m_rGPSDatalock.vUnlock();

      m_rSensorDatalock.s16Lock();
      trSensorData otrSensorData = m_rSensorData;
      m_rSensorDatalock.vUnlock();

      m_rVehicleDatalock.s16Lock();
      trVehicleData otrVehicleData = m_rVehicleData;
      m_rVehicleDatalock.vUnlock();

      NmeaEncoder oNmeaEncoder(otrGPSData, otrSensorData,otrVehicleData);
      t_String szNmeaGGASentence;
      t_String szNmeaRMCSentence;
      //t_String szNmeaPASCDSentence;
      t_String szNmeaPAGCDSentence;

      //! @Note:
      //! Send GPS data if GPGGA, GPRMC, GPGSV and/or GPHT sentences are requested
      //! Send GPS data if PACSD, PAGCD and/or PAACD sentences are requested
      t_Bool bTransferGpsData = false;
      t_Bool bTransferDrData = false;

      m_NmeaSentencesListlock.s16Lock();
      //! Check and get GPGGA sentence if required.
      if (m_NmeaSentencesList.end() !=
            std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPGGA))
      {
         szNmeaGGASentence = oNmeaEncoder.szGetNmeaGGASentence(e8NMEA_SOURCE_GPS);
         bTransferGpsData = true;
      }
      //! Check and get GPRMC sentence if required.
      if (m_NmeaSentencesList.end() !=
            std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPRMC))
      {
         szNmeaRMCSentence = oNmeaEncoder.szGetNmeaRMCSentence(e8NMEA_SOURCE_GPS);
         bTransferGpsData = true;
      }
      //! Check and get PASCD sentence if required.
//      if (m_NmeaSentencesList.end() !=
//         std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_PASCD))
//      {
//         szNmeaPASCDSentence = oNmeaEncoder.szGetNmeaPASCDSentence(true);
//         bTransferDrData = true;
//      }
      //! Check and get PAGCD sentence if required.
      if (m_NmeaSentencesList.end() !=
         std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_PAGCD))
      {
         szNmeaPAGCDSentence = oNmeaEncoder.szGetNmeaPAGCDSentence(true);
         bTransferDrData = true;
      }
      m_NmeaSentencesListlock.vUnlock();

      spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
      if (NULL != poExtCompMgr)
      {
         spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
         if (NULL != poExtCmdNavDataIntf)
         {
            if (true == bTransferGpsData)
            {
               poExtCmdNavDataIntf->bTransferGPSData(0, m_szSelDevSerialNo, szNmeaGGASentence, szNmeaRMCSentence, "", "");
            }
            if (true == bTransferDrData)
            {
               poExtCmdNavDataIntf->bTransferDrData(0, m_szSelDevSerialNo, "", szNmeaPAGCDSentence, "");
            }
         } //if (NULL != poExtCmdNavDataIntf)
      }//if (NULL != poExtCompMgr)
   }//if ((0 != m_u32SelDevHandle)&&...)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vOnData(const trSensorData&...)
***************************************************************************/
t_Void spi_tclDipoDataService::vOnData(const trSensorData& rfcorSensorData)
{
   //ETG_TRACE_USR1(("spi_tclDipoDataService::vOnData() entered \n"));

   //! Store Sensor Data
   //@Note: It will be used along with Location data to provode NMEA data
   //once Location data is received.
   m_rSensorDatalock.s16Lock();
   m_rSensorData = rfcorSensorData;
   m_rSensorDatalock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vOnData(const trVehicleData&...)
***************************************************************************/
t_Void spi_tclDipoDataService::vOnData(const trVehicleData& rfcoVehicleData, t_Bool bSolicited)
{
   //ETG_TRACE_USR1(("spi_tclDipoDataService::vOnData() entered \n"));

   //! Store Vehicle Data
   SPI_INTENTIONALLY_UNUSED(bSolicited);
   m_rVehicleDatalock.s16Lock();
   m_rVehicleData = rfcoVehicleData;
   m_rVehicleDatalock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vSetSensorDataAvailablility(...)
***************************************************************************/
t_Void spi_tclDipoDataService::vSetSensorDataAvailablility(
      const trDataServiceConfigData& rfrDataServiceConfigInfo)
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::vSetSensorDataAvailablility() entered "));

   //! Store config data
   m_rDipoDataServiceConfigDatalock.s16Lock();
   m_rDipoDataServiceConfigData = rfrDataServiceConfigInfo;
   m_enGeoCoordinateSystemType = m_rDipoDataServiceConfigData.enGeoCoordinateSystemType;
   m_rDipoDataServiceConfigDatalock.vUnlock();
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vSetDataSubscription();
***************************************************************************/
t_Void spi_tclDipoDataService::vSetDataSubscription(t_Bool bSubscribe)
{
   /*lint -esym(40,fvSubscribeForData)fvSubscribeForData Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDipoDataService::vSetDataSubscription() entered: bSubscribe = %u \n",
         ETG_ENUM(BOOL, bSubscribe)));

   //! Forward Data subscription/unsubscription request to DataService manager
   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      m_rDataServiceCb.fvSubscribeForData(bSubscribe, e8VEHICLE_DATA);

      if (true == bIsGNSSModeSupported())
      {
         //@Note: Currently only GPS data is being used. In future if DeadReckoning
         //data is required (for PAACD, PAGCD), need to subscribe for DR data.
         t_Bool bIsGPSsupported = m_poDataServiceSettings->bGetCarPlayGPSDataSupportflag();
         if(true == bIsGPSsupported)//Fix for NCG3D-165262
         {
            m_rDataServiceCb.fvSubscribeForData(bSubscribe, e8GPS_DATA);
         }
         m_rDataServiceCb.fvSubscribeForData(bSubscribe, e8GNSS_DATA);
      }
   }//if (NULL != m_rDataServiceCb.fvSubscribeForData)
} //! end of vSetDataSubscription()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDipoDataService::bIsGNSSModeSupported()
***************************************************************************/
t_Bool spi_tclDipoDataService::bIsGNSSModeSupported()
{
   //@Note: It is assumed to be GNSS mode if StartLocationData(GPGGA, GPRMC and/or PAGCD) is received.
   //PASCD is not considered here since it is provided in both GNSS & Sensors mode.

   m_NmeaSentencesListlock.s16Lock();
   t_Bool bGNSSMode =
      (m_NmeaSentencesList.end() != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPGGA)) ||
      (m_NmeaSentencesList.end() != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPRMC)) ||
      (m_NmeaSentencesList.end() != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_PAGCD));
   m_NmeaSentencesListlock.vUnlock();

   ETG_TRACE_USR1(("[DESC]:spi_tclDipoDataService::bIsGNSSModeSupported: %d ", ETG_ENUM(BOOL, bGNSSMode)));

   return bGNSSMode;
} //! end of bIsGNSSModeSupported()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDipoDataService::bIsSensorsModeSupported()
***************************************************************************/
t_Bool spi_tclDipoDataService::bIsSensorsModeSupported()
{
   //@Note: Currently only PASCD is supported for Sensors mode.
   //It is assumed to be Sensors mode if StartLocationData(only PASCD) is received.
   //PASCD sentences has to be sent irrespective of whether GPS data comes or not.

   m_NmeaSentencesListlock.s16Lock();
   t_Bool bSensorsMode = (m_NmeaSentencesList.end() != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_PASCD));
   m_NmeaSentencesListlock.vUnlock();

   ETG_TRACE_USR1(("[DESC]:spi_tclDipoDataService::bIsSensorsModeSupported: %d ", ETG_ENUM(BOOL, bSensorsMode)));
   return bSensorsMode;
} //! end of bIsSensorsModeSupported()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vSendSensorsModeData()
***************************************************************************/
t_Void spi_tclDipoDataService::vSendSensorsModeData()
{
   //! Send PASCD sentence update here only for Sensor mode variants
   if (
      (0 != m_u32SelDevHandle)
      &&
      (true == m_bIsLocDataTransferEnabled)
      /*&&
      (true == bIsSensorsModeSupported())*/
      )
   {
      m_rGPSDatalock.s16Lock();
      trGPSData otrGPSData = m_rGPSData;
      m_rGPSDatalock.vUnlock();

      m_rSensorDatalock.s16Lock();
      trSensorData otrSensorData = m_rSensorData;
      m_rSensorDatalock.vUnlock();

      m_rVehicleDatalock.s16Lock();
      trVehicleData otrVehicleData = m_rVehicleData;
      m_rVehicleDatalock.vUnlock();

      NmeaEncoder oNmeaEncoder(otrGPSData, otrSensorData,otrVehicleData);
      t_Bool bUseGpsData = bIsGNSSModeSupported();
      t_String szNmeaPASCDSentence = oNmeaEncoder.szGetNmeaPASCDSentence(bUseGpsData);

      spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
      if (NULL != poExtCompMgr)
      {
         spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
         if (NULL != poExtCmdNavDataIntf)
         {
            poExtCmdNavDataIntf->bTransferDrData(0, m_szSelDevSerialNo, szNmeaPASCDSentence, "", "");
         }
      }
   }//if ((0 != m_u32SelDevHandle)&&...)
} //! end of vSendSensorsModeData()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vStartSensorsModeUpdate()
***************************************************************************/
t_Void spi_tclDipoDataService::vStartSensorsModeUpdate()
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::vStartSensorsModeUpdate() entered "));

   Timer* poTimer = Timer::getInstance();
   if (true == bSensorsModeTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:vStartSensorsModeUpdate: Sensors Mode Timer already running "));
   }
   else if (NULL != poTimer)
   {
      poTimer->StartTimer(srSensorsModeTimerID, scou32SensorsModeUpdateTimerVal,
            scou32SensorsModeUpdateTimerVal, this,
            &spi_tclDipoDataService::bSensorsModeTimerCb, NULL);
      bSensorsModeTimerRunning = true;

      ETG_TRACE_USR2(("[DESC]:vStartSensorsModeUpdate: Sensors Mode Timer started "));
   }//End of if (NULL != poTimer)

} //! end of vStartSensorsModeUpdate()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDipoDataService::vStopSensorsModeUpdate()
***************************************************************************/
t_Void spi_tclDipoDataService::vStopSensorsModeUpdate()
{
   ETG_TRACE_USR1(("spi_tclDipoDataService::vStopSensorsModeUpdate() entered "));

   Timer* poTimer = Timer::getInstance();
   if ((true == bSensorsModeTimerRunning) && (NULL != poTimer))
   {
      poTimer->CancelTimer(srSensorsModeTimerID);
      bSensorsModeTimerRunning = false;

      ETG_TRACE_USR2(("[DESC]:vStopSensorsModeUpdate: Sensors Mode Timer stopped "));
   }//End of if (NULL != poTimer)

} //! end of vStopSensorsModeUpdate()

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclDipoDataService::bSensorsModeTimerCb
 ***************************************************************************/
t_Bool spi_tclDipoDataService::bSensorsModeTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   //ETG_TRACE_USR1(("spi_tclDipoDataService::bSensorsModeTimerCb entered \n"));

   spi_tclDipoDataService* poDipoDataService = static_cast<spi_tclDipoDataService*> (pvObject);
   if (NULL != poDipoDataService)
   {
      poDipoDataService->vSendSensorsModeData();
   }//if (NULL != poDipoDataService)

   return true;
} //! end of bSensorsModeTimerCb()

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDipoDataService::vSetGeoCoordinateSystem(...)
 ******************************************************************************************************/
t_Void spi_tclDipoDataService::vSetGeoCoordinateSystem(tenGeoCoordinateSystemType enGeoCoordinateSystemType)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDipoDataService::vSetGeoCoordinateSystem() Entered with Geo Coordinate System = %d",enGeoCoordinateSystemType));

   m_rDipoDataServiceConfigDatalock.s16Lock();
   m_enGeoCoordinateSystemType = enGeoCoordinateSystemType;
   m_rDipoDataServiceConfigDatalock.vUnlock();
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostGPRMCDataStatusValues(...)
 ******************************************************************************************************/
t_Void spi_tclDipoDataService::vPostGPRMCDataStatusValues(t_U32 u32DeviceId,t_U8 u8GPRMCDataStatusValues)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDipoDataService::vPostGPRMCDataStatusValues() Entered with Device ID=%d, GPRMC Data Status Values=%d",
            u32DeviceId, u8GPRMCDataStatusValues));

   /**********************************************************************************************************
    * MP_FI <bit name="GPRMCDataStatusValueA" value="0"/> i.e. Decimal Value - 1 : scou8GPRMCDataStatusValueA
    * MP_FI <bit name="GPRMCDataStatusValueV" value="1"/> i.e. Decimal Value - 2 : scou8GPRMCDataStatusValueV
    * MP_FI <bit name="GPRMCDataStatusValueX" value="2"/> i.e. Decimal Value - 4 : scou8GPRMCDataStatusValueX
    *********************************************************************************************************/

   m_rDipoDataServiceConfigDatalock.s16Lock();
   switch(m_rDipoDataServiceConfigData.enGeoCoordinateSystemType)
   {
      case e8GEO_COORDINATE_SYSTEM_WGS :
      {
         m_enGeoCoordinateSystemType = ((scou8GPRMCDataStatusValueA & u8GPRMCDataStatusValues) == scou8GPRMCDataStatusValueA) ?
                                e8GEO_COORDINATE_SYSTEM_WGS : e8GEO_COORDINATE_SYSTEM_UNKNOWN;
      }
         break;
      case e8GEO_COORDINATE_SYSTEM_GCJ :
      {
         m_enGeoCoordinateSystemType = ((scou8GPRMCDataStatusValueX & u8GPRMCDataStatusValues) == scou8GPRMCDataStatusValueX) ?
                               e8GEO_COORDINATE_SYSTEM_GCJ : e8GEO_COORDINATE_SYSTEM_UNKNOWN;
      }
         break;
      default :
         m_enGeoCoordinateSystemType = e8GEO_COORDINATE_SYSTEM_UNKNOWN;
   }
   m_rDipoDataServiceConfigDatalock.vUnlock();
   ETG_TRACE_USR1(("[PARAM]::spi_tclDipoDataService::vPostGPRMCDataStatusValues() Evaluated Coordinate System = %d",
            ETG_ENUM(COORDINATE_SYSTEM,m_enGeoCoordinateSystemType)));

}
//lint -restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
