/*!
 *******************************************************************************
 * \file             spi_tclDiPoConnection.cpp
 * \brief            DiPo Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPo Connection class to handle ios devices capable of DiPo
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 10.01.2013 |  Pruthvi Thej Nagaraju       | Initial Version
 13.02.2013 |  Shihabudheen P M            | Added 1. vOnDiPODeviceConnect
                                                   2. vOnDiPODeviceDisConnect
 31.07.2014 |  Ramya Murthy                | SPI feature configuration via LoadSettings()
 05.11.2014 |  Ramya Murthy                | Added response for Application metadata.
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)    | Added Lint comments to suppress C++11 Errors

 06.05.2015 |  Tejaswini HB                | Lint Fix
 01.03.2017 |  Shiva Kumar G               | R12N Plugin Adaptations
 \endverbatim
 ******************************************************************************/

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include "ahl_if.h"

#include <unistd.h>
#include <cstdlib>
#include "BaseTypes.h"
#include "Timer.h"
#include "spi_tclDiPOCmdDiscoverer.h"
#include "spi_tclDiPOCmdSession.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPoConnection.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "spi_tclMediator.h"

#include "spi_tclExtCmdAppleDiscoverer.h"
#include "spi_tclExtCompManager.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
      #include "trcGenProj/Header/spi_tclDiPoConnection.cpp.trc.h"
   #endif
#endif


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
static spi_tclDiPOManager* spoDiPOManager = NULL;
static spi_tclDiPOCmdDiscoverer* spoDiPOCmdDisc = NULL;
static spi_tclDiPOCmdSession* spoDiPOCmdSession = NULL;
spi_tclDiPoConnection* spi_tclDiPoConnection::m_poDiPOConnection = NULL;

static timer_t stConnectDeviceTimerIndex = 0;
static t_U8 su8NoOfConnectRequests = 0;

static timer_t stSessionTimerIndex = 0;
static const t_U32 scou32SessionTimerInterval = 10000;

//CPW Timers
static const t_U8 scou8NoOfTrials = 16;
static const t_U32 scou32TimerInterval = 1000; //1000ms * 16 times = 15sec re-trial for connect to succeed after device discovery

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/


//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::spi_tclDiPoConnection
 ***************************************************************************/
spi_tclDiPoConnection::spi_tclDiPoConnection():
m_u32CurrSelectedDevice(0),
m_enCarplaySettingStatus(e8USAGE_ENABLED),
m_enConnectTriggerReason(tenConnectTriggerReason::e8CONNECT_UNKNOWN)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::spi_tclDiPoConnection entered"));
   
   m_poDiPOConnection = this;

   spoDiPOManager = spi_tclDiPOManager::getInstance();
   if (NULL != spoDiPOManager)
   {
      spoDiPOCmdDisc = spoDiPOManager->poGetDiPODiscInstance();

	   //Register for the callbacks from the required callbacks.
	   spoDiPOManager->bRegisterObject((spi_tclDiPORespDiscoverer*)this);	
       spoDiPOManager->bRegisterObject((spi_tclDiPORespSession*)this);

   }//if(NULL != spoVncManager )
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::~spi_tclDiPoConnection
 ***************************************************************************/
spi_tclDiPoConnection::~spi_tclDiPoConnection()
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::~spi_tclDiPoConnection entered"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::vOnLoadSettings
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType)
{
   SPI_INTENTIONALLY_UNUSED(rfrheadUnitInfo);
   SPI_INTENTIONALLY_UNUSED(enCertificateType);
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnLoadSettings entered"));
   if (NULL != spoDiPOManager)
   {
      spi_tclDiPOCmdSession* pDiPOCmdSession = spoDiPOManager->poGetDiPOSessionInstance();
      if(pDiPOCmdSession)
      {
         pDiPOCmdSession->vOnLoadSettings();
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::vOnLoadSettingsCompleted
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vOnLoadSettingsCompleted()
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnLoadSettingsCompleted entered"));
   if (NULL != spoDiPOManager)
   {
      spi_tclDiPOCmdSession* pDiPOCmdSession = spoDiPOManager->poGetDiPOSessionInstance();
      if(pDiPOCmdSession)
      {
         pDiPOCmdSession->vOnLoadSettingsCompleted();
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnection::vOnSaveSettings
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vOnSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnSaveSettings entered"));
   m_u32CurrSelectedDevice = 0;
   if (NULL != spoDiPOManager)
   {
      spi_tclDiPOCmdSession* pDiPOCmdSession = spoDiPOManager->poGetDiPOSessionInstance();
      if(pDiPOCmdSession)
      {
         pDiPOCmdSession->vOnSaveSettings();
      }
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::bStartDeviceDetection
 ***************************************************************************/
t_Bool spi_tclDiPoConnection::bStartDeviceDetection()
{
   t_Bool bRetVal = true;

   ETG_TRACE_USR1(("spi_tclDiPoConnection::bStartDeviceDetection bRetVal = %d \n", bRetVal));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::vOnAddDevicetoList
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vOnAddDevicetoList(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnAddDevicetoList entered"));
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::bSetDevProjUsage
 ***************************************************************************/
t_Bool spi_tclDiPoConnection::bSetDevProjUsage(tenEnabledInfo enServiceStatus)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::bSetDevProjUsage entered"));
   m_enCarplaySettingStatus = enServiceStatus;
   return true; //! TODO
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vRegisterCallbacks(trConnCallbacks &ConnCallbacks)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vRegisterCallbacks entered"));
   m_rDiPoConnCallbacks = ConnCallbacks;
}


/***************************************************************************
** FUNCTION:  spi_tclDiPoConnection::vOnSelectDevice()
***************************************************************************/
t_Void spi_tclDiPoConnection::vOnSelectDevice(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionType enDevConnType,
               tenDeviceConnectionReq enDevConnReq, tenEnabledInfo enDAPUsage,
               tenEnabledInfo enCDBUsage, tenSelectReason enSelectionReason,
               const trUserContext corUsrCntxt, tenDeviceType enDeviceType)
{

    SPI_INTENTIONALLY_UNUSED(enSelectionReason);
    SPI_INTENTIONALLY_UNUSED(corUsrCntxt);
    SPI_INTENTIONALLY_UNUSED(enDAPUsage);
    SPI_INTENTIONALLY_UNUSED(enCDBUsage);
    SPI_INTENTIONALLY_UNUSED(enDevConnType);
    SPI_INTENTIONALLY_UNUSED(enDeviceType);

   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnSelectDevice entered"));

   spi_tclDeviceIDDataIntf oDeviceIDIntf;
   t_String szBTAddress;
   oDeviceIDIntf.vGetBTAddress(szBTAddress, cou32DeviceHandle);

   m_u32CurrSelectedDevice = cou32DeviceHandle;
   //Commented the below variables since they are not being used.
   //tenTriggerType enTriggerType = e8DIPO_TRIGGER_SELECT_DEVICE;
   //tenSessionStatus enSessionStatus = e8_SESSION_UNKNOWN; //Dummy value. will not process it
   //tenDeviceConnectionStatus enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;//Dummy value . will not process
   if (enDevConnReq == e8DEVCONNREQ_SELECT)
   {
      vSetInitalModes(szBTAddress);

      if (NULL != spoDiPOCmdDisc)
      {
         //@TODO - Lock to be added
         su8NoOfConnectRequests++;
         m_sczDeviceBTAddr = szBTAddress;
         m_enConnectTriggerReason = tenConnectTriggerReason::e8CONNECT_ON_USER_TRIGGER;
         spoDiPOCmdDisc->vConnectDevice(szBTAddress.c_str());
      }

   }
   else if ((NULL != spoDiPOCmdDisc) && (enDevConnReq == e8DEVCONNREQ_DESELECT))
   {
      spoDiPOCmdDisc->vDisconnectDevice(szBTAddress.c_str(),tenConnectTriggerReason::e8CONNECT_ON_USER_TRIGGER);
   }
}

/***************************************************************************
** FUNCTION:  spi_tclDiPoConnection::vPostSelctDeviceResult()
***************************************************************************/
t_Void spi_tclDiPoConnection::vPostSelectDeviceResult(tenResponseCode enResponse,
    tenErrorCode enErrorType)
{
   SPI_INTENTIONALLY_UNUSED(enResponse);
   SPI_INTENTIONALLY_UNUSED(enErrorType);
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vPostSelectDeviceResult entered"));

}

/***************************************************************************
** FUNCTION:  spi_tclDiPoConnection::vSetRoleSwitchRequestedInfo()
***************************************************************************/
t_Void spi_tclDiPoConnection::vSetRoleSwitchRequestedInfo(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vSetRoleSwitchRequestedInfo entered"));
   m_u32CurrSelectedDevice = cou32DeviceHandle;
}


/***************************************************************************
** FUNCTION:  spi_tclDiPoConnection::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclDiPoConnection::vOnSelectDeviceResult(const t_U32 cou32DeviceHandle,
                                                    const tenDeviceConnectionReq coenConnReq,
                                                    const tenResponseCode coenRespCode,
                                                    tenDeviceCategory enDevCat, tenSelectReason enSelectionReason,
                                                    tenDeviceType enDeviceType)
{
   /*lint -esym(40,fvDeviceConnection)fvDeviceConnection Undeclared identifier */

   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enSelectionReason);
   SPI_INTENTIONALLY_UNUSED(enDeviceType);

   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnSelectDeviceResult entered DeviceId= 0x%x", cou32DeviceHandle));

   if( (e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode))
   {
      if(0 != stConnectDeviceTimerIndex)
      {
         vStopTimer();
         su8NoOfConnectRequests = 0;
         m_sczDeviceBTAddr.clear();
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vOnConnectDeviceResp(...
***************************************************************************/
t_Void spi_tclDiPoConnection::vOnConnectDeviceResp(const char* pczBTMACAddress,
   t_Bool bIsConnectSucceeded)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnConnectDeviceResp-BTMACAddress:%s",
            pczBTMACAddress));
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnConnectDeviceResp-Response:%d,No.of trials:%d,Max.Trials:%d,ConnectTrigger:%d",
            bIsConnectSucceeded,su8NoOfConnectRequests,scou8NoOfTrials,m_enConnectTriggerReason));

   if((true == bIsConnectSucceeded) || (scou8NoOfTrials <= su8NoOfConnectRequests))
   {
      su8NoOfConnectRequests = 0;
      m_sczDeviceBTAddr.clear();

      if(false == bIsConnectSucceeded)
      {
         //Send the select device result to DeviceSelector
         vSendSelectDeviceResult(e8SELECTION_FAILED);
         m_enConnectTriggerReason = tenConnectTriggerReason::e8CONNECT_UNKNOWN;
      }
      else
      {
         m_enConnectTriggerReason = tenConnectTriggerReason::e8CONNECT_UNKNOWN;
      }
   }
   else
   {
      vStartTimer();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vOnDisconnectDeviceResp(...
***************************************************************************/
t_Void spi_tclDiPoConnection::vOnDisconnectDeviceResp(const char* pczBTMACAddress,
   t_Bool bIsDisconnectSucceeded)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnDisconnectDeviceResp-BTMACAddress:%s",
      pczBTMACAddress));
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnDisconnectDeviceResp-Response:%d",
      bIsDisconnectSucceeded));
   if(NULL != (m_rDiPoConnCallbacks.fvSelectDeviceResult))
   {
      (m_rDiPoConnCallbacks.fvSelectDeviceResult)(e8NO_ERRORS);
   }
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vOnSessionMsg(...
***************************************************************************/
t_Void spi_tclDiPoConnection::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
         tenDiPOSessionTransport enSessionDipoTransport,
         t_String szSessionIPAddress)
{
   /*lint -esym(40,fvUpdateSessionStatus)fvUpdateSessionStatus Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vOnSessionMsg entered with state: %d, Session Transport:%d, Session IP Address:%s",
            ETG_ENUM(DIPO_SESSION_STATE, enDiPOSessionState),ETG_ENUM(DIPO_SESSION_TRANSPORT,enSessionDipoTransport),szSessionIPAddress.c_str()));

   if (e8DIPO_SESSION_START == enDiPOSessionState)
   {
      //Establish iAP2OverCarPlay, if the session is started over WiFi transport
      if (e8_DIPO_SESSION_TRANSPORT_WIFI == enSessionDipoTransport)
      {
         vEstablishiAP2OverCarPlaySession(m_u32CurrSelectedDevice);
      }

      //Send the select device result to DeviceSelector
      vSendSelectDeviceResult(e8NO_ERRORS);

      //Update the session transport information
      if(NULL != (m_rDiPoConnCallbacks.fvSessionTransport))
      {
         tenSessionTransport enSessionTransport = (e8_DIPO_SESSION_TRANSPORT_USB == enSessionDipoTransport)? e8_SESSION_TRANSPORT_USB: e8_SESSION_TRANSPORT_WIFI;
         (m_rDiPoConnCallbacks.fvSessionTransport)(m_u32CurrSelectedDevice, e8DEV_TYPE_DIPO, enSessionTransport);
      }

      //Updating the Session IP Address
      if(NULL != (m_rDiPoConnCallbacks.fvOnSessionMessageStoreIPAddress))
      {
         (m_rDiPoConnCallbacks.fvOnSessionMessageStoreIPAddress)(m_u32CurrSelectedDevice, szSessionIPAddress);
      }

      m_enConnectTriggerReason = tenConnectTriggerReason::e8CONNECT_UNKNOWN;
   }

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vStartTimer()
***************************************************************************/
t_Void spi_tclDiPoConnection::vStartTimer()
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vStartTimer: Timer Index:%p",stConnectDeviceTimerIndex));
   t_Bool bIsTimerStarted = false;

   Timer* poTimer = Timer::getInstance();
   if( (NULL != poTimer) && (NULL != m_poDiPOConnection) )
   {
      if(true == poTimer->StartTimer( stConnectDeviceTimerIndex, scou32TimerInterval,0, m_poDiPOConnection,bOnTimerCallback,NULL ))
      {
         ETG_TRACE_USR2(("spi_tclDiPoConnection::vStartTimer: Timer started successfully. ConnectRequest will be sent after specified timeout"));
         bIsTimerStarted = true;
      }
   }//if(( NULL != poTimer)
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclDiPoConnection::vStartTimer: Either poTimer or m_poDiPOConnection is NULL"));
   }

   if(false == bIsTimerStarted)
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclDiPoConnection::vStartTimer: Error in starting Timer. Send the error response"));
      su8NoOfConnectRequests = 0;
      stConnectDeviceTimerIndex = 0;

      vSendSelectDeviceResult(e8SELECTION_FAILED);
      m_enConnectTriggerReason = tenConnectTriggerReason::e8CONNECT_UNKNOWN;
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vStopTimer()
***************************************************************************/
t_Void  spi_tclDiPoConnection::vStopTimer()
{
   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(stConnectDeviceTimerIndex);
      //Reset the Timer Index
      stConnectDeviceTimerIndex = 0; 
      ETG_TRACE_USR4(("[ERR]:spi_tclDiPoConnection::vStopTimer: Timer stopped"));
   }//if( NULL != poTimer)
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoConnection::bOnTimerCallback()
***************************************************************************/
t_Bool spi_tclDiPoConnection::bOnTimerCallback(timer_t timerID , 
                                              t_Void *pObject, 
                                              const t_Void *pcoUserData)
{
   t_Bool bRet = false;
   ETG_TRACE_USR1(("spi_tclDiPoConnection::bOnTimerCallback"));

   SPI_INTENTIONALLY_UNUSED(pcoUserData);
   SPI_INTENTIONALLY_UNUSED(pObject);

   if ( (stConnectDeviceTimerIndex == timerID) && (NULL != m_poDiPOConnection) && (NULL != spoDiPOCmdDisc) )
   {
      //Stop the timer to reset the timer index to zero.
      m_poDiPOConnection->vStopTimer();
      su8NoOfConnectRequests++;
      ETG_TRACE_USR1(("spi_tclDiPoConnection::bOnTimerCallback:Sending ConnectDevice request for %s",m_poDiPOConnection->m_sczDeviceBTAddr.c_str()));
      spoDiPOCmdDisc->vConnectDevice(m_poDiPOConnection->m_sczDeviceBTAddr);

      bRet = true;
   }
   else
   {
       ETG_TRACE_ERR(("spi_tclDiPoConnection::bOnTimerCallback: m_poDiPOConnection or spoDiPOCmdDisc is NULL"));
   }

   return bRet;

}
/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vEstablishiAP2OverCarPlaySession(...
***************************************************************************/
t_Void spi_tclDiPoConnection::vEstablishiAP2OverCarPlaySession(t_U32 u32DeviceID)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vEstablishiAP2OverCarPlaySession:DeviceID:0x%x",u32DeviceID));

   spi_tclDeviceIDDataIntf oDeviceIDIntf;
   t_String szBTAddress;
   oDeviceIDIntf.vGetBTAddress(szBTAddress, u32DeviceID);

   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdAppleDiscovererIntf *poExtCmdAppleDisc = NULL;
   if (NULL != poExtCompMgr)
   {
      poExtCmdAppleDisc = poExtCompMgr->poGetCmdAppleDiscovererIntfInst();
   }

   if (NULL != poExtCmdAppleDisc)
   {
      poExtCmdAppleDisc->vStartiAP2OverCarPlay(szBTAddress);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vSendSelectDeviceResult(...
***************************************************************************/
t_Void spi_tclDiPoConnection::vSendSelectDeviceResult(tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDiPoConnection::vSendSelectDeviceResult:ErrorCode:%d",enErrorCode));

   /*lint -esym(40,fvSelectDeviceResult)fvSelectDeviceResult Undeclared identifier */
   if(NULL != (m_rDiPoConnCallbacks.fvSelectDeviceResult))
   {
      (m_rDiPoConnCallbacks.fvSelectDeviceResult)(enErrorCode);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoConnection::vSetInitalModes()
 ***************************************************************************/
t_Void spi_tclDiPoConnection::vSetInitalModes(const t_String& corfszBTAddress)
{
   //! Evaluate inital modes here only for CPW disabled variants
   //! (BT address is needed to check BT playback state of CP USB device, which is available only after Bonjour Discovery)

   spi_tclMediator *poMediator = spi_tclMediator::getInstance();

   if ((NULL != m_poConnSettings) && (false == m_poConnSettings->bReadCPWStatus()) && (NULL != poMediator))
   {
      poMediator->vOnEvaluateInitalModes(corfszBTAddress);
   }

   //! Read and send initial modes data to carplayd
   if (NULL != spoDiPOManager)
   {
      spoDiPOCmdSession = spoDiPOManager->poGetDiPOSessionInstance();
      if (NULL != spoDiPOCmdSession)
      {
         t_Bool bAudioLastMode = false, bDisplayLastMode = false;
         if (NULL != m_poConnSettings)
         {
            bDisplayLastMode = m_poConnSettings->bIsDisplayLastModeProjection();
            bAudioLastMode = m_poConnSettings->bIsAudioLastModeProjection();
         }
         spoDiPOCmdSession->vSendLastModeData(bAudioLastMode, bDisplayLastMode);
      }
   }
}

//lint -restore
