/*!
 *******************************************************************************
 * \file             spi_tclDiPoConnection.h
 * \brief            DiPo Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPo Connection class to handle ios devices capable of DiPo
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 10.01.2013 |  Pruthvi Thej Nagaraju       | Initial Version
 13.02.2014 |  Shihabudheen P M            | added functions
                                             1.vOnDiPODeviceConnect()
                                             2.vOnDiPODeviceDisConnect()
 05.11.2014 |  Ramya Murthy                | Added response for Application metadata.
 01.03.2017 |  Shiva Kumar G               | R12N Plugin Adaptations
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDIPOCONNECTION_H_
#define SPI_TCLDIPOCONNECTION_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Lock.h"
#include "DiPOTypes.h"
#include "spi_tclConnection.h"
#include "spi_tclDiPOConnectionIntf.h"
#include "spi_tclDiPORespDiscoverer.h"
#include "spi_tclDiPORespSession.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

typedef enum
{
   //! DiPO activated.
   e8DIPO_ACTIVE = 0,

   //! Dipo not active
   e8DIPO_NOT_ACTIVE = 1,

   //! DiPO Activating automatically(Automatic device selection)
   e8SIPO_SPI_ACTIVATING = 2,

   //! DiPO activating by the user
   e8DIPO_HMI_ACTIVATING = 3,

   //! DiPO deactivating by SPI internally
   e8DIPO_SPI_DEACTIVATING = 4,

   //! DiPO deactivating by HMI internally
   e8DIPO_HMI_DEACTIVATING = 5
}tenDiPOState;


typedef enum 
{
   //! Trigger is the select device
   e8DIPO_TRIGGER_SELECT_DEVICE = 0,

   //! trigger is the select device response
   e8DIPO_TRIGGER_SELECT_DEVICE_RESULT = 1,

   //! Trigger other than select device
   e8DIPO_TRIGGER_DEVICELIST_UPDATE = 2
}tenTriggerType;

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
class spi_tclMPlayClientHandler;

/*!
 * \class spi_tclDiPoConnection
 * \brief DiPo Connection class to handle ios devices capable of DiPo
 */

class spi_tclDiPoConnection: public spi_tclConnection, //! Base Connection Class
	public spi_tclDiPOConnectionIntf, public spi_tclDiPORespDiscoverer , public spi_tclDiPORespSession //!DiPO connection manager interface.
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::spi_tclDiPoConnection
       ***************************************************************************/
      /*!
       * \fn     spi_tclDiPoConnection()
       * \brief  Default Constructor
       * \sa      ~spi_tclDiPoConnection()
       **************************************************************************/
      spi_tclDiPoConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::~spi_tclDiPoConnection
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclDiPoConnection()
       * \brief  Destructor
       * \sa     spi_tclDiPoConnection()
       **************************************************************************/
      ~spi_tclDiPoConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnLoadSettings
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettings()
       * \brief  Called on loadsettings
       *         Optional interface
       * \param rfrheadUnitInfo : Head unit information
       * \param enCertificateType : CertificateType to be used for authentication
       **************************************************************************/
      virtual t_Void vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnLoadSettingsCompleted
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettingsCompleted()
       * \brief  Invoked to notify that load settings has completed
       **************************************************************************/
      virtual t_Void vOnLoadSettingsCompleted();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::vOnSaveSettings
       ***************************************************************************/
      /*!
       * \fn     vOnSaveSettings()
       * \brief  Called on savesettings
       *         Optional interface
       * \retval none
       **************************************************************************/
      virtual t_Void vOnSaveSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \fn     bStartDeviceDetection()
       * \brief  Starts device detection
       * \retval returns true on successful initialization and false on failure
       **************************************************************************/
       t_Bool bStartDeviceDetection();

       /***************************************************************************
        ** FUNCTION:  spi_tclDiPoConnection::vSetDeviceSwitchInfo
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceSwitchInfo
        * \brief  Set device switch information
        * \param  rfrDeviceSwitchInfo : Device switch info
        **************************************************************************/
       t_Void vSetDeviceSwitchInfo(trDeviceSwitchInfo & /*rfrDeviceSwitchInfo*/) {};

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::vOnAddDevicetoList
       ***************************************************************************/
      /*!
       * \fn     vOnAddDevicetoList()
       * \brief  To be called when a new device has to be added to device list
       * \param  cou32DeviceHandle: Device handle of the device to be added to
       *         the device list
       **************************************************************************/
      t_Void vOnAddDevicetoList(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::bSetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bSetDevProjUsage()
       * \brief  Called when the SPI featured is turned ON or OFF by the user
       **************************************************************************/
      t_Bool bSetDevProjUsage(tenEnabledInfo enServiceStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoConnection::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *        callbacks.
       * \param rfrConnCallbacks : reference to the callback structure
       *        populated by the caller
       **************************************************************************/
      t_Void vRegisterCallbacks(trConnCallbacks &ConnCallbacks);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDiPoConnection::vOnSelectDevice()
      ***************************************************************************/
      /*!
      * \fn     t_Void vOnSelectDevice()
      * \brief  Function to get the deviec selection.
      * \param  cou32DeviceHandle : [IN] Device handle
      * \param  enDevConnReq : [IN] Selection status
      **************************************************************************/
      virtual t_Void vOnSelectDevice(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionType enDevConnType,
               tenDeviceConnectionReq enDevSelReq, tenEnabledInfo enDAPUsage,
               tenEnabledInfo enCDBUsage, tenSelectReason enSelectionReason,
               const trUserContext corUsrCntxt, tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDiPoConnection::vPostSelctDeviceResult()
      ***************************************************************************/
      /*!
      * \fn     t_Void vPostSelctDeviceResult()
      * \brief  Function to post the device selection statu.
      * \param  enResponse : [IN] Operation status.
      * \param  enErrorType : [IN] error value.
      **************************************************************************/
      virtual t_Void vPostSelectDeviceResult(tenResponseCode enResponse, 
         tenErrorCode enErrorType);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDiPoConnection::vSetRoleSwitchRequestedInfo()
      ***************************************************************************/
      /*!
      * \fn     t_Void vSetRoleSwitchRequestedInfo()
      * \brief  Function to post the device selection statu.
      * \param  cou32DeviceHandle : [IN] Roleswitch requested device ID
      * \retVal NONE
      **************************************************************************/
      virtual t_Void vSetRoleSwitchRequestedInfo(const t_U32 cou32DeviceHandle);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDiPoConnection::vOnSelectDeviceResult()
      ***************************************************************************/
      /*!
      * \fn      t_Void vOnSelectDeviceResult
      * \brief   To perform the actions that are required, after the select device is
      *           successful/failed
      * \pram    cou32DeviceHandle  : [IN] Uniquely identifies the target Device.
      * \pram    enDevSelReq : [IN] Identifies the Connection Request.
      * \pram    coenRespCode: [IN] Response code. Success/Failure
      * \pram    enDevCat    : [IN] Device Category. ML/DiPo
      * \retval  t_Void
      **************************************************************************/
      t_Void vOnSelectDeviceResult(const t_U32 cou32DeviceHandle,
         const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode,
         tenDeviceCategory enDevCat, tenSelectReason enSelectionReason, tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

   private:
      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vOnConnectDeviceResp(...
     ***************************************************************************/
     /*!
     * \fn     t_Void vOnConnectDeviceResp(
     *                       const char* pczBTMACAddress, t_Bool bIsConnectSucceeded)
     * \brief   To Post the ConnectDevice Response
     * \param   pczBTMACAddress : [IN] BT MAC Address of the device
     * \param   bIsConnectSucceeded : [IN] true - ConnectDevice success
     *                                     false - ConnectDevice failure
     * \retval  t_Void
     ***************************************************************************/
     t_Void vOnConnectDeviceResp(const char* pczBTMACAddress, t_Bool bIsConnectSucceeded);

     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vOnDisconnectDeviceResp(...
     ***************************************************************************/
     /*!
     * \fn      t_Void vOnDisconnectDeviceResp(
     *                       const char* pczBTMACAddress, t_Bool bIsConnectSucceeded)
     * \brief   To Post the DisconnectDevice Response
     * \param   pczBTMACAddress : [IN] BT MAC Address of the device
     * \param   bIsDisconnectSucceeded : [IN] true - DisconnectDevice success
     *                                     false - DisconnectDevice failure
     * \retval  t_Void
     ***************************************************************************/
     t_Void vOnDisconnectDeviceResp(const char* pczBTMACAddress, t_Bool bIsDisconnectSucceeded);

     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vOnSessionMsg(...
     ***************************************************************************/
     /*!
     * \fn      t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
     *                                       tenDiPOSessionTransport enSessionTransport,
     *                                       t_String szSessionIPAddress)
     * \brief   To Post the session message to SPI, when the session state changes.
     * \param   enDiPOSessionState : [IN] Session State : Session Started,Session_Ended,
     *                                                    CarPlayd_PlugIn_Loaded or UnLoaded.
     * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
     *                                    Applicable, only when SessionState is Session Started/ended.
     * \param   szSessionIPAddress : [IN] IP Address of the active carplay device.
     *                                    used, only if the Session Transport is WiFi
     * \retval  t_Void
     ***************************************************************************/
     t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                                  tenDiPOSessionTransport enSessionTransport,
                                  t_String szSessionIPAddress);					  
					  
	 
     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vStartTimer()
     ***************************************************************************/
     /*!
     * \fn      t_Void vStartTimer()
     * \brief   To start the timer
     * \retval  t_Void
     * \sa      vStopTimer(),bOnTimerCallback()
     **************************************************************************/
     t_Void vStartTimer();
     
     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vStopTimer()
     ***************************************************************************/
     /*!
     * \fn      t_Void vStopTimer()
     * \brief   To stop the timer
     * \retval  t_Void
     * \sa      vStartTimer()
     **************************************************************************/
     t_Void vStopTimer();
     
     /***************************************************************************
     ** FUNCTION:  t_Bool spi_tclDiPoConnection::bOnTimerCallback()
     ***************************************************************************/
     /*!
     * \fn      static t_Bool bOnTimerCallback(timer_t timerID , 
     *                                         t_Void *pObject, 
     *                                         const t_Void *pcoUserData)
     * \brief   Timer Callback Function to send connectdevice on timer expiry
     * \param   timerID : [IN] Id of the timer for which the timer is expired
     * \param   pObject : [IN] pointer to the object for which the callback is related to
     * \param   pcoUserData : [IN] User context data
     * \retval  t_Bool
     * \sa      vStartTimer()
     **************************************************************************/
     static t_Bool bOnTimerCallback(timer_t timerID , 
                                    t_Void *pObject, 
                                    const t_Void *pcoUserData);
     
     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vEstablishiAP2OverCarPlaySession(...
     ***************************************************************************/
     /*!
     * \fn      t_Void vEstablishiAP2OverCarPlaySession(t_U32 u32DeviceID)
     * \brief   Method to trigger iAP2 Over CarPlay session establishment
     * \param   u32DeviceID : [IN] Unique Device ID
     * \retval  t_Void
     ***************************************************************************/
     t_Void vEstablishiAP2OverCarPlaySession(t_U32 u32DeviceID);

     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoConnection::vSendSelectDeviceResult(...
     ***************************************************************************/
     /*!
     * \fn      t_Void vSendSelectDeviceResult(tenErrorCode enErrorCode)
     * \brief   To send the select device result to device selector
     * \param   enErrorCode : [IN] Error Code
     * \retval  t_Void
     ***************************************************************************/
     t_Void vSendSelectDeviceResult(tenErrorCode enErrorCode);
     
     /***************************************************************************
      ** FUNCTION:  spi_tclDiPoConnection::vSetInitalModes()
      ***************************************************************************/
     /*!
     * \fn      t_Void vSetInitalModes(const t_String& corfszBTAddress)
     * \brief   Method to send initial audio and video mode data to carplayd
     *            before session establishment
     * \param   corfszBTAddress : [IN] Unique Device ID
     * \retval  t_Void
     ***************************************************************************/
     t_Void vSetInitalModes(const t_String& corfszBTAddress);



     //! Callbacks for ConnMngr to register. These callbacks will be used to
     //! inform device detection and device disconnection to connection manager
     trConnCallbacks m_rDiPoConnCallbacks;
     
     //! Current selected device
     t_U32 m_u32CurrSelectedDevice; 
     
     //! Status of carplay service status.
     tenEnabledInfo m_enCarplaySettingStatus;
     
     trDiPODeviceInfo m_rDipoDeviceinfo;
     
     t_String m_sczDeviceBTAddr;
     
     static spi_tclDiPoConnection* m_poDiPOConnection;
     
     //! lock object(based on mutext).
     Lock m_oLock;

     //Current ConnectDevice is in progress for
     tenConnectTriggerReason m_enConnectTriggerReason;

};
#endif // SPI_TCLDIPOCONNECTION_H_
