  /*!
 *******************************************************************************
 * \file         spi_tclDiPoBluetooth.h
 * \brief        DiPO Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 10.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 24.11.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented BT block/unblock for GM
 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLDIPOBLUETOOTH_H
#define _SPI_TCLDIPOBLUETOOTH_H

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "Timer.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_WifiTypedefs.h"
#include "spi_tclBluetoothDevBase.h"
#include "spi_tclDiPORespBluetooth.h"
#include "spi_tclExtRespBluetooth.h"
#include "spi_tclExtRespWiFi.h"
#include "spi_tclExtRespAppleDiscoverer.h"


/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/* Forward Declarations. */
class spi_tclBluetoothIntf;
class spi_tclBluetoothClient;
class spi_tclDiPORespBluetooth;
class spi_tclExtRespWiFi;


/*!
* \class spi_tclDiPoBluetooth
* \brief This is the DiPO Bluetooth class that handles the Bluetooth
*        connection logic during a DiPO device session
*/
class spi_tclDiPoBluetooth : public spi_tclBluetoothDevBase,public spi_tclDiPORespBluetooth,public spi_tclExtRespWiFi,
public spi_tclExtRespAppleDiscoverer, public spi_tclExtRespBluetooth
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoBluetooth::spi_tclDiPoBluetooth(...)
   ***************************************************************************/
   /*!
   * \fn      spi_tclDiPoBluetooth(spi_tclBluetoothIntf* poBTInterface,
   *             tenBTDisconnectStrategy enBTDisconnStrategy)
   * \brief   Parameterized Constructor
   * \param   [IN] poBTInterface: Pointer to Bluetooth manager interface
   ***************************************************************************/
   spi_tclDiPoBluetooth(spi_tclBluetoothIntf* poBTInterface);

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPoBluetooth::~spi_tclDiPoBluetooth();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclDiPoBluetooth()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclDiPoBluetooth();

   /*********Start of functions overridden from spi_tclBluetoothDevBase*******/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when SelectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when SelectDevice operation is complete, with the result
   *          of the operation. 
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest(
   ***************************************************************************/
   /*!
   * \fn      vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
   * \brief   Called when DeselectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceResponse(
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when DeselectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);


   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDiPoBluetooth::vOnAutoSelectionFailureForWireless()
    ***************************************************************************/
   /*!
    * \fn      vOnAutoSelectionFailureForWireless()
    * \brief   Interface to set a Bluetooth limitation in HU.
    * \param   [IN] rfcszBTAddress: BT address of device
    * \param   [IN] enTechnology: Technology for which BT limitation mode is required
    * \param   [IN] enCommChannel: Communication channel of the device
    * \param   [IN] enAction: Type of action for which BT limitation mode is required
    * \retval  t_Bool: True - if request to set limitation mode is sent successfully, false
    **************************************************************************/
   t_Void vOnAutoSelectionFailureForWireless(const t_String& rfcszBTAddress, tenBTTechnology enTechnology,
            tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction);

   /*********End of functions overridden from spi_tclBluetoothDevBase*********/

   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/
   /***************************************************************************
    ** FUNCTION: spi_tclDiPoBluetooth(const spi_tclDiPoBluetooth &rfcoBluetooth)
    ***************************************************************************/
   /*!
    * \fn      spi_tclDiPoBluetooth(const spi_tclDiPoBluetooth &rfcoBluetooth)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclDiPoBluetooth(const spi_tclDiPoBluetooth &rfcoBluetooth);

   /***************************************************************************
    ** FUNCTION: const spi_tclDiPoBluetooth & operator=(
    **                                 const spi_tclDiPoBluetooth &rfcoBluetooth);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclDiPoBluetooth & operator=(const spi_tclDiPoBluetooth &rfcoBluetooth);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclDiPoBluetooth & operator=(
            const spi_tclDiPoBluetooth &rfcoBluetooth);

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclDiPoBluetooth::vDeselectDipoDevice()
   ***************************************************************************/
   /*!
   * \fn      vDeselectDipoDevice()
   * \brief   Requests for deselection of current active DiPO device.
   * \retval  None
   **************************************************************************/
   t_Void vDeselectDipoDevice();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnDisableBluetooth(...
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vOnDisableBluetooth(t_String szBTMACAddr,
   *                     ,tenDiPOSessionTransport enSessionTransport)
   * \brief   To Post the disableBluetooth update to registered clients
   * \param   szBTMACAddr : [IN] Unique ID of the device.
   * \param   enSessionTransport : [IN] Session transport
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnDisableBluetooth(t_String szBTMACAddr,tenDiPOSessionTransport enSessionTransport);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnWiFiConfig(...)
   ***************************************************************************/
   /*!
    * \fn      t_Void vPostWiFiConfig()
    * \brief   Called when ActiveWifiSetup structure is sent by WBL Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN] rfcorWiFiConfig: structure of type trWiFiConfig
	* \param   [IN] corvecStationsInfo: List of connected stations.
    * \sa      None
    **************************************************************************/
   t_Void vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo);
   
   /***************************************************************************
   ** FUNCTION:  tenBTCommunicationChannel spi_tclDiPoBluetooth::enGetBTCommnChannel(...
   ***************************************************************************/
   /*!
   * \fn      tenBTCommunicationChannel enGetBTCommnChannel(tenDeviceConnectionType enDeviceConnectionType)
   * \brief   To get the Bt Communication channel that needs to be populated in SetBTLimitationMode request
   * \param   [IN]enDeviceConnectionType : [IN] USB Connected/Wireless Connected
   * \retval  tenBTCommunicationChannel
   ***************************************************************************/   
   tenBTCommunicationChannel enGetBTCommnChannel(tenDeviceConnectionType enDeviceConnectionType);
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostDeviceListMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostDeviceListMsg()
    * \brief   Called on devicelist update
    * \param   corfrDeviceInfo: a trDeviceInfo type structure
    * \sa      None
    **************************************************************************/
    t_Void vPostDeviceListMsg(const trDeviceInfo &rfcoDeviceInfo);

    /***************************************************************************
     ** FUNCTION:  spi_tclDiPoBluetooth::bBTLimModeTimerCb
     ***************************************************************************/
    /*!
     * \fn     bBTLimModeTimerCb
     * \brief  called on expiry of BT Lim Mode timer
     * \param  rTimerID: ID of the timer which has expired
     * \param  pvObject: pointer to object passed while starting the timer
     * \param  pvUserData: data passed during start of the timer
     **************************************************************************/
    static t_Bool bBTLimModeTimerCb(timer_t rTimerID, t_Void *pvObject,
             const t_Void *pvUserData);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostBTServiceAvailabilityMsg()
     ***************************************************************************/
    /*!
     * \fn      t_Void vPostBTConnectionChangedMsg()
     * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
     * \sa      None
     **************************************************************************/
    virtual t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostBTLimitationModeMsg()
     ***************************************************************************/
    /*!
     * \fn      t_Void vPostBTLimitationModeMsg()
     * \brief   Called when BT limitation mode message is  sent by BT Client Handler.
     * \sa      None
     **************************************************************************/
    virtual t_Void vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo);

    /***************************************************************************
     ** FUNCTION:  t_Bool spi_tclDiPoBluetooth::bSetBTLimitationMode()
     ***************************************************************************/
    /*!
     * \fn      bSetBTLimitationMode()
     * \brief   Interface to set a Bluetooth limitation in HU.
     * \param   [IN] rfcszBTAddress: BT address of device
     * \param   [IN] rfcszBTDeviceName: BT device name
     * \param   [IN] enTechnology: Technology for which BT limitation mode is required
     * \param   [IN] enCommChannel: Communication channel of the device
     * \param   [IN] enAction: Type of action for which BT limitation mode is required
     * \retval  t_Bool: True - if request to set limitation mode is sent successfully, false
     **************************************************************************/
    t_Void vSetBTLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
             tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction);
   
    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vCanclePreviousBTLimTimer()
     ***************************************************************************/
    /*!
     * \fn      t_Void vCanclePreviousBTLimTimer()
     * \brief   Cancle Previous BT Limitation Timer
     * \sa      None
     **************************************************************************/
     t_Void vCanclePreviousBTLimTimer();

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** BT Manager interface pointer
   ***************************************************************************/
   spi_tclBluetoothIntf* const   m_cpoBTInterface;

   //!structure to hold WiFi Config
   trWiFiConfig m_rWiFiConfig;

   t_String m_szSelectionInProgress;

   trBTLimitationModeInfo m_rBTLimitationModeInfo;

   //! Stores the BT LIM Mode timer ID
   timer_t m_rBTLimModeTimerID;

   //! store the device connection type in case disableBT msg needs to be processed later
   tenDeviceConnectionType m_enDeviceConnectionType;
   
   //! Action of pending SetBTLimitationMode request
   tenBTLimitationAction m_enPendingBTLimAction;

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

};

#endif // _SPI_TCLDIPOBLUETOOTH_H

