/*!
 *******************************************************************************
 * \file             spi_tclDiPoAudio.h
 * \brief            Implements the Audio functionality for Digital IPOd Out using 
                     interface to Real VNC SDK through VNC Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for Digital IPOd Out
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                          | Modifications
 29.10.2013 |  Hari Priya E R(RBEI/ECP2)       | Initial Version
 04.04.2014 |  Shihabudheen P M(RBEI/ECP2)     | Added 1.bStartAudio()

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef SPI_TCLDIPOAUDIO_H
#define SPI_TCLDIPOAUDIO_H

#include "BaseTypes.h"
#include "DiPOTypes.h"
#include "Timer.h"
#include "Lock.h"
#include "spi_tclAudioDevBase.h"
#include "spi_tclDiPORespAudio.h"
#include "spi_tclDiPORespSession.h"
#include "spi_tclDiPORespRsrcMngr.h"

/**
 *  class definitions.
 */

class spi_tclDiPOCmdRsrcMngr;
/**
 * This class implements the Audio functionality for Digital IPOd Out
 */
class spi_tclDiPoAudio : public spi_tclAudioDevBase, public spi_tclDiPORespAudio,
         public spi_tclDiPORespSession, public spi_tclDiPORespRsrcMngr
{
public:
	 /***************************************************************************
     *********************************PUBLIC*************************************
     ***************************************************************************/

    /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudio::spi_tclDiPoAudio();
    ***************************************************************************/
    /*!
    * \fn      spi_tclDiPoAudio()
    * \brief   Default Constructor
    **************************************************************************/
	spi_tclDiPoAudio();

	/***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudio::~spi_tclDiPoAudio();
    ***************************************************************************/
    /*!
    * \fn      ~spi_tclDiPoAudio()
    * \brief   Virtual Destructor
    **************************************************************************/
	virtual ~spi_tclDiPoAudio();

  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoAudio::bStartAudio(t_U32,tString,CbAudioResp)
   ***************************************************************************/
   /*!
   * \fn      bStartAudio(t_U32 u32DeviceId, tString szAudioDev,
   * \brief   Start Streaming of Audio from the CE Device to the Audio Output
	*          Device assigned by the Audio Manager for the Source.
	*          Mandatory Interface to be implemented.
	* \param   [u32DeviceId]: Unique Identifier for the Connected Device.
	*		     [szAudioDev]: ALSA Audio Device
   *          [enAudDir]    :Specify the Audio Direction(Alternate or Main Audio).
	* \retval  Bool value
    **************************************************************************/
	virtual t_Bool bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
              tenAudioDir enAudDir);

  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoAudio::bStartAudio(t_U32,tString,t_String, 
   **				tenAudioDir)
   ***************************************************************************/
   /*!
   * \fn      bStartAudio(t_U32 u32DeviceId, tString szAudioDev,
   * \brief   Start Streaming of Audio from the CE Device to the Audio Output
	*          Device assigned by the Audio Manager for the Source.
	*          Mandatory Interface to be implemented.
	* \param  [u32DeviceId]: Unique Identifier for the Connected Device.
	*		  [szAudioDev]: ALSA Audio Out Device 
   *          [szInputAudioDev]: Audio In Device name
   *		  [enAudDir]    :Specify the Audio Direction(Phone or VR Audio).
	* \retval  Bool value
    **************************************************************************/
   virtual t_Bool bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
              t_String szInputAudioDev, tenAudioDir enAudDir);

  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoAudio::bStopAudio(t_U32, CbAudioResp)
   ***************************************************************************/
   /*!
   * \fn      vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir)
   * \brief   Stop Streaming of Audio from the CE Device to the Audio Output
   *          Device assigned by the Audio Manager for the Source.
   *          Mandatory Interface to be implemented.
   * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
   * \param   [enAudDir]   : Specify the Audio Direction.
   * \param   [bIsPaused]  : Indicates if source is paused or stopped
   * \retval  None
   **************************************************************************/
   virtual t_Void vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused);

  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoAudio::bSelectAudioDevice(t_U32,
   ***************************************************************************/
   /*!
   * \fn      bSelectAudioDevice(t_U32 u32DeviceId CbAudioResp cbSelectDevResp)
   * \brief   Perfom necessary actions specific to a device selection like
	*          obtaining audio capabilities of device, supported modes etc
	* \param   [u32DeviceId]: Unique Identifier for the Connected Device.
	*		   [cbSelectDevResp]: Callback function provided to notify response
	*          for Select Device.
	* \retval  Bool value
    **************************************************************************/
	virtual t_Bool bSelectAudioDevice(t_U32 u32DeviceId);

  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDiPoAudio::bIsAudioLinkSupported(t_U32,
	*                                            tenAudioLink)
   ***************************************************************************/
   /*!
   * \fn      bIsAudioLinkSupported(t_U32 u32DeviceId)
   * \brief   Perfom necessary actions specific to a device on deselection.
	*          Optional Interface for implementation.
	* \param   [u32DeviceId]: Unique Identifier for the Connected Device.
	*          [enLink]: Specify the Audio Link Type for which Capability is
	*          requested. Mandatory interface to be implemented.
	* \retval  Bool value, TRUE if Supported, FALSE otherwise
    **************************************************************************/
	virtual t_Bool bIsAudioLinkSupported(t_U32 u32DeviceId, tenAudioLink enLink);

  /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDiPoAudio::vDeselectAudioDevice()
   ***************************************************************************/
   /*!
   * \fn      vDeselectAudioDevice(t_U32 u32DeviceId)
   * \brief   Perfom necessary actions specific to a device on deselection.
	*          Optional Interface for implementation.
	* \param   [u32DeviceId]: Unique Identifier for the Connected Device.
	* \retval  Bool value
    **************************************************************************/
	virtual t_Void vDeselectAudioDevice(t_U32 u32DeviceId);

  /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudio::vRegisterCallbacks
    ***************************************************************************/
   /*!
    * \fn     vRegisterCallbacks()
    * \brief  interface for the creator class to register for the required
    *        callbacks.
    * \param rfrAudCallbacks : reference to the callback structure
    *        populated by the caller
    **************************************************************************/
   t_Void vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks);

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclDiPoAudio::bSetAudioBlockingMode()
   ***************************************************************************/
   /*!
   * \fn     t_Bool bSetAudioBlockingMode(const t_U32 cou32DevId,
   *         const tenBlockingMode coenBlockingMode)
   * \brief  Interface to set the audio blocking mode.
   * \param  cou32DevId       : [IN] Uniquely identifies the target Device.
   * \param  coenBlockingMode : [IN] Identifies the Blocking Mode.
   * \retval t_Bool
   **************************************************************************/
   t_Bool bSetAudioBlockingMode(const t_U32 cou32DevId,
      const tenBlockingMode coenBlockingMode);

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclDiPoAudio::vOnAudioError()
    ***************************************************************************/
   /*!
    * \fn    t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
    * \brief  Interface to set the audio error.
    * \param  enAudDir       : [IN] Uniquely identifies the target Device.
    * \param  enAudioError : [IN] Audio Error
    **************************************************************************/
   t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDiPoAudio::bInitializeAudioPlayback(t_U32)
    ***************************************************************************/
   /*!
    * \fn      bInitializeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
    * \brief   Perform necessary actions to prepare for an Audio Playback.
    *          Function will be called prior to a Play Command from Audio Manager.
    *          Mandatory Interface to be implemented by Device Class.
    * \param   [u32DeviceId]: Unique Identifier for the Connected Device
    * \param   [enAudDir]: Audio route being allocated
    * \param   [enAudSampleRate]: Specifies the Audio Sample Rate.
	* \param   [enNativeSamplingRate]: Specifies the Native Audio Sample Rate.
    * \retval  Bool value
    **************************************************************************/
   t_Bool bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,
         tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDiPoAudio::bFinalizeAudioPlayback(t_U32,tenAudioDir)
    ***************************************************************************/
   /*!
    * \fn      bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
    * \brief   Perform necessary actions on completion of Audio Playback.
    *          Function will be called after to a Stop Command from Audio Manager.
    * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
   * \param   [enAudDir]: Audio route being deallocated
    * \retval  Bool value
    **************************************************************************/
   t_Bool bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir);

   /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclDiPoAudio::vUpdateDeviceSelection()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUpdateDeviceSelection()
   * \brief   To update the device selection.
   * \param   u32DevID : [IN] Device ID.
   * \param   enDevCat : [IN] Category of the device
   * \param   enDeviceConnReq : [IN] Select/ deselect.
   * \param   enRespCode : [IN] Response code (success/failure)
   * \param   enErrorCode : [IN] Error
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vUpdateDeviceSelection(t_U32 u32DevID, tenDeviceCategory enDevCat,
         tenDeviceConnectionReq enDeviceConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode);

private:

   /***************************************************************************
    *********************************PRIVATE************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vOnModesChangedUpdate
    ***************************************************************************/
   /*!
    * \fn     vOnModesChangedUpdate()
    * \brief  Informs when audio mode changes on the carplay device
    **************************************************************************/
   t_Void vOnModesChangedUpdate(const trDiPOModeState& corfrDiPOModeState);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnAudioAllocateRequest(...
    ***************************************************************************/
   /*!
    * \fn      t_Void vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
    *                         tenDiPOMainAudioType enAudioType,tenAudioReqType enAudioReqType,
    *                         const trDiPOAudioFormat& corfrDiPOAudioFormat)
    * \brief   To Post the Audio allocate/de-allocate request
    * \param   tenAudioChannelType : [IN] Audio channel type
    * \param   enAudioType : [IN] Alternate audio/ default audio
    * \param   enAudioReqType : [IN] Audio allocate/deallocate request
    * \param   corfrDiPOAudioFormat : [IN] Audio format which includes sample rate, bits per channel..
    * \retval  t_Void
    ***************************************************************************/
   t_Void vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
            tenDiPOMainAudioType enAudioType,
            tenAudioReqType enAudioReqType,
            const trDiPOAudioFormat& corfrDiPOAudioFormat);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnDuckAudioMsg(...
    ***************************************************************************/
   /*!
    * \fn      t_Void vOnDuckAudioMsg(t_Double dFinalVolume,
    *                         t_Double dDurationInMs)
    * \brief   To Post the duck audio request
    * \param   dFinalVolume : [IN] Final volume after ducking
    * \param   dDurationInMs : [IN] Ramp down duration in msec
    * \retval  t_Void
    ***************************************************************************/
   t_Void vOnDuckAudioMsg(t_Double dFinalVolume,t_Double dDurationInMs);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnSessionMsg(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
   *                                       tenDiPOSessionTransport enSessionTransport,
   *                                       t_String szSessionIPAddress)
   * \brief   To Post the session message to SPI, when the session state changes.
   * \param   enDiPOSessionState : [IN] Session State : Session Started,Session_Ended,
   *                                                    CarPlayd_PlugIn_Loaded or UnLoaded.
   * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
   *                                    Applicable, only when SessionState is Session Started/ended.
   * \param   szSessionIPAddress : [IN] IP Address of the active carplay device.
   *                                    used, only if the Session Transport is WiFi
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
            tenDiPOSessionTransport enSessionTransport,
            t_String szSessionIPAddress);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vHandleSessionEnd
    ***************************************************************************/
   /*!
    * \fn     vHandleSessionEnd()
    * \brief  Informs when dipo session ends
    **************************************************************************/
   t_Void vHandleSessionEnd();

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudio::enGetAudioInDataSet(...)
    ***************************************************************************/
   /*!
    * \fn     vHandleDeallocRequest()
    * \brief  Returns audio data set to be used by ECNR service
    * \param  enAudioDir: Audio direction type enumeration
    * \param  enAudioSamplingRate: Audio sampling rate enumeration
    **************************************************************************/
   tenAudioInDataSet enGetAudioInDataSet(tenAudioDir enAudioDir,
         tenAudioSamplingRate enAudioSamplingRate);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudio::vActivateChannel(...)
    ***************************************************************************/
   /*!
    * \fn     vActivateChannel(tenAudioDir enAudDir)
    * \brief  Function to request for activation of audio channel
    * \param  enAudDir : Audio direction type
    * \retval None
    **************************************************************************/
   t_Void vActivateChannel(tenAudioDir enAudDir, tenAudioSamplingRate enSamplingRate,
         tenAudioSamplingRate enNativeSamplingRate = e8AUD_SAMPLERATE_DEFAULT);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudio::vDeactivateChannel(...)
    ***************************************************************************/
   /*!
    * \fn     vDeactivateChannel(tenAudioDir enAudDir, tenSrcAvailabilityReason)
    * \brief  Function to request for deactivation of audio channel
    * \param  enAudDir : Audio direction type
    * \param  enSrcAvlReason : source availability reason code
    * \retval None
    **************************************************************************/
   t_Void vDeactivateChannel(tenAudioDir enAudDir, tenSrcAvailabilityReason enSrcAvlReason=e8REASON_NOMEDIA);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudio::vMuteAudio(...)
    ***************************************************************************/
   /*!
    * \fn     vMuteAudio(t_Bool bMute)
    * \brief  Function to request for audio source mute
    * \param  bMute : TRUE if Mute required, else FALSE for Unmute
    * \retval None
    **************************************************************************/
   t_Void vMuteAudio(t_Bool bMute);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPoAudio::vSendAudioStatusChange(...)
    ***************************************************************************/
   /*!
    * \fn     vSendAudioStatusChange(tenAudioStatus enAudStatus)
    * \brief  Function to notify audio status changes
    * \param  enAudStatus : Audio status type
    * \retval None
    **************************************************************************/
   t_Void vSendAudioStatusChange(tenAudioStatus enAudStatus);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vProcessChangedAudSrc()
    ***************************************************************************/
   /*!
    * \fn     vProcessChangedMainAudSrc()
    * \brief  Processes the incoming Audio message if the source has changed
    **************************************************************************/
   t_Void vProcessChangedMainAudSrc(const tenAudioDir enRequestedAudSrc,
            const tenAudioSamplingRate enRequestedSamplingRate,
            const tenAudioSamplingRate enRequestedNativeSamplingRate);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vProcessUnchangedAudSrc()
    ***************************************************************************/
   /*!
    * \fn     vProcessUnchangedMainAudSrc()
    * \brief  Processes the incoming Audio message if the source has not changed
    **************************************************************************/
   t_Void vProcessUnchangedMainAudSrc( const tenAudioSamplingRate enRequestedSamplingRate,
         const tenAudioSamplingRate enRequestedNativeSamplingRate);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vHandleAllocRequest
    ***************************************************************************/
   /*!
    * \fn     vHandleAllocRequest()
    * \brief  Processes audio allocation request from the device.
    * \param   tenAudioChannelType : [IN] Audio channel type
    * \param   enAudioType : [IN] Alternate audio/ default audio
    * \param   corfrDiPOAudioFormat : [IN] Audio format which includes sample rate, bits per channel..
    **************************************************************************/
   t_Void vHandleAllocRequest(AudioChannelType enAudioChannelType,
            tenDiPOMainAudioType enAudioType,
            const trDiPOAudioFormat& corfrDiPOAudioFormat);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vHandleDeallocRequest
    ***************************************************************************/
   /*!
    * \fn     vHandleDeallocRequest()
    * \brief  Processes audio deallocation request from the device.
    **************************************************************************/
   t_Void vHandleDeallocRequest(const AudioChannelType coenAudioChannelType);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::bAudRestoreTimerCb
    ***************************************************************************/
   /*!
    * \fn     bAudRestoreTimerCb
    * \brief  called on expiry of selection timer
    * \param  rTimerID: ID of the timer which has expired
    * \param  pvObject: pointer to object passed while starting the timer
    * \param  pvUserData: data passed during start of the timer
    **************************************************************************/
   static t_Bool bAudRestoreTimerCb(timer_t rTimerID, t_Void *pvObject,
            const t_Void *pvUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::bAudReleaseTimerCb
    ***************************************************************************/
   /*!
    * \fn     bAudReleaseTimerCb
    * \brief  called on expiry of audio release timer
    * \param  rTimerID: ID of the timer which has expired
    * \param  pvObject: pointer to object passed while starting the timer
    * \param  pvUserData: data passed during start of the timer
    **************************************************************************/
   static t_Bool bAudReleaseTimerCb(timer_t rTimerID, t_Void *pvObject,
            const t_Void *pvUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vdVocoderSampleRateCheck
    ***************************************************************************/
   /*!
    * \fn     vdVocoderSampleRateCheck
    * \brief  called to set the proper value for Audio Sample Rate
    * \param  dVocoderSampleRate: Sample Rate recieved from ADIT.
    **************************************************************************/
   t_U16 u16VocoderSampleRatefromDouble(t_Double dVocoderSampleRate);

   //! Map to keep the Audio sampling rate info
   std::map<t_U16, tenAudioSamplingRate> m_mapAudioSamplingRate;

   //! Stores the currently active audio source
   tenDiPOEntity m_enCurrentAudioMode;

   //! Stores the currently active Main audio direction
   tenAudioDir m_enCurrAudioMainSrc;

   //! Stores the currently active Alternate audio direction
   tenAudioDir m_enCurrAudioAltSrc;

   //! Sets dipo transfer type
   tenDiPOTransferType m_enTransferType;

   //! Stores current device used in carplay session
   t_U32 m_u32DeviceHandle;

   t_Bool m_bIsSessionActive;

   //! Stores the current sampling rate of the main channel
   tenAudioSamplingRate m_enAudioSamplingRate;

   //! Stores the enabled/disabled status of Audio Ducking feature.
   t_Bool m_bAudDuckEnabled;

   Lock m_oSessionStateLock;

   tenCarplayStreamState m_enMediaStreamState;

   Lock m_oStreamStateLock;

   //! Structure object for Function pointers . 
   //! This will be used by Audio Manager to register for response callbacks from ML and DiPo Audio
   trAudioCallbacks m_rAudCallbacks;

   //! Pointer to Dipo cmd resource manager
   spi_tclDiPOCmdRsrcMngr* m_poDiPOCmdRsrcMngr;
};
#endif // SPI_TCLDIPOAUDIO_H
