/***********************************************************************/
/*!
* \file  spi_tclDiPoAppMngr.h
* \brief DiPo App Mngr Implementation
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPo App Mngr Implementation
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
16.02.2014  | Shiva Kumar Gurija    | Initial Version
20.03.2014  | Shihabudheen P M      | Modified 1.vLaunchApp
27.06.2014  | Hari Priya E R        | Included Siri Action handling in LaunchApp
18.07.2014  | Shihabudheen P M      | Added vSetVehicleConfig
05.06.2015  | Tejaswini HB          | Added lint comments to suppress C++11 errors
06.05.2015  | Tejaswini HB          | Lint Fix
17.07.2015  | Sameer Chandra        | Memory leak fix
15.10.2015  | Shihabudheen P M      | Added vUpdateConfigInfo

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
//#define VNC_USE_STDINT_H                   //Commented to Fix Lint errors,May be required for future use
#include <string.h>
#include <chrono>
#include "DiPOTypes.h"
#include "RespRegister.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdMetaData.h"
#include "spi_tclAppMngrDefines.h"
#include "spi_tclDiPOCmdVideo.h"
#include "spi_tclDiPOCmdInput.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPoAppMngr.h"

#include "spi_tclExtCmdNavData.h"

using namespace std;

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_APPMNGR
#include "trcGenProj/Header/spi_tclDiPoAppMngr.cpp.trc.h"
#endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
static spi_tclDiPOManager* spoDiPOManager = NULL;
static spi_tclDiPOCmdVideo* spoDiPOCmdVideo = NULL;
static spi_tclDiPOCmdInput* spoDiPOCmdInput = NULL;

static const t_U32 cou32PhoneCallTimerInterval = 1000U;
static const t_U32 cou32NumberOfSecondsInOneHour = 3600;
static const t_U32 cou32NumberOfSecondsInOneMinute = 60;
static const t_U32 cou32NumberOfMinutessInOneHour = 60;

/***************************************************************************
** FUNCTION:  spi_tclDiPoAppMngr::spi_tclDiPoAppMngr()
***************************************************************************/
spi_tclDiPoAppMngr::spi_tclDiPoAppMngr():m_u32SelDevHandle(0),
                                         m_u64CurrDeviceTimeInSeconds(0),
                                         m_rPhoneCallTimerID(0), m_bPhoneCallTimerRunning(false),
                                         m_rDeviceTimeTimerID(0), m_bDeviceTimeTimerRunning(false)
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::spi_tclDiPoAppMngr()"));
   
   spoDiPOManager = spi_tclDiPOManager::getInstance();
   if (NULL != spoDiPOManager)
   {
       spoDiPOCmdVideo = spoDiPOManager->poGetDiPOVideoInstance();
       spoDiPOCmdInput = spoDiPOManager->poGetDiPOInputInstance();
   }//if(NULL != spoVncManager )
}

/***************************************************************************
** FUNCTION:  spi_tclDiPoAppMngr::~spi_tclDiPoAppMngr()
***************************************************************************/
spi_tclDiPoAppMngr::~spi_tclDiPoAppMngr()
{
   ETG_TRACE_USR1((" spi_tclDiPoAppMngr::~spi_tclDiPoAppMngr()"));
   m_u32SelDevHandle = 0;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoConnection::vOnSessionMsg(...
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
         tenDiPOSessionTransport enSessionDipoTransport,
         t_String szSessionIPAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vOnSessionMsg entered with state: %d, Session Transport:%d, Session IP Address:%s",
            ETG_ENUM(DIPO_SESSION_STATE, enDiPOSessionState),
            ETG_ENUM(DIPO_SESSION_TRANSPORT,enSessionDipoTransport),
            szSessionIPAddress.c_str()));

   if(e8DIPO_SESSION_START == enDiPOSessionState)
   {
      vTriggerNowPlayingStatus();//Fix for NCG3D-221927
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoAppMngr::bInitialize()
***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bInitialize()
{
   ETG_TRACE_USR1((" spi_tclDiPoAppMngr::bInitialize()" ));
   //Add code
   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
      pRespRegister->bRegisterObject(dynamic_cast<spi_tclExtRespMetaData*>(this));
      pRespRegister->bRegisterObject(dynamic_cast<spi_tclExtRespNavData*>(this));
      pRespRegister->bRegisterObject(dynamic_cast<spi_tclDiPORespSession*>(this));
   }
   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vUnInitialize()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vUnInitialize()
{
   ETG_TRACE_USR1((" spi_tclDiPoAppMngr::vUnInitialize()"));
   //Add code
}
   /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclDiPoAppMngr::vRegisterAppMngrCallbacks()
   ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vRegisterAppMngrCallbacks(const trAppMngrCallbacks& corfrAppMngrCbs)
{
   //Add code
   m_rAppMngrCallbacks = corfrAppMngrCbs;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   /*lint -esym(40,fpvSelectDeviceResult)fpvSelectDeviceResult Undeclared identifier */
   ETG_TRACE_USR1((" spi_tclDiPoAppMngr::vSelectDevice:Dev-0x%x",corfrSelectReq.m_u32DeviceHandle));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq.m_enDevConnReq);

   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdMetaDataIntf *poExtCmdMetaDataIntf = NULL;
   if(NULL != poExtCompMgr)
   {
	   poExtCmdMetaDataIntf = poExtCompMgr->poGetCmdMetaDataIntfInst();
   }
   if(NULL != poExtCmdMetaDataIntf)
   {
	   (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) ? poExtCmdMetaDataIntf->vRegisterForMetadataProperties() :
                                              poExtCmdMetaDataIntf->vUnregisterForMetadataProperties();
   }

   if(NULL != m_rAppMngrCallbacks.fpvSelectDeviceResult)
   {
      (m_rAppMngrCallbacks.fpvSelectDeviceResult)(true);
   }//if(NULL != m_rAppMngrCallbacks.fpvSelectDeviceResult)
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoAppMngr::bLaunchApp()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vLaunchApp(const t_U32 cou32DevId, 
                                      t_U32 u32AppHandle,
                                      const trUserContext& rfrcUsrCntxt, 
                                      tenDiPOAppType enDiPOAppType, 
                                      t_String szTelephoneNumber, 
                                      tenEcnrSetting enEcnrSetting)
{
	/*lint -esym(40,fpvLaunchAppResult)fpvLaunchAppResult Undeclared identifier */
   ETG_TRACE_USR1((" spi_tclDiPoAppMngr::vLaunchApp:DevID:0x%x AppID:0x%x,AppType:%d",
      cou32DevId,u32AppHandle,ETG_ENUM(APPLICATION_TYPE,enDiPOAppType)));
   SPI_INTENTIONALLY_UNUSED(enEcnrSetting);

   t_Bool bIsSiriAction = false;

   //! URL of the application to launch
   t_Char szAppUrl[MAX_STR_LEN]={0};
   SiriAction enSiriAction=SiriAction_NA;

   // Assigning the URL values for different Apps as per the Apple Specification
   switch ((t_U8) enDiPOAppType)
   {
      case e8DIPO_NOT_USED:
      case e8DIPO_NO_URL:
      {
         // Message to launch the home screen
         strncpy(szAppUrl, "", MAX_STR_LEN);
      }
         break;
      case e8DIPO_MAPS:
      {
         // Message to launch the Maps
         strncpy(szAppUrl, "maps:", MAX_STR_LEN);
      }
         break;
      case e8DIPO_MOBILEPHONE:
      {
         // Message to launch the Mobilephone application
         strncpy(szAppUrl, "mobilephone:", MAX_STR_LEN);
      }
         break;
      case e8DIPO_TEL_NUMBER:
      {
         // Message to launch the phone call with given number
         strncpy(szAppUrl, "tel:", MAX_STR_LEN);
         strncat(szAppUrl, szTelephoneNumber.c_str(), szTelephoneNumber.size());
      }
         break;
      case e8DIPO_SIRI_PREWARN:
      {
         enSiriAction = SiriAction_Prewarm;
         bIsSiriAction = true;
      }
         break;

      case e8DIPO_SIRI_BUTTONDOWN:
      {
         enSiriAction = SiriAction_ButtonDown;
         bIsSiriAction = true;
      }
         break;
      case e8DIPO_SIRI_BUTTONUP:
      {
         enSiriAction = SiriAction_ButtonUp;
         bIsSiriAction = true;
      }
         break;
      case e8DIPO_MUSIC:
      {
         strncpy(szAppUrl, "music:", MAX_STR_LEN);
      }
         break;
      case e8DIPO_NOWPLAYING:
      {
         strncpy(szAppUrl, "nowplaying:", MAX_STR_LEN);
      }
      break;
      default:
         bIsSiriAction = false;
         break;
   }//End of switch


   if (true == bIsSiriAction)
   {
      if(NULL != spoDiPOCmdInput)
      {
         spoDiPOCmdInput->vUpdateSiriActionEvent(enSiriAction);
      }
   }
   else
   {
      if(NULL != spoDiPOCmdVideo)
      {
         spoDiPOCmdVideo -> vRequestUI(szAppUrl);
      }
   }


   if( NULL != m_rAppMngrCallbacks.fpvLaunchAppResult)
   {
      (m_rAppMngrCallbacks.fpvLaunchAppResult)(cou32DevId,u32AppHandle,enDiPOAppType,
               e8NO_ERRORS,rfrcUsrCntxt);
   }//if( NULL != m_rAppMngrCallbacks.fpvLaunchAppResult)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vTerminateApp()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vTerminateApp(const t_U32 cou32DevId,
                                         const t_U32 cou32AppId,
                                         const trUserContext& rfrcUsrCntxt)
{
   /*lint -esym(40,fpvTerminateAppResult)fpvTerminateAppResult Undeclared identifier */
   ETG_TRACE_USR1((" %s entered", __PRETTY_FUNCTION__));

   //Post the success as a Result to tcl App Mngr
   if( NULL != m_rAppMngrCallbacks.fpvTerminateAppResult )
   {
      (m_rAppMngrCallbacks.fpvTerminateAppResult)(cou32DevId,cou32AppId,
         e8NO_ERRORS,rfrcUsrCntxt);
   }//if( NULL != m_rAppMngrCallbacks.fpvTerminateAppResult )
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vGetAppIconData()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vGetAppIconData(t_String szAppIconUrl, 
                                           const trUserContext& rfrcUsrCntxt)
{
	/*lint -esym(40,fpvCbAppIconDataResult)fpvCbAppIconDataResult Undeclared identifier */
   ETG_TRACE_USR1((" %s entered", __PRETTY_FUNCTION__));
   SPI_INTENTIONALLY_UNUSED(szAppIconUrl);
   //Post rsponse to HMI
   if( NULL != m_rAppMngrCallbacks.fpvCbAppIconDataResult)
   {
      (m_rAppMngrCallbacks.fpvCbAppIconDataResult)(e8ICON_INVALID,NULL,0,rfrcUsrCntxt);
   }//if( NULL != m_rAppMngrCallbacks.fpvCbAppIconDataResult)
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoAppMngr::bCheckAppValidity()
***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bCheckAppValidity(const t_U32 cou32DevId, 
                                             const t_U32 cou32AppId)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(cou32AppId);
   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vSetVehicleConfig()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vSetVehicleConfig(tenVehicleConfiguration enVehicleConfig,
                         t_Bool bSetConfig)
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vSetVehicleConfig: Vehicle Config:%d, Enabled-%d",
   ETG_ENUM(VEHICLE_MODE,enVehicleConfig), ETG_ENUM(ENABLED_INFO,bSetConfig)));

   vUpdateConfigInfo(enVehicleConfig);
   if(NULL != spoDiPOCmdVideo)
   {
     spoDiPOCmdVideo -> vSetVehicleMode(enVehicleConfig);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vUpdateConfigInfo()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vUpdateConfigInfo(const tenVehicleConfiguration enVehicleConfig)
{

   switch(enVehicleConfig)
   {
      case e8PARK_MODE:
      case e8DRIVE_MODE:
      {
         if(NULL != m_poAppSettings)
         {
            m_poAppSettings->vSetDriveModeInfo(enVehicleConfig);
         }
      }
      break;
      case e8_DAY_MODE:
      case e8_NIGHT_MODE:
      {
         if(NULL != m_poAppSettings)
         {
            m_poAppSettings->vSetNightModeInfo(enVehicleConfig);
         }
      }
      break;
      default:
         break;
   }
}


t_Void spi_tclDiPoAppMngr::vAppMetaDataCallback(
         const trAppMediaMetaData& rfcorApplicationMediaMetaData,
         const trUserContext& rfcorUsrCntxt)
{

   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vAppMetaDataCallback  Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaMetaData)
   {
      (m_rAppMngrCallbacks.fvAppMediaMetaData)(rfcorApplicationMediaMetaData, rfcorUsrCntxt);
   }

}

t_Void spi_tclDiPoAppMngr::vAppMediaPlaytimeCb(
         const trAppMediaPlaytime& rfcorApplicationMediaPlaytime,
         const trUserContext& rfcorUsrCntxt)
{

   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vAppMediaPlaytimeCb  Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaPlaytime)
   {
      (m_rAppMngrCallbacks.fvAppMediaPlaytime)(rfcorApplicationMediaPlaytime,rfcorUsrCntxt);
   }

}

t_Void spi_tclDiPoAppMngr::vAppPhoneDataCb(
   const trAppPhoneData& rfcorApplicationPhoneData,
   const trUserContext& rfcorUsrCntxt)
{

   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vAppPhoneDataCb  Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppPhoneData)
   {
     (m_rAppMngrCallbacks.fvAppPhoneData)(rfcorApplicationPhoneData,rfcorUsrCntxt);
   }
}

t_Void spi_tclDiPoAppMngr::vAppMediaAlbumArtCb(const trAppMediaAlbumArt& rAppMediaAlbumArt, const trUserContext& rUserContext)
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vAppMediaAlbumArtCb  Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaAlbumArt)
   {
     (m_rAppMngrCallbacks.fvAppMediaAlbumArt)(rAppMediaAlbumArt,rUserContext);
   }
}

t_Void spi_tclDiPoAppMngr::vGetMediaObjectAlbumArt(const t_U32& corfu32DeviceHandle,
          const t_String& corfsAlbumArt, const trUserContext& corfrUsrCntxt)
{
   SPI_INTENTIONALLY_UNUSED(corfu32DeviceHandle);
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdMetaDataIntf *poExtCmdMetaDataIntf = NULL;
   if(NULL != poExtCompMgr)
   {
      poExtCmdMetaDataIntf = poExtCompMgr->poGetCmdMetaDataIntfInst();
   }
   if(NULL != poExtCmdMetaDataIntf)
   {
      poExtCmdMetaDataIntf->bGetDiPOAlbumArt(corfsAlbumArt,corfrUsrCntxt);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppMediaMetaDataMsg()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppMediaMetaDataMsg(const trAppMediaMetaData &rfcoAppMediaMetaData,
         const trUserContext &rfcoUserContext)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vPostAppMediaMetaDataMsg Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaMetaData)
   {
      (m_rAppMngrCallbacks.fvAppMediaMetaData)(rfcoAppMediaMetaData, rfcoUserContext);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppMediaAlbumArtMsg()
***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppMediaAlbumArtMsg(const trAppMediaAlbumArt &rfcoAppMediaAlbumArt,
         const trUserContext &rfcoUserContext)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vPostAppMediaAlbumArtMsg Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaAlbumArt)
   {
      (m_rAppMngrCallbacks.fvAppMediaAlbumArt)(rfcoAppMediaAlbumArt,rfcoUserContext);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppMediaPlaytimeMsg()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppMediaPlaytimeMsg(const trAppMediaPlaytime &rfcoAppMediaPlaytime,
         const trUserContext &rfcoUserContext)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vAppMetaDataCallback Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppMediaPlaytime)
   {
      (m_rAppMngrCallbacks.fvAppMediaPlaytime)(rfcoAppMediaPlaytime,rfcoUserContext);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppPhoneDataMsg()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppPhoneDataMsg(const trAppPhoneData& rfcorAppPhoneData,
         const trUserContext &rfcoUserContext)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vPostAppPhoneDataMsg Entered"));
   if(NULL != m_rAppMngrCallbacks.fvAppPhoneData)
   {
      (m_rAppMngrCallbacks.fvAppPhoneData)(rfcorAppPhoneData,rfcoUserContext);
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppDeviceTimeUpdateMsg()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppDeviceTimeUpdateMsg(const trDeviceTime& rfcoDeviceTime)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vPostAppDeviceTimeUpdateMsg Entered"));
   ETG_TRACE_USR2(("[FUNC]spi_tclDiPoAppMngr::vPostAppDeviceTimeUpdateMsg - rfcou64CurrDevTimeInSeconds = %d", rfcoDeviceTime.u64CurrDevTimeInSeconds));

   m_u64CurrDeviceTimeInSecondsLock.s16Lock();
   m_u64CurrDeviceTimeInSeconds = rfcoDeviceTime.u64CurrDevTimeInSeconds;
   m_u64CurrDeviceTimeInSecondsLock.vUnlock();

   vStartDeviceTimeProgressTimer();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vTriggerNowPlayingStatus()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vTriggerNowPlayingStatus()
{
   ETG_TRACE_USR1(("[DESC]: spi_tclDiPoAppMngr::vTriggerNowPlayingStatus entered"));
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdMetaDataIntf *poExtCmdMetaDataIntf = NULL;
   if(NULL != poExtCompMgr)
   {
      poExtCmdMetaDataIntf = poExtCompMgr->poGetCmdMetaDataIntfInst();
   }
   if (NULL != poExtCmdMetaDataIntf)
   {
      poExtCmdMetaDataIntf->vTriggerNowPlayingStatus();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostAppPhoneCallDataMsg()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostAppPhoneCallDataMsg(const trAppPhoneCallData& rfcorAppPhoneCallData,
         const trUserContext &rfcoUserContext)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::vPostAppPhoneCallDataMsg Entered"));
   SPI_INTENTIONALLY_UNUSED(rfcoUserContext);

   vAppPhoneDataSyncList(rfcorAppPhoneCallData);
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vPostAppPhoneCallDataMsg Displaying PhoneCallData"));
   vDisplayPhoneCallData();

   vSendAppPhoneCallData();

   if(true == bIsAnyCallInProgress())
   {
      vStartPhoneCallProgressTimer();
   }
   else
   {
      vStopPhoneCallProgressTimer();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vSendAppPhoneCallData()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vSendAppPhoneCallData()
{

   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vSendAppPhoneCallData() Entered"));
   m_rAppPhoneCallDataLock.s16Lock();
   trAppPhoneCallData orAppPhoneCallData = m_rAppPhoneCallData;
   m_rAppPhoneCallDataLock.vUnlock();

   if(NULL != m_rAppMngrCallbacks.fvAppPhoneCallData)
   {
      (m_rAppMngrCallbacks.fvAppPhoneCallData)(orAppPhoneCallData,corEmptyUsrContext);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bIsAnyCallInProgress()
 ***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bIsAnyCallInProgress()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::bIsAnyCallInProgress() Entered"));

   t_Bool anyCallInProgress = false;

   m_rAppPhoneCallDataLock.s16Lock();
   for (auto spiPhoneCallMetaData : m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList)
   {
      if((e8CALL_ACTIVE == spiPhoneCallMetaData.enPhoneCallState) ||
               (e8CALL_HELD == spiPhoneCallMetaData.enPhoneCallState) )
      {
         anyCallInProgress = true;
         break;
      }
   }
   m_rAppPhoneCallDataLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::bIsAnyCallInProgress() left with %d",ETG_ENUM(BOOL,anyCallInProgress)));

   return anyCallInProgress;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vAppPhoneDataSyncList()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vAppPhoneDataSyncList(const trAppPhoneCallData& rfcovecAppPhoneCallData)//rfcorPhoneCallExtendedMetadata)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vAppPhoneDataSyncList() Entered"));

   trAppPhoneCallData orAppPhoneCallData = rfcovecAppPhoneCallData;
   m_rAppPhoneCallDataLock.s16Lock();

   //update CallProgress values to vector received from MPlay client handler with values in
   //m_rAppPhoneCallData and then replace the m_rAppPhoneCallData with new vector
   if (0 != m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList.size())
   {
      for (auto& mpPhoneCallMetaData : orAppPhoneCallData.tvecPhoneCallExtendedMetaDataList )
      {
         for(auto spiPhoneCallMetaData : m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList)
         {
            if (0 == mpPhoneCallMetaData.szCallUUID.compare(spiPhoneCallMetaData.szCallUUID))
            {
               mpPhoneCallMetaData.rCallProgress = spiPhoneCallMetaData.rCallProgress;
               mpPhoneCallMetaData.rCallProgressRefFromDevice = spiPhoneCallMetaData.rCallProgressRefFromDevice;

               break;
            }
         }
      }
      m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList.clear();
   }

   m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList = orAppPhoneCallData.tvecPhoneCallExtendedMetaDataList;

   m_rAppPhoneCallDataLock.vUnlock();

   vUpdateOffsetBasedOnStartTimeStamp();
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vAppPhoneDataSyncList() PhoneCallData after vUpdateOffsetBasedOnStartTimeStamp"));
   vDisplayPhoneCallData();
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vUpdateOffsetBasedOnStartTimeStamp()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vUpdateOffsetBasedOnStartTimeStamp()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vUpdateOffsetBasedOnStartTimeStamp() Entered"));

   m_u64CurrDeviceTimeInSecondsLock.s16Lock();
   t_U64 u64CurrDeviceTimeInSeconds = m_u64CurrDeviceTimeInSeconds;
   m_u64CurrDeviceTimeInSecondsLock.vUnlock();

   m_rAppPhoneCallDataLock.s16Lock();
   for(auto& spiPhoneCallMetaData : m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList)
   {
      if ((0 == spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Hour) &&
               (0 == spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Minute) &&
               (0 == spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Second) &&
               ((e8CALL_ACTIVE == spiPhoneCallMetaData.enPhoneCallState) || (e8CALL_HELD == spiPhoneCallMetaData.enPhoneCallState)))
      {
         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp: CallUUID = %s",spiPhoneCallMetaData.szCallUUID.c_str()));
         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp: Call State = %d",ETG_ENUM(PHONE_CALL_STATE, spiPhoneCallMetaData.enPhoneCallState)));
         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp: StartTimeStamp = %s",std::to_string(spiPhoneCallMetaData.u64StartTimeStamp).c_str()));
         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp: Curr Device Time = %s",std::to_string(u64CurrDeviceTimeInSeconds).c_str()));

         t_S64 offset = (u64CurrDeviceTimeInSeconds - spiPhoneCallMetaData.u64StartTimeStamp);

         if(0 < offset)
         {
            spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Hour = static_cast<t_U8>(offset/cou32NumberOfSecondsInOneHour);
            offset%=cou32NumberOfSecondsInOneHour;
            spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Minute = static_cast<t_U8>(offset/cou32NumberOfSecondsInOneMinute);
            offset%=cou32NumberOfSecondsInOneMinute;
            spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Second = static_cast<t_U8>(offset);
         }

         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp CallProgress = %d hour %d minutes %d seconds",spiPhoneCallMetaData.rCallProgress.u8Hour,
                  spiPhoneCallMetaData.rCallProgress.u8Minute, spiPhoneCallMetaData.rCallProgress.u8Second));
         ETG_TRACE_USR1(("vUpdateOffsetBasedOnStartTimeStamp CallProgressRefFromDevice = %d hour %d minutes %d seconds",spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Hour,
                  spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Minute, spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Second));
      }
   }
   m_rAppPhoneCallDataLock.vUnlock();
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vIncrementCurrentDeviceTime (...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vIncrementCurrentDeviceTime ()
{
   ETG_TRACE_USR2(("[DESC]::spi_tclDiPoAppMngr::vIncrementCurrentDeviceTime () Entered - m_u64CurrDeviceTimeInSeconds = %d",
         m_u64CurrDeviceTimeInSeconds));
   m_u64CurrDeviceTimeInSecondsLock.s16Lock();
   ++m_u64CurrDeviceTimeInSeconds;
   m_u64CurrDeviceTimeInSecondsLock.vUnlock();
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vIncrementPhoneCallTimer(...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vIncrementPhoneCallTimer(trTime& rCallProgressTimer)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vIncrementPhoneCallTimer() Entered "));

   rCallProgressTimer.u8Second++;
   if(cou32NumberOfSecondsInOneMinute == rCallProgressTimer.u8Second)
   {
      rCallProgressTimer.u8Second = 0;
      rCallProgressTimer.u8Minute++;

      if(cou32NumberOfMinutessInOneHour == rCallProgressTimer.u8Minute)
      {
         rCallProgressTimer.u8Minute = 0;
         rCallProgressTimer.u8Hour++;
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vStartPhoneCallProgressTimer(...)
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vStartPhoneCallProgressTimer()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vStartPhoneCallProgressTimer() Entered"));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (false == m_bPhoneCallTimerRunning))
   {
      poTimer->StartTimer(m_rPhoneCallTimerID, cou32PhoneCallTimerInterval, cou32PhoneCallTimerInterval, this,
               &spi_tclDiPoAppMngr::bPhoneCallTimerCb, NULL);
      m_bPhoneCallTimerRunning = true;
      ETG_TRACE_USR4(("Phone call Timer started "));
   } //End of if (NULL != poTimer)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vStopPhoneCallProgressTimer(...)
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vStopPhoneCallProgressTimer()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vStopPhoneCallProgressTimer() Entered"));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == m_bPhoneCallTimerRunning))
   {
      poTimer->CancelTimer(m_rPhoneCallTimerID);
      m_bPhoneCallTimerRunning = false;
      ETG_TRACE_USR4(("Phone call Timer stopped "));
   } //End of if (NULL != poTimer)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vStartDeviceTimeProgressTimer(...)
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vStartDeviceTimeProgressTimer()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vStartDeviceTimeProgressTimer() Entered"));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (false == m_bDeviceTimeTimerRunning))
   {
      poTimer->StartTimer(m_rDeviceTimeTimerID, cou32PhoneCallTimerInterval, cou32PhoneCallTimerInterval, this,
               &spi_tclDiPoAppMngr::bDeviceTimeTimerCb, NULL);
      m_bDeviceTimeTimerRunning = true;
      ETG_TRACE_USR4(("Device Time Timer started "));
   } //End of if (NULL != poTimer)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vStopDeviceTimeProgressTimer(...)
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vStopDeviceTimeProgressTimer()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vStopDeviceTimeProgressTimer() Entered"));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == m_bDeviceTimeTimerRunning))
   {
      poTimer->CancelTimer(m_rDeviceTimeTimerID);
      m_bDeviceTimeTimerRunning = false;
      ETG_TRACE_USR4(("Device Time Timer stopped "));
   } //End of if (NULL != poTimer)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vResetPhoneCallProgressTimer()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vResetPhoneCallProgressTimer(trTime& rCallProgress)
{
   rCallProgress.u8Hour = 0;
   rCallProgress.u8Minute = 0;
   rCallProgress.u8Second = 0;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vUpdatePhoneCallProgress()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vUpdatePhoneCallProgress()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vUpdatePhoneCallProgress() Entered"));

   m_rAppPhoneCallDataLock.s16Lock();
   for(auto& PhoneCallMetaData : m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList)
   {
      auto PhoneCallProgress = (e8CALL_ACTIVE == PhoneCallMetaData.enPhoneCallState) ||
                               (e8CALL_HELD == PhoneCallMetaData.enPhoneCallState);
      if(PhoneCallProgress)
      {
         //only if status is active/held, the call timer is incremented
         //vIncrementPhoneCall(PhoneCallMetaData);
         vIncrementPhoneCallTimer(PhoneCallMetaData.rCallProgress);
         vIncrementPhoneCallTimer(PhoneCallMetaData.rCallProgressRefFromDevice);
      }
      else
      {
         vResetPhoneCallProgressTimer(PhoneCallMetaData.rCallProgress);
         vResetPhoneCallProgressTimer(PhoneCallMetaData.rCallProgressRefFromDevice);
      }
   }
   m_rAppPhoneCallDataLock.vUnlock();
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bPhoneCallTimerCb(...)
 ******************************************************************************************************/
t_Bool spi_tclDiPoAppMngr::bPhoneCallTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::bPhoneCallTimerCb() Entered"));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   spi_tclDiPoAppMngr* poDiPoAppMngr = static_cast<spi_tclDiPoAppMngr*>(pvObject);
   if(NULL != poDiPoAppMngr)
   {
      poDiPoAppMngr->vUpdatePhoneCallProgress();

      ETG_TRACE_USR1(("bPhoneCallTimerCb : PhoneCallData"))
      poDiPoAppMngr->vDisplayPhoneCallData();

      poDiPoAppMngr->vSendAppPhoneCallData();

   }

   return true;
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bDeviceTimeTimerCb(...)
 ******************************************************************************************************/
t_Bool spi_tclDiPoAppMngr::bDeviceTimeTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::bDeviceTimeTimerCb() Entered"));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   spi_tclDiPoAppMngr* poDiPoAppMngr = static_cast<spi_tclDiPoAppMngr*>(pvObject);
   if(NULL != poDiPoAppMngr)
   {
      poDiPoAppMngr->vIncrementCurrentDeviceTime ();
   }

   return true;
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vResetAllPhoneCallData(...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vResetAllPhoneCallData()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vResetAllPhoneCallData() Entered"));

   //Reset call data list on CP end
   m_rAppPhoneCallDataLock.s16Lock();
   m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList.clear();
   m_rAppPhoneCallDataLock.vUnlock();

   m_u64CurrDeviceTimeInSecondsLock.s16Lock();
   m_u64CurrDeviceTimeInSeconds = 0;
   m_u64CurrDeviceTimeInSecondsLock.vUnlock();

   vStopDeviceTimeProgressTimer();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vDisplayPhoneCallData()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vDisplayPhoneCallData()
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vDisplayPhoneCallData() Entered"));

   m_rAppPhoneCallDataLock.s16Lock();
   for(auto& spiPhoneCallMetaData : m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList)
   {
      ETG_TRACE_USR4(("vDisplayPhoneCallData PhoneCallData of CallUUID = %s",spiPhoneCallMetaData.szCallUUID.c_str()));
      ETG_TRACE_USR4(("vDisplayPhoneCallData Call State = %d",ETG_ENUM(PHONE_CALL_STATE, spiPhoneCallMetaData.enPhoneCallState)));
      ETG_TRACE_USR4(("vDisplayPhoneCallData StartTimeStamp = %s",std::to_string(spiPhoneCallMetaData.u64StartTimeStamp).c_str()));
      ETG_TRACE_USR4(("vDisplayPhoneCallData CallProgress = %d hour %d minutes %d seconds",spiPhoneCallMetaData.rCallProgress.u8Hour,
               spiPhoneCallMetaData.rCallProgress.u8Minute, spiPhoneCallMetaData.rCallProgress.u8Second));
      ETG_TRACE_USR4(("vDisplayPhoneCallData CallProgressRefFromDevice = %d hour %d minutes %d seconds",spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Hour,
               spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Minute, spiPhoneCallMetaData.rCallProgressRefFromDevice.u8Second));
   }
   m_rAppPhoneCallDataLock.vUnlock();

}
/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostDiPORouteGuidanceUpdateMsg(...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostDiPORouteGuidanceUpdateMsg(const trDiPORouteGuidanceUpdate &corfrDiPORouteGuidanceUpdate)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vPostDiPORouteGuidanceUpdateMsg() Entered"));
   if(0 != m_u32SelDevHandle)
   {
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
      trTurnByTurnInfo rTurnByTurnInfo;
      rTurnByTurnInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
      rTurnByTurnInfo.u32DeviceHandle = m_u32SelDevHandle;
      rTurnByTurnInfo.rCarPlayTBTUpdate.isDiPoRouteGuidanceUpdateValid = true;
      rTurnByTurnInfo.rCarPlayTBTUpdate.isDiPoRouteGuidanceManeuverUpdateValid = false;
      rTurnByTurnInfo.rCarPlayTBTUpdate.rDiPORouteGuidanceUpdate = corfrDiPORouteGuidanceUpdate;
      if(NULL != m_rAppMngrCallbacks.fvUpdateTurnByTurnInfo)
      {
         m_rAppMngrCallbacks.fvUpdateTurnByTurnInfo(rTurnByTurnInfo);
      }
#else
      if (NULL != m_rAppMngrCallbacks.fvUpdateCarPlayRouteGuidanceUpdate)
      {
         trDiPORouteGuidanceUpdate rDiPORouteGuidanceUpdate = corfrDiPORouteGuidanceUpdate;
         rDiPORouteGuidanceUpdate.u32DeviceHandle = m_u32SelDevHandle;
         m_rAppMngrCallbacks.fvUpdateCarPlayRouteGuidanceUpdate(rDiPORouteGuidanceUpdate);
      }
#endif
   }
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vPostDiPORouteGuidanceManeuverUpdateMsg(...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vPostDiPORouteGuidanceManeuverUpdateMsg(
                                   const trDiPORouteGuidanceManeuverUpdate &corfrDiPORouteGuidanceManeuverUpdate)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vPostDiPORouteGuidanceManeuverUpdateMsg() Entered"));
   if (0 != m_u32SelDevHandle)
   {
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
      trTurnByTurnInfo rTurnByTurnInfo;
      rTurnByTurnInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
      rTurnByTurnInfo.u32DeviceHandle = m_u32SelDevHandle;
      rTurnByTurnInfo.rCarPlayTBTUpdate.isDiPoRouteGuidanceUpdateValid = false;
      rTurnByTurnInfo.rCarPlayTBTUpdate.isDiPoRouteGuidanceManeuverUpdateValid = true;
      rTurnByTurnInfo.rCarPlayTBTUpdate.rDiPORouteGuidanceManeuverUpdate = corfrDiPORouteGuidanceManeuverUpdate;
      if(NULL != m_rAppMngrCallbacks.fvUpdateTurnByTurnInfo)
      {
         m_rAppMngrCallbacks.fvUpdateTurnByTurnInfo(rTurnByTurnInfo);
      }
#else
      if (NULL != m_rAppMngrCallbacks.fvUpdateCarPlayRouteGuidanceManeuverUpdate)
      {
         trDiPORouteGuidanceManeuverUpdate rDiPORouteGuidanceManeuverUpdate = corfrDiPORouteGuidanceManeuverUpdate;
         rDiPORouteGuidanceManeuverUpdate.u32DeviceHandle = m_u32SelDevHandle;
         m_rAppMngrCallbacks.fvUpdateCarPlayRouteGuidanceManeuverUpdate(rDiPORouteGuidanceManeuverUpdate);
      }
#endif
   }
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vOnSelectDeviceResult(...)
 ******************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vOnSelectDeviceResult(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq,
                                                 const tenResponseCode coenRespCode)
{
    ETG_TRACE_USR1(("[DESC]::spi_tclDiPoAppMngr::vOnSelectDeviceResult() cou32DevId = %d, coenConnReq = %d coenRespCode = %d",
                             cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq), ETG_ENUM(RESPONSE_CODE, coenRespCode)));

    spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
    spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;

    if(NULL != poExtCompMgr)
    {
        poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
    }

    if ((e8DEVCONNREQ_SELECT == coenConnReq) && (e8SUCCESS == coenRespCode))
    {
        //Store the selected device handle
        m_u32SelDevHandle = cou32DevId;
      //@TODO: vDipoStartRouteGuidanceUpdate to be removed once HMI adapts to bOnStartRouteGuidanceUpdates interface
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
      if(NULL != poExtCmdNavDataIntf && true == m_bDiPOConfigData.bTBTUpdateEnabled)
      {
         poExtCmdNavDataIntf->vDipoStartRouteGuidanceUpdate(cou32DevId);
      }
#else
      if ((NULL != poExtCmdNavDataIntf) && (false == m_rStartRouteGuidanceUpdates.vecDisplayComponent.empty()))
      {
         //when the session is active, send StartRouteGuidanceUpdates request to handle AutoStartStop RouteGuidance
         ETG_TRACE_USR1(("[DESC]: spi_tclDiPoAppMngr::vOnSelectDeviceResult - Sending pending StartRouteGuidance requests"));

         poExtCmdNavDataIntf->bOnStartRouteGuidanceUpdates(m_u32SelDevHandle, m_rStartRouteGuidanceUpdates);
      }
#endif
   }
   else if (e8DEVCONNREQ_DESELECT == coenConnReq)
   {
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
      if(NULL != poExtCmdNavDataIntf && true == m_bDiPOConfigData.bTBTUpdateEnabled)
      {
         poExtCmdNavDataIntf->vDipoStopRouteGuidanceUpdate(cou32DevId);
      }
#else
      trStopRouteGuidanceUpdates rStopRouteGuidanceUpdates;
      //when the session is deactivated, send StopRouteGuidanceUpdates request to handle AutoStartStop RouteGuidance
      ETG_TRACE_USR1(("[DESC]: spi_tclDiPoAppMngr::vOnSelectDeviceResult - Sending auto StopRouteGuidance requests"));

      // On device deselection, send the stop route guidance updates to handle AutoStartStop RouteGuidance.
      // Do not clear the map as it is required to send StartRouteGuidance request on next device selection.
      if (false == m_rStartRouteGuidanceUpdates.vecDisplayComponent.empty())
      {
         rStopRouteGuidanceUpdates.vecDisplayComponent.assign(m_rStartRouteGuidanceUpdates.vecDisplayComponent.begin(), m_rStartRouteGuidanceUpdates.vecDisplayComponent.end());
         poExtCmdNavDataIntf->bOnStopRouteGuidanceUpdates(m_u32SelDevHandle, rStopRouteGuidanceUpdates);
      }

      // Clear the route guidance updates on session termination
      vClearRouteGuidanceUpdates();
#endif
      // Reset Phone Call Timer data
      vResetAllPhoneCallData();
      //Clear the selected device handle
      m_u32SelDevHandle = 0;
   }
}

/***********************************************************************************************************************
** FUNCTION:  spi_tclDiPoAppMngr::vSetConfigData(..)
************************************************************************************************************************/
t_Void spi_tclDiPoAppMngr::vSetConfigData(const trConfigData& rfrConfigData)
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vSetConfigData entered"));
   m_bDiPOConfigData = rfrConfigData;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bOnRegRouteGuidanceDisplayConf()
 ***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bOnRegRouteGuidanceDisplayConf(
         const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::bOnRegRouteGuidanceDisplayConf Entered"));
   t_Bool bRet = false;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;

   if (NULL != poExtCompMgr)
   {
      poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
   }
   if (NULL != poExtCmdNavDataIntf)
   {
      bRet = poExtCmdNavDataIntf->bOnRegRouteGuidanceDisplayConf(corfvecRouteGuidanceDisplayComponent);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bOnStartRouteGuidanceUpdates()
 ***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bOnStartRouteGuidanceUpdates(trStartRouteGuidanceUpdates& corfrStartRouteGuidanceUpdates)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::bOnStartRouteGuidanceUpdates Entered"));
   t_Bool bRet = false;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;

   if (NULL != poExtCompMgr)
   {
      poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
      if (NULL != poExtCmdNavDataIntf)
      {
         if (true == corfrStartRouteGuidanceUpdates.bNeedAutoStartStopTBTUpdates)
         {
            for(auto enDisplayComponent : corfrStartRouteGuidanceUpdates.vecDisplayComponent)
            {
               auto itrDispComp = std::find(m_rStartRouteGuidanceUpdates.vecDisplayComponent.begin(), m_rStartRouteGuidanceUpdates.vecDisplayComponent.end(), enDisplayComponent);
               if (m_rStartRouteGuidanceUpdates.vecDisplayComponent.end() == itrDispComp)
               {
                  m_rStartRouteGuidanceUpdates.vecDisplayComponent.push_back(enDisplayComponent);
               }
            }
            m_rStartRouteGuidanceUpdates.bSourceName |= corfrStartRouteGuidanceUpdates.bSourceName;
            m_rStartRouteGuidanceUpdates.bSourceSupportsRouteGuidance |= corfrStartRouteGuidanceUpdates.bSourceSupportsRouteGuidance;

            // Handling Auto Start/Stop RouteGuidanceUpdates based on session status
            if (0 != m_u32SelDevHandle)
            {
               // If CarPlay session is active, send StartRouteGuidanceUpdates request to phone
               bRet = poExtCmdNavDataIntf->bOnStartRouteGuidanceUpdates(m_u32SelDevHandle, corfrStartRouteGuidanceUpdates);
            }
            else
            {
               // If CarPlay session is not active, push the request to queue and send it once CarPlay session becomes active
               ETG_TRACE_USR2(("[DESC]:bOnStartRouteGuidanceUpdates - CarPlay session inactive. StartRouteGuidanceUpdates request added to queue and will be sent after CarPlay session is active"));
            }
         }
         else
         {
            if (0 != m_u32SelDevHandle)
            {
               bRet = poExtCmdNavDataIntf->bOnStartRouteGuidanceUpdates(m_u32SelDevHandle,
                        corfrStartRouteGuidanceUpdates);
            }
            else
            {
               ETG_TRACE_ERR(("[ERR]:spi_tclDiPoAppMngr::bOnStartRouteGuidanceUpdates - CarPlay session inactive, StartRouteGuidanceUpdates request discarded"));
            }
         }
      }
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::bOnStopRouteGuidanceUpdates()
 ***************************************************************************/
t_Bool spi_tclDiPoAppMngr::bOnStopRouteGuidanceUpdates(trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDiPoAppMngr::bOnStopRouteGuidanceUpdates Entered"));
   t_Bool bRet = false;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;

   if (NULL != poExtCompMgr)
   {
      poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
   }
   // Clear the StartRouteGuidance request from the map for the components that has received StopRouteGuidance request
   for(auto enDisplayComponent : corfStopRouteGuidanceUpdates.vecDisplayComponent)
   {
      auto itrDispComp = std::find(m_rStartRouteGuidanceUpdates.vecDisplayComponent.begin(), m_rStartRouteGuidanceUpdates.vecDisplayComponent.end(), enDisplayComponent);
      if (m_rStartRouteGuidanceUpdates.vecDisplayComponent.end() != itrDispComp)
      {
         m_rStartRouteGuidanceUpdates.vecDisplayComponent.erase(itrDispComp);
      }
   }
   if (NULL != poExtCmdNavDataIntf)
   {
      if (0 != m_u32SelDevHandle)
      {
         bRet = poExtCmdNavDataIntf->bOnStopRouteGuidanceUpdates(m_u32SelDevHandle, corfStopRouteGuidanceUpdates);
      }
      else
      {
         ETG_TRACE_USR1(("CarPlay session is inactive. Hence, StopRouteGuidance request is discarded"));
      }
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vClearRouteGuidanceUpdates()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vClearRouteGuidanceUpdates()
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vClearRouteGuidanceUpdates entered"));
   trDiPORouteGuidanceUpdate rDiPORouteGuidanceUpdate;
   rDiPORouteGuidanceUpdate.u32DeviceHandle = m_u32SelDevHandle;
   rDiPORouteGuidanceUpdate.bIsDisplayComponentListAvailable = true;
   // Send the DisplayComponentList as empty so that updates are sent to all components
   rDiPORouteGuidanceUpdate.vecDisplayComponentList.clear();
   rDiPORouteGuidanceUpdate.bIsRouteGuidanceStateAvailable = true;
   // Set to No Route so that HMI clears the cache
   rDiPORouteGuidanceUpdate.enRouteGuidanceState = e8DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET;
   m_rAppMngrCallbacks.fvUpdateCarPlayRouteGuidanceUpdate(rDiPORouteGuidanceUpdate);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAppMngr::vRestoreSettings()
 ***************************************************************************/
t_Void spi_tclDiPoAppMngr::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclDiPoAppMngr::vRestoreSettings entered"));
   m_rStartRouteGuidanceUpdates.vecDisplayComponent.clear();
   m_rStartRouteGuidanceUpdates.bSourceName = false;
   m_rStartRouteGuidanceUpdates.bSourceSupportsRouteGuidance = false;
   m_rStartRouteGuidanceUpdates.bNeedAutoStartStopTBTUpdates = false;
}
///////////////////////////////////////////////////////////////////////////////

//lint -restore
// <EOF>
