
/*!
*******************************************************************************
* \file          spi_tclDiPODeviceAdapter.cpp
* \brief         CarPlay Device Adapter impmentation.
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    CarPlay Device Adapter impmentation
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
10.11.2016 |  Shiva Kumar G               | Initial Version
09.05.2019  |BDU6KOR                      | Gen4 COmpiler warnings resolved

\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "Timer.h"
#include "IPCMessageQueue.h"
#include "spi_tclDiPODeviceAdapter.h"

#include "spi_tclDiPOMsgQInterface.h"
#include "spi_tclDiPODiscovererDispatcher.h"
#include "spi_tclDiPOAdapterMsgQInterface.h"


#define SPI_ENABLE_DLT //enable DLT
#define SPI_LOG_CLASS Spi_CarPlay
#include "Trace.h"
LOG_IMPORT_CONTEXT(Spi_CarPlay);

/******************************************************************************
| typedefs (scope: module-global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
ICarPlayDeviceReceiver* spi_tclDiPODeviceAdapter::m_poCarplayDeviceAdapterRcvr = NULL;
spi_tclDiPODeviceAdapter* spi_tclDiPODeviceAdapter::m_poDiPODeviceAdapter = NULL;


/***************************************************************************
** FUNCTION: spi_tclDiPODeviceAdapter::spi_tclDiPODeviceAdapter()
***************************************************************************/
spi_tclDiPODeviceAdapter::spi_tclDiPODeviceAdapter()
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::spi_tclDiPODeviceAdapter()"));
}


/***************************************************************************
** FUNCTION: spi_tclDiPODeviceAdapter::~spi_tclDiPODeviceAdapter()
***************************************************************************/
spi_tclDiPODeviceAdapter::~spi_tclDiPODeviceAdapter()
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::~spi_tclDiPODeviceAdapter()"));
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclDiPODeviceAdapter::Initialize()
***************************************************************************/
t_Bool spi_tclDiPODeviceAdapter::Initialize(ICarPlayDeviceReceiver& rfrDeviceAdapterRcvr)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::Initialize entered"));
   m_poDiPODeviceAdapter = this;
   m_poCarplayDeviceAdapterRcvr = &rfrDeviceAdapterRcvr;
   vSetDeviceDiscoveryState(true);
   return true;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPODeviceAdapter::OnDeviceUpdated()
***************************************************************************/
t_Void spi_tclDiPODeviceAdapter::OnDeviceUpdated(const CarPlayDevice& corfrCarPlayDeviceInfo, t_Bool bDeviceConnStatus)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::OnDeviceUpdated: is Device Connected-%d", ETG_ENUM(BOOL, bDeviceConnStatus)));

   trDeviceInfoMsg orDeviceInfoMsg;
   //Set Message Type
   orDeviceInfoMsg.enMsgType = e8DEVICE_INFO_MESSAGE;
   orDeviceInfoMsg.bDeviceConnStatus = bDeviceConnStatus;

   if (NULL != corfrCarPlayDeviceInfo.deviceName)
   {
       ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::OnDeviceUpdated: Device Name:%s", (char*)corfrCarPlayDeviceInfo.deviceName));
       strncpy(orDeviceInfoMsg.szDeviceName, corfrCarPlayDeviceInfo.deviceName, MAX_STR_LEN - 1);
   }
   if (NULL != corfrCarPlayDeviceInfo.btMac)
   {
       ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::OnDeviceUpdated: BT MAC:%s", (char*)corfrCarPlayDeviceInfo.btMac));
       strncpy(orDeviceInfoMsg.szBTMACAddress, szCovertBTAddrToSPIFormat(corfrCarPlayDeviceInfo.btMac).c_str(), MAX_STR_LEN - 1);
   }
   if (NULL != corfrCarPlayDeviceInfo.deviceVersion)
   {
       ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::OnDeviceUpdated: Device Version:%s", (char*)corfrCarPlayDeviceInfo.deviceVersion));
       strncpy(orDeviceInfoMsg.szDeviceVersion, corfrCarPlayDeviceInfo.deviceVersion, MAX_STR_LEN - 1);
   }

   spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
   t_Bool bStatus = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&orDeviceInfoMsg, sizeof(trDeviceInfoMsg)) : false;

   ETG_TRACE_USR4(("spi_tclDiPODeviceAdapter::OnDeviceUpdated(): Message sent status:%d", ETG_ENUM(BOOL, bStatus)));
}


/***************************************************************************
** FUNCTION: t_Void spi_tclDiPODeviceAdapter::vConnectDevice()
***************************************************************************/
t_Void spi_tclDiPODeviceAdapter::vConnectDevice(const char* cszBTMACAddress)
{

   if (NULL != m_poDiPODeviceAdapter)
   {
      if (NULL != cszBTMACAddress)
      {
         ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vConnectDevice:BT MAC Address:%s",  cszBTMACAddress));

	     CAEStatus enConnectStatus = m_poDiPODeviceAdapter->enSendConnectDeviceReq(cszBTMACAddress);
		 
		 ETG_TRACE_USR2(("spi_tclDiPODeviceAdapter::vConnectDevice:ConnectionStatus:%d",ETG_ENUM(DIPO_CONNECT_ERROR,enConnectStatus)));
		 
		 m_poDiPODeviceAdapter->vSendConnectDeviceResp(cszBTMACAddress,(CarPlayAdapterError_NoError == enConnectStatus));

      }
      else
      {
         ETG_TRACE_ERR(("[ERR]::spi_tclDiPODeviceAdapter::vConnectDevice: cszBTMACAddress is NULL"));
         m_poDiPODeviceAdapter->vSendConnectDeviceResp(cszBTMACAddress, CarPlayAdapterError_Generic);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::spi_tclDiPODeviceAdapter::vConnectDevice: m_poDiPODeviceAdapter is NULL"));
	  m_poDiPODeviceAdapter->vSendConnectDeviceResp(cszBTMACAddress, CarPlayAdapterError_Generic);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPODeviceAdapter::vDisconnectDevice()
***************************************************************************/
t_Void spi_tclDiPODeviceAdapter::vDisconnectDevice(const char* cszBTMACAddress,
                                                  tenConnectTriggerReason enConnectTriggerReason)
{
   if ((NULL != m_poCarplayDeviceAdapterRcvr) && (NULL != cszBTMACAddress))
   {
      ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vDisconnectDevice:BT MAC Address:%s", cszBTMACAddress));
      if(tenConnectTriggerReason::e8CONNECT_UNKNOWN == enConnectTriggerReason)
      {
          ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vDisconnectDevice:DisconnectTrigger:%d", 
          ETG_ENUM(CONNECT_TRIGGER,(t_U32)tenConnectTriggerReason::e8CONNECT_UNKNOWN)));
      }
      else if(tenConnectTriggerReason::e8CONNECT_ON_USER_TRIGGER == enConnectTriggerReason)
      {
          ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vDisconnectDevice:DisconnectTrigger:%d", 
          ETG_ENUM(CONNECT_TRIGGER,(t_U32)tenConnectTriggerReason::e8CONNECT_ON_USER_TRIGGER)));          
      }
      else
      {
          ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vDisconnectDevice:DisconnectTrigger:%d", 
          ETG_ENUM(CONNECT_TRIGGER,(t_U32)tenConnectTriggerReason::e8CONNECT_ON_RETRIAL)));              
      }
      CAEStatus enDisconnectStatus = m_poCarplayDeviceAdapterRcvr->DisconnectDevice(cszBTMACAddress);

      ETG_TRACE_USR4(("spi_tclDiPODeviceAdapter::vDisconnectDevice():DisonnectDevice Result:%d",
         ETG_ENUM(DIPO_CONNECT_ERROR, enDisconnectStatus)));

	  if( tenConnectTriggerReason::e8CONNECT_ON_RETRIAL != enConnectTriggerReason)
	  {
         //Prepare a new message and send the response back to SPI Process.
          trDisconnectDeviceRespMsg orDisconnectDeviceRespMsg;
          orDisconnectDeviceRespMsg.enMsgType = e8DEVICE_DISCONNECT_RESP_MESSAGE;

          //length of the string may change during conversion from SPI to CarPlay or CarPlay to PSI BT Address format.
          //check for array out of bounds and proper NULL termination of the strings.
          size_t sBTMacAddrLen = strlen(cszBTMACAddress);
          if ( (sBTMacAddrLen < MAX_STR_LEN) && (NULL != m_poDiPODeviceAdapter) )
          {
              strncpy(orDisconnectDeviceRespMsg.szBTMACAddress, m_poDiPODeviceAdapter->szCovertBTAddrToSPIFormat(cszBTMACAddress).c_str(), sBTMacAddrLen);
              orDisconnectDeviceRespMsg.szBTMACAddress[sBTMacAddrLen] = '\0';
          }
          else
          {
              ETG_TRACE_ERR(("[ERR]:spi_tclDiPODeviceAdapter::vDisconnectDevice:Received BT Address is invalid or m_poDiPODeviceAdapter is NULL."));
          }

          orDisconnectDeviceRespMsg.bDisconnectResult = (CarPlayAdapterError_NoError == enDisconnectStatus);

          spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
          t_Bool bStatus = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&orDisconnectDeviceRespMsg, sizeof(trDisconnectDeviceRespMsg)) : false;

          ETG_TRACE_USR2(("spi_tclDiPODeviceAdapter::vDisconnectDevice(): Response sent status = %d", ETG_ENUM(STATUS, bStatus)));
	  }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::spi_tclDiPODeviceAdapter::vConnectDevice: cszBTMACAddress or m_poCarplayDeviceAdapter is NULL"));
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPODeviceAdapter::vSetDeviceDiscoveryState()
***************************************************************************/
t_Void spi_tclDiPODeviceAdapter::vSetDeviceDiscoveryState(t_Bool bStartDeviceDiscovery)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vSetDeviceDiscoveryState:StartDiscovery:%d",
      ETG_ENUM(BOOL,bStartDeviceDiscovery)));

   if (NULL != m_poCarplayDeviceAdapterRcvr)
   {
      CAEStatus enSetDeviceDiscoveryStatus = CarPlayAdapterError_Generic;
      if (true == bStartDeviceDiscovery)
      {
         enSetDeviceDiscoveryStatus = m_poCarplayDeviceAdapterRcvr->StartCarPlayDeviceDiscovery();
      }//if (true == poDeviceDiscoveryMsg->bStartDeviceDiscovery)
      else
      {
         enSetDeviceDiscoveryStatus = m_poCarplayDeviceAdapterRcvr->StopCarPlayDeviceDiscovery();
      }

      ETG_TRACE_USR4(("spi_tclDiPODeviceAdapter::vSetDeviceDiscoveryState::Result:%d",
         ETG_ENUM(DIPO_CONNECT_ERROR, enSetDeviceDiscoveryStatus)));
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclDiPODeviceAdapter::vSetDeviceDiscoveryState:m_poCarplayDeviceAdapterRcvr is NULL"));
   }
}

/***************************************************************************
** FUNCTION: std::string spi_tclDiPODeviceAdapter::szCovertBTAddrToSPIFormat(const char* pcocCarplaydBTAddr)
***************************************************************************/
std::string spi_tclDiPODeviceAdapter::szCovertBTAddrToSPIFormat(const char* pcocCarplaydBTAddr)
{
   //! Initialize string with the BT MAC address
   t_String szBTAddress;
   if(NULL != pcocCarplaydBTAddr)
   {
      szBTAddress = pcocCarplaydBTAddr;
   }

   //! Remove all instances of ':' character from string and convert to uppercase.
   //! (Example: If szBTMACAddress is "28:e1:4c:df:30:72", format string as "28E14CDF3072"
   if (false == szBTAddress.empty())
   {
      szBTAddress.erase(std::remove(szBTAddress.begin(), szBTAddress.end(), ':'), szBTAddress.end());
      std::transform(szBTAddress.begin(), szBTAddress.end(), szBTAddress.begin(), ::toupper);
   }

   ETG_TRACE_USR4(("spi_tclDiPODeviceAdapter::szCovertBTAddrToSPIFormat() left with %s ", szBTAddress.c_str()));
   return szBTAddress;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPODeviceAdapter::vSendConnectDeviceResp()
***************************************************************************/
t_Void spi_tclDiPODeviceAdapter::vSendConnectDeviceResp(const char* cszBTMACAddress,
                                                        t_Bool bIsConnectSuccess)
{
     if (NULL != cszBTMACAddress)
     {
        ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::vSendConnectDeviceResp:bIsConnectSuccess:%d, BT MAC Address:%s", ETG_ENUM(BOOL,
            bIsConnectSuccess), cszBTMACAddress));

        //Prepare a new message and send the response back to SPI Process.
        trConnectDeviceRespMsg orConnectDeviceRespMsg;
        orConnectDeviceRespMsg.enMsgType = e8DEVICE_CONNECT_RESP_MESSAGE;

        //length of the string may change during conversion from SPI to CarPlay or CarPlay to PSI BT Address format.
        //check for array out of bounds and proper NULL termination of the strings.
        size_t sBTMacAddrLen = strlen(cszBTMACAddress);
        if ((sBTMacAddrLen < MAX_STR_LEN) && (NULL != m_poDiPODeviceAdapter))
        {
           strncpy(orConnectDeviceRespMsg.szBTMACAddress,
                    m_poDiPODeviceAdapter->szCovertBTAddrToSPIFormat(cszBTMACAddress).c_str(),
                    sBTMacAddrLen);
        }
        else
        {
           ETG_TRACE_ERR(("[ERR]:spi_tclDiPODeviceAdapter::vSendConnectDeviceResp:Received BT Address seems to be invalid. Length is more than defined."));
        }

        orConnectDeviceRespMsg.bConnectResult = bIsConnectSuccess;

        spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
        t_Bool bStatus = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&orConnectDeviceRespMsg, sizeof(trConnectDeviceRespMsg)) : false;
        ETG_TRACE_USR4(("spi_tclDiPODeviceAdapter::vSendConnectDeviceResp(): Response sent status:%d", ETG_ENUM(BOOL,
                 bStatus)));
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::spi_tclDiPODeviceAdapter::vSendConnectDeviceResp: cszBTMACAddress is NULL"));
   }
}

/***************************************************************************
** FUNCTION:  CAEStatus spi_tclDiPODeviceAdapter::enSendConnectDeviceReq()
***************************************************************************/
CAEStatus spi_tclDiPODeviceAdapter::enSendConnectDeviceReq(const char* cszBTMACAddress)
{
   CAEStatus enConnectStatus = CarPlayAdapterError_Generic;
   
   if (NULL != m_poCarplayDeviceAdapterRcvr)
   {
      enConnectStatus = m_poCarplayDeviceAdapterRcvr->ConnectDevice(cszBTMACAddress);
   
      ETG_TRACE_USR1(("spi_tclDiPODeviceAdapter::enSendConnectDeviceReq():ConnectDevice Result:%d",
        ETG_ENUM(DIPO_CONNECT_ERROR, enConnectStatus)));
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::spi_tclDiPODeviceAdapter::enSendConnectDeviceReq: m_poCarplayDeviceAdapter is NULL"));
   }
   
   return enConnectStatus;
}

/////////////////////////////////////<EOF>//////////////////////////////////////////
