/*!
 *******************************************************************************
 * \file             spi_tclDiscovererSettings.cpp
 * \brief            Project specific settings for Device discovery
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Project specific settings for device discovery
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 19.01.2017 |  Noopur R Kalawatia          | Initial Version


 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "FileHandler.h"
#include "crc.h"
#include "StringHandler.h"
#include "XmlDocument.h"
#include "XmlReader.h"
#include "spi_tclDiscovererSettings.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclDiscovererSettings.cpp.trc.h"
#endif
#endif

static const t_S32 scos32AnyProductID = 0;
using namespace shl::xml;

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDiscovererSettings::~spi_tclDiscovererSettings
 ***************************************************************************/
spi_tclDiscovererSettings::~spi_tclDiscovererSettings()
{
    m_oLocksetDevCat.s16Lock();
    m_setDevCat.clear();
    m_oLocksetDevCat.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiscovererSettings::vIntializeSPISettings
 ***************************************************************************/
t_Void spi_tclDiscovererSettings::vIntializeSPISettings()
{
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::vIntializeSPISettings() entered \n"));
   //Check the validity of the xml file
   if (!m_szDeviceExclusionListPath.empty())
   {
      spi::io::FileHandler oPolicySettingsFile(m_szDeviceExclusionListPath.c_str(), spi::io::SPI_EN_RDONLY);
      if (true == oPolicySettingsFile.bIsValid())
      {
         tclXmlDocument oXmlDoc(m_szDeviceExclusionListPath);
         tclXmlReader oXmlReader(&oXmlDoc, this);
         oXmlReader.bRead("DISCOVERERSETTINGS");
      } // if (true == oPolicySettingsFile.bIsValid())
   }
   else
   {
      ETG_TRACE_USR1((" spi_tclDiscovererSettings::vIntializeSPISettings() : No blacklist available\n"));
   }

   if (!m_szDeviceInclusionListPath.empty())
   {
      spi::io::FileHandler oPolicySettingsFile(m_szDeviceInclusionListPath.c_str(), spi::io::SPI_EN_RDONLY);
      if (true == oPolicySettingsFile.bIsValid())
      {
         tclXmlDocument oXmlDoc(m_szDeviceInclusionListPath);
         tclXmlReader oXmlReader(&oXmlDoc, this);
         oXmlReader.bRead("DISCOVERERSETTINGS");
      } // if (true == oPolicySettingsFile.bIsValid())
   }
   else
   {
      ETG_TRACE_USR1((" spi_tclDiscovererSettings::vIntializeSPISettings() : No whitelist available\n"));
   }
   vDisplaySPISettings();
}


/***************************************************************************
 ** FUNCTION:  spi_tclDiscovererSettings::bIsDeviceWhitelistingEnabled
 ***************************************************************************/
t_Bool spi_tclDiscovererSettings::bIsDeviceWhitelistingEnabled()
{
   return m_bIsDeviceWhitelistingEnabled;
}

/***************************************************************************
 *  ** FUNCTION: t_Void spi_tclDiscovererSettings::vSetWhitelistEnabledCat
 ***************************************************************************/
t_Void spi_tclDiscovererSettings::vSetWhitelistEnabledCat(tenDeviceCategory enDevCat)
{
    ETG_TRACE_USR1((" spi_tclDiscovererSettings::vSetWhitelistEnabledCat() entered"));
    m_oLocksetDevCat.s16Lock();
	 m_setDevCat.insert(enDevCat);
    m_oLocksetDevCat.vUnlock();
}

/***************************************************************************
 *  ** FUNCTION: t_Bool spi_tclDiscovererSettings::bIsWhitelistEnabledCat
 ***************************************************************************/
t_Bool spi_tclDiscovererSettings::bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory)
{
    ETG_TRACE_USR1((" spi_tclDiscovererSettings::bIsWhitelistEnabledCat() entered"));
    m_oLocksetDevCat.s16Lock();
    t_Bool bRetVal = false;
    std::set<tenDeviceCategory>::iterator itDevCat;
    for (itDevCat = m_setDevCat.begin(); itDevCat != m_setDevCat.end(); ++itDevCat)
    {
        ETG_TRACE_USR2(("bIsWhitelistEnabledCat::Device category for which whitelisting is enabled is %d",
             ETG_ENUM(DEVICE_CATEGORY, *itDevCat)));
        if(enChosenDeviceCategory == *itDevCat)
        {
            bRetVal = true;
            break;
        }
    }
    m_oLocksetDevCat.vUnlock();
    return bRetVal;
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDiscovererSettings::spi_tclDiscovererSettings
 ***************************************************************************/
spi_tclDiscovererSettings::spi_tclDiscovererSettings()
{
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::spi_tclDiscovererSettings() entered \n"));
   m_bUSBPort1 = true;
   m_bUSBPort2 = true;
   m_bUSBPort3 = true;
   m_bIsDeviceWhitelistingEnabled = false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiscovererSettings::vDisplaySPISettings
 ***************************************************************************/
t_Void spi_tclDiscovererSettings::vDisplaySPISettings()
{
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::vDisplaySPISettings() entered \n"));
   ETG_TRACE_USR2((" AAP Blacklisted Devices  List : \n"));

      for(auto itMapExclusionList = m_mapAAPDevExclusionList.begin();
               itMapExclusionList != m_mapAAPDevExclusionList.end();
               itMapExclusionList++)
      {
         ETG_TRACE_USR2((" VendorID = %d\t", (itMapExclusionList->first)));
         ETG_TRACE_USR2((" ProductID = %d\n",(itMapExclusionList->second)));
      }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscovererSettings::bIsDeviceBlacklisted()
 ***************************************************************************/
t_Bool spi_tclDiscovererSettings::bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID,
         tenDeviceCategory enDeviceCtegory)
{
   SPI_INTENTIONALLY_UNUSED(enDeviceCtegory);
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::bIsDeviceBlacklisted() entered \n"));
   t_Bool bRetVal = false;

   if (!m_mapAAPDevExclusionList.empty())
   {
      auto itDevList = m_mapAAPDevExclusionList.find(s32VendorID);

      if (itDevList != m_mapAAPDevExclusionList.end())
      {
         ETG_TRACE_ERR(("[ERR]:bIsDeviceBlacklisted: Vendor ID matches "));
         auto itRange = m_mapAAPDevExclusionList.equal_range(itDevList->first);

         for (auto itProdIDs = itRange.first; itProdIDs != itRange.second; itProdIDs++)
         {
            if ((s32ProductID == itProdIDs->second) || (scos32AnyProductID == (itProdIDs->second)))
            {
               bRetVal = true;
               ETG_TRACE_ERR(("[ERR]spi_tclDiscovererSettings::bIsDeviceBlacklisted - Is Device Blacklisted = %d ", ETG_ENUM(BOOL,
                        bRetVal)));
            }
         }
      }
      ETG_TRACE_USR1(("spi_tclDiscovererSettings ::bIsDeviceBlacklisted:  ProductID = %d, VendorID = %d, bRetVal = %d", s32ProductID, s32VendorID, bRetVal));
   }
   else
   {
      ETG_TRACE_USR1((" spi_tclDiscovererSettings::bIsDeviceBlacklisted() Device exclusion list is empty \n"));
   }

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiscovererSettings::vIsDeviceWhitelisted()
 ***************************************************************************/
t_Void spi_tclDiscovererSettings::vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory &rfenDevCat)
{
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::vIsDeviceWhitelisted() entered \n"));

   if (SPI_MAP_NOT_EMPTY(m_mapMLDevInclusionList))
   {
      std::pair<t_S32,t_S32> mapMLDevInclusionList = std::make_pair(s32VendorID, s32ProductID);

      std::map< std::pair<t_S32,t_S32>,tenDeviceCategory >::iterator itmapMLDevInclusionList =
         m_mapMLDevInclusionList.find(mapMLDevInclusionList);

      //! Whitelisting is added to check whether a device that is connected to HU can be allowed for a
      //! particular technology. In case of AIVI project, only A+T box has to be allowed for ML and other
      //! devices must be blocked for ML. This is done by checking whether the vendor id and product id of
      //! the device and the category that device is allowed for.

      if (itmapMLDevInclusionList != m_mapMLDevInclusionList.end())
      {
          if(rfenDevCat == e8DEV_TYPE_UNKNOWN)
          {
              rfenDevCat = itmapMLDevInclusionList->second;
          }
          else if(rfenDevCat != itmapMLDevInclusionList->second)
          {
              rfenDevCat = e8DEV_TYPE_UNKNOWN;
          }
          ETG_TRACE_USR1(("spi_tclDiscovererSettings ::vIsDeviceWhitelisted: Device with ProductID = %d, VendorID = %d is whitelisted", s32ProductID, s32VendorID));
      }
      else
      {
          rfenDevCat = e8DEV_TYPE_UNKNOWN;
          ETG_TRACE_USR1(("spi_tclDiscovererSettings ::vIsDeviceWhitelisted: Device with ProductID = %d, VendorID = %d is not whitelisted", s32ProductID, s32VendorID));
      }//if(itmapMLDevInclusionList != m_mapMLDevInclusionList

   }//if (SPI_MAP_NOT_EMPTY(m_mapMLDevInclusionList))
   else
   {
      ETG_TRACE_USR1((" spi_tclDiscovererSettings::vIsDeviceWhitelisted() Device exclusion list is empty \n"));
   }
}

/***************************************************************************
 *  ** FUNCTION:  spi_tclDiscovererSettings::bXmlReadNode
 ***************************************************************************/
t_Bool spi_tclDiscovererSettings::bXmlReadNode(xmlNodePtr poNode)
{
   ETG_TRACE_USR1((" spi_tclDiscovererSettings::bXmlReadNode() entered \n"));
   t_Bool bRetVal = false;
   t_String szNodeName;
   t_S32 s32VendorID, s32ProductID = 0;

   if (NULL != poNode)
   {
      szNodeName = (const t_Char*) (poNode->name);
   }
   if ("DEVICE_EXCLUSION_LIST" == szNodeName)
   {
      bRetVal = bGetAttribute("VENDORID", poNode, s32VendorID);
      bRetVal = bGetAttribute("PRODUCTID", poNode, s32ProductID) && bRetVal;
      if (true == bRetVal)
      {
         m_mapAAPDevExclusionList.insert(std::pair<t_S32, t_S32>(s32VendorID, s32ProductID));
      }
   }
   else if("DEVICE_INCLUSION_LIST" == szNodeName)
   {
      t_S32 s32Value = 0;
      bRetVal = bGetAttribute("VENDORID", poNode, s32VendorID);
      bRetVal = bGetAttribute("PRODUCTID", poNode, s32ProductID) && bRetVal;
      bRetVal = bGetAttribute("DEVICE_CATEGORY", poNode, s32Value) && bRetVal;
      tenDeviceCategory enDevCat = ((s32Value >= 0) && (s32Value<= e8DEV_TYPE_DEFAULT)) ?
                                    (tenDeviceCategory) s32Value : e8DEV_TYPE_UNKNOWN;
      if (true == bRetVal)
      {
          std::pair<t_S32,t_S32> mapMLDevInclusionList = std::make_pair(s32VendorID,s32ProductID);
          m_mapMLDevInclusionList.insert(std::pair<std::pair<t_S32,t_S32>,tenDeviceCategory>(mapMLDevInclusionList,enDevCat));
          ETG_TRACE_USR1(("spi_tclDiscovererSettings: Device with ProductID = %d, VendorID = %d cat: %d", s32ProductID, s32VendorID,enDevCat));
          //! m_bIsDeviceWhitelistingEnabled variable is used to tell that whitelisting is enabled for the project
          //! (when there is atleast one DEVICE_INCLUSION_LIST node in the xml file)
          m_bIsDeviceWhitelistingEnabled = true;
          vSetWhitelistEnabledCat(enDevCat);
      }
   }

   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiscovererSettings::vSetDeviceInclusionListPath()
***************************************************************************/
t_Void spi_tclDiscovererSettings::vSetDeviceListPath(t_String rfszDeviceInclusionListPath,t_String rfszDeviceExclusionListPath)
{
   ETG_TRACE_USR1(("spi_tclDiscovererSettings::vSetDeviceInclusionListPath\n"));
   m_szDeviceInclusionListPath = rfszDeviceInclusionListPath;
   m_szDeviceExclusionListPath = rfszDeviceExclusionListPath;
}


/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiscovererSettings::bIsDeviceWhitelisted
 ***************************************************************************/
t_Bool spi_tclDiscovererSettings::bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID, tenDeviceCategory enReqDevCat)
{
   ETG_TRACE_USR1(("spi_tclDiscovererSettings::bIsDeviceWhitelisted Product Id=%u Vendor d=%u",u32ProductID,u32VendorID));
   tenDeviceCategory enWhitelistedCategory = e8DEV_TYPE_UNKNOWN;

   vIsDeviceWhitelisted(u32ProductID, u32VendorID, enWhitelistedCategory);
   t_Bool bIsDeviceWhitelisted = (true == bIsDeviceWhitelistingEnabled()) && (enReqDevCat == enWhitelistedCategory);
   ETG_TRACE_USR1(("spi_tclDiscovererSettings::bIsDeviceWhitelisted: For %d,  is %s", ETG_ENUM(DEVICE_CATEGORY, enReqDevCat), (true == bIsDeviceWhitelisted) ? "whitelisted" : "not whitelisted"));
   return bIsDeviceWhitelisted;
}


