/*!
 *******************************************************************************
 * \file             spi_tclConnSettings.cpp
 * \brief            Project specific settings for Connection Management
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Project specific settings for Connection Management
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 11.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 24.11.2014 |  Shiva Kumar Gurija          | XML Validation
 08.11.2016 |  Noopur R K                  | Session Configuration for AAP
 17.02.2017 |  Noopur R K                  | Changes for Blacklisting of devices
 18.07.2017 |  Noopur R K                  | Added method GetGeneralRestictions

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "FileHandler.h"
#include "crc.h"
#include "StringHandler.h"
#include "XmlDocument.h"
#include "XmlReader.h"
#include "spi_tclConnSettings.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclConnSettings.cpp.trc.h"
#endif
#endif

static const t_Char scoczG3GXmlConfigFile[] = "/opt/bosch/spi/xml/policy.xml";
static const t_U8 scou8DefaultHistorySize = 10;
static const t_U8 scou8DefaultTimeOnStartup = 30;


using namespace shl::xml;
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclConnSettings::spi_tclConnSettings
 ***************************************************************************/
spi_tclConnSettings::spi_tclConnSettings() :
                  m_enSelPrority(e8PRIORITY_DEVICELIST_HISTORY),
                  m_szPerStoragePath("/opt/bosch/DeviceHistory.db"),
                  m_u32DevHistorySize(scou8DefaultHistorySize),
                  m_bXMLValidationEnabled(false),
                  m_enCertificateType(e8_CERTIFICATETYPE_DEVELOPER),
                  m_enAndroidDevicePreference(e8DEV_TYPE_ANDROIDAUTO),
                  m_enAppleDevicePreference(e8DEV_TYPE_DIPO),
                  m_enStoredSelectionMode(e16DEVICESEL_UNKNOWN),
                  m_enAndroidDevicePreferencefromDatapool(e8DEV_TYPE_UNKNOWN),
                  m_enAppleDevicePreferencefromDatapool(e8DEV_TYPE_UNKNOWN),
                  m_bIsCarlifeIOSEnabled(FALSE)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   //! Values initialized to default values (Used if xml policy not found)
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnSettings::~spi_tclConnSettings
 ***************************************************************************/
spi_tclConnSettings::~spi_tclConnSettings()
{

}
/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclConnSettings::bIntializeConnSettings
 ***************************************************************************/
t_Void spi_tclConnSettings::vIntializeConnSettings()
{
   const t_Char* pczConfigFilePath = NULL;
   pczConfigFilePath = scoczG3GXmlConfigFile;

   //Check the validity of the xml file
   spi::io::FileHandler oPolicySettingsFile(pczConfigFilePath,
            spi::io::SPI_EN_RDONLY);
   if (true == oPolicySettingsFile.bIsValid())
   {
      tclXmlDocument oXmlDoc(pczConfigFilePath);
      tclXmlReader oXmlReader(&oXmlDoc, this);
      oXmlReader.bRead("CONNMNGR");
   } // if (true == oPolicySettingsFile.bIsValid())

   vDisplayConnSettings();
}

/***************************************************************************
 ** FUNCTION:  tenDeviceSelectionMode spi_tclConnSettings::enGetDeviceSelectionMode
 ***************************************************************************/
tenDeviceSelectionMode spi_tclConnSettings::enGetDeviceSelectionMode(tenDeviceCategory enTechnologyType)
{
   tenDeviceSelectionMode enSelectionMode = e16DEVICESEL_UNKNOWN;
   std::map<tenDeviceCategory, tenDeviceSelectionMode>::iterator itMap = m_mapenSelMode.find(enTechnologyType);
   if(m_mapenSelMode.end() != itMap)
   {
      enSelectionMode = itMap->second;
   }
   ETG_TRACE_USR1((" [DESC]  spi_tclConnSettings::enGetDeviceSelectionMode(): Selection mode for %d is %d \n",
		   ETG_ENUM(DEVICE_CATEGORY, enTechnologyType), ETG_ENUM(SELECTION_MODE, enSelectionMode)));
   return enSelectionMode;
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetDeviceSelectionMode()
 ***************************************************************************/
t_Void spi_tclConnSettings::vSetDeviceSelectionMode(tenDeviceSelectionMode enDevSelectionMode, tenDeviceCategory enTechnologyType)
{
   if(e8DEV_TYPE_DEFAULT == enTechnologyType)
   {
      m_mapenSelMode[e8DEV_TYPE_DIPO] = enDevSelectionMode;
      m_mapenSelMode[e8DEV_TYPE_MIRRORLINK] = enDevSelectionMode;
      m_mapenSelMode[e8DEV_TYPE_ANDROIDAUTO] = enDevSelectionMode;
      m_mapenSelMode[e8DEV_TYPE_MYSPIN] = enDevSelectionMode;
      m_mapenSelMode[e8DEV_TYPE_CARLIFE] = enDevSelectionMode;
      m_mapenSelMode[e8DEV_TYPE_ONCAR] = enDevSelectionMode;
   }
   m_mapenSelMode[enTechnologyType] = enDevSelectionMode;
}

/***************************************************************************
 ** FUNCTION:  tenDeviceCategory spi_tclConnSettings::enGetTechnologyPreference
 ***************************************************************************/
tenDeviceCategory spi_tclConnSettings::enGetTechnologyPreference(tenDeviceType enDeviceType) const
{
   return (e8_ANDROID_DEVICE == enDeviceType)?m_enAndroidDevicePreference:m_enAppleDevicePreference;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclConnSettings::enGetCertificateType
 ***************************************************************************/
tenCertificateType spi_tclConnSettings::enGetCertificateType() const
{
   return m_enCertificateType;
}

/***************************************************************************
 ** FUNCTION:  tenSelModePriority spi_tclConnSettings::enGetSelectionModePriority
 ***************************************************************************/
tenSelModePriority spi_tclConnSettings::enGetSelectionModePriority()
{
   return m_enSelPrority;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vGetPersistentStoragePath
 ***************************************************************************/
t_Void spi_tclConnSettings::vGetPersistentStoragePath(
         t_String &rfszPersStoragePath)
{
   rfszPersStoragePath = m_szPerStoragePath;
}

/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclConnSettings::u32GetDeviceHistorySize
 ***************************************************************************/
t_U32 spi_tclConnSettings::u32GetDeviceHistorySize()
{
   return m_u32DevHistorySize;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclConnSettings::bIsXMLValidationEnabled
***************************************************************************/
t_Bool spi_tclConnSettings::bIsXMLValidationEnabled()  const
{
   //Add code
   return m_bXMLValidationEnabled;
}



/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConnSettings::vGetDeviceExclusionListPath
 ***************************************************************************/
t_Void spi_tclConnSettings::vGetDeviceExclusionListPath(
         t_String &rfszDeviceExclusionListPath)
{
   rfszDeviceExclusionListPath = m_szDeviceExclusionListPath;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vGetDeviceInclusionListPath
 ***************************************************************************/
t_Void spi_tclConnSettings::vGetDeviceInclusionListPath(
         t_String &rfszDeviceInclusionListPath)
{
   rfszDeviceInclusionListPath = m_szDeviceInclusionListPath;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConnSettings::bGetCarlifeIOSFeatureSupport
 ***************************************************************************/
t_Bool spi_tclConnSettings::bGetCarlifeIOSFeatureSupport()
{
   return m_bIsCarlifeIOSEnabled;
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/*************************************************************************
 ** FUNCTION:  virtual bXmlReadNode(xmlNode *poNode)
 *************************************************************************/
t_Bool spi_tclConnSettings::bXmlReadNode(xmlNodePtr poNode)
{
   t_S32 s32Value = 0;
   t_Bool bRetVal = false;
   t_String szNodeName;

   if (NULL != poNode)
   {
      szNodeName = (const char *) (poNode->name);
   } // if (NULL != poNode)

   if ("DEVICE_SELECTION_MODE" == szNodeName)
   {
     //! compare device selection mode from datapool. if the value is unknown then read it from policy.xml (virgin start)
     if(e16DEVICESEL_UNKNOWN == m_enStoredSelectionMode)
     {
        bRetVal = bGetAttribute("DEFAULT_MODE", poNode, s32Value);
        SPI_NORMAL_ASSERT(false == bRetVal);
        m_mapenSelMode[e8DEV_TYPE_DEFAULT] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                                  (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     }
     else
     {
        bRetVal = true;
		ETG_TRACE_USR4(("bXmlReadNode return value: %d",ETG_ENUM(BOOL,bRetVal)));
        m_mapenSelMode[e8DEV_TYPE_DEFAULT] = m_enStoredSelectionMode;
     }

     bRetVal = bGetAttribute("ANDROID_AUTO", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_ANDROIDAUTO] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     bRetVal = bGetAttribute("MIRRORLINK", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_MIRRORLINK] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     bRetVal = bGetAttribute("CARPLAY", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_DIPO] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     bRetVal = bGetAttribute("MYSPIN", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_MYSPIN] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     bRetVal = bGetAttribute("CARLIFE", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_CARLIFE] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;
     bRetVal = bGetAttribute("ONCAR", poNode, s32Value);
     SPI_NORMAL_ASSERT(false == bRetVal);
     m_mapenSelMode[e8DEV_TYPE_ONCAR] = ((s32Value >= 0) && (s32Value <= e16DEVICESEL_SEMI_AUTOMATIC))?
                               (tenDeviceSelectionMode) s32Value: e16DEVICESEL_AUTOMATIC;

   } //if ("DEVICE_SELECTION_MODE" == szNodeName)


   else if ("PREFERRED_TECHNOLOGY" == szNodeName)
   {
	   //Technology preference is read from the Datapool, if the Datapool value is unknow then it is read from the policy.xml
	   //Android Devices
	   if(e8DEV_TYPE_UNKNOWN == m_enAndroidDevicePreferencefromDatapool)
	   {
		   bRetVal = bGetAttribute("ANDROID_DEVICE", poNode, s32Value);
		   SPI_NORMAL_ASSERT(false == bRetVal);
		   m_enAndroidDevicePreference = ((s32Value >= 0) && (s32Value < e8DEV_TYPE_MAX)) ?
				   (tenDeviceCategory) s32Value: e8DEV_TYPE_ANDROIDAUTO;

	   }
	   else
	   {
		   bRetVal = true;
		   ETG_TRACE_USR4(("Android Devices bXmlReadNode return value: %d",ETG_ENUM(BOOL,bRetVal)));
		   m_enAndroidDevicePreference = m_enAndroidDevicePreferencefromDatapool;
	   }

	   //Apple Devices
	   if(e8DEV_TYPE_UNKNOWN == m_enAppleDevicePreferencefromDatapool)
	   {
		   bRetVal = bGetAttribute("APPLE_DEVICE", poNode, s32Value);
		   SPI_NORMAL_ASSERT(false == bRetVal);
		   m_enAppleDevicePreference = ((s32Value >= 0) && (s32Value < e8DEV_TYPE_MAX)) ?
				   (tenDeviceCategory) s32Value: e8DEV_TYPE_DIPO;
	   }
	   else
	   {
		   bRetVal = true;
		   ETG_TRACE_USR4(("Apple Devices bXmlReadNode return value: %d",ETG_ENUM(BOOL,bRetVal)));
		   m_enAppleDevicePreference = m_enAppleDevicePreferencefromDatapool;
	   }
   } //if ("PREFERRED_TECHNOLOGY" == szNodeName)

   else if ("CERTIFICATES" == szNodeName)
   {
      bRetVal = bGetAttribute("CERTIFICATE_TYPE", poNode, s32Value);
      SPI_NORMAL_ASSERT(false == bRetVal);
      m_enCertificateType = ((s32Value >= 0) && (s32Value<= e8_CERTIFICATETYPE_FFS)) ?
                                    (tenCertificateType) s32Value : e8_CERTIFICATETYPE_DEVELOPER;
   } //if ("MULTISESSION_SUPPORT" == szNodeName)

   else if ("PERSISTENT_STORAGE" == szNodeName)
   {
      bRetVal = bGetAttribute("STORAGE_PATH", poNode, m_szPerStoragePath);
      SPI_NORMAL_ASSERT(false == bRetVal);
   } //if ("PERSISTENT_STORAGE" == szNodeName)

   else if ("DEVICE_HISTORY_SIZE" == szNodeName)
   {
      bRetVal = bGetAttribute("MAXIMUM_SIZE", poNode, s32Value);
      SPI_NORMAL_ASSERT(false == bRetVal);
      m_u32DevHistorySize = (s32Value >= 0) ? s32Value : scou8DefaultHistorySize;
   } //if ("PERSISTENT_STORAGE" == szNodeName)
   else if("ENABLE_XML_VALIDATION" == szNodeName)
   {
      bRetVal = bGetAttribute("BOOL",poNode,m_bXMLValidationEnabled);
      SPI_NORMAL_ASSERT(false == bRetVal);
   }//else if("ENABLE_XML_VALIDATION" == szNodeName)
   else if ("MYSPIN_IAP_ATTRIBUTES" == szNodeName)
   {
      bRetVal = bGetAttribute("AUDIOSUPPORT", poNode,m_rMySPINAppInfo.rMySPINIapAppInfo.bAudioSupport);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("APPNAME", poNode,m_rMySPINAppInfo.rMySPINIapAppInfo.szAppName);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("BUNDLESEEDID", poNode,m_rMySPINAppInfo.rMySPINIapAppInfo.szBundleSeedId);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("DEVICE_ID", poNode,m_rMySPINAppInfo.rMySPINIapAppInfo.szDeviceId);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("PROTOCOL", poNode,m_rMySPINAppInfo.rMySPINIapAppInfo.szProtocol);
      SPI_NORMAL_ASSERT(false == bRetVal);

      t_Bool bIsAutoLaunch = false;
      bRetVal = bGetAttribute("AUTOLAUNCH", poNode,bIsAutoLaunch);
      SPI_NORMAL_ASSERT(false == bRetVal);
      (bIsAutoLaunch)? ( m_rMySPINAppInfo.rMySPINIapAppInfo.enIapLaunchType = e8LAUNCH_WITHOUT_USER_ALERT):
                       ( m_rMySPINAppInfo.rMySPINIapAppInfo.enIapLaunchType = e8LAUNCH_WITH_USER_ALERT );
   }
   else if ("MYSPIN_AOAP_ATTRIBUTES" == szNodeName)
   {
      bRetVal = bGetAttribute("AUDIOSUPPORT", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.bAudioSupport);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("DESCRIPTION", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szDescription);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("MODEL", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szVehicleModelName);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("URL", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szUrl);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("VERSION", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szVersion);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("SERIAL", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szSerial);
      SPI_NORMAL_ASSERT(false == bRetVal);
      bRetVal = bGetAttribute("VENDORNAME", poNode,m_rMySPINAppInfo.rMySPINAoapAppInfo.szVendorName);
      SPI_NORMAL_ASSERT(false == bRetVal);
   }
   else if("DEVICE_EXCLUSION_LIST" == szNodeName)
   {
      bRetVal = bGetAttribute("VALUE" , poNode ,m_szDeviceExclusionListPath);
      SPI_NORMAL_ASSERT(false == bRetVal);
   }
   else if("DEVICE_INCLUSION_LIST" == szNodeName)
   {
      bRetVal = bGetAttribute("VALUE" , poNode ,m_szDeviceInclusionListPath);
      SPI_NORMAL_ASSERT(false == bRetVal);
   }else if ("ENABLE_CARLIFE_IOS" == szNodeName )
   {
      bRetVal = bGetAttribute("BOOL" , poNode ,m_bIsCarlifeIOSEnabled);
      SPI_NORMAL_ASSERT(false == bRetVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vDisplayConnSettings
 ***************************************************************************/
t_Void spi_tclConnSettings::vDisplayConnSettings()
{
   ETG_TRACE_USR2(("/****************Connection Settings ******************/\n"));
   for (auto itmap = m_mapenSelMode.begin(); itmap != m_mapenSelMode.end(); itmap++)
   {
      ETG_TRACE_USR2((" Device Selection Mode for %d = %d", ETG_ENUM(DEVICE_CATEGORY, itmap->first),
               ETG_ENUM(SELECTION_MODE,itmap->second)));
   }
   ETG_TRACE_USR2(("Android Device Preference = %d ",ETG_ENUM(DEVICE_CATEGORY,m_enAndroidDevicePreference)));
   ETG_TRACE_USR2(("Apple Device Preference = %d ",ETG_ENUM(DEVICE_CATEGORY,m_enAppleDevicePreference)));
   ETG_TRACE_USR2((" Persistent Storage Path  = %s",m_szPerStoragePath.c_str()));
   ETG_TRACE_USR2((" Device History Size Limit= %d ",m_u32DevHistorySize));
   ETG_TRACE_USR2((" XML Validation Enabled= %d", ETG_ENUM(BOOL,m_bXMLValidationEnabled) ));
   ETG_TRACE_USR2((" Certificates type = %d \n", ETG_ENUM(CERTIFICATE_TYPE,m_enCertificateType) ));
   ETG_TRACE_USR2((" mySPIN iOS App AudioSupport = %d ", m_rMySPINAppInfo.rMySPINIapAppInfo.bAudioSupport));
   ETG_TRACE_USR2((" mySPIN iOS App BundelID = %s ", m_rMySPINAppInfo.rMySPINIapAppInfo.szBundleSeedId.c_str()));
   ETG_TRACE_USR2((" mySPIN iOS App DeviceID = %s ", m_rMySPINAppInfo.rMySPINIapAppInfo.szDeviceId.c_str()));
   ETG_TRACE_USR2((" mySPIN iOS App Protocol = %s ", m_rMySPINAppInfo.rMySPINIapAppInfo.szProtocol.c_str()));
   ETG_TRACE_USR2((" mySPIN iOS App Name = %s ", m_rMySPINAppInfo.rMySPINIapAppInfo.szAppName.c_str()  ));
   ETG_TRACE_USR2((" mySPIN iOS App Launch type  = %d ", m_rMySPINAppInfo.rMySPINIapAppInfo.enIapLaunchType ));
   ETG_TRACE_USR2((" Device Exclusion List Storage Path  = %s",m_szDeviceExclusionListPath.c_str()));
   ETG_TRACE_USR2((" Device Inclusion List Storage Path  = %s",m_szDeviceInclusionListPath.c_str()));
   ETG_TRACE_USR2((" Carlife IOS Feature support  = %d",ETG_ENUM(BOOL,m_bIsCarlifeIOSEnabled)));
}

/***************************************************************************
     

 ** FUNCTION:  t_Void spi_tclConnSettings::vGetMySPINIapAppInfo
 ***************************************************************************/
t_Void spi_tclConnSettings::vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo)
{
   rMySPINIapAppInfo = m_rMySPINAppInfo.rMySPINIapAppInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vGetMySPINAoapAppInfo
 ***************************************************************************/
t_Void spi_tclConnSettings::vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo)
{
   rfrMySPINAoapAppInfo = m_rMySPINAppInfo.rMySPINAoapAppInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vStoreSelectionModefromDatapool
 ***************************************************************************/
t_Void spi_tclConnSettings::vStoreSelectionModefromDatapool(tenDeviceSelectionMode enStoredSelectionMode)
{
   m_enStoredSelectionMode = enStoredSelectionMode;
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConnSettings::vStoreDevicePreferencefromDatapool
 ***************************************************************************/
t_Void spi_tclConnSettings::vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool, tenDeviceCategory enAppleDevicePreferencefromDatapool)
{
   m_enAndroidDevicePreferencefromDatapool = enAndroidDevicePreferencefromDatapool;
   m_enAppleDevicePreferencefromDatapool = enAppleDevicePreferencefromDatapool;
}
