/*!
 *******************************************************************************
 * \file             spi_tclAudioSettings.cpp
 * \brief            Settings class provides the interface to
 retrieve the project policy configurations
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Settings Audio class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                                  | Modifications
 28.10.2013 |  Hari Priya E R(RBEI/ECP2)               | Initial Version
 21.11.2013 |  Raghavendra S (RBEI/ECP2)               | Implementation of redefined
 Audio Settings Interface
 27/01/2014 | Vinoop (RBEI/ECP2)                       | Implemented parser for extracting
 Audio settings from xml file
 23.10.2014 | Hari Priya E R(RBEI/ECP2)                | Included policy value to 
 enable/disable audio blocking
 14.08.2015 | Shiva Kumar Gurija                       | Extensions for ML Dynamic Audio
 24.02.2016 | Ramya Murthy                             | Included default audio type for CarPlay
 24.04.2016 |  Chaitra Srinivasa                       | Changes to RTP In support from settings
 13.03.2018 | Ramya Murthy                             | Baidu Carlife sink record configuration
 
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "SPITypes.h"
#include "spi_tclAudioSettings.h"
#include "FileHandler.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclAudioSettings.cpp.trc.h"
#endif

using namespace std;
static const t_Char cosG3GXmlConfigFile[] = "/opt/bosch/spi/xml/policy.xml";

/***************************************************************************
 ** FUNCTION:  spi_tclAudioSettings::spi_tclAudioSettings();
 ***************************************************************************/
spi_tclAudioSettings::spi_tclAudioSettings() :
                  m_RTPPayload(0),
                  m_AudioIPL(0),
                  m_bAudBlocking_AppCat(false),
                  m_bAudBlocking_GlobalMute(false),
                  m_bEnableMLDynAudio(false),
                  m_bMLDynAudioInPolicy(false),
                  m_EnableMediaStreamRec(false),
                  m_EnableGuidanceStreamRec(false),
                  m_bRTPInSupport(false),
                  m_bMySPINAdvAudioEnabled(false),
                  m_enLastModeSupport(e8LASTMODE_DISABLED),
                  m_bMuteEnabled(false),
                  m_bVolumeLockeEnabled(false)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   for (t_U8 u8AudSources = 0; u8AudSources < e8AUD_INVALID; u8AudSources++)
   {
      m_au8Sources[u8AudSources] = 0;
   }
   // vReadAudioSettings();

} //spi_tclAudioSettings::spi_tclAudioSettings()

/***************************************************************************
 ** FUNCTION:  spi_tclVideoSettings::vReadAudioSettings()
 ***************************************************************************/
void spi_tclAudioSettings::vReadAudioSettings()
{
   t_Char* szConfigFilePath = NULL;
   szConfigFilePath = const_cast<t_Char*>(cosG3GXmlConfigFile);

   spi::io::FileHandler oPolicySettingsFile(szConfigFilePath, spi::io::SPI_EN_RDONLY);
   if (true == oPolicySettingsFile.bIsValid())
   {
      tclXmlDocument *poXmlDoc = new tclXmlDocument(szConfigFilePath);
      SPI_NORMAL_ASSERT(NULL == poXmlDoc);

      tclXmlReader* poXmlReader = new tclXmlReader(poXmlDoc, this);
      SPI_NORMAL_ASSERT(NULL == poXmlReader);
      if (NULL != poXmlReader)
      {
         poXmlReader->bRead("AUDIO");
      }
      RELEASE_MEM(poXmlReader);
      RELEASE_MEM(poXmlDoc);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioSettings::~spi_tclAudioSettings();
 ***************************************************************************/
spi_tclAudioSettings::~spi_tclAudioSettings()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
} //spi_tclAudioSettings::~spi_tclAudioSettings()

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclAudioSettings::u8GetSourceNumber(tenAudioDir enAudDir)const
 ***************************************************************************/
t_U8 spi_tclAudioSettings::u8GetSourceNumber(tenAudioDir enAudDir) const
{
   t_U8 u8SrcNum = 0;

   if (e8AUD_INVALID > enAudDir)
   {
      u8SrcNum = m_au8Sources[enAudDir];
   }

   return u8SrcNum;

} //t_U8 spi_tclAudioSettings::u8GetSourceNumber(tenAudioDir enAudDir)const

/***************************************************************************
 ** FUNCTION: t_Void vGetRTPPayloadType(tU32& rfu32RTPPayload)
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetRTPPayloadType(t_U32& rfu32RTPPayload) const
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::bGetRTPPayloadType %d \n", m_RTPPayload));
   rfu32RTPPayload = m_RTPPayload;
} //t_Bool vGetRTPPayloadType(tU32& rfu32RTPPayload)

/***************************************************************************
 ** FUNCTION: t_Void vGetRTPPayloadType(tU32& rfu32RTPPayload)
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetAudioIPL(t_U32& rfu32AudioIPL) const
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::bGetAudioIPL %d \n", m_AudioIPL));
   rfu32AudioIPL = m_AudioIPL;
} //t_Bool vGetAudioIPL(tU32& rfu32AudioIPL)

/***************************************************************************
 ** FUNCTION: t_Bool bGetAudioBlockingTrigger()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetAudBlockTriggerBasedOnCat() const
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::bGetAudBlockTriggerBasedOnCat: Enabled- %d ", ETG_ENUM(BOOL,
            m_bAudBlocking_AppCat)));

   return m_bAudBlocking_AppCat;
}

/***************************************************************************
 ** FUNCTION: t_Bool vSetAudBlockTriggerBasedOnCat()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::vSetAudBlockTriggerBasedOnCat: Enabled- %d ", ETG_ENUM(BOOL,
            bIsML11OrAboveDevice)));

   m_bAudBlocking_AppCat = bIsML11OrAboveDevice;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAudBlockTriggerBasedOnGlobaleMute()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetAudBlockTriggerBasedOnGlobaleMute() const
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::bGetAudBlockTriggerBasedOnGlobalMute: Enabled- %d ", ETG_ENUM(BOOL,
            m_bAudBlocking_GlobalMute)));

   return m_bAudBlocking_GlobalMute;
}

/***************************************************************************
 ** FUNCTION: t_Bool vGetAudioSettingsData()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vDisplayAudioSettings()
{
   for (t_U8 u8AudSources = 0; u8AudSources < e8AUD_INVALID; u8AudSources++)
   {
      ETG_TRACE_USR4(("Source Number  = %d ", m_au8Sources[u8AudSources]));
   }

   ETG_TRACE_USR4((" RTP Payload = %d", m_RTPPayload));
   ETG_TRACE_USR4((" Audio IPL = %d ", m_AudioIPL));
   ETG_TRACE_USR4((" Audio Blocking Based On AppCat Enabled = %d ", ETG_ENUM(BOOL, m_bAudBlocking_AppCat)));
   ETG_TRACE_USR4((" Audio Blocking Based On Global Mute Enabled = %d ", ETG_ENUM(BOOL, m_bAudBlocking_GlobalMute)));
   ETG_TRACE_USR4((" RTP In support = %d ", ETG_ENUM(BOOL, m_bRTPInSupport)));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vGetAudioPipeConfig()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig)
{
   ETG_TRACE_USR1((" spi_tclAudioSettings::vGetAudioPipeConfig  \n"));
   rfmapAudioPipeConfig = m_mapszAudioPipeConfig;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetMLDynAudioSupport()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetMLDynAudioSupport() const
{
   //Add code
   return m_bEnableMLDynAudio;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vSetMLDynAudioSupport()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR2(("spi_tclAudioSettings::vSetMLDynAudioSupport - Dynamic Audio support Enable = %d", ETG_ENUM(BOOL,
            bIsML11OrAboveDevice)));
   m_bEnableMLDynAudio = bIsML11OrAboveDevice;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAAPMediaStreamRecEnabled()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetAAPMediaStreamRecEnabled() const
{
   //Add code
   return m_EnableMediaStreamRec;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAAPGuidanceStreamRecEnabled()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetAAPGuidanceStreamRecEnabled() const
{
   //Add code
   return m_EnableGuidanceStreamRec;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclAudioSettings::szGetAudSinkMaxUnackedFrames()
 ***************************************************************************/
t_String spi_tclAudioSettings::szGetAudSinkMaxUnackedFrames() const
{
   //! Send MaxUnackedFrames data
   ETG_TRACE_USR4((" spi_tclAudioSettings::szGetAudSinkMaxUnackedFrames() "
            "Maximum unacked frames - %s", m_szAudSinkMaxUnackedFrames.c_str()));
   return m_szAudSinkMaxUnackedFrames;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetRTPInSupport()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetRTPInSupport() const
{
   //! Get whether RTPIn is supported or not
   return m_bRTPInSupport;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclAudioSettings::szGetAudSinkMaxUnackedFrames()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetMySIPNAdvAudioSupport() const
{

   return m_bMySPINAdvAudioEnabled;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclAudioSettings::szGetDisableRealTimePriorityAudio()
 ***************************************************************************/
t_String spi_tclAudioSettings::szGetDisableRealTimePriorityAudio() const
{
   //! Send MaxUnackedFrames data
   ETG_TRACE_USR4((" spi_tclAudioSettings::szGetDisableRealTimePriorityAudio() "
            "disable real time priority - %s", m_szDisableRealTimePriorityAud.c_str()));
   return m_szDisableRealTimePriorityAud;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclAudioSettings::szGetAudThreadsRealTimePriority()
 ***************************************************************************/
t_String spi_tclAudioSettings::szGetAudThreadsRealTimePriority() const
{
   //! Send MaxUnackedFrames data
   ETG_TRACE_USR4((" spi_tclAudioSettings::szGetAudThreadsRealTimePriority() "
            "thread real time priority  - %s", m_szAudThreadsRealTimePriority.c_str()));
   return m_szAudThreadsRealTimePriority;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclAudioSettings::szGetAudSinkThresholdMs()
 ***************************************************************************/
t_String spi_tclAudioSettings::szGetAudSinkThresholdMs() const
{
   //! Send MaxUnackedFrames data
   ETG_TRACE_USR4((" spi_tclAudioSettings::szGetAudSinkThresholdMs() "
            "Audio Sink Threshold value - %s", m_szAudSinkThresholdMs.c_str()));
   return m_szAudSinkThresholdMs;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclMediaSinkConfig()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetBdclMediaSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   mapAudioConfig = m_mapBdclMediaAudioSinkConfig;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclTTSSinkConfig()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetBdclTTSSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   mapAudioConfig = m_mapBdclTTSAudioSinkConfig;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclVRSinkConfig()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetBdclVRSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   mapAudioConfig = m_mapBdclVRAudioSinkConfig;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclVRSourceConfig()
 ***************************************************************************/
t_Void spi_tclAudioSettings::vGetBdclVRSourceConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   mapAudioConfig = m_mapBdclVRAudioSourceConfig;
}

/***************************************************************************
 ** FUNCTION: tenLastModeSupport spi_tclAudioSettings::bGetLastModeSupport()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetLastModeSupport(tenDeviceCategory enDevCat) const
{
   ETG_TRACE_USR2(("[PARAM]:spi_tclResourceMngrSettings::enGetLastModeSupport(): Last mode support value - %d", ETG_ENUM(LASTMODE_SUPPORT,
            m_enLastModeSupport)));

   t_Bool bIsLastModeSupported = false;
   switch (enDevCat)
   {
      case e8DEV_TYPE_DIPO:
         bIsLastModeSupported = (e8LASTMODE_DISABLED != m_enLastModeSupport);
         break;
      case e8DEV_TYPE_UNKNOWN:
      case e8DEV_TYPE_MAX:
      case e8DEV_TYPE_DEFAULT:
         bIsLastModeSupported = false;
         break;
      default:
         bIsLastModeSupported = (e8LASTMODE_ALLTECH == m_enLastModeSupport);
         break;

   }
   return bIsLastModeSupported;
}

/***************************************************************************
 ** FUNCTION: tenLastModeSupport spi_tclAudioSettings::enGetLastModeSupport()
 ***************************************************************************/
tenLastModeSupport spi_tclAudioSettings::enGetLastModeSupport() const
{
   ETG_TRACE_USR4(("spi_tclAudioSettings::enGetLastModeSupport(): Last mode support value - %d", ETG_ENUM(LASTMODE_SUPPORT,
            m_enLastModeSupport)));
   return m_enLastModeSupport;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAudioMuteEnabled()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetAudioMuteEnabled() const
{
   ETG_TRACE_USR4(("spi_tclAudioSettings::bGetAudioMuteEnabled(): Last mode support value - %d", ETG_ENUM(BOOL,
            m_bMuteEnabled)));
   return m_bMuteEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetVolumeLockEnabled()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetVolumeLockEnabled() const
{
   ETG_TRACE_USR4(("spi_tclAudioSettings::bGetVolumeLockEnabled(): Last mode support value - %d", ETG_ENUM(BOOL,
            m_bVolumeLockeEnabled)));
   return m_bVolumeLockeEnabled;
}
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclAudioSettings::bGetMLDynAudioApplicability()
 ***************************************************************************/
t_Bool spi_tclAudioSettings::bGetMLDynAudioApplicability() const
{
   ETG_TRACE_USR4(("m_bMLDynAudioInPolicy = %d", ETG_ENUM(BOOL, m_bMLDynAudioInPolicy)));
   return m_bMLDynAudioInPolicy;
}
/*************************************************************************
 ** FUNCTION:  virtual bXmlReadNode(xmlNode *poNode)
 *************************************************************************/
t_Bool spi_tclAudioSettings::bXmlReadNode(xmlNodePtr poNode)
{
   t_String szattribName;
   t_S32 u32iValue = 0;
   t_Bool bRetVal = false;
   t_String szNodeName;
   if (NULL != poNode)
   {
      szNodeName = (const char *) (poNode->name);
   } //if (NULL != poNode)

   if ("SOURCE_NUM" == szNodeName)
   {
      //get entertain source number
      szattribName = "SRC_ENT";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_ENT  = %d\n", u32iValue));
      m_au8Sources[e8AUD_MAIN_OUT] = static_cast<t_U8>(u32iValue);

      //get Phone source number
      szattribName = "SRC_PHONE";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_PHONE  = %d\n", u32iValue));
      m_au8Sources[e8AUD_PHONE_IN] = static_cast<t_U8>(u32iValue);

      //get voice rec source number
      szattribName = "SRC_VR";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_VR  = %d\n", u32iValue));
      m_au8Sources[e8AUD_VR_IN] = static_cast<t_U8>(u32iValue);

      //get MIX with out duck source number
      szattribName = "SRC_MIX";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_MIX  = %d\n", u32iValue));
      m_au8Sources[e8AUD_MIX_OUT] = static_cast<t_U8>(u32iValue);

      //get ALERT source number
      szattribName = "SRC_ALERT";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_ALERT  = %d\n", u32iValue));
      m_au8Sources[e8AUD_ALERT_OUT] = static_cast<t_U8>(u32iValue);

      //get  MIX with Duck number
      szattribName = "SRC_DUCK";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_DUCK  = %d\n", u32iValue));
      m_au8Sources[e8AUD_DUCK] = static_cast<t_U8>(u32iValue);

      //get  temporary number
      szattribName = "SRC_TRANSIENT";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_TRANSIENT  = %d ", u32iValue));
      m_au8Sources[e8AUD_TRANSIENT] = static_cast<t_U8>(u32iValue);

      //get  ML stereo source number
      szattribName = "SRC_STEREO_MIX";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_STEREO_MIX  = %d ", u32iValue));
      m_au8Sources[e8AUD_STEREO_MIX_OUT] = static_cast<t_U8>(u32iValue);

      //get default source number
      szattribName = "SRC_DEFAULT";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_DEFAULT  = %d\n", u32iValue));
      m_au8Sources[e8AUD_DEFAULT] = static_cast<t_U8>(u32iValue);

      //get default source number
      szattribName = "SRC_VR_STEREO";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4(("SRC_VR_STEREO  = %d\n", u32iValue));
      m_au8Sources[e8AUD_VR_STEREO] = static_cast<t_U8>(u32iValue);

   } //if ("SOURCE_NUM" == szNodeName)

   else if ("RTP_PAYLOAD" == szNodeName)
   {
      //get the RTP Payload types supported details
      szattribName = "PAYLOAD";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4((" RTP Payload = %d\n", u32iValue));
      m_RTPPayload = u32iValue;

   } //else if ("RTP_PAYLOAD" == szNodeName)
   else if ("AUDIO_IPL" == szNodeName)
   {
      //get the Initial playback latency 
      szattribName = "IPL";
      bRetVal = bGetAttribute(szattribName, poNode, u32iValue);
      ETG_TRACE_USR4((" Audio IPL = %d\n", u32iValue));
      m_AudioIPL = u32iValue;
   } //else if ("AUDIO_IPL" == szNodeName)
   else if ("AUDIO_BLOCKING_CAT" == szNodeName)
   {
      //get whether teh audio blocking needs to enabled, based on app category
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bAudBlocking_AppCat);
      ETG_TRACE_USR4((" Audio Blocking Based On App Category = %d\n", ETG_ENUM(BOOL, m_bAudBlocking_AppCat)));
   } //else if ("AUDIO_BLOCKING_CAT" == szNodeName)
   else if ("AUDIO_BLOCKING_GLOBAL_MUTE" == szNodeName)
   {
      //get whether teh audio blocking needs to enabled, based on global mute trigger
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bAudBlocking_GlobalMute);
      ETG_TRACE_USR4((" Audio Blocking Based On Global Mute Enabled = %d", ETG_ENUM(BOOL, m_bAudBlocking_GlobalMute)));
   } //else if("AUDIO_BLOCKING_GLOBAL_MUTE" == szNodeName)
   else if ("AUDIO_CONFIGURATION" == szNodeName)
   {
      //get the audio configurations
      szattribName = "MAIN_AUDIO_24kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_24kHz]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_24kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_24kHz].c_str()));
      szattribName = "MAIN_AUDIO_16kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_16kHz]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_16kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_16kHz].c_str()));
      szattribName = "MAIN_AUDIO_8kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_8kHz]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_8kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_8kHz].c_str()));
      szattribName = "MAIN_AUDIO_ALERT";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_ALERT]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_ALERT = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_ALERT].c_str()));
      szattribName = "MAIN_AUDIO_MEDIA_BROWSING";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_BROWSING]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_MEDIA_BROWSING = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_BROWSING].c_str()));
      szattribName = "MAIN_AUDIO_MEDIA_STANDALONE";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_MEDIA_STANDALONE = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE].c_str()));
      szattribName = "MAIN_AUDIO_MEDIA_STANDALONE_BDCL";
      bRetVal = bGetAttribute(szattribName,
               poNode,
               m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE_BDCL]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_MEDIA_STANDALONE_BDCL = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE_BDCL].c_str()));
      szattribName = "MAIN_AUDIO_SPEECH";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_SPEECH]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_SPEECH = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_SPEECH].c_str()));
      szattribName = "MAIN_AUDIO_DUMMY";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_DUMMY]);
      ETG_TRACE_USR4((" Audio configuration  MAIN_AUDIO_DUMMY = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_DUMMY].c_str()));
      szattribName = "ALTERNATE_AUDIO";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_ALTERNATEAUDIO]);
      ETG_TRACE_USR4((" Audio configuration  ALTERNATE_AUDIO = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_ALTERNATEAUDIO].c_str()));
      szattribName = "AUDIO_IN_24kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_24kHz]);
      ETG_TRACE_USR4((" Audio configuration  AUDIO_IN_24kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_24kHz].c_str()));
      szattribName = "AUDIO_IN_16kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_16kHz]);
      ETG_TRACE_USR4((" Audio configuration  AUDIO_IN_16kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_16kHz].c_str()));
      szattribName = "AUDIO_IN_8kHz";
      bRetVal = bGetAttribute(szattribName, poNode, m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_8kHz]);
      ETG_TRACE_USR4((" Audio configuration  AUDIO_IN_8kHz = %s\n", m_mapszAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_8kHz].c_str()));
   } //else if ("AUDIO_CONFIGURATION" == szNodeName)
   else if ("ML_DYN_AUDIO" == szNodeName)
   {
      //get whether the dynamic audio source swicth for ML is enabled
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bMLDynAudioInPolicy);
      ETG_TRACE_USR4((" ML Dynamic Audio Enabled in policy- %d", ETG_ENUM(BOOL, m_bMLDynAudioInPolicy)));
   } //else if("ML_DYN_AUDIO" == szNodeName)
   else if ("AAP_AUD_ENDPOINT_REC" == szNodeName)
   {
      szattribName = "MEDIA_STREAM";
      bRetVal = bGetAttribute(szattribName, poNode, m_EnableMediaStreamRec);
      ETG_TRACE_USR4((" AAP Media Stream Recording Enabled - %d", ETG_ENUM(BOOL, m_EnableMediaStreamRec)));

      szattribName = "GUIDANCE_STREAM";
      bRetVal = bGetAttribute(szattribName, poNode, m_EnableGuidanceStreamRec);
      ETG_TRACE_USR4((" AAP Guidance Stream Recording Enabled - %d", ETG_ENUM(BOOL, m_EnableGuidanceStreamRec)));
   } //else if("AAP_AUD_ENDPOINT_REC" == szNodeName)
   else if ("RTPIN_SUPPORT" == szNodeName)
   {
      //get whether the RTP_IN support is enabled
      szattribName = "VALUE";
      bRetVal = bGetAttribute(szattribName, poNode, m_bRTPInSupport);
      ETG_TRACE_USR4((" RTPIN support - %d", ETG_ENUM(BOOL, m_bRTPInSupport)));
   } //else if("RTP_IN" == szNodeName)
   else if ("MSPIN_ADVANCED_AUDIO_SUPPORT" == szNodeName)
   {
      //Check if mySPIN Advanced Audio is supported or not
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bMySPINAdvAudioEnabled);
   }
   else if ("AAP_AUD_SINK_CONFIG" == szNodeName)
   {
      //get the max-unacked-frames value for Audio Sink Endpoint configuration
      szattribName = "MAX_UNACKED_FRAMES";
      bRetVal = bGetAttribute(szattribName, poNode, m_szAudSinkMaxUnackedFrames);
      ETG_TRACE_USR4((" Audio Sink Max Unacked Frames - %s", m_szAudSinkMaxUnackedFrames.c_str()));

      szattribName = "DISABLE_REAL_TIME_PRIORITY_AUDIO";
      bRetVal = bGetAttribute(szattribName, poNode, m_szDisableRealTimePriorityAud);
      ETG_TRACE_USR4((" Disable the real time Priority audio - %s", m_szDisableRealTimePriorityAud.c_str()));

      szattribName = "AUDIO_THREADS_REAL_TIME_PRIORITY";
      bRetVal = bGetAttribute(szattribName, poNode, m_szAudThreadsRealTimePriority);
      ETG_TRACE_USR4((" Audio Threads Real time priority - %s", m_szAudThreadsRealTimePriority.c_str()));

      szattribName = "AUDIO_SINK_THRESHOLD_MS";
      bRetVal = bGetAttribute(szattribName, poNode, m_szAudSinkThresholdMs);
      ETG_TRACE_USR4((" Audio Sink Threshold value - %s", m_szAudSinkThresholdMs.c_str()));
   }
   else if (("BDCL_MEDIA_SINK_CONFIGURATION" == szNodeName) || ("BDCL_TTS_SINK_CONFIGURATION" == szNodeName)
            || ("BDCL_VR_SINK_CONFIGURATION" == szNodeName))
   {
      t_String szKey, szValue;
      std::map<t_String, t_String> mapAudSinkConfig;

      szattribName = "STREAM_TYPE_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: STREAM_TYPE_KEY = %s", szKey.c_str()));
      szattribName = "STREAM_TYPE_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: STREAM_TYPE_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "CHANNELS_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: CHANNELS_KEY = %s", szKey.c_str()));
      szattribName = "CHANNELS_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: CHANNELS_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "SAMPLE_RATE_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: SAMPLE_RATE_KEY = %s", szKey.c_str()));
      szattribName = "SAMPLE_RATE_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: SAMPLE_RATE_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "BITS_PER_SAMPLE_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BITS_PER_SAMPLE_KEY = %s", szKey.c_str()));
      szattribName = "BITS_PER_SAMPLE_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BITS_PER_SAMPLE_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "DISABLE_RT_PRIO_AUD_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: DISABLE_RT_PRIO_AUD_KEY = %s", szKey.c_str()));
      szattribName = "DISABLE_RT_PRIO_AUD_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: DISABLE_RT_PRIO_AUD_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "THREADS_RT_PRIO_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: THREADS_RT_PRIO_KEY = %s", szKey.c_str()));
      szattribName = "THREADS_RT_PRIO_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: THREADS_RT_PRIO_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "BUFFER_MIN_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BUFFER_MIN_KEY = %s", szKey.c_str()));
      szattribName = "BUFFER_MIN_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BUFFER_MIN_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "BUFFER_MAX_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BUFFER_MAX_KEY = %s", szKey.c_str()));
      szattribName = "BUFFER_MAX_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: BUFFER_MAX_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "RECORD_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: RECORD_KEY = %s", szKey.c_str()));
      szattribName = "RECORD_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Sink configuration: RECORD_VAL = %s", szValue.c_str()));
      mapAudSinkConfig[szKey] = szValue;

      if ("BDCL_MEDIA_SINK_CONFIGURATION" == szNodeName)
      {
         m_mapBdclMediaAudioSinkConfig = mapAudSinkConfig;
      }
      else if ("BDCL_TTS_SINK_CONFIGURATION" == szNodeName)
      {
         m_mapBdclTTSAudioSinkConfig = mapAudSinkConfig;
      }
      else if ("BDCL_VR_SINK_CONFIGURATION" == szNodeName)
      {
         m_mapBdclVRAudioSinkConfig = mapAudSinkConfig;
      }
   }
   else if ("BDCL_VR_SOURCE_CONFIGURATION" == szNodeName)
   {
      t_String szKey, szValue;

      szattribName = "CHANNELS_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: CHANNELS_KEY = %s", szKey.c_str()));
      szattribName = "CHANNELS_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: CHANNELS_VAL = %s", szValue.c_str()));
      m_mapBdclVRAudioSourceConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "SAMPLE_RATE_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: SAMPLE_RATE_KEY = %s", szKey.c_str()));
      szattribName = "SAMPLE_RATE_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: SAMPLE_RATE_VAL = %s", szValue.c_str()));
      m_mapBdclVRAudioSourceConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "BITS_PER_SAMPLE_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: BITS_PER_SAMPLE_KEY = %s", szKey.c_str()));
      szattribName = "BITS_PER_SAMPLE_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: BITS_PER_SAMPLE_VAL = %s", szValue.c_str()));
      m_mapBdclVRAudioSourceConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "DISABLE_RT_PRIO_AUD_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: DISABLE_RT_PRIO_AUD_KEY = %s", szKey.c_str()));
      szattribName = "DISABLE_RT_PRIO_AUD_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: DISABLE_RT_PRIO_AUD_VAL = %s", szValue.c_str()));
      m_mapBdclVRAudioSourceConfig[szKey] = szValue;
      szKey = "";
      szValue = "";

      szattribName = "THREADS_RT_PRIO_KEY";
      bRetVal = bGetAttribute(szattribName, poNode, szKey);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: THREADS_RT_PRIO_KEY = %s", szKey.c_str()));
      szattribName = "THREADS_RT_PRIO_VAL";
      bRetVal = bGetAttribute(szattribName, poNode, szValue);
      ETG_TRACE_USR4((" BDCL Audio Source configuration: THREADS_RT_PRIO_VAL = %s", szValue.c_str()));
      m_mapBdclVRAudioSourceConfig[szKey] = szValue;

   }

   else if ("LASTMODE_SUPPORT" == szNodeName)
   {
      szattribName = "VALUE";
      t_S32 s32LastModeSupport = 0;
      bRetVal = bGetAttribute(szattribName, poNode, s32LastModeSupport);
      m_enLastModeSupport = static_cast<tenLastModeSupport>(s32LastModeSupport);
      ETG_TRACE_USR4(("Last mode support value = %d", ETG_ENUM(LASTMODE_SUPPORT, m_enLastModeSupport)));
   }

   else if ("AUDIO_MUTE" == szNodeName)
   {
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bMuteEnabled);
      ETG_TRACE_USR4(("Audio mute enabled value = %d", ETG_ENUM(BOOL, m_bMuteEnabled)));
   }

   else if ("AUDIO_VOL_LOCK" == szNodeName)
   {
      szattribName = "BOOL";
      bRetVal = bGetAttribute(szattribName, poNode, m_bVolumeLockeEnabled);
      ETG_TRACE_USR4(("Audio volume lock enabled value = %d", ETG_ENUM(BOOL, m_bVolumeLockeEnabled)));
   }
   return bRetVal;
}

