/*!
 *******************************************************************************
 * \file             spi_tclAudioSettings.h
 * \brief            Settings class provides the interface to 
                     retrieve the project policy configurations
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   SmartPhone Integration
 DESCRIPTION:    Settings Audio class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                                  | Modifications
 28.10.2013 |  Hari Priya E R(RBEI/ECP2)               | Initial Version
 18.11.2013 |  Raghavendra S (RBEI/ECP2)               | Redefinition of Interface
 27/01/2014 |  Vinoop(RBEI/ECP2)                       |Implemented parser for extracting 
                                                          Audio settings from xml file
14.08.2015  | Shiva Kumar Gurija                       | Extensions for ML Dynamic Audio

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef SPI_TCLAUDIOSETTINGS_H
#define SPI_TCLAUDIOSETTINGS_H

#include "SPITypes.h"
#include "XmlDocument.h"
#include "Xmlable.h"
#include "XmlReader.h"

using namespace shl::xml;

/**
 *  class definitions.
 */

/**
 * This is the Settings class provides the interface to 
   retrieve the project policy configurations
 */

class spi_tclAudioSettings:public tclXmlReadable
{
   public:

      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclAudioSettings::spi_tclAudioSettings()
       ***************************************************************************/
      /*!
       * \fn      spi_tclAudioSettings()
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclAudioSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclAudioSettings::~spi_tclAudioSettings()
       ***************************************************************************/
      /*!
       * \fn      virtual ~spi_tclAudioSettings()
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclAudioSettings();

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclAudioSettings::u8GetSourceNumber(tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      u8GetSourceNumber( tenAudioDir enAudDir)
       * \brief   Function to get the Source Number for Audio Link & selected device
       *          category.
       * \param   [enAudDir]: Audio Link Direction
       * \retval  t_U8 value: Source Number
       **************************************************************************/
      t_U8 u8GetSourceNumber(tenAudioDir enAudDir) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclAudioSettings::vGetRTPPayloadType
       (t_U32& rfu32RTPPayload)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetRTPPayloadType(t_U32& u32RTPPayload)
       * \brief   Method to get the RTPPayload type
       * \param   [rfu32RTPPayload]: Reference to the RTP Payload type
       * \retval  NONE
       **************************************************************************/
      t_Void vGetRTPPayloadType(t_U32& rfu32RTPPayload) const;

      //TO BE CHECKED

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclAudioSettings::vGetAudioIPL
       (tU32& rfu32AudioIPL)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAudioIPL(tU32& rfu32AudioIPL)
       * \brief   Method to get the Audio IPL
       * \param   [rfu32AudioIPL]: Reference to the Audio IPL
       * \retval  t_Bool value: TRUE if Setting is Valid, FALSE Otherwise
       **************************************************************************/
      t_Void vGetAudioIPL(t_U32& rfu32AudioIPL) const;

       /***************************************************************************
       ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAudioBlockingTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnCat()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on 
       *          Application category or not. This feature works only in conjunction with the 
       *          Dynamic Audio feature
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnCat() const;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclAudioSettings::vSetAudBlockTriggerBasedOnCat()
      ***************************************************************************/
     /*!
      * \fn      t_Void vSetAudBlockTriggerBasedOnCat()
      * \brief   Method to Set the Audio Blocking enable/disable trigger based on
      *          Application category or not. This feature works only in conjunction with the
      *          Dynamic Audio feature
      * \retval  t_Void
      **************************************************************************/
      t_Void vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice);

       /***************************************************************************
       ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAudBlockTriggerBasedOnGlobaleMute()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on 
       *          Global mute is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclAudioSettings::vGetAudioSettingsData()
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetAudioSettingsData()
       * \brief  To print the audio settings data
       * \retval  t_Void
       **************************************************************************/
      t_Void vDisplayAudioSettings();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclAudioSettings::vGetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetAudioPipeConfig()
       * \brief  provides audio configuration for the specified audio steam
       * \param  [OUT]rfrmapAudioPipeConfig : returns audio configuration as string
       **************************************************************************/
      t_Void vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig);
	  
      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclAudioSettings::bGetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioSupport() const
       * \brief  To Get whether ML Audio is supported or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMLDynAudioSupport() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclAudioSettings::vSetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     vSetMLDynAudioSupport() const
       * \brief  To Set whether ML Audio is supported or not.
       *         (For ML 1.0 = Not Supported, For Ml 1.1 onwards = supported)
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice); //bSetMLDynAudioSupport

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAAPMediaStreamRecEnabled()
      ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPMediaStreamRecEnabled() const
       * \brief  To Get whether AAP Media audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPMediaStreamRecEnabled() const;

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAAPGuidanceStreamRecEnabled()
      ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPGuidanceStreamRecEnabled() const
       * \brief  To Get whether AAP Guidance audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPGuidanceStreamRecEnabled() const;

      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::szGetAudSinkMaxUnackedFrames()
      ***************************************************************************/
      /*!
       * \fn     szGetAudSinkMaxUnackedFrames() const
       * \brief  To Get "max-unacked-frames" value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudSinkMaxUnackedFrames() const;
	  
      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclAudioSettings::bGetRTPInSupport()
      ***************************************************************************/
      /*!
       * \fn     t_Bool bGetRTPInSupport() const
       * \brief  To Get whether RTPIn support is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetRTPInSupport() const;	  

      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::bGetMySIPNAdvAudioSupport()
      ***************************************************************************/
      /*!
       * \fn     bGetMySIPNAdvAudioSupport() const
       * \brief  To Get support for mySPIN Advanced Audio
       * \retval  t_String
       **************************************************************************/
      t_Bool bGetMySIPNAdvAudioSupport() const;
      
      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::szGetDisableRealTimePriorityAudio()
      ***************************************************************************/
      /*!
      * \fn     szGetDisableRealTimePriorityAudio() const
      * \brief  To Get the value of disabling real time priority audio for Audio Sink configuration
      * \retval  t_String
      **************************************************************************/
      t_String szGetDisableRealTimePriorityAudio() const;

      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::szGetAudThreadsRealTimePriority()
      ***************************************************************************/
      /*!
      * \fn     szGetAudThreadsRealTimePriority() const
      * \brief  To Get Audio threads Real time priority value for Audio Sink configuration
      * \retval  t_String
      **************************************************************************/
      t_String szGetAudThreadsRealTimePriority() const;

      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::szGetAudSinkThresholdMs()
      ***************************************************************************/
      /*!
      * \fn     szGetAudSinkThresholdMs() const
      * \brief  To Get Audio Sink Threshold configuration
      * \retval  t_String
      **************************************************************************/
      t_String szGetAudSinkThresholdMs() const;

      /***************************************************************************
      ** FUNCTION: t_String spi_tclAudioSettings::vGetBdclMediaSinkConfig()
      ***************************************************************************/
      /*!
      * \fn     vGetBdclMediaSinkConfig() const
      * \brief  To Get BDCL Media Audio Sink configuration
      * \retval  t_String
      **************************************************************************/
      t_Void vGetBdclMediaSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclTTSSinkConfig()
      ***************************************************************************/
      /*!
      * \fn     vGetBdclTTSSinkConfig() const
      * \brief  To Get BDCL TTS Audio Sink configuration
      * \retval  t_Void
      **************************************************************************/
      t_Void vGetBdclTTSSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclVRSinkConfig()
      ***************************************************************************/
      /*!
      * \fn     vGetBdclVRSinkConfig() const
      * \brief  To Get BDCL VR Audio Sink configuration
      * \retval  t_Void
      **************************************************************************/
      t_Void vGetBdclVRSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclAudioSettings::vGetBdclVRSourceConfig()
      ***************************************************************************/
      /*!
      * \fn     vGetBdclVRSourceConfig() const
      * \brief  To Get BDCL VR Audio Source configuration
      * \retval  t_Void
      **************************************************************************/
      t_Void vGetBdclVRSourceConfig(std::map< t_String, t_String >& mapAudioConfig) const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclAudioSettings::bGetLastModeSupport()
       ***************************************************************************/
       /*!
       * \fn     bGetLastModeSupport() const
       * \brief  To Get last mode support configuration for a device category
       * \param  enDevCat [IN]: Device category
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetLastModeSupport(tenDeviceCategory enDevCat) const;
  
      /***************************************************************************
      ** FUNCTION: tenLastModeSupport spi_tclAudioSettings::enGetLastModeSupport()
      ***************************************************************************/
      /*!
      * \fn     enGetLastModeSupport() const
      * \brief  To Get last mode support configuration
      * \retval  tenLastModeSupport

      **************************************************************************/
      tenLastModeSupport enGetLastModeSupport() const;

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclAudioSettings::bGetAudioMuteEnabled()
      ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAudioMuteEnabled() const
       * \brief  To Get whether audio mute is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudioMuteEnabled() const;

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclAudioSettings::bGetVolumeLockEnabled()
      ***************************************************************************/
      /*!
       * \fn     t_Bool bGetVolumeLockEnabled() const
       * \brief  To Get whether Volume lock is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetVolumeLockEnabled() const;

	  /***************************************************************************
	   ** FUNCTION: t_Bool spi_tclAudioSettings::bGetMLDynAudioApplicability()
	   ***************************************************************************/
       /*!
       * \fn     t_Bool bGetMLDynAudioApplicability() const
	   * \brief  To Get whether ML Dnamic Audio is applicable or not from policy.
	   * \retval  t_Bool
		 **************************************************************************/
       t_Bool bGetMLDynAudioApplicability() const;

      /***************************************************************************
       ** FUNCTION:  spi_tclAudioSettings::vReadAudioSettings()
       ***************************************************************************/
      /*!
       * \retval  t_Void
       **************************************************************************/
      t_Void vReadAudioSettings();

   private:
      /***************************************************************************
      *********************************PRIVATE************************************
      ***************************************************************************/

      /*************************************************************************
       ** FUNCTION:  virtual bXmlReadNode(xmlNode *poNode)
       *************************************************************************/
      /*!
       * \fn     virtual bool bXmlReadNode(xmlNode *poNode)
       * \brief  virtual function to read data from a xml node
       * \param  poNode : [IN] pointer to xml node
       * \retval bool : true if success, false otherwise.
       *************************************************************************/
      virtual bool bXmlReadNode(xmlNodePtr poNode);

      t_U8 m_au8Sources[e8AUD_INVALID];
      t_U32 m_RTPPayload;
      t_U32 m_AudioIPL;

      //! Member variable to store whether the Audio Blocking based on Application Category feature is enabled or not
      t_Bool m_bAudBlocking_AppCat;
      //! Member variable to store whether the Audio Blocking based on Global Mute feature is enabled or not
      t_Bool m_bAudBlocking_GlobalMute;

      tmapAudioPipeConfig  m_mapszAudioPipeConfig;
      t_Bool m_bEnableMLDynAudio;
	  //Only to read Dynamic audio supported value from policy
      t_Bool m_bMLDynAudioInPolicy; 
      //! Indicates whether Media stream audio recording is enabled.
      t_Bool m_EnableMediaStreamRec;
      //! Indicates whether Guidance stream audio recording is enabled.
      t_Bool m_EnableGuidanceStreamRec;

      //! Indicates Max-unacked-Frames count for Audio Sink Endpoint
      t_String m_szAudSinkMaxUnackedFrames;

      //! Indicates whether RTP_IN support enabled or not
      t_Bool m_bRTPInSupport;
	  
	  //! Indicates if mySPIN Advanced Audio is enabled
      t_Bool m_bMySPINAdvAudioEnabled;

      t_String m_szDisableRealTimePriorityAud;
      
      t_String m_szAudThreadsRealTimePriority;
      
      t_String m_szAudSinkThresholdMs;

      std::map< t_String, t_String > m_mapBdclMediaAudioSinkConfig;

      std::map< t_String, t_String > m_mapBdclTTSAudioSinkConfig;

      std::map< t_String, t_String > m_mapBdclVRAudioSinkConfig;

      std::map< t_String, t_String > m_mapBdclVRAudioSourceConfig;
	  
      //! Indicates the last mode support that is enabled for a given project based on policy settings
      tenLastModeSupport m_enLastModeSupport;

      t_Bool m_bMuteEnabled;

      t_Bool m_bVolumeLockeEnabled;

      /***************************************************************************
       ****************************END OF PRIVATE *********************************
       ***************************************************************************/

}; //spi_tclAudioSettings

#endif //_SPI_TCLAUDIOSETTINGS_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>

