/***********************************************************************/
/*!
* \file  spi_tclDynamicConfigurationBase.h
* \brief  Implementation of the Class spi_tclDynamicConfigurationBase
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Common  Base Class  for SPI dynamic configuration 
AUTHOR:         kmo6kor
COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
07.06.2018  | kmo6kor               | Initial Version
18.07.2017  | Noopur R K            | Added u16GetGeneralRestrictions method
\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
#include "spi_tclDynamicConfigurationBase.h"
#include "RandomIdGenerator.h"
#include "DirHandler.h"
//! Includes for Trace files
#include "Trace.h"
#include "FileHandler.h"
#include "StringHandler.h"
#include <sstream>
#include <fstream>

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclDynamicConfigurationBase.cpp.trc.h"
#endif
#endif

#define CC_LABEL_FILE_PATH "/opt/bosch/spi/cc-label.txt"
#define DEFAULT_SPI_VERSION "RBEI_MAIN_G3G_SPI_XXXX"

//General Restriction info for AAP and CP
// 0 for Android auto denotes that clock,phone signal, battery level will be displayed on the HU.
static const t_U16 scou16AAPGeneralRestrictionInfo = 0x00;
static const t_U16 scou16DipoGeneralRestrictionInfo = 0x00;

#define DPI_FOUR_EIGHTY 160
#define DPI_SEVEN_TWENTY 240
#define DPI_TEN_EIGHTY 360

/***************************************************************************
** FUNCTION: spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase()
***************************************************************************/
spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase(spi_tclDynamicConfigurationReaderBase* poDynamicConfigurationReader):
m_poDynamicConfigurationReader(poDynamicConfigurationReader),
m_u8DipoParkModeRestrictionInfo(0)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase Entered"));
}

/***************************************************************************
** FUNCTION: spi_tclDynamicConfigurationBase::~spi_tclDynamicConfigurationBase()
***************************************************************************/
spi_tclDynamicConfigurationBase::~spi_tclDynamicConfigurationBase()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::~spi_tclDynamicConfigurationBase Entered"));
   RELEASE_MEM(m_poDynamicConfigurationReader);
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vSetMLNotificationOnOff()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetMLNotificationOnOff:bSetNotificationsOn-%d",
      ETG_ENUM(BOOL,bSetNotificationsOn)));
   if(NULL != m_poDynamicConfigurationReader)
   {
   //Store setting in datapool.
   m_poDynamicConfigurationReader->bWriteMLNotificationSetting(bSetNotificationsOn);
   }
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bGetMLNotificationEnabledInfo()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bGetMLNotificationEnabledInfo() const
{
   t_Bool bNotificationsOn = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
      bNotificationsOn = m_poDynamicConfigurationReader->bReadMLNotificationSetting();
      ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bGetMLNotificationEnabledInfo: bNotificationsOn-%d",
      ETG_ENUM(BOOL,bNotificationsOn)));
   }
   return bNotificationsOn;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::vSetDeviceSelectionMode()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType)
{
   SPI_INTENTIONALLY_UNUSED(enTechnologyType);
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetDeviceSelectionMode"));
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bWriteSelectionMode(enSelectionMode);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetMLNotiSettingVal()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetMLNotiSettingVal() entered "));
   // By default set to false
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bWriteMLNotificationSetting(bMLNotiSettingVal);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetMLLinkEnableSetVal()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetMLLinkEnableSetVal() entered "));
   t_Bool bRetVal = false;
   // By default ML usage is enabled
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteMLEnableSetting(enMLLinkEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetDipoEnableSetVal()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetDipoEnableSetVal() entered "));
   t_Bool bRetVal = false;
   // By default Dipo usage is enabled
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteDipoEnableSetting(enDipoEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetAAPEnableSetVal()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetAAPEnableSetVal() entered "));
   t_Bool bRetVal = false;
   // By default the AAP usage is enabled
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteAAPEnableSetting(enAAPEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetCarlifeEnableSetVal()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetCarlifeEnableSetVal() entered "));
   t_Bool bRetVal = false;
   // By default the Carlife usage is enabled
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteCarlifeEnableSetting(enCarlifeEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetSteeringWheelPos()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetSteeringWheelPos() entered "));
   // Default steering wheel position is set as unknown drive side
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bWriteDriveSideInfo(enSteeringWheelPos);
   }
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetSelectMode()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetSelectMode(tenDeviceSelectionMode enSelectMode)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetSelectMode() entered "));
   // Default select mode is set to unknown device
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bWriteSelectionMode(enSelectMode);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetAndroidDeviceTechnologyPreference()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetAndroidDeviceTechnologyPreference() entered "));
   t_Bool bRetVal = false;
   // Default technology preference for Android Devices is set as Android Auto
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteAndroidDeviceTechnologyPreference(enAndroidDeviceTechPrefVal);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bSetAppleDeviceTechnologyPreference()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bSetAppleDeviceTechnologyPreference() entered "));
   t_Bool bRetVal = false;
   // Default technology preference Apple Devices is set as DIPO
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWriteAppleDeviceTechnologyPreference(enAppleDeviceTechPrefVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::u8ReadVirginStartSetting()
***************************************************************************/
t_U8 spi_tclDynamicConfigurationBase::u8ReadVirginStartSetting()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::u8ReadVirginStartSetting entered"));
   t_U8 u8VirginStart = 0;
   if(NULL != m_poDynamicConfigurationReader)
   {
      u8VirginStart = m_poDynamicConfigurationReader->u8ReadVirginStartSetting();
      ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vReadVirginStartSetting: u8VirginStart-%d",
      ETG_ENUM(U8,u8VirginStart)));
   }
   return u8VirginStart;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vWriteVirginStartSetting()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vWriteVirginStartSetting(t_U8 u8DefaultVgStrt)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vWriteVirginStartSetting entered"));
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->u8WriteVirginStartSetting(u8DefaultVgStrt);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bWriteAudioLastMode()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bWriteAudioLastMode(t_Bool bAudioLastMode)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bWriteAudioLastMode() entered - Audio last mode: %d", ETG_ENUM(BOOL,bAudioLastMode)));
   t_Bool bRetVal = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
      bRetVal = m_poDynamicConfigurationReader->bWriteAudioLastMode(bAudioLastMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclDynamicConfigurationBase::bWriteAudioLastMode - Value is written successfully = %d", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bWriteDisplayLastMode()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bWriteDisplayLastMode(t_Bool bDisplayLastMode)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bWriteDisplayLastMode() entered - Display last mode: %d", ETG_ENUM(BOOL,bDisplayLastMode)));
   t_Bool bRetVal = false;
   if( NULL != m_poDynamicConfigurationReader )
   {
      bRetVal = m_poDynamicConfigurationReader->bWriteDisplayLastMode(bDisplayLastMode);
   }
    ETG_TRACE_USR2(("[DESC]: spi_tclDynamicConfigurationBase::bWriteDisplayLastMode - Value is written successfully = %d", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}


/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bIsDisplayLastModeProjection()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bIsDisplayLastModeProjection()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bIsDisplayLastModeProjection() entered "));
   t_Bool bDisplayLastMode = false;
   if( NULL != m_poDynamicConfigurationReader )
   {
     m_poDynamicConfigurationReader->vReadDisplayLastMode(bDisplayLastMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclDynamicConfigurationBase::bIsDisplayLastModeProjection - Display last mode is projection = %d", ETG_ENUM(BOOL, bDisplayLastMode)));
   return bDisplayLastMode;
}

/***************************************************************************
 ** FUNCTION:  t_U8 spi_tclDynamicConfigurationBase::u8GetDipoRestrictionInfo()
 ***************************************************************************/
t_U8 spi_tclDynamicConfigurationBase::u8GetDipoRestrictionInfo(t_Bool bDriveRestrictions)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::u8GetDipoRestrictionInfo() entered "));
   t_U8 u8RestrictionsInfo = m_u8DipoParkModeRestrictionInfo;
   if ((true == bDriveRestrictions) && (NULL != m_poDynamicConfigurationReader))
   {
      m_poDynamicConfigurationReader->bReadDiPODriveRestrictionInfo(u8RestrictionsInfo);
      ETG_TRACE_USR2(("spi_tclDynamicConfigurationBase::u8GetDipoRestrictionInfo : DiPO Restrictions:%d ", u8RestrictionsInfo));
   }
   return u8RestrictionsInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDipoFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetDipoFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo, const t_U8 cou8DriveModeRestrictionInfo)
{
   //! Not required to keep this information since CarPlay cannot support
   // two different values for feature lockout.
   m_u8DipoParkModeRestrictionInfo = cou8ParkModeRestrictionInfo;
   if( NULL != m_poDynamicConfigurationReader)
   {
      t_Bool bStatus = m_poDynamicConfigurationReader->bWriteDiPODriveRestrictionInfo(cou8DriveModeRestrictionInfo);
      tenResponseCode enResponseCode = (false == bStatus) ? e8FAILURE:e8SUCCESS;
      ETG_TRACE_USR2(("spi_tclDynamicConfigurationBase :: vSetDipoFeatureRestrictions Set Drive restriction is completed with status : %d ", enResponseCode));
   }
}

/***************************************************************************
 ** FUNCTION:  t_String spi_tclDynamicConfigurationBase::szGetVehicleId()
 ** Function returns Vehicle Id after reading it from DataPool.
 ***************************************************************************/
t_String spi_tclDynamicConfigurationBase::szGetVehicleId()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::szGetVehicleId() entered "));
   t_String szVehicleIdentifier;
   if(NULL != m_poDynamicConfigurationReader)
   {
      //Read Vehicle Id if available.
      m_poDynamicConfigurationReader->bReadVehicleId(szVehicleIdentifier);

      ETG_TRACE_USR1(("spi_tclDynamicConfiguration::szGetVehicleId() entered and value of vehicle Id after reading from datapool is :%s",szVehicleIdentifier.c_str()));
      //If Vehicle Id is not available,then generate the vehicle Id using Random Generator Algorithm
      //and write it to DataPool,Read and return it.
      ETG_TRACE_USR1(("length of Vehicle Id string after reading from datapool is :%d",szVehicleIdentifier.length()));
      if (0 == szVehicleIdentifier.length())
      {
         t_String szBTMac;
         m_poDynamicConfigurationReader->bReadBluetoothMacAddress(szBTMac);
         RandomIdGenerator oRandomIdGenerator(szBTMac);
         szVehicleIdentifier=oRandomIdGenerator.szgenerateRandomId();
         m_poDynamicConfigurationReader->bWriteVehicleId(szVehicleIdentifier);
         ETG_TRACE_USR1(("spi_tclDynamicConfiguration::szGetVehicleId() entered and value of vehicle Id after writing in datapool and then reading is :%s",szVehicleIdentifier.c_str()));
      }
   }

   return szVehicleIdentifier;
}

/***************************************************************************
 ** FUNCTION:  t_U16 spi_tclDynamicConfigurationBase::u16GetGeneralRestrictions()
***************************************************************************/
t_U16 spi_tclDynamicConfigurationBase::u16GetGeneralRestrictions(tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::u16GetGeneralRestrictions entered with device category = %d",
            ETG_ENUM( DEVICE_CATEGORY, enDeviceCategory)));
   t_U16 u16GeneralRestrictionInfo = 0;

   t_Bool bStatus = false;
   if( NULL != m_poDynamicConfigurationReader)
   {
	   if(enDeviceCategory == e8DEV_TYPE_ANDROIDAUTO)
	   {
	      bStatus = m_poDynamicConfigurationReader->bReadAAPGeneralRestrictionInfo(u16GeneralRestrictionInfo);
	      if(false == bStatus)
	      {
	         u16GeneralRestrictionInfo = scou16AAPGeneralRestrictionInfo;
	      }
	   }//(enDeviceCategory == e8DEV_TYPE_ANDROIDAUTO)
	   else if(enDeviceCategory == e8DEV_TYPE_DIPO)
	   {
	      bStatus = m_poDynamicConfigurationReader->bReadDipoGeneralRestrictionInfo(u16GeneralRestrictionInfo);
	      if (false == bStatus)
	      {
	         u16GeneralRestrictionInfo = scou16DipoGeneralRestrictionInfo;
	      }
	   }//(enDeviceCategory == e8DEV_TYPE_DIPO)
   }
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetGeneralRestrictions entered with device category = %d has the general restriction"
            "info = %d",
            ETG_ENUM( DEVICE_CATEGORY, enDeviceCategory),u16GeneralRestrictionInfo));
   tenResponseCode enResponseCode = (false == bStatus) ? e8FAILURE:e8SUCCESS;
   ETG_TRACE_USR1(("Set General restriction info completed with status : %d ", enResponseCode));
   return u16GeneralRestrictionInfo;
}

/***************************************************************************
** FUNCTION:  tenDriveSideInfo spi_tclDynamicConfigurationBase::enGetDriveSideInfo(...)
***************************************************************************/
tenDriveSideInfo spi_tclDynamicConfigurationBase::enGetDriveSideInfo()
{
   //! Initialise return value to default.
   tenDriveSideInfo enDriveSideInfo = e8LEFT_HAND_DRIVE;
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bReadDriveSideInfo(enDriveSideInfo);
   }
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enGetDriveSideInfo() left with enDriveSideInfo = %d ",
         ETG_ENUM(DRIVE_SIDE_TYPE, enDriveSideInfo)));
   return enDriveSideInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfoToDatapool()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfoToDatapool(tenDriveSideInfo enDriveSideInfo)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetDriveSideInfoToDatapool entered"));
   SPI_INTENTIONALLY_UNUSED(enDriveSideInfo);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfo()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfo(const tenVehicleConfiguration enVehicleConfig)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfiguration::vSetDriveSideInfo() entered "));
   //!Note: Nothing to do here. Value always read from KDS.
   SPI_INTENTIONALLY_UNUSED(enVehicleConfig);
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bIsAudioLastModeProjection()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bIsAudioLastModeProjection()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bIsAudioLastModeProjection() entered "));
   t_Bool bAudioLastMode = false;
   if( NULL != m_poDynamicConfigurationReader )
   {
      m_poDynamicConfigurationReader->vReadAudioLastMode(bAudioLastMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclDynamicConfigurationBase::bIsAudioLastModeProjection - Audio last mode is projection = %d", ETG_ENUM(BOOL, bAudioLastMode)));
   return bAudioLastMode;
}

/***************************************************************************
 ** FUNCTION:  t_String spi_tclDynamicConfigurationBase::szGetSPIVersion()
 ***************************************************************************/
t_String spi_tclDynamicConfigurationBase::szGetSPIVersion() const
{
  ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase()::szGetSPIVersion entered "));
  t_String szSPIVersion;
  if( NULL != m_poDynamicConfigurationReader )
  {
     if(true == m_poDynamicConfigurationReader->bReadSPIVersion(szSPIVersion))
     {
        if("Unknown" == szSPIVersion)
        {
           ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase()::szGetSPIVersion SPI Version not set in DP"));
           spi::io::FileHandler oFile(CC_LABEL_FILE_PATH,spi::io::SPI_EN_RDONLY);
           if(true == oFile.bIsValid())
           {
              ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase()::szGetSPIVersion opened cc-label.txt"));
              int size = oFile.s32GetSize();
              if(size > 0)
              {
                 t_Char str[size+1] = {'\0'};
                 oFile.bFRead(str,size);
                 szSPIVersion = str;
                 std::istringstream iss(szSPIVersion); //stringstream makes it easy to browse through words in a string
                 iss >> szSPIVersion;
                 iss >> szSPIVersion;
                 m_poDynamicConfigurationReader->bWriteSPIVersion(szSPIVersion); //write the SPI Version to DP
              }
           }
        }
     }
  }
  if(szSPIVersion.size() == 0 || "Unknown" == szSPIVersion)
  {
     //Fail safe for oDatapool.bReadSPIVersion() and oFile.bIsValid()
     szSPIVersion = DEFAULT_SPI_VERSION;
  }
  return szSPIVersion;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::bReadSelectionMode(tenDeviceSelectionMode &enSelectionMode)
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::bReadSelectionMode(tenDeviceSelectionMode &enSelectionMode)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bReadSelectionMode() entered "));
   if( NULL != m_poDynamicConfigurationReader )
   {
      m_poDynamicConfigurationReader->bReadSelectionMode(enSelectionMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclDynamicConfigurationBase::bReadSelectionMode - SelectionMode in datapool is = %d",enSelectionMode));
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAndroidDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bReadAndroidDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bReadAndroidDeviceTechnologyPreference() entered "));
   t_Bool bRet = false;
   if( NULL != m_poDynamicConfigurationReader )
   {
      bRet = m_poDynamicConfigurationReader->bReadAndroidDeviceTechnologyPreference(enTechnologyPreference);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAndroidDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bReadAppleDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bReadAppleDeviceTechnologyPreference() entered "));
   t_Bool bRet = false;
   if( NULL != m_poDynamicConfigurationReader )
   {
      bRet = m_poDynamicConfigurationReader->bReadAppleDeviceTechnologyPreference(enTechnologyPreference);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadDipoEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadDipoEnableSettingVal() const
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enReadDipoEnableSettingVal() entered "));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if( NULL != m_poDynamicConfigurationReader )
   {
	   enEnabledInfo = m_poDynamicConfigurationReader->bReadDipoEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadAAPEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadAAPEnableSettingVal() const
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enReadAAPEnableSettingVal() entered "));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if( NULL != m_poDynamicConfigurationReader )
   {
	   enEnabledInfo = m_poDynamicConfigurationReader->bReadAAPEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadmySPINEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadmySPINEnableSettingVal() const
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enReadmySPINEnableSettingVal() entered "));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if( NULL != m_poDynamicConfigurationReader )
   {
	   enEnabledInfo = m_poDynamicConfigurationReader->bReadmySPINEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadCarlifeEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadCarlifeEnableSettingVal() const
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enReadCarlifeEnableSettingVal() entered "));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if( NULL != m_poDynamicConfigurationReader )
   {
	   enEnabledInfo = m_poDynamicConfigurationReader->bReadCarlifeEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclDynamicConfigurationBase::enReadMLEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadMLEnableSettingVal() const
{
    ETG_TRACE_USR1((" spi_tclDynamicConfigurationBase::enReadMLEnableSettingVal() entered"));
    tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
    if(NULL != m_poDynamicConfigurationReader)
    {
    	enEnabledInfo = m_poDynamicConfigurationReader->bReadMLEnableSetting();
    }
    return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vClearVehicleId()
 ** Setting Vehicle Id to DataPool.
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vClearVehicleId()
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vClearVehicleId() entered "));
   t_String szVehicleIdentifier;
   if(NULL != m_poDynamicConfigurationReader)
   {
      m_poDynamicConfigurationReader->bWriteVehicleId(szVehicleIdentifier);
   }
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vClearVehicleId() entered and value of vehicle Id after writing in datapool is :%s",szVehicleIdentifier.c_str()));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetScreenConfigs()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetScreenConfigs"));


   ETG_TRACE_USR4(("Screen Width = %d",corfrDispLayerAttr.u16ScreenWidth));
   ETG_TRACE_USR4(("Screen Height = %d",corfrDispLayerAttr.u16ScreenHeight));
   ETG_TRACE_USR4(("Screen Width(mm) = %d",corfrDispLayerAttr.u16ScreenWidthMm));
   ETG_TRACE_USR4(("Screen Height(mm) = %d",corfrDispLayerAttr.u16ScreenHeightMm));
   ETG_TRACE_USR4(("Display type = %u",ETG_ENUM(DISPLAY_TYPE,corfrDispLayerAttr.enDisplayType)));

   ETG_TRACE_USR4(("Vector size is %d",corfrDispLayerAttr.vecDisplayLayerAttr.size()));

   for(t_U8 u8Index=0;u8Index<corfrDispLayerAttr.vecDisplayLayerAttr.size();u8Index++)
   {
      ETG_TRACE_USR4(("------------- %d Screen Configuration ----------",
         ETG_ENUM(DEVICE_CATEGORY,corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].enDevCat)));

      ETG_TRACE_USR4(("Video Layer Id = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16VideoLayerID));
      ETG_TRACE_USR4(("Video Surface Id = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16VideoSurfaceID));
      ETG_TRACE_USR4(("Touch Layer Id = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16TouchLayerID));
      ETG_TRACE_USR4(("Touch Surface Id  = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16TouchSurfaceID));
      ETG_TRACE_USR4(("Codec Resolution : Width in pixels = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16LayerWidth));
      ETG_TRACE_USR4(("Codec Resolution : Height in pixels = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16LayerHeight));
      ETG_TRACE_USR4(("Display Screen Width(mm) = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16DisplayWidthMm));
      ETG_TRACE_USR4(("Display Screen Height(mm) = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16DisplayHeightMm));
      ETG_TRACE_USR4(("Display Touch type = %u",ETG_ENUM(DISPLAY_TOUCH_TYPE,corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].enTouchEventType)));
      ETG_TRACE_USR4(("Pixel Aspect Ratio = %s",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].szPixelAspectRatio.c_str()));
      ETG_TRACE_USR4(("Dots per Inch = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16dpi));
      ETG_TRACE_USR4(("Priority Index = %d",corfrDispLayerAttr.vecDisplayLayerAttr[u8Index].u16PriorityIndex));
   } //for(t_U8 u8Index=0;u8Index<corfrDispLayerAttr.

   m_oLock.s16Lock();
   //Set the screen configurations in data pool, if there is any change in the current values
   if(m_rDisplayAttributes != corfrDispLayerAttr)
   {
	   if(NULL != m_poDynamicConfigurationReader)
	   {
		   if(true != m_poDynamicConfigurationReader->bWriteScreenAttributes(corfrDispLayerAttr))
	       {
			   ETG_TRACE_ERR(("spi_tclDynamicConfiguration::vSetScreenConfigs: Error in setting values in datapool"));
	       }//if(true != m_poDatapoolReader).
	   }
	   //Update the member variables of this class, to use in current power cycle.
       m_rDisplayAttributes = corfrDispLayerAttr;
   }//if(corfrDispLayerAttr != m_rDisplayAttributes)
   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetGeneralRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory, t_U16 u16GeneralRestrictionInfo)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetGeneralRestrictions() Entered with Device Category = %d",
         ETG_ENUM( DEVICE_CATEGORY, enDeviceCategory)));

   t_Bool bStatus = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
	   if(enDeviceCategory == e8DEV_TYPE_ANDROIDAUTO)
	   {
	      ETG_TRACE_USR1(("[PARAM]::vSetGeneralRestrictions()AAP General Restriction = %d",
	               u16GeneralRestrictionInfo));
	      bStatus = m_poDynamicConfigurationReader->bWriteAAPGeneralRestrictionInfo(u16GeneralRestrictionInfo);
	   }
	   else if(enDeviceCategory == e8DEV_TYPE_DIPO)
	   {
	      ETG_TRACE_USR1(("[PARAM]::vSetGeneralRestrictions()Dipo General Restriction = %d",
	               u16GeneralRestrictionInfo));
	      bStatus = m_poDynamicConfigurationReader->bWriteDipoGeneralRestrictionInfo(u16GeneralRestrictionInfo);
	   }
   }
   tenResponseCode enResponseCode = (false == bStatus) ? e8FAILURE:e8SUCCESS;
   ETG_TRACE_USR1(("Set General restriction info completed with status : %d ", enResponseCode));
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::vSetMySPINEnableSetVal()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnablel)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vSetMySPINEnableSetVal() entered "));
   // By default the AAP usage is enabled
   t_Bool bRetVal = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
	   bRetVal = m_poDynamicConfigurationReader->bWritemySPINEnableSetting(enmySPINLinkEnablel);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vGetDPI()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetDPI(const t_U32& rfu8ScreenHeight,t_U16& rfu8DPI)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetDPI() entered "));
   if (rfu8ScreenHeight <= 480)
   {
      rfu8DPI = DPI_FOUR_EIGHTY;
   }
   else if ((480 < rfu8ScreenHeight) || (rfu8ScreenHeight <= 720))
   {
      rfu8DPI = DPI_SEVEN_TWENTY;
   }
   else
   {
      rfu8DPI = DPI_TEN_EIGHTY;
   }
}

/***************************************************************************
 ** FUNCTION   :t_Void spi_tclDynamicConfigurationBase::vGetDisplayLayerAttributeData()
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetDisplayLayerAttributeData(trVideoConfigData& rVideoConfigData,
                                                          const trDisplayLayerAttributes& corfrDispLayerAttr)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetDisplayLayerAttributeData entered"));
   rVideoConfigData.u32LayerId = corfrDispLayerAttr.u16VideoLayerID;
   rVideoConfigData.u32SurfaceId = corfrDispLayerAttr.u16VideoSurfaceID;
   rVideoConfigData.u32TouchLayerId = corfrDispLayerAttr.u16TouchLayerID;
   rVideoConfigData.u32TouchSurfaceId = corfrDispLayerAttr.u16TouchSurfaceID;

   rVideoConfigData.u32ProjScreen_Width = corfrDispLayerAttr.u16LayerWidth;
   rVideoConfigData.u32ProjScreen_Height = corfrDispLayerAttr.u16LayerHeight;

   rVideoConfigData.u32ProjScreen_Width_Mm = corfrDispLayerAttr.u16DisplayWidthMm;
   rVideoConfigData.u32ProjScreen_Height_Mm = corfrDispLayerAttr.u16DisplayHeightMm;
   rVideoConfigData.enTouchEventType = corfrDispLayerAttr.enTouchEventType;
   memset(rVideoConfigData.szPixelAspectRatio, '\0', MAXKEYSIZE);

   if (!corfrDispLayerAttr.szPixelAspectRatio.empty())
   {
      //If HMI sets the value
      strncpy(rVideoConfigData.szPixelAspectRatio, corfrDispLayerAttr.szPixelAspectRatio.c_str(), MAXKEYSIZE-1);
   }
   else
   {
      //If HMI doesn't set the value, then Default value is used
      t_String szPixelAspectRatio;
      vGetPixelAspectRatio(szPixelAspectRatio);
      strncpy(rVideoConfigData.szPixelAspectRatio, szPixelAspectRatio.c_str(), MAXKEYSIZE-1);
   }

   if (0 != corfrDispLayerAttr.u16dpi)
   {
      //If HMI sets the value
      rVideoConfigData.u16dpi = corfrDispLayerAttr.u16dpi;
   }
   else
   {
      //If HMI doesn't set the value, then Default value is used - 160 dpi for 480p and 240dpi for 720p
      vGetDPI(rVideoConfigData.u32Screen_Height, rVideoConfigData.u16dpi);

   }

   //remove these two statements, once HMI implements Screen width & height in MM
   if ((0 == rVideoConfigData.u32ProjScreen_Width_Mm) && (0 != m_rDisplayAttributes.u16ScreenWidth) && (0
            != m_rDisplayAttributes.u16ScreenWidthMm))
   {
      rVideoConfigData.u32ProjScreen_Width_Mm = (t_U32) ((float) (corfrDispLayerAttr.u16LayerWidth)
               / (m_rDisplayAttributes.u16ScreenWidth)) * (m_rDisplayAttributes.u16ScreenWidthMm);
   }//if( (0 == rfrVideoConfig.u32ProjSc

   if ((0 == rVideoConfigData.u32ProjScreen_Height_Mm) && (0 != m_rDisplayAttributes.u16ScreenHeight) && (0
            != m_rDisplayAttributes.u16ScreenHeightMm))
   {
      rVideoConfigData.u32ProjScreen_Height_Mm = (t_U32) ((float) (corfrDispLayerAttr.u16LayerHeight)
               / (m_rDisplayAttributes.u16ScreenHeight)) * (m_rDisplayAttributes.u16ScreenHeightMm);
   }//if( (0 == rfrVideoConfig.u32ProjScreen_Heig

   //Assign the priority index.
   rVideoConfigData.u16PriorityIndex = corfrDispLayerAttr.u16PriorityIndex;
}

/***************************************************************************
 ** FUNCTION   :t_Void spi_tclDynamicConfigurationBase::vGetDisplayAttributeData()
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetDisplayAttributeData(trVideoConfigData& rfrVideoConfigData)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetDisplayAttributeData entered"));
   rfrVideoConfigData.u32ResolutionSupported = FOUR_EIGHTY;
   rfrVideoConfigData.u32Screen_Width = m_rDisplayAttributes.u16ScreenWidth;
   rfrVideoConfigData.u32Screen_Height = m_rDisplayAttributes.u16ScreenHeight;
   rfrVideoConfigData.u32Screen_Width_Millimeter = m_rDisplayAttributes.u16ScreenWidthMm;
   rfrVideoConfigData.u32Screen_Height_Millimeter = m_rDisplayAttributes.u16ScreenHeightMm;
   rfrVideoConfigData.enDisplayType = m_rDisplayAttributes.enDisplayType;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vPrintVideoConfig(...)
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vPrintVideoConfig(const trVideoConfigData& corfrVideoConfig)
{
   //! print video configuration data retrieved by reading EOL value and
   //! get the video configuration data from the look up table
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vPrintVideoConfig entered"));
   ETG_TRACE_USR4(("Screen Width = %d",corfrVideoConfig.u32Screen_Width));
   ETG_TRACE_USR4(("Screen Height = %d",corfrVideoConfig.u32Screen_Height));
   ETG_TRACE_USR4(("Screen Width(mm) = %d",corfrVideoConfig.u32Screen_Width_Millimeter));
   ETG_TRACE_USR4(("Screen Height(mm) = %d",corfrVideoConfig.u32Screen_Height_Millimeter));
   ETG_TRACE_USR4(("Display type = %u",ETG_ENUM(DISPLAY_TYPE,corfrVideoConfig.enDisplayType)));
   ETG_TRACE_USR4(("Display Touch type = %u",ETG_ENUM(DISPLAY_TOUCH_TYPE,corfrVideoConfig.enTouchEventType)));
   ETG_TRACE_USR4(("Layer Id = %d",corfrVideoConfig.u32LayerId));
   ETG_TRACE_USR4(("Surface Id = %d",corfrVideoConfig.u32SurfaceId));
   ETG_TRACE_USR4(("Touch Layer Id = %d",corfrVideoConfig.u32TouchLayerId));
   ETG_TRACE_USR4(("Touch Surface Id  = %d",corfrVideoConfig.u32TouchSurfaceId));
   ETG_TRACE_USR4(("Proj Screen Width = %d",corfrVideoConfig.u32ProjScreen_Width));
   ETG_TRACE_USR4(("Proj Screen Height = %d",corfrVideoConfig.u32ProjScreen_Height));
   ETG_TRACE_USR4(("Proj Screen Width(mm) = %d",corfrVideoConfig.u32ProjScreen_Width_Mm));
   ETG_TRACE_USR4(("Proj Screen Height(mm) = %d",corfrVideoConfig.u32ProjScreen_Height_Mm));
   ETG_TRACE_USR4(("Proj ScreenPixel aspect ratio= %s",corfrVideoConfig.szPixelAspectRatio));
   ETG_TRACE_USR4(("Dots Per Inch = %d",corfrVideoConfig.u16dpi));
   ETG_TRACE_USR4(("Priority Index = %d",corfrVideoConfig.u16PriorityIndex));
   ETG_TRACE_USR4(("szDisplayDev = %s",corfrVideoConfig.szDisplayDev));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::vfindVideoDevPath()
 ***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vfindVideoDevPath(t_String& rfszVideoDevice)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vfindVideoDevPath entered"));
   const t_String szSysPath = "/sys/class/video4linux"; //base folder where video device ned to be searched
   rfszVideoDevice.clear(); //initialize as empty

   shl::io::DirHandler oDirHandler(szSysPath.c_str());
   if (oDirHandler.bIsValid())
   {
      struct dirent * entry = NULL;
      const t_String szM2mDevName = "m2m"; //this is the video device name to be searched
      std::ostringstream oss;

      //read through each directory in the selected folder
      while ((entry = oDirHandler.prReadDir()) != NULL)
      {
         t_String dirName = entry->d_name;
         if (dirName == "." || dirName == "..")
         {
            continue;
         }

         //derive the complete name of the file to be read
         oss.str("");
         oss << szSysPath << "/" << dirName << "/name";

         //try to open the video device file, if it exist
         std::ifstream is(oss.str().c_str());
         t_String  name;
         if(is)
         {
            //read the data from the file
            std::ostringstream ss;
            ss << is.rdbuf();
            name = ss.str();
            is.close();
         }

         //check if m2m device is found
         size_t m2mDevFound = name.find(szM2mDevName);
         if (m2mDevFound != std::string::npos)
         {
            t_String devicePrefix = "/dev/";
            //the actual device file exist at /dev/*
            rfszVideoDevice = devicePrefix + dirName;
            ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vfindVideoDevPath - found M2M device : %s", rfszVideoDevice.c_str()));
            break;
         }
      }
   }
}
/***************************************************************************
** FUNCTION   :t_Void spi_tclDynamicConfigurationBase::vGetVideoConfigData()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetVideoConfigData(tenDeviceCategory enDevCat,
                                                tvecVideoConfigList& rfvecVideoConfigList)
{

   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetVideoConfigData - %d",ETG_ENUM(DEVICE_CATEGORY,enDevCat)));
   m_oLock.s16Lock();

   trVideoConfigData rVideoConfigData;
   trDisplayLayerAttributes rDisplayLayerAttributes;


   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetVideoConfigData size- %d",m_rDisplayAttributes.vecDisplayLayerAttr.size()));

   t_String szVideoDevice;
   vfindVideoDevPath(szVideoDevice);
   // Read the technology specific configurations
   for(t_U8 u8Index=0; u8Index<m_rDisplayAttributes.vecDisplayLayerAttr.size();u8Index++)
   {
      ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetVideoConfigData Index %d Category- %d",u8Index,ETG_ENUM(DEVICE_CATEGORY,m_rDisplayAttributes.vecDisplayLayerAttr[u8Index].enDevCat)));
      if(enDevCat == m_rDisplayAttributes.vecDisplayLayerAttr[u8Index].enDevCat)
      {
         memset(rVideoConfigData.szDisplayDev, '\0', PATH_LENGTH);
         strncpy(rVideoConfigData.szDisplayDev, szVideoDevice.c_str(), PATH_LENGTH-1);
         vGetDisplayAttributeData(rVideoConfigData);
         rDisplayLayerAttributes = m_rDisplayAttributes.vecDisplayLayerAttr[u8Index];
         vGetDisplayLayerAttributeData(rVideoConfigData, rDisplayLayerAttributes);
         //Store the video config data into the vector variable.
         rfvecVideoConfigList.push_back(rVideoConfigData);

         //Print the current display config for the technology
         vPrintVideoConfig(rVideoConfigData);
      }//if(enDevCat == m_rDisplayAttributes.v
   }// for(t_U8 u8Index=0; u8Index<m_rDisplayAttributes.vecDi
   m_oLock.vUnlock();
}

/***************************************************************************
** FUNCTION   :t_Void spi_tclDynamicConfigurationBase::vGetPrimaryDisplayConfiguration()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetPrimaryDisplayConfiguration(const tvecVideoConfigList& corfvecVideoConfigList,
                                       trVideoConfigData& rVideoConfigData)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclDynamicConfigurationBase::vGetPrimaryDisplayConfiguration entered"));
   for (t_U8 u8Index = 0; u8Index < corfvecVideoConfigList.size(); u8Index++)
   {
      if (PRIMARY_INDEX == corfvecVideoConfigList[u8Index].u16PriorityIndex)
      {
         rVideoConfigData = corfvecVideoConfigList[u8Index];
      }
   }
}


/***************************************************************************
** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bWriteOnCarEnableSetting()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bWriteOnCarEnableSetting()"));
   t_Bool bRet = false;
   if(NULL != m_poDynamicConfigurationReader )
   {
      bRet = m_poDynamicConfigurationReader->bWriteOnCarEnableSetting(enOnCarLinkEnable);
   }
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bReadVehicleManufacturerInfofromDatapool()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bReadVehicleManufacturerInfofromDatapool(trVehicleManufacturerInformation& rfrVehicleManufacturerInformation)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bReadVehicleManufacturerInfofromDatapool()"));
   SPI_INTENTIONALLY_UNUSED(rfrVehicleManufacturerInformation);
   return false;
}

/***************************************************************************
** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadOnCarEnableSetting()
***************************************************************************/
tenEnabledInfo spi_tclDynamicConfigurationBase::enReadOnCarEnableSetting() const
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::enReadOnCarEnableSetting()"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if(NULL != m_poDynamicConfigurationReader )
   {
      enEnabledInfo = m_poDynamicConfigurationReader->enReadOnCarEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vGetVehicleBTAddress()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vGetVehicleBTAddress(t_String &szVehicleBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vGetVehicleBTAddress()"));
   if(NULL != m_poDynamicConfigurationReader )
   {
	   m_poDynamicConfigurationReader->bReadBluetoothMacAddress(szVehicleBTAddress);
   }
   ETG_TRACE_USR2(("spi_tclDynamicConfigurationBase::vGetVehicleBTAddress = %s ", szVehicleBTAddress.c_str()));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vOnVehicleBTAddress()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vOnVehicleBTAddress(t_String &szVehicleBTAddress)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vOnVehicleBTAddress()"));
   if(NULL != m_poDynamicConfigurationReader )
   {
	   m_poDynamicConfigurationReader->bWriteBluetoothMacAddress(szVehicleBTAddress);
   }
   ETG_TRACE_USR2(("spi_tclDynamicConfigurationBase::vOnVehicleBTAddress = %s ", szVehicleBTAddress.c_str()));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vRegisterKeyCodes()
***************************************************************************/
t_Void spi_tclDynamicConfigurationBase::vRegisterKeyCodes(
   const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardKeys,
   const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftKeys)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::vRegisterKeyCodes entered"));
   //Register the Key info in data pool, if there is any change in the current values
   if(NULL != m_poDynamicConfigurationReader)
   {
      for(auto const &TechKeys : corfrmapTechSuppHardKeys)
      {
         m_poDynamicConfigurationReader->bWriteHardKeyInfo(TechKeys.second,TechKeys.first);
      }
      for(auto const &TechKeys : corfrmapTechSuppSoftKeys)
      {
         m_poDynamicConfigurationReader->bWriteSoftKeyInfo(TechKeys.second,TechKeys.first);
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bGetKeyInfo()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bGetKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{

   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bGetKeyInfo entered"));
   t_Bool bretVal = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
      bretVal = m_poDynamicConfigurationReader->bReadKeyCodeInfo(rfrsetKeyCodes,enCategory);
   }
   return bretVal;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bGetSoftKeyInfo()
***************************************************************************/
t_Bool spi_tclDynamicConfigurationBase::bGetSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{

   ETG_TRACE_USR1(("spi_tclDynamicConfigurationBase::bGetKeyInfo entered"));
   t_Bool bretVal = false;
   if(NULL != m_poDynamicConfigurationReader)
   {
      bretVal = m_poDynamicConfigurationReader->bReadSoftKeyInfo(rfrsetKeyCodes,enCategory);
   }
   return bretVal;
}
