/***********************************************************************/
/*!
 * \file  Datapool.cpp
 * \brief Wrapper class for Datapool usage
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Wrapper class for Datapool usage
 AUTHOR:         Ramya Murthy
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 13.05.2014  | Ramya Murthy          | Initial Version
 11.09.2014  | Shihabudheen P M      | Modified for BT MAC address writing and reading
 25.06.2015  | Shiva kaumr G         | Dynamic display configuration
 14.03.2016  | Chaitra Srinivasa     | Default settings
 18.07.2017  | Noopur R K            | Added bReadDiPODriveRestrictionInfo,bWriteGeneralRestrictionInfo,
 bReadAAPGeneralRestrictionInfo, bWriteAAPGeneralRestrictionInfo

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "Datapool.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_FI

#include "dp_smartphoneintegration_if.h"

// Includes all header files for FI of datapool
#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/Datapool.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
typedef dp_tclSpiDpInternalDataMLNotificationSetting dp_tFiMLNotifSetting;
typedef dp_tclSpiDpInternalDataMirrorLinkEnableSetting dp_tFiMLEnableSetting;
typedef dp_tclSpiDpInternalDataDipoEnableSetting dp_tFiDipoEnableSetting;
typedef dp_tclSpiDpInternalDataAAPEnableSetting dp_tFiAAPEnableSetting;
typedef dp_tclSpiDpInternalDataScreenAttributes dp_tFiScreenAttr;
typedef dp_tclSpiDpInternalDataDiPODriveRestrictionInfo dp_tDriveRestInfo;
typedef dp_tclSpiDpInternalDataSteeringWheelPosition dp_tSteeringWheelPos;
typedef dp_tclSpiDpInternalDataVehicleID dp_tVehicleId;
typedef dp_tclSpiDpInternalDataSelectionMode dp_tSelectionMode;
typedef dp_tclSpiDpInternalDataVirginStart dp_tVirginStart;
typedef dp_tclSpiDpInternalDataSPIVersion dp_tSPIVersion;
typedef dp_tclSpiDpInternalDataAAPGeneralRestrictionInfo dp_tAAPGeneralRestInfo;
typedef dp_tclSpiDpInternalDataDipoGeneralRestrictionInfo dp_tDipoGeneralRestInfo;
typedef dp_tclSpiDpInternalDatamySPINEnableSetting dp_tFimySPINEnableSetting;
typedef dp_tclSpiDpInternalDataCarlifeEnableSetting dp_tFiCarlifeEnableSetting;
typedef dp_tclSpiDpInternalDataVehicleManufacturerInformation dp_tSPIVMInfo;
typedef dp_tclSpiDpInternalDataDisplayLastMode dp_tDisplayLastMode;
typedef dp_tclSpiDpInternalDataAudioLastMode dp_tAudioLastMode;
typedef dp_tclSpiDpInternalDataAndroidDeviceTechnologyPreference dp_tAdroidDeviceTechnologyPreference;
typedef dp_tclSpiDpInternalDataAppleDeviceTechnologyPreference   dp_tAppleDeviceTechnologyPreference;
typedef dp_tclSpiDpInternalDataOnCarEnableSetting                dp_tFiOnCarEnableSetting;
typedef dp_tclSpiDpInternalDatau32AASuppHardKeysArr              dp_tFiu32AASuppHardKeysArr;
typedef dp_tclSpiDpInternalDatau32DiPoSuppHardKeysArr            dp_tFiu32DiPoSuppHardKeysArr;
typedef dp_tclSpiDpInternalDatau32DiPoSuppSoftKeysArr            dp_tFiu32DiPoSuppSoftKeysArr;

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

#define MAX_MAC_ADDRESS_STRLEN 20
#define MAX_VEHICLE_ID_STRLEN  128
#define MAX_SPI_VERSION_STRLEN 255

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  Datapool::Datapool(const trGPSData& rfcorGpsData)
 ***************************************************************************/
Datapool::Datapool()
{
   ETG_TRACE_USR1(("Datapool() entered \n"));
}

/***************************************************************************
 ** FUNCTION: Datapool::~Datapool()
 ***************************************************************************/
Datapool::~Datapool()
{
   ETG_TRACE_USR1(("~Datapool() entered \n"));
   m_mapTechSuppSoftkeys.clear();
   m_mapTechSuppHardkeys.clear();
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::bReadMLEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::bReadMLEnableSetting() const
{
   //! Create an object of Datapool FI class & read the data
   dp_tFiMLEnableSetting oMLEnableSetting;
   tenEnabledInfo enMLEnabled = (tenEnabledInfo) oMLEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadMLEnableSetting: left with bMLEnabled = %u \n", enMLEnabled));
   return enMLEnabled;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::bReadDipoEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::bReadDipoEnableSetting() const
{
   //! Create an object of Datapool FI class & read the data
   dp_tFiDipoEnableSetting oDipoEnableSetting;
   tenEnabledInfo enDiPOEnabled = (tenEnabledInfo) oDipoEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadDipoEnableSetting: left with bDiPOEnabled = %u \n", enDiPOEnabled));
   return enDiPOEnabled;
}
/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::bReadAAPEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::bReadAAPEnableSetting() const
{
   //! TODO Create an object of Datapool FI class & read the data
   dp_tFiAAPEnableSetting oAAPEnableSetting;
   tenEnabledInfo enAAPEnabled = (tenEnabledInfo) oAAPEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadAAPEnableSetting: left with bAAPEnabled = %u \n", enAAPEnabled));
   return enAAPEnabled;
}
/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::enReadOnCarEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::enReadOnCarEnableSetting() const
{
   //! TODO Create an object of Datapool FI class & read the data
   dp_tFiOnCarEnableSetting oOnCarEnableSetting;
   tenEnabledInfo enOnCarEnabled = (tenEnabledInfo) oOnCarEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::enReadOnCarEnableSetting: left with bOnCarEnabled = %u \n", enOnCarEnabled));
   return enOnCarEnabled;
}
/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::bReadmySPINEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::bReadmySPINEnableSetting() const
{
   //! Create an object of Datapool FI class & read the data
   dp_tFimySPINEnableSetting omySPINEnableSetting;
   tenEnabledInfo enmySPINEnabled = (tenEnabledInfo) omySPINEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadmySPINEnableSetting: left with bmySPINEnabled = %u \n", enmySPINEnabled));
   return enmySPINEnabled;
}
/***************************************************************************
 ** FUNCTION:  tenEnabledInfo Datapool::bReadCarlifeEnableSetting()
 ***************************************************************************/
tenEnabledInfo Datapool::bReadCarlifeEnableSetting() const
{
   //! TODO Create an object of Datapool FI class & read the data
   dp_tFiCarlifeEnableSetting oCarlifeEnableSetting;
   tenEnabledInfo enCarlifeEnabled = (tenEnabledInfo) oCarlifeEnableSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadCarlifeEnableSetting: left with bCarlifeEnabled = %u \n", enCarlifeEnabled));
   return enCarlifeEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadMLNotificationSetting()
 ***************************************************************************/
t_Bool Datapool::bReadMLNotificationSetting() const
{
   //! Create an object of Datapool FI class & read the data
   dp_tFiMLNotifSetting oMLNotifSetting;
   t_Bool bMLNotifEnabled = oMLNotifSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::bReadMLNotificationSetting: left with bMLNotifEnabled = %u \n", bMLNotifEnabled));
   return bMLNotifEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_U8 Datapool::u8ReadVirginStartSetting()
 ***************************************************************************/
t_U8 Datapool::u8ReadVirginStartSetting() const
{
   //! Create an object of Datapool FI class & read the data
   dp_tVirginStart oVirginStartSetting;
   t_U8 u8VirginStart = (tenEnabledInfo) oVirginStartSetting.tGetData();

   ETG_TRACE_USR4(("Datapool::u8ReadVirginStartSetting: left with u8VirginStart = %u \n", u8VirginStart));
   return u8VirginStart;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteMLEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWriteMLEnableSetting(tenEnabledInfo enMirrorLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWriteMLEnableSetting: entered with enMirrorLinkEnable = %u \n", enMirrorLinkEnable));

   //! Create an object of Datapool FI class & set the data
   dp_tFiMLEnableSetting oMLEnableSetting;
   t_S32 s32WriteError = oMLEnableSetting.s32SetData(enMirrorLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteMLEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDipoEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWriteDipoEnableSetting(tenEnabledInfo enDipoLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWriteDipoEnableSetting: entered with enDipoLinkEnable = %u \n", enDipoLinkEnable));

   //! Create an object of Datapool FI class & set the data
   dp_tFiDipoEnableSetting oDipoEnableSetting;
   t_S32 s32WriteError = oDipoEnableSetting.s32SetData(enDipoLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteDipoEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}
/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAAPEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWriteAAPEnableSetting(tenEnabledInfo enAAPLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWriteAAPEnableSetting: entered with enAAPLinkEnable = %u \n", enAAPLinkEnable));
   //! Create an object of Datapool FI class & set the data
   dp_tFiAAPEnableSetting oAAPEnableSetting;
   t_S32 s32WriteError = oAAPEnableSetting.s32SetData(enAAPLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteAAPEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteOnCarEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWriteOnCarEnableSetting: entered with enOnCarLinkEnable = %u \n", enOnCarLinkEnable));
   //! Create an object of Datapool FI class & set the data
   dp_tFiOnCarEnableSetting oOnCarEnableSetting;
   t_S32 s32WriteError = oOnCarEnableSetting.s32SetData(enOnCarLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteOnCarEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWritemySPINEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWritemySPINEnableSetting(tenEnabledInfo enmySPINLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWritemySPINEnableSetting: entered with enmySPINLinkEnable = %u \n", enmySPINLinkEnable));
   //! Create an object of Datapool FI class & set the data
   dp_tFimySPINEnableSetting omySPINEnableSetting;
   t_S32 s32WriteError = omySPINEnableSetting.s32SetData(enmySPINLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWritemySPINEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteCarlifeEnableSetting(tenEnabledInfo...)
 ***************************************************************************/
t_Bool Datapool::bWriteCarlifeEnableSetting(tenEnabledInfo enCarlifeLinkEnable)
{
   ETG_TRACE_USR1(("Datapool::bWriteCarlifeEnableSetting: entered with enCarlifeLinkEnable = %u \n", enCarlifeLinkEnable));
   //! Create an object of Datapool FI class & set the data
   dp_tFiCarlifeEnableSetting oCarlifeEnableSetting;
   t_S32 s32WriteError = oCarlifeEnableSetting.s32SetData(enCarlifeLinkEnable);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteCarlifeEnableSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteMLNotificationSetting(t_Bool...)
 ***************************************************************************/
t_Bool Datapool::bWriteMLNotificationSetting(t_Bool bNotificationEnabled)
{
   ETG_TRACE_USR1(("Datapool::bWriteMLNotificationSetting: entered with bNotificationEnabled = %u \n", bNotificationEnabled));

   //! Create an object of Datapool FI class & set the data
   dp_tFiMLNotifSetting oMLNotifSetting;
   t_S32 s32WriteError = oMLNotifSetting.s32SetData(bNotificationEnabled);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteMLNotificationSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteBluetoothMacAddress(t_String...)
 ***************************************************************************/
t_Bool Datapool::bWriteBluetoothMacAddress(t_String &szBluetoothMacAddr)
{
   ETG_TRACE_USR1(("Datapool::bWriteBluetoothMacAddress: entered with Bluetooth MAC address = %s \n", szBluetoothMacAddr.c_str()));

   dp_tclSpiDpInternalDataBluetoothMACAddress oBluetoothMacAddr;
   t_S32 s32WriteError = oBluetoothMacAddr.s32SetData(const_cast<t_Char *>(szBluetoothMacAddr.c_str()));

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteBluetoothMacAddress: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_U8 Datapool::u8WriteVirginStartSetting()
 ***************************************************************************/
t_U8 Datapool::u8WriteVirginStartSetting(t_U8 u8VirginStart)
{
   ETG_TRACE_USR1(("Datapool::u8WriteVirginStartSetting: entered with u8VirginStart = %u \n", u8VirginStart));

   //! Create an object of Datapool FI class & set the data
   dp_tVirginStart oVirginStartSetting;
   t_S32 s32WriteError = oVirginStartSetting.s32SetData(u8VirginStart);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::u8WriteVirginStartSetting: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadBluetoothMacAddress(t_String...)
 ***************************************************************************/
t_Bool Datapool::bReadBluetoothMacAddress(t_String &szBluetoothMacAddr)
{
   ETG_TRACE_USR1(("Datapool::bReadBluetoothMacAddress: entered"));
   t_Char * szMacAddress = new char[MAX_MAC_ADDRESS_STRLEN];

   dp_tclSpiDpInternalDataBluetoothMACAddress oBluetoothMacAddr;
   t_S32 s32ReadDataLength = oBluetoothMacAddr.s32GetData(szMacAddress, MAX_MAC_ADDRESS_STRLEN);

   if (0 != s32ReadDataLength)
   {
      szBluetoothMacAddr = szMacAddress;
   }
   else
   {
      ETG_TRACE_ERR(("Datapool::bReadBluetoothMacAddress: Read error occured "));
   }
   delete[] szMacAddress;
   return (0 != s32ReadDataLength);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::vStringCopy(t_Char* cDestination,const t_String szSource)
 ***************************************************************************/
t_Void Datapool::vStringCopy(t_Char* cDestination, const t_String szSource)
{
   if (szSource.length() < MAX_SIZE_CHAR_ARRAY)
   {
      memset(cDestination, '\0', MAX_SIZE_CHAR_ARRAY);
      strncpy(cDestination, szSource.c_str(), MAX_SIZE_CHAR_ARRAY);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteScreenAttributes()
 ***************************************************************************/
t_Bool Datapool::bWriteScreenAttributes(const trDisplayAttributes& corfrDispAttr)
{
   ETG_TRACE_USR1(("Datapool::bWriteScreenAttributes() entered"));

   dp_tFiScreenAttr oScreenAttr;
   trScreenProperties rScreenProp;
   rScreenProp.u16ScreenHeight = corfrDispAttr.u16ScreenHeight;
   rScreenProp.u16ScreenWidth = corfrDispAttr.u16ScreenWidth;
   rScreenProp.u16ScreenHeightMm = corfrDispAttr.u16ScreenHeightMm;
   rScreenProp.u16ScreenWidthMm = corfrDispAttr.u16ScreenWidthMm;
   rScreenProp.enDPDisplayType =
            (e8RESISTIVE_SCREEN == corfrDispAttr.enDisplayType) ? e8DP_RESISTIVE_SCREEN : e8DP_CAPACITIVE_SCREEN;
   rScreenProp.u16DisplayLayerAttributesSize = static_cast<t_U16>(corfrDispAttr.vecDisplayLayerAttr.size());

   for (t_U8 u8Index = 0; u8Index < corfrDispAttr.vecDisplayLayerAttr.size(); u8Index++)
   {
      const trDisplayLayerAttributes& corfrDispLayerAttr = corfrDispAttr.vecDisplayLayerAttr[u8Index];
      trProjLayerAttributes& rfrProjLayerAttributes = rScreenProp.aProjLayerAttr[u8Index];
      rfrProjLayerAttributes.enDPDevCat = static_cast<tenDPDevCat>(corfrDispLayerAttr.enDevCat);
      rfrProjLayerAttributes.u16VideoLayerID = corfrDispLayerAttr.u16VideoLayerID;
      rfrProjLayerAttributes.u16VideoSurfaceID = corfrDispLayerAttr.u16VideoSurfaceID;
      rfrProjLayerAttributes.u16TouchLayerID = corfrDispLayerAttr.u16TouchLayerID;
      rfrProjLayerAttributes.u16TouchSurfaceID = corfrDispLayerAttr.u16TouchSurfaceID;
      rfrProjLayerAttributes.u16LayerHeight = corfrDispLayerAttr.u16LayerHeight;
      rfrProjLayerAttributes.u16LayerWidth = corfrDispLayerAttr.u16LayerWidth;
      rfrProjLayerAttributes.u16DisplayHeightMm = corfrDispLayerAttr.u16DisplayHeightMm;
      rfrProjLayerAttributes.u16DisplayWidthMm = corfrDispLayerAttr.u16DisplayWidthMm;
      rfrProjLayerAttributes.enDPTouchEventType =
               (e8SINGLE_TOUCH == corfrDispLayerAttr.enTouchEventType) ? e8DP_SINGLETOUCH : e8DP_MULTITOUCH;
      vStringCopy(rfrProjLayerAttributes.cPixelAspectRatio, corfrDispLayerAttr.szPixelAspectRatio);
      rfrProjLayerAttributes.u16dpi = corfrDispLayerAttr.u16dpi;
      rfrProjLayerAttributes.u16PriorityIndex = corfrDispLayerAttr.u16PriorityIndex;
   } //for(t_U8 u8Index=0;u8Index<rfrDispAttr;u8Index++)

   t_S32 s32Error = oScreenAttr.s32SetData(rScreenProp);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteScreenAttributes: Error = %u", s32Error));
   } //if (DP_S32_NO_ERR != s32Error)
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadScreenAttributes()
 ***************************************************************************/
t_Bool Datapool::bReadScreenAttributes(trDisplayAttributes& rfrDispAttr)
{
   ETG_TRACE_USR1(("Datapool::bReadScreenAttributes() entered"));

   dp_tFiScreenAttr oScreenAttr;
   trScreenProperties rScreenProp;
   t_S32 s32Error = oScreenAttr.s32GetData(rScreenProp);

   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadScreenAttributes: Read error occured - %u ", s32Error));
   } //if(DP_S32_NO_ERR != s32Error)
   else
   {
      rfrDispAttr.u16ScreenHeight = rScreenProp.u16ScreenHeight;
      rfrDispAttr.u16ScreenWidth = rScreenProp.u16ScreenWidth;
      rfrDispAttr.u16ScreenHeightMm = rScreenProp.u16ScreenHeightMm;
      rfrDispAttr.u16ScreenWidthMm = rScreenProp.u16ScreenWidthMm;
      rfrDispAttr.enDisplayType =
               (e8DP_RESISTIVE_SCREEN == rScreenProp.enDPDisplayType) ? e8RESISTIVE_SCREEN : e8CAPACITIVE_SCREEN;
      //Iterate the loop only for the number of display layer attributes populated in the datapool
      t_U16 u16DisplayLayerAttributesSize = rScreenProp.u16DisplayLayerAttributesSize;

      ETG_TRACE_ERR(("Datapool::bReadScreenAttributes: Vector size- %d ", u16DisplayLayerAttributesSize));

      for (t_U8 u8Index = 0; u8Index < u16DisplayLayerAttributesSize; u8Index++)
      {
         trDisplayLayerAttributes rfrDispLayerAttr;
         const trProjLayerAttributes& corfrProjLayerAttributes = rScreenProp.aProjLayerAttr[u8Index];
         rfrDispLayerAttr.enDevCat = static_cast<tenDeviceCategory>(corfrProjLayerAttributes.enDPDevCat);
         rfrDispLayerAttr.u16VideoLayerID = corfrProjLayerAttributes.u16VideoLayerID;
         rfrDispLayerAttr.u16VideoSurfaceID = corfrProjLayerAttributes.u16VideoSurfaceID;
         rfrDispLayerAttr.u16TouchLayerID = corfrProjLayerAttributes.u16TouchLayerID;
         rfrDispLayerAttr.u16TouchSurfaceID = corfrProjLayerAttributes.u16TouchSurfaceID;
         rfrDispLayerAttr.u16LayerHeight = corfrProjLayerAttributes.u16LayerHeight;
         rfrDispLayerAttr.u16LayerWidth = corfrProjLayerAttributes.u16LayerWidth;
         rfrDispLayerAttr.u16DisplayHeightMm = corfrProjLayerAttributes.u16DisplayHeightMm;
         rfrDispLayerAttr.u16DisplayWidthMm = corfrProjLayerAttributes.u16DisplayWidthMm;
         rfrDispLayerAttr.enTouchEventType =
                  (e8DP_SINGLETOUCH == corfrProjLayerAttributes.enDPTouchEventType) ? e8SINGLE_TOUCH : e8MULTI_TOUCH;
         rfrDispLayerAttr.szPixelAspectRatio = corfrProjLayerAttributes.cPixelAspectRatio;
         rfrDispLayerAttr.u16dpi = corfrProjLayerAttributes.u16dpi;
         rfrDispLayerAttr.u16PriorityIndex = corfrProjLayerAttributes.u16PriorityIndex;
         rfrDispAttr.vecDisplayLayerAttr.push_back(rfrDispLayerAttr);
      }
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadDiPODriveRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bReadDiPODriveRestrictionInfo(t_U8 &cu8DiPORestrictionInfo)
{
   dp_tDriveRestInfo oDriveRestInfo;
   t_S32 s32Error = oDriveRestInfo.s32GetData(cu8DiPORestrictionInfo);

   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadDiPODriveRestrictionInfo: Read error occured - %u ", s32Error));
   }
   else
   {
      ETG_TRACE_USR1(("Datapool::bReadDiPODriveRestrictionInfo() value %d:", cu8DiPORestrictionInfo));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDiPODriveRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteDiPODriveRestrictionInfo(t_U8 cu8DiPORestrictionInfo)
{
   dp_tDriveRestInfo oDriveRestInfo;
   t_U8 u8DriveRestInfo = 0;
   t_S32 s32Error = oDriveRestInfo.s32GetData(u8DriveRestInfo);

   if (DP_S32_NO_ERR == s32Error)
   {
      if (cu8DiPORestrictionInfo != u8DriveRestInfo)
      {
         s32Error = oDriveRestInfo.s32SetData(cu8DiPORestrictionInfo);
      }
   }

   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWrieDiPODriveRestrictionInfo: Write error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAAPGeneralRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bReadAAPGeneralRestrictionInfo(t_U16 &cu8AAPGeneralRestrictionInfo)
{
   dp_tAAPGeneralRestInfo oAAPGeneralRestInfo;
   t_S32 s32Error = oAAPGeneralRestInfo.s32GetData(cu8AAPGeneralRestrictionInfo);

   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadAAPGeneralRestrictionInfo: Read error occured - %u ", s32Error));
   }
   else
   {
      ETG_TRACE_USR1(("Datapool::bReadAAPGeneralRestrictionInfo() value %d:", cu8AAPGeneralRestrictionInfo));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAAPGeneralRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteAAPGeneralRestrictionInfo(t_U16 u16AAPGeneralRestrictionInfo)
{
   dp_tAAPGeneralRestInfo oAAPGeneralRestInfo;
   t_U16 u16AAPGeneralRestInfo = 0;

   t_S32 s32Error = oAAPGeneralRestInfo.s32GetData(u16AAPGeneralRestInfo);

   if (DP_S32_NO_ERR == s32Error)
   {
      if (u16AAPGeneralRestrictionInfo != u16AAPGeneralRestInfo)
      {
         s32Error = oAAPGeneralRestInfo.s32SetData(u16AAPGeneralRestrictionInfo);
      }
   }
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteAAPGeneralRestrictionInfo: Write error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadDiPODriveRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bReadDipoGeneralRestrictionInfo(t_U16 &cu8DipoGeneralRestrictionInfo)
{
   dp_tDipoGeneralRestInfo oDipoGeneralRestInfo;
   t_S32 s32Error = oDipoGeneralRestInfo.s32GetData(cu8DipoGeneralRestrictionInfo);

   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadDipoGeneralRestrictionInfo: Read error occured - %u ", s32Error));
   }
   else
   {
      ETG_TRACE_USR1(("Datapool::bReadDipoGeneralRestrictionInfo() value %d:", cu8DipoGeneralRestrictionInfo));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDipoGeneralRestrictionInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteDipoGeneralRestrictionInfo(t_U16 u16DipoGeneralRestrictionInfo)
{
   ETG_TRACE_USR1(("Datapool::bWriteDipoGeneralRestrictionInfo() value %d:", u16DipoGeneralRestrictionInfo));
   dp_tDipoGeneralRestInfo oDipoGeneralRestInfo;
   t_U16 u16DipoGeneralRestInfo = 0;

   t_S32 s32Error = oDipoGeneralRestInfo.s32GetData(u16DipoGeneralRestInfo);

   if (DP_S32_NO_ERR == s32Error)
   {
      if (u16DipoGeneralRestrictionInfo != u16DipoGeneralRestInfo)
      {
         s32Error = oDipoGeneralRestInfo.s32SetData(u16DipoGeneralRestrictionInfo);
      }
   }
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteDipoGeneralRestrictionInfo: Write error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}
/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadDriveSideInfo()
 ***************************************************************************/
t_Bool Datapool::bReadDriveSideInfo(tenDriveSideInfo &rfenDriveSideInfo)
{
   dp_tSteeringWheelPos oSteeringWheelPos;
   tenSteeringWheelPosition enPosition;
   t_S32 s32Error = oSteeringWheelPos.s32GetData(enPosition);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadDriveSideInfo: Read error occured - %u ", s32Error));
   }
   else
   {
      rfenDriveSideInfo = (tenDriveSideInfo) enPosition;
      ETG_TRACE_USR2(("Datapool::bReadDriveSideInfo: value read  - %d ", rfenDriveSideInfo));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDriveSideInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteDriveSideInfo(const tenDriveSideInfo enDriveSideInfo)
{
   dp_tSteeringWheelPos oSteeringWheelPos;
   tenSteeringWheelPosition enCurData;
   t_S32 s32Error = oSteeringWheelPos.s32GetData(enCurData);
   if (DP_S32_NO_ERR == s32Error)
   {
      if (enDriveSideInfo != (tenDriveSideInfo) enCurData)
      {
         s32Error = oSteeringWheelPos.s32SetData((tenSteeringWheelPosition) enDriveSideInfo);
      }
   }
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWrieDiPODriveRestrictionInfo: Write error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadVehicleId()
 ***************************************************************************/
t_Bool Datapool::bReadVehicleId(t_String &szVehicleIdentifier)
{
   ETG_TRACE_USR1(("Datapool::bReadVehicleId: entered"));
   t_Char * szVehicleid = new char[MAX_VEHICLE_ID_STRLEN];
   dp_tVehicleId oVehicleId;
   t_S32 s32ReadDataLength = oVehicleId.s32GetData(szVehicleid, MAX_VEHICLE_ID_STRLEN);
   if (0 != s32ReadDataLength)
   {
      szVehicleIdentifier = szVehicleid;
      ETG_TRACE_USR1(("Datapool::bReadVehicleId: entered,length of Vehicle Id is :%d", szVehicleIdentifier.length()));
   }
   else
   {
      ETG_TRACE_ERR(("Datapool::bReadVehicleId: Read error occured "));
   }
   RELEASE_ARRAY_MEM(szVehicleid);
   return (0 != s32ReadDataLength);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteVehicleId()
 ***************************************************************************/
t_Bool Datapool::bWriteVehicleId(t_String &szVehicleIdentifier)
{
   ETG_TRACE_USR1(("Datapool::bWriteVehicleId: entered with VehicleId = %s \n", szVehicleIdentifier.c_str()));
   dp_tVehicleId oVehicleId;
   t_S32 s32WriteError = oVehicleId.s32SetData(const_cast<t_Char *>(szVehicleIdentifier.c_str()));

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteVehicleId: Write error = %u! \n", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteSelectionMode()
 ***************************************************************************/
t_Bool Datapool::bWriteSelectionMode(tenDeviceSelectionMode enSelectionMode)
{
   ETG_TRACE_USR1(("Datapool::bWriteSelectionMode: entered with enSelectionMode = %d \n", ETG_ENUM(SELECTION_MODE,
            enSelectionMode)));
   dp_tSelectionMode oSelectionMode;
   t_U8 u8SelectionMode = enSelectionMode;
   t_S32 s32Error = oSelectionMode.s32SetData(u8SelectionMode);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteSelectionMode: Write error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadSelectionMode()
 ***************************************************************************/
t_Bool Datapool::bReadSelectionMode(tenDeviceSelectionMode &enSelectionMode)
{
   dp_tSelectionMode oSelectionMode;
   t_U8 u8DeviceSelectionMode = 0;
   t_S32 s32Error = oSelectionMode.s32GetData(u8DeviceSelectionMode);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadSelectionMode: Read error occured - %u ", s32Error));
   }
   else
   {
      enSelectionMode = (tenDeviceSelectionMode) u8DeviceSelectionMode;
      ETG_TRACE_USR2(("Datapool::bReadSelectionMode: value read  - %d ", ETG_ENUM(SELECTION_MODE, enSelectionMode)));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteSPIVersion()
 ***************************************************************************/
t_Bool Datapool::bWriteSPIVersion(t_String szSPIVersion)
{
   ETG_TRACE_USR1(("Datapool::bWriteSPIVersion: entered with SPI Version = %s \n", szSPIVersion.c_str()));
   dp_tSPIVersion oSPIVersion;
   t_S32 s32WriteError = oSPIVersion.s32SetData(const_cast<t_Char *>(szSPIVersion.c_str()));

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("Datapool::bWriteSPIVersion: Write error occurred - %u ", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadSPIVersion()
 ***************************************************************************/
t_Bool Datapool::bReadSPIVersion(t_String &szSPIVersion)
{
   ETG_TRACE_USR1(("Datapool::bReadSPIVersion: entered"));
   dp_tSPIVersion oSPIVersion;
   t_Char szVersion[MAX_SPI_VERSION_STRLEN];
   t_S32 s32ReadDataLength = oSPIVersion.s32GetData(szVersion, MAX_SPI_VERSION_STRLEN);
   if (0 != s32ReadDataLength)
   {
      szSPIVersion = szVersion;
      ETG_TRACE_USR1(("Datapool::bReadSPIVersion: SPI Version is :%s", szSPIVersion.c_str()));
   }
   return (0 != s32ReadDataLength);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteVehicleManufacturerInformation()
 ***************************************************************************/
t_Bool Datapool::bWriteVehicleManufacturerInformation(
         const trVehicleManufacturerInformation& corfrVehicleManufacturerInformation)
{
   ETG_TRACE_USR1(("Datapool::bWriteVehicleManufacturerInformation() entered"));

   dp_tSPIVMInfo oVMInfo;
   trVMInfo rVehicleManufacturerInfo;

   //Values read from HMI are set in datapool
   vStringCopy(rVehicleManufacturerInfo.cVehicleMake, corfrVehicleManufacturerInformation.szVehicleMake);
   vStringCopy(rVehicleManufacturerInfo.cVehicleModel, corfrVehicleManufacturerInformation.szVehicleModel);
   vStringCopy(rVehicleManufacturerInfo.cVehicleModelYear, corfrVehicleManufacturerInformation.szVehicleModelYear);
   vStringCopy(rVehicleManufacturerInfo.cVehicleHUSwVersion, corfrVehicleManufacturerInformation.szVehicleHUSwVersion);
   vStringCopy(rVehicleManufacturerInfo.cVehicleHUHwVersion, corfrVehicleManufacturerInformation.szVehicleHUHwVersion);
   vStringCopy(rVehicleManufacturerInfo.cVehicleHUMake, corfrVehicleManufacturerInformation.szVehicleHUMake);
   vStringCopy(rVehicleManufacturerInfo.cVehicleHUModel, corfrVehicleManufacturerInformation.szVehicleHUModel);
   vStringCopy(rVehicleManufacturerInfo.cVehicleHUSwBuild, corfrVehicleManufacturerInformation.szVehicleHUSwBuild);

   rVehicleManufacturerInfo.enDriverPositionInfo =
            static_cast<tenDriverPositionInfo>(corfrVehicleManufacturerInformation.enDriverSideInfo);
   rVehicleManufacturerInfo.bIsNativeNavSupported = corfrVehicleManufacturerInformation.bIsNativeNavSupported;
   rVehicleManufacturerInfo.bIsRotaryCtrlSupported = corfrVehicleManufacturerInformation.bIsRotaryCtrlSupported;
   rVehicleManufacturerInfo.u16SupportedSPITechnologiesInfo =
            corfrVehicleManufacturerInformation.u16SupportedSPITechnologiesInfo;

   vStringCopy(rVehicleManufacturerInfo.rOEMIconDetails.cOEMName,
            corfrVehicleManufacturerInformation.rOEMIconDetails.szOEMName);
   vStringCopy(rVehicleManufacturerInfo.rOEMIconDetails.cOEMPath,
            corfrVehicleManufacturerInformation.rOEMIconDetails.szOEMPath);

   rVehicleManufacturerInfo.enFidelityType =
            static_cast<tenFidelityTouchType>(corfrVehicleManufacturerInformation.enFidelity);

   t_S32 s32Error = oVMInfo.s32SetData(rVehicleManufacturerInfo);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteVehicleManufacturerInformation: Error = %u", s32Error));
   } //if (DP_S32_NO_ERR != s32Error)

   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadVehicleManufacturerInformation()
 ***************************************************************************/
t_Bool Datapool::bReadVehicleManufacturerInformation(
         trVehicleManufacturerInformation& rfrVehicleManufacturerInformation)
{
   ETG_TRACE_USR1(("Datapool::bReadVehicleManufacturerInformation() entered"));

   dp_tSPIVMInfo oVMInfo;
   trVMInfo rVehicleManufacturerInfo;
   t_S32 s32Error = oVMInfo.s32GetData(rVehicleManufacturerInfo);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadVehicleManufacturerInformation: Read error occured - %u ", s32Error));
   } //if(DP_S32_NO_ERR != s32Error)
   else
   {
      rfrVehicleManufacturerInformation.szVehicleMake = rVehicleManufacturerInfo.cVehicleMake;
      rfrVehicleManufacturerInformation.szVehicleModel = rVehicleManufacturerInfo.cVehicleModel;
      rfrVehicleManufacturerInformation.szVehicleModelYear = rVehicleManufacturerInfo.cVehicleModelYear;
      rfrVehicleManufacturerInformation.szVehicleHUSwVersion = rVehicleManufacturerInfo.cVehicleHUSwVersion;
      rfrVehicleManufacturerInformation.szVehicleHUHwVersion = rVehicleManufacturerInfo.cVehicleHUHwVersion;
      rfrVehicleManufacturerInformation.szVehicleHUMake = rVehicleManufacturerInfo.cVehicleHUMake;
      rfrVehicleManufacturerInformation.szVehicleHUModel = rVehicleManufacturerInfo.cVehicleHUModel;
      rfrVehicleManufacturerInformation.szVehicleHUSwBuild = rVehicleManufacturerInfo.cVehicleHUSwBuild;
      rfrVehicleManufacturerInformation.enDriverSideInfo =
               (e8DP_LEFT_HAND_DRIVE == rVehicleManufacturerInfo.enDriverPositionInfo) ?
                        e8LEFT_HAND_DRIVE : e8RIGHT_HAND_DRIVE;
      rfrVehicleManufacturerInformation.bIsNativeNavSupported = rVehicleManufacturerInfo.bIsNativeNavSupported;
      rfrVehicleManufacturerInformation.bIsRotaryCtrlSupported = rVehicleManufacturerInfo.bIsRotaryCtrlSupported;
      rfrVehicleManufacturerInformation.u16SupportedSPITechnologiesInfo =
               rVehicleManufacturerInfo.u16SupportedSPITechnologiesInfo;
      rfrVehicleManufacturerInformation.rOEMIconDetails.szOEMName = rVehicleManufacturerInfo.rOEMIconDetails.cOEMName;
      rfrVehicleManufacturerInformation.rOEMIconDetails.szOEMPath = rVehicleManufacturerInfo.rOEMIconDetails.cOEMPath;
      rfrVehicleManufacturerInformation.enFidelity =
               (e8DP_LOW_FIDELITY == rVehicleManufacturerInfo.enFidelityType) ? e8LOW_FIDELITY : e8HIGH_FIDELITY;

   }

   return (DP_S32_NO_ERR == s32Error);

}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDisplayLastMode()
 ***************************************************************************/
t_Bool Datapool::bWriteDisplayLastMode(const t_Bool& corfbDispLastMode)
{
   ETG_TRACE_USR1(("Datapool::bWriteDisplayLastMode: entered with display last mode as projection: %d", ETG_ENUM(BOOL,
            corfbDispLastMode)));
   dp_tDisplayLastMode oDisplayLastMode;
   t_S32 s32WriteError = oDisplayLastMode.s32SetData(corfbDispLastMode);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("[ERR]::Datapool::bWriteDisplayLastMode: Write error occurred: %u ", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Void Datapool:vReadDisplayLastMode()
 ***************************************************************************/
t_Void Datapool::vReadDisplayLastMode(t_Bool& rbDispLastMode)
{
   ETG_TRACE_USR1(("Datapool::vReadDisplayLastMode: entered"));
   dp_tDisplayLastMode oDisplayLastMode;
   tBool bDispLastMode = false;
   t_S32 s32Error = oDisplayLastMode.s32GetData(bDispLastMode);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("[ERR]::Datapool::vReadDisplayLastMode: Read error occured - %u ", s32Error));
   }
   else
   {
      rbDispLastMode = bDispLastMode;
      ETG_TRACE_USR2(("[DESC]: Datapool::vReadDisplayLastMode: display last mode is projection: %d", ETG_ENUM(BOOL,
               rbDispLastMode)));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAudioLastMode()
 ***************************************************************************/
t_Bool Datapool::bWriteAudioLastMode(const t_Bool& corfbAudioLastMode)
{
   ETG_TRACE_USR1(("Datapool::bWriteAudioLastMode: entered with audio last mode as projection: %d", ETG_ENUM(BOOL,
            corfbAudioLastMode)));
   dp_tAudioLastMode oAudioLastMode;
   t_S32 s32WriteError = oAudioLastMode.s32SetData(corfbAudioLastMode);

   if (DP_S32_NO_ERR != s32WriteError)
   {
      ETG_TRACE_ERR(("[ERR]::Datapool::bWriteAudioLastMode: Write error occurred: %u", s32WriteError));
   }
   return (DP_S32_NO_ERR == s32WriteError);
}

/***************************************************************************
 ** FUNCTION:  t_Void Datapool::vReadAudioLastMode()
 ***************************************************************************/
t_Void Datapool::vReadAudioLastMode(t_Bool& rbAudioLastMode)
{
   ETG_TRACE_USR1(("Datapool::vReadAudioLastMode: entered"));
   dp_tAudioLastMode oAudioLastMode;
   tBool bAudioLastMode = false;
   t_S32 s32Error = oAudioLastMode.s32GetData(bAudioLastMode);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("[ERR]::Datapool::vReadAudioLastMode: Read error occured - %u ", s32Error));
   }
   else
   {
      rbAudioLastMode = bAudioLastMode;
      ETG_TRACE_USR2(("[DESC]: Datapool::vReadAudioLastMode: audio last mode is projection: %d", ETG_ENUM(BOOL,
               rbAudioLastMode)));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAndroidDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool Datapool::bWriteAndroidDeviceTechnologyPreference(const tenDeviceCategory coenTechnologyPreference)
{
   ETG_TRACE_USR1(("Datapool::bWriteAndroidDeviceTechnologyPreference: entered with enTechnologyPreference = %d \n", ETG_ENUM(DEVICE_CATEGORY,
            coenTechnologyPreference)));
   dp_tAdroidDeviceTechnologyPreference oAndroidDeviceTechnologyPreference;
   t_U8 u8DeviceCategory = coenTechnologyPreference;
   t_S32 s32Error = oAndroidDeviceTechnologyPreference.s32SetData(u8DeviceCategory);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteAndroidDeviceTechnologyPreference: Read error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAndroidDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool Datapool::bReadAndroidDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference)
{
   dp_tAdroidDeviceTechnologyPreference oAndroidDeviceTechnologyPreference;
   t_U8 u8DeviceCategory = 0;
   t_S32 s32Error = oAndroidDeviceTechnologyPreference.s32GetData(u8DeviceCategory);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadAndroidDeviceTechnologyPreference: Read error occured - %u ", s32Error));
   }
   else
   {
      enTechnologyPreference = (tenDeviceCategory) u8DeviceCategory;
      ETG_TRACE_USR2(("Datapool::bReadAndroidDeviceTechnologyPreference: value read  - %d ", ETG_ENUM(DEVICE_CATEGORY,
               enTechnologyPreference)));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAppleDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool Datapool::bWriteAppleDeviceTechnologyPreference(const tenDeviceCategory coenTechnologyPreference)
{
   ETG_TRACE_USR1(("Datapool::bWriteAppleDeviceTechnologyPreference: entered with enTechnologyPreference = %d \n", ETG_ENUM(DEVICE_CATEGORY,
            coenTechnologyPreference)));
   dp_tAppleDeviceTechnologyPreference oAppleDeviceTechnologyPreference;
   t_U8 u8DeviceCategory = coenTechnologyPreference;
   t_S32 s32Error = oAppleDeviceTechnologyPreference.s32SetData(u8DeviceCategory);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteAppleDeviceTechnologyPreference: Read error occured - %u ", s32Error));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAppleDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool Datapool::bReadAppleDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference)
{
   dp_tAppleDeviceTechnologyPreference oAppleDeviceTechnologyPreference;
   t_U8 u8DeviceCategory = 0;
   t_S32 s32Error = oAppleDeviceTechnologyPreference.s32GetData(u8DeviceCategory);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadAppleDeviceTechnologyPreference: Read error occured - %u ", s32Error));
   }
   else
   {
      enTechnologyPreference = (tenDeviceCategory) u8DeviceCategory;
      ETG_TRACE_USR2(("Datapool::bReadAppleDeviceTechnologyPreference: value read  - %d ", ETG_ENUM(DEVICE_CATEGORY,
               enTechnologyPreference)));
   }
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_String Datapool::szGetSoftwareVersion()
 ***************************************************************************/
t_String Datapool::szGetSoftwareVersion()
{
   t_U8  u8DataBuf[KDS_MAX_ENTRY_LENGTH];
   t_String szSwVersion;
   dp_tclregVersionsCustomerVersion oData;
   t_U8 u8Result = oData.u8GetData((char*)u8DataBuf, KDS_MAX_ENTRY_LENGTH);
   if (u8Result == DP_U8_ELEM_STATUS_VALID)
   {
      u8DataBuf[KDS_MAX_ENTRY_LENGTH-1] = 0; //string termination
      ETG_TRACE_USR4(("[PARAM]Datapool::szGetSoftwareVersion- Software Version = %s", u8DataBuf));
      szSwVersion = t_String((const char*)u8DataBuf);
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:Datapool: Failed to Read Software Version from datapool: error %d", u8Result));
   }

   return szSwVersion;

}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteHardKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteHardKeyInfo(const std::set<tenKeyCode>& corfrsetSuppHardKeys,
                                   tenDeviceCategory enDeviceCategory)

{
   ETG_TRACE_USR1(("Datapool::bWriteHardKeyInfo entered %d",ETG_ENUM(DEVICE_CATEGORY,enDeviceCategory)));
   t_Bool   bretVal = false;

   switch(enDeviceCategory)
   {
    case e8DEV_TYPE_ANDROIDAUTO:
         bretVal = bWriteAAHardKeyInfo(corfrsetSuppHardKeys);
         break;
    case e8DEV_TYPE_DIPO:
         bretVal = bWriteDiPoHardKeyInfo(corfrsetSuppHardKeys);
         break;
    default:
         ETG_TRACE_ERR(("Datapool::bWriteHardKeyInfo Category Not Supported"));
         break;
   }
   return bretVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteSoftKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteSoftKeyInfo(const std::set<tenKeyCode>& corfrsetSuppSoftKeys,
                                   tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("Datapool::bWriteSoftKeyInfo entered %d",ETG_ENUM(DEVICE_CATEGORY,enDeviceCategory)));
   t_Bool   bretVal = false;

   switch(enDeviceCategory)
   {
    case e8DEV_TYPE_DIPO:
         bretVal = bWriteDiPoSoftKeyInfo(corfrsetSuppSoftKeys);
         break;
    default:
         ETG_TRACE_ERR(("Datapool::bWriteSoftKeyInfo Category Not Supported"));
         break;
   }
   return bretVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteAAHardKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteAAHardKeyInfo(const std::set<tenKeyCode>& corfrsetAASuppHardKeys)
{
   ETG_TRACE_USR1(("Datapool::bWriteAAHardKeyInfo entered"));
   dp_tFiu32AASuppHardKeysArr oAASuppKeys;
   std::set<tenKeyCode> rfrsetSuppHardKeys;
   tU8 u8KeyCount = 0;
   tU32 u32arrAAKeys[scou8dpMaxKeys] = {0};
   t_U8 u8InitStatus = oAASuppKeys.u8GetElementStatus();

   bReadAAKeyCodeInfo(rfrsetSuppHardKeys);

   //If there are no elements in the datapool map[AndroidAuto]
   //should not have any elements.
   if(rfrsetSuppHardKeys.empty())
   {
      m_mapTechSuppHardkeys[e8DEV_TYPE_ANDROIDAUTO].clear();
   }

   // Handle Ignition Cycle case.On ignition cycle the map will be 
   // empty.If client registers already in previous ignition cycle
   // we need to append to the existing keys.
   else if ((m_mapTechSuppHardkeys[e8DEV_TYPE_ANDROIDAUTO].empty()) &&
       (u8InitStatus != DP_U8_ELEM_STATUS_INITVAL))
   {
      m_mapTechSuppHardkeys[e8DEV_TYPE_ANDROIDAUTO].insert
                (rfrsetSuppHardKeys.begin(),rfrsetSuppHardKeys.end());
   }

   m_mapTechSuppHardkeys[e8DEV_TYPE_ANDROIDAUTO].insert
               (corfrsetAASuppHardKeys.begin(),corfrsetAASuppHardKeys.end());

   for (auto enKeys : m_mapTechSuppHardkeys[e8DEV_TYPE_ANDROIDAUTO])
   {
     if(u8KeyCount < scou8dpMaxKeys)
     {
       u32arrAAKeys[u8KeyCount]  = static_cast<tU32>(enKeys);
       u8KeyCount++;
     }
     else
     {
       ETG_TRACE_ERR(("Datapool::bWriteAAHardKeyCodeInfo:Key limit has reached"));
     }
   }

   t_S32 s32Error = oAASuppKeys.s32SetData(u32arrAAKeys,scou8dpMaxKeys);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteAAHardKeyCodeInfo: Error = %u", s32Error));
   }//if (DP_S32_NO_ERR != s32Error)
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDiPoHardKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteDiPoHardKeyInfo(const std::set<tenKeyCode>& corfrsetDiPoSuppKeys)
{
   ETG_TRACE_USR1(("Datapool::bWriteDiPoHardKeyInfo entered"));
   dp_tFiu32DiPoSuppHardKeysArr oDiPoSuppKeys;
   tU8 u8KeyCount = 0;
   std::set<tenKeyCode> rfrsetSuppHardKeys;
   tU32 u32arrDiPoKeys[scou8dpMaxKeys] = {0};
   tU8 u8InitStatus = oDiPoSuppKeys.u8GetElementStatus();

   bReadDiPoKeyCodeInfo(rfrsetSuppHardKeys);

   //If there are no elements in the datapool map[Dipo] should not have any elements.
   if(rfrsetSuppHardKeys.empty())
   {
      m_mapTechSuppHardkeys[e8DEV_TYPE_DIPO].clear();
   }

   // Handle Ignition Cycle case.On ignition cycle the map will be 
   // empty.If client registers already in previous ignition cycle
   // we need to append to the existing keys.
   else if ((m_mapTechSuppHardkeys[e8DEV_TYPE_DIPO].empty()) &&
       (u8InitStatus != DP_U8_ELEM_STATUS_INITVAL))
   {
      m_mapTechSuppHardkeys[e8DEV_TYPE_DIPO].insert
                (rfrsetSuppHardKeys.begin(),rfrsetSuppHardKeys.end());
   }

   m_mapTechSuppHardkeys[e8DEV_TYPE_DIPO].insert(corfrsetDiPoSuppKeys.begin(),corfrsetDiPoSuppKeys.end());
   for (auto enKeys : m_mapTechSuppHardkeys[e8DEV_TYPE_DIPO])
   {
      if(u8KeyCount < scou8dpMaxKeys)
      {
         u32arrDiPoKeys[u8KeyCount]  = static_cast<tU32>(enKeys);
         u8KeyCount++;
      }
      else
      {
         ETG_TRACE_ERR(("Datapool::bWriteDiPoHardKeyCodeInfo:Key limit has reached"));
      }
   }
   t_S32 s32Error = oDiPoSuppKeys.s32SetData(u32arrDiPoKeys,scou8dpMaxKeys);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteDiPoHardKeyCodeInfo: Error = %u", s32Error));
   }//if (DP_S32_NO_ERR != s32Error)
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bWriteDiPoSoftKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bWriteDiPoSoftKeyInfo(const std::set<tenKeyCode>& corfrsetDiPoSuppSoftKeys)
{
   ETG_TRACE_USR1(("Datapool::bWriteDiPoSoftKeyInfo entered"));
   dp_tFiu32DiPoSuppSoftKeysArr oDiPoSuppKeys;
   tU8 u8KeyCount = 0;
   tU32 u32arrDiPoKeys[scou8dpMaxKeys] = {0};
   std::set<tenKeyCode> rfrsetSuppSoftKeys;
   tU8 u8InitStatus = oDiPoSuppKeys.u8GetElementStatus();

   bReadDiPoSoftKeyInfo(rfrsetSuppSoftKeys);

   //If there are no elements in the datapool softmap[Dipo] should not have any elements.
   if(rfrsetSuppSoftKeys.empty())
   {
      m_mapTechSuppSoftkeys[e8DEV_TYPE_DIPO].clear();
   }

   // Handle Ignition Cycle case.On ignition cycle the map will be 
   // empty.If client registers already in previous ignition cycle
   // we need to append to the existing keys.
   else if ((m_mapTechSuppSoftkeys[e8DEV_TYPE_DIPO].empty()) &&
       (u8InitStatus != DP_U8_ELEM_STATUS_INITVAL))
   {
      m_mapTechSuppSoftkeys[e8DEV_TYPE_DIPO].insert
                (rfrsetSuppSoftKeys.begin(),rfrsetSuppSoftKeys.end());
   }

   m_mapTechSuppSoftkeys[e8DEV_TYPE_DIPO].insert(corfrsetDiPoSuppSoftKeys.begin(),corfrsetDiPoSuppSoftKeys.end());

   for (auto enKeys : m_mapTechSuppSoftkeys[e8DEV_TYPE_DIPO])
   {
      if(u8KeyCount < scou8dpMaxKeys)
      {
        u32arrDiPoKeys[u8KeyCount]  = static_cast<tU32>(enKeys);
        u8KeyCount++;
      }
      else
      {
        ETG_TRACE_ERR(("Datapool::bWriteDiPoSoftKeyInfo:Key limit has reached"));
      }
   }
   t_S32 s32Error = oDiPoSuppKeys.s32SetData(u32arrDiPoKeys,scou8dpMaxKeys);
   if (DP_S32_NO_ERR != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bWriteDiPoSoftKeyInfo: Error = %u", s32Error));
   }//if (DP_S32_NO_ERR != s32Error)
   return (DP_S32_NO_ERR == s32Error);
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadKeyCodeInfo()
 ***************************************************************************/
t_Bool Datapool::bReadKeyCodeInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{
   ETG_TRACE_USR1(("Datapool::bReadKeyCodeInfo entered %d",ETG_ENUM(DEVICE_CATEGORY,enCategory)));
   t_Bool   bretVal = false;

   switch(enCategory)
   {
    case e8DEV_TYPE_ANDROIDAUTO:
         bretVal = bReadAAKeyCodeInfo(rfrsetKeyCodes);
         break;
    case e8DEV_TYPE_DIPO:
         bretVal = bReadDiPoKeyCodeInfo(rfrsetKeyCodes);
         break;
    default:
         ETG_TRACE_ERR(("Datapool::bReadKeyCodeInfo Category Not Supported"));
         break;
   }
   return bretVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadSoftKeyInfo()
 ***************************************************************************/
t_Bool Datapool::bReadSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{
   ETG_TRACE_USR1(("Datapool::bReadSoftKeyInfo entered %d",ETG_ENUM(DEVICE_CATEGORY,enCategory)));
   t_Bool   bretVal = false;

   switch(enCategory)
   {
    case e8DEV_TYPE_DIPO:
         bretVal = bReadDiPoSoftKeyInfo(rfrsetKeyCodes);
         break;
    default:
         ETG_TRACE_ERR(("Datapool::bReadSoftKeyInfo Category Not Supported"));
         break;
   }
   return bretVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool Datapool::bReadAAKeyCodeInfo()
***************************************************************************/
t_Bool Datapool::bReadAAKeyCodeInfo(std::set<tenKeyCode>& rfrsetAASuppKeys)
{
   ETG_TRACE_USR1(("Datapool::bReadAAKeyCodeInfo entered"));
   dp_tFiu32AASuppHardKeysArr oSuppKeys;
   tU32 u32arrAAKeys[scou8dpMaxKeys] = {0};
   t_Bool  bretVal = true;
   t_S32 s32Error = oSuppKeys.s32GetData(u32arrAAKeys,scou8dpMaxKeys);
   t_U8 u8InitStatus = oSuppKeys.u8GetElementStatus();

   if(u8InitStatus == DP_U8_ELEM_STATUS_INITVAL)
   {
      bretVal = false;
   }
   else if (DP_S32_NO_ERR != s32Error && scou8dpMaxKeys != s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadKeyAACodeInfo: Read error occured - %u ", s32Error));
      bretVal = false;
   }//if(DP_S32_NO_ERR != s32Error)
   else
   {
      for(tU8 u8KeyCount = 0;u8KeyCount<scou8dpMaxKeys;u8KeyCount++)
      {
         if( 0 == u32arrAAKeys[u8KeyCount])
         {
            continue;
         }
         else
         {
            rfrsetAASuppKeys.insert(static_cast<tenKeyCode>(u32arrAAKeys[u8KeyCount]));
         }
      }
   }
   return bretVal;
}
/***************************************************************************
** FUNCTION:  Datapool::bReadDiPoKeyCodeInfo()
***************************************************************************/
t_Bool Datapool::bReadDiPoKeyCodeInfo(std::set<tenKeyCode>& rfrsetDiPoSuppKeys)
{
   ETG_TRACE_USR1(("Datapool::bDiPoReadKeyCodeInfo entered"));
   dp_tFiu32DiPoSuppHardKeysArr oSuppKeys;
   tU32 u32arrDiPoKeys[scou8dpMaxKeys] = {0};
   t_Bool  bretVal = true;
   t_S32 s32Error = oSuppKeys.s32GetData(u32arrDiPoKeys,scou8dpMaxKeys);
   t_U8 u8InitStatus = oSuppKeys.u8GetElementStatus();
   if(u8InitStatus == DP_U8_ELEM_STATUS_INITVAL)
   {
      bretVal = false;
   }
   else if (DP_S32_NO_ERR != s32Error && scou8dpMaxKeys!=s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadDiPoKeyCodeInfo: Read error occured - %u ", s32Error));
      bretVal = false;
   }//if(DP_S32_NO_ERR != s32Error)
   else
   {
      for(tU8 u8KeyCount = 0;u8KeyCount<scou8dpMaxKeys;u8KeyCount++)
      {
         if( 0 == u32arrDiPoKeys[u8KeyCount])
         {
            continue;
         }
         else
         {
            rfrsetDiPoSuppKeys.insert(static_cast<tenKeyCode>(u32arrDiPoKeys[u8KeyCount]));
         }
      }
   }
   return bretVal;
}
/***************************************************************************
** FUNCTION:  Datapool::bReadDiPoSoftKeyInfo()
***************************************************************************/
t_Bool Datapool::bReadDiPoSoftKeyInfo(std::set<tenKeyCode>& rfrsetDiPoSuppSoftKeys)
{
   ETG_TRACE_USR1(("Datapool::bReadDiPoSoftKeyInfo entered"));
   dp_tFiu32DiPoSuppSoftKeysArr oSuppKeys;
   tU32 u32arrDiPoKeys[scou8dpMaxKeys] = {0};
   t_Bool  bretVal = true;
   t_S32 s32Error = oSuppKeys.s32GetData(u32arrDiPoKeys,scou8dpMaxKeys);
   t_U8 u8InitStatus = oSuppKeys.u8GetElementStatus();
   if(u8InitStatus == DP_U8_ELEM_STATUS_INITVAL)
   {
      bretVal = false;
   }
   else if (DP_S32_NO_ERR != s32Error && scou8dpMaxKeys!=s32Error)
   {
      ETG_TRACE_ERR(("Datapool::bReadDiPoKeyCodeInfo: Read error occured - %u ", s32Error));
      bretVal = false;
   }//if(DP_S32_NO_ERR != s32Error)
   else
   {
      for(tU8 u8KeyCount = 0;u8KeyCount<scou8dpMaxKeys;u8KeyCount++)
      {
         if( 0 == u32arrDiPoKeys[u8KeyCount])
         {
            continue;
         }
         else
         {
            rfrsetDiPoSuppSoftKeys.insert(static_cast<tenKeyCode>(u32arrDiPoKeys[u8KeyCount]));
         }
      }
   }
   return bretVal;
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>
