  /*!
 *******************************************************************************
 * \file         spi_tclWiFi.cpp
 * \brief        Wifi Manager class that provides interface to delegate
                 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Wifi Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
10.02.2017  |  Unmukt Jain (RBEI/ECP2)          | Initial Version
 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLWIFI_H_
#define _SPI_TCLWIFI_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "BaseTypes.h"
#include "spi_WifiTypedefs.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclWifiRespIntf.h"
#include "spi_tclExtRespWiFi.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclExtRespOOBT.h"
#include "spi_tclWiFiSettingsIntf.h"
#include "Lock.h"
#include <set>

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/
class spi_tclWiFiDevBase;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/* Forward Declarations. */
class spi_tclExtCmdWiFiIntf;
/*!
* \class spi_tclWiFi
* \brief This is the main Wifi Manager class that provides interface to delegate
*        the execution of command and handle response.
*/
class spi_tclWiFi : public spi_tclLifeCycleIntf, public spi_tclExtRespWiFi, public spi_tclSelectionIntf, public spi_tclExtRespOOBT
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclWiFi::spi_tclWiFi()
   ***************************************************************************/
   /*!
   * \fn      spi_tclWiFi()
   * \brief   Default Constructor
   ***************************************************************************/
	explicit spi_tclWiFi(spi_tclWifiRespIntf *poWifiRespIntf);

   /***************************************************************************
   ** FUNCTION:  spi_tclWiFi::~spi_tclWiFi();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclWiFi()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclWiFi();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclWiFi::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to Initialize
    * \sa      bUnInitialize()
    **************************************************************************/
   virtual t_Bool bInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclWiFi::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
    * \brief   Method to UnInitialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bUnInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclWiFi::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
    * \sa      vSaveSettings()
    **************************************************************************/
   virtual t_Void vLoadSettings();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vSaveSettings()
    ***************************************************************************/
   /*!
    * \fn      vSaveSettings()
    * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
    * \sa      vLoadSettings()
    **************************************************************************/
   virtual t_Void vSaveSettings();
   
   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vInitiateWirelessDiscovery()
    ***************************************************************************/
   /*!
    * \fn      vInitiateWirelessDiscovery()
    * \brief   vInitiateWirelessDiscovery Method. Invoked by HMI to start CPW session.
    **************************************************************************/
   t_Void vInitiateWirelessDiscovery(tenDeviceCategory enDevCategory, const t_String& coszBTMACAddress = "");

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vGetWiFiConfig()
    ***************************************************************************/
   /*!
    * \fn      vGetWiFiConfig()
    * \brief   vGetWiFiConfig Method. Method to fetch WiFi Config.
    * \param   rfrWiFiConfig[OUT]: trWiFiConfig structure type
    **************************************************************************/
   t_Void vGetWiFiConfig(trWiFiConfig &rfrWiFiConfig);
   
    /***************************************************************************
    ** FUNCTION:  spi_tclWiFi::vSelectDevice
    ***************************************************************************/
   /*!
    * \fn     vSelectDevice
    * \brief  Device selection request
    * \param  corfrSelectReq : Device selection request details
    **************************************************************************/
   t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclWiFi::vSelectDeviceResult
    ***************************************************************************/
   /*!
    * \fn     t_Void vSelectDeviceResult
    * \brief  device selection result
    * \param  corfrSelectReq : Device selection result details
    * \param  enErrorCode : Error code
    **************************************************************************/
   t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

   /***************************************************************************
    ** FUNCTION:  spi_tclWiFi::vUpdateSelectionState
    ***************************************************************************/
   /*!
    * \fn     vUpdateSelectionState
    * \brief  Informs if the device selection is not initiated for a reported device
    * \param  cou32DeviceHandle: Device handle
    **************************************************************************/
   t_Void  vSetDevSelectorBusyStatus(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enDevSelectionState);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFi::vGetWiFiCredentials()
   ***************************************************************************/
   /*!
	* \fn      t_Void vGetWiFiCredentials(tenDeviceCategory enDevCategory,,const trUserContext& corfrUsrCntxt)
    * \brief   Method to get the WiFi credentials to share with a device
    * \param   szBTMACAddress : BTMACAddress
    * \param  [IN] corfrUsrCntxt: User Context Details.
    * \retval  t_Void
    **************************************************************************/	 
	t_Void vGetWiFiCredentials(t_String szBTMACAddress,const trUserContext& corfrUsrCntxt);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFi::vSendWiFiCredentials()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSendWiFiCredentials(t_String szBTMACAddress)
   * \brief   To send WiFi credentials to registered clients after User consent.
   * \param  [IN] szBTMACAddress: Indicates Unique ID of the device.
   * \sa      None
   **************************************************************************/  
   t_Void vSendWiFiCredentails(t_String szBTMACAddress);

   /***************************************************************************
   ** FUNCTION:  spi_tclExtCmdWiFiIntf spi_tclWiFi::poFetchExtCmdWiFiIntf()
   ***************************************************************************/
   /*!
   * \fn      poFetchExtCmdWiFiIntf()
   * \brief   Gets the poFetchExtCmdWiFiIntf pointer from ExtCompManager
   * \retval  None
   **************************************************************************/
   spi_tclExtCmdWiFiIntf* poFetchExtCmdWiFiIntf();


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFi::vHandleOOBTPairing()
   ***************************************************************************/
   /*!
   * \fn      t_Void vHandleOOBTPairing(t_String szBTMACAddress)
   * \brief   To send PrepareSetup
   * \sa      None
   **************************************************************************/
   t_Void vHandleOOBTComplete();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFi::vSetWiFiSettingsInstance(spi_tclWiFiSettingsIntf* poWiFiSettingsIntf)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetWiFiSettingsInstance()
   * \brief   Method to set Wifi Settings instance.
   * \param   poWiFiSettingsIntf : pointer to WiFiSettingsIntf.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetWiFiSettingsInstance(spi_tclWiFiSettingsIntf* poWiFiSettingsIntf);
   
   /***************************************************************************
    ** FUNCTION:  spi_tclWiFi::vRestoreSettings
    ***************************************************************************/
   /*!
    * \fn     t_Void vRestoreSettings()
    * \brief  vRestoreSettings Method. Invoked on factory reset to clear the Private Data
    * \param  None
    **************************************************************************/
   t_Void vRestoreSettings();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFiSetupHndlr::vSetUserDeselect()
    ***************************************************************************/
   /*!
    * \fn      vSetUserDeselect(t_String szBTMACAddress, t_Bool bUserDeslect)
    * \param   szBTMACAddress : [IN] BTMACAddress
    * \param   bUserDeslect: [IN] - UserDeselect falg
    **************************************************************************/
   t_Void vSetUserDeselect(const t_String& corszBTMACAddress, t_Bool bUserDeslect);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFiSetupHndlr::vSetExchangeCredentails()
    ***************************************************************************/
   /*!
    * \fn      vSetExchangeCredentails(t_Bool bSetExchangeCredentails)
    * \param   bSetExchangeCredentails: [IN] - Credentials Exchange falg
    **************************************************************************/
   t_Void vSetExchangeCredentails(t_Bool bSetExchangeCredentails);

   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiSetupHndlr::vOnConfigureWiFiResult(...)
   ***************************************************************************/
   /*!
    * \fn      t_Void vPostConfigureWiFiResult()
    * \brief   Called when ConfigureWiFi operation result is sent by WBL Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN] bConfigureWiFiResult: Result of ConfigureWiFi operation
    * \sa      None
    **************************************************************************/
   t_Void vPostConfigureWiFiResult(t_Bool bConfigureWiFiResult);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWiFiSetupHndlr::vOnWiFiConfig(...)
   ***************************************************************************/
   /*!
    * \fn      t_Void vPostWiFiConfig()
    * \brief   Called when ActiveWifiSetup structure is sent by WBL Client Handler.
    *          Mandatory interface to be implemented.
    * \param   [IN] rfcorWiFiConfig: structure of type trWiFiConfig
	* \param   [IN] corvecStationsInfo: List of connected stations.
    * \sa      None
    **************************************************************************/
   t_Void vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo);
   
   /***************************************************************************
    ** FUNCTION:  virtual t_Void spi_tclWiFi::vPostOOBTComplete()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostOOBTComplete()
    * \brief   Called when Conflicts Trigger property is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostOOBTComplete();

   	 

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclWiFi(const spi_tclWiFi &rhs)
    ***************************************************************************/
   /*!
    * \fn      spi_tclBluetooth(const spi_tclWiFi &rhs)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclWiFi(const spi_tclWiFi &rhs);

   /***************************************************************************
    ** FUNCTION: const spi_tclWiFi & operator=(const spi_tclWiFi &rhs);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclWiFi & operator=(const spi_tclWiFi &rhs);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclWiFi & operator=(const spi_tclWiFi &rhs);
   
   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclWiFi::bIsCarPlayWirelessAPSetUp()
    ***************************************************************************/
   /*!
    * \fn      t_Bool bIsCarPlayWirelessAPSetUp()
    * \brief   Method to check whether CarPlay Wirelss AP is setup
	* \retval  t_Bool
    **************************************************************************/ 
   t_Bool bIsCarPlayWirelessAPSetUp();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vSendSelectDeviceResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSendSelectDeviceResult()
    * \brief   Method to send the select device result
	* \param   enErrorCode : [IN] Response code
	* \retval  t_Void
    **************************************************************************/    
   t_Void vSendSelectDeviceResult(tenErrorCode enErrorCode);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vSendInitiateWirelessDiscoveryResult()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSendInitiateWirelessDiscoveryResult(t_Bool bSuccess)
    * \brief   Method to send the InitiateWirelessDiscovery result to HMI
	* \param   bSuccess : [IN] Success /Failure
	* \retval  t_Bool
    **************************************************************************/     
   t_Void vSendInitiateWirelessDiscoveryResult(t_Bool bSuccess);
 

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vHandleDeviceDeselection()
    ***************************************************************************/
   /*!
    * \fn      t_Void vHandleDeviceDeselection(t_U32 u32DevID,tenDeviceCategory enDevCat,
	*                                          tenDeviceConnectionType enDeviceConnType)
    * \brief   Method to handle device deselection requests
	* \param   u32DevID : [IN] Device ID
	* \param   enDevCat : [IN] Device Category
	* \retval  t_Void
    **************************************************************************/ 
    t_Void vHandleDeviceDeselection(t_U32 u32DevID,tenDeviceCategory enDevCat,tenDeviceConnectionType enDeviceConnType);

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclWiFi::bHandleInitiateWirelessDiscovery()
    ***************************************************************************/
   /*!
    * \fn      t_Bool bHandleInitiateWirelessDiscovery()
    * \brief   Method to Handle the InitiateWirelessDiscovery
    * \Param   enDevCategory - Device Category to differentiate for Handling Differently for other Technologies
    * \Param   coszBTMACAddress - BTMAC address of the Device
    **************************************************************************/
   t_Bool bHandleInitiateWirelessDiscovery(tenDeviceCategory enDevCategory, const t_String& coszBTMACAddress);

   /***************************************************************************
    ** FUNCTION:  spi_tclWiFi::vRegisterCallbacks
    ***************************************************************************/
   /*!
    * \fn     vRegisterCallbacks
    * \brief  Registers for callbacks to AAP/DiPo classes
    **************************************************************************/
   t_Void vRegisterCallbacks();

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclWiFi::bValidateClient
    ***************************************************************************/
   /*!
    * \fn     bValidateClient
    * \brief  Validates the index whether it is within the range or not
    * \param  u8Index [IN]: Index
    **************************************************************************/
   t_Bool bValidateClient(t_U8 u8Index);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vDeselectionOnDisconnection
    ***************************************************************************/
   /*!
    * \fn     vDeselectionOnDisconnection
    * \brief  Handles Device Deselection
    * \param  cou32DeviceHandle [IN]: Device handle
    * \param  enSelectionReason : Selection Reason for Deselection
    **************************************************************************/
   t_Void vDeselectionOnDisconnection(const t_U32 cou32DeviceHandle, tenSelectReason enSelectionReason);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWiFi::vHandleDeviceSelection
    ***************************************************************************/
   /*!
    * \fn     vHandleDeviceSelection
    * \brief  Handles Device Selection
    * \param  cou32DeviceHandle [IN]: Device handle
    **************************************************************************/
   t_Void vHandleDeviceSelection(const t_U32 cou32DeviceHandle);

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclWiFi::bIsPreferredFrequency()
    ***************************************************************************/
   /*!
    * \fn      t_Bool bIsPreferredFrequency()
    * \brief   Method to check whether CarPlay Wirelss AP is setup with Preferred Frequency
    * \retval  t_Bool
    **************************************************************************/
   t_Bool bIsPreferredFrequency();

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   //!Wifi Response Interface pointer
   spi_tclWifiRespIntf    *m_poWifiRespIntf;

   //!structure to hold WiFi Config
   trWiFiConfig m_rWiFiConfig;
   
   t_Bool m_bSelectDeviceRespPending;
   
   t_Bool m_bInitiateWirelessDiscoveryRespPending;
   
   t_String m_szWiFiCredentialExchangePendingForDevice;
   
   trUserContext m_GetWiFiCredentialsContext;

   spi_tclWiFiDevBase* m_poWiFiDevBase[e8DEV_TYPE_MAX]; 
   
   t_Bool m_bIsAPPoweredOn;

   spi_tclWiFiSettingsIntf* m_poWiFiSettings;

   //! Initiate Wireless Discovery Request Pending
   t_Bool m_bInitiateWirelessDiscoveryRequestPending;

   //!Set that stores the Device BT Mac with which Credentials are already exchanged
   std::set<t_String> m_setAccessoryWifiInfoExchangedDevices;

   Lock m_CredentialsExchangeLock;

   //! Share the credentials on the PREACTIVATED and ACTIVATING
    t_Bool m_bExchangeCredentials;

    tenWifiFrequency m_enPreferredFrequency;

   /***************************************************************************
   ******************************END OF PRIVATE********************************
   ***************************************************************************/

};

#endif // _SPI_TCLWIFI_H_

