/*!
 *******************************************************************************
 * \file             spi_WifiTypedefs.h
 * \brief            Type defines used by Wifi classes
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Type defines used by Wifi classes
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
01.12.2016  |  Unmukt Jain (RBEI/ECP2)          | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef _SPI_WIFITYPEDEFS_H_
#define _SPI_WIFITYPEDEFS_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <string>
#include <vector>
#include <functional> 
#include "SPITypes.h"

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_String scoszTechnology_CPW = "CarPlayWireless";
static const t_String scoszTechnology_AAW = "AndroidAutoWireless";
static const t_String scoszTechnology_Default = "Normal";
static const t_String scoszFrequencyBand_2_4Ghz = "2.4 GHz";
static const t_String scoszFrequencyBand_5Ghz = "5 GHz";

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

typedef enum
{
   e8_Unknown = 0,
   e8_2_dot_4_GHz = 1,
   e8_5_dot_0_GHz = 2
}tenWifiFrequency;

typedef enum
{
    e8_NONE = 0,
    e8_WEP  = 1,
    e8_WPA  = 2,
    e8_WPA2 = 3
}tenWifiSecurityType;

typedef enum
{
   e8_TECHNOLOGY_UNKNOWN = 0,
   e8_TECHNOLOGY_ALLOWED = 1,
   e8_TECHNOLOGY_NOTALLOWED = 2
}tenTechnologyStatus;


typedef std::map<t_U16,t_U16> tMapSupportedChannels;
typedef std::map<tenDeviceCategory,tenTechnologyStatus> tMapSupportedTechnologies;

/* structure to store supported frequency info **/
struct trFrequencyDetails
{
   tMapSupportedChannels mapSuppChannels;
   tMapSupportedTechnologies mapSuppTechnologies;
};

typedef std::map<tenWifiFrequency,trFrequencyDetails> tMapSupportedFrequencies;
typedef std::pair<tenWifiFrequency,trFrequencyDetails> tPairSupportedFrequencies;

/* structure to provide information of a STA connected to HU WiFi AP*/
struct trStationInfo
{
      t_String szIPAddress;
      t_String szDeviceName;
      t_String szMacAddress;

      trStationInfo():szIPAddress(),szDeviceName(), szMacAddress(){}

      trStationInfo& operator=(const trStationInfo& corfrStationInfo)
      {
         szIPAddress = corfrStationInfo.szIPAddress;
         szDeviceName = corfrStationInfo.szDeviceName;
         szMacAddress = corfrStationInfo.szMacAddress;
         return *this;
      }

};

struct trWiFiConfig
{
    t_String szSSID;
    t_String szPassphrase;
    t_String szInterface;
    t_String szMode;
    t_String szType;
    t_U32 u32CurrentOperatingChannel;
    tenWifiFrequency enFrequency;
    tenWifiSecurityType enSecurity;
    t_Bool bVisible;
    t_Bool bPowered;

    trWiFiConfig():   u32CurrentOperatingChannel(0),
                      enFrequency(tenWifiFrequency::e8_Unknown),
                      enSecurity(tenWifiSecurityType::e8_NONE),
                      bVisible(false),
                      bPowered(false)
    {

    }
};

//! Callback signatures definition
typedef std::function<t_Void(t_Bool)> vOnConfigureWiFiResult;
typedef std::function<t_Void(const trWiFiConfig &, const std::vector<trStationInfo>&)> vOnWiFiConfig;
typedef std::function<t_Void(const tMapSupportedFrequencies&)> vOnSupportedFrequencyUpdate;

typedef std::function<t_Void(const t_U32, tenSelectReason)> tfvDeviceDeselectionOnDisconnection;
typedef std::function<t_Void(const t_U32)> tfvDeviceSelectionOnConnection;
//! \brief   Structure holding the callbacks to WiFi Manager. Used by WiFi Client to notify
//!          WiFi Manager about WiFi Source events.
struct trWiFiCallbacks
{
      //! Called with result of ConfigureWifi method
      vOnConfigureWiFiResult    fvOnConfigureWiFiResult;

      //! Called with WifiAPConfig structure
      vOnWiFiConfig fvOnWiFiConfig;

      tfvDeviceDeselectionOnDisconnection fvDeviceDeselectionOnDisconnection;
      tfvDeviceSelectionOnConnection fvDeviceSelectionOnConnection;
      //! Called to indicate update in supported frequencies
      vOnSupportedFrequencyUpdate fvOnSupportedFrequencies;

      trWiFiCallbacks() :
         fvOnConfigureWiFiResult(NULL),
         fvOnWiFiConfig(NULL),
         fvDeviceDeselectionOnDisconnection(NULL),
         fvDeviceSelectionOnConnection(NULL),
         fvOnSupportedFrequencies(NULL)
      {
      }
};


struct trWiFiAPConfig
{
	t_String szSSID;
    t_String szPassphrase;	
    t_U16 u16OperatingChannel;
    tenWifiSecurityType enSecurity;	
	
	trWiFiAPConfig(): u16OperatingChannel(0),
                      enSecurity(tenWifiSecurityType::e8_NONE){}
};

#endif //_SPI_WIFITYPEDEFS_H_
