/*!
 *******************************************************************************
 * \file              spi_tclInputHandler.h
 * \brief             Main SPI input handler 
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Main input handler that creates ML and DiPo Input handlers
 AUTHOR:        Hari Priya E R (RBEI/ECP2)
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 25.11.2013 |  Hari Priya E R              | Initial Version
 12.03.2014 |  Hari Priya E R              | Included changes for handling ML and DiPO 
                                             Input handlers based on Device Type
 06.04.2014 |  Ramya Murthy                | Initialisation sequence implementation
 03.07.2014 |  Hari Priya E R              | Added changes related to Input Response interface
 31.07.2014 |  Ramya Murthy                | SPI feature configuration via LoadSettings()
 13.03.2015 |  Sameer Chandra		       | SelectDevice Implementation for AAP.
 25.06.2015 |  Sameer Chandra              | Added ML XDeviceKey Support for PSA
 15.07.2015 |  Sameer Chandra              | Added new method vProcessKnobKeyEvents

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLINPUTHANDLER_H_
#define SPI_TCLINPUTHANDLER_H_

/******************************************************************************
 | includes:
 | 1)RealVNC sdk - includes
 | 2)Typedefines
 |----------------------------------------------------------------------------*/
#include <set>
#include "SPITypes.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclInputRespIntf.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclInputHandlerSettingsIntf.h"


/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

class spi_tclInputDevBase;

/*!
 * \class spi_tclInputHandler
 * \brief Main input handler that creates ML and DiPo Input handlers
 *
 */

class spi_tclInputHandler :public spi_tclInputRespIntf, public spi_tclLifeCycleIntf, public spi_tclSelectionIntf
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION: spi_tclInputHandler::spi_tclInputHandler(spi_tclInputRespIntf* ..)
   ***************************************************************************/
   /*!
   * \fn      spi_tclInputHandler(spi_tclInputRespIntf* poInputRespIntf)
   * \brief   Parameterised constructor
   * \param   poInputRespIntf: [IN]Input Response Interface Pointer
   * \sa      ~spi_tclInputHandler()
   **************************************************************************/
   spi_tclInputHandler(spi_tclInputRespIntf* poInputRespIntf);

   /***************************************************************************
   ** FUNCTION:  spi_tclInputHandler::~spi_tclInputHandler()
   ***************************************************************************/
   /*!
   * \fn      spi_tclInputHandler()
   * \brief   destructor
   * \sa      spi_tclInputHandler()
   **************************************************************************/
   ~spi_tclInputHandler();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclInputHandler::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      bInitialize()
   * \brief   Method to Initialize
   * \sa      bUnInitialize()
   **************************************************************************/
   virtual t_Bool bInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclInputHandler::bUnInitialize()
   ***************************************************************************/
   /*!
   * \fn      bUnInitialize()
   * \brief   Method to UnInitialize
   * \sa      bInitialize()
   **************************************************************************/
   virtual t_Bool bUnInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclInputHandler::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
    * \sa      vSaveSettings()
    **************************************************************************/
   virtual t_Void vLoadSettings();

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclInputHandler::vSaveSettings()
   ***************************************************************************/
   /*!
   * \fn      vSaveSettings()
   * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
   * \sa      vLoadSettings()
   **************************************************************************/
   virtual t_Void vSaveSettings();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vProcessTouchEvent
   ***************************************************************************/
   /*!
   * \fn      vProcessTouchEvent(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
   trTouchData &rfrTouchData)
   * \brief   Receives the Touch events and forwards its to 
   either ML or DiPO Input handlers
   * \param   u32DeviceHandle : [IN] unique identifier to ML Server
   * \pram    enDevCat        : [IN] Identifies the Connection Request.
   * \param   rfrTouchData    : [IN] reference to touch data structure which contains
   *          touch details received /ref trTouchData
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessTouchEvent(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
      trTouchData &rfrTouchData);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vProcessKeyEvents
   ***************************************************************************/
   /*!
   * \fn      vProcessKeyEvents(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
   tenKeyMode enKeyMode, tenKeyCode enKeyCode)
   * \brief   Receives hard key events and forwards it to either 
   ML or DiPO Input handlers
   * \param   u32DeviceHandle : [IN] unique identifier to ML Server
   * \pram    enDevCat        : [IN] Identifies the Connection Request.
   * \param   enKeyMode       : [IN] indicates keypress or keyrelease
   * \param   enKeyCode       : [IN] unique key code identifier
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessKeyEvents(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
      tenKeyMode enKeyMode, tenKeyCode enKeyCode);


   /***************************************************************************
    ** FUNCTION:  spi_tclInputHandler::vSelectDevice
    ***************************************************************************/
   /*!
    * \fn     vSelectDevice
    * \brief  Device selection request
    * \param  corfrSelectReq : Device selection request details
    **************************************************************************/
   t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclInputHandler::vCbSelectDeviceResp()
   ***************************************************************************/
   /*!
   * \fn     t_Void vCbSelectDeviceResp(const t_U32 cou32DevId,
   *         const tenErrorCode coenErrorCode)
   * \brief  Method to Send the Select Device response to mediator
   * \param  cou32DevId       : [IN] Uniquely identifies the target Device.
   * \param  coenErrorCode    : [IN] Type of the error.
   * \retval t_Void
   **************************************************************************/
   t_Void vCbSelectDeviceResp(const t_U32 cou32DevId,
       const tenErrorCode coenErrorCode);

   /***************************************************************************
    ** FUNCTION:  spi_tclInputHandler::vSelectDeviceResult
    ***************************************************************************/
   /*!
    * \fn     vSelectDeviceResult
    * \brief  device selection result
    * \param  corfrSelectReq : Device selection result details
    * \param  enErrorCode : Error code
    **************************************************************************/
   t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vProcessKnobKeyEvents
   ***************************************************************************/
   /*!
   * \fn      vProcessKnobKeyEvents(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
   t_S8 s8EncoderDeltaCnt)
   * \brief   Receives Knob Encoder Delta Changes and forwards it to either
   *          ML/DiPO/AAP Input handlers
   * \param   u32DeviceHandle : [IN] unique identifier of the Server
   * \pram    enDevCat        : [IN] Identifies the Connection Request.
   * \param   s8EncoderDeltaCnt : [IN] Encoder delta counts.
   * \retval  NONE
   **************************************************************************/
   t_Void vProcessKnobKeyEvents(t_U32 u32DeviceHandle,tenDeviceCategory enDevCat,
         t_S8 s8EncoderDeltaCnt);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vGetKeyIconData()
   ***************************************************************************/
   /*!
   * \fn    t_Void vGetKeyIconData(t_String szKeyIconUrl,
   *         tenDeviceCategory enDevCat,
   *         const trUserContext& rfrcUsrCntxt)
   * \brief  To Get the application icon data
   * \param  szAppIconUrl  : [IN] Key Icon URL
   * \pram   enDevCat      : [IN] Identifies the Connection Category
   * \param  rfrcUsrCntxt  : [IN] User Context
   * \retval  t_Void
   **************************************************************************/
   t_Void vGetKeyIconData(const t_U32 cou32DevId,
      t_String szKeyIconUrl,
      tenDeviceCategory enDevCat,
      const trUserContext& rfrcUsrCntxt);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vCbKeyIconDataResult()
   ***************************************************************************/
   /*!
   * \fn    t_Void vCbKeyIconDataResult(const t_U8* pcu8AppIconData,t_U32 u32Len,
   *          const trUserContext& rfrcUsrCntxt)
   * \brief   Method to send the Get Key Icon data response to HMI
   * \param  pcu8AppIconData  : [IN] Image stream
   * \param  u32Len   : [IN] data stream length
   * \param  rfrcUsrCntxt  : [IN] User context
   * \retval  t_Void
   **************************************************************************/
   t_Void vCbKeyIconDataResp(const t_U32 cou32DevId,
                             const t_U8* pcu8KeyIconData,
                             t_U32 u32Len, const trUserContext& rfrcUsrCntxt);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vCbXDeviceKeyInfoResp()
   ***************************************************************************/
   /*!
   * \fn    t_Void vCbXDeviceKeyInfoResp(const t_U32 cou32DevId,
                                t_U16 u16NumXDevices,
                                std::vector<trXDeviceKeyDetails> vecrXDeviceKeyDetails)
   * \brief   Method to send the Get Key Icon data response to HMI
   * \param  cou32DevId  : [IN] Device ID.
   * \param  u16NumXDevices   : [IN] Number of X-Device keys
   * \param  vecrXDeviceKeyDetails  : [IN] X-Device keys details.
   * \param  u16ValidityOfData  : [IN] Validity Indicator
   * \retval  t_Void
   **************************************************************************/
   t_Void vCbServerKeyCapInfoResp(const t_U32 cou32DevId,
                                t_U16 u16NumXDevices,
                                trMLSrvKeyCapabilities rSrvKeyCapabilities,
                                t_U16 u16ValidityOfData);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclInputHandler::vSetInputHandlerSettingsInstance(spi_tclInputHandlerSettingsIntf* poInputHandlerSettingsIntf)
    ***************************************************************************/
    /*!
    * \fn      t_Void vSetInputHandlerSettingsInstance()
    * \brief   Method to set Input settings instance.
    * \param   poInputHandlerSettingsIntf  : pointer to InputHandlerSettingsIntf.
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetInputHandlerSettingsInstance(spi_tclInputHandlerSettingsIntf* poInputHandlerSettingsIntf);

   /***************************************************************************
      ** FUNCTION:  t_Void spi_tclInputHandler::vGetInputHandlerConfig()
      ***************************************************************************/
      /*!
      * \fn      t_Void vGetInputHandlerConfig()
      * \brief   Method to print the Input Handler configuration to read KDS
      * \retval  t_Void
      **************************************************************************/
   t_Void vGetInputHandlerConfig();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::bCheckKeyValidity()
   ***************************************************************************/
   /*!
   * \fn      t_Void bCheckKeyValidity()
   * \brief   Method to Check the validity of keys sent by Clients
   * \param   corfrmapTechSuppHardKeys      :[IN] Supported Hard Keys sent by clients
   * \param   corfrmapTechSuppSoftKeys      :[IN] Supported SoftKeys sent by clients
   * \param   rfrmapTechUnSuppHardKeys      :[OUT] UnSupported Hard Key Info
   * \param   rfrmapTechUnSuppSoftKeys      :[OUT] UnSupported Soft Key Info
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bCheckKeyValidity(
              const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardkeys,
              const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftkeys,
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechUnSuppHardkeys,
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechUnSuppSoftkeys);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vRegisterKeyCodes()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRegisterKeyCodes()
   * \brief   Method to Register Key codes sent by clients
   * \param   rfrmapTechSuppHardKeys :[IN] Supported Hard Key Info sent by clients
   * \param   rfrmapTechSuppSoftKeys :[IN] Supported Soft Key Info sent by clients
   * \param   trUserContext :[IN] User Context
   * \retval  t_Bool
   **************************************************************************/
   t_Void vRegisterKeyCodes(
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppHardkeys,
              std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppSoftkeys,
              const trUserContext& rfcorUsrCntxt);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclInputHandler::vResetKeyInfo()
   ***************************************************************************/
   /*!
   * \fn      t_Void vResetKeyInfo()
   * \brief   Method to update technologies the factory reset has happened
   * \retval  t_Void
   **************************************************************************/   
   t_Void vResetKeyInfo();




   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/


   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE**********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclInputHandler::spi_tclInputHandler(const spi_tclInputHandler..
   ***************************************************************************/
   /*!
   * \fn      spi_tclInputHandler(const spi_tclInputHandler &corfrSrc)
   * \brief   Copy constructor, will not be implemented.
   * \note    This is a technique to disable the Copy constructor for this class.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclInputHandler(const spi_tclInputHandler& corfrSrc);

   /***************************************************************************
   ** FUNCTION:  spi_tclInputHandler& spi_tclInputHandler::operator= (const..
   ***************************************************************************/
   /*!
   * \fn      spi_tclInputHandler& operator= (const spi_tclInputHandler &corfrSrc)
   * \brief   Assignment Operator, will not be implemented.
   * \note    This is a technique to disable the assignment operator for this class.
   *          So if an attempt for the assignment is made linker complains.
   * \param   corfrSrc : [IN] Source Object
   **************************************************************************/
   spi_tclInputHandler& operator = (const spi_tclInputHandler& corfrSrc);

   /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclInputHandler::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRegisterCallbacks()
   * \brief   To Register for the asynchronous responses that are required from
   *          ML/DiPo/AAP Input Handler
   * \retval  t_Void
   **************************************************************************/
   t_Void vRegisterCallbacks();

   /***************************************************************************
   ** FUNCTION:  t_Bool  spi_tclVideo::bValidateClient()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bValidateClient(t_U8 cou8Index)
   * \brief   To validate the client index. check whether it is in the range of
   *          the Video clients Array
   * \param   cou8Index : [IN] Index in the video clients Array
   * \retval  t_Bool
   **************************************************************************/
   inline t_Bool bValidateClient(const t_U8 cou8Index);
   //! Member Variable - pointer to the Base class for ML and DiPo Input Handlers
   spi_tclInputDevBase* m_poInputDevBase[NUM_INPUT_CLIENTS];

   //!Input Response Interface Pointer
   spi_tclInputRespIntf* m_poInputRespIntf;

   spi_tclInputHandlerSettingsIntf* m_poInputHandlerSettings;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

};

#endif /* SPI_TCLINPUTHANDLER_H_ */
