/*!
 *******************************************************************************
 * \file             spi_tclDeviceSelector.cpp
 * \brief            Handles device diagnosis
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles device diagnosis
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 21.09.2018 |  Rajendra Naik Vadthe 	   | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclDiagnosisResp.h"
#include "spi_tclDiagnosis.h"
#include "spi_tclDiscovererSettings.h"
#include "SPITypes.h"
#include "Timer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY//Raj::Need to check
#include "trcGenProj/Header/spi_tclDiagnosis.cpp.trc.h"
#endif
#endif

//Timer limit for device visibility is 50sec
static const t_U32 scou32DeviceConnectionTimeout = 50000;

/***************************************************************************
 ** FUNCTION:  spi_tclDiagnosis::spi_tclDiagnosis
 ***************************************************************************/
spi_tclDiagnosis::spi_tclDiagnosis(spi_tclDiagnosisResp *poRespInterface, spi_tclMediatorIntf* poMediator,
         spi_tclDiscovererSettingsIntf* poDiscovSettings) :
                  m_poMediator(poMediator),
                  m_poDiagnosisResp(poRespInterface),
                  m_oDeviceConnectionTimerIndex(0),
                  m_u32ValidDeviceHandle(0),
                  m_bIsMirrorlinkSupported(false),
                  m_poDiscovererSettings(poDiscovSettings)
{
   ETG_TRACE_USR1(("Creating spi_tclDiagnosis "));
   //m_u32CurrSelectedDevice = 0;
   //bInitialize();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiagnosis::~spi_tclDiagnosis
 ***************************************************************************/
spi_tclDiagnosis::~spi_tclDiagnosis()
{
   ETG_TRACE_USR1(("Destroying spi_tclDeviceSelector  "));
   m_poDiagnosisResp = NULL;
   m_poMediator = NULL;
   m_poDiscovererSettings = NULL;
   m_bIsMirrorlinkSupported = false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiagnosis::bInitialize
 ***************************************************************************/
t_Bool spi_tclDiagnosis::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclDiagnosis::bInitialize "));
   vRegisterCallbacks();   //Raj::Need to check
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIAGLOG    
   trSpiFeatureSupport rSpiFeatureSupp;
   if ((nullptr != m_poDiscovererSettings))
   {
      m_poDiscovererSettings->vGetSpiFeatureSupport(rSpiFeatureSupp);
   }
   m_bIsMirrorlinkSupported = rSpiFeatureSupp.bMirrorLinkSupported();
   if (m_bIsMirrorlinkSupported)
   {
      ETG_TRACE_USR1(("spi_tclDiagnosis::bInitialize MirrorLink is supported \n"));
      vStartDeviceConnectionTimer();
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclDiagnosis::bInitialize Diaglog results are not reported \n"));
   }
#endif
   //! Even if starting timer fails, initialize can return true as this
   //! doen't affect major functionality
   //! Currently timer is not used by any projects
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiagnosis::vUnInitialize()
 ***************************************************************************/
t_Void spi_tclDiagnosis::vUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclDiagnosis::vUnInitialize "));
   //! Add code: Currently callback deregistration not required.
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclDiagnosis::vRegisterCallbacks()
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vRegisterCallbacks() Entered"));
   //! Register for callbacks
   trDevDiagnosisCallback rDevDiagnosisCbs;
   rDevDiagnosisCbs.fvOnDeviceDisconnectionStatus = std::bind(&spi_tclDiagnosis::vDeviceDisconnectionCb, this,
   SPI_FUNC_PLACEHOLDERS_1);

   rDevDiagnosisCbs.fvOnDeviceConnectionStatus = std::bind(&spi_tclDiagnosis::vDeviceConnectionCb,
            this,
            SPI_FUNC_PLACEHOLDERS_3);

   if (NULL != m_poMediator)
   {
      m_poMediator->vRegisterCallbacks(rDevDiagnosisCbs);
   } // if (NULL != m_poMediator)

}

/*******************************************************************
 ** FUNCTION: t_Void spi_tclDiagnosis::vStartDeviceConnectionTimer()
 ********************************************************************/
t_Void spi_tclDiagnosis::vStartDeviceConnectionTimer()
{
   ETG_TRACE_USR2(("spi_tclDeviceDiscoverer::vStartDeviceConnectionTimer entered"));
   Timer* poTimer = Timer::getInstance();
   if (0 == m_oDeviceConnectionTimerIndex)
   {
      if ((NULL != poTimer)
               && (false
                        == poTimer->StartTimer(m_oDeviceConnectionTimerIndex,
                                 scou32DeviceConnectionTimeout,
                                 0,
                                 this,
                                 &spi_tclDiagnosis::bDeviceConnectionTimerCb,
                                 NULL)))
      {
         ETG_TRACE_ERR(("vStartDeviceConnectionTimer: Error in starting Timer"));
      }
   }
}

/******************************************************
 ** t_Bool spi_tclDiagnosis::bDeviceConnectionTimerCb()
 *******************************************************/
t_Bool spi_tclDiagnosis::bDeviceConnectionTimerCb(timer_t timerID, t_Void *pObject, const t_Void *pcoUserData)
{
   ETG_TRACE_USR2(("spi_tclDiagnosis::bDeviceConnectionTimerCb entered"));
   t_Bool bRet = false;

   SPI_INTENTIONALLY_UNUSED(pcoUserData);
   spi_tclDiagnosis* poDiagnosis = static_cast<spi_tclDiagnosis*>(pObject);
   ETG_TRACE_USR4(("[PARAM]:timerID: %p", timerID));
   if (NULL != poDiagnosis)
   {
      if (timerID == poDiagnosis->m_oDeviceConnectionTimerIndex)
      {
         poDiagnosis->m_oDeviceConnectionTimerIndex = 0;
         bRet = true;
      }
      //WRITE_INTO_DTC if DTC_POINTER is not alive....In the next IGN cycle if device is still not visible...Is it sufficient if we write into DTC only once in previous IGN cycle
      //! Send Device DTC Status change
      if (NULL != poDiagnosis)
      {
         poDiagnosis->vHandleDeviceConnection(e8DTC_NOT_CONNECTED);
      } //if (NULL != m_poDiscoveryResp)
   }
   return bRet;
}

/******************************************************
 ** t_Void spi_tclDiagnosis::vHandleDeviceConnection()
 *******************************************************/
t_Void spi_tclDiagnosis::vHandleDeviceConnection(tenDTCStatus enDTCStatus)
{

   if (NULL != m_poDiagnosisResp)
   {
      m_poDiagnosisResp->vPostDeviceDTCStatus(enDTCStatus);
   }
}

/******************************************************************
 ** FUNCTION: t_Void spi_tclDiagnosis::vStopDeviceConnectionTimer()
 *******************************************************************/
t_Void spi_tclDiagnosis::vStopDeviceConnectionTimer()
{
   //Stop the timer incase HU got PID and VID of the device
   ETG_TRACE_USR2(("spi_tclDiagnosis::vStopDeviceConnectionTimer entered"));
   Timer* poTimer = Timer::getInstance();
   if (0 != m_oDeviceConnectionTimerIndex)
   {
      if (NULL != poTimer)
      {
         poTimer->CancelTimer(m_oDeviceConnectionTimerIndex);
         //Reset the Timer Index
         m_oDeviceConnectionTimerIndex = 0;
         ETG_TRACE_USR2(("vStopDeviceConnectionTimer: Timer stopped"));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiagnosis::vDeviceConnection
 ***************************************************************************/
t_Void spi_tclDiagnosis::vDeviceConnection(t_U32 u32ProductID, t_U32 u32VendorID, t_U32 u32DeviceHandle)
{
   if (NULL != m_poDiscovererSettings)
   {
      if (true == m_poDiscovererSettings->bIsDeviceWhitelisted(u32ProductID, u32VendorID, e8DEV_TYPE_MIRRORLINK))
      {
         vStopDeviceConnectionTimer();
         if (NULL != m_poDiagnosisResp)
         {
            m_poDiagnosisResp->vPostDeviceDTCStatus(e8DTC_CONNECTED);
            m_u32ValidDeviceHandle = u32DeviceHandle;
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiagnosis::vDeviceConnectionCb
 ***************************************************************************/
t_Void spi_tclDiagnosis::vDeviceConnectionCb(const t_U32 u32ProductID, const t_U32 u32VendorID,
         const t_U32 cou32DeviceHandle)
{

   ETG_TRACE_USR1((" spi_tclDiagnosis::vDeviceConnectionCb received for cou32DeviceHandle = 0x%x u32ProductID=%u u32VendorID=%u", cou32DeviceHandle, u32ProductID, u32VendorID));
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIAGLOG   
   if (true == m_bIsMirrorlinkSupported)
   {
      ETG_TRACE_USR1(("spi_tclDiagnosis::vDeviceConnectionCb  MirrorLink is supported \n"));
      vDeviceConnection(u32ProductID, u32VendorID, cou32DeviceHandle);
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclDiagnosis::vDeviceConnectionCb  Diaglog results are not reported \n"));
   }
#endif
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vDeviceDisconnectionCb
 ***************************************************************************/
t_Void spi_tclDiagnosis::vDeviceDisconnectionCb(const t_U32 cou32DeviceHandle)
{
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIAGLOG
   ETG_TRACE_USR1((" spi_tclDiagnosis::vDeviceDisconnectionCb:"
                     " cou32DeviceHandle = 0x%x", cou32DeviceHandle));
   if (true == m_bIsMirrorlinkSupported)
   {
      if (NULL != m_poDiagnosisResp && m_u32ValidDeviceHandle == cou32DeviceHandle)
      {
         m_poDiagnosisResp->vPostDeviceDTCStatus(e8DTC_NOT_CONNECTED);
      }
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclDiagnosis::vDeviceDisconnectionCb  Diaglog results are not reported \n"));
   }
#endif   
}

