/*!
 *******************************************************************************
 * \file         spi_tclDataService.cpp
 * \brief        Data Service Manager class that provides interface to delegate
 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Data Service Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 27.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 14.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented sending GPS data to
 MediaPlayer client.
 06.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Initialisation sequence implementation
 13.06.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for: 
 (1) MPlay FI extn to start/stop loc info
 (2) VDSensor data integration
 (3) NMEA-PASCD sentence for DiPO
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)         | SPI feature configuration via LoadSettings()
 13.10.2014 |  Hari Priya E R (RBEI/ECP2)       | Added interface for Vehicle Data for PASCD
 and interface with Data Service Response
 05.02.2015 |  Ramya Murthy (RBEI/ECP2)         | Added interface to set availability of LocationData
 28.03.2015 |  SHITANSHU SHEKHAR (RBEI/ECP2)    | Initialized AAP data service object and
 Implemented enGetDataServiceIndex()
 23.04.2015 |  Ramya Murthy (RBEI/ECP2)         | Added interface to set region code
 28.05.2015 |  Tejaswini H B(RBEI/ECP2)         | Added Lint comments to suppress C++11 Errors
 25.06.2015 |  Tejaswini HB (RBEI/ECP2)         | Featuring out Android Auto
 30.10.2015 |  Shiva Kumar G                    | Implemented ReportEnvironment Data feature
 03.12.2015 |  SHITANSHU SHEKHAR                | Implemented vSetFeatureRestrictions() function.
 16.03.2016 |  Ramya Murthy                     | Revised implementation of data subscription
 01.01.2017 |  Noopur R K                       | Added changes for Delete device interface implementation
 01.03.2017 |  Shiva Kumar G                    | Changes to enable/disable CarPlay feature    
 20.03.2019 | Roveena Francy Lobo               | Added vOnLanguageSetting() for private cdb
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclFactory.h"
#include "spi_tclMediator.h"
#include "spi_tclMPlayClientHandler.h"
#include "RespRegister.h"
#include "spi_tclDataServiceDevBase.h"
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDipoDataService.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAAPDataService.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINDataService.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLDataService.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarDataService.h"
#endif

#include "LocationDataSimulator.h"
#include "spi_tclDataService.h"
#include "spi_LoopbackTypes.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdSensorData.h"
#include "spi_tclExtCmdPositionData.h"
#include "spi_tclExtCmdClock.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
#include "trcGenProj/Header/spi_tclDataService.cpp.trc.h"
#endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDataServicesIndex, GetDataService_t*> DataService;
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDataService::spi_tclDataService();
 ***************************************************************************/
spi_tclDataService::spi_tclDataService(ahl_tclBaseOneThreadApp* poMainApp,
         spi_tclDataServiceRespIntf* poDataSrvcRespIntf) :
                  spi_tclSelectionIntf(e32COMPID_DATASERVICE),
                  m_poMainApp(poMainApp),
                  m_enSelDevCategory(e8DEV_TYPE_UNKNOWN),
                  m_poDataSrvcRespIntf(poDataSrvcRespIntf),
                  m_poDataServiceSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclDataService() entered "));
   SPI_NORMAL_ASSERT(NULL == m_poMainApp);
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      m_poDataSvcHandlersLst[u8Index] = NULL;
   }
} //!end of spi_tclDataService()

/***************************************************************************
 ** FUNCTION:  spi_tclDataService::~spi_tclDataService();
 ***************************************************************************/
spi_tclDataService::~spi_tclDataService()
{
   ETG_TRACE_USR1(("~spi_tclDataService() entered "));
   m_poDataSrvcRespIntf = NULL;
   m_poMainApp = NULL;
   m_poDataServiceSettings = NULL;
} //!end of ~spi_tclDataService()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDataService::bInitialize()
 ***************************************************************************/
t_Bool spi_tclDataService::bInitialize()
{
   //! Create Pos_Fi client handler
   SPI_NORMAL_ASSERT(NULL == m_poMainApp);
   t_Bool bInitSuccess = false;
   /*lint -esym(40,fvSubscribeForLocationData) fvSubscribeForLocationData Undeclared identifier */
   /*lint -esym(40,fvSubscribeForEnvData) fvSubscribeForEnvData Undeclared identifier */
   /*lint -esym(40,_1) _1 Undeclared identifier */
   /*lint -esym(40,_2) _2 Undeclared identifier */
   //! Create the Data Service handlers
   //! Initialize DataService manager callbacks structure
   trDataServiceCb rDataServiceCb;
   rDataServiceCb.fvSubscribeForData = std::bind(&spi_tclDataService::vOnSubscribeForData, this,
   SPI_FUNC_PLACEHOLDERS_2);
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK	  
   ETG_TRACE_USR1(("spi_tclDataService::bInitialize DataService[e8_DATASVC_ML_INDEX]=%p",DataService[e8_DATASVC_ML_INDEX]));
   if ((NULL != DataService[e8_DATASVC_ML_INDEX]))
   {
      ETG_TRACE_USR2(("[DESC]::spi_tclDataService::bInitialise() before obj m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX] = %p \n", m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX]));
      m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX] = DataService[e8_DATASVC_ML_INDEX](rDataServiceCb);
      SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX]);
   }
   ETG_TRACE_USR2(("[DESC]::spi_tclDataService::bInitialise() after obj success m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX] = %p \n", m_poDataSvcHandlersLst[e8_DATASVC_ML_INDEX]));
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poDataSvcHandlersLst[e8_DATASVC_DIPO_INDEX] = new spi_tclDipoDataService(rDataServiceCb);
   SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_DIPO_INDEX]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
   m_poDataSvcHandlersLst[e8_DATASVC_ANDROIDAUTO_INDEX] = new spi_tclAAPDataService(rDataServiceCb);
   SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_ANDROIDAUTO_INDEX]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
   m_poDataSvcHandlersLst[e8_DATASVC_MYSPIN_INDEX]= new (std::nothrow) spi_tclMySPINDataService(rDataServiceCb);
   SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_MYSPIN_INDEX]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
   m_poDataSvcHandlersLst[e8_DATASVC_BDCL_INDEX]= new (std::nothrow) spi_tclBDCLDataService(rDataServiceCb);
   SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_BDCL_INDEX]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
   m_poDataSvcHandlersLst[e8_DATASVC_ONCAR_INDEX]= new (std::nothrow) spi_tclOnCarDataService(rDataServiceCb);
   SPI_NORMAL_ASSERT(NULL == m_poDataSvcHandlersLst[e8_DATASVC_ONCAR_INDEX]);
#endif

   //! Initialise the Data Service handlers
   //! Set the bInit to false, if all the SPI technologies cannot be initialized
   //! else return true if one of the SPI technologies can be initialized
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         bInitSuccess = m_poDataSvcHandlersLst[u8Index]->bInitialise() || bInitSuccess;
      }
   } //for (t_U8 u8Index = 0;...)

   RespRegister *pRespRegister = RespRegister::getInstance();
   if (NULL != pRespRegister)
   {
      pRespRegister->bRegisterObject((spi_tclExtRespPositionData*) this);
      pRespRegister->bRegisterObject((spi_tclExtRespSensorData*) this);
      pRespRegister->bRegisterObject((spi_tclExtRespNavData*) this);
      pRespRegister->bRegisterObject((spi_tclExtRespClock*) this);
   }

   ETG_TRACE_USR2(("[DESC]::spi_tclDataService::bInitialise() left with result = %u \n", ETG_ENUM(BOOL, bInitSuccess)));
   return bInitSuccess;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclDataService::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclDataService::bUnInitialize() entered "));

   t_Bool bUninitSuccess = true;

   //! Uninitialize & delete the Data Service handlers
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         t_Bool bIsSuccess = m_poDataSvcHandlersLst[u8Index]->bUninitialise();
         bUninitSuccess = bUninitSuccess && bIsSuccess;
      }
      RELEASE_MEM(m_poDataSvcHandlersLst[u8Index]);
   } //for (t_U8 u8Index = 0;...)

   ETG_TRACE_USR2(("[DESC]::bUnInitialize() left with result = %u ", bUninitSuccess));
   return bUninitSuccess;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vLoadSettings()
 ***************************************************************************/
t_Void spi_tclDataService::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclDataService::vLoadSettings() entered "));
   tenRegion enRegion = e8_WORLD;
   if (NULL != m_poDataServiceSettings)
   {
      enRegion = m_poDataServiceSettings->enGetRegion();
   }
   vSetRegion(enRegion);
   //Enable Location data based on System variant.
   t_Bool bIsDRLocData = false;
   vSetLocDataAvailablility(bIsDRLocData);
   //Add code
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vSetDataServiceSettingsInstance(m_poDataServiceSettings);
         m_poDataSvcHandlersLst[u8Index]->vLoadSettings();
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSaveSettings()
 ***************************************************************************/
t_Void spi_tclDataService::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclDataService::vSaveSettings() entered "));
   //Add code
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclDataService::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vSelectDevice() entered: "
            "DeviceHandle = %u, DevConnectionRequest = %u, Device Category = %u ", corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            corfrSelectReq.m_enDevConnReq), ETG_ENUM(DEVICE_CATEGORY, corfrSelectReq.m_enDevCategory)));

   t_U8 u8ServiceLstIndx = enGetDataServiceIndex(corfrSelectReq.m_enDevCategory);

   if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   {
      m_poDataSvcHandlersLst[u8ServiceLstIndx]->vSelectDevice(corfrSelectReq.m_u32DeviceHandle,
               corfrSelectReq.m_enDevConnReq);
   }

   spi_tclMediator *poMediator = spi_tclMediator::getInstance();

   //! Data service can always return true as the service is not critical for feature to work
   if (NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_DATASERVICE, e8NO_ERRORS);
   } //if(NULL != poMediator)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnSelectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclDataService::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vOnSelectDeviceResult() entered: "
            "DeviceHandle = %u, DevConnectionRequest = %u, DeviceCategory = %d, ErrorCode = %u ", corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            corfrSelectReq.m_enDevConnReq), ETG_ENUM(DEVICE_CATEGORY, corfrSelectReq.m_enDevCategory), ETG_ENUM(ERROR_CODE,
            enErrorCode)));

   tenDeviceCategory enDevCat = corfrSelectReq.m_enDevCategory;

   //! Trigger DataService handler based on SelectDevice/Deselect device operation success
   if (e8DEV_TYPE_UNKNOWN != enDevCat)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(enDevCat);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         if (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq)
         {
            tenResponseCode enRespCode = (e8NO_ERRORS == enErrorCode) ? e8SUCCESS : e8FAILURE;
            m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnSelectDeviceResult(corfrSelectReq.m_u32DeviceHandle,
                     corfrSelectReq.m_enDevConnReq,
                     enRespCode);
            //Store SelectedDevice category
            m_enSelDevCategory = (e8NO_ERRORS == enErrorCode) ? enDevCat : e8DEV_TYPE_UNKNOWN;
            ;
         }
         else if ((e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq) && (enDevCat == m_enSelDevCategory))
         {
            m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnDeselectDeviceResult(corfrSelectReq.m_u32DeviceHandle);
            //Clear SelectedDevice category
            m_enSelDevCategory = e8DEV_TYPE_UNKNOWN;
         }
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } //if (e8DEV_TYPE_UNKNOWN != enDevCat)

} //!end of vOnSelectDeviceResult()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vStartStopLocationData(...)
 ***************************************************************************/
t_Void spi_tclDataService::vStartStopLocationData(t_Bool bStartLocData,
         const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList, t_U8 u8RequestedBy)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vStartStopLocationData() entered: "
            "bStartLocData = %u, rfcoNmeaSentencesList size = %u ", ETG_ENUM(BOOL, bStartLocData), rfcoNmeaSentencesList.size()));

   if (e8DEV_TYPE_DIPO != m_enSelDevCategory)
   {
      ETG_TRACE_USR2(("[DESC]::vStartStopLocationData received before CarPlay device is active. "));
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)

   //! Forward request to Dipo DataService handler
   //! @Note: StartLocationData or StopLocationData message is processed regardless of whether CP device is
   //! active/inactive in SPI, since the messages can arrive before CarPlay selection is complete (PSARCC21-553).

   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {

      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         if ((true == bStartLocData) && (false == rfcoNmeaSentencesList.empty()))
         {
            m_poDataSvcHandlersLst[u8Index]->vStartLocationData(rfcoNmeaSentencesList, u8RequestedBy);
         }
         else if (false == bStartLocData)
         {
            m_poDataSvcHandlersLst[u8Index]->vStopLocationData(rfcoNmeaSentencesList, u8RequestedBy);
         }
         else
         {
            ETG_TRACE_ERR(("[ERR]::vStartStopLocationData: Empty NMEA sentences list received! "));
         }
      }
   }

} //!end of vStartStopLocationData()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnGPSData(const trGPSData& rfrcGpsData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnGPSData(const trGPSData& rfrcGpsData)
{
   //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnGPSData: "
            "Latitude = %f degree, Longitude = %f degree, Heading = %f degree, Heading Direction = %u, "
            "Speed = %f cm/s, Altitude = %f m, ExactTime(ms) = %u ", static_cast<t_Float>(rfrcGpsData.d64Latitude), static_cast<t_Float>(rfrcGpsData.d64Longitude), static_cast<t_Float>(rfrcGpsData.dHeading), ETG_ENUM(HEADING_DIR,
            rfrcGpsData.enHeadingDir), rfrcGpsData.f32Speed, rfrcGpsData.f32Altitude, rfrcGpsData.u16ExactTime));

   //@Note: Posix time printed separately since its a 64-bit value
   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnGPSData: PosixTime = %s ", std::to_string(rfrcGpsData.PosixTime).c_str()));

   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnGPSData: "
            "SensorType = %d, TurnRate = %f centidegree/s, Acceleration = %d ", rfrcGpsData.enSensorType, rfrcGpsData.f32TurnRate, rfrcGpsData.s16Acceleration));

   //! Send data to Selected device's DataService handler
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnData(rfrcGpsData);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnSensorData(const trSensorData& rfrcSensorData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnSensorData(const trSensorData& rfrcSensorData)
{
   //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnSensorData: "
            "GnssQuality = %u, enGnssMode = %u, SatellitesUsed = %u, "
            "HDOP = %f, GeoidalSeparation = %f, TimeStamp = %u ", ETG_ENUM(GNSS_QUALITY, rfrcSensorData.enGnssQuality), ETG_ENUM(GNSS_MODE,
            rfrcSensorData.enGnssMode), rfrcSensorData.u16SatellitesUsed, static_cast<t_Float>(rfrcSensorData.dHDOP), static_cast<t_Float>(rfrcSensorData.dGeoidalSeparation), rfrcSensorData.u32TimeStamp));

   //@Note: Posix time printed separately since its a 64-bit value
   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnSensorData: PosixTime = %lld ", rfrcSensorData.PosixTime));

   //! Send data to Selected device's DataService handler
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnData(rfrcSensorData);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnAccSensorData
 ** (const std::vector<trAccSensorData>& corfvecrAccSensorData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
{
   //! Send data to Selected device's DataService handler
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnAccSensorData(corfvecrAccSensorData);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnGyroSensorData
 ** (const std::vector<trGyroSensorData>& rfcoSensorData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
{
   //! Send data to Selected device's DataService handler
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnGyroSensorData(corfvecrGyroSensorData);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnVehicleData(const trVehicleData rfrcVehicleData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnVehicleData(const trVehicleData rfrcVehicleData, t_Bool bSolicited)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vOnVehicleData: "
            "Speed = %d, GearPosition = %u, ParkBrakeActive = %u, VehMovState = %u ", rfrcVehicleData.s16Speed, ETG_ENUM(GEARPOSITION,
            rfrcVehicleData.enGearPosition), ETG_ENUM(BOOL, rfrcVehicleData.bParkBrakeActive), ETG_ENUM(VEH_MOV_STATE,
            rfrcVehicleData.enVehMovState)));

   //! Send data to Selected device's DataService handler
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vOnData(rfrcVehicleData, bSolicited);
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSetLocDataAvailablility(t_Bool...)
 ***************************************************************************/
t_Void spi_tclDataService::vSetLocDataAvailablility(t_Bool bIsDeadReckonedData)
{
   ETG_TRACE_USR1(("spi_tclDataService::vSetLocDataAvailablility) entered "));
   trDataServiceConfigData rDataServiceConfigData;

   if (NULL != m_poDataServiceSettings)
   {
      rDataServiceConfigData.bLocDataAvailable = (m_poDataServiceSettings->bGetLocDataEnabled())
               && (m_poDataServiceSettings->bIsLocDataEnabled());
      rDataServiceConfigData.bAcclData = m_poDataServiceSettings->bGetAccelerometerDataEnabled();
      rDataServiceConfigData.bDeadReckonedData = bIsDeadReckonedData;
      rDataServiceConfigData.bEnvData = m_poDataServiceSettings->bGetEnvDataSubEnabled();
      rDataServiceConfigData.bGearStatus = m_poDataServiceSettings->bGetGearStatusEnabled();
      rDataServiceConfigData.bGyroData = m_poDataServiceSettings->bGetGyroDataEnabled();
      rDataServiceConfigData.bParkBrakeData = m_poDataServiceSettings->bGetParkBrakeDataEnabled();
      rDataServiceConfigData.u32SpeedTimerIntervalInMs = m_poDataServiceSettings->u32GetSpeedTimerInterval();
      rDataServiceConfigData.enGeoCoordinateSystemType = m_poDataServiceSettings->enGetGeoCoordinateSystem();
   } //if (NULL != m_poDataServiceSettings)

   ETG_TRACE_USR4(("[PARAM]:vSetLocDataAvailablility: "
            "LocDataAvailable = %d , IsDeadReckonedData = %d, "
            "AcclData = %d, GyroData = %d, GearData = %d, EnvData = %d, "
            "Park Brake Data = %d, Speed frequency = %d", rDataServiceConfigData.bLocDataAvailable, rDataServiceConfigData.bDeadReckonedData, rDataServiceConfigData.bAcclData, rDataServiceConfigData.bGyroData, rDataServiceConfigData.bGearStatus, rDataServiceConfigData.bEnvData, rDataServiceConfigData.bParkBrakeData, rDataServiceConfigData.u32SpeedTimerIntervalInMs));

   //! Send data to device's DataService handler
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vSetSensorDataAvailablility(rDataServiceConfigData);
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSimulateGPSDataTransfer(t_U32 u32DataRateInMs)
 ***************************************************************************/
t_Void spi_tclDataService::vSimulateGPSDataTransfer(t_U32 u32DataRateInMs)
{
   ETG_TRACE_USR1(("spi_tclDataService::vSimulateGPSDataTransfer() entered "));

   trGPSData rSampleGpsData;
   LocationDataSimulator* poLocSimulator = LocationDataSimulator::getInstance();
   if (NULL != poLocSimulator)
   {
      /*lint -esym(40,fvOnSimGpsData) fvOnSimGpsData Undeclared identifier */
      /*lint -esym(40,_2) _2 Undeclared identifier */
      //! Initialise callbacks to be handed over to
      //! Pos_Fi client handler for receiving GPS data.
      trLocDataSimulationCb rLocDataSimCb;
      rLocDataSimCb.fvOnSimGpsData = std::bind(&spi_tclDataService::vOnGPSData, this,
      SPI_FUNC_PLACEHOLDERS_1);

      poLocSimulator->vRegisterCallbacks(rLocDataSimCb);
      poLocSimulator->vSimulateGpsData(u32DataRateInMs);
   } //if (NULL != poLocSimulator)
}

/***************************************************************************
 *********************************PRIVATE************************************
 ***************************************************************************/

/***************************************************************************
 **FUNCTION: tenDataServicesIndex spi_tclDataService::enGetDataServiceIndex(tenDeviceCategory)
 ***************************************************************************/
tenDataServicesIndex spi_tclDataService::enGetDataServiceIndex(tenDeviceCategory enDataCategory)
{
   tenDataServicesIndex enDataSvcIndex = e8_DATASVC_ML_INDEX; //Initialise to avoid Lint warning;
   switch (enDataCategory)
   {
      case e8DEV_TYPE_DIPO:
         enDataSvcIndex = e8_DATASVC_DIPO_INDEX;
         break;
      case e8DEV_TYPE_MIRRORLINK:
         enDataSvcIndex = e8_DATASVC_ML_INDEX;
         break;
      case e8DEV_TYPE_ANDROIDAUTO:
         enDataSvcIndex = e8_DATASVC_ANDROIDAUTO_INDEX;
         break;
      case e8DEV_TYPE_MYSPIN:
         enDataSvcIndex = e8_DATASVC_MYSPIN_INDEX;
         break;
      case e8DEV_TYPE_CARLIFE:
         enDataSvcIndex = e8_DATASVC_BDCL_INDEX;
         break;
      case e8DEV_TYPE_ONCAR:
         enDataSvcIndex = e8_DATASVC_ONCAR_INDEX;
         break;
      default:
         ETG_TRACE_ERR(("[ERR]::spi_tclDataService::enGetDataServiceIndex() Invalid data received"));
         break;
   } //switch(enDataCategory)

   return enDataSvcIndex;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDataService::vSetRegion(...)
 ***************************************************************************/
t_Void spi_tclDataService::vSetRegion(tenRegion enRegion)
{
   ETG_TRACE_USR1(("spi_tclDataService::vSetRegion() entered: Region = %u ", enRegion));

   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vSetRegion(enRegion);
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnSubscribeForData(...)
 ***************************************************************************/
t_Void spi_tclDataService::vOnSubscribeForData(t_Bool bSubscribe, tenSensorDataType enDataType)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vOnSubscribeForData() entered: "
            "Subscribe = %u, DataType = %u ", ETG_ENUM(SUBSCRIPTION_TYPE, bSubscribe), ETG_ENUM(SENSOR_DATA_TYPE,
            enDataType)));

   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdPositionDataIntf *poExtCmdPositionDataIntf = NULL;
   spi_tclExtCmdSensorDataIntf *poExtCmdSensorDataIntf = NULL;
   spi_tclExtCmdClockIntf *poExtCmdClockIntf = NULL;
   if (NULL != poExtCompMgr)
   {
      poExtCmdPositionDataIntf = poExtCompMgr->poGetCmdPositionDataIntfInst();
      poExtCmdSensorDataIntf = poExtCompMgr->poGetCmdSensorDataIntfInst();
      poExtCmdClockIntf = poExtCompMgr->poGetCmdClockIntfInst();
   }
   if (true == bSubscribe)
   {
      switch (enDataType)
      {
         case e8GPS_DATA:
         case e8DEADRECKONING_DATA:
         {
            if (NULL != poExtCmdPositionDataIntf)
            {
               poExtCmdPositionDataIntf->vStartLocData(enDataType);
            }
         }
            break;

         case e8GNSS_DATA:
         case e8ACCELEROMETER_DATA:
         case e8GYROSCOPE_DATA:
         {
            if (NULL != poExtCmdSensorDataIntf)
            {
               poExtCmdSensorDataIntf->vStartSensorData(enDataType);
            }
         }
            break;

         case e8VEHICLE_DATA:
         {
            //@Note: Nothing to do. Data is provided by HMI.
         }
            break;
         case e8VDCLOCK_DATA:
         {
            if (NULL != poExtCmdClockIntf)
            {
               poExtCmdClockIntf->vStartClock();
            }
         }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]:spi_tclDataService vOnSubscribeForData: Invalid data type %d", ETG_ENUM(SENSOR_DATA_TYPE,
                     enDataType)));
            break;

      } //! End of switch
   } //! End of if(true == bSubscribe)
   else
   {
      switch (enDataType)
      {
         case e8GPS_DATA:
         case e8DEADRECKONING_DATA:
         {
            if (NULL != poExtCmdPositionDataIntf)
            {
               poExtCmdPositionDataIntf->vStopLocData(enDataType);
            }
         }
            break;

         case e8GNSS_DATA:
         case e8ACCELEROMETER_DATA:
         case e8GYROSCOPE_DATA:
         {
            if (NULL != poExtCmdSensorDataIntf)
            {
               poExtCmdSensorDataIntf->vStopSensorData(enDataType);
            }
         }
            break;

         case e8VEHICLE_DATA:
         {
            //@Note: Nothing to do. Data is provided by HMI.
         }
            break;
         case e8VDCLOCK_DATA:
         {
            if (NULL != poExtCmdClockIntf)
            {
               poExtCmdClockIntf->vStopClock();
            }
         }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]:spi_tclDataService vOnSubscribeForData: Invalid data type %d", ETG_ENUM(SENSOR_DATA_TYPE,
                     enDataType)));
            break;

      } //! End of switch
   } //! End of else
} //! End of vOnSubscribeForData()

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vOnEnvironmentData()
 ***************************************************************************/
t_Void spi_tclDataService::vOnEnvironmentData(t_Bool bValidTempUpdate, t_Double dTemp, t_Bool bValidPressureUpdate,
         t_Double dPressure)
{
   //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
   ETG_TRACE_USR4(("[PARAM]::spi_tclDataService::vOnEnvironmentData: "
            "Temp = %f, IsValidTemp = %u, Pressure = %f, IsValidPressure = %u ", static_cast<t_Float>(dTemp), ETG_ENUM(BOOL,
            bValidTempUpdate), static_cast<t_Float>(dPressure), ETG_ENUM(BOOL, bValidPressureUpdate)));

   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vOnEnvironmentData(bValidTempUpdate, dTemp, bValidPressureUpdate, dPressure);
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDataService::vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
         const t_U8 cou8ParkModeRestrictionInfo, const t_U8 cou8DriveModeRestrictionInfo)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vSetFeatureRestrictions() Entered with Device Category = %d", ETG_ENUM(DEVICE_CATEGORY,
            enDevCategory)));

   t_U8 u8ServiceLstIndx = enGetDataServiceIndex(enDevCategory);
   if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   {
      //send park/drive restriction to corresponding device category
      m_poDataSvcHandlersLst[u8ServiceLstIndx]->vSetFeatureRestrictions(cou8ParkModeRestrictionInfo,
               cou8DriveModeRestrictionInfo);
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostPositionDataResult(...)
 ***************************************************************************/
t_Void spi_tclDataService::vPostPositionDataResult(trDeadReckoningData rDeadReckoningData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostPositionDataResult() Entered"));
   vOnGPSData(rDeadReckoningData);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostSensorDataResult(...)
 ***************************************************************************/
t_Void spi_tclDataService::vPostSensorDataResult(trSensorData rSensorData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostSensorDataResult() Entered"));
   vOnSensorData(rSensorData);
}

/*************************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostAccSensorDataResult(...)
 *************************************************************************************************************/
t_Void spi_tclDataService::vPostAccSensorDataResult(const std::vector<trAccSensorData> &corfrAccSensorData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostAccSensorDataResult() Entered"));
   vOnAccSensorData(corfrAccSensorData);
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostGyroSensorDataResult(...)
 ******************************************************************************************************/
t_Void spi_tclDataService::vPostGyroSensorDataResult(const std::vector<trGyroSensorData> &corfrGyroSensorData)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostGyroSensorDataResult() Entered"));
   vOnGyroSensorData(corfrGyroSensorData);
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostNavDataResult(...)
 ******************************************************************************************************/
t_Void spi_tclDataService::vPostNavDataResult(t_Bool bStartLocData,
         const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList, t_U8 u8RequestedBy)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostNavDataResult() Entered"));
   vStartStopLocationData(bStartLocData, rfcoNmeaSentencesList, u8RequestedBy);

}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostGPRMCDataStatusValues(...)
 ******************************************************************************************************/
t_Void spi_tclDataService::vPostGPRMCDataStatusValues(t_U32 u32DeviceId, t_U8 u8GPRMCDataStatusValues)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostGPRMCDataStatusValues() Entered"));

   //! Send data to device's DataService handler.
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vPostGPRMCDataStatusValues(u32DeviceId, u8GPRMCDataStatusValues);
      }
   } //for (t_U8 u8Index = 0;...)
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSetGeoCoordinateSystem(...)
 ******************************************************************************************************/
t_Void spi_tclDataService::vSetGeoCoordinateSystem(tenGeoCoordinateSystemType enGeoCoordinateSystemType)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vSetGeoCoordinateSystem() Entered with Geo Coordinate System = %d", enGeoCoordinateSystemType));

   //! Send data to device's DataService handler
   for (t_U8 u8Index = 0; u8Index < e8_DATASVC_INDEX_SIZE; ++u8Index)
   {
      if (NULL != m_poDataSvcHandlersLst[u8Index])
      {
         m_poDataSvcHandlersLst[u8Index]->vSetGeoCoordinateSystem(enGeoCoordinateSystemType);
      }
   } //for (t_U8 u8Index = 0;...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSetDataServiceSettingsInstance()
 ***************************************************************************/
t_Void spi_tclDataService::vSetDataServiceSettingsInstance(spi_tclDataServiceSettingsIntf* poDataServiceSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclDataService::vSetDataServiceSettingsInstance entered"));
   if (NULL != poDataServiceSettingsIntf)
   {
      ETG_TRACE_USR1(("spi_tclDataService::vSetDataServiceSettingsInstance: Setting m_poDataServiceSettings val"));
      m_poDataServiceSettings = poDataServiceSettingsIntf;
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vSetGeneralRestrictions(...)
 ***************************************************************************/
t_Void spi_tclDataService::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory, t_U16 u16GeneralRestrInfo)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vSetGeneralRestrictions() Entered with Device Category = %d", ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCategory)));

   t_U8 u8ServiceLstIndx = enGetDataServiceIndex(enDeviceCategory);
   if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   {
      //send the ETC status
      m_poDataSvcHandlersLst[u8ServiceLstIndx]->vSetGeneralRestrictions(u16GeneralRestrInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclResourceMngr::vSetETCStatus(t_Bool bETCStatus)
 ***************************************************************************/
t_Void spi_tclDataService::vSetETCStatus(t_Bool bETCStatus)
{
   ETG_TRACE_USR1(("spi_tclDataService::vSetETCStatus entered"));
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vSetETCStatus(bETCStatus);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   ETG_TRACE_USR1(("spi_tclDataService::vSetETCStatus left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vGetDataServiceConfig()
 ***************************************************************************/
t_Void spi_tclDataService::vGetDataServiceConfig()
{
   ETG_TRACE_USR1(("spi_tclDataService::vGetDataServiceConfig entered"));
   if (NULL != m_poDataServiceSettings)
   {
      m_poDataServiceSettings->enGetGeoCoordinateSystem();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnClockData(const t_U8 cou8TimeFormat)
 ***************************************************************************/
t_Void spi_tclDataService::vOnClockData(const t_U8 cou8TimeFormat)
{
   ETG_TRACE_USR4(("spi_tclDataService::vOnClockData: TimeFormat = %u", cou8TimeFormat));
   ETG_TRACE_USR1(("SelectDevCategory =%d", m_enSelDevCategory));
   //! Send data to Selected device's DataService handler. At present, vOnClockData is only for mirrorlink.
   //So,m_enSelDevCategory is e8DEV_TYPE_MIRRORLINK and u8ServiceLstIndx is e8_DATASVC_ML_INDEX
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);
      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnTimeFormat(cou8TimeFormat);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vPostClockResult(const t_U8 cou8TimeFormat)
 ***************************************************************************/
t_Void spi_tclDataService::vPostClockResult(const t_U8 cou8TimeFormat)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vPostClockResult() Entered"));
   vOnClockData(cou8TimeFormat);
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting)
 ***************************************************************************/
t_Void spi_tclDataService::vOnSendLanguageSetting(tenLanguageSetting enLanguageSetting)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclDataService::vOnSendLanguageSetting entered with vLanguageSetting = %d", enLanguageSetting));
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);

      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnSendLanguageSetting(enLanguageSetting);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   ETG_TRACE_USR1(("spi_tclDataService::vOnSendLanguageSetting left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState)
 ***************************************************************************/
t_Void spi_tclDataService::vOnVehiclePowerModeState(tenVehiclePowerMode enVehiclePowerModeState)
{
   ETG_TRACE_USR4(("spi_tclDataService::vOnVehiclePowerModeState entered with VehiclePower Mode as %d", ETG_ENUM(VEHICLE_POWER_MODE,
            enVehiclePowerModeState)));

   //! Send vehicle Power mode state to MLDataService (Particularly to A+T box only)
   ETG_TRACE_USR1(("SelectDevCategory =%d", m_enSelDevCategory));
   //! Send data to Selected device's DataService handler. At present, vOnVehiclePowerModeState is only for mirrorlink.
   //So,m_enSelDevCategory is e8DEV_TYPE_MIRRORLINK and u8ServiceLstIndx is e8_DATASVC_ML_INDEX
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);
      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         ETG_TRACE_USR1(("ML Null"));
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnVehiclePowerMode(enVehiclePowerModeState);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData)
 ***************************************************************************/
t_Void spi_tclDataService::vOnAmbientColorData(const trAmbientColorData& corfrAmbientColorData)
{
   ETG_TRACE_USR2(("[DESC]:vOnAmbientColorData() Contains RGBAlpha details as follows "));
   ETG_TRACE_USR2(("[DESC]:vOnAmbientColorData() Contains Red Value %2f", corfrAmbientColorData.fRed));
   ETG_TRACE_USR2(("[DESC]:vOnAmbientColorData() Contains Green Value %2f", corfrAmbientColorData.fGreen));
   ETG_TRACE_USR2(("[DESC]:vOnAmbientColorData() Contains Blue Value %2f", corfrAmbientColorData.fBlue));
   ETG_TRACE_USR2(("[DESC]:vOnAmbientColorData() Contains Alpha Value %2f", corfrAmbientColorData.fAlpha));

   //! Send vehicle Power mode state to MLDataService (Particularly to A+T box only)
   ETG_TRACE_USR1(("SelectDevCategory =%d", m_enSelDevCategory));
   //! Send data to Selected device's DataService handler. At present, vOnAmbientColorData is only for mirrorlink.
   //So,m_enSelDevCategory is e8DEV_TYPE_MIRRORLINK and u8ServiceLstIndx is e8_DATASVC_ML_INDEX
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);
      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnAmbientColor(corfrAmbientColorData);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDataService::vDistanceToEmptyTank(...)
 ***************************************************************************/
t_Void spi_tclDataService::vDistanceToEmptyTank(t_S32 s32DistToEmptyTank)
{
   ETG_TRACE_USR1(("SelectDevCategory =%d", m_enSelDevCategory));
   //So,m_enSelDevCategory is e8DEV_TYPE_MIRRORLINK and u8ServiceLstIndx is e8_DATASVC_ML_INDEX
   if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)
   {
      t_U8 u8ServiceLstIndx = enGetDataServiceIndex(m_enSelDevCategory);
      if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
      {
         m_poDataSvcHandlersLst[u8ServiceLstIndx]->vOnDistanceToEmptyTank(s32DistToEmptyTank);
      } //if (NULL != m_poDataSvcHandlersLst[u8ServiceLstIndx])
   } // if (e8DEV_TYPE_UNKNOWN != m_enSelDevCategory)   
}
//lint ?restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
