/*!
 *******************************************************************************
 * \file             spi_tclLegalDisclaimer.cpp
 * \brief            Handles legal disclaimer requirements for SPI technology
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles legal disclaimer requirements for SPI technology
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 28.10.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclLegalDisclaimer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclLegalDisclaimer.cpp.trc.h"
#endif
#endif

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::spi_tclLegalDisclaimer
 ***************************************************************************/
spi_tclLegalDisclaimer::spi_tclLegalDisclaimer(spi_tclDeviceListIntf* poDeviceList, spi_tclLegalDisclaimerResp* pResp,
         spi_tclMediatorIntf *poMediator) :
         spi_tclSelectionIntf(e32COMPID_LEGALDISCLAIMER), m_pDeviceList(poDeviceList), m_pLegalDiscResp(pResp),
         m_pMediator(poMediator)
{
   ETG_TRACE_USR1((" spi_tclLegalDisclaimer::spi_tclLegalDisclaimer() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::~spi_tclLegalDisclaimer
 ***************************************************************************/

spi_tclLegalDisclaimer::~spi_tclLegalDisclaimer()
{
   ETG_TRACE_USR1((" spi_tclLegalDisclaimer::~spi_tclLegalDisclaimer() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::bValidateAuthorization
 ***************************************************************************/
t_Bool spi_tclLegalDisclaimer::bValidateAuthorization(const t_U32 cou32DeviceHandle)
{
   //! check if it is a valid request
   t_Bool bValidRequest = false;
   m_oDeviceAuthLock.s16Lock();
   if (m_mapAuthorizationStatus.end() != m_mapAuthorizationStatus.find(cou32DeviceHandle))
   {
      bValidRequest = true;
   }
   m_oDeviceAuthLock.vUnlock();

   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::bValidateAuthorization() Validity of authorization Request from HMI = %d ", ETG_ENUM(BOOL, bValidRequest)));
   return bValidRequest;
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::vSetDeviceAuthorization
 ***************************************************************************/

t_Void spi_tclLegalDisclaimer::vSetDeviceAuthorization(const t_U32 cou32DeviceHandle,
         tenUserAuthorizationStatus enUserAuthStatus)
{
   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::vSetDeviceAuthorization: Authorization Request received for DeviceID = 0x%x "
            "with Authorization status = %d ", cou32DeviceHandle, ETG_ENUM(AUTHORIZATION_STATUS,enUserAuthStatus)));

   //! check if it is a valid request
   t_Bool bValidRequest = bValidateAuthorization(cou32DeviceHandle);
   if ((NULL != m_pDeviceList) && (true == bValidRequest))
   {
      t_U32 u32SelectedDevice = m_pDeviceList->u32GetSelectedDevice();
      trDeviceInfo rDeviceInfo;
      t_Bool bValidDevice = m_pDeviceList->bGetDeviceInfo(cou32DeviceHandle, rDeviceInfo);

      //!update the list with new authorization status
      m_pDeviceList->vSetUserAuthorizationStatus(cou32DeviceHandle, enUserAuthStatus);
      vUpdateDeviceAuthorization(cou32DeviceHandle, enUserAuthStatus);

      if ((true == bValidDevice) && (0 == u32SelectedDevice)
               && (e8_USER_AUTH_UNKNOWN == rDeviceInfo.enUserAuthorizationStatus))
      {
         //! Proceed with device selection if user has authorized the device
         tenErrorCode enErrorCode =
                  (e8_USER_AUTH_AUTHORIZED == enUserAuthStatus) ?
                           e8NO_ERRORS: e8OPERATION_CANCELLED_BY_USER;
         vSendSelectDeviceResponse(enErrorCode);
      }
      //! If there is a device already selected then deselect the device if user unauthorizes it
      else if (e8_USER_AUTH_CANCELLED == enUserAuthStatus)
      {
         vSendSelectDeviceResponse(e8OPERATION_CANCELLED_BY_USER);
      }
      else if (e8_USER_AUTH_UNAUTHORIZED == enUserAuthStatus)
      {
         if (NULL != m_pMediator)
         {
            m_pMediator->vPostAutoDeviceSelection(u32SelectedDevice, e8DEVCONNREQ_DESELECT);
         } // if (NULL != poMediator)
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::vSelectDevice
 ***************************************************************************/

t_Void spi_tclLegalDisclaimer::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("[DESC]  spi_tclLegalDisclaimer::vSelectDevice() : Selection Request received by Legal Disclaimer "));
   //! Abort selection if none of the projection technologies are enabled for the reported devicetype
   if (false == bIsProjectionEnabled(corfrSelectReq.m_enDeviceType))
   {
      vSendSelectDeviceResponse(e8PROJECTION_NOT_ENABLED);
   }
   //! Proceed with selection if authorization is not required for this device (Known Device)
   else if(false == bIsAuthorizationRequired(corfrSelectReq.m_u32DeviceHandle, corfrSelectReq.m_enDeviceType))
   {
      vSendSelectDeviceResponse(e8NO_ERRORS);
   }
   //! Send HMI Authorization request if a new device is connected to the system
   else if(NULL != m_pDeviceList)
   {
      m_pDeviceList->vSetUserAuthorizationStatus(corfrSelectReq.m_u32DeviceHandle, e8_USER_AUTH_UNKNOWN);
      vUpdateDeviceAuthorization(corfrSelectReq.m_u32DeviceHandle, e8_USER_AUTH_UNKNOWN);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::vSelectDeviceResult
 ***************************************************************************/

t_Void spi_tclLegalDisclaimer::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::vSelectDeviceResult() Selection Result received by Legal Disclaimer "));
   //! Send Authroization status update to clear list when user disconnects the device
   if(e8DEVICE_DISCONNECTED_DURING_SELECTION == enErrorCode)
   {
      vUpdateDeviceAuthorization(corfrSelectReq.m_u32DeviceHandle, e8_USER_AUTH_ABORTED);
   }
   m_oDeviceAuthLock.s16Lock();
   if(m_mapAuthorizationStatus.end() != m_mapAuthorizationStatus.find(corfrSelectReq.m_u32DeviceHandle))
   {
      m_mapAuthorizationStatus.erase(corfrSelectReq.m_u32DeviceHandle);
   }
   m_oDeviceAuthLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::vSendSelectDeviceResponse
 ***************************************************************************/

t_Void spi_tclLegalDisclaimer::vSendSelectDeviceResponse(tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::vSendSelectDeviceResponse() Responding to Device selection Request "));
   if(NULL != m_pMediator)
   {
      m_pMediator->vPostSelectDeviceRes(enGetComponentID(), enErrorCode);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::bIsAuthorizationRequired
 ***************************************************************************/
t_Bool spi_tclLegalDisclaimer::bIsAuthorizationRequired(const t_U32 cou32DeviceHandle, tenDeviceType enDeviceType)
{
   SPI_INTENTIONALLY_UNUSED(enDeviceType);
   t_Bool bisAuthorizationReqd = false;
   //! Is the projection status set to "user confirmation needed"?
   if (NULL != m_pDeviceList)
   {
      tenUserAuthorizationStatus enAuthStatus = m_pDeviceList->enGetUserAuthorizationStatus(cou32DeviceHandle);
      bisAuthorizationReqd=(e8_USER_AUTH_UNKNOWN == enAuthStatus) ;
   }

   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::bIsAuthorizationRequired () Authorization required  status for Device ID 0x%x is %d",
               cou32DeviceHandle, ETG_ENUM(BOOL, bisAuthorizationReqd)));
   return bisAuthorizationReqd;
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::bIsAuthorizationRequired
 ***************************************************************************/
t_Void spi_tclLegalDisclaimer::vUpdateDeviceAuthorization(const t_U32 cou32DeviceHandle, tenUserAuthorizationStatus enUserAuthStatus)
{
   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::vUpdateDeviceAuthorization(): Updating device authorization status to HMI for Device 0x%x ", cou32DeviceHandle));

   tenDeviceType enDevType = e8_UNKNOWN_DEVICE;
   //! Send the authorization status to the response class
   if (NULL != m_pDeviceList)
   {
      enDevType = m_pDeviceList->enGetDeviceType(cou32DeviceHandle);
   }
   if(NULL != m_pLegalDiscResp)
   {
      m_oDeviceAuthLock.s16Lock();
      m_mapAuthorizationStatus[cou32DeviceHandle] = enUserAuthStatus;
      m_oDeviceAuthLock.vUnlock();
      m_pLegalDiscResp->vUpdateDeviceAuthorizationStatus(cou32DeviceHandle,enDevType, enUserAuthStatus);
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer::bIsProjectionEnabled
 ***************************************************************************/
t_Bool spi_tclLegalDisclaimer::bIsProjectionEnabled(tenDeviceType enDeviceType)
{
   t_Bool bIsProjectionEnabled = false;

   if (NULL != m_pDeviceList)
   {
      //! Check if any projection technology is enabled for android device
      if (e8_ANDROID_DEVICE == enDeviceType)
      {
         tenEnabledInfo enAndroidAutoEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enMirrorlinkEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enMySPINEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enCarlifeEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enOnCarEnabled = e8USAGE_UNKNOWN;
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_ANDROIDAUTO, enAndroidAutoEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MIRRORLINK, enMirrorlinkEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCarlifeEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_ONCAR, enOnCarEnabled, enDeviceType);
         bIsProjectionEnabled = ((e8USAGE_DISABLED != enMirrorlinkEnabled)
                  || (e8USAGE_DISABLED != enAndroidAutoEnabled)
				  || (e8USAGE_DISABLED != enMySPINEnabled)
				  || (e8USAGE_DISABLED != enCarlifeEnabled)
				  || (e8USAGE_DISABLED != enOnCarEnabled));
      }
      //! Check if any projection technology is enabled for apple device
      else if (e8_APPLE_DEVICE == enDeviceType)
      {
         tenEnabledInfo enCarplayEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enMySPINEnabled = e8USAGE_UNKNOWN;
         tenEnabledInfo enCarlifeEnabled = e8USAGE_UNKNOWN;
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_DIPO, enCarplayEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_MYSPIN, enMySPINEnabled, enDeviceType);
         m_pDeviceList->bGetDevProjUsage(e8DEV_TYPE_CARLIFE, enCarlifeEnabled, enDeviceType);
         bIsProjectionEnabled = ((e8USAGE_DISABLED != enCarplayEnabled)
				  || (e8USAGE_DISABLED != enMySPINEnabled)
				  || (e8USAGE_DISABLED != enCarlifeEnabled));
      }
   }
   ETG_TRACE_USR1(("[DESC] spi_tclLegalDisclaimer::bIsProjectionEnabled() The projection technologies enabled status for DeviceType %d is  %d ",
            ETG_ENUM(DEVICE_TYPE, enDeviceType), ETG_ENUM(BOOL, bIsProjectionEnabled)));
   return bIsProjectionEnabled;
}
