/*!
 *******************************************************************************
 * \file             spi_tclDeviceSelector.cpp
 * \brief            Handles select and deselection of device
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles select and deselection of device
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 10.12.2014 | Shihabudheen P M             | Changed for blocking device usage 
 preference updates during
 select/deselect is in progress.
 05.02.2015 |  Ramya Murthy                | Changes to set different intervals in
 component response timer for different SPI components

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclDevSelResp.h"
#include "spi_tclDeviceSelector.h"
#include "spi_tclSelectionValidator.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclDeviceSelector.cpp.trc.h"
#endif
#endif

static t_U32 su32CompResponseTime10s = 10000;
static t_U32 su32CompResponseTime15s = 15000;
static t_U32 su32CompResponseTime25s = 25000;
static t_U32 su32CompResponseTime26s = 26000;
static t_U32 su32CompResponseTime60s = 60000;
//! Changing connection response time to 25 seconds as mirrorlink switch might sometime take >20 seconds
static t_U32 su32ConnCompResponseTimeinms = su32CompResponseTime25s;
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

static const tenCompID azSelectionSequence[][e32COMPID_SIZE] =
#define SPI_DEVICE_SELECTOR_SELECTION_SEQUENCE
#include "SelectionConfiguration.cfg"
#undef SPI_DEVICE_SELECTOR_SELECTION_SEQUENCE

static const tenCompID azDeSelectionSequence[][e32COMPID_SIZE] =
#define SPI_DEVICE_SELECTOR_DESELECTION_SEQUENCE
#include "SelectionConfiguration.cfg"
#undef SPI_DEVICE_SELECTOR_DESELECTION_SEQUENCE

t_U32 spi_tclDeviceSelector::m_u32CurrSelectedDevice = 0;

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::spi_tclDeviceSelector
 ***************************************************************************/
spi_tclDeviceSelector::spi_tclDeviceSelector(spi_tclDevSelResp *poRespInterface, spi_tclDeviceListIntf *poDeviceList,
         spi_tclMediatorIntf* poMediator, spi_tclConnSettingsIntf *poConnSettings,
         std::map<tenCompID, spi_tclSelectionIntf*> &rfrComponents) :
                  m_poMediator(poMediator),
                  m_poDevSelResp(poRespInterface),
                  /* m_u32CurrSelectedDevice(0),*/
                  m_bDeviceSwitchRequest(false),
                  m_bIsDevSelectorBusy(false),
                  m_enSelectedComp(e32COMPID_UNKNOWN),
                  m_rCompRespTimerID(0),
                  m_u32RecentlyConnectedDevice(0),
                  m_pDeviceList(poDeviceList),
                  m_poConnSettings(poConnSettings),
                  m_mapComponents(rfrComponents)

{
   ETG_TRACE_USR1(("Creating spi_tclDeviceSelector "));
   m_u32CurrSelectedDevice = 0;
   //bInitialize();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::~spi_tclDeviceSelector
 ***************************************************************************/
spi_tclDeviceSelector::~spi_tclDeviceSelector()
{
   ETG_TRACE_USR1(("Destroying spi_tclDeviceSelector  "));

   Timer* poTimer = Timer::getInstance();
   if ((0 != m_rCompRespTimerID) && (NULL != poTimer))
   {
      poTimer->CancelTimer(m_rCompRespTimerID);
   }

   m_poDevSelResp = NULL;
   m_poMediator = NULL;
   m_rCompRespTimerID = 0;
   m_bDeviceSwitchRequest = false;
   m_bIsDevSelectorBusy = false;
   m_u32CurrSelectedDevice = 0;
   m_enSelectedComp = e32COMPID_UNKNOWN;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::bInitialize
 ***************************************************************************/
t_Bool spi_tclDeviceSelector::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::bInitialize "));
   vRegisterCallbacks();
   //! Even if starting timer fails, initialize can return true as this
   //! doen't affect major functionality
   //! Currently timer is not used by any projects
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSelector::vUnInitialize()
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vUnInitialize "));
   //! Add code: Currently callback deregistration not required.
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSelector::vLoadSettings()
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vLoadSettings "));
   //! Cancel component response timer if there is a state change from ON to OFF
   Timer* poTimer = Timer::getInstance();
   if ((0 != m_rCompRespTimerID) && (NULL != poTimer))
   {
      poTimer->CancelTimer(m_rCompRespTimerID);
   }
   //! Reset the flags on loadsettings
   m_bIsDevSelectorBusy = false;
   m_enSelectedComp = e32COMPID_UNKNOWN;
   m_bDeviceSwitchRequest = false;
   su32ConnCompResponseTimeinms = su32CompResponseTime25s;
   //!load selection sequences
   vLoadSelectionSequence();
   vLoadDeselectionSequence();

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDeviceSelector::vSaveSettings()
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vSaveSettings "));

   //! Cancel device selection as no response is expected form connection manager
   //! after savesettings. This is because there is no updates from mediaplayer
   if (true == m_bIsDevSelectorBusy)
   {
      vReceiveSelectDeviceResCb(m_enSelectedComp, e8UNKNOWN_ERROR);
   }
}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vRestoreSettings "));
   //! Cancel the device selection on restore factory setting
   if (true == m_bIsDevSelectorBusy)
   {
      vReceiveSelectDeviceResCb(m_enSelectedComp, e8OPERATION_CANCELLED_BY_FACTORY_RESET);
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclDeviceSelector::vRestoreSettings Selection Not Busy"));
      t_U32 u32CurrSelectedDevice = u32GetSelectedDeviceId();
      if (0 != u32CurrSelectedDevice)
      {
         ETG_TRACE_USR1(("spi_tclDeviceSelector::vRestoreSettings u32CurrSelectedDevice -  %d", u32CurrSelectedDevice));
         vDeviceDeselectionOnFactoryReset(u32CurrSelectedDevice, e8DEVCONNREQ_DESELECT);

      }
      else
      {
         ETG_TRACE_USR1(("spi_tclDeviceSelector::vRestoreSettings u32CurrSelectedDevice -  %d"
                  ".Since No device is selected/Connected to the system, Sending the response back.", u32CurrSelectedDevice));
         //clearing the private data after factory reset
         if (NULL != m_poDevSelResp)
         {
            m_poDevSelResp->vClearPrivateDataOnfactoryReset();
         }
      }

   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vSelectDevice
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSelectDevice(trSelectDeviceRequest &rfrSelectDevInfo)
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vSelectDevice cou32DeviceHandle = 0x%x,"
            " enDevConnReq = %d , Device category = %d SelectReason = %d, Device Connection Type:%d", rfrSelectDevInfo.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            rfrSelectDevInfo.m_enDevConnReq), ETG_ENUM(DEVICE_CATEGORY, rfrSelectDevInfo.m_enDevCategory), ETG_ENUM(SELECT_REASON,
            rfrSelectDevInfo.m_enSelectionReason), ETG_ENUM(CONNECTION_TYPE, rfrSelectDevInfo.m_enDevConnType)));

   tenErrorCode enErrorCode = e8NO_ERRORS;

   if ((e8_UNKNOWN_DEVICE == rfrSelectDevInfo.m_enDeviceType) && (NULL != m_pDeviceList))
   {
      rfrSelectDevInfo.m_enDeviceType = m_pDeviceList->enGetDeviceType(rfrSelectDevInfo.m_u32DeviceHandle);
   }
   //! Error hanlding for device deselection issued for unknown device category
   if ((NULL != m_pDeviceList) && (e8DEVCONNREQ_DESELECT == rfrSelectDevInfo.m_enDevConnReq)
            && (e8DEV_TYPE_UNKNOWN == rfrSelectDevInfo.m_enDevCategory))
   {
      rfrSelectDevInfo.m_enDevCategory = m_pDeviceList->enGetDeviceCategory(m_u32CurrSelectedDevice);
   }

   if(NULL != m_pDeviceList)
   {
      m_pDeviceList->vSetDeviceSelectionReason(rfrSelectDevInfo.m_u32DeviceHandle, rfrSelectDevInfo.m_enSelectionReason);
   }

   //lock is added to avoid the synchronisation issues - 
   //This lock added as Dlink was getting detected while the selection was in progress for the Device for CP/AAP - After Ignition Cycle

   m_oSelectionValidatorLock.s16Lock();
   spi_tclSelectionValidator oDeviceValidator(m_pDeviceList, m_u32CurrSelectedDevice, m_bIsDevSelectorBusy);
   oDeviceValidator.vValidateSelection(rfrSelectDevInfo, enErrorCode);
   m_oSelectionValidatorLock.vUnlock();

   vUpdateDevSelectorBusyStatus(true, rfrSelectDevInfo.m_u32DeviceHandle);

   if (e8NO_ERRORS == enErrorCode)
   {
      //! Store the request info
      m_rSelectDevReqInfo = rfrSelectDevInfo;

      //! Set the device category for selection
      if ((NULL != m_pDeviceList) && (e8DEV_TYPE_UNKNOWN != rfrSelectDevInfo.m_enDevCategory))
      {
         m_pDeviceList->vSetDeviceCategory(rfrSelectDevInfo.m_u32DeviceHandle, rfrSelectDevInfo.m_enDevCategory);
      }

      //! Check if any other device is selected
      vEvaluateDeviceSelectionReq();

   } // if(e8NO_ERRORS == enErrorCode)
   else
   {
      if (e8RESOURCE_BUSY != enErrorCode)
      {
         vUpdateDevSelectorBusyStatus(false, rfrSelectDevInfo.m_u32DeviceHandle);
      }
      //! Post Result directly to HMI if processing the request is not required
      vSendSelectDeviceResultToHMI(rfrSelectDevInfo, enErrorCode);
      //! update current selection state to HMI
      vUpdateDevSelectionInfo(rfrSelectDevInfo.m_u32DeviceHandle, enErrorCode, rfrSelectDevInfo.m_enDevConnReq);

      //clearing the private data after factory reset, if the Error Code is OPERATION_REJECTED as A+T Box will not deselected,
      //But we have to clear the Private Data on the Factory reset
      if ((e8OPERATION_REJECTED == enErrorCode) && (e8_FACTORY_RESET == rfrSelectDevInfo.m_enSelectionReason)
               && (e8DEVCONNREQ_DESELECT == rfrSelectDevInfo.m_enDevConnReq))
      {
         if (NULL != m_poDevSelResp)
         {
            m_poDevSelResp->vClearPrivateDataOnfactoryReset();
         }

      }
   }      // else part of if(e8NO_ERRORS == enErrorCode)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vReceiveSelectDeviceResCb
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vReceiveSelectDeviceResCb(tenCompID enCompID, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector:vReceiveSelectDeviceResCb: Device selection result received from component = %d, Reported ErrorCode = %d  threadID = %d", ETG_ENUM(COMPONENT_ID,
            enCompID), ETG_ENUM(ERROR_CODE, enErrorCode), SPI_THREADID ));

   if (enCompID == m_enSelectedComp)
   {
      //! Cancel the timer if the response is received for the current component
      Timer* poTimer = Timer::getInstance();
      if ((0 != m_rCompRespTimerID) && (NULL != poTimer))
      {
         poTimer->CancelTimer(m_rCompRespTimerID);
      }
      trSelectDeviceRequest rSelDevInfo =
               (false == m_bDeviceSwitchRequest) ? m_rSelectDevReqInfo : m_rDeselectDevReqInfo;

      //! On Successful Select device result, send device selection request for next
      //! component

      tenDeviceCategory enDevCat = e8DEV_TYPE_UNKNOWN;
      if (NULL != m_pDeviceList)
      {
         enDevCat = m_pDeviceList->enGetDeviceCategory(rSelDevInfo.m_u32DeviceHandle);
      }

      //Updating the Device Category - Based on Category set by HMI through AOC popup
      if ((enDevCat != e8DEV_TYPE_UNKNOWN) && (enDevCat != e8DEV_TYPE_DEFAULT))
      {
         m_rSelectDevReqInfo.m_enDevCategory = enDevCat;
         rSelDevInfo.m_enDevCategory = enDevCat;
      }

      //! Clear the current selected component ID
      m_enSelectedComp = e32COMPID_UNKNOWN;
      //! Trigger device selection sequence if the previous component has returned success
      //! Trigger device deselection sequence even if the deselection for previous component failed
      if ((e8NO_ERRORS == enErrorCode) || (e8DEVCONNREQ_DESELECT == rSelDevInfo.m_enDevConnReq))
      {
         tenCompID enNextCompID = enGetNextComponentID(enDevCat, rSelDevInfo.m_enDevConnReq, enCompID);
         //! Check if selection request is sent to all components. If so then send the result to HMI
         //! and other subcomponents
         if (e32COMPID_UNKNOWN == enNextCompID)
         {
            vSendSelectDeviceResult(rSelDevInfo, e8NO_ERRORS);
         }
         else
         {
            vSendDeviceSelectionReq(rSelDevInfo, enNextCompID);
         }
      } // if (true == bSelDevRes)
      else
      {
         //! Send Error to HMI if device selection failure is returned
         vSendSelectDeviceResult(rSelDevInfo, enErrorCode);
      } // else case for  if (true == bSelDevRes)
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vDeviceDisconnectionCb
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vDeviceDisconnectionCb(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vDeviceDisconnectionCb"
            " Device-0x%x is disconnected .", cou32DeviceHandle));
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vDeviceDisconnectionCb:"
            " m_u32CurrSelectedDevice = 0x%x", m_u32CurrSelectedDevice));
   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_pDeviceList)
   {
      enDevCategory = m_pDeviceList->enGetDeviceCategory(cou32DeviceHandle);
   }

   if (cou32DeviceHandle == m_u32CurrSelectedDevice)
   {
      //! Send device deselection request on device disconnection
      vOnAutomaticDeviceSelection(m_u32CurrSelectedDevice, e8DEVCONNREQ_DESELECT);
      m_u32CurrSelectedDevice = 0;
   } // if(cou32DeviceHandle == m_u32CurrSelectedDevice)

   //! Cancel Device selection if disconnection is received during device selection except for mirrorlink
   //! Mirrorlink can still disconnect and reconnect when a device selection is in progress because of the switch
   else if ((true == m_bIsDevSelectorBusy) && (cou32DeviceHandle == m_rSelectDevReqInfo.m_u32DeviceHandle)
            && (e8DEV_TYPE_MIRRORLINK != enDevCategory))
   {
      vReceiveSelectDeviceResCb(m_enSelectedComp, e8DEVICE_DISCONNECTED_DURING_SELECTION);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vDeviceConnectionCb
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vDeviceConnectionCb(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vDeviceConnectionCb received for cou32DeviceHandle = 0x%x", cou32DeviceHandle));

   tenDeviceSelectionMode enSelMode = e16DEVICESEL_MANUAL;
   tenDeviceConnectionType enDeviceConnType = e8UNKNOWN_CONNECTION;

   if ((NULL != m_poConnSettings) && (NULL != m_pDeviceList))
   {
      m_u32RecentlyConnectedDevice = cou32DeviceHandle;
      tenDeviceCategory enDevCategory = m_pDeviceList->enGetDeviceCategory(cou32DeviceHandle);
      tenDeviceType enDevType = m_pDeviceList->enGetDeviceType(cou32DeviceHandle);
      enSelMode = m_poConnSettings->enGetDeviceSelectionMode(enDevCategory);
      enDeviceConnType = m_pDeviceList->enGetDeviceConnType(cou32DeviceHandle);


      trSelectDeviceRequest rSelDevReq;
      rSelDevReq.m_u32DeviceHandle = cou32DeviceHandle;
      rSelDevReq.m_enDevConnReq = e8DEVCONNREQ_SELECT;
      rSelDevReq.m_enDevCategory = enDevCategory;
      rSelDevReq.m_enDeviceType = enDevType;
      rSelDevReq.m_enDevConnType = enDeviceConnType;

      spi_tclSelectionValidator oDeviceValidator(m_pDeviceList, m_u32CurrSelectedDevice, m_bIsDevSelectorBusy);

      //! Automatically select the device if validation is successful
      if ((e16DEVICESEL_MANUAL != enSelMode) && (true == oDeviceValidator.bDeviceValidforAutoSelection(rSelDevReq)))
      {
         vOnAutomaticDeviceSelection(cou32DeviceHandle, e8DEVCONNREQ_SELECT);
      }
      else
      {
         //! Inform connection manager about not selecting the connected device only if the Selection  is not in PROGRESS.
         //! This is required to answer to roleswitch required request from mediaplayer
         //! TODO - Device Connection Type should be maintained differently for the different transport
            m_oLockComponents.s16Lock();
            for(auto itCompMap = m_mapComponents.begin() ; itCompMap != m_mapComponents.end(); itCompMap++ )
            {
               itCompMap->second->vUpdateSelectionState(m_u32RecentlyConnectedDevice, e8_SELECTION_STATE_DEVICE_NOT_SELECTED ,enDeviceConnType);
            }
            m_oLockComponents.vUnlock();

            m_u32RecentlyConnectedDevice = 0;

      }

      ETG_TRACE_USR4(("[PARAM] spi_tclDeviceSelector::vDeviceConnectionCb m_u32CurrSelectedDevice = 0x%x, m_bIsDevSelectorBusy =%d", cou32DeviceHandle, ETG_ENUM(BOOL,
               m_bIsDevSelectorBusy)));
   } //if (NULL != m_pDeviceList)
}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vApplySelectionStrategy
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vApplySelectionStrategy(tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSelector::vApplySelectionStrategy : Currently Selected device = %d enDeviceCategory=%d", m_u32CurrSelectedDevice, ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCategory)));
   t_U32 u32SelectedDevice = 0;

   tenDeviceSelectionMode enSelMode = e16DEVICESEL_MANUAL;
   tenDeviceSelectionMode enTechnologySpecificSelMode = e16DEVICESEL_MANUAL;
   if (NULL != m_poConnSettings)
   {
      enSelMode = m_poConnSettings->enGetDeviceSelectionMode(e8DEV_TYPE_DEFAULT);
      enTechnologySpecificSelMode = m_poConnSettings->enGetDeviceSelectionMode(enDeviceCategory);
   }

   ETG_TRACE_USR4(("[PARAM] vApplySelectionStrategy Selection mode = %d ", ETG_ENUM(SELECTION_MODE, enSelMode)));

   if (e16DEVICESEL_MANUAL != enSelMode)
   {
      if (e8DEV_TYPE_UNKNOWN != enDeviceCategory)
      {
         u32SelectedDevice = u32GetLastConnectedDevice(enDeviceCategory);
      }
      else
      {
         u32SelectedDevice = u32GetLastConnectedDevice();
      }

   }

   //! Send selection request for the chosen device if no other device is already selected
   if ((0 != u32SelectedDevice) && (0 == m_u32CurrSelectedDevice)
            && ((e16DEVICESEL_MANUAL != enTechnologySpecificSelMode)
                     && (e16DEVICESEL_UNKNOWN != enTechnologySpecificSelMode)))
   {
      vOnAutomaticDeviceSelection(u32SelectedDevice, e8DEVCONNREQ_SELECT);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSelector::vContinueWithSelection
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vContinueWithSelection(t_U32 u32DeviceHandle, tenDeviceConnectionReq enDevConnReq,
         tenDeviceCategory enDevCategory, tenSelectReason enSelectReason)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSelector::vContinueWithSelection: Received %d request for DeviceID:0x%x,Category:%d,SelectReason:%d ", ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), u32DeviceHandle, enDevCategory, ETG_ENUM(SELECT_REASON, enSelectReason)));

   tenDeviceConnectionType enDevConnType = enGetDeviceConnectionType(u32DeviceHandle, enDevConnReq, enDevCategory);
   if (NULL != m_pDeviceList)
   {
      if (e8DEV_TYPE_UNKNOWN == enDevCategory)
      {
         enDevCategory = m_pDeviceList->enGetDeviceCategory(u32DeviceHandle);
      }
   } //if (NULL != m_pDeviceList)

   //! trigger automatic device selection if device selector is not busy and device handle is valid
   if ((0 != u32DeviceHandle) && (false == m_bIsDevSelectorBusy))
   {
      //! Dummy user context sent as select device is internally triggered
      trUserContext rUsrCntxt;
      trSelectDeviceRequest rSelDeviceReq;
      rSelDeviceReq.m_u32DeviceHandle = u32DeviceHandle;
      rSelDeviceReq.m_enDevConnReq = enDevConnReq;
      rSelDeviceReq.m_enDevConnType = enDevConnType;
      rSelDeviceReq.m_enDevCategory = enDevCategory;
      rSelDeviceReq.m_enSelectionReason = enSelectReason;
      rSelDeviceReq.m_corUsrCntxt = rUsrCntxt;
      vSelectDevice(rSelDeviceReq);
   } // if (0 != cou32DeviceHandle)

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vOnAutomaticDeviceSelection
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vOnAutomaticDeviceSelection(const t_U32 cou32DeviceHandle,
         tenDeviceConnectionReq enDevConnReq, tenDeviceCategory enDevCategory, tenSelectReason enSelectReason)
{
   ETG_TRACE_USR1(("[DESC] spi_tclDeviceSelector::vOnAutomaticDeviceSelection: Received %d request without user trigger for DeviceID:0x%x,Category:%d,SelectReason:%d", ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), cou32DeviceHandle, enDevCategory, ETG_ENUM(SELECT_REASON, enSelectReason)));

   vContinueWithSelection(cou32DeviceHandle, enDevConnReq, enDevCategory, enSelectReason);
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDeviceSelector::bIsConnRetrialRequired
 ***************************************************************************/
t_Bool spi_tclDeviceSelector::bIsConnRetrialRequired(t_U32 u32DeviceID, tenDeviceConnectionReq enDevConnReq,
         tenDeviceCategory enDevCategory, tenSelectReason enSelectReason)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::bIsConnRetrialRequired:DevID:0x%x,ConnectionRequest:%d,Category:%d,SelectReason:%d", u32DeviceID, ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), enDevCategory, ETG_ENUM(SELECT_REASON, enSelectReason)));

   t_Bool bIsRetrialReq = false;

   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_USB;
   if (NULL != m_pDeviceList)
   {
      enDevCategory = m_pDeviceList->enGetDeviceCategory(u32DeviceID);
      enSessionTransport = m_pDeviceList->enGetSessionTransport(u32DeviceID);
   }

   //currently re-trial is required only for CarPlay wireless session end.
   t_Bool bIsDeselectionForCPW = ((e8DEV_TYPE_DIPO == enDevCategory) && (e8DEVCONNREQ_DESELECT == enDevConnReq)
            && (e8_SESSION_TRANSPORT_WIFI == enSessionTransport));

   if (true == bIsDeselectionForCPW)
   {
      //Retrial is required only if the deselection is triggered because of unknown reason or due to WiFi setting change on HU or iPhone or WiFi intereference.
      //This is valid for only CPW as of now. - Retry is not needed if the Deselect is because of HMI Trigger/User trigger/SPM State off or Factory reset
      t_Bool bIsRetryValidForCurSelectReason = ((e8_REASON_UNKNOWN == enSelectReason)
               || (e8_HU_WIFI_AP_OFF == enSelectReason) || (e8_STATION_DISCONNECTED == enSelectReason));

      //Check if the deselect is already progress for the device.	
      trDeviceSelectionInfo rDeviceSelectionInfo;
      if (NULL != m_pDeviceList)
      {
         m_pDeviceList->vGetDeviceSelectionInfo(u32DeviceID, rDeviceSelectionInfo);
      }
      t_Bool bIsDeselectInProgress = ((e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS
               == rDeviceSelectionInfo.enDeviceSelectionState)
               || (e8_SELECTION_STATE_DEVICE_NOT_SELECTED == rDeviceSelectionInfo.enDeviceSelectionState));

      //Perform retrial only if the device is connected over USB and de-selection is not progress for the same device.
      t_Bool bIsDeviceUSBConnected = false;
      if (NULL != m_pDeviceList)
      {
         bIsDeviceUSBConnected = (e8USB_CONNECTED == m_pDeviceList->enGetDeviceConnType(u32DeviceID))
                  && (e8DEV_CONNECTED == m_pDeviceList->enGetDeviceConnStatus(u32DeviceID));
      }

      ETG_TRACE_USR2(("spi_tclDeviceSelector::bIsConnRetrialRequired:Is de-selection in progress for the device:%d, De-select reason:%d", ETG_ENUM(BOOL,
               bIsDeselectInProgress), m_rDeselectDevReqInfo.m_enSelectionReason));

      ETG_TRACE_USR2(("spi_tclDeviceSelector::bIsConnRetrialRequired:DeviceDeselectonInProgress:%d,is retry valid for Select Reason:%d, is USB connected:%d", ETG_ENUM(BOOL,
               bIsDeselectInProgress), ETG_ENUM(BOOL, bIsRetryValidForCurSelectReason), ETG_ENUM(BOOL,
               bIsDeviceUSBConnected)));

      if ((false == bIsDeselectInProgress) && (true == bIsRetryValidForCurSelectReason)
               && (true == bIsDeviceUSBConnected))
      {
         bIsRetrialReq = true;
      }
   }
   ETG_TRACE_USR4(("spi_tclDeviceSelector::bIsConnRetrialRequired:RetVal:%d", ETG_ENUM(BOOL, bIsRetrialReq)));
   return bIsRetrialReq;

}

/***************************************************************************
 ** FUNCTION:  tenDeviceConnectionType spi_tclDeviceSelector::enGetDeviceConnectionType
 ***************************************************************************/
tenDeviceConnectionType spi_tclDeviceSelector::enGetDeviceConnectionType(t_U32 cou32DeviceHandle,
         tenDeviceConnectionReq enDevConnReq, tenDeviceCategory enDevCategory)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::enGetDeviceConnectionType:DevID:0x%x,ConnectionRequest:%d,Category:%d", cou32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), enDevCategory));

   tenDeviceConnectionType enDevConnType = e8UNKNOWN_CONNECTION;
   if (NULL != m_pDeviceList)
   {
      if ((e8DEV_TYPE_DIPO == enDevCategory) && (e8DEVCONNREQ_DESELECT == enDevConnReq))
      {
         tenSessionTransport enSessionTransport = m_pDeviceList->enGetSessionTransport(cou32DeviceHandle);
         switch (enSessionTransport)
         {
            case e8_SESSION_TRANSPORT_USB:
            {
               enDevConnType = e8USB_CONNECTED;
            }
               break;
            case e8_SESSION_TRANSPORT_WIFI:
            default:
            {
               enDevConnType = e8WIRELESS_CONNECTED;
            }
               break;
         }
      }
      else
      {
         enDevConnType = m_pDeviceList->enGetDeviceConnType(cou32DeviceHandle);
      }
   } //if (NULL != m_pDeviceList)

   return enDevConnType;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vOnSetUserDeselect
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vOnSetUserDeselect(const t_U32 cou32DeviceHandle, t_Bool bUserDeselect)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vOnSetUserDeselect: Setting user deselection flag for DeviceHandle = 0x%x ", cou32DeviceHandle));

   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->vSetUserDeselectionFlag(cou32DeviceHandle, bUserDeselect);
   } //if (NULL != m_pDeviceList)
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclDeviceSelector::u32GetSelectedDeviceHandle()
 ***************************************************************************/
t_U32 spi_tclDeviceSelector::u32GetSelectedDeviceHandle() const
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::u32GetSelectedDeviceHandle = 0x%x ", m_u32CurrSelectedDevice));
   return m_u32CurrSelectedDevice;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDeviceSelector::vSetDeviceSelectionMode()
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSetDeviceSelectionMode(tenDeviceSelectionMode enDeviceSelectionMode,
         tenDeviceCategory enTechnologyType)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vSetDeviceSelectionMode [DESC] Request to set the selection mode to %d ", ETG_ENUM(DEVICE_SELECTION_MODE,
            enDeviceSelectionMode)));

   if (NULL != m_poConnSettings)
   {
      //! Trigger automatic device selection if HMI has changes from manula to automatic selection mode
      tenDeviceSelectionMode enStoredDevSelMode = m_poConnSettings->enGetDeviceSelectionMode(enTechnologyType);
      if ((enDeviceSelectionMode != enStoredDevSelMode))
      {
         //! Store the setting if selection mode changes
         m_poConnSettings->vSetDeviceSelectionMode(enDeviceSelectionMode, enTechnologyType);
         //! Apply selection strategy if the device selection mode has changed from manual to automatic
         if (e16DEVICESEL_AUTOMATIC == enDeviceSelectionMode)
         {
            vApplySelectionStrategy();
         }
      }
   }
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vRegisterCallbacks()
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vRegisterCallbacks() Entered"));
   /*lint -esym(40,fvOnSelectDeviceRes) fvOnSelectDeviceRes Undeclared identifier */
   /*lint -esym(40,fvOnDeviceDisconnection)fvOnDeviceDisconnection Undeclared identifier */
   /*lint -esym(40,fvAutoSelectDevice) fvAutoSelectDevice Undeclared identifier */
   /*lint -esym(40,fvSetUserDeselect) fvSetUserDeselect Undeclared identifier */
   /*lint -esym(40,fvApplySelStrategy) fvApplySelStrategy Undeclared identifier */
   /*lint -esym(40,fvApplySelStrategy) fvApplySelStrategy Undeclared identifier */
   /*lint -esym(40,_1) _1 Undeclared identifier */
   /*lint -esym(40,_2) _2 Undeclared identifier */
   //! Register for callbacks
   trSelectDeviceCallbacks rSelDevCbs;
   rSelDevCbs.fvOnSelectDeviceRes = std::bind(&spi_tclDeviceSelector::vReceiveSelectDeviceResCb,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   rSelDevCbs.fvOnDeviceDisconnection = std::bind(&spi_tclDeviceSelector::vDeviceDisconnectionCb,
            this,
            std::placeholders::_1);

   rSelDevCbs.fvOnDeviceConnection = std::bind(&spi_tclDeviceSelector::vDeviceConnectionCb,
            this,
            std::placeholders::_1);

   rSelDevCbs.fvAutoSelectDevice = std::bind(&spi_tclDeviceSelector::vOnAutomaticDeviceSelection,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3,
            std::placeholders::_4);

      rSelDevCbs.fvSetUserDeselect = std::bind(
            &spi_tclDeviceSelector::vOnSetUserDeselect, this,
            SPI_FUNC_PLACEHOLDERS_2);

   rSelDevCbs.fvApplySelStrategy = std::bind(&spi_tclDeviceSelector::vApplySelectionStrategy,
            this,
            std::placeholders::_1);

   rSelDevCbs.fvDeviceChanged = std::bind(&spi_tclDeviceSelector::vDeviceChanged, this, std::placeholders::_1);

   rSelDevCbs.fvOnHUWiFiAPStateChange = std::bind(&spi_tclDeviceSelector::vOnHUWiFiAPStateChange,
            this,
            std::placeholders::_1);

   rSelDevCbs.fvOnDeviceConnectsToWirelessAP = std::bind(&spi_tclDeviceSelector::vIsPreSelectionValidcb, this,
   SPI_FUNC_PLACEHOLDERS_2);


   rSelDevCbs.fvGetDeviceSelectorCurrentSelectionState = std::bind(&spi_tclDeviceSelector::vGetDeviceSelectorCurrentSelectionState, this,
            SPI_FUNC_PLACEHOLDERS_1);

   rSelDevCbs.fvOnRetrailSetDeviceSelectionState = std::bind(&spi_tclDeviceSelector::vOnRetrailSetSelectionState, this,
            SPI_FUNC_PLACEHOLDERS_1);

   if (NULL != m_poMediator)
   {
      m_poMediator->vRegisterCallbacks(rSelDevCbs);
   } // if (NULL != m_poMediator)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vSendDeviceSelectionReq
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSendDeviceSelectionReq(const trSelectDeviceRequest &corfrSelDevinfo, tenCompID enCompID)
{
   ETG_TRACE_USR2(("spi_tclDeviceSelector::vSendDeviceSelectionReq: Sending %d request for DeviceHandle = 0x%x to component =%d Transport:%d", ETG_ENUM(CONNECTION_REQ,
            corfrSelDevinfo.m_enDevConnReq), corfrSelDevinfo.m_u32DeviceHandle, ETG_ENUM(COMPONENT_ID, enCompID), ETG_ENUM(CONNECTION_TYPE,
            corfrSelDevinfo.m_enDevConnType)));

   m_enSelectedComp = enCompID;
   //! Start the component response timer and wait for the response
   vStartComponentResponseTimer(corfrSelDevinfo.m_enDevCategory);

   //! Send Device Selection request to interested SPI components
   m_oLockComponents.s16Lock();
   std::map<tenCompID, spi_tclSelectionIntf*>::iterator itCompMap = m_mapComponents.find(enCompID);
   if (m_mapComponents.end() != itCompMap)
   {
      itCompMap->second->vSelectDevice(corfrSelDevinfo);
   }
   else
   {
      ETG_TRACE_ERR((" Unknown component ID = %d ", enCompID));
   }
   m_oLockComponents.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vEvaluateDeviceSelectionReq
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vEvaluateDeviceSelectionReq()
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vEvaluateDeviceSelectionReq(): Evaluating device selection request: Currently Selected Device =0x%x ", m_u32CurrSelectedDevice));

   tenDeviceConnectionStatus enConnStatus = e8DEV_NOT_CONNECTED;
   tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE;
   tenDeviceConnectionType enDevConnType = e8UNKNOWN_CONNECTION;
   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_pDeviceList)
   {
      enDevCategory = m_pDeviceList->enGetDeviceCategory(m_rSelectDevReqInfo.m_u32DeviceHandle);
      enDeviceType = m_pDeviceList->enGetDeviceType(m_rSelectDevReqInfo.m_u32DeviceHandle);

      enDevConnType = m_pDeviceList->enGetDeviceConnType(m_rSelectDevReqInfo.m_u32DeviceHandle);
      if ((e8DEV_TYPE_DIPO == enDevCategory) && (e8DEVCONNREQ_DESELECT == m_rSelectDevReqInfo.m_enDevConnReq))
      {
         tenSessionTransport enSessionTransport =
                  m_pDeviceList->enGetSessionTransport(m_rSelectDevReqInfo.m_u32DeviceHandle);
         switch (enSessionTransport)
         {
            case e8_SESSION_TRANSPORT_USB:
            {
               enDevConnType = e8USB_CONNECTED;
            }
               break;
            case e8_SESSION_TRANSPORT_WIFI:
            default:
            {
               enDevConnType = e8WIRELESS_CONNECTED;
            }
               break;
         }
      }
      //! Switch Device Case:
      //! Send device disconnect if some other device is selected
      if ((e8DEVCONNREQ_SELECT == m_rSelectDevReqInfo.m_enDevConnReq) && (0 != m_u32CurrSelectedDevice))
      {
         ETG_TRACE_USR2((" Device %d already selected. Deselecting it ... ", m_u32CurrSelectedDevice));
         //! Send Device deselection request
         m_bDeviceSwitchRequest = true;
         m_rDeselectDevReqInfo.m_u32DeviceHandle = m_u32CurrSelectedDevice;
         m_rDeselectDevReqInfo.m_enSelectionReason = e8_REASON_USER_TRIGGER;
         m_rDeselectDevReqInfo.m_enDevConnReq = e8DEVCONNREQ_DESELECT;
         m_rDeselectDevReqInfo.m_enDevCategory = enDevCategory;
         m_rDeselectDevReqInfo.m_enDevConnType = enDevConnType;
         m_rDeselectDevReqInfo.m_enDeviceType = enDeviceType;

         //! on Switch device set the user deselection flag for the device being deselected
         m_pDeviceList->vSetUserDeselectionFlag(m_rDeselectDevReqInfo.m_u32DeviceHandle, true);
         m_pDeviceList->vSetUserDeselectionFlag(m_rSelectDevReqInfo.m_u32DeviceHandle, false);

         //! Send the deselection request
         vStartDeviceSelection(m_rDeselectDevReqInfo);
      } // if ((e8DEVCONNREQ_SELECT == m_rSelectDevReqInfo.m_enDevConnReq)
      else
      {
         //! Normal Selection/Deselection Case
         //! Set user deselection flag if the device is deselected by the user
         //! via HMI. Otherwise reset the flag. this flag will be read before
         //! triggering automatic selection

         enConnStatus = m_pDeviceList->enGetDeviceConnStatus(m_rSelectDevReqInfo.m_u32DeviceHandle);
         m_rSelectDevReqInfo.m_enDevCategory = enDevCategory;
         m_rSelectDevReqInfo.m_enDevConnType = enDevConnType;
         m_rSelectDevReqInfo.m_enDeviceType = enDeviceType;
         if (e8DEV_CONNECTED == enConnStatus)
         {
            //! Set the user deselection flag if deselection is triggered as a result
            //! of user action (any HMI trigger that results in deselection)when device is connected
            //! other than the switch device case
            t_Bool bIsUserDeselect = ((e8DEVCONNREQ_DESELECT == m_rSelectDevReqInfo.m_enDevConnReq)
                     && (e8_REASON_USER_TRIGGER == m_rSelectDevReqInfo.m_enSelectionReason) && (false == m_bDeviceSwitchRequest));

            ETG_TRACE_USR4((" spi_tclDeviceSelector::vEvaluateDeviceSelectionReq() bisHMIDeselect =%d "
                     " m_rDeselectDevReqInfo.m_u32DeviceHandle = %d m_rSelectDevReqInfo.m_u32DeviceHandle = %d, Device Connection Type:%d", ETG_ENUM(
                              BOOL, bIsUserDeselect), m_rDeselectDevReqInfo.m_u32DeviceHandle, m_rSelectDevReqInfo.m_u32DeviceHandle,ETG_ENUM(CONNECTION_TYPE,m_rSelectDevReqInfo.m_enDevConnType)));

            m_pDeviceList->vSetUserDeselectionFlag(m_rSelectDevReqInfo.m_u32DeviceHandle, bIsUserDeselect);
         }
         //! corrected the selection sequence for mirrorlink in case of deselect and select from HMI
         vStartDeviceSelection(m_rSelectDevReqInfo);
      } //else case for if (e8DEVCONNREQ_SELECT ...
   } //if (NULL != m_pDeviceList)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::bDeviceSelectionComplete
 ***************************************************************************/
t_Bool spi_tclDeviceSelector::bDeviceSelectionComplete(const t_U32 cou32DeviceHandle,
         tenDeviceConnectionReq enDevConnReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::bDeviceSelectionComplete: Device selection process completed for DeviceHandle =0x%x, %d request"
            "Resulting ErrorCode= %d Internal deselection required flag = %d ", cou32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), ETG_ENUM(ERROR_CODE, enErrorCode), ETG_ENUM(BOOL, m_bDeviceSwitchRequest)));

   t_Bool bProcessComplete = true;

   if (e8NO_ERRORS == enErrorCode)
   {
      m_u32CurrSelectedDevice = (e8DEVCONNREQ_SELECT == enDevConnReq) ? m_rSelectDevReqInfo.m_u32DeviceHandle : 0;
   }

   //! Check if the result is for internal deselection
   if ((e8NO_ERRORS == enErrorCode) && ((true == m_bDeviceSwitchRequest)))
   {
      bProcessComplete = false;
      m_bDeviceSwitchRequest = false;
   } // if (e8NO_ERRORS == enErrorCode)
   return bProcessComplete;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vSendSelectDeviceResult
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSendSelectDeviceResultToHMI(trSelectDeviceRequest &rfrSelDevReq,
         tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vSendSelectDeviceResultToHMI: Sending device selection result to HMI"
            " cou32DeviceHandle =0x%x, enDevConnReq = %d, enErrorCode = %d, SelectionReason = %d, DeviceConnectionType:%d", rfrSelDevReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,
            rfrSelDevReq.m_enDevConnReq), ETG_ENUM(ERROR_CODE, enErrorCode), ETG_ENUM(SELECT_REASON_UNKNOWN,
            rfrSelDevReq.m_enSelectionReason), ETG_ENUM(CONNECTION_TYPE, rfrSelDevReq.m_enDeviceType)));

   //! Populate response code based on error

   ETG_TRACE_USR1(("spi_tclDeviceSelector::vSendSelectDeviceResultToHMI: SrcAppID = %d ", rfrSelDevReq.m_corUsrCntxt.u32SrcAppID));
   //! Send the result to HMI if trigger is from HMI and if its is not the deslection request for switch device case
   if ((NULL != m_poDevSelResp) && (0 != rfrSelDevReq.m_corUsrCntxt.u32SrcAppID) && (false == m_bDeviceSwitchRequest))
   {
      m_poDevSelResp->vPostSelectDeviceResult(rfrSelDevReq, enErrorCode, rfrSelDevReq.m_corUsrCntxt);
   } //if ((NULL != m_poDevSelResp) &&
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vSendSelectDeviceResult
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vSendSelectDeviceResult(const trSelectDeviceRequest &corfrSelDevReq,
         tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vSendSelectDeviceResult: Sending %d request's result for DeviceHandle =0x%x to other components"
            " enErrorCode = %d, SelectionReason = %d,DeviceConnectionType:%d ", ETG_ENUM(CONNECTION_REQ,
            corfrSelDevReq.m_enDevConnReq), corfrSelDevReq.m_u32DeviceHandle, ETG_ENUM(ERROR_CODE, enErrorCode), ETG_ENUM(SELECT_REASON,
            corfrSelDevReq.m_enSelectionReason), ETG_ENUM(CONNECTION_TYPE, corfrSelDevReq.m_enDeviceType)));

   t_Bool bDevSelectDone = bDeviceSelectionComplete(corfrSelDevReq.m_u32DeviceHandle,
            corfrSelDevReq.m_enDevConnReq,
            enErrorCode);

   if (true == bDevSelectDone)
   {
      vSendSelectDeviceResultToHMI(m_rSelectDevReqInfo, enErrorCode);
   }

   t_Bool bIsDeviceSwitch = ((e8DEVCONNREQ_DESELECT == corfrSelDevReq.m_enDevConnReq) && (false == bDevSelectDone));
   m_rSelectDevReqInfo.m_bisSwitchRequired = bIsDeviceSwitch;

   //! Send Device Selection request to interested SPI components
   m_oLockComponents.s16Lock();
   std::map<tenCompID, spi_tclSelectionIntf*>::iterator itCompMap = m_mapComponents.begin();
   for (itCompMap = m_mapComponents.begin(); itCompMap != m_mapComponents.end(); itCompMap++)
   {
      itCompMap->second->vSelectDeviceResult(m_rSelectDevReqInfo, enErrorCode);
   }
   m_oLockComponents.vUnlock();

   //update selection progress state to not applicable once selection is completed 
   m_pDeviceList->vSetSelectionProgressState(corfrSelDevReq.m_u32DeviceHandle,
            e8_SELECTION_PROGRESS_STATE_NOT_APPLICABLE);

   if ((e8DEVCONNREQ_DESELECT == corfrSelDevReq.m_enDevConnReq)
            && (e8_STATION_DISCONNECTED == corfrSelDevReq.m_enSelectionReason))
   {
      enErrorCode = e8STA_DISCONNECTED;
   }
   //! update current selection state to HMI
   vUpdateDevSelectionInfo(corfrSelDevReq.m_u32DeviceHandle, enErrorCode, corfrSelDevReq.m_enDevConnReq);

   //clearing the private data after factory reset
   if ((NULL != m_poDevSelResp)
            && (((e8_FACTORY_RESET == corfrSelDevReq.m_enSelectionReason)
                     && (e8DEVCONNREQ_DESELECT == corfrSelDevReq.m_enDevConnReq))
                     || (e8OPERATION_CANCELLED_BY_FACTORY_RESET == enErrorCode)))
   {
      m_poDevSelResp->vClearPrivateDataOnfactoryReset();
   }

   m_rSelectDevReqInfo.m_bisSwitchRequired = false;

   if (true == bDevSelectDone)
   {
      //! reset the selected component after the result is posted to HMI
      if (e8RESOURCE_BUSY != enErrorCode)
      {
         //! Remove the busy flag
         vUpdateDevSelectorBusyStatus(false, corfrSelDevReq.m_u32DeviceHandle);
         m_enSelectedComp = e32COMPID_UNKNOWN;
      }
   }
   //! Switch device case. Selection is pending for the other device
   else
   {
      //! Send the selection request
      vStartDeviceSelection(m_rSelectDevReqInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vStartComponentResponseTimer
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vStartComponentResponseTimer(tenDeviceCategory enDevCategory)
{
   t_U32 u32TimerInterval_ms = 0;

   //! Determine the timer interval based on component
   switch (m_enSelectedComp)
   {
      case e32COMPID_AUDIO: //! Mirrorlink audio is taking more than 10 seconds to respond. Wait for 25 seconds
      case e32COMPID_BLUETOOTH:
      {
         u32TimerInterval_ms = su32CompResponseTime25s;
         break;
      } //case e32COMPID_BLUETOOTH:
      case e32COMPID_RESOURCEMNGR:
      case e32COMPID_DISCOVERER:
      {
         u32TimerInterval_ms = su32ConnCompResponseTimeinms;
         //! For Android auto reduce the duration of component response timer to 10sec
         if ((e8DEV_TYPE_ANDROIDAUTO == enDevCategory) && (su32CompResponseTime25s == su32ConnCompResponseTimeinms))
         {
            u32TimerInterval_ms = su32CompResponseTime10s;
         }

         if (e8DEV_TYPE_DIPO == enDevCategory)
         {
            //In case of CarPlay Wireless, iPhone may take little longer to connect to Wi-Fi AP after Wi-Fi credentials are provided.
            //Hence timer is set to 45sec
            u32TimerInterval_ms = su32CompResponseTime60s;
         }

         if (e8DEV_TYPE_CARLIFE == enDevCategory)
         {
            //In case of Carlife iPhone since both discoverer and switcher timer is 25s,
            //whenever roleswitch is failed discoverer timer expires first and then switcher timer.
            //This was leading to two FAILURE response from discoverer. So increasing the discoverer timer to 26s.
            u32TimerInterval_ms = su32CompResponseTime26s;
         }
         break;
      }
      case e32COMPID_CONNECTIONMANAGER:
      {
         u32TimerInterval_ms =
                  (((e8DEV_TYPE_CARLIFE == enDevCategory) || (e8DEV_TYPE_ONCAR == enDevCategory))) ?
                           (su32CompResponseTime15s) : (su32CompResponseTime10s);

         //In case of CarPlay, since the device disconnections are not used device discovery could return immediately during selection sequence.
         // if there was session with the same device just before current selection. In this case we require to perform re-trial and try to establish session.
         if (e8DEV_TYPE_DIPO == enDevCategory)
         {
            u32TimerInterval_ms = su32CompResponseTime25s;
         }
         break;
      }
         //! Below components need user action. Hence timer will not be started
      case e32COMPID_LEGALDISCLAIMER:
      case e32COMPID_UNKNOWN:
      case e32COMPID_TECHNOLOGYSELECTOR:
      case e32COMPID_WIFI:
      {
         u32TimerInterval_ms = 0;
      }
         break;
      case e32COMPID_APPMANAGER:
      case e32COMPID_VIDEO:
      default:
      {
         u32TimerInterval_ms = su32CompResponseTime10s;
         break;
      }		 //default:
   }		 //switch (m_enSelectedComp)

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (0 != u32TimerInterval_ms))
   {
      timer_t rTimerID;
      //! Start timer and wait for the response from the component
      poTimer->StartTimer(rTimerID, u32TimerInterval_ms, 0, this, &spi_tclDeviceSelector::bCompRespTimerCb, NULL);
      m_rCompRespTimerID = rTimerID;
   }
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vStartComponentResponseTimer: Waiting for %d component to respond back in %d milliseconds threadID = %d", ETG_ENUM(COMPONENT_ID,
            m_enSelectedComp), u32TimerInterval_ms, SPI_THREADID ));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vUpdateDevSelectorBusyStatus
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vUpdateDevSelectorBusyStatus(t_Bool bSelectorStatus, t_U32 cou32DeviceHandle)
{
   SPI_INTENTIONALLY_UNUSED(cou32DeviceHandle);
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vUpdateDevSelectorBusyStatus Current State = %d "
            "Requested State =%d", ETG_ENUM(BOOL, m_bIsDevSelectorBusy), ETG_ENUM(BOOL, bSelectorStatus)));
   m_bIsDevSelectorBusy = bSelectorStatus;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::enGetNextComponentID
 ***************************************************************************/
tenCompID spi_tclDeviceSelector::enGetNextComponentID(tenDeviceCategory enDeviceCat,
         tenDeviceConnectionReq enDevConnReq, tenCompID enCurrentCompID)
{
   tenCompID enNextCompID = e32COMPID_UNKNOWN;

   if (e8DEVCONNREQ_SELECT == enDevConnReq)
   {
      if (m_mapSelectionSequence.end() != m_mapSelectionSequence.find(enDeviceCat))
      {
         ETG_TRACE_USR1((" spi_tclDeviceSelector::Device category found: size = %d ", m_mapSelectionSequence[enDeviceCat].size()));
         std::vector<tenCompID>::iterator itVecSequence;
         for (itVecSequence = m_mapSelectionSequence[enDeviceCat].begin();
                  itVecSequence != m_mapSelectionSequence[enDeviceCat].end(); itVecSequence++)
         {
            if (enCurrentCompID == *itVecSequence)
            {
               itVecSequence++;
               if (itVecSequence != m_mapSelectionSequence[enDeviceCat].end())
               {
                  enNextCompID = *itVecSequence;
               }
               else
               {
                  enNextCompID = *(m_mapSelectionSequence[enDeviceCat].begin());
               }
               break;
            }
         }
      }
   }
   else if (e8DEVCONNREQ_DESELECT == enDevConnReq)
   {
      if (m_mapDeselectionSequence.end() != m_mapDeselectionSequence.find(enDeviceCat))
      {
         std::vector<tenCompID>::iterator itVecSequence;
         for (itVecSequence = m_mapDeselectionSequence[enDeviceCat].begin();
                  itVecSequence != m_mapDeselectionSequence[enDeviceCat].end(); itVecSequence++)
         {
            if (enCurrentCompID == *itVecSequence)
            {
               itVecSequence++;
               if (itVecSequence != m_mapDeselectionSequence[enDeviceCat].end())
               {
                  enNextCompID = *itVecSequence;
               }
               else
               {
                  enNextCompID = *(m_mapDeselectionSequence[enDeviceCat].begin());
               }
               break;
            }
         }
      }
   }

   ETG_TRACE_USR1((" spi_tclDeviceSelector::enGetNextComponentID Device Category = %d, Selection Request = %d,"
            " Next ComponentID = %d enCurrentCompID = %d", ETG_ENUM(DEVICE_CATEGORY, enDeviceCat), ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), ETG_ENUM(COMPONENT_ID, enNextCompID), ETG_ENUM(COMPONENT_ID, enCurrentCompID)));
   return enNextCompID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vLoadSelectionSequence
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vLoadSelectionSequence()
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vLoadSelectionSequence entered "));
   std::vector<tenCompID> vecMLSelectionSequence(azSelectionSequence[e8DEV_TYPE_DIPO],
            (azSelectionSequence[e8DEV_TYPE_DIPO] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_DIPO] = vecMLSelectionSequence;

   std::vector<tenCompID> vecCarplaySelectionSequence(azSelectionSequence[e8DEV_TYPE_MIRRORLINK],
            (azSelectionSequence[e8DEV_TYPE_MIRRORLINK] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_MIRRORLINK] = vecCarplaySelectionSequence;

   std::vector<tenCompID> vecAAPSelectionSequence(azSelectionSequence[e8DEV_TYPE_ANDROIDAUTO],
            (azSelectionSequence[e8DEV_TYPE_ANDROIDAUTO] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_ANDROIDAUTO] = vecAAPSelectionSequence;

   std::vector<tenCompID> vecmySPINSelectionSequence(azSelectionSequence[e8DEV_TYPE_MYSPIN],
            (azSelectionSequence[e8DEV_TYPE_MYSPIN] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_MYSPIN] = vecmySPINSelectionSequence;

   std::vector<tenCompID> vecBDCLSelectionSequence(azSelectionSequence[e8DEV_TYPE_CARLIFE],
            (azSelectionSequence[e8DEV_TYPE_CARLIFE] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_CARLIFE] = vecBDCLSelectionSequence;

   std::vector<tenCompID> vecOnCarSelectionSequence(azSelectionSequence[e8DEV_TYPE_ONCAR],
            (azSelectionSequence[e8DEV_TYPE_ONCAR] + e32COMPID_SIZE));
   m_mapSelectionSequence[e8DEV_TYPE_ONCAR] = vecOnCarSelectionSequence;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vLoadDeselectionSequence
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vLoadDeselectionSequence()
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vLoadDeselectionSequence entered "));
   std::vector<tenCompID> vecMLSelectionSequence(azDeSelectionSequence[e8DEV_TYPE_DIPO],
            (azDeSelectionSequence[e8DEV_TYPE_DIPO]
                     + ((sizeof(azDeSelectionSequence[e8DEV_TYPE_DIPO]) / sizeof(tenCompID)))));
   m_mapDeselectionSequence[e8DEV_TYPE_DIPO] = vecMLSelectionSequence;

   std::vector<tenCompID> vecCarplaySelectionSequence(azDeSelectionSequence[e8DEV_TYPE_MIRRORLINK],
            (azDeSelectionSequence[e8DEV_TYPE_MIRRORLINK]
                     + ((sizeof(azDeSelectionSequence[e8DEV_TYPE_MIRRORLINK]) / sizeof(tenCompID)))));
   m_mapDeselectionSequence[e8DEV_TYPE_MIRRORLINK] = vecCarplaySelectionSequence;

   std::vector<tenCompID> vecAAPSelectionSequence(azDeSelectionSequence[e8DEV_TYPE_ANDROIDAUTO],
            (azDeSelectionSequence[e8DEV_TYPE_ANDROIDAUTO]
                     + ((sizeof(azDeSelectionSequence[e8DEV_TYPE_ANDROIDAUTO]) / sizeof(tenCompID)))));
   m_mapDeselectionSequence[e8DEV_TYPE_ANDROIDAUTO] = vecAAPSelectionSequence;

   std::vector<tenCompID> vecMySpinDeselectionSequence(azDeSelectionSequence[e8DEV_TYPE_MYSPIN],
            (azDeSelectionSequence[e8DEV_TYPE_MYSPIN] + e32COMPID_SIZE));
   m_mapDeselectionSequence[e8DEV_TYPE_MYSPIN] = vecMySpinDeselectionSequence;

   std::vector<tenCompID> vecBDCLDeselectionSequence(azDeSelectionSequence[e8DEV_TYPE_CARLIFE],
            (azDeSelectionSequence[e8DEV_TYPE_CARLIFE]
                     + ((sizeof(azDeSelectionSequence[e8DEV_TYPE_CARLIFE]) / sizeof(tenCompID)))));
   m_mapDeselectionSequence[e8DEV_TYPE_CARLIFE] = vecBDCLDeselectionSequence;

   std::vector<tenCompID> vecOnCarDeSelectionSequence(azDeSelectionSequence[e8DEV_TYPE_ONCAR],
            (azDeSelectionSequence[e8DEV_TYPE_ONCAR]
                     + ((sizeof(azDeSelectionSequence[e8DEV_TYPE_ONCAR]) / sizeof(tenCompID)))));
   m_mapDeselectionSequence[e8DEV_TYPE_ONCAR] = vecOnCarDeSelectionSequence;

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::u32GetLastConnectedDevice
 ***************************************************************************/
t_U32 spi_tclDeviceSelector::u32GetLastConnectedDevice()
{
   /*lint -esym(40,rDeviceInfo) rDeviceInfo Undeclared identifier */
   /*lint -esym(40,u32DeviceHandle) u32DeviceHandle Undeclared identifier */
   /*lint -esym(40,enDeviceConnectionStatus) enDeviceConnectionStatus Undeclared identifier */
   /*lint -esym(40,second) second Undeclared identifier */
   /*lint -esym(40,enDeviceUsageEnabled) enDeviceUsageEnabled Undeclared identifier */
   /*lint -esym(40,bIsUSBConnected) bIsUSBConnected Undeclared identifier */
   /*lint -esym(40,bIsUserDeselected) bIsUserDeselected Undeclared identifier */
   /*lint -esym(40,enDeviceUsageEnabled) enDeviceUsageEnabled Undeclared identifier */

   std::vector<trDeviceInfo> vecDeviceInfoList;
   t_U32 u32LastConnectedDevice = 0;
   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->vGetConnectedDeviceList(vecDeviceInfoList);
      //! Iterate through the device list and find the device for
      //! selection based on project requirements
      //! Select the recently connected device for automatic selection
      //! if the device is not deselected by user and if the Device usage flag is enabled
      //! and if the device validation is successful

      spi_tclSelectionValidator oDeviceValidator(m_pDeviceList, m_u32CurrSelectedDevice, m_bIsDevSelectorBusy);

      for (auto itVec = vecDeviceInfoList.begin(); itVec != vecDeviceInfoList.end(); itVec++)
      {
         trSelectDeviceRequest rSelDevReq;
         rSelDevReq.m_u32DeviceHandle = itVec->u32DeviceHandle;
         rSelDevReq.m_enDevCategory = itVec->enDeviceCategory;
         rSelDevReq.m_enDevConnReq = e8DEVCONNREQ_SELECT;
         rSelDevReq.m_enDeviceType = itVec->rProjectionCapability.enDeviceType;
         if (true == oDeviceValidator.bDeviceValidforAutoSelection(rSelDevReq))
         {
            u32LastConnectedDevice = itVec->u32DeviceHandle;
         }
      } // for (auto itvec = rvecDeviceInfo.begin()...
   }

   ETG_TRACE_USR4(("u32GetLastConnectedDevice :Last connected Device = %d ", u32LastConnectedDevice));
   return u32LastConnectedDevice;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vUpdateDevSelectionInfo(t_Bool)
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vUpdateDevSelectionInfo(const t_U32 cou32DeviceHandle, tenErrorCode enErrorcode,
         tenDeviceConnectionReq enDeviceConnectionReq)
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vUpdateDevSelectionInfo DeviceID = 0x%x, Error = %d ", cou32DeviceHandle, ETG_ENUM(ERROR_CODE,
            enErrorcode)));

   trDeviceSelectionInfo rStoredSelectDeviceInfo;
   tenDeviceConnectionType enDeviceConnectionType = e8UNKNOWN_CONNECTION;
   t_Bool bIsSelectError = false;
   t_Bool bIsReenumerationInProgress = false;
   t_Bool bDeviceSelectionInfoUpdation = false;

//! Below switch case is based on this table
//   ErrorCode               DeviceConnectionReq   SelectionErrorType                    SelectionState
//   e8NO_ERRORS             SELECT               DEVICE_SELECTION_ERROR_NONE            SELECTION_STATE_DEVICE_SELECTED
//   e8NO_ERRORS             DESELECT             DEVICE_SELECTION_ERROR_NONE            SELECTION_STATE_DEVICE_NOT_SELECTED
//   e8RESOURCE_BUSY         SELECT/DESELECT      DEVICE_SELECTION_ERROR_RESOURCE_BUSY   SELECTION_STATE_DEVICE_NOT_SELECTED
//   e8OPERATION_REJECTED    SELECT/DESELECT      DEVICE_SELECTION_ERROR_RESOURCE_BUSY   SELECTION_STATE_DEVICE_NOT_SELECTED
//   e8INVALID_DEV_HANDLE    SELECT/DESELECT      DEVICE_SELECTION_ERROR_NOT_CONNECTED   SELECTION_STATE_DEVICE_NOT_SELECTED
//   e8DEVICE_NOT_CONNECTED  SELECT/DESELECT      DEVICE_SELECTION_ERROR_NOT_CONNECTED   SELECTION_STATE_DEVICE_NOT_SELECTED
//   e8SELECTION_FAILED      SELECT               DEVICE_SELECTION_DEVICE_ERROR          SELECTION_STATE_DEVICE_SELECTION_ERROR
//   e8UNKNOWN_ERROR         SELECT               DEVICE_SELECTION_DEVICE_ERROR          SELECTION_STATE_DEVICE_SELECTION_ERROR
//   e8SELECTION_FAILED      DESELECT             DEVICE_SELECTION_DEVICE_ERROR          SELECTION_STATE_DEVICE_DESELECTION_ERROR
//   e8UNKNOWN_ERROR         DESELECT             DEVICE_SELECTION_DEVICE_ERROR          SELECTION_STATE_DEVICE_DESELECTION_ERROR
//   e8CONNECT_TIME_OUT      SELECT               DEVICE_SELECTION_DEVICE_ERROR          SELECTION_STATE_DEVICE_SELECTION_ERROR
//   e8FATAL_ERROR           SELECT               DEVICE_SELECTION_DEVICE_ERROR_FATAL    SELECTION_STATE_DEVICE_SELECTION_ERROR

   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->vGetDeviceSelectionInfo(cou32DeviceHandle, rStoredSelectDeviceInfo);
      bIsSelectError = m_pDeviceList->bIsSelectError(cou32DeviceHandle);
      bIsReenumerationInProgress = (e8_SELECTION_STATE_DEVICE_REENUMERATION_INPROGRESS
               == rStoredSelectDeviceInfo.enDeviceSelectionState);
   }

   switch (enErrorcode)
   {
      case e8NO_ERRORS:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NONE;
         if (e8DEVCONNREQ_SELECT == enDeviceConnectionReq)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTED;
         }
         //! Don't overwrite user deselection state until HMI does a selection
         else if ((e8DEVCONNREQ_DESELECT == enDeviceConnectionReq)
                  && (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState))
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         }
         if ((NULL != m_pDeviceList) && (true == bIsSelectError))
         {
            rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_DEVICE_ERROR;
         }

      }
         break;
      case e8OPERATION_CANCELLED_BY_USER:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_CANCELLED_BY_USER;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         }

      }
         break;
      case e8PROJECTION_NOT_ENABLED:
      case e8OPERATION_REJECTED:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_RESOURCE_BUSY;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         }
      }
         break;
      case e8DEVICE_NOT_CONNECTED:
      case e8INVALID_DEV_HANDLE:
      case e8DEVICE_DISCONNECTED_DURING_SELECTION:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NOT_CONNECTED;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         }
      }
         break;
      case e8FATAL_ERROR:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_DEVICE_ERROR_FATAL;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState =
                     (e8DEVCONNREQ_SELECT == enDeviceConnectionReq) ?
                              e8_SELECTION_STATE_DEVICE_SELECTION_ERROR : e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
         }
      }
         break;
      case e8PHONE_UNSUPPORTED:
      {
         ETG_TRACE_USR1(("e8PHONE_UNSUPPORTED ERROR "));
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_PHONE_UNSUPPORTED;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState =
                     (e8DEVCONNREQ_SELECT == enDeviceConnectionReq) ?
                              e8_SELECTION_STATE_DEVICE_SELECTION_ERROR : e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
         }
      }
         break;
      case e8APP_NOT_INSTALLED:
      {
         ETG_TRACE_USR1(("e8APP_NOT_INSTALLED ERROR "));
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_APP_NOT_INSTALLED;
         if (e8_SELECTION_STATE_DEVICE_USER_DESELECTED != rStoredSelectDeviceInfo.enDeviceSelectionState)
         {
            rStoredSelectDeviceInfo.enDeviceSelectionState =
                     (e8DEVCONNREQ_SELECT == enDeviceConnectionReq) ?
                              e8_SELECTION_STATE_DEVICE_SELECTION_ERROR : e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
         }
      }
         break;
         //! Need not update the SelectionInfo for the below mentioned use-cases only for the device where in SELECTION/DESELECTION in Progress or Device is SELECTED
         //! 1. SELECTION IN PROGRESS - SELECT/DESELECT request - e8RESOURCE_BUSY
         //! 2. DESELECTION IN PROGRESS - SELECT/DESELECT request - e8RESOURCE_BUSY
         //! 3. SELECTD - SELECT Request - e8UNSUPPORTED_OPERATION
         //! 4. DESELECTED - DESELECT Request - e8UNSUPPORTED_OPERATION
      case e8RESOURCE_BUSY:
      case e8UNSUPPORTED_OPERATION:
      {
         //! Nothing todo
         bDeviceSelectionInfoUpdation = true;
      }
         break;

      case e8AP_SETUP_ERROR:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_WIFI_AP_SETUP_ERROR;
         rStoredSelectDeviceInfo.enDeviceSelectionState =
                  (e8DEVCONNREQ_SELECT == enDeviceConnectionReq) ?
                           e8_SELECTION_STATE_DEVICE_SELECTION_ERROR : e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
      }
         break;

         //! Information needed by HMI to inform the user about the Device leaving the CPW AP.
         //! As of now, no Project have such popup to the user, In future the if any Project requests this is the Implementation.
      case e8STA_DISCONNECTED:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_STA_DISCONNECTED;
         rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
      }
         break;

      case e8DEVICE_SWITCH_FAILED:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_SWITCH_FAILED;
         rStoredSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTION_ERROR;
      }
         break;

      case e8SELECTION_FAILED:
      case e8UNKNOWN_ERROR:
      case e8CONNECT_TIME_OUT:
      default:
      {
         rStoredSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_DEVICE_ERROR;
         rStoredSelectDeviceInfo.enDeviceSelectionState =
                  (e8DEVCONNREQ_SELECT == enDeviceConnectionReq) ?
                           e8_SELECTION_STATE_DEVICE_SELECTION_ERROR : e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
      }
         break;
   }

   //! updating the Device Selector current state
   vUpdateCurrentDeviceSelectorState(cou32DeviceHandle,
            enErrorcode,
            enDeviceConnectionReq,
            bDeviceSelectionInfoUpdation);

   if ((NULL != m_pDeviceList) && (false == bIsReenumerationInProgress))
   {
      m_pDeviceList->vSetDeviceSelectionInfo(cou32DeviceHandle, rStoredSelectDeviceInfo);
   }

   if(NULL != m_pDeviceList)
   {
       m_pDeviceList->vSetDeviceSelectionErrorType(cou32DeviceHandle,rStoredSelectDeviceInfo.enDeviceSelectionErrorType);
       enDeviceConnectionType = m_pDeviceList->enGetDeviceConnType(cou32DeviceHandle);
   }

   m_oLockComponents.s16Lock();
   for(auto itCompMap = m_mapComponents.begin() ; itCompMap != m_mapComponents.end(); itCompMap++ )
   {
      itCompMap->second->vUpdateSelectionState(cou32DeviceHandle, rStoredSelectDeviceInfo.enDeviceSelectionState, enDeviceConnectionType);
   }
   m_oLockComponents.vUnlock();

}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vStartDeviceSelection
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vStartDeviceSelection(trSelectDeviceRequest &rSelectDevReqInfo)
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vStartDeviceSelection Starting device selection sequence:DeviceConnectionType:%d,SelectionType:%d ", ETG_ENUM(CONNECTION_TYPE,
            rSelectDevReqInfo.m_enDeviceType), ETG_ENUM(CONNECTION_REQ, rSelectDevReqInfo.m_enDevConnReq)));

   tenCompID enCompID = e32COMPID_UNKNOWN;

   if ((e8DEVCONNREQ_SELECT == rSelectDevReqInfo.m_enDevConnReq)
            && (false == m_mapSelectionSequence[rSelectDevReqInfo.m_enDevCategory].empty()))
   {
      enCompID = m_mapSelectionSequence[rSelectDevReqInfo.m_enDevCategory].front();
   }
   else if ((e8DEVCONNREQ_DESELECT == rSelectDevReqInfo.m_enDevConnReq)
            && (false == m_mapDeselectionSequence[rSelectDevReqInfo.m_enDevCategory].empty()))
   {
      enCompID = m_mapDeselectionSequence[rSelectDevReqInfo.m_enDevCategory].front();
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclDeviceSelector::vStartDeviceSelection Invalid request, %d", ETG_ENUM(CONNECTION_REQ,
               rSelectDevReqInfo.m_enDevConnReq)));
   }

   if (e32COMPID_UNKNOWN != enCompID)
   {
      trDeviceSelectionInfo rDeviceSelectionInfo;
      rDeviceSelectionInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NONE;
      rDeviceSelectionInfo.enDeviceSelectionState =
               (e8DEVCONNREQ_SELECT == rSelectDevReqInfo.m_enDevConnReq) ?
                        e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS :
                        e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS;
      m_pDeviceList->vSetDeviceSelectionInfo(rSelectDevReqInfo.m_u32DeviceHandle, rDeviceSelectionInfo);
      //! Set the deviceselector status as busy while processing the request
      vUpdateDevSelectorBusyStatus(true, rSelectDevReqInfo.m_u32DeviceHandle);

      //! update the Device selection state Details
      m_oDeviceSelectionDetailsLock.s16Lock();
      m_rDeviceSelectionDetails.u32DeviceHandle = rSelectDevReqInfo.m_u32DeviceHandle;
      m_rDeviceSelectionDetails.enDeviceSelectionState = rDeviceSelectionInfo.enDeviceSelectionState;
      m_oDeviceSelectionDetailsLock.vUnlock();


      m_oLockComponents.s16Lock();

      for (auto itCompMap = m_mapComponents.begin(); itCompMap != m_mapComponents.end(); itCompMap++)
      {
         itCompMap->second->vUpdateSelectionState(rSelectDevReqInfo.m_u32DeviceHandle, rDeviceSelectionInfo.enDeviceSelectionState, rSelectDevReqInfo.m_enDevConnType);
      }
      m_oLockComponents.vUnlock();
      vSendDeviceSelectionReq(rSelectDevReqInfo, enCompID);
   }
   else
   {
      vUpdateDevSelectorBusyStatus(false, rSelectDevReqInfo.m_u32DeviceHandle);
   }
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::bCompRespTimerCb
 ***************************************************************************/
t_Bool spi_tclDeviceSelector::bCompRespTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);

   spi_tclDeviceSelector* poDevSelector = static_cast<spi_tclDeviceSelector*>(pvObject);
   ETG_TRACE_ERR(("spi_tclDeviceSelector::bCompRespTimerCb : Component %d did not respond back for "
            "device selection request in specified time: Triggering cleanup.. threadID = %p", ETG_ENUM(COMPONENT_ID,
            poDevSelector->m_enSelectedComp), SPI_THREADID ));
   ETG_TRACE_ERR(("spi_tclDeviceSelector::bCompRespTimerCb : rTimerID %p m_rCompRespTimerID %p threadID %d ", rTimerID, poDevSelector->m_rCompRespTimerID, SPI_THREADID ));
   Timer* poTimer = Timer::getInstance();

   //!  If the select device response is not received before timer expiry
   //! then assume the response from the component as false. this is done to
   //! avoid device selector to be blocked in busy state.
   if ((NULL != poDevSelector) && (NULL != poTimer))
   {
      poDevSelector->vReceiveSelectDeviceResCb(poDevSelector->m_enSelectedComp, e8SELECTION_FAILED);
      poTimer->CancelTimer(rTimerID);
   }
   return true;
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::u32GetSelectedDeviceId
 ***************************************************************************/
t_U32 spi_tclDeviceSelector::u32GetSelectedDeviceId()
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::u32GetSelectedDeviceId()  m_u32CurrSelectedDevice = 0x%x", m_u32CurrSelectedDevice));
   return m_u32CurrSelectedDevice;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vDeviceChanged
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vDeviceChanged(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vDeviceChanged Entered() "));
   if ((true == m_bIsDevSelectorBusy) && (cou32DeviceHandle == m_rSelectDevReqInfo.m_u32DeviceHandle)
            && (e8DEV_TYPE_DIPO == m_rSelectDevReqInfo.m_enDevCategory)
            && (e8WIRELESS_CONNECTED == m_rSelectDevReqInfo.m_enDevConnType)
            && (e8DEVCONNREQ_SELECT == m_rSelectDevReqInfo.m_enDevConnReq))
   {
      vReceiveSelectDeviceResCb(m_enSelectedComp, e8DEVICE_DISCONNECTED_DURING_SELECTION);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vDeviceDeselectionOnFactoryReset
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vDeviceDeselectionOnFactoryReset(const t_U32 cou32DeviceHandle,
         tenDeviceConnectionReq enDevConnReq)
{

   ETG_TRACE_USR1(("spi_tclDeviceSelector::vDeviceDeselectionOnFactoryReset() Enetered"));
   tenDeviceCategory enDevCategory = m_pDeviceList->enGetDeviceCategory(cou32DeviceHandle);

   tenDeviceConnectionType enDeviceConnectionType = m_pDeviceList->enGetDeviceConnType(cou32DeviceHandle);
   //! trigger automatic device selection if device selector is not busy and device handle is valid
   if ((0 != cou32DeviceHandle) && (false == m_bIsDevSelectorBusy))
   {
      //! Dummy user context sent as select device is internally triggered
      trUserContext rUsrCntxt;
      trSelectDeviceRequest rSelDeviceReq;
      rSelDeviceReq.m_u32DeviceHandle = cou32DeviceHandle;
      rSelDeviceReq.m_enDevConnReq = enDevConnReq;
      rSelDeviceReq.m_enDevConnType = enDeviceConnectionType;
      rSelDeviceReq.m_enDevCategory = enDevCategory;
      rSelDeviceReq.m_enSelectionReason = e8_FACTORY_RESET;
      rSelDeviceReq.m_corUsrCntxt = rUsrCntxt;
      vSelectDevice(rSelDeviceReq);
   } // if (0 != cou32DeviceHandle)

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vOnHUWiFiAPStateChange
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vOnHUWiFiAPStateChange(t_Bool bIsSetToOn)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vOnHUWiFiAPStateChange():bIsSetToOn:%d", ETG_ENUM(BOOL, bIsSetToOn)));

   t_U32 u32DevID = u32GetSelectedDeviceId();

   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN;
   tenSessionTransport enSessionTransport = e8_SESSION_TRANSPORT_USB;

   if (NULL != m_pDeviceList)
   {
      enDevCategory = m_pDeviceList->enGetDeviceCategory(u32DevID);
      enSessionTransport = m_pDeviceList->enGetSessionTransport(u32DevID);
   }

   //If the CPW is session is active WiFi is turned OFF and then trigger de-select
   if ((false == bIsSetToOn) && (e8DEV_TYPE_DIPO == enDevCategory) && (e8_SESSION_TRANSPORT_WIFI == enSessionTransport))
   {
      ETG_TRACE_USR4(("spi_tclDeviceSelector::vOnHUWiFiAPStateChange():WiFi is turned off during CPW session. Trigger deselection"));
      vOnAutomaticDeviceSelection(u32DevID, e8DEVCONNREQ_DESELECT, e8DEV_TYPE_DIPO, e8_HU_WIFI_AP_OFF);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::u32GetLastConnectedDevice
 ***************************************************************************/
t_U32 spi_tclDeviceSelector::u32GetLastConnectedDevice(tenDeviceCategory enDeviceCategory)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::u32GetLastConnectedDevice Entered with "
            "the Device Category is %d", ETG_ENUM(DEVICE_CATEGORY, enDeviceCategory)));
   std::vector<trDeviceInfo> vecDeviceInfoList;
   tenDeviceCategory enDeviceCategoryForSelection = e8DEV_TYPE_UNKNOWN;
   t_U32 u32LastConnectedDeviceForSelection = 0;

   if (NULL != m_pDeviceList)
   {
      m_pDeviceList->vGetConnectedDeviceList(vecDeviceInfoList);
      //! Iterate through the device list and find the device for
      //! selection based on project requirements
      //! Select the recently connected device for automatic selection
      //! if the device is not deselected by user and if the Device usage flag is enabled

      for (auto itVec = vecDeviceInfoList.begin(); itVec != vecDeviceInfoList.end(); itVec++)
      {
         //please add the check here for the all the technologies supported by Android Device
         if ((enDeviceCategory == itVec->enDeviceCategory)
                  || ((e8DEV_TYPE_ANDROIDAUTO == enDeviceCategory)
                           && (e8_ANDROID_DEVICE == itVec->rProjectionCapability.enDeviceType))
                  || ((e8DEV_TYPE_CARLIFE == enDeviceCategory)
                           && (e8_ANDROID_DEVICE == itVec->rProjectionCapability.enDeviceType))
                  || ((e8DEV_TYPE_MYSPIN == enDeviceCategory)
                           && (e8_ANDROID_DEVICE == itVec->rProjectionCapability.enDeviceType))
                  || ((e8DEV_TYPE_ONCAR == enDeviceCategory)
                           && (e8_ANDROID_DEVICE == itVec->rProjectionCapability.enDeviceType)))
         {
            enDeviceCategoryForSelection = itVec->enDeviceCategory;
            u32LastConnectedDeviceForSelection = itVec->u32DeviceHandle;
            break;
         }
      } // for (auto itvec = rvecDeviceInfo.begin()...
   }

   ETG_TRACE_USR4(("u32GetLastConnectedDeviceForSelection :Last connected Device for Selection "
            "on Changing the Device Usage Preference = %d and has the Device Category as %d", u32LastConnectedDeviceForSelection, ETG_ENUM(DEVICE_CATEGORY,
            enDeviceCategoryForSelection)));
   return u32LastConnectedDeviceForSelection;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vIsPreSelectionValidcb
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vIsPreSelectionValidcb(const t_U32 cou32DeviceHandle, t_Bool &rfrbIsPreSelectionValid)
{
   ETG_TRACE_USR1(("spi_tclDeviceSelector::vIsPreSelectionValidcb Entered with "
            "the Device Handle is %d", cou32DeviceHandle));

   rfrbIsPreSelectionValid = bIsPreSelectionValid(cou32DeviceHandle);

   ETG_TRACE_USR1(("spi_tclDeviceSelector::vIsPreSelectionValidcb -Device Handle - %0x valid for Autoselection %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            rfrbIsPreSelectionValid)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::bIsPreSelectionValid
 ***************************************************************************/
t_Bool spi_tclDeviceSelector::bIsPreSelectionValid(const t_U32 cou32DeviceHandle)
{
   //1. Check whether the device is existing or not - Return False if not found (bIsDeviceValid)
   //2. If found in the device list, check whether the device was deselcted manually -Return False if the deselected manually (bGetUserDeselectionFlag)
   //3. If found in the device list and auto deselected, then check whether any other device is selected or Selection in progress. - Return False if so

   ETG_TRACE_USR1(("spi_tclDeviceSelector::bIsPreSelectionValid Entered with "
            "the Device Handle is %d", cou32DeviceHandle));

   t_Bool bUserDeselectSet = false;
   std::map<t_U32, trEntireDeviceInfo> mapDeviceInfoList;
   tenDeviceSelectionState enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
   t_Bool bIsPre_SelectionValid = false;

   if (NULL != m_pDeviceList)
   {
      //get the entire device list info
      m_pDeviceList->vGetEntireDeviceList(mapDeviceInfoList);
      bUserDeselectSet = m_pDeviceList->bGetUserDeselectionFlag(cou32DeviceHandle);



      //find the device in the entire device list
      auto itMapDevList = mapDeviceInfoList.find(cou32DeviceHandle);
      if (itMapDevList != mapDeviceInfoList.end())
      {
         //Device is found in the list
         //check whether it was userdeselected
         if (false == bUserDeselectSet)
         {
            bIsPre_SelectionValid = true;
            ETG_TRACE_USR1(("spi_tclDeviceSelector::bIsPreSelectionValid Automatically deselected"));
            for (auto itMapDeviceList = mapDeviceInfoList.begin(); itMapDeviceList != mapDeviceInfoList.end();
                     itMapDeviceList++)
            {

               enDeviceSelectionState = itMapDeviceList->second.rDeviceInfo.rDeviceSelectionInfo.enDeviceSelectionState;
               if ((enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS)
                        || (enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTED)
                        || (enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_DESELECTION_INPROGRESS))
               {
                  bIsPre_SelectionValid = false;
                  break;
               }

            }
         }
      }
      ETG_TRACE_USR1(("spi_tclDeviceSelector::bIsPreSelectionValid  with UserDeselect flag set to %d", ETG_ENUM(BOOL,
               bUserDeselectSet)));
   }

   ETG_TRACE_USR1(("spi_tclDeviceSelector::bIsPreSelectionValid -Device Handle - %0x valid for Autoselection %d", cou32DeviceHandle, ETG_ENUM(BOOL,
            bIsPre_SelectionValid)));
   return bIsPre_SelectionValid;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vGetDeviceSelectorCurrentSelectionState(
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vGetDeviceSelectorCurrentSelectionState(trDeviceSelectionStateDetails& rfrDeviceSelectionDetails)
{
   m_oDeviceSelectionDetailsLock.s16Lock();
   rfrDeviceSelectionDetails.u32DeviceHandle = m_rDeviceSelectionDetails.u32DeviceHandle ;
   rfrDeviceSelectionDetails.enDeviceSelectionState = m_rDeviceSelectionDetails.enDeviceSelectionState;
   m_oDeviceSelectionDetailsLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclDeviceSelector::vGetDeviceSelectorCurrentSelectionState Left with Selection State Details with Device handle %d having Selection State %d ",
            rfrDeviceSelectionDetails.u32DeviceHandle, ETG_ENUM(SELECTION_STATE, rfrDeviceSelectionDetails.enDeviceSelectionState)));

}


/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vUpdateCurrentDeviceSelectorState()
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vUpdateCurrentDeviceSelectorState(const t_U32 cou32DeviceHandle, tenErrorCode enErrorcode,
         tenDeviceConnectionReq enDeviceConnectionReq, t_Bool bDeviceSelectionInfoUpdation)
{
   ETG_TRACE_USR1((" spi_tclDeviceSelector::vUpdateCurrentDeviceSelectorState DeviceID = 0x%x, Error = %d ", cou32DeviceHandle, ETG_ENUM(ERROR_CODE,
            enErrorcode)));

   tenResponseCode enRespCode = (e8NO_ERRORS == enErrorcode) ? e8SUCCESS : e8FAILURE;

   //! update the Device Selection state Details accordingly
   m_oDeviceSelectionDetailsLock.s16Lock();
   m_rDeviceSelectionDetails.u32DeviceHandle = cou32DeviceHandle;
   switch (enRespCode)
   {
      case e8SUCCESS:
         if (enDeviceConnectionReq == e8DEVCONNREQ_SELECT)
         {
            m_rDeviceSelectionDetails.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTED;
         }
         else
         {
            m_rDeviceSelectionDetails.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
         }
         break;

      case e8FAILURE:
         if (false == bDeviceSelectionInfoUpdation)
         {
            if (enDeviceConnectionReq == e8DEVCONNREQ_SELECT)
            {
               m_rDeviceSelectionDetails.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTION_ERROR;
            }
            else
            {
               m_rDeviceSelectionDetails.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_DESELECTION_ERROR;
            }
         }
         break;
   }

   ETG_TRACE_USR1(("spi_tclDeviceSelector::vUpdateCurrentDeviceSelectorState  Left Selection State Details with Device handle %d having Selection State %d ", m_rDeviceSelectionDetails.u32DeviceHandle, ETG_ENUM(SELECTION_STATE,
            m_rDeviceSelectionDetails.enDeviceSelectionState)));
   m_oDeviceSelectionDetailsLock.vUnlock();

}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector::vOnRetrailSetSelectionState(
 ***************************************************************************/
t_Void spi_tclDeviceSelector::vOnRetrailSetSelectionState(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclConnMngr::vOnRetrailSetSelectionState() entered with device handle %d", cou32DeviceHandle));

   trDeviceSelectionInfo rSelectDeviceInfo;
   if(NULL != m_pDeviceList)
   {
      m_pDeviceList->vGetDeviceSelectionInfo(cou32DeviceHandle,rSelectDeviceInfo);

      if(e8_SELECTION_STATE_DEVICE_RETRY_INPROGRESS == rSelectDeviceInfo.enDeviceSelectionState)
      {
         rSelectDeviceInfo.enDeviceSelectionErrorType = e8_DEVICE_SELECTION_ERROR_NONE;
         rSelectDeviceInfo.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTED;

         m_oDeviceSelectionDetailsLock.s16Lock();
         m_rDeviceSelectionDetails.u32DeviceHandle = cou32DeviceHandle;
         m_rDeviceSelectionDetails.enDeviceSelectionState = e8_SELECTION_STATE_DEVICE_SELECTED;
         m_oDeviceSelectionDetailsLock.vUnlock();

         m_pDeviceList->vSetDeviceSelectionInfo(cou32DeviceHandle, rSelectDeviceInfo);
      }
   }
}

//lint ?restore
