/*!
 *******************************************************************************
 * \file             spi_tclTechnologySelector.h
 * \brief            Responsible for handling technology selection from HMI
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Responsible for handling technology selection from HMI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 1.11.2016 |  Pruthvi Thej Nagaraju       | Initial Version
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLTECHNOLOGYSELECTOR_H_
#define SPI_TCLTECHNOLOGYSELECTOR_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclSelectionIntf.h"
#include "spi_tclTechnologySelectorResp.h"
#include "spi_tclDeviceListIntf.h"
#include "spi_tclMediatorIntf.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
struct trDeviceUsageEnabledConfig
{
   tenEnabledInfo enOverallSetting;
   tenEnabledInfo enDeviceSetting;
   tenEnabledInfo enEffectiveSetting;

   trDeviceUsageEnabledConfig& operator=(const trDeviceUsageEnabledConfig& corfrDeviceUsageEnabledConfig)
   {
      if( &corfrDeviceUsageEnabledConfig != this )
      {
         enOverallSetting = corfrDeviceUsageEnabledConfig.enOverallSetting;
         enDeviceSetting = corfrDeviceUsageEnabledConfig.enDeviceSetting;
         enEffectiveSetting = corfrDeviceUsageEnabledConfig.enEffectiveSetting;
      }   //if( &corfrAAPVideoFocusState != this )
      return *this;
   }
};

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
class spi_tclTechnologySelector: public spi_tclSelectionIntf
{
   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::spi_tclTechnologySelector
       ***************************************************************************/
      /*!
       * \fn     spi_tclTechnologySelector
       * \brief  Parameterized constructor
       * \param  poDeviceList : pointer to devicelist
       * \param  pResp : pointer to send asynchronous responses to listener
       * \param  poMediator : pointer to mediator
       **************************************************************************/
      spi_tclTechnologySelector(spi_tclDeviceListIntf* poDeviceList, spi_tclTechnologySelectorResp* pResp, spi_tclMediatorIntf* poMediator);

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::~spi_tclTechnologySelector
       ***************************************************************************/
      /*!
       * \fn     !spi_tclTechnologySelector
       * \brief  destructor
       **************************************************************************/
      ~spi_tclTechnologySelector();

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::vSetTechnologySelection
       ***************************************************************************/
      /*!
       * \fn     vSetTechnologySelection
       * \brief  To set the technology preference
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enUserAuthStatus : Authorization status
       **************************************************************************/
      t_Void vSetTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference);

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::vSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vSelectDevice
       * \brief  device selection request
       * \param  corfrSelectReq : Device selection request details
       **************************************************************************/
      virtual t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::vSelectDeviceResult
       ***************************************************************************/
      /*!
       * \fn     vSelectDeviceResult
       * \brief  device selection result
       * \param  corfrSelectReq : Device selection result details
       **************************************************************************/
      virtual t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

   private:
      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::vSendSelectDeviceResponse
       ***************************************************************************/
      /*!
       * \fn     vSendSelectDeviceResponse
       * \brief  send device selection response back to deviceselector
       * \param  enErrorCode :Error code
       **************************************************************************/
      virtual t_Void vSendSelectDeviceResponse(tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::vUpdateTechnologySelection
       ***************************************************************************/
      /*!
       * \fn     vUpdateTechnologySelection
       * \brief  Helper function to send technology selection responses to listener
       * \param  cou32DeviceHandle : Device ID
       * \param  enTechnologyPreference : Technology preference
       **************************************************************************/
      t_Void vUpdateTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference);

      /***************************************************************************
       ** FUNCTION:  spi_tclTechnologySelector::enGetEffectiveTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     enGetEffectiveTechnologyPreference
       * \brief  calculate the effective technology usage value
       * \param  u32DeviceHandle :device handle
       * \retval the effective technology enabled info for the device handle
       **************************************************************************/
      tenEnabledInfo enGetEffectiveTechnologyPreference(const t_U32 u32DeviceHandle);

      //! Device list pointer
      spi_tclDeviceListIntf* m_pDeviceList;
      //! TechnologySelector pointer
      spi_tclTechnologySelectorResp* m_pTechSelectorResp;
      //! Mediator pointer
      spi_tclMediatorIntf* m_pMediator;
      //! Indicates if the device selection request is pending
      t_Bool m_bIsSelectRequestPending;
};



#endif /* SPI_TCLTECHNOLOGYSELECTOR_H_ */
