/*!
 *******************************************************************************
 * \file             spi_tclSelectionValidator.h
 * \brief            Validate device selection request
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Validate device selection request
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 14.11.2016 |  Pruthvi Thej Nagaraju       | Initial Version
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLSELECTIONVALIDATOR_H_
#define SPI_TCLSELECTIONVALIDATOR_H_


#include "spi_tclDeviceListIntf.h"
#include "spi_tclMediatorIntf.h"


class spi_tclSelectionValidator
{
   public:

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::spi_tclSelectionValidator
       ***************************************************************************/
      /*!
       * \fn     spi_tclSelectionValidator
       * \brief  Constructor
       * \param  poDeviceList : Device list handler
       * \param  cou32SelectedDevice : selected device handle
       * \param  cobBusyStatus : Device selector busy status
       **************************************************************************/
      spi_tclSelectionValidator(spi_tclDeviceListIntf* poDeviceList, const t_U32 cou32SelectedDevice,
               const t_Bool cobBusyStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::spi_tclSelectionValidator
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclSelectionValidator
       * \brief  destructor
       * \param  corfrSelectReq : Device selection request details
       **************************************************************************/
      ~spi_tclSelectionValidator();

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::vValidateSelection
       ***************************************************************************/
      /*!
       * \fn     vValidateSelection
       * \brief  device selection validation. This in turn validated deviceid and the request
       * \param  corfrSelectReq : Device selection request details
       * \param  rfenErrorCode : Returns NO_ERRORS if validation is successful else
       *                         will return error code
       **************************************************************************/
       t_Void vValidateSelection(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode &rfenErrorCode);


      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::u32GetCurrSelectedDevice
       ***************************************************************************/
       /*!
        * \fn     u32GetCurrSelectedDevice
        * \brief  returns currently selected device
        * \param  corfrSelectReq : Device selection request details
        **************************************************************************/
      t_U32 u32GetCurrSelectedDevice() const
      {
         return m_u32CurrSelectedDevice;
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bDeviceSelectorBusy
       ***************************************************************************/
      /*!
       * \fn     bDeviceSelectorBusy
       * \brief  returns current status of deviceselector
       **************************************************************************/
      t_Bool bDeviceSelectorBusy() const
      {
         return m_bisDeviceSelectorBusy;
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::vValidateSelectDeviceRequest
       ***************************************************************************/
      /*!
       * \fn     vValidateSelectDeviceRequest
       * \brief  validates the device id for selection
       * \param  corfrSelectReq : Device selection request details
       * \param  rfenErrorCode : Returns NO_ERRORS if validation is successful else
       *                         will return error code
       * \retval true: if the device selection request id valid else false
       **************************************************************************/
      t_Bool bValidateDeviceID(const trSelectDeviceRequest &corfrSelDevReq, tenErrorCode &rfenErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bDeviceValidforAutoSelection
       ***************************************************************************/
      /*!
       * \fn     bDeviceValidforAutoSelection
       * \brief  validates the selected device request for automatic selection
       * \param  corfrSelectReq : Device selection request details
       * \retval true: if the device selection request id valid else false
       **************************************************************************/
      t_Bool bDeviceValidforAutoSelection(const trSelectDeviceRequest &corfrSelDevReq);

   private:
      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bIsProjectionEnabled
       ***************************************************************************/
      /*!
       * \fn     bIsProjectionEnabled
       * \brief  checks if the projection is enabled for the specified device type
       * \param  enDeviceType : Device selection request details
       * \retval true if projection is enabled
       **************************************************************************/
      t_Bool bIsProjectionEnabled(tenDeviceType enDeviceType);


      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bIsProjectionEnabled
       ***************************************************************************/
      /*!
       * \fn     bIsProjectionEnabled
       * \brief  checks if the projection is enabled for the specified device category
       * \param  enDeviceType : Device selection request details
       * \retval true if projection is enabled
       **************************************************************************/
      t_Bool bIsProjectionEnabled(tenDeviceCategory enDevCat, tenDeviceType enDeviceType);


      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bValidateSelectDeviceRequest
       ***************************************************************************/
      /*!
       * \fn     bValidateSelectDeviceRequest
       * \brief  validates the selected device request
       * \param  corfrSelectReq : Device selection request details
       * \param  rfenErrorCode : Returns NO_ERRORS if validation is successful else
       *                         will return error code
       * \retval true: if the device selection request id valid else false
       **************************************************************************/
      t_Bool bValidateSelectDeviceRequest(const trSelectDeviceRequest &corfrSelDevReq, tenErrorCode &rfenErrorCode);


      /***************************************************************************
       ** FUNCTION:  spi_tclSelectionValidator::bIsDeviceWhiteListed
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceWhiteListed
       * \brief  Check whether the Device is white listed
       * \param  cou32DeviceHandle : Device Handle
       * \retval true: if the device is white listed else false
       **************************************************************************/
      t_Bool bIsDeviceWhiteListed(const t_U32 cou32DeviceHandle);

      //! Device list handler
      spi_tclDeviceListIntf* m_pDeviceList;
      //! Currently selected device
      const t_U32 m_u32CurrSelectedDevice;
      //! Device selector status
      const t_Bool m_bisDeviceSelectorBusy;
};



#endif /* SPI_TCLSELECTIONVALIDATOR_H_ */
