/*!
 *******************************************************************************
 * \file             spi_tclDeviceSelector.h
 * \brief            Handles select and deselection of device
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Handles select and deselection of device
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 10.12.2014 | Shihabudheen P M             | Changed for blocking device usage 
                                             preference updates during 
                                             select/deselect is in progress. 

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICESELECTOR_H_
#define SPI_TCLDEVICESELECTOR_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDeviceListIntf.h"
#include "spi_tclMediatorIntf.h"
#include "spi_tclConnSettingsIntf.h"
#include "spi_tclSelectionIntf.h"
#include "Timer.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

//! Forward declarations
class spi_tclDevSelResp;
/*!
 * \class spi_tclDeviceSelector
 * \brief Handles selection  and deselection of device
 */

class spi_tclDeviceSelector
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::spi_tclDeviceSelector
       ***************************************************************************/
      /*!
       * \fn     spi_tclDeviceSelector(spi_tclDevSelResp *poRespInterface)
       * \brief  Parameterized Constructor
       * \param  poRespInterface: Pointer to response interface class
       * \sa      ~spi_tclDeviceSelector()
       **************************************************************************/
      spi_tclDeviceSelector(spi_tclDevSelResp *poRespInterface,
               spi_tclDeviceListIntf *poDeviceList, spi_tclMediatorIntf* poMediator,
               spi_tclConnSettingsIntf *poConnSettings,
               std::map<tenCompID, spi_tclSelectionIntf*> &rfrComponents);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::~spi_tclDeviceSelector
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclDeviceSelector()
       * \brief  Destructor
       * \sa     spi_tclDeviceSelector()
       **************************************************************************/
      virtual ~spi_tclDeviceSelector();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceSelector::bInitialize()
       ***************************************************************************/
      /*!
       * \fn      bInitialize()
       * \brief   Method to Initialize Device selector: Register for callbacks
       * \retval  true: if the initialization of Device selector is successful,
       *          false: on failure
       * \sa      bUnInitialize()
       **************************************************************************/
      virtual t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSelector::vUnInitialize()
       ***************************************************************************/
      /*!
       * \fn      vUnInitialize()
       * \brief   Method to UnInitialize Device selector: Unregister callbacks.
       * \sa      bInitialize()
       **************************************************************************/
      virtual t_Void vUnInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSelector::vLoadSettings()
       ***************************************************************************/
      /*!
       * \fn      vLoadSettings()
       * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
       * \sa      vSaveSettings()
       **************************************************************************/
      virtual t_Void vLoadSettings();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclDeviceSelector::vSaveSettings()
       ***************************************************************************/
      /*!
       * \fn      vSaveSettings()
       * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
       * \sa      vLoadSettings()
       **************************************************************************/
      virtual t_Void vSaveSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vRestoreSettings
       ***************************************************************************/
      /*!
       * \fn     vRestoreSettings
       * \brief  Resets selection timer
       **************************************************************************/
      t_Void vRestoreSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vSelectDevice()
       * \brief  Interface to select or deselect a particular device
       *         specified by device handle
       * \param  rfrSelectDevReqInfo : Device selection info
       **************************************************************************/
      t_Void vSelectDevice(trSelectDeviceRequest &rfrSelectDevInfo);


      /***************************************************************************
      ** FUNCTION: t_U32 spi_tclDeviceSelector::u32GetSelectedDeviceHandle()
      ***************************************************************************/
      /*!
      * \fn     u32GetSelectedDeviceHandle()
      * \brief  It provides the Device Handle of currently selected device, if any.
      *         If no device is selected, returns 0xFFFFFFFF.
      * \return Device Handle of Selected Device.
      **************************************************************************/
      t_U32 u32GetSelectedDeviceHandle() const;

     /***************************************************************************
      ** FUNCTION: t_U32 spi_tclDeviceSelector::u32GetSelectedDeviceId()
      ***************************************************************************/
      /*!
      * \fn     u32GetSelectedDeviceId()
      * \brief  It provides the Device Handle of currently selected device, if any.
      *         If no device is selected, returns 0xFFFFFFFF.
      * \return Device Handle of Selected Device.
      **************************************************************************/
      static t_U32 u32GetSelectedDeviceId();

      /***************************************************************************
       ** FUNCTION:  t_Void vSetDeviceSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDeviceSelectionMode(
       * \brief   Method to set the device selection mode to automatic/manual. Changes
       *          will take into effect on successive connection
       * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
       * \param   enTechnologyType : Device category
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vApplySelectionStrategy
       ***************************************************************************/
      /*!
       * \fn     vApplySelectionStrategy()
       * \brief  Called to trigger device selection automatically
       * param   Default argument - enDeviceCategory
       **************************************************************************/
      t_Void vApplySelectionStrategy(tenDeviceCategory enDeviceCategory = e8DEV_TYPE_UNKNOWN);

	  /***************************************************************************
	   ** FUNCTION:  spi_tclDeviceSelector::vOnHUWiFiAPStateChange
	   ***************************************************************************/
	  /*!
	   * \fn     vOnHUWiFiAPStateChange(t_Bool bIsSetToOn)
      * \brief  Called when WiFi state changes.
	  * \param  enSelectReason : Reason for Trigger
	  * \retval t_Void
      **************************************************************************/ 
	  t_Void vOnHUWiFiAPStateChange(t_Bool bIsSetToOn);
	  

   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /***************************************************************************
      ** FUNCTION: spi_tclDeviceSelector(const spi_tclDeviceSelector &corfobjRhs)
      ***************************************************************************/
      /*!
      * \fn      spi_tclDeviceSelector(const spi_tclDeviceSelector &corfobjRhs)
      * \brief   Copy constructor not implemented hence made protected to prevent
      *          misuse
      **************************************************************************/
      spi_tclDeviceSelector(const spi_tclDeviceSelector &corfobjRhs);

      /***************************************************************************
      ** FUNCTION: const spi_tclDeviceSelector & operator=(const spi_tclDeviceSelector &corfobjRhs);
      ***************************************************************************/
      /*!
      * \fn      const spi_tclDeviceSelector & operator=(const spi_tclDeviceSelector &corfobjRhs);
      * \brief   assignment operator not implemented hence made protected to
      *          prevent misuse
      **************************************************************************/
      const spi_tclDeviceSelector & operator=(
         const spi_tclDeviceSelector &corfobjRhs);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  Registers callbacks with mediator
       **************************************************************************/
      t_Void vRegisterCallbacks();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vReceiveSelectDeviceResCb
       ***************************************************************************/
      /*!
       * \fn     vReceiveSelectDeviceResCb()
       * \brief  Callback to receive response for Select Device
       * \param  enCompID: Component ID which posts select device result
       * \param  bSelDevRes: true if Selecting Device by enCompID is success.
       *         otherwise false
       **************************************************************************/
      t_Void vReceiveSelectDeviceResCb(tenCompID enCompID, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vDeviceDisconnectionCb
       ***************************************************************************/
      /*!
       * \fn     vDeviceDisconnectionCb()
       * \brief  Callback function for Device Disconnection
       * \param  cou32DeviceHandle: Device handle of the disconnected device
       **************************************************************************/
      t_Void vDeviceDisconnectionCb(const t_U32 cou32DeviceHandle) ;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vDeviceConnectionCb
       ***************************************************************************/
      /*!
       * \fn     vDeviceConnectionCb()
       * \brief  Callback function for Device Connection
       * \param  cou32DeviceHandle: Device handle of the Connected device
       **************************************************************************/
      t_Void vDeviceConnectionCb(const t_U32 cou32DeviceHandle) ;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vOnAutomaticDeviceSelection
       ***************************************************************************/
      /*!
       * \fn     vOnAutomaticDeviceSelection()
       * \brief  Called when a device is selected or deselected automatically
       * \param  cou32DeviceHandle: Device handle
       * \param  enDevConnReq: informs if the device has to be selected or unselected
       * \param  enDeviceCategory: Device category
	   * \param enSelectReason : Reason for Trigger
       **************************************************************************/
      t_Void vOnAutomaticDeviceSelection(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionReq enDevConnReq, 
			   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN,
			   tenSelectReason enSelectReason=e8_REASON_UNKNOWN);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vDeviceDeselectionOnFactoryReset
       ***************************************************************************/
      /*!
       * \fn     vDeviceDeselectionOnFactoryReset()
       * \brief  Called when a device is selected or deselected automatically on Factory Reset
       * \param  cou32DeviceHandle: Device handle
       * \param  enDevConnReq: informs if the device has to be selected or unselected
       * \param  enDeviceCategory: Device category
       **************************************************************************/
      t_Void vDeviceDeselectionOnFactoryReset(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionReq enDevConnReq);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vOnSetUserDeselect
       ***************************************************************************/
      /*!
       * \fn     vOnSetUserDeselect()
       * \brief  Called when a device is deselected due to user action
       * \param  cou32DeviceHandle: Device handle
       * \param  bUserDeselect: User Deselection Flag
       **************************************************************************/
      t_Void vOnSetUserDeselect(const t_U32 cou32DeviceHandle, t_Bool bUserDeselect);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vSendDeviceSelectionReq
       ***************************************************************************/
      /*!
       * \fn     vSendDeviceSelectionReq()
       * \brief  Sends select device call to other SPI components
       * \param  corfrSelDevinfo : Structure containing information about the device
       *                           to be selected
       * \param  enCompID: Component ID which posts select device result
       **************************************************************************/
      t_Void vSendDeviceSelectionReq(const trSelectDeviceRequest &corfrSelDevinfo,
               tenCompID enCompID);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vEvaluateDeviceSelectionReq
       ***************************************************************************/
      /*!
       * \fn     vEvaluateDeviceSelectionReq()
       * \brief  Checks if some other device is selected. If so first deselect the
       *         already selected device and then send select device request
       **************************************************************************/
      t_Void vEvaluateDeviceSelectionReq();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::bDeviceSelectionComplete
       ***************************************************************************/
      /*!
       * \fn     bDeviceSelectionComplete()
       * \brief  Evaluates whether the select device request is complete or not
       * \param  cou32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDevConnReq    : Identifies the Connection Request.
       * \param  enErrorCode     : Returns the error code
       * \retval true: if the processing of select device request is complete.
       *         false: If further actions are needed to complete current select
       *                device request
       **************************************************************************/
      t_Bool bDeviceSelectionComplete(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionReq enDevConnReq, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vSendSelectDeviceResult
       ***************************************************************************/
      /*!
       * \fn     vSendSelectDeviceResult()
       * \brief  Sends select device result to interested SPI components
       * \param  corfrSelDevinfo : Structure containing information about the selection request
       * \param  enErrorCode     : Error code
       **************************************************************************/
      t_Void vSendSelectDeviceResult(const trSelectDeviceRequest &corfrSelDevReq, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vSendSelectDeviceResultToHMI
       ***************************************************************************/
      /*!
       * \fn     vSendSelectDeviceResultToHMI()
       * \brief  Sends select device result to HMI
       * \param  corfrSelDevinfo : Structure containing information about the selection request
       * \param  enErrorCode     : Error code
       **************************************************************************/
      t_Void vSendSelectDeviceResultToHMI(trSelectDeviceRequest &rfrSelDevReq, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vStartComponentResponseTimer
       ***************************************************************************/
      /*!
       * \fn     vStartComponentResponseTimer
       * \brief  Starts timer to monitor response from SPI components
       **************************************************************************/
      t_Void vStartComponentResponseTimer(tenDeviceCategory enDevCategory);

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceSelector::vUpdateDevSelectorBusyStatus(t_Bool)
      ***************************************************************************/
      /*!
      * \fn     vUpdateDevSelectorBusyStatus
      * \brief  Update busy status for Device selector
      * \param  bSelectorStatus :[IN] True if selector is busy, false otherwise.
      **************************************************************************/
      t_Void vUpdateDevSelectorBusyStatus(t_Bool bSelectorStatus,t_U32 cou32DeviceHandle);
	  
      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceSelector::enGetNextComponentID(t_Bool)
      ***************************************************************************/
      /*!
      * \fn     enGetNextComponentID
      * \brief  Provides the next component to be selected 
      * \param  enDeviceCat   : Device category
      * \param  enDevConnReq   : Device selection request
      * \param  enCurrentCompID:ID of Component whose selection has just completed
      **************************************************************************/
      tenCompID enGetNextComponentID(tenDeviceCategory enDeviceCat,
                    tenDeviceConnectionReq enDevConnReq, tenCompID enCurrentCompID);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vLoadSelectionSequence
       ***************************************************************************/
      /*!
       * \fn     vLoadSelectionSequence
       * \brief  Loads device selector selection sequence. to be called on loadsettings
       **************************************************************************/
      t_Void vLoadSelectionSequence();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vLoadDeselectionSequence
       ***************************************************************************/
      /*!
       * \fn     vLoadDeselectionSequence
       * \brief  Loads device selector deselection sequence. to be called on loadsettings
       **************************************************************************/
      t_Void vLoadDeselectionSequence();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::u32GetLastConnectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetLastConnectedDevice
       * \brief  Returns device handle of device which is last connected
       * \retval Device handle of the last connected device. returns 0 if no valid
       *         device is connected
       **************************************************************************/
      t_U32 u32GetLastConnectedDevice();

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceSelector::vUpdateDevSelectionInfo(t_Bool)
      ***************************************************************************/
      /*!
      * \fn     vUpdateDevSelectionInfo
      * \brief  Update device selection info for cou32DeviceHandle
      * \param  cou32DeviceHandle :[IN] Device handle
      * \param  enErrorcode :[IN] Error Code
      * \param  enDeviceConnectionReq :[IN] Selection request type
      **************************************************************************/
      t_Void vUpdateDevSelectionInfo(const t_U32 cou32DeviceHandle,
               tenErrorCode enErrorcode, tenDeviceConnectionReq enDeviceConnectionReq);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vStartDeviceSelection
       ***************************************************************************/
      /*!
       * \fn     vStartDeviceSelection
       * \brief  Start device selection sequence
       **************************************************************************/
      t_Void vStartDeviceSelection(trSelectDeviceRequest &rSelectDevReqInfo);


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vDeviceChanged
       ***************************************************************************/
      /*!
       * \fn     vDeviceChanged
        * \param  cou32DeviceHandle :[IN] Device handle
       **************************************************************************/
      t_Void vDeviceChanged(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::bCompRespTimerCb
       ***************************************************************************/
      /*!
       * \fn     bCompRespTimerCb
       * \brief  called on expiry of component response timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
       **************************************************************************/
      static t_Bool bCompRespTimerCb(timer_t rTimerID, t_Void *pvObject,
               const t_Void *pvUserData);
			   
      /***************************************************************************
       ** FUNCTION:  tenDeviceConnectionType spi_tclDeviceSelector::enGetDeviceConnectionType
       ***************************************************************************/
      /*!
       * \fn     tenDeviceConnectionType enGetDeviceConnectionType()
       * \brief  Method to get the transport on which device is connected.
       * \param  cou32DeviceHandle: Device handle
       * \param  enDevConnReq: informs if the device has to be selected or unselected
       * \param  enDeviceCategory: Device category
	   * \retval tenDeviceConnectionType: ConnectionType of the device
       **************************************************************************/
      tenDeviceConnectionType enGetDeviceConnectionType(t_U32 cou32DeviceHandle, 
	                                                    tenDeviceConnectionReq enDevConnReq,  
	                                                    tenDeviceCategory enDevCategory);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceSelector::bIsConnRetrialRequired
       ***************************************************************************/
      /*!
       * \fn     tenDeviceConnectionType bIsConnRetrialRequired()
       * \brief  Method to check whether the retry mechanism is required to start session again.
       * \param  cou32DeviceHandle: Device handle
       * \param  enDevConnReq: informs if the device has to be selected or unselected
       * \param  enDeviceCategory: Device category
	   * \param  enSelectReason : Reason for Select/Deselect
	   * \retval t_Bool: TRUE - if retry required.
       **************************************************************************/
      t_Bool bIsConnRetrialRequired(t_U32 cou32DeviceHandle, 
	                                tenDeviceConnectionReq enDevConnReq,  
	                                tenDeviceCategory enDevCategory,
									tenSelectReason enSelectReason);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSelector::vContinueWithSelection
       ***************************************************************************/
      /*!
       * \fn     tenDeviceConnectionType vContinueWithSelection()
       * \brief  Method to perform selection/deselection.
       * \param  cou32DeviceHandle: Device handle
       * \param  enDevConnReq: informs if the device has to be selected or unselected
       * \param  enDeviceCategory: Device category
	   * \param  enSelectReason : Reason for Select/Deselect
	   * \retval t_Void
       **************************************************************************/
      t_Void vContinueWithSelection(t_U32 u32DeviceHandle, 
	  tenDeviceConnectionReq enDevConnReq,  
	  tenDeviceCategory enDevCategory,
	  tenSelectReason enSelectReason=e8_REASON_UNKNOWN);


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::u32GetLastConnectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetLastConnectedDevice
       * \brief  Returns device handle of device which is last connected
       * \retval Device handle of the last connected device for Selection on Changing the DeviceUsagePreference. returns 0 if no valid
       *         device is connected
       **************************************************************************/
      t_U32 u32GetLastConnectedDevice(tenDeviceCategory enDeviceCategory);


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSelector::vIsPreSelectionValidcb
       ***************************************************************************/
      /*!
       * \fn     vIsPreSelectionValidcb
       * \brief  Returns boolean value, True : if the pre-selection request is valid for the device selection request once the device is connects to carplaywirless AP
       * \param  cou32DeviceHandle           : [IN] Device handle
       * \param  rfrbIsPreSelectionValid     : [OUT] boolean value
       **************************************************************************/
      t_Void vIsPreSelectionValidcb(const t_U32 cou32DeviceHandle, t_Bool &rfrbIsPreSelectionValid);
	  
	        /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceSelector::bIsPreSelectionValid(const t_U32 cou32DeviceHandle)
       ***************************************************************************/
      /*!
       * \fn     vIsPreSelectionValidcb
       * \brief  Returns boolean value, True : if the pre-selection request is valid for the device selection request once the device is connects to carplaywirless AP
       * \param  cou32DeviceHandle           : [IN] Device handle
       **************************************************************************/
      t_Bool bIsPreSelectionValid(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceSelector::vGetDeviceSelectorCurrentSelectionState(const t_U32 cou32DeviceHandle)
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceSelectorCurrentSelectionState
       * \Breif  Gets the Current Selection State
       * \param  rfrDeviceSelectionDetails           : [OUT] structure to hold the DevceSelection Details
       **************************************************************************/
      t_Void vGetDeviceSelectorCurrentSelectionState(trDeviceSelectionStateDetails& rfrDeviceSelectionDetails);

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceSelector::vUpdateCurrentDeviceSelectorState()
      ***************************************************************************/
      /*!
      * \fn     vUpdateCurrentDeviceSelectorState
      * \brief  Update device selection info for cou32DeviceHandle
      * \param  cou32DeviceHandle :[IN] Device handle
      * \param  enErrorcode :[IN] Error Code
      * \param  enDeviceConnectionReq :[IN] Selection request type
      **************************************************************************/
      t_Void vUpdateCurrentDeviceSelectorState(const t_U32 cou32DeviceHandle,
               tenErrorCode enErrorcode, tenDeviceConnectionReq enDeviceConnectionReq, t_Bool bDeviceSelectionInfoUpdation);
			   
      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vOnRetrailSetSelectionState
       ***************************************************************************/
      /*!
       * \fn     vOnRetrailSetSelectionState
       * \brief  updates the Selection state
       * \param cou32DeviceHandle: Device Handle of the device
       **************************************************************************/
      t_Void vOnRetrailSetSelectionState(const t_U32 cou32DeviceHandle);

      //! pointer to Mediator for posting messages
      spi_tclMediatorIntf *m_poMediator;

      //! pointer to Device selection response
      spi_tclDevSelResp *m_poDevSelResp;

      //! Device to be deselected
      t_Bool m_bDeviceSwitchRequest;

      //! Currently selected device
      static t_U32 m_u32CurrSelectedDevice;

      //! Store current select device request
      trSelectDeviceRequest m_rSelectDevReqInfo;

      //! Store info of device to be deselected
      trSelectDeviceRequest m_rDeselectDevReqInfo;

      //! Set when busy with another request for device selection
      t_Bool m_bIsDevSelectorBusy;

      //! Stores the current component ID
      tenCompID m_enSelectedComp;

      //! Stores the component response timer ID
      timer_t m_rCompRespTimerID;

      //! Stores selection sequence
      std::map<tenDeviceCategory, std::vector<tenCompID>> m_mapSelectionSequence;

      //! Stores deselection sequence
      std::map<tenDeviceCategory, std::vector<tenCompID>> m_mapDeselectionSequence;

      //! Currently reported device connections
      t_U32 m_u32RecentlyConnectedDevice;

      //! pointer to devicelist interface
      spi_tclDeviceListIntf *m_pDeviceList;

      //! Pointer to connection settings Interface
      spi_tclConnSettingsIntf *m_poConnSettings;

      //! Stores component pointers
      std::map<tenCompID, spi_tclSelectionIntf*> m_mapComponents;

      //! Lock to protect components map
      Lock m_oLockComponents;

      //! Lock to avoid sync issues while accessing validator
      Lock m_oSelectionValidatorLock;

      //! Structure that stores the Selection State
      trDeviceSelectionStateDetails m_rDeviceSelectionDetails;

      //! Lock to avoid sync issues while accessing validator
      Lock m_oDeviceSelectionDetailsLock;
};

/*! } */
#endif /* SPI_TCLDEVICESELECTOR_H_ */
