/*!
 *******************************************************************************
 * \file             spi_tclDeviceListHandler.h
 * \brief            Interface to access Device List
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Interface to access Device List
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.11.2016 |  Pruthvi Thej Nagaraju       | Initial Version
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICELISTINTF_H_
#define SPI_TCLDEVICELISTINTF_H_


/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_ConnMngrTypeDefines.h"

/*!
 * \class spi_tclDeviceListIntf
 * \brief Interface for Device List handling
 */
class spi_tclDeviceListIntf
{

   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::spi_tclDeviceListIntf
       ***************************************************************************/
      /*!
       * \fn     spi_tclDeviceListIntf()
       * \brief  Default Constructor
       * \sa      ~spi_tclDeviceListIntf()
       **************************************************************************/
      spi_tclDeviceListIntf(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::~spi_tclDeviceListIntf
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclDeviceListIntf()
       * \brief   Destructor
       * \sa     spi_tclDeviceListIntf()
       **************************************************************************/
      virtual ~spi_tclDeviceListIntf(){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bRestoreDeviceList
       ***************************************************************************/
      /*!
       * \fn     bRestoreDeviceList
       * \brief  Reads database to restore device history to device list
       * \retval true: if the history was added successfully to device list
       *         false: on Error
       **************************************************************************/
      virtual t_Bool bRestoreDeviceList()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bSaveDeviceList
       ***************************************************************************/
      /*!
       * \fn     bSaveDeviceList
       * \brief  Saves the device list to persistent memory device list
       **************************************************************************/
      virtual t_Void bSaveDeviceList()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vClearDeviceList
       ***************************************************************************/
      /*!
       * \fn     vClearDeviceList
       * \brief  clears Device List
       **************************************************************************/
      virtual t_Void vClearDeviceList()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enAddDeviceToList
       ***************************************************************************/
      /*!
       * \fn     enAddDeviceToList
       * \brief  Adds device to device list
       * \param cou32DeviceHandle: Device Handle to be added to device list
       * \param rfrDeviceInfo: Device Info of the device to be added to list
       **************************************************************************/
      virtual tenDeviceStatusInfo enAddDeviceToList(const t_U32 cou32DeviceHandle,
               const trDeviceInfo &rfrDeviceInfo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bRemoveDeviceFromList
       ***************************************************************************/
      /*!
       * \fn     bRemoveDeviceFromList
       * \brief  Removes device to device list
       * \param  coU32DeviceHandle:  Device Handle to be removed from device list
       **************************************************************************/
      virtual t_Bool bRemoveDeviceFromList(const t_U32 coU32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceList
       * \brief  interface to get SPI devices list
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with all SPI devices
       **************************************************************************/
      virtual t_Void vGetDeviceList(const t_U32 cou32DeviceHandle, std::vector<trDeviceInfo>& rfvecDeviceInfoList)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetConnectedDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetConnectedDeviceList
       * \brief  interface to get SPI devices list with only connected devices
       * \param  rfvecDeviceInfoList: reference to the device list which will
       *         be populated with the connected SPI devices
       **************************************************************************/
      virtual t_Void vGetConnectedDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetEntireDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetEntireDeviceList
       * \brief  interface to get SPI devices list with complete device info
       * \param  rfrmapDeviceInfoList: reference to the device list which will
       *         be populated with the detected SPI devices
       **************************************************************************/
      virtual t_Void vGetEntireDeviceList(std::map<t_U32, trEntireDeviceInfo> &rfrmapDeviceInfoList)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetUserDeselectionFlag
       ***************************************************************************/
      /*!
       * \fn     vSetUserDeselectionFlag
       * \brief  Sets the flag when user deselects the device from HMI
       * \param  cou32DeviceHandle :Device Handle
       * \param  bState : indicates the value of UserDeselectionFlag
       **************************************************************************/
      virtual t_Void vSetUserDeselectionFlag(const t_U32 cou32DeviceHandle, t_Bool bState)=0;

       /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bGetUserDeselectionFlag
       ***************************************************************************/
      /*!
       * \fn     bGetUserDeselectionFlag
       * \brief  returns the flag when user deselects the device from HMI
       * \param  cou32DeviceHandle :Device Handle
       * \retval  bState : indicates the value of UserDeselectionFlag
       **************************************************************************/
      virtual t_Bool bGetUserDeselectionFlag(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceListIntf::u32GetNoofConnectedDevices
      ***************************************************************************/
     /*!
      * \fn     u32GetNoofConnectedDevices
      * \brief  returns no of connected devices
      * \retval  t_U32 : indicates the no of connected devices
      **************************************************************************/
      virtual t_U32 u32GetNoofConnectedDevices()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceValidity
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceValidity
       * \brief  Sets the validity of the device (based on certification)
       * \param  u32DeviceHandle : Device handle of which the validity has to be set
       * \param  bDeviceValid: true if device is certified else false
       **************************************************************************/
      virtual t_Void vSetDeviceValidity(const t_U32 cou32DeviceHandle,
               t_Bool bDeviceValid)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bIsDeviceValid
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceValid
       * \brief  Interface to get device information for a particular device handle
       * \param  cou32DeviceHandle: Device handle of the  device
       * \retval true: if the device is valid otherwise false
       **************************************************************************/
      virtual t_Bool bIsDeviceValid(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bGetDeviceInfo
       ***************************************************************************/
      /*!
       * \fn     bGetDeviceInfo
       * \brief  Interface to get device information for a particular device handle
       * \param  u32DeviceHandle: Device handle of the  device
       * \param  rfrDeviceInfo: Device information for the requested device
       **************************************************************************/
      virtual t_Bool bGetDeviceInfo(t_U32 u32DeviceHandle,
               trDeviceInfo &rfrDeviceInfo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceName
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceName
       * \brief  Sets the device name of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszDeviceName : Device Name to be set
       **************************************************************************/
      virtual t_Void vSetDeviceName(const t_U32 cou32DeviceHandle, t_String &rfrszDeviceName)=0;
      
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetDeviceName
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceName
       * \brief  To get the device name of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device
       * \param rfrszDeviceName : Device Name to be get
       **************************************************************************/      
      virtual t_Void vGetDeviceName(t_U32 cou32DeviceHandle, t_String &rfszDeviceName) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vAddDeviceToHistory
       ***************************************************************************/
      /*!
       * \fn     vAddDeviceToHistory
       * \brief  Adds device to history
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       **************************************************************************/
      virtual t_Void vAddDeviceToHistory(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vRemoveDeviceFromHistory
       ***************************************************************************/
      /*!
       * \fn     vRemoveDeviceFromHistory
       * \brief  removes device from history
       * \param  cou32DeviceHandle: Device Handle of the device to be removed from history
       **************************************************************************/
      virtual t_Void vRemoveDeviceFromHistory(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bGetDeviceHistory
       ***************************************************************************/
      /*!
       * \fn     bGetDeviceHistory
       * \brief  Interface to get device history
       * \param  rfvecDevHistory: reference to vector containing device history
       **************************************************************************/
      virtual t_Bool bGetDeviceHistory(std::vector<trEntireDeviceInfo> &rfvecDevHistory) const = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceCategory
       * \brief  Returns Device category for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      virtual tenDeviceCategory enGetDeviceCategory(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceCategory
       * \brief  Sets the Device category for the device
       * \param  cou32DeviceHandle :Device Handle
       * \param  enDeviceCategory : indicates the device category to be used for selection
       **************************************************************************/
      virtual t_Void vSetDeviceCategory(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceConnType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceConnType
       * \brief  Returns Device connection type for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      virtual tenDeviceConnectionType enGetDeviceConnType(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceConnStatus
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceConnStatus
       * \brief  Returns Device connection status for the provided device ID
       * \param  cou32DeviceHandle: Device Handle for which device
       *         category is requested
       **************************************************************************/
      virtual tenDeviceConnectionStatus enGetDeviceConnStatus(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetBTAddress
       ***************************************************************************/
      /*!
       * \fn     vSetBTAddress
       * \brief  Sets the BT Address of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param corfrszBTAddress : BT Address to be set
       **************************************************************************/
      virtual t_Void vSetBTAddress(const t_U32 cou32DeviceHandle,
            const t_String& corfrszBTAddress)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetBTAddress
       ***************************************************************************/
      /*!
       * \fn     vGetBTAddress
       * \brief  Gets the BT Address of the device : cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle whose connection status is requested
       * \param  rfszBTAddress: [OUT] BT address for the requested device handle
       **************************************************************************/
      virtual t_Void vGetBTAddress(const t_U32 cou32DeviceHandle, t_String &rfszBTAddress)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceSelection
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelection
       * \brief  Sets the bDeviceSelection flag if the device was selected during
       *         a power cycle
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      virtual t_Void vSetDeviceSelection(const t_U32 cou32DeviceHandle, t_Bool bSelectDev)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::u32GetSelectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetSelectedDevice
       * \brief  returns the currently selected device
       * \retval Device Handle of the currently selected device. O if not found
       **************************************************************************/
      virtual t_U32 u32GetSelectedDevice()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bWasDeviceSelected
       ***************************************************************************/
      /*!
       * \fn     bWasDeviceSelected
       * \brief  Returns whether a device was selected or not previously.
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      virtual t_Bool bWasDeviceSelected(const t_U32 cou32DeviceHandle)=0;


      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceConnectionStatus
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectionStatus
       * \brief  Sets the Device Connection status flag for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle for which device
       *         Connection is requested
       **************************************************************************/
      virtual t_Void vSetDeviceConnectionStatus(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionStatus enConnStatus)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bIsDeviceConnected
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceConnected
       * \brief  Returns whether a device was Connected or not previously
       * \param  cou32DeviceHandle: Device Handle for which device
       *         selection is requested
       **************************************************************************/
      virtual t_Bool bIsDeviceConnected(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDAPSupport
       ***************************************************************************/
      /*!
       * \fn     vSetDAPSupport
       * \brief  Sets the DAP Supported field for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle
       **************************************************************************/
      virtual t_Void vSetDAPSupport(const t_U32 cou32DeviceHandle, t_Bool bDAPSupported)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bIsDAPSupported
       ***************************************************************************/
      /*!
       * \fn     bIsDAPSupported
       * \brief  Returns the DAP Support status of cou32DeviceHandle
       * \param  cou32DeviceHandle: Device Handle
       * \retval true: If device supports DAP. false otherwise
       **************************************************************************/
      virtual t_Bool bIsDAPSupported(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
      ** FUNCTION: virtual t_Void spi_tclDeviceListIntf::vSetDeviceUsagePreference
      ***************************************************************************/
      /*!
      * \fn     vSetDeviceUsagePreference
      * \brief  Interface to set the preference for the usage of the connected
      *         Mirror Link/DiPO device. This can be set for individual devices or
      *         for all the connected Mirror Link/DiPO devices.
      * \param  cou32DeviceHandle  : Unique handle which identifies the device.
      *              If the value is 0xFFFFFFFF, then this function sets the overall
      *              preference usage for Mirror Link/DiPO.
      * \param  enEnabledInfo    : Enable Information.
      **************************************************************************/
      virtual t_Void vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenEnabledInfo enEnabledInfo) = 0;

      /***************************************************************************
      ** FUNCTION: virtual t_Void spi_tclDeviceListIntf::vGetDeviceUsagePreference
      ***************************************************************************/
      /*!
      * \fn     vGetDeviceUsagePreference
      * \brief  Interface to get the preference for the usage of the
      *         connected Mirror Link/iPod Out device during startup.
      * \param  cou32DeviceHandle  : Unique handle which identifies the device.
      *              If the value is 0xFFFFFFFF, then this function sets the overall
      *              preference usage for Mirror Link/DiPO.
      * \param  [OUT] rfenEnabledInfo : Enable Information.
      **************************************************************************/
      virtual t_Void vGetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenEnabledInfo& rfenEnabledInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bSetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bSetDevProjUsage
       * \brief  Called when the SPI feature is turned ON or OFF by the user
       * \param  enSPIType: indicates the SPI Type (Mirrorlink, Dipo ..)
       * \param  enSPIState : indicates the service status of enSPIType
       **************************************************************************/
      virtual t_Bool bSetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo enSPIState, tenDeviceType enDeviceType)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bGetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bGetDevProjUsage
       * \brief  Returns the present SPI Feature Stata
       * \param  enSPIType: SPI service type for which state is
       *         requested (Mirrorlink, DiPo ..)
       * \param  rfrEnabledInfo: [OUT]Indicates if enSPIType is enabled or not
       * \retval true if enSPIType is enabled.
       *         false if enSPIType is disabled
       **************************************************************************/
      virtual t_Bool bGetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo &rfrEnabledInfo, tenDeviceType enDeviceType)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::u32GetLastSelectedDevice
       ***************************************************************************/
      /*!
       * \fn     u32GetLastSelectedDevice
       * \brief  Returns the last selected device
       * \retval Device handle of the selected device
       **************************************************************************/
      virtual t_U32 u32GetLastSelectedDevice()=0;

     /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetSelectError
       ***************************************************************************/
      /*!
       * \fn     vSetSelectError
       * \brief  set to true if the error occurs on device side during selection
       *         Ex: Role switch failure
       * \param cou32DeviceHandle: Device handle of the selected device
       * \parama bIsError value of the device error flag
       **************************************************************************/
      virtual t_Void vSetSelectError(const t_U32 cou32DeviceHandle, t_Bool bIsError)=0;

     /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bIsSelectError
       ***************************************************************************/
      /*!
       * \fn     bIsSelectError
       * \brief  Returns true if the error occurs on device side during selection
      *         Ex: Role switch failure
       * \param cou32DeviceHandle: Device handle of the selected device
       * \retval value of the device error flag
       **************************************************************************/
      virtual t_Bool bIsSelectError(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vClearPrivateData
       ***************************************************************************/
      /*!
       * \fn     vClearPrivateData
       * \brief  Clears previously connected devices from the list
       **************************************************************************/
      virtual t_Void vClearPrivateData()=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::bGetUSBConnectedFlag
       ***************************************************************************/
      /*!
       * \fn     bGetUSBConnectedFlag
       * \brief  Returns true if when USB connection for the device is active
       * \param cou32DeviceHandle: Device handle
       * \retval true if when USB connection for the device is active else false
       **************************************************************************/
      virtual t_Bool bGetUSBConnectedFlag(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vClearPrivateData
       ***************************************************************************/
      /*!
       * \fn     vClearPrivateData
       * \brief  Set USB connection for the device (active/inactive)
       * \param cou32DeviceHandle: Device handle
       * \param bUSBConnected: Indicates whether USB connection is active or not
       **************************************************************************/
      virtual t_Void vSetUSBConnectedFlag(const t_U32 cou32DeviceHandle, t_Bool bUSBConnected)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetUserPreference
       ***************************************************************************/
      /*!
       * \fn     enGetUserPreference
       * \brief  Returns user preference SPI technology for the device
       * \param cou32DeviceHandle: Device handle
       **************************************************************************/
      virtual tenUserPreference enGetUserPreference(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetUserPreference
       ***************************************************************************/
      /*!
       * \fn     vSetUserPreference
       * \brief   Sets user preference SPI technology for the device
       * \param cou32DeviceHandle: Device handle
       * \param  enUserPref: user preference SPI technology for the device
       **************************************************************************/
      virtual t_Void vSetUserPreference(const t_U32 cou32DeviceHandle, tenUserPreference enUserPref)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetSPISupport
       ***************************************************************************/
      /*!
       * \fn     enGetSPISupport
       * \brief  Get SPI technology support for indicated device handle and category
       * \param  cou32DeviceHandle: Device handle
       * \param  enSPIType: SPI technology supporet to be checked
       * \retval Indicates if this SPi technology is supported
       **************************************************************************/
      virtual tenSPISupport enGetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetSPISupport
       ***************************************************************************/
      /*!
       * \fn     vSetSPISupport
       * \brief  Sets SPI technology support for indicated device handle and category
       * \param  cou32DeviceHandle: Device handle
       * \param  enSPIType: SPI technology supporet to be checked
       * \param enSPISupport: indicates SPi technology is supported
       **************************************************************************/
      virtual t_Void vSetSPISupport(const t_U32 cou32DeviceHandle, tenDeviceCategory enSPIType,
               tenSPISupport enSPISupport)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetUserAuthorizationStatus
       ***************************************************************************/
      /*!
       * \fn     vSetUserAuthorizationStatus
       * \brief  Sets device authorization status
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      virtual t_Void vSetUserAuthorizationStatus(const t_U32 cou32DeviceHandle, tenUserAuthorizationStatus enUserAuthStatus)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetUserAuthorizationStatus
       ***************************************************************************/
      /*!
       * \fn     enGetUserAuthorizationStatus
       * \brief  Get device authorization status
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      virtual tenUserAuthorizationStatus enGetUserAuthorizationStatus(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceType
       * \brief  Gets the device type
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      virtual tenDeviceType enGetDeviceType(const t_U32 cou32DeviceHandle)=0;
      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceListIntf::enGetUSBPortType
      ***************************************************************************/
      /*!
      * \fn     enGetUSBPortType
      * \brief  Gets the USB port type
      * \param  cou32DeviceHandle: Device handle
      **************************************************************************/
      virtual tenUSBPortType enGetUSBPortType(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceSelectionInfo
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelectionInfo
       * \brief  Sets the device selection info for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  corfrDeviceSelectioninfo: Device selection info
       **************************************************************************/
      virtual t_Void vSetDeviceSelectionInfo(const t_U32 cou32DeviceHandle, const trDeviceSelectionInfo &corfrDeviceSelectioninfo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceSelectionErrorType
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelectionErrorType
       * \brief  Sets the device selection error type for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  enDeviceSelectionErrorType: Device selection error type
       **************************************************************************/
      virtual t_Void vSetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle,const tenDeviceSelectionErrorType coenDeviceSelectionErrorType)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceSelectionErrorType
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceSelectionErrorType
       * \brief  Gets the device selection error type for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      virtual tenDeviceSelectionErrorType enGetDeviceSelectionErrorType(const t_U32 cou32DeviceHandle)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetDeviceSelectionInfo
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceSelectionInfo
       * \brief  Get the device selection info for cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle
       * \param  rfrDeviceSelectioninfo: [OUT]Device selection info
       **************************************************************************/
      virtual t_Void vGetDeviceSelectionInfo(const t_U32 cou32DeviceHandle, trDeviceSelectionInfo &rfrDeviceSelectioninfo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetSpiFeatureSupport(const trSpiFeatureSupport&)
       ***************************************************************************/
      /*!
       * \fn     vSetSpiFeatureSupport
       * \brief  Sets SPI feature support for all the technologies (as per KDS)
       * \param  corfrSpiFeatureSupp: [IN] reference for SPI features supported structure
       **************************************************************************/
      virtual t_Void vSetSpiFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupp)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::u32GetDeviceIDFromBTAddress
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromBTAddress(coszBTAddress)
       * \brief  Returns DeviceID for the Device having BT address coszBTAddress
       * \param  coszBTAddress: [IN] BTAddress of the device
       * \retval  t_U32: Device ID having BTAddress coszBTAddress
       **************************************************************************/
      virtual t_U32 u32GetDeviceIDFromBTAddress(const t_String &coszBTAddress)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::u32GetDeviceIDFromSerialNo
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromSerialNo(coszSerialNo)
       * \brief  Returns DeviceID for the Device having Serial number coszSerialNo
       * \param  coszSerialNo: [IN] Serial number of the device
       * \retval  t_U32: Device ID having Serial number coszSerialNo
       **************************************************************************/
      virtual t_U32 u32GetDeviceIDFromSerialNo(const t_String &coszSerialNo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetSessionTransportFromSerialNumber()
       ***************************************************************************/
      /*!
      * \brief  Get Transport on which session is established using SerialNumber
      * \param  coszSerialNo: [IN] Serial Number
      * \retval  enSessionTransport: Session Transport Type
      **************************************************************************/
      virtual tenSessionTransport enGetSessionTransportFromSerialNumber(const t_String coszSerialNo)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::u32GetDeviceIDFromName
       ***************************************************************************/
      /*!
       * \fn     u32GetDeviceIDFromName(coszDeviceName)
       * \brief  Returns DeviceID for the Device having Device name coszDeviceName
       * \param  coszDeviceName: [IN]  Device name  of the device
       * \retval  t_U32: Device ID having  Device name  coszDeviceName
       **************************************************************************/
      virtual t_U32 u32GetDeviceIDFromName(const t_String &coszDeviceName) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vGetSerialNumber
       ***************************************************************************/
      /*!
       * \fn     vGetSerialNumber
       * \brief  Returns Serial number for the Device with ID cou32DeviceID
       * \param  cou32DeviceID: [IN] Device ID
       * \param  coszSerialNo: [OUT] Serial number f the device
       **************************************************************************/
      virtual t_Void vGetSerialNumber(t_String &coszSerialNo, const t_U32 cou32DeviceID)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::enGetDeviceProfile
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceProfile(const t_U32 cou32DeviceID)
       * \brief  Returns Device PRofile for the Device with ID cou32DeviceID
       * \param  cou32DeviceID: [IN] Device ID
       * \retval  tenDeviceProfile: Device profile
       **************************************************************************/
      virtual tenDeviceProfile enGetDeviceProfile(const t_U32 cou32DeviceID) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListHandler::vSetDeviceProfile
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceProfile(const t_U32 cou32DeviceID)
       * \brief  Sets Device PRofile for the Device with ID cou32DeviceID
       * \param  cou32DeviceID: [IN] Device ID
       * \retval  enDeviceProfile: Device profile
       **************************************************************************/
      virtual t_Void vSetDeviceProfile(const t_U32 cou32DeviceID, tenDeviceProfile enDeviceProfile) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetSessionTransport()
       ***************************************************************************/
      /*!
       * \fn     vSetSessionTransport
       * \brief  Sets Transport on which session is established
       * \param  cou32DeviceHandle: [IN] Device ID
       * \param  enSessionTransport: [IN] Session Transport
       **************************************************************************/
      virtual t_Void vSetSessionTransport(const t_U32 cou32DeviceHandle, tenSessionTransport enSessionTransport)=0;
	  
	        /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetSelectionProgressState()
       ***************************************************************************/
      /*!
       * \fn     vSetSelectionProgressState
       * \brief  Sets Selection Progress state
       * \param  cou32DeviceHandle: [IN] Device ID
       * \param  enDeviceSelectionProgressState: [IN] Selection Progress state     
       **************************************************************************/
      virtual t_Void vSetSelectionProgressState(const t_U32 cou32DeviceHandle, tenDeviceSelectionProgressState enDeviceSelectionProgressState)=0;
	  

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::enGetSessionTransport()
       ***************************************************************************/
      /*!
       * \fn     enGetSessionTransport
       * \brief  Get Transport on which session is established
       * \param  cou32DeviceHandle: [IN] Device ID
       * \retval  enSessionTransport: Session Transport Type
       **************************************************************************/
      virtual tenSessionTransport enGetSessionTransport(const t_U32 cou32DeviceHandle)=0;
	  
	  /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetTechnologySelectionInDeviceListHandler
       ***************************************************************************/
      /*!
       * \fn     vSetTechnologySelection
       * \brief  Add the device to the device history
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enTechnologyPreference : Technology preference
       **************************************************************************/
      virtual t_Void vSetTechnologySelection(const t_U32 cou32DeviceHandle, tenDeviceCategory enTechnologyPreference)=0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::rGetDeviceConnectionCountInfo
       ***************************************************************************/
      /*!
       * \fn     rGetDeviceConnectionCountInfo
       * \brief  Returns device connection count information
       * \retval Structure containing device connection count information
       **************************************************************************/
      virtual trDeviceConnectionCountInfo rGetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle)=0;

	   /***************************************************************************
       ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceConnectionCountInfo
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectionCountInfo
       * \brief  Adding Device connection info to device list
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  rfrNewDeviceConnectionCountInfo : Structure containing device connection count information
       **************************************************************************/
      virtual t_Void vSetDeviceConnectionCountInfo(const t_U32 cou32DeviceHandle,trDeviceConnectionCountInfo &rfrNewDeviceConnectionCountInfo)=0;




      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclDeviceListIntf::enGetTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory enGetTechnologyPreference()
       * \brief  returns Device type preference
       * \param   coenDeviceType : Device Type
       **************************************************************************/
      virtual tenDeviceCategory enGetTechnologyPreference(const tenDeviceType coenDeviceType)=0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceListIntf::vSetTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetTechnologyPreference()
       * \brief  returns Device type preference
       * \param  coenDeviceType : Device Type
       * \param  coenTechnologyPref : Technology preference
       **************************************************************************/
      virtual t_Void vSetTechnologyPreference(const tenDeviceType coenDeviceType ,const tenDeviceCategory coenTechnologyPref)=0;

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::vSetVendorProductID
        ***************************************************************************/
       /*!
        * \fn     vSetVendorProductID
        * \brief  Updates vendor ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  cou32VendorID : Vendor ID
        * \param  cou32ProductID : Product ID
        **************************************************************************/
       virtual t_Void vSetVendorProductID(const t_U32 cou32DeviceHandle,
                      const t_U32 cou32VendorID,const t_U32 cou32ProductID)=0;
   
       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::u32GetVendorID
        ***************************************************************************/
       /*!
        * \fn     u32GetVendorID
        * \brief  Get vendor ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        **************************************************************************/
       virtual t_U32 u32GetVendorID(const t_U32 cou32DeviceHandle)=0;
      
       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::u32GetProductID
        ***************************************************************************/
       /*!
        * \fn     u32GetProductID
        * \brief  Get product ID
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        **************************************************************************/
       virtual t_U32 u32GetProductID(const t_U32 cou32DeviceHandle)=0;   


       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceWhitelistedInfo
        ***************************************************************************/
       /*!
        * \fn     vSetDeviceWhitelistedInfo
        * \brief  Set the Device Whitelisting Info
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  bIsDeviceWhiteListed : True if the Device is white listed
        **************************************************************************/
       virtual t_Void vSetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle, const t_Bool bIsDeviceWhiteListed) = 0;

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceWhitelistedInfo
        ***************************************************************************/
       /*!
        * \fn     bGetDeviceWhitelistedInfo
        * \brief  Get the Device Whitelisting Info
        * \param  cou32DeviceHandle : Uniquely identifies the target Device.
        * \retval    Return true if the Device is white listed
        **************************************************************************/
       virtual t_Bool bGetDeviceWhitelistedInfo(const t_U32 cou32DeviceHandle) = 0;

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::vSetSessionIP
        ***************************************************************************/
       /*!
        * \fn     vSetSessionIP
        * \brief  Sets the Session IP of the device identified by device handle
        * \param cou32DeviceHandle: Device Handle of the device to be added to history
        * \param rfrszDeviceName : Session IP to be set
        **************************************************************************/
       virtual t_Void vSetSessionIP(const t_U32 cou32DeviceHandle, t_String &rfrszSessionIP) = 0;

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::szGetSessionIP
        ***************************************************************************/
       /*!
        * \fn     szGetSessionIP
        * \brief  Get the Session IP of the device identified by device handle
        * \param cou32DeviceHandle: Device Handle of the device to be added to history
        **************************************************************************/
       virtual t_String szGetSessionIP(const t_U32 cou32DeviceHandle) = 0;

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::u32GetDeviceIDFromIPAddress
        ***************************************************************************/
       /*!
        * \fn     u32GetDeviceIDFromIPAddress(coszBTAddress)
        * \brief  Returns DeviceID for the Device having IP address coszBTAddress
        * \param  coszBTAddress: [IN] BTAddress of the device
        * \retval  t_U32: Device ID having IPTAddress coszIPAddress
        **************************************************************************/
        virtual t_U32 u32GetDeviceIDFromIPAddress(const t_String &coszIPAddress) = 0;

        /***************************************************************************
         ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceConnectionType()
         ***************************************************************************/
        /*!
        * \fn     vSetDeviceConnectionType
        * \brief  Sets Device Connection Type
        * \param  cou32DeviceHandle: [IN] Device ID
        * \param  enDeviceConnectionType: [IN] Device Connection Type
        **************************************************************************/
        virtual t_Void vSetDeviceConnectionType(const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnectionType) = 0 ;

        /***************************************************************************
         ** FUNCTION:  spi_tclDeviceListIntf::bUpdateDeviceName
         ***************************************************************************/
        /*!
         * \fn     bUpdateDeviceName
         * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
         * \param  [IN] szDeviceName : Device Name
         ***************************************************************************/
         virtual t_Bool bUpdateDeviceName(const t_U32 cou32DeviceHandle,t_String szDeviceName) = 0;


       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceListIntf::vEvaluateEnableInfoForCarlife
        ***************************************************************************/
       /*!
        * \fn     vEvaluateEnableInfoForCarlife
        * \brief  Evaluates device projection settings for Carlife technology
        * \param  rfrEnabledInfo: [OUT]Indicates if enSPIType is enabled or not
        * \param  enDeviceType: Indicates device type - android or apple
        * \retval  t_U32: Device ID having IPTAddress coszIPAddress
        **************************************************************************/
        virtual t_Void vEvaluateEnableInfoForCarlife(tenEnabledInfo &rfrEnabledInfo, tenDeviceType enDeviceType) = 0;

        /***************************************************************************
         ** FUNCTION: t_Void spi_tclDeviceListIntf::vSetDeviceSelectionReason
         ***************************************************************************/
        /*!
         * \fn     vSetDeviceSelectionReason
         * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
         * \param  [IN] enSelectReason : Device Selection reason
         ***************************************************************************/
        virtual t_Void vSetDeviceSelectionReason(const t_U32 cou32DeviceHandle, tenSelectReason enSelectReason) = 0;
		
		/***************************************************************************
         ** FUNCTION:  spi_tclDeviceListIntf::enGetLastActiveSessionTransport()
         ***************************************************************************/
        /*!
        * \brief  Get Transport on which session is established in the last trail
        * \param  cou32DeviceHandle: [IN] Device ID
        * \retval  enSessionTransport: Session Transport Type
        **************************************************************************/
        virtual tenSessionTransport enGetLastActiveSessionTransport(const t_U32 cou32DeviceHandle) = 0;

        /***************************************************************************
         ** FUNCTION:  spi_tclDeviceListIntf::vSetDeviceSwitchState
         ***************************************************************************/
        /*!
         * \fn     vSetDeviceSwitchState
         * \brief  Sets the Device Switch State
         * \param  cou32DeviceHandle [IN]:Device Handle
         * \param  enDeviceSwitchState[IN] : Device Switch state - PROJECTION/DEFAULT
         **************************************************************************/
        virtual t_Void vSetDeviceSwitchState(const t_U32 cou32DeviceHandle, tenUSBSwitchState enDeviceSwitchState) = 0;

        /***************************************************************************
         ** FUNCTION:  spi_tclDeviceListIntf::enGetDeviceSwitchState
         ***************************************************************************/
        /*!
         * \fn     enGetDeviceSwitchState
         * \brief  returns the Device Switch State
         * \param  cou32DeviceHandle :Device Handle
         * \retval enDeviceSwitchState : indicates Device Switch state - PROJECTION/DEFAULT
         **************************************************************************/
        virtual tenUSBSwitchState enGetDeviceSwitchState(const t_U32 cou32DeviceHandle) = 0;

        /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDeviceListIntf::vUpdateBTLimitedFunctionality
        ***************************************************************************/
        /*!
        * \fn       t_Void vUpdateBTLimitedFunctionality
        * \brief   To Post the BT Limitated Functionality
        * \param   cou32DeviceHandle : [IN] Device Handle
        * \param   m_bIsDeviceBTFunctionalityLimited : [IN] BT Limited functionality is set or not
        * \retval  t_Void
        **************************************************************************/
        virtual t_Void vUpdateBTLimitedFunctionality(const t_U32 cou32DeviceHandle, t_Bool bIsDeviceBTFunctionalityLimited) = 0;
};

#endif /* SPI_TCLDEVICELISTINTF_H_ */
