/*!
 *******************************************************************************
 * \file             spi_tclConnection.h
 * \brief            Base Connection class
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Base class for Connection classes. Provides basic connection
                 interfaces to be implemented by derived classes.
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 10.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 05.11.2014 |  Ramya Murthy                | Added callback for Application metadata.
 18.07.2017 |  Noopur R K                  | Added method vSetGeneralRestrictions.
 \endverbatim
 ******************************************************************************/
#ifndef SPI_TCLCONNECTION_H_
#define SPI_TCLCONNECTION_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <functional> //! C++11
#include "SPITypes.h"
#include "spi_ConnMngrTypeDefines.h"
#include <map>


#include "spi_tclConnSettingsIntf.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/


/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
/*!
 * \brief Index to various SPI Connections
 */
enum tenConnPointersIndex
{
   e8_CONN_ML_INDEX = 0,
   e8_CONN_DIPO_INDEX = 1,
   e8_CONN_AAUTO_INDEX = 2,
   e8_CONN_MYSPIN_INDEX = 3,

   //! Please add connection pointers above this in consecutive order
   //! This is required to maintain the size of enum
   e8_CONN_INDEX_SIZE
};

//!@TODO SPI Metadata info to be added

/*!
 * \class spi_tclConnection
 * \brief Base class for Connection classes. Provides basic connection
 *         interface to be implemented by derived classes
 */

class spi_tclConnection
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::spi_tclConnection
       ***************************************************************************/
      /*!
       * \fn     spi_tclConnection()
       * \brief  Default Constructor
       * \sa      ~spi_tclConnection()
       **************************************************************************/
      spi_tclConnection():m_poConnSettings(NULL)
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::~spi_tclConnection
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclConnection()
       * \brief  virtual Destructor
       * \sa     spi_tclConnection()
       **************************************************************************/
      virtual ~spi_tclConnection()
      {
    	  m_poConnSettings = NULL;
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::bInitializeConnection
       ***************************************************************************/
      /*!
       * \fn     bInitializeConnection()
       * \brief  required initializations.
       *         Mandatory interface
       * \retval returns true on successful initialization and false on failure
       **************************************************************************/
      virtual t_Bool bInitializeConnection() {return true;}

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnLoadSettings
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettings()
       * \brief  Called on loadsettings
       *         Optional interface
       * \param rfrheadUnitInfo : Head unit information
       * \param enCertificateType : CertificateType to be used for authentication
       **************************************************************************/
      virtual t_Void vOnLoadSettings(trHeadUnitInfo & /*rfrheadUnitInfo*/, tenCertificateType /*enCertificateType*/)
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnLoadSettingsCompleted
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettingsCompleted()
       * \brief  Invoked to notify that load settings has completed
       **************************************************************************/
      virtual t_Void vOnLoadSettingsCompleted()
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnSaveSettings
       ***************************************************************************/
      /*!
       * \fn     vOnSaveSettings()
       * \brief  Called on savesettings
       *         Optional interface
       * \retval none
       **************************************************************************/
      virtual t_Void vOnSaveSettings()
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \fn     bStartDeviceDetection()
       * \brief  Starts device detection
       * \retval returns true on successful detection and false on failure
       **************************************************************************/
      virtual t_Bool bStartDeviceDetection() {return true;}

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vUnInitializeConnection
       ***************************************************************************/
      /*!
       * \fn     vUnInitializeConnection()
       * \brief  Uninitialization of sdk's etc
       *         Mandatory interface
       **************************************************************************/
      virtual t_Void vUnInitializeConnection() {}

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *        callbacks.
       *         Mandatory interface
       * \param rfrConnCallbacks : reference to the callback structure
       *        populated by the caller
       **************************************************************************/
      virtual t_Void vRegisterCallbacks(trConnCallbacks &rfrConnCallbacks) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnAddDevicetoList
       ***************************************************************************/
      /*!
       * \fn     vOnAddDevicetoList()
       * \brief  called when a new device has to be added to device list
       *         Optional interface
       * \param  cou32DeviceHandle: Device handle of the device to be added to
       *         the device list
       **************************************************************************/
      virtual t_Void vOnAddDevicetoList(const t_U32 /*cou32DeviceHandle*/)
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::bSetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bSetDevProjUsage()
       * \brief  Called when the SPI featured is turned ON or OFF by the user.
       *         Mandatory interface
       * \param  enServiceStatus : Sets the particular SPI service ON or OFF
       **************************************************************************/
      virtual t_Bool bSetDevProjUsage(tenEnabledInfo /*enServiceStatus*/) {return true;}

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vOnSelectDevice
       * \brief  Called when a device is selected by the user. Optional interface
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDevConnType   : Identifies the Connection Type.
       * \param  enDevSelReq    : Identifies the Connection Request.
       * \param  enDAPUsage      : Identifies Usage of DAP for the selected ML device.
       *              This value is not considered for de-selection of device.
       * \param  enCDBUsage      : Identifies Usage of CDB for the selected ML device.
       *              This value is not considered for de-selection of device
       * \param  enSelectionReason   :  [IN] Describes whether the device Selection or Deselection is
       *               triggered due to a user action.
       * \param  corUsrCntxt      : User Context Details.
       **************************************************************************/
      virtual t_Void vOnSelectDevice(const t_U32 /*cou32DeviceHandle*/,
               tenDeviceConnectionType /*enDevConnType*/,
               tenDeviceConnectionReq /*enDevSelReq*/, tenEnabledInfo /*enDAPUsage*/,
               tenEnabledInfo /*enCDBUsage*/, tenSelectReason /*enSelectionReason*/,
               const trUserContext /*corUsrCntxt*/, tenDeviceType /*enDeviceType = e8_UNKNOWN_DEVICE*/)
      {
         //! TODO RD Replace the arguments into a structure
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vSetRoleSwitchRequestedInfo
       ***************************************************************************/
      /*!
       * \fn     vSetRoleSwitchRequestedInfo
       * \brief  Function to upddate the roleswitch requested info.
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       **************************************************************************/
      virtual t_Void vSetRoleSwitchRequestedInfo(const t_U32 /*cou32DeviceHandle*/)
      {
      }

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConnection::vOnSelectDeviceResult()
      ***************************************************************************/
      /*!
      * \fn      t_Void vOnSelectDeviceResult
      * \brief   To perform the actions that are required, after the select device is
      *           successful/failed
      * \pram    cou32DeviceHandle  : [IN] Uniquely identifies the target Device.
      * \pram    enDevSelReq : [IN] Identifies the Connection Request.
      * \pram    coenRespCode: [IN] Response code. Success/Failure
      * \pram    enDevCat    : [IN] Device Category. ML/DiPo
      * \param  enSelectReason   :  [IN] Describes whether the device Selection or Deselection is
      *               triggered due to a user action.
      * \retval  t_Void
      **************************************************************************/
      virtual t_Void vOnSelectDeviceResult(const t_U32 /*cou32DeviceHandle*/,
         const tenDeviceConnectionReq /*coenConnReq*/,
         const tenResponseCode /*coenRespCode*/,
         tenDeviceCategory /*enDevCat*/,tenSelectReason /*enSelectionReason*/, tenDeviceType /*enDeviceType = e8_UNKNOWN_DEVICE*/)
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vSetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSwitchInfo
       * \brief  Set device switch information
       * \param  rfrDeviceSwitchInfo : Device switch info
       **************************************************************************/
      virtual t_Void vSetDeviceSwitchInfo(trDeviceSwitchInfo & /*rfrDeviceSwitchInfo*/)
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vSetGeneralRestrictions
       ***************************************************************************/
      /*!
       * \fn     vSetGeneralRestrictions
       * \brief  Set general restriction information
       * \param  void
       **************************************************************************/
      virtual t_Void vSetGeneralRestrictions(t_U16 /*u16GeneralRestrictionInfo*/)
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vUpdateDeviceConnectionCountInfo
       ***************************************************************************/
      /*!
       * \fn     vUpdateDeviceConnectionCountInfo
       * \brief  Update device Connection Count information
       * \param  rDeviceConnectionCountInfo : Structure for Device connection count info
       **************************************************************************/
      virtual t_Void vUpdateDeviceConnectionCountInfo(trDeviceConnectionCountInfo & /*rDeviceConnectionCountInfo*/)
      {

      }

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConnection::vRetryConnection()
      ***************************************************************************/
      /*!
      * \fn      t_Void vRetryConnection()
      * \brief   Method to retry to establish session
      * \pram    u32DeviceID  : [IN] Device Handle.
      * \retval  t_Void
      **************************************************************************/
      virtual t_Void vRetryConnection(t_U32 /*u32DeviceID*/){}

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConnection::vSetConnectionsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetConnectionsettingsInstance()
       * \brief   Method to set connection settings instance.
       * \param   poConnSettingsIntf  : pointer to ConnsettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetConnectionsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
      {
         if( NULL!= poConnSettingsIntf)
         {
            m_poConnSettings = poConnSettingsIntf;
         }
      }
	  
	  /***************************************************************************
      ** FUNCTION:  spi_tclConnection::vSetSelectionProgressState
      **************************************************************************/
     /*!
      * \fn      vSetSelectionProgressState()
      * \brief   set selection state to BDCL Connmanger
      * \retval  t_Void
      *************************************************************************/
      virtual t_Void vOnSetSelectionProgressState(tenDeviceSelectionState /*enDeviceSelectionState*/){}

   protected:

      spi_tclConnSettingsIntf* m_poConnSettings;

   private:

      /***************************************************************************
       *********************************PRIVATE************************************
       ***************************************************************************/

       /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnDeviceConnection
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceConnection()
       * \brief  Callback to be called on new device detection by the derived class
       * \param cou32DeviceHandle: Device handle of the detected device
       * \param corfrDeviceInfo: reference to the structure containing device info
       **************************************************************************/
      virtual t_Void vOnDeviceConnection(const t_U32 /*cou32DeviceHandle*/,
               const trDeviceInfo & /*corfrDeviceInfo*/) {};

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnDeviceDisconnection
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceDisconnection()
       * \brief Callback to be called on device disconnection by the derived class
       * \param cou32DeviceHandle: Device Handle of the disconnected device
       **************************************************************************/
      virtual t_Void vOnDeviceDisconnection(const t_U32 /*cou32DeviceHandle*/) {};

};

typedef spi_tclConnection* GetConnection_t();
extern std::map<tenDeviceCategory,GetConnection_t*> Connection;

/*! } */
#endif // SPI_TCLCONNECTION_H_
